/*
 * Copyright (C) 2010 Motorola
 * Copyright (C) 2011-2012 Canonical, Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef WAUDIO_H
#define WAUDIO_H

#include <stdint.h>
#include <stddef.h>

#define ANDROID_AUDIO_WRAPPER_ERROR_START 0x9000000
typedef unsigned long uint32;

typedef int status_type;
enum{
	OK = 0,
	NO_ERROR = 0,
//UNKNOWN_ERROR,
//INIT_FAIL = -ENODEV, 
	INVALID_PARAM = ANDROID_AUDIO_WRAPPER_ERROR_START,
};

 /* As a convenience, if a callback is supplied, a handler thread
     * is automatically created with the appropriate priority. This thread
     * invokes the callback when a new buffer becomes availlable or an underrun condition occurs.
     * Parameters:
     *
     * event:   type of event notified (see enum AudioTrack::event_type).
     * user:    Pointer to context for use by the callback receiver.
     * info:    Pointer to optional parameter according to event type:
     *          - EVENT_MORE_DATA: pointer to AudioTrack::Buffer struct. The callback must not write
     *          more bytes than indicated by 'size' field and update 'size' if less bytes are
     *          written.
     *          - EVENT_UNDERRUN: unused.
     *          - EVENT_LOOP_END: pointer to an int indicating the number of loops remaining.
     *          - EVENT_MARKER: pointer to an uin32_t containing the marker position in frames.
     *          - EVENT_NEW_POS: pointer to an uin32_t containing the new position in frames.
     *          - EVENT_BUFFER_END: unused.
     */

typedef void (*callback_t)(int event, void* user, void *info);

typedef enum _stream_type{
DEFAULT = -1,
VOICE_CALL = 0,
SYSTEM = 1,
RING = 2,
MUSIC = 3,
ALARM = 4,
NOTIFICATION = 5,
BLUETOOTH_SCO = 6,
ENFORCED_AUDIBLE = 7,
DTMF = 8,
TTS = 9,
NUM_STREAM_TYPES
}stream_type;

/* Events used by AudioTrack callback function (audio_track_cblk_t).
     */
enum _event_type {
        EVENT_MORE_DATA = 0,        // Request to write more data to PCM buffer.
        EVENT_UNDERRUN = 1,         // PCM buffer underrun occured.
        EVENT_LOOP_END = 2,         // Sample loop end was reached; playback restarted from loop start if loop count was not 0.
        EVENT_MARKER = 3,           // Playback head is at the specified marker position (See setMarkerPosition()).
        EVENT_NEW_POS = 4,          // Playback head is at a new position (See setPositionUpdatePeriod()).
        EVENT_BUFFER_END = 5        // Playback head is at the end of the buffer.
}event_type;

/* Do not change these values without updating their counterparts
 * in media/java/android/media/MediaRecorder.java!
 */
typedef enum _audio_source {
    AUDIO_SOURCE_DEFAULT = 0,
    AUDIO_SOURCE_MIC = 1,
    AUDIO_SOURCE_VOICE_UPLINK = 2,
    AUDIO_SOURCE_VOICE_DOWNLINK = 3,
    AUDIO_SOURCE_VOICE_CALL = 4,
    AUDIO_SOURCE_CAMCORDER = 5,
    AUDIO_SOURCE_VOICE_RECOGNITION = 6,
    AUDIO_SOURCE_MAX = AUDIO_SOURCE_VOICE_RECOGNITION,

    AUDIO_SOURCE_LIST_END  // must be last - used to validate audio source type
}audio_source;


// Audio sub formats (see AudioSystem::audio_format).
enum pcm_sub_format {
        PCM_SUB_16_BIT          = 0x1, // must be 1 for backward compatibility
        PCM_SUB_8_BIT           = 0x2, // must be 2 for backward compatibility
};

 // Audio format consists in a main format field (upper 8 bits) and a sub format field (lower 24 bits).
 // The main format indicates the main codec type. The sub format field indicates options and parameters
 // for each format. The sub format is mainly used for record to indicate for instance the requested bitrate
 // or profile. It can also be used for certain formats to give informations not present in the encoded
 // audio stream (e.g. octet alignement for AMR).
typedef enum _audio_format {
        INVALID_FORMAT      = -1,
        FORMAT_DEFAULT      = 0,
        PCM                 = 0x00000000, // must be 0 for backward compatibility
        MP3                 = 0x01000000,
        AMR_NB              = 0x02000000,
        AMR_WB              = 0x03000000,
        AAC                 = 0x04000000,
        HE_AAC_V1           = 0x05000000,
        HE_AAC_V2           = 0x06000000,
        VORBIS              = 0x07000000,
        MAIN_FORMAT_MASK    = 0xFF000000,
        SUB_FORMAT_MASK     = 0x00FFFFFF,
        // Aliases
        PCM_16_BIT          = (PCM|PCM_SUB_16_BIT),
        PCM_8_BIT          = (PCM|PCM_SUB_8_BIT)
    }audio_format;

// Channel mask definitions must be kept in sync with JAVA values in /media/java/android/media/AudioFormat.java
typedef enum _audio_channels {
        // output channels
        CHANNEL_OUT_FRONT_LEFT = 0x4,
        CHANNEL_OUT_FRONT_RIGHT = 0x8,
        CHANNEL_OUT_FRONT_CENTER = 0x10,
        CHANNEL_OUT_LOW_FREQUENCY = 0x20,
        CHANNEL_OUT_BACK_LEFT = 0x40,
        CHANNEL_OUT_BACK_RIGHT = 0x80,
        CHANNEL_OUT_FRONT_LEFT_OF_CENTER = 0x100,
        CHANNEL_OUT_FRONT_RIGHT_OF_CENTER = 0x200,
        CHANNEL_OUT_BACK_CENTER = 0x400,
        CHANNEL_OUT_MONO = CHANNEL_OUT_FRONT_LEFT,
        CHANNEL_OUT_STEREO = (CHANNEL_OUT_FRONT_LEFT | CHANNEL_OUT_FRONT_RIGHT),
        CHANNEL_OUT_QUAD = (CHANNEL_OUT_FRONT_LEFT | CHANNEL_OUT_FRONT_RIGHT |
                CHANNEL_OUT_BACK_LEFT | CHANNEL_OUT_BACK_RIGHT),
        CHANNEL_OUT_SURROUND = (CHANNEL_OUT_FRONT_LEFT | CHANNEL_OUT_FRONT_RIGHT |
                CHANNEL_OUT_FRONT_CENTER | CHANNEL_OUT_BACK_CENTER),
        CHANNEL_OUT_5POINT1 = (CHANNEL_OUT_FRONT_LEFT | CHANNEL_OUT_FRONT_RIGHT |
                CHANNEL_OUT_FRONT_CENTER | CHANNEL_OUT_LOW_FREQUENCY | CHANNEL_OUT_BACK_LEFT | CHANNEL_OUT_BACK_RIGHT),
        CHANNEL_OUT_7POINT1 = (CHANNEL_OUT_FRONT_LEFT | CHANNEL_OUT_FRONT_RIGHT |
                CHANNEL_OUT_FRONT_CENTER | CHANNEL_OUT_LOW_FREQUENCY | CHANNEL_OUT_BACK_LEFT | CHANNEL_OUT_BACK_RIGHT |
                CHANNEL_OUT_FRONT_LEFT_OF_CENTER | CHANNEL_OUT_FRONT_RIGHT_OF_CENTER),
        CHANNEL_OUT_ALL = (CHANNEL_OUT_FRONT_LEFT | CHANNEL_OUT_FRONT_RIGHT |
                CHANNEL_OUT_FRONT_CENTER | CHANNEL_OUT_LOW_FREQUENCY | CHANNEL_OUT_BACK_LEFT | CHANNEL_OUT_BACK_RIGHT |
                CHANNEL_OUT_FRONT_LEFT_OF_CENTER | CHANNEL_OUT_FRONT_RIGHT_OF_CENTER | CHANNEL_OUT_BACK_CENTER),

        // input channels
        CHANNEL_IN_LEFT = 0x4,
        CHANNEL_IN_RIGHT = 0x8,
        CHANNEL_IN_FRONT = 0x10,
        CHANNEL_IN_BACK = 0x20,
        CHANNEL_IN_LEFT_PROCESSED = 0x40,
        CHANNEL_IN_RIGHT_PROCESSED = 0x80,
        CHANNEL_IN_FRONT_PROCESSED = 0x100,
        CHANNEL_IN_BACK_PROCESSED = 0x200,
        CHANNEL_IN_PRESSURE = 0x400,
        CHANNEL_IN_X_AXIS = 0x800,
        CHANNEL_IN_Y_AXIS = 0x1000,
        CHANNEL_IN_Z_AXIS = 0x2000,
        CHANNEL_IN_VOICE_UPLINK = 0x4000,
        CHANNEL_IN_VOICE_DNLINK = 0x8000,
        CHANNEL_IN_MONO = CHANNEL_IN_FRONT,
        CHANNEL_IN_STEREO = (CHANNEL_IN_LEFT | CHANNEL_IN_RIGHT),
        CHANNEL_IN_ALL = (CHANNEL_IN_LEFT | CHANNEL_IN_RIGHT | CHANNEL_IN_FRONT | CHANNEL_IN_BACK|
                CHANNEL_IN_LEFT_PROCESSED | CHANNEL_IN_RIGHT_PROCESSED | CHANNEL_IN_FRONT_PROCESSED | CHANNEL_IN_BACK_PROCESSED|
                CHANNEL_IN_PRESSURE | CHANNEL_IN_X_AXIS | CHANNEL_IN_Y_AXIS | CHANNEL_IN_Z_AXIS |
                CHANNEL_IN_VOICE_UPLINK | CHANNEL_IN_VOICE_DNLINK)
    }audio_channels;



typedef void* TrackHandle;

typedef void* RecordHandle;

typedef struct _buffer_struct{
//   enum {
//            MUTE    = 0x00000001
//        };
        uint32_t    flags;
        int         channelCount;
        int         format;
        size_t      frameCount;
        size_t      size;
        union {
            void*       raw;
            short*      i16;
            int8_t*     i8;
        };
}buffer_struct;

/*Audio Track related interface*/
TrackHandle CreateAudioTrack();

void DestoryAudioTrack(TrackHandle handle);

void AudioTrack_start(TrackHandle handle);

void AudioTrack_stop(TrackHandle handle);

status_type AudioTrack_getposition(TrackHandle handle, uint32 *position);

//status_type AudioTrack_setposition(TrackHandle handle,int32_t position);

size_t AudioTrack_write(TrackHandle handle,const void *buffer, size_t size);

status_type AudioTrack_set(TrackHandle handle,stream_type streamtype, uint32 samplerate,audio_format format, audio_channels hannel);

void AudioTrack_pause(TrackHandle handle);

status_type AudioTrack_set_asyn(TrackHandle handle,stream_type streamtype, uint32 samplerate,audio_format format, audio_channels hannel,callback_t cbf,void* user);

uint32 AudioTrack_latency(TrackHandle handle);

void AudioTrack_mute(TrackHandle handle, int muted);

/*Audio Record related interface*/
RecordHandle CreateAudioRecord();

void DestoryAudioRecord(RecordHandle handle);

void AudioRecord_start(RecordHandle handle);

void AudioRecord_stop(RecordHandle handle);

status_type AudioRecord_getposition(RecordHandle handle,int32_t *position);

//status_type AudioRecord_setposition(RecordHandle handle,int32_t position);

size_t AudioRecord_read(RecordHandle handle,void *buffer, size_t size);

status_type AudioRecord_set(RecordHandle handle,audio_source inputsource);

//void AudioRecord_pause(RecordHandle handle);

uint32 AudioRecord_latency(RecordHandle handle);

/*Audio System related interface*/
status_type AudioSystem_getStreamVolumeIndex(stream_type streamtype,int *index );


status_type AudioSystem_getStreamMute(stream_type streamtype, int *mute );


status_type AudioSystem_setStreamVolumeIndex(stream_type streamtype,int index );


status_type AudioSystem_setStreamMute(stream_type streamtype, int mute );

#endif

