#!/usr/bin/env python

# This file is part of Atabake
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Authors: Artur Duque de Souza <artur.souza@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

__author__ = "Artur Duque de Souza / Leonardo Sobral Cunha"
__author_email__ = "artur.souza@openbossa.org / leonardo.cunha@openbossa.org"

import os
import sys
import gobject
import select
import signal
import StringIO
import shlex
import subprocess
import logging

import atabake.lib.utils as utils
from atabake.lib.errors import *
from atabake.lib.player import Player
from atabake.lib.player_session import PlayerSession
from atabake.lib.config_file import AtabakeConfigFile


log = logging.getLogger("atabake.player.mplayer")

class AnswerException(Exception):
    """Exception to be raised when there is a problem
    inside _read_ans function
    """
    pass


class MPlayerBackend(Player):
    """
    MPlayer backend for playing audio/video files.

    This class spawns an mplayer process for playing the requested url.
    """
    name = "mplayer"
    exec_name = "mplayer"
    mplayer_path = utils.which(exec_name)
    priority = 1
    enabled = True

    def __init__(self, url, xid, get_state, eos_cb, error_cb, buff_cb):
        Player.__init__(self, url, xid, get_state, eos_cb, error_cb, buff_cb)
        self._init_state()
        self.conf = AtabakeConfigFile()
        self.volume = None
        self.pause_cookie = 0
        self.last_position = 0

    def _init_state(self):
        self.proc = None
        self.info = {}
        self.started_playback = False
        self.out_watcher = -1
        self.error_watcher = -1
        Player.reset_player(self)

    def _reset_state(self):
        if self.out_watcher >= 0:
            gobject.source_remove(self.out_watcher)
        if self.error_watcher >= 0:
            gobject.source_remove(self.error_watcher)
        self._init_state()

    @staticmethod
    def _kill_proc(proc):
        if proc is None:
            return

        def died():
            return proc.poll() is not None

        if died():
            proc.wait()
            return

        log.debug("mplayer didn't die, sending SIGTERM ...")
        os.kill(proc.pid, signal.SIGTERM)
        time.sleep(0.2)

        if died():
            proc.wait()
            return

        log.debug("mplayer didn't die, sending SIGKILL ...")
        os.kill(proc.pid, signal.SIGKILL)
        proc.wait()

    def _run_mplayer(self, input=None, output=None, error=None):
        log.debug("Executing MPlayer with: %s", self.mplayer_cmd)
        self.proc = subprocess.Popen(self.mplayer_cmd, stdin=input,
                                     stdout=output, stderr=error,
                                     close_fds=True)

    def _setup_mplayer_cmd(self):
        self.mplayer_cmd = [self.mplayer_path, "-quiet", "-slave",
                            "-noconsolecontrols", "-nojoystick",
                            "-nolirc", "-nomouseinput", "-osdlevel", "0",
                            "-idx", "-vo"]

        vo = "xv"
        if self.conf.has_option("Mplayer", "vo"):
            vo = self.conf.get("Mplayer", "vo")
        elif os.path.exists("/etc/osso_software_version"):
            osso_file = open("/etc/osso_software_version")
            osso_version = osso_file.readline()
            if osso_version.find("2006") >= 0 or osso_version.find("HACKER") >= 0:
                vo = "omapfb"
            osso_file.close()

        self.mplayer_cmd.append(vo)

        if self.conf.has_option("Mplayer", "opts"):
            opts = self.conf.get("Mplayer", "opts")
            for option in opts.split():
                self.mplayer_cmd.append(option)

        ignore_keys = "/usr/share/atabake/ignore_keys.conf"
        if os.path.exists(ignore_keys):
            self.mplayer_cmd.extend(("-input", "conf=" + ignore_keys))

        if len(self.preferences) > 0:
            for pref, value in self.preferences.iteritems():
                if pref == "fullscreen" and value:
                    self.mplayer_cmd.append("-fs")

                elif pref == "nocache" and value:
                    self.mplayer_cmd.append("-nocache")
                else:
                    self.mplayer_cmd.append(pref)
                    self.mplayer_cmd.append(value)
                log.debug("Loaded: %s:%s -> %s" % (pref, value,
                                                   self.mplayer_cmd))

                log.debug("Loaded preferences for mplayer")

        if self.xid:
            log.info("Setting xid (0x%x) in mplayer", self.xid)
            self.mplayer_cmd.append("-wid")
            self.mplayer_cmd.append("0x%x" % (self.xid))

        # to play from file
        self.mplayer_cmd.append(self.url)

    def _cmd(self, msg, pausing_keep=True):
        try:
            if not pausing_keep or not self.is_paused():
                prefix = ""
            else:
                prefix = "pausing_keep "

            cmd = "%s%s\n" % (prefix, msg)
            self.proc.stdin.write(cmd)
            self.proc.stdin.flush()
            log.debug("MPlayer command: %r", cmd)
        except Exception, e:
            log.error("Error writing MPlayer command %r: %s", cmd, e)
            if self.proc:
                try:
                    self.proc.stdin.close()
                    self._kill_proc(self.proc)
                except Exception, e:
                    log.error(e, exc_info=True)
                    raise
                finally:
                    self.proc = None

    def _handle_eos(self):
        log.debug("In handle eos")
        if not self.is_holded():
            if self.proc is not None:
                self.proc.wait()

            if not self.is_idle() and self.eos_callback:
                log.debug("Quitting, emit EOS")
                self.eos_callback()

            self._reset_state()

    def _register_io_handlers(self):
        flags = gobject.IO_IN | gobject.IO_PRI | gobject.IO_ERR | gobject.IO_HUP
        self.out_watcher = gobject.io_add_watch(self.proc.stdout,
                                                flags, self._io_handler)
        self.error_watcher = gobject.io_add_watch(self.proc.stderr,
                                                  flags, self._error_handler)

    def _error_handler(self, fd_, flags):
        # Right now we just look for file not found errors
        if self.proc:
            line = self._readline(fd=self.proc.stderr)
            if line.find("File not found") >= 0:
                log.error("File not found")
                self.error_callback(AtabakeException.ERROR_FILE_NOT_FOUND)
                return False
            else:
                log.debug("Ignored MPlayer error output during parse: %r", line)
        return True

    def _io_handler(self, fd, flags):
        # FIXME: _io_handler and _read_ans read the same output pipe
        if flags & gobject.IO_ERR:
            log.debug("Error reading MPlayer output")
            return False
        elif flags & gobject.IO_HUP:
            log.debug("Exit signal received from MPlayer")
            self._handle_eos()
            return False
        else:
            # FIXME: we need to change if -> while to parse the A: status
            if self.is_playing():
                line = self._readline()
                if line.startswith("A:") and line.endswith("\r"):
                    pieces = line.split()
                    pos = float(pieces[1])
                    return True
                elif line in ("Exiting... (Quit)\n", "Exiting... (End of file)\n"):
                    self._handle_eos()
                    return False
                elif line == "Starting playback...\n":
                    log.debug("Starting playback!")
                    self.buffering_callback(100.0)

                    self.started_playback = True
                    log.debug("Restoring volume: %s", self.volume)
                    if self.volume is not None:
                        self._cmd("volume %d 1" % self.volume)

                    while line:
                        log.debug("emptying buffer, line = %r", line)
                        line = self._readline(100)
                        if line and line.startswith("A:") and line.endswith("\r"):
                            break
                    return True
                elif line.find("Cache fill") >= 0:
                    percentage = line.replace(" ", "").split(":")[1]
                    percentage = float(percentage.split("%")[0])
                    self.buffering_callback(percentage)
                    return True
                else:
                    log.debug("Ignored MPlayer output during parse: %r", line)
            return True

    def _readline(self, timeout=0, fd=None):
        if not self.proc:
            return ""

        p = select.poll()
        flag_err = select.POLLERR | select.POLLHUP

        if fd is None:
            rfile = self.proc.stdout
        else:
            rfile = fd

        p.register(rfile, flag_err | select.POLLIN | select.POLLPRI)
        buf = StringIO.StringIO()

        while (self.is_playing() or self.is_paused()):

            lst = p.poll(timeout)
            if not lst:
                line = buf.getvalue()
                buf.close()
                log.debug("Time out!: %r", line)
                return line

            for fd, flags in lst:
                if flags & flag_err:
                    log.error("Error reading MPlayer: %s, "
                              "flags=%x", fd, flags)
                    return ""

                c = rfile.read(1)
                buf.write(c)

                if c == "\n" or c == "\r":
                    line = buf.getvalue()
                    buf.close()
                    return line

    def _read_ans(self, prefix="", timeout=0, tries=1):
        if not self.started_playback:
            return None

        while tries and (self.is_playing() or self.is_paused()):
            tries -= 1
            line = self._readline(timeout)

            if not line:
                continue

            last = line[-1]
            if last == "\n" or last == "\r":
                line = line[: -1]

            if not prefix:
                return line
            elif line.startswith(prefix):
                return line[len(prefix):]
            else:
                log.debug("Ignored MPlayer output in read ans: %r", line)

        # exceeded our tries
        raise AnswerException("Maximum number of tries")

    ## Player controls

    def play(self):
        try:
            self._setup_mplayer_cmd()
            self._run_mplayer(input=subprocess.PIPE, output=subprocess.PIPE,
                              error=subprocess.PIPE)
            self._register_io_handlers()
        except Exception, e:
            log.error("Failed to execute \"%s\": %s", self.mplayer_cmd, e)
            self._kill_proc(self.proc)
            raise PlayError(e)

    def pause(self):
        # XXX: kill mplayer after some time while paused
        # because it's high energy consumption
        def stop_player(cookie):
            if self.is_paused() and self.pause_cookie == cookie:
                self.stop()
            return False

        try:
            # mplayer isn't running
            if self.is_paused() and not self.proc:
                self.play()
                self.seek(self.last_postition)
                return

            if not self.is_paused() and not self._is_video():
                self.pause_cookie += 1
                gobject.timeout_add(60000, stop_player, self.pause_cookie)
                self.last_postition = self.get_position()

            self._cmd("pause")
        except Exception, e:
            log.error(e, exc_info=True)
            raise PauseError(e)

    def stop(self):
        def stop_mplayer(proc):
            self._kill_proc(proc)
            return False

        try:
            proc = self.proc
            self._cmd("quit")
            self._reset_state()

            # give some time to mplayer shut itself down
            gobject.timeout_add(1000, stop_mplayer, proc)
        except Exception, e:
            log.error(e, exc_info=True)
            raise StopError(e)

    def is_seekable(self):
        return True

    def seek(self, pos):
        try:
            # convert pos from milisecs (int32) to secs (float)
            seek_set = 2
            pos_secs = pos / 1000.0
            self._cmd("seek %s %s" % (pos_secs, seek_set))
        except Exception, e:
            log.error(e, exc_info=True)
            raise SeekError(e)

        return (pos_secs, seek_set)

    ## Getters

    def get_position(self):
        try:
            self._cmd("get_time_pos")
            pos_secs = self._read_ans("ANS_TIME_POSITION=",
                                      timeout=1000, tries=10)
        except AnswerException, e:
            log.warning(e, exc_info=True)
            return 0
        except Exception, e:
            log.error(e, exc_info=True)
            raise GetPositionError(e)

        pos = pos_secs and float(pos_secs.replace(",", ".")) or 0.0
        pos = int(pos * 1000.0)
        log.debug("Got position %d ms from mplayer", pos)
        return pos

    def get_duration(self):
        try:
            self._cmd("get_time_length")
            duration_secs = self._read_ans("ANS_LENGTH=",
                                           timeout=1000, tries=10)
        except AnswerException, e:
            log.warning(e, exc_info=True)
            return 0
        except Exception, e:
            log.warning(e, exc_info=True)
            raise GetDurationError(e)

        dur = duration_secs and float(duration_secs.replace(",", ".")) or 0.0
        dur = int(dur * 1000.0)
        log.debug("Got duration %d ms from mplayer", dur)
        return dur

    def get_media_details(self):
        def format_str(line):
            return line

        def format_float(line):
            return line.replace(",", ".")

        def format_res(line):
            (width, height) = line.split(" x ")
            return (width, height)

        if not self.proc or not self.started_playback:
            return {}

        spec = (("audio bitrate", "get_audio_bitrate",
                    "ANS_AUDIO_BITRATE=", format_str),
                ("audio codec", "get_audio_codec",
                    "ANS_AUDIO_CODEC=", format_str),
                ("audio samples", "get_audio_samples",
                    "ANS_AUDIO_SAMPLES=", format_str),
                ("video bitrate", "get_video_bitrate",
                    "ANS_VIDEO_BITRATE=", format_str),
                ("video res", "get_video_resolution",
                    "ANS_VIDEO_RESOLUTION=", format_res),
                ("video codec", "get_video_codec",
                    "ANS_VIDEO_CODEC=", format_str),
                ("album", "get_meta_album",
                    "ANS_META_ALBUM=", format_str),
                ("artist", "get_meta_artist",
                    "ANS_META_ARTIST=", format_str),
                ("genre", "get_meta_genre",
                    "ANS_META_GENRE=", format_str),
                ("comment", "get_meta_comment",
                    "ANS_META_COMMENT=", format_str),
                ("title", "get_meta_title",
                    "ANS_META_TITLE=", format_str),
                ("track", "get_meta_track",
                    "ANS_META_TRACK=", format_str),
                ("year", "get_meta_year",
                    "ANS_META_YEAR=", format_str),
                ("length", "get_time_length",
                    "ANS_LENGTH=", format_float),
               )
        info = {}

        try:
            for key, cmd, ans_prefix, conv_func in spec:
                self._cmd(cmd)
                v = self._read_ans(ans_prefix, timeout=1000, tries=10)
                if not v:
                    continue

                v = shlex.split(v)
                if len(v) == 1:
                    v = v[0]

                value = conv_func(v)
                if value:
                    if key == "video res":
                        (width, height) = value
                        info["width"] = width
                        info["height"] = height
                    else:
                        info[key] = value

        except Exception, e:
            log.error(e, exc_info=True)
            raise GetMediaDetailsError(e)

        self.info = info
        log.info(str(info))
        return info

    def _is_video(self):
        try:
            self._cmd("get_video_bitrate")
            video_bitrate = self._read_ans("ANS_VIDEO_BITRATE=",
                                           timeout=1000, tries=10)
        except Exception, e:
            log.error(e, exc_info=True)
            raise GetMediaDetailsError(e)

        if len(video_bitrate) < 4:
            return False
        else:
            return True

    ## Setters

    def set_volume(self, value):
        self.volume = value
        try:
            self._cmd("volume %d 1" % value)
            log.debug("Volume set to %s", value)
        except Exception, e:
            log.error(e, exc_info=True)
            raise SetVolumeError(e)

    def set_fullscreen(self, status):
        try:
            if self.is_idle():
                self.preferences["fullscreen"] = bool(status)
            else:
                self._cmd("vo_fullscreen %d" % status)
        except Exception, e:
            log.error(e, exc_info=True)
            raise SetFullScreenError(e)

    def set_video_window(self, xid):
        self.xid = xid
        log.debug("Setting xid in mplayer")
