#! /bin/bash
# Set Axyl website permissions
# Execute this script from inside an Axyl website root directory to
# ensure that permissions are set correctly.
#
# With no parameters passed on the command-line, this script will
# try to find out the Webserver user and group, and assume the
# Web Administrator is 'wwwadmin'. Optionally you may pass these
# in as follows:
#      $1  -  Webserver Administrator username
#      $2  -  Webserver username
#      $3  -  Webserver group (if not supplied, assumed same as user)
#
# Ownership and group settings will then be based on these values, with
# most of the website owned by the Axyl User, and a group of the
# Webserver Administrator. Only the 'var' sub-directory differs from
# this, with a group of Webserver Group.
#
# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
set -e

# Sanity checks on where we are..
if [ ! -f "./application.xml" -o ! -d "./lib" ]
then
  echo "you must be in the root directory of an Axyl website to run"
  echo "this script. Change directory first."
  exit 1
fi

if [ `pwd` = "/" ]
then
  echo "you cannot execute this script from the root directory!"
  exit 1
fi

# Get common funcs and config vars etc.
# Acquire the local configuration..
CONFDIR=/etc/axyl
CONF=${CONFDIR}/axyl.conf
if [ ! -f $CONF ] ; then
  echo "Axyl configuration file $CONF not found!"
  exit 2
else
  . $CONF
  if [ ! -d $AXYL_HOME ] ; then
    echo "FATAL: the Axyl root directory '$AXYL_HOME' does not exist."
    exit 6
  fi
  . ${AXYL_HOME}/install/axyl-common.sh
fi

# Defaults for webserver users and group..
WWWUSER=$APACHE_USER
WWWGROUP=$APACHE_GROUP
WWWADMIN=wwwadmin

# Deal with passed-in parameters..
if [ "$1" != "" ] ; then
  WWWADMIN=$1
  if [ "$2" != "" ] ; then
    WWWUSER=$2
    if [ "$3" != "" ] ; then
      WWWGROUP=$3
    else
      WWWGROUP=$WWWUSER
    fi
  fi
fi
mkuser $WWWUSER system $WWWGROUP any /var/www default "WWW User"

# Web hosting administration group..
mkgroup $WWWADMIN any

# Generally writeable only for axyl and WWW administrators..
chown -R ${AXUSER}:${WWWADMIN} .
chmod -R 0664 .
chmod -R g+sw .
find . -type d -exec chmod a+x {} \;

# Executable Perl scripts..
find . -type f -name "*.pl" -exec chmod ug+x {} \;

# Php executables in the scripts directory..
if [ -d scripts ] ; then
  find scripts -type f -name "*.php" -exec chmod ug+x {} \;
fi

# Make all shell scripts executable..
scripts=`find . -type f -exec grep -l -E -e "(\#\![[:space:]]*\/bin\/bash)|(\#\![[:space:]]*\/bin\/sh)" {} \;`
if [ "$scripts" != "" ] ; then
  for script in $scripts ; do
    chmod ug+x $script
  done
fi

# Except var directory is writeable to webserver too..
chgrp -R ${WWWUSER} ./var

echo "permissions have been set."

exit 0

# END