/*
 * Bickley - a meta data management framework.
 * Copyright © 2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <string.h>
#include <stdlib.h>

#include <glib.h>
#include <gconf/gconf-client.h>

#include <bickley/bkl-db.h>
#include <bickley/bkl-entry.h>
#include <bickley/bkl-item-extended.h>

#include "bkl-finder.h"
#include "bkl-indexer.h"
#include "bkl-orbiter.h"
#include "bkl-path-finder.h"
#include "bkl-source.h"
#include "bkl-source-gconf.h"

typedef struct _BklSourceGConf {
    BklSource source;

    GConfClient *gconf;
    GHashTable *finders;

    GSList *watched_uris;

    int uri_count;

    /* List of the uris in the database and the
       time they were last indexed at */
    GHashTable *last_indexed;
    GHashTable *dead_uris;
} BklSourceGConf;

typedef struct _BklSourceGConfClass {
    BklSourceClass parent_class;
} BklSourceGConfClass;

G_DEFINE_TYPE (BklSourceGConf, bkl_source_gconf, BKL_TYPE_SOURCE);

#define BICKLEY_GCONF_DIR "/apps/bickley"
#define WATCH_URI_KEY "/apps/bickley/watch_uris"

static void
bkl_source_gconf_finalize (GObject *object)
{
    G_OBJECT_CLASS (bkl_source_gconf_parent_class)->finalize (object);
}

static void
bkl_source_gconf_dispose (GObject *object)
{
    G_OBJECT_CLASS (bkl_source_gconf_parent_class)->dispose (object);
}

static void
get_more_work (gpointer key,
               gpointer value,
               gpointer data)
{
    BklFinder *finder = (BklFinder *) value;
    gboolean *more_work = data;

    *more_work |= bkl_finder_lazy_dig (finder);
}

static void
delete_dead_keys (gpointer key,
                  gpointer value,
                  gpointer userdata)
{
    BklSource *source = userdata;
    BklItem *item = NULL;
    GError *error = NULL;
    char *uri = key;

    item = bkl_db_get_item (source->db, uri, &error);
    if (error != NULL) {
        if (error->code != KOZO_DB_ERROR_KEY_NOT_FOUND) {
            g_warning ("Error getting item for %s: %s", uri, error->message);
            return;
        }

        g_error_free (error);
        error = NULL;
    }

    if (item) {
        bkl_indexer_remove_item (source, item, source->db->db);
    }

    bkl_db_remove (source->db, uri, &error);
    if (error != NULL) {
        g_warning ("(%s): Error removing dead uri %s: %s",
                   source->name, uri, error->message);
        g_error_free (error);
    }
    bkl_source_uri_deleted (source, uri);
}

static gboolean
bkl_source_gconf_do_work (BklSource *source)
{
    BklSourceGConf *gconf = (BklSourceGConf *) source;
    gboolean more_work = FALSE;

    g_hash_table_foreach (gconf->finders, get_more_work, &more_work);
    source->more_work = more_work;

    if (more_work == FALSE) {
        if (gconf->dead_uris) {
            /* Now we've spidered everything we know what we can delete, yey! */
            g_print ("We have %d dead uris\n",
                     g_hash_table_size (gconf->dead_uris));
            g_hash_table_foreach (gconf->dead_uris, delete_dead_keys, gconf);

            g_hash_table_destroy (gconf->dead_uris);
            gconf->dead_uris = NULL;
        }
    }

    return more_work;
}

static void
bkl_source_gconf_investigate_uris (BklSource *source,
                                   GPtrArray *files)
{
    BklSourceGConf *gconf = (BklSourceGConf *) source;
    GPtrArray *fa;
    int i;

    fa = g_ptr_array_new ();

    for (i = 0; i < files->len; i++) {
        InvestigateUri *iu = files->pdata[i];
        char *uri = g_file_get_uri (iu->file);
        gpointer value;

        /* When we see a URI remove it from the dead_uris table */
        if (gconf->dead_uris) {
            g_hash_table_remove (gconf->dead_uris, uri);
        }

        value = g_hash_table_lookup (gconf->last_indexed, uri);
        if (value != NULL) {
            if (iu->mtime.tv_sec <= GPOINTER_TO_INT (value)) {
#if 0
                g_print ("(%s): Ignoring %s as new mtime %d is less than old %d\n",
                         source->name, uri, iu->mtime.tv_sec,
                         GPOINTER_TO_INT (value));
#endif
                g_free (uri);

                continue;
            }
        }

        g_ptr_array_add (fa, uri);
    }

    gconf->uri_count += bkl_orbiter_investigate_files (source, fa);
    /* investigate_files takes ownership of the uris */
    g_ptr_array_free (fa, TRUE);
}

static void
bkl_source_gconf_remove_uris (BklSource *source,
                              GPtrArray *files)
{
    BklSourceGConf *gconf = (BklSourceGConf *) source;
    int i;

    for (i = 0; i < files->len; i++) {
        char *uri;
        BklItem *item;
        GError *error = NULL;

        uri = g_file_get_uri (files->pdata[i]);

        item = bkl_db_get_item (source->db, uri, &error);
        if (error != NULL) {
            if (error->code != KOZO_DB_ERROR_KEY_NOT_FOUND) {
                return;
            }

            g_error_free (error);
            error = NULL;
        }

        if (item) {
            bkl_indexer_remove_item (source, item, source->db->db);

            g_object_unref (item);

            bkl_db_remove (source->db, uri, &error);
            if (error) {
                g_warning ("(%s): Error removing %s: %s", source->name, uri,
                           error->message);
                g_error_free (error);
                g_free (uri);
                return;
            }

            bkl_source_uri_deleted (source, uri);
        }

        if (gconf->dead_uris) {
            g_hash_table_remove (gconf->dead_uris, uri);
        }
        g_free (uri);
    }
}

static GPtrArray *
copy_string_array (GPtrArray *original)
{
    GPtrArray *clone;
    int i;

    if (original == NULL) {
        return NULL;
    }

    clone = g_ptr_array_sized_new (original->len);
    for (i = 0; i < original->len; i++) {
        g_ptr_array_add (clone, g_strdup (original->pdata[i]));
    }

    return clone;
}

static gboolean
bkl_source_gconf_add_item (BklSource  *source,
                           const char *uri,
                           BklItem    *item,
                           GError    **error)
{
    BklItem *old;
    BklSourceGConf *gconf = (BklSourceGConf *) source;

    old = bkl_db_get_item (source->db, uri, error);
    if (*error != NULL) {
        if ((*error)->code != KOZO_DB_ERROR_KEY_NOT_FOUND) {
            return FALSE;
        }

        g_error_free (*error);
        *error = NULL;
    }

    if (old == NULL) {
        bkl_db_add_item (source->db, uri, item, error);

        bkl_indexer_index_item (source, item, source->db->db, error);
        bkl_source_uri_added (source, uri);
    } else {
        BklItemExtended *ext = (BklItemExtended *) old;
        GPtrArray *tags;

        /* Copy the extended details from the existing item
           as these shouldn't change on update */
        bkl_item_extended_set_use_count
            ((BklItemExtended *) item, bkl_item_extended_get_use_count (ext));
        bkl_item_extended_set_last_used
            ((BklItemExtended *) item, bkl_item_extended_get_last_used (ext));
        bkl_item_extended_set_rating
            ((BklItemExtended *) item, bkl_item_extended_get_rating (ext));
        bkl_item_extended_set_pinned
            ((BklItemExtended *) item, bkl_item_extended_get_pinned (ext));

        tags = bkl_item_extended_get_tags (ext);
        bkl_item_extended_set_tags ((BklItemExtended *) item,
                                    copy_string_array (tags));

        bkl_db_replace_item (source->db, uri, item, error);
        bkl_indexer_update_item (source, item, old, source->db->db);
        g_object_unref (old);

        bkl_source_uri_changed (source, uri);
    }

    gconf->uri_count--;
    if (gconf->uri_count == 0) {
        bkl_source_completed (source);
    }

    return TRUE;
}

static void
add_uri_watch (BklSourceGConf *source,
               const char     *uri)
{
    BklFinder *finder;

    finder = g_hash_table_lookup (source->finders, uri);
    if (G_UNLIKELY (finder)) {
        return;
    }

    finder = bkl_path_finder_new ((BklSource *) source, uri);
    g_print ("(%s) Added finder for %s\n", ((BklSource *)source)->name, uri);
    g_hash_table_insert (source->finders, finder->base_uri, finder);

    bkl_source_in_progress ((BklSource *) source);
    bkl_orbiter_start_worker ();
}

static void
remove_uri_watch (BklSourceGConf *source,
                  const char     *uri)
{
    BklFinder *finder;

    finder = g_hash_table_lookup (source->finders, uri);
    if (G_UNLIKELY (finder == NULL)) {
        return;
    }

    g_hash_table_remove (source->finders, uri);
    bkl_finder_destroy (finder);
}

static void
calculate_watch_changes (BklSourceGConf *source,
                         GSList         *list)
{
    GSList *added = NULL, *removed = NULL, *l;
    gboolean free_added = FALSE, free_removed = FALSE;

    if (source->watched_uris == NULL) {
        added = list;
    } else {
        free_added = TRUE;
        for (l = list; l; l = l->next) {
            const char *uri = l->data;
            gboolean found = FALSE;
            GSList *k;

            for (k = source->watched_uris; k; k = k->next) {
                const char *uri2 = k->data;

                if (strcmp (uri, uri2) == 0) {
                    found = TRUE;
                    break;
                }
            }

            if (found == FALSE) {
                added = g_slist_append (added, (char *) uri);
            }
        }
    }

    if (list == NULL) {
        removed = source->watched_uris;
    } else {
        free_removed = TRUE;
        for (l = source->watched_uris; l; l = l->next) {
            const char *uri = l->data;
            gboolean found = FALSE;
            GSList *k;

            for (k = list; k; k = k->next) {
                const char *uri2 = k->data;

                if (strcmp (uri, uri2) == 0) {
                    found = TRUE;
                    break;
                }
            }

            if (found == FALSE) {
                removed = g_slist_append (removed, (char *) uri);
            }
        }
    }

    for (l = removed; l; l = l->next) {
        const char *uri = l->data;

        remove_uri_watch (source, uri);
    }

    if (free_removed) {
        g_slist_free (removed);
    }

    for (l = added; l; l = l->next) {
        const char *uri = l->data;

        add_uri_watch (source, uri);
    }

    if (free_added) {
        g_slist_free (added);
    }
}

static void
watched_uri_list_changed (GConfClient *client,
                          guint        cnxn_id,
                          GConfEntry  *entry,
                          gpointer     userdata)
{
    BklSourceGConf *source = (BklSourceGConf *) userdata;
    GConfValue *value;
    GSList *new_list, *uri_list = NULL;

    value = gconf_entry_get_value (entry);
    new_list = gconf_value_get_list (value);

    for (; new_list; new_list = new_list->next) {
        GConfValue *value = new_list->data;

        /* The string is destroyed when the value is destroyed */
        uri_list = g_slist_append (uri_list,
                                   (char *) gconf_value_get_string (value));
    }

    calculate_watch_changes (source, uri_list);
    g_slist_free (uri_list);
}

static gboolean
bkl_gconf_init (BklSourceGConf *source)
{
    GError *error = NULL;
    GSList *watched_uris, *w;

    source->gconf = gconf_client_get_default ();
    if (source->gconf == NULL) {
        g_warning ("Error getting gconf client");
        return FALSE;
    }

    gconf_client_add_dir (source->gconf, BICKLEY_GCONF_DIR,
                          GCONF_CLIENT_PRELOAD_ONELEVEL, &error);
    if (error != NULL) {
        g_warning ("Error adding %s: %s", BICKLEY_GCONF_DIR, error->message);
        g_error_free (error);
        return FALSE;
    }

    watched_uris = gconf_client_get_list (source->gconf,
                                          WATCH_URI_KEY,
                                          GCONF_VALUE_STRING, &error);
    if (error != NULL) {
        g_warning ("Error getting watched uris: %s", error->message);
        g_error_free (error);

        g_object_unref (source->gconf);
        source->gconf = NULL;
        return FALSE;
    }

    /* Listen for changes to the watch list */
    gconf_client_notify_add (source->gconf, WATCH_URI_KEY,
                             watched_uri_list_changed, NULL, NULL, NULL);

    calculate_watch_changes (source, watched_uris);
    for (w = watched_uris; w; w = w->next) {
        g_free (w->data);
    }
    g_slist_free (watched_uris);

    return TRUE;
}

static gboolean
get_last_indexed (KozoDB     *db,
                  const char *key,
                  KozoEntry  *entry,
                  gpointer    userdata)
{
    BklSourceGConf *source = (BklSourceGConf *) userdata;
    KozoField *field;
    const char *data;
    glong t;

    field = bkl_entry_get_field (entry, BKL_FIELD_FILE);
    if (field == NULL) {
        return TRUE;
    }

    data = bkl_file_field_get_string (field, BKL_FILE_FIELD_MODIFICATION_TIME);
    if (data == NULL || *data == '\0') {
        return TRUE;
    }

    t = strtol (data, NULL, 10);
    g_hash_table_insert (source->last_indexed, g_strdup (key),
                         GINT_TO_POINTER (t));
    g_hash_table_insert (source->dead_uris, g_strdup (key), NULL);

    kozo_field_free (field);

    return TRUE;
}


static gboolean
kozo_init (BklSourceGConf *source,
           const char     *name)
{
    BklSource *s = (BklSource *) source;
    GError *error = NULL;

    s->db = bkl_db_get (name, &error);
    if (s->db == NULL) {
        g_print ("fialed: %p\n", error);
        if (error->code == KOZO_DB_ERROR_VERSION_MISMATCH) {
            g_warning ("The database for %s is out of date.\nIt should be removed and regenerated.\n%s", name, error->message);
        } else {
            g_warning ("Error getting Bickley database %s: %s",
                       name, error->message);
        }
        g_error_free (error);

        return FALSE;
    }

    source->last_indexed = g_hash_table_new_full (g_str_hash, g_str_equal,
                                                  g_free, NULL);

    /* This could probably be done quicker with some sort of prefix tree */
    source->dead_uris = g_hash_table_new_full (g_str_hash, g_str_equal,
                                               g_free, NULL);
    kozo_db_foreach (s->db->db, get_last_indexed, source);
    return TRUE;
}

static void
add_special_dir_watches (BklSourceGConf *source)
{
    char *uri;
    GUserDirectory special_dirs[] = { G_USER_DIRECTORY_MUSIC,
                                      G_USER_DIRECTORY_PICTURES,
                                      G_USER_DIRECTORY_VIDEOS,
                                      G_USER_DIRECTORY_DOWNLOAD,
                                      G_USER_N_DIRECTORIES };
    int i;

    for (i = 0; special_dirs[i] != G_USER_N_DIRECTORIES; i++) {
        const char *filename;

        filename = g_get_user_special_dir (special_dirs[i]);
        if (filename == NULL) {
            continue;
        }

        uri = g_filename_to_uri (g_get_user_special_dir (special_dirs[i]),
                                 NULL, NULL);
        add_uri_watch (source, uri);
        g_free (uri);
    }
}

static void
bkl_source_gconf_class_init (BklSourceGConfClass *klass)
{
    GObjectClass *o_class = (GObjectClass *) klass;
    BklSourceClass *s_class = (BklSourceClass *) klass;

    o_class->finalize = bkl_source_gconf_finalize;
    o_class->dispose = bkl_source_gconf_dispose;

    s_class->do_work = bkl_source_gconf_do_work;
    s_class->investigate_files = bkl_source_gconf_investigate_uris;
    s_class->remove_files = bkl_source_gconf_remove_uris;
    s_class->add_item = bkl_source_gconf_add_item;
}

void
bkl_source_gconf_init (BklSourceGConf *gconf)
{
}

/* FIXME: This should all go into init */
BklSource *
bkl_source_gconf_new (const char *name)
{
    BklSourceGConf *gconf;
    BklSource *source;

    gconf = g_object_new (BKL_TYPE_SOURCE_GCONF, NULL);
    source = (BklSource *) gconf;

    source->name = g_strdup (name);

    if (kozo_init (gconf, name) == FALSE) {
        g_free (source->name);

        g_object_unref (gconf);
        return NULL;
    }

    gconf->finders = g_hash_table_new_full (g_str_hash, g_str_equal,
                                            g_free, NULL);
    gconf->uri_count = 0;

    if (g_getenv ("BKL_ORBITER_NO_SPECIAL") == NULL) {
        add_special_dir_watches (gconf);
    }

    bkl_gconf_init (gconf);

    source->more_work = TRUE;

    return source;
}
