#!/system/bin/sh

# (C) Copyright Canonical 2011,2012

# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.

# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.

# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

set -e

usage() {
    echo "usage: lxc-shutdown -n name [-w] [-r]"
    echo "  Cleanly shut down a container."
    echo "  -w: wait for shutdown to complete."
    echo "  -r: reboot (ignore -w)."
    echo "  -t timeout: wait at most timeout seconds (implies -w), then kill"
    echo "              the container."
}

alarm() {
    pid=$1
    timeout=$2
    sleep $timeout
    kill $pid
}

dolxcstop()
{
    echo "Calling lxc-stop on $lxc_name"
    lxc-stop -n $lxc_name
    exit 0
}

usage_err() {
    [ -n "$1" ] && echo "$1" >&2
    usage
    exit 1
}

optarg_check() {
    [ -n "$2" ] || usage_err "option '$1' requires an argument"
}

timeout="-1"

reboot=0
dowait=0

while [ $# -gt 0 ]; do
    opt="$1"
    shift
    case "$opt" in
    -h|--help)
        usage
        exit 0
        ;;
    -n|--name)
        optarg_check $opt "$1"
        lxc_name=$1
        shift
        ;;
    -w|--wait)
        dowait=1
        ;;
    -r|--reboot)
        reboot=1
        ;;
    -t|--timeout)
        optarg_check $opt "$1"
        timeout=$1
        dowait=1
        shift
        ;;
    --)
        break;;
    -?)
        usage_err "unknown option '$opt'"
        ;;
    -*)
        # split opts -abc into -a -b -c
        set -- $(echo "${opt#-}" | sed 's/\(.\)/ -\1/g') "$@"
        ;;
    *)
        usage_err "unknown option '$opt'"
        exit 1
        ;;
    esac
done

if [ -z "$lxc_name" ]; then
    echo "no container name specified"
    usage
    exit 1
fi

if ! id | grep -q "uid=0"; then
   echo "This command has to be run as root"
   exit 1
fi

pid=$(lxc-info -n ufa -p 2>/dev/null | while read field PID;  do echo $PID; done)
if [ "$pid" = "-1" ]; then
    echo "$lxc_name is not running"
    exit 1
fi

if [ $reboot -eq 1 ]; then
    kill -s INT $pid
    exit 0
else
    kill -s PWR $pid
fi

if [ $dowait -eq 0 ]; then
    exit 0
fi

if [ $timeout != "-1" ]; then
    trap dolxcstop EXIT
    alarm $$ $timeout &
    alarmpid=$!
fi

while ! lxc-info -n $lxc_name --state-is STOPPED; do
    sleep 1
done

if [ $timeout != "-1" ]; then
    kill $alarmpid
fi

echo "Container $lxc_name has shut down"

exit 0
