#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import os
import shutil
import logging

from ui import AlbumArtScreen
from album_view import AlbumView
from terra.core.manager import Manager
from terra.core.controller import Controller

__all__ = ("AlbumArtAudioController",)

log = logging.getLogger("canola.ui.album_art")

mger = Manager()
OptionsControllerMixin = mger.get_class("OptionsControllerMixin")
AlbumArtOptionsModelFolder = mger.get_class("Model/Options/Folder/Media/Audio/AlbumArt")


class AlbumArtAudioController(Controller, OptionsControllerMixin):
    terra_type = "Controller/Folder/Media/Audio/AlbumArt"

    def __init__(self, model, evas, parent):
        Controller.__init__(self, model, evas, parent)
        self.evas = evas
        self._load_model()
        self._setup_view()
        OptionsControllerMixin.__init__(self)

    def _load_model(self):
        self.model.load()
        len_albums = len(self.model.children)
        if len_albums == 0:
            self.current = None
        elif len_albums == 1:
            self.current = 0
        else:
            self.current = self.model.current or 0

    def calc_prev(self, value):
        return (value - 1) % len(self.model.children)

    def calc_next(self, value):
        return (value + 1) % len(self.model.children)

    def _cb_titles_clicked(self, view, index):
        # callback for AudioModel on_clicked
        # the first children is "albums"
        self.model.children[self.current].load()
        audio_model = self.model.children[self.current].children[index]
        self.model.children[self.current].current = index
        view.select_item(index)
        self.parent.use(audio_model)

    def _setup_labels(self, prev_album=None,
                      current_album=None, next_album=None):
        if not prev_album:
            prev = self.calc_prev(self.current)
            prev_album = self.model.children[prev]

        if not current_album:
            current_album = self.model.children[self.current]

        if not next_album:
            next = self.calc_next(self.current)
            next_album = self.model.children[next]

        self.view.setup_album_labels(prev_album.name, prev_album.artist,
                                     current_album.name,
                                     current_album.artist,
                                     next_album.name, next_album.artist)

    def _do_load_children(self, *albums):
        for album in albums:
            album.load()

    def _setup_view(self):
        self.view = AlbumArtScreen(self.evas, self.parent.view)
        if self.current is not None:
            self.view.started_next_move = self._started_next_move
            self.view.started_prev_move = self._started_prev_move
            self.view.finished_next_move = self._load_next_cover
            self.view.finished_prev_move = self._load_prev_cover
            self.view.setup_cover_widgets(self._cb_titles_clicked,
                                          self._cb_cover_clicked)

            prev = self.calc_prev(self.current)
            next = self.calc_next(self.current)
            prev_album = self.model.children[prev]
            current_album = self.model.children[self.current]
            next_album = self.model.children[next]

            self.view.prev_album.set_cover(prev_album.cover)
            self.view.current_album.set_cover(current_album.cover)
            self.view.next_album.set_cover(next_album.cover)

            # setup labels
            self._setup_labels(prev_album, current_album, next_album)

            # setup initial titles lists
            self._do_load_children(current_album)
            self.view.current_album.set_list(current_album.children)

            self._setup_aux_prev()
            self._setup_aux_next()

    def _started_next_move(self):
        next = self.calc_next(self.current)
        next_album = self.model.children[next]
        self._do_load_children(next_album)
        self.view.next_album.set_list(next_album.children)

    def _started_prev_move(self):
        prev = self.calc_prev(self.current)
        prev_album = self.model.children[prev]
        self._do_load_children(prev_album)
        self.view.prev_album.set_list(prev_album.children)

    def _clean_current_album(self, idx):
        self.model.children[idx].unload()
        self.view.current_album.titles.hide()
        self.view.current_album.titles.delete()

    def _load_next_cover(self):
        old = self.current
        self.current = self.calc_next(self.current)
        self._setup_labels()

        # clean old prev_aux_album
        if self.view.prev_aux_album.cover:
            self.view.prev_aux_album.cover.delete()

        # clean current_album's titles
        self._clean_current_album(old)

        self.view.prev_aux_album.set_cover_image(self.view.prev_album)
        self.view.prev_album.set_cover_image(self.view.current_album)
        self.view.prev_album.titles = None
        self.view.prev_album.w_cover_set = None
        self.view.current_album.set_new_album(self.view.next_album)
        self.view.next_album.set_cover_image(self.view.next_aux_album)
        self.view.next_album.titles = None
        self.view.next_album.w_cover_set = None

        self._setup_aux_next()

    def _load_prev_cover(self):
        old = self.current
        self.current = self.calc_prev(self.current)
        self._setup_labels()

        # clen old next_aux_album
        if self.view.next_aux_album.cover:
            self.view.next_aux_album.cover.delete()

        # clean current_album's titles
        self._clean_current_album(old)

        self.view.next_aux_album.set_cover_image(self.view.next_album)
        self.view.next_album.set_cover_image(self.view.current_album)
        self.view.next_album.titles = None
        self.view.next_album.w_cover_set = None
        self.view.current_album.set_new_album(self.view.prev_album)
        self.view.prev_album.set_cover_image(self.view.prev_aux_album)
        self.view.prev_album.titles = None
        self.view.prev_album.w_cover_set = None

        self._setup_aux_prev()

    def _setup_aux_prev(self):
        self.view.prev_aux_album = AlbumView(self.view.w_prev_aux, self.view,
                                             self._cb_titles_clicked,
                                             self._cb_cover_clicked)
        new_prev = self.calc_prev(self.current - 1)
        new_prev_album = self.model.children[new_prev]
        self.view.prev_aux_album.set_cover(new_prev_album.cover)

    def _setup_aux_next(self):
        self.view.next_aux_album = AlbumView(self.view.w_next_aux, self.view,
                                             self._cb_titles_clicked,
                                             self._cb_cover_clicked)
        new_next = self.calc_next(self.current + 1)
        new_next_album = self.model.children[new_next]
        self.view.next_aux_album.set_cover(new_next_album.cover)

    def _cb_cover_clicked(self, grid, index):
        cur_album = self.view.current_album
        cover_path = cur_album._cover_path
        old_cover = os.path.join(cover_path, "cover.jpg")
        old_small = os.path.join(cover_path, "cover-small.jpg")
        new_small = os.path.basename(cur_album.w_cover_set.elements[index])
        tmp = new_small.split("-small-")
        new_cover = "%s-%s" % (tmp[0], tmp[1])
        if not os.path.islink(old_cover):
            try:
                files = os.listdir(cover_path)
            except OSError:
                log.warning("The directory does not exists")
                return

            if os.path.islink(old_small):
                os.unlink(old_small)

            for cover_idx in range(99):
                cover = os.path.join(cover_path,
                                     "cover-%s.jpg" % cover_idx)
                cover_small = os.path.join(cover_path,
                                           "cover-small-%s.jpg" % cover_idx)

                if not cover in files and os.path.exists(old_cover):
                    os.rename(old_cover, cover)
                    if not cover_small in files and os.path.exists(old_small):
                        os.rename(old_small, cover_small)
                    break
        else:
            os.unlink(old_cover)
            os.unlink(old_small)

        try:
            if os.path.exists(os.path.join(cover_path, new_cover)):
                os.symlink(new_small, old_small)
                os.symlink(new_cover, old_cover)
        except Exception, e:
            log.warning("Could not create link to new cover: %s", e)
            try:
                shutil.copy(os.path.join(cover_path, new_small), old_small)
                shutil.copy(os.path.join(cover_path, new_cover), old_cover)
            except Exception, e:
                log.warning("Trying to setup non cover on "
                            "vfat file system: %s", e)

        try:
            cur_album.set_cover(os.path.join(cover_path, new_cover))
        except Exception, e:
            log.error("Could not setup cover: %s", e)

        self.model.updated = True

    def delete(self):
        self.view.delete()
        if self.model.children:
            self.model.children[self.current].unload()
        self.model.unload()

    def back(self):
        if not self.view.animating:
            self.parent.back()

    def go_home(self):
        if not self.view.animating:
            self.parent.go_home()

    def options_model_get(self):
        return AlbumArtOptionsModelFolder(None, self)
