# -*- coding: utf-8 -*-
#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import logging
import os
import re
import md5
from time import mktime

from ecore import timer_add

from terra.core.manager import Manager
from terra.core.plugin_prefs import PluginPrefs
from terra.core.terra_object import TerraObject
from terra.utils.encoding import to_utf8

from common import STATE_INITIAL, STATE_DOWNLOAD_DIALOG_OPENING, \
    STATE_DOWNLOADING, STATE_PAUSED, STATE_DOWNLOADED, STATE_QUEUED


log = logging.getLogger("plugins.canola-core.ondemand.model.mixin")
mger = Manager()
db = mger.canola_db

CanolaError = mger.get_class("Model/Notify/Error")
SystemProps = mger.get_class("SystemProperties")

DownloadManager = mger.get_class("DownloadManager")
download_mger = DownloadManager()
network = mger.get_status_notifier("Network")

SIZE_REQUIREMENT = 10 * (1024 ** 2) # 10 Mb


from htmllib import HTMLParser

def fix_html(s):
    # remove img tags
    s = re.sub("[ ]*<img.*?>[ ]*", " ", s)
    # remove problematic html entities
    s = re.sub("&nbsp;", "", s)
    # fix unescaped html embedded in xml
    try:
        p = HTMLParser(None)
        p.save_bgn()
        p.feed(s)
        return p.save_end()
    except:
        return s


class OnDemandMixin(TerraObject):
    terra_type = "Model/Media/Generic/OnDemand"
    feed_table = None
    item_table = None
    download_group = None
    refresh_inteval = 0.5

    layout = {
        1: ("(id INTEGER PRIMARY KEY, feed_id VARCHAR, "
            "remote_uri VARCHAR UNIQUE, local_path VARCHAR, "
            "title VARCHAR, artist VARCHAR, rating INTEGER, "
            "cover VARCHAR, desc VARCHAR, date INTEGER, pos INTEGER, "
            "has_file INTEGER)"),

        2: ("(id integer primary key, feed_id integer, "
            "remote_uri varchar unique, local_path varchar, "
            "title varchar, artist varchar, rating integer, "
            "desc varchar, date integer, pos integer, "
            "state integer)"),

        3: ("(id integer primary key, feed_id integer, "
            "remote_uri varchar unique, uri varchar, size integer, "
            "title varchar, artist varchar, rating integer, "
            "desc varchar, read integer, date integer, pos integer, "
            "state integer)")
    }

    statements = {
        'update-from-1':
            "INSERT INTO %(item-table)s (feed_id, remote_uri, uri, "
            "title, artist, rating, desc, date, pos, state) "
            "SELECT feed_id, remote_uri, local_path, title, "
            "artist, rating, desc, date, pos, has_file "
            "FROM %(item-table)s_backup",

        'update-from-2':
            "INSERT INTO %(item-table)s (feed_id, remote_uri, uri, "
            "title, artist, rating, desc, date, pos, state) "
            "SELECT feed_id, remote_uri, local_path, title, "
            "artist, rating, desc, date, pos, state "
            "FROM %(item)s_backup",

        'create':           "CREATE TABLE IF NOT EXISTS "
                            "%(item-table)s %(layout)s",
        'table-info':       "PRAGMA TABLE_INFO(%(item-table)s)",
        'insert':           "INSERT INTO %(item-table)s "
                            "(feed_id, remote_uri, uri, title, "
                            "artist, rating, desc, read, date, pos) "
                            "values (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)",
        'select-all':       "SELECT * FROM %(item-table)s ",
        'select':           "SELECT * FROM %(item-table)s "
                            "WHERE feed_id == ? "
                            "ORDER BY date DESC",
        'select-one':       "SELECT * FROM %(item-table)s "
                            "WHERE feed_id = ? AND id = ?",
        'select-parent-title': "SELECT title FROM %(feed-table)s "
                            "WHERE id == ?",
        'uri-path-by-feed': "SELECT remote_uri, uri FROM %(item-table)s "
                            "WHERE feed_id = ?",
        'delete-by-feed':   "DELETE FROM %(item-table)s "
                            "WHERE feed_id = ?",
        'exists':           "SELECT * FROM %(item-table)s "
                            "WHERE remote_uri == ?",
        'update-pos':       "UPDATE %(item-table)s SET pos = ? "
                            "WHERE id == ?",
        'update-rating':    "UPDATE %(item-table)s SET rating = ? "
                            "WHERE id == ?",
        'update-uri':       "UPDATE %(item-table)s SET uri = ? "
                            "WHERE id == ?",
        'update-read':      "UPDATE %(item-table)s SET read = ? "
                            "WHERE id == ?",
        'update-filesize':  "UPDATE %(item-table)s SET size = ? "
                            "WHERE id == ?",
        'entries':          "SELECT COUNT(*) FROM %(item-table)s",
    }

    @classmethod
    def preprocess_query(cls, query, **kargs):
        params = { 'feed-table': cls.feed_table,
                   'item-table': cls.item_table }
        params.update(kargs)
        real_query = query % params
        return real_query

    @classmethod
    def execute_stmt(cls, stmt, *args, **kargs):
        query = cls.preprocess_query(cls.statements[stmt], **kargs)
        return db.execute(query, args)

    @classmethod
    def execute_stmt_with_cursor(cls, stmt, cur, *args, **kargs):
        query = cls.preprocess_query(cls.statements[stmt], **kargs)
        return cur.execute(query, args).fetchall()

    def __init__(self, parent):
        TerraObject.__init__(self)

        self.parent = parent
        self._feed_id = None
        self.id = None

        self.desc = None
        self.date = None

        self._unheard = True

        self._rating = 0
        self._last_pos = None
        self._uri = None
        self._filesize = -1

        self._data_state = STATE_INITIAL
        self._progress = 0.0
        self.delete_shown = False

        self.downloader = None
        self.velocity = 0
        self.bytes_written = 0

        self.stored_bytes_written = 0
        self.times_synced = 0

        self.callback_progress = None
        self.callback_network = None
        self.callback_unheard = None
        self.callback_state = None

    def inform_property_changed(self):
        if self.parent is not None and self in self.parent.children:
            self.parent.child_property_changed(self)

    def feed_id_get(self):
        return self._feed_id

    def feed_id_set(self, value, cache=None):
        def get_cover():
            path = PluginPrefs("settings").get("cover_path")
            cover = os.path.join(path,
                "podcasts/%s/cover.jpg" % self.feed_id)
            return cover

        def get_album():
            rows = self.execute_stmt('select-parent-title', self.feed_id)
            if rows:
                album = to_utf8(rows[0][0] or "")
            else:
                album = ""
            return album

        self._feed_id = int(value)

        if cache is None:
            self.cover = get_cover()
            self.album = get_album()
        else:
            key = "album" + str(value)
            if key not in cache:
                cache[key] = get_album()
            self.album = cache[key]

            key = "cover" + str(value)
            if key not in cache:
                cache[key] = get_cover()
            self.cover = cache[key]

    feed_id = property(feed_id_get, feed_id_set)

    # accesses db
    def __set_filesize(self, value):
        if value is None:
            log.warning("Trying to set filesize of None")
            return

        if self.id is None or value == self._filesize:
            return

        self.execute_stmt('update-filesize', value, self.id)
        self._filesize = value

    def __get_filesize(self):
        return self._filesize

    filesize = property(__get_filesize, __set_filesize)

    # accesses db
    def __set_rating(self, value):
        # NOTE: we allow None

        if self.id is not None:
            self.execute_stmt('update-rating', value, self.id)
            self._rating = value

    def __get_rating(self):
        return self._rating

    rating = property(__get_rating, __set_rating)

    # accesses db
    def __set_last_pos(self, value):
        # NOTE: we allow None

        if self.id is not None:
            self.execute_stmt('update-pos', value, self.id)
            self._last_pos = value

    def __get_last_pos(self):
        return self._last_pos

    last_pos = property(__get_last_pos, __set_last_pos)

    # accesses db
    def __set_uri(self, value):
        if value is None:
            log.warning("Trying to set uri of None")
            return

        if self.id is not None:
            self.execute_stmt('update-uri', value, self.id)
            self._uri = value

    def __get_uri(self):
        return self._uri

    uri = property(__get_uri, __set_uri)

    # accesses db
    def __set_data_state(self, value):
        if value is None:
            log.warning("Trying to set data_state of None")
            return

        self._data_state = value

        self.inform_property_changed()

        if self.callback_state is not None:
            self.callback_state(value)

    def __get_data_state(self):
        return self._data_state

    data_state = property(__get_data_state, __set_data_state)

    def __set_progress(self, value):
        self._progress = value
        if self.callback_progress is not None:
            self.callback_progress(value)

    def __get_progress(self):
        return self._progress

    progress = property(__get_progress, __set_progress)

    def __get_activatable(self):
        return bool(self._progress != 0.0)

    activatable = property(__get_activatable)

    def __set_unheard(self, value):
        if self.id is None:
            return

        if value is None:
            log.warning("Trying to set unheard of None")
            return

        self.execute_stmt('update-read', int(not value), self.id)
        self._unheard = value

        self.inform_property_changed()

        if self.callback_unheard is not None:
            self.callback_unheard(value)

    def __get_unheard(self):
        return self._unheard

    unheard = property(__get_unheard, __set_unheard)

    def on_download_started_cb(self):
        self.data_state = STATE_DOWNLOADING
        self.add_progress_updater()

    def on_queued_cb(self):
        self.data_state = STATE_QUEUED
        self.remove_progress_updater()

    def on_paused_cb(self):
        self.data_state = STATE_PAUSED
        self.remove_progress_updater()

    def on_cancelled_cb(self):
        self.data_state = STATE_INITIAL
        self.remove_progress_updater()
        self.finish_cancelled_download()

    def on_download_ended_cb(self, exception, mimetype):
        self.remove_progress_updater()

        if exception is not None:
            msg = exception.message or exception.msg or str(exception)
            if self.progress > 0:
                self.show_error(msg, STATE_PAUSED)
            else:
                self.show_error(msg, STATE_INITIAL)
            return
        else:
            # Do not change state here, is done after the anim ends
            self.finish_successful_download()

    def add_progress_updater(self):
        if self.downloader is None:
            return

        if not hasattr(self, "timer") or not self.timer:
            self.bytes_written = -1
            self.velocity = 0
            self.timer = timer_add(self.refresh_inteval, self.sync_progress)
            log.debug("Added a progress timer for '%s'" % self.uri)

    def remove_progress_updater(self):
        # in case download was extremely quick we might
        # not have called the sync_progress, so do it here
        self.sync_progress()

        if not hasattr(self, "timer") or not self.timer:
            return

        if self.timer is not None:
            log.debug("Removed a progress timer for '%s'" % self.uri)
            self.timer.stop()
            self.timer.delete()
            self.timer = None

    def sync_progress(self, *ignore):
        if self.downloader is None:
            return False

        self.times_synced += 1

        bytes_written, self.filesize = self.downloader.get_progress()

        if self.times_synced % 4 == 0:
            if self.bytes_written != -1:
                velocity = (bytes_written - self.stored_bytes_written) \
                    / (1024 * self.refresh_inteval * 4)
                self.velocity = (self.velocity + velocity) / 2 # normalization
            self.stored_bytes_written = self.bytes_written

        if self.filesize != -1:
            if self.filesize > 0:
                self.progress = (bytes_written / float(self.filesize))
            else:
                self.progress = 0.0
        else:
            log.debug("Couldn't get size of '%s'" % self.remote_uri)

        self.bytes_written = bytes_written

        if self.progress >= 1.0:
            return False
        else:
            return True

    def unload(self):
        # unload only means that we exited/refreshed the screen, if
        # we /scheduled/ downloads we still want them to start.
        if self.data_state == STATE_DOWNLOAD_DIALOG_OPENING:
            self.fetch()

        self.remove_progress_updater()

        self._remove_download_process()
        self.callback_progress = None
        self.callback_network = None
        # do not dispose download item, keep downloading

    def show_error(self, msg, state):
        self.data_state = state

        if self.parent is None:
            return

        err = CanolaError(msg)
        if self.parent.callback_notify:
            self.parent.callback_notify(err)

    def best_save_location_get(self, filename):
        # Always get latest settings
        path = PluginPrefs("settings").get('download_path', None)

        if path is None:
            path = SystemProps.DEFAULT_DIR_AUDIO

        path = os.path.join(path, "OnDemand")

        try:
            SystemProps().prepare_write_path(path,
                "Download folder", SIZE_REQUIREMENT)
        except Exception, e:
            self.show_error(e.message, STATE_INITIAL)
            return False

        self.uri = os.path.join(path, str(self.id) + "_" + filename)
        return True

    def _sanitize_filename(self, filename):
        filename = filename.replace('%20', ' ')
        return filename

    def _add_download_process(self):
        def setup_process():
            self.downloader = download_mger.add(self.remote_uri,
                self.uri, data=(self.feed_id, self.id, self.title),
                group=self.download_group)
            self.connect_callbacks(self.downloader)

        if self.uri is not None and os.path.exists(self.uri + ".info"):
            # if we have an info file corresponding to the uri we will
            # download to the same file (resume if exists)
            setup_process()
        else:
            basename = os.path.basename(self.remote_uri)
            uri_list = basename.split("?")
            filename = self._sanitize_filename(uri_list[0])
            log.debug("Name for file being downloaded : %s", filename)

            if len(uri_list) > 1:
                suffix = md5.md5(uri_list[1]).hexdigest()
                try:
                    filename, ext = filename.split(".")
                    filename += "-%s.%s" % (suffix, ext)
                except ValueError, e:
                    filename += "-%s" % suffix

            if self.best_save_location_get(filename):
                setup_process()
            # else: didn't succeed and an error was shown, so we do not
            # set up a download process.

    def _remove_download_process(self):
        if self.downloader:
            self.disconnect_callbacks(self.downloader)
            self.downloader = None

    def connect_callbacks(self, downloader):
        self.downloader.on_download_started_add(self.on_download_started_cb)
        self.downloader.on_cancelled_add(self.on_cancelled_cb)
        self.downloader.on_paused_add(self.on_paused_cb)
        self.downloader.on_queued_add(self.on_queued_cb)
        self.downloader.on_finished_add(self.on_download_ended_cb)

    def disconnect_callbacks(self, downloader):
        self.downloader.on_download_started_remove(self.on_download_started_cb)
        self.downloader.on_cancelled_remove(self.on_cancelled_cb)
        self.downloader.on_paused_remove(self.on_paused_cb)
        self.downloader.on_queued_remove(self.on_queued_cb)
        self.downloader.on_finished_remove(self.on_download_ended_cb)

    def fetch(self):
        if self.downloader is None:
            return

        self.data_state = STATE_DOWNLOADING
        self.downloader.start(True)
        self.progress = 0.0

    def pause(self):
        if self.downloader is not None:
            self.downloader.pause()
            self.data_state = STATE_PAUSED

    def resume(self):
        self.sync_progress()
        self.data_state = STATE_DOWNLOADING
        if self.downloader is not None:
            self.downloader.start(True)
        else:
            self._add_download_process()
            self.fetch()

    def finish_successful_download(self):
        self.progress = 1.0
        self._remove_download_process()
        if os.path.exists(self.uri):
            self.sync_progress()
            self.data_state = STATE_DOWNLOADED

    def finish_cancelled_download(self):
        self.progress = 0.0
        self._remove_download_process()
        self.data_state = STATE_INITIAL

    def purge(self):
        self.progress = 0.0
        if os.path.exists(self.uri or ""):
            try:
                os.unlink(self.uri)
            except OSError, e:
                msg = e.strerror + ": '" +  e.filename + "'."
                self.show_error(msg, STATE_INITIAL)
        if self.downloader is not None:
            self.downloader.cancel()
        else:
            self.data_state = STATE_INITIAL

    def _insert(self, cur=None):
        values = (self.feed_id, self.remote_uri, self.uri,
                  self.title, self.artist, self.rating, self.desc,
                  int(not self.unheard), self.date, self.last_pos)
        cur = cur or db.get_cursor()
        self.execute_stmt_with_cursor('insert', cur, *values)
        self.id = cur.lastrowid

    def commit(self):
        already_exists = self.execute_stmt('exists', self.remote_uri)
        if already_exists:
            db.commit()
        else:
            self._insert()

    def get_uri_from_feed_entry(self, entry):
        try:
            return to_utf8(entry.enclosures[0].href)
        except:
            if entry.has_key("link"):
                uri = to_utf8(entry.link)
                return uri
        return None

    def use_data_from_db_row(self, row, cache):
        self.id = row[0]
        self.feed_id_set(row[1], cache)

        self.remote_uri = to_utf8(row[2])
        self.name = self.title = to_utf8(row[5] or "")
        self.artist = to_utf8(row[6] or "")
        self.desc = to_utf8(row[8] or "").replace("<li>", "<li>• ")
        self.date = row[10]

        # do not update db (we read from db) and use private properties
        self._unheard = bool(not row[9])
        self._uri = (row[3]) and to_utf8(row[3]) or None
        self._filesize = row[4] or -1
        self._rating = row[7] or 0
        self._last_pos = row[11]

        self._progress = 0

    def use_data_from_feed_entry(self, entry):
        self.remote_uri = self.get_uri_from_feed_entry(entry)
        self.name = self.title = to_utf8(entry.title)
        self.artist = entry.has_key("author") and to_utf8(entry.author) or ""
        self.genre = "Podcast"
        self.desc = entry.has_key("description") and \
            fix_html(to_utf8(entry.description)) or ""

        if entry.has_key("updated_parsed") and entry.updated_parsed is not None:
            self.date = int(mktime(entry.updated_parsed))

        # Do not update db, so use private fields:
        self._rating = 0
        self._data_state = STATE_INITIAL

        self._progress = 0
