#!/bin/bash

CWD="$PWD"
PREFIX=/usr/local
BINDIR=
SYSCONFDIR=
DATAROOTDIR=
DBUSDIR=`pkg-config --variable=session_bus_services_dir dbus-1`
DESKTOPDIR=
ICONDIR=
EDJE_CC=${EDJE_CC:-edje_cc}
PYTHON=${PYTHON:-python}
TMPDIR=${TMPDIR:-/tmp}
PLUGINS="canola-core"

die() {
    echo "ERROR: $*" >&2
    exit 1
}

log() {
    echo "LOG: $*" >&2
}

help() {
    cat <<EOF
$0 [options] <command1> [.. <commandN>]

where command:

 compile                      compile data into binaries, themes, modules.
 clean                        remove files generated at compile time.
 install                      install files on system.

where options:

Paths:
 -p, --prefix=PREFIX          base where to install files
                              (default: $PREFIX)
 -b, --bindir=BINDIR          where to install binaries
                              (default: PREFIX/bin)
 -d, --datarootdir=DIR        base where to install read-only and architecture
                              independent data
                              (default: PREFIX/share)
 -s, --sysconfdir=DIR         read-only single machine configuration
                              (default: PREFIX/etc)
     --dbus-services-dir=DIR  where dbus loads its services files.
                              (default: $DBUSDIR)
     --desktopdir=DIR         where to install desktop files.
                              (default: $DATAROOTDIR/applications)
     --icondir=DIR            where to install icon files.
                              (default: $DATAROOTDIR/icons)

Tools Paths:
     --with-edje-cc=PATH      path to edje_cc binary.
     --with-python=PATH       path to python binary.

Misc Options:
 -h, --help                   this message.


environment variables:

 EDJE_CC                      path to edje_cc binary, same as --with-edje-cc.
 PYTHON                       path to python binary, same as --with-python.
 DESTDIR                      when installing, use this as prefix for all files.

EOF
}


TEMP=`getopt -o p:b:s:d:h \
    --long "prefix:,bindir:,sysconfdir:,datarootdir:,dbus-services-dir:,desktopdir:,icondir:,with-edje-cc:,with-python:,help" \
    -- "$@"`
if [ $? != 0 ]; then
    die "Terminating..."
fi

eval set -- "$TEMP"
while true; do
    case "$1" in
        -p|--prefix)
            PREFIX="$2"
            shift 2
            ;;
        -b|--bindir)
            BINDIR="$2"
            shift 2
            ;;
        -d|--datarootdir)
            DATAROOTDIR="$2"
            shift 2
            ;;
        -s|--sysconfdir)
            SYSCONFDIR="$2"
            shift 2
            ;;
        --dbus-services-dir)
            DBUSDIR="$2"
            shift 2
            ;;
        --desktopdir)
            DESKTOPDIR="$2"
            shift 2
            ;;
        --icondir)
            ICONDIR="$2"
            shift 2
            ;;
        --with-edje-cc)
            EDJE_CC="$2"
            shift 2
            ;;
        --with-python)
            PYTHON="$2"
            shift 2
            ;;
        -h|--help)
            help
            exit 0
            ;;
        --)
            shift
            break
            ;;
        *)
            die "Internal error parsing command line arguments: $1"
            ;;
    esac
done

[ -z "$BINDIR" ] && BINDIR="$PREFIX/bin"
[ -z "$DATAROOTDIR" ] && DATAROOTDIR="$PREFIX/share"
[ -z "$SYSCONFDIR" ] && SYSCONFDIR="$PREFIX/etc"
[ -z "$DBUSDIR" ] && DBUSDIR="$DATAROOTDIR/dbus-1/services"
[ -z "$DESKTOPDIR" ] && DESKTOPDIR="$DATAROOTDIR/applications"
[ -z "$ICONDIR" ] && ICONDIR="$DATAROOTDIR/icons"

DATADIR="$DATAROOTDIR/canola"
THEMESDIR="$DATADIR/themes"
PLUGINSDIR="$DATADIR/plugins"

EDJE_CC=`which $EDJE_CC`
[ -z "$EDJE_CC" -o ! -x "$EDJE_CC" ] && die "no edje_cc binary found."

PYTHON=`which $PYTHON`
[ -z "$PYTHON" -o ! -x "$PYTHON" ] && die "no python binary found."

cat <<EOF
Summary:

   Base paths:
      PREFIX..........: $PREFIX
      BINDIR..........: $BINDIR
      DATAROOTDIR.....: $DATAROOTDIR
      SYSCONFDIR......: $SYSCONFDIR
      DBUSDIR.........: $DBUSDIR
      DESKTOPDIR......: $DESKTOPDIR
      ICONDIR.........: $ICONDIR

   Project paths:
      DATADIR.........: $DATADIR
      THEMESDIR.......: $THEMESDIR
      PLUGINSDIR......: $PLUGINSDIR

   Tools paths:
      EDJE_CC.........: $EDJE_CC
      PYTHON..........: $PYTHON
EOF


install_data() {
    SRC=${1:?missing source.}
    DST=${2:?missing destination.}

    /usr/bin/install -D -m 644 "$SRC" "$DESTDIR$DST" || \
        die "could not install data file '$SRC' to '$DESTDIR$DST'"
}

install_pyscript() {
    SRC=${1:?missing source.}
    DST=${2:?missing destination.}

    if ! head -n1 "$SRC" | grep '^#!/.*python' >/dev/null 2>/dev/null; then
        die "not a python script: $SRC"
    fi

    T=`tempfile`
    cat "$SRC" | sed -e "s:#!/.*$:#!$PYTHON:" > "$T"

    /usr/bin/install -D -m 755 "$T" "$DESTDIR$BINDIR/$DST" || \
        die "could not install python script file '$SRC' to '$DESTDIR$BINDIR/$DST'"

    rm -- "$T"
}

install_script() {
    SRC=${1:?missing source.}
    DST=${2:?missing destination.}

    if head -n1 "$SRC" | grep '^#!/.*python' >/dev/null 2>/dev/null; then
        install_pyscript "$SRC" "$DST"
    else
        /usr/bin/install -D -m 755 "$SRC" "$DESTDIR$BINDIR/$DST" || \
            die "could not install script file '$SRC' to '$DESTDIR$BINDIR/$DST'"
    fi
}

themes_compile() {
    log "compiling themes..."
    cd $CWD || die "could not cd $CWD"
    for d in $CWD/data/themes/*; do
        THEME=`basename $d`
        [ x"$THEME" = "xchecker" ] && continue
        log "   $THEME"
        (cd "$d" && $EDJE_CC $THEME.edc) || die "could not build theme: $THEME"
    done
    log "compiling themes: done."
}

themes_clean() {
    log "cleaning themes..."
    for d in $CWD/data/themes/*; do
        THEME=`basename $d`
        [ x"$THEME" = "xchecker" ] && continue
        [ ! -f "$d/$THEME.edj" ] && continue
        log "   $THEME"
        rm -- "$d/$THEME.edj" || die "could not rm $d/$THEME.edj"
    done
    log "cleaning themes: done."
}

themes_install() {
    log "installing themes..."
    cd $CWD || die "could not cd $CWD"
    for d in $CWD/data/themes/*; do
        THEME=`basename $d`
        [ x"$THEME" = "xchecker" ] && continue
        log "   $THEME"
        install_data "$d/$THEME.edj" "$THEMESDIR/$THEME.edj"
    done
    log "installing themes: done."
}

etk_themes_compile() {
    log "compiling etk (conf) themes..."
    cd $CWD || die "could not cd $CWD"

    log "   default"
    (cd "$CWD/data/etk-themes/default" &&
        $EDJE_CC -id images default.edc conf.edj) || \
            die "could not build theme: default"

    log "compiling etk (conf) themes: done."
}

etk_themes_install() {
    log "installing etk (conf) themes..."

    log "   default"
    install_data "$CWD/data/etk-themes/default/conf.edj" "$THEMESDIR/conf.edj"

    log "installing etk (conf) themes: done."
}

etk_themes_clean() {
    log "cleaning etk (conf) themes..."

    if [ -f "$CWD/data/etk-themes/default/conf.edj" ]; then
        log "   default"
        rm -- "$CWD/data/etk-themes/default/conf.edj"
    fi

    log "cleaning etk (conf) themes: done."
}


plugin_compile() {
    COLLECTION=${1:?missing collection name.}
    log "   compiling python plugin $COLLECTION..."
    cd $CWD || die "could not cd $CWD"
    $PYTHON -OO -m compileall -d $COLLECTION $CWD/plugins/$COLLECTION || \
        die "could not compile python plugin $COLLECTION"
    $PYTHON -m compileall -d $COLLECTION $CWD/plugins/$COLLECTION || \
        die "could not compile python plugin $COLLECTION"
    log "   compiling python plugin $COLLECTION: done."
}

plugin_zip() {
    COLLECTION=${1:?missing collection name.}
    log "   zip python plugin $COLLECTION..."
    cd $CWD/plugins || die "could not cd $CWD/plugins"
    zip -q -9 $COLLECTION.zip \
        $COLLECTION/plugins.info \
        `find -type f -name '*.pyo' -o -name '*.pyc' -o -name '*.py'` || \
        die "could not zip plugin $COLLECTION"
    log "   zip python plugin $COLLECTION: done."
}

plugins_compile() {
    log "compile python plugins..."
    for p in $PLUGINS; do
        plugin_compile $p
    done
    log "compile python plugins: done."
}

plugins_zip() {
    log "zip python plugins..."
    for p in $PLUGINS; do
        plugin_zip $p
    done
    log "zip python plugins: done."
}

plugins_install() {
    log "installing python plugins..."
    for p in $PLUGINS; do
        log "   plugin: $p as $PLUGINSDIR/$p.zip"
        install_data "$CWD/plugins/$p.zip" "$PLUGINSDIR/$p.zip"
    done
    log "installing python plugins: done."
}

plugins_clean() {
    log "cleaning python plugins..."
    for p in $PLUGINS; do
        find "$CWD/plugins/$p" -type f \
            -name '*.pyo' -o -name '*.pyc' -print0 | xargs -0 rm -f
        if [ -f "$CWD/plugins/$p.zip" ]; then
            rm "$CWD/plugins/$p.zip" || die "could not rm $CWD/plugins/$p.zip"
        fi
    done
    log "cleaning python plugins: done."
}

icon_install() {
    SRC=${1:?missing source.}
    DST=${2:?missing destination.}

    log "   icon: $SRC as $ICONDIR/$DST"
    install_data "$CWD/data/icons/$SRC" "$ICONDIR/$DST"
}

icons_install() {
    log "installing icons..."
    icon_install "scalable.png" "canola.png"
    log "installing icons: done."
}

install_data_fix_canola_path() {
    SRC=${1:?missing source.}
    DST=${2:?missing destination.}

    T=`tempfile`
    cat "$SRC" | sed -e "s:/usr/bin/canola:$BINDIR/canola:g" > "$T"
    install_data "$T" "$DST"
    rm -- "$T"
}

desktop_install() {
    log "install canola.desktop in $DESKTOPDIR/"
    install_data_fix_canola_path "$CWD/data/canola.desktop" "$DESKTOPDIR/canola.desktop"
}

dbus_install() {
    log "install canola.service in $DBUSDIR/"
    install_data_fix_canola_path "$CWD/data/canola.service" "$DBUSDIR/canola.service"
}

presets_install() {
    log "install canola.presets in $DATADIR/"
    install_data "$CWD/data/canola.presets" "$DATADIR/canola.presets"
}

bins_install() {
    log "installing binaries..."
    mkdir -p "$DESTDIR/$BINDIR"
    for b in $CWD/bin/*; do
        [ ! -x "$b" ] && continue
        bn=`basename "$b"`
        log "   "`basename "$bn"`
        install_script "$b" "$bn"
    done
    log "installing binaries: done."
}

conf_install() {
    log "generating $SYSCONFDIR/canola.conf..."
    T=`tempfile`
    cat > "$T" <<EOF
[terra]
app_user_dir = .canola
plugins_dir = $PLUGINSDIR
themes_dir = $THEMESDIR
plugins_cfg = $DATADIR/plugins.pickle
presets_cfg = $DATADIR/canola.presets
EOF

    install_data "$T" "$SYSCONFDIR/canola.conf"
    rm -- "$T"
    log "generating $SYSCONFDIR/canola.conf: done."
}

for arg in $@; do
    case "$arg" in
        compile)
            themes_compile
            etk_themes_compile
            plugins_compile
            plugins_zip
            ;;
        clean)
            themes_clean
            etk_themes_clean
            plugins_clean
            ;;
        install)
            themes_install
            etk_themes_install
            plugins_install
            icons_install
            desktop_install
            dbus_install
            presets_install
            bins_install
            conf_install
            ;;
        *)
            die "unknown command: $1"
            ;;
    esac
done

