#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import os
import array
import logging
import ecore
import edje

try:
    from pysqlite2 import dbapi2 as sqlite
except ImportError:
    from sqlite3 import dbapi2 as sqlite

import dbus
try:
    from e_dbus import DBusEcoreMainLoop
    DBusEcoreMainLoop(set_as_default=True)
except Exception:
    import dbus.ecore

from terra.core.terra_object import TerraObject
from terra.core.manager import Manager
from terra.core.controller import Controller, get_controller_for_model, \
    get_controller_class_for_model
from terra.core.task import Task, get_task_controller_for_task
from terra.core.model import Model
from terra.core.plugin_prefs import PluginPrefs
from terra.ui.window import MainWindow
from terra.ui.screen import Screen


log = logging.getLogger("canola.main_controller")

mger = Manager()
Notify = mger.get_class("Model/Notify")
WaitNotify = mger.get_class("Model/WaitNotify")
PluginNotFoundError = mger.get_class("Model/Notify/Error/PluginNotFound")
PlayerHook = mger.get_class("Hook/Player")
RemotePlayer = mger.get_class("Controller/RemotePlayer")
RemotePlayerModel = mger.get_class("Model/Notify/RemotePlayer")


class CanolaDB(object):
    def __init__(self):
        self.filename = os.path.join(os.path.expanduser("~"),
                                     ".canola", "canola.db")
        self.connection = sqlite.connect(self.filename)
        self.connection.text_factory = str

    def get_cursor(self):
        """Return a Canola DB cursor.
        Classes must avoid at all costs getting their own cursors. For all
        sane uses 'execute' must be used instead.

        For those cases where a cursor is really necessary for optimization
        one should take extra care to close it properly after use.
        """
        log.debug_warning("Returning db's cursor")
        return self.connection.cursor()

    def execute(self, stmt, *args):
        """Returns the result of 'query'

        This function is safe and handles the DB cursor automatically, in
        opposition to 'get_cursor' that leaves the cursor handling to
        the caller thus being unsafe.
        """
        rows = self.connection.execute(stmt, *args).fetchall()
        # Commit if doing anything but a 'select' statement
        if not stmt[0] in ('S', 's'):
            log.debug("Executing SQL: %s", stmt)
            self.connection.commit()
        return rows

    def commit(self):
        self.connection.commit()
        log.debug("Commited info to db: %s", self.filename)

    def reconnect(self):
        self.connection.close()
        self.connection = sqlite.connect(self.filename)
        self.connection.text_factory = str
        log.debug("Reconnected db: %s", self.filename)


def dbus_activate(bus, service):
    try:
        bo = bus.get_object("org.freedesktop.DBus",
                            "/org/freedesktop/DBus",
                            introspect=False)
        bi = dbus.Interface(bo, "org.freedesktop.DBus")
        bi.StartServiceByName(service, dbus.UInt32(0))
    except Exception, e:
        log.error("Could not activate %r: %s", service, e, exc_info=True)


class CanolaDaemon(object):
    def __init__(self):
        self.bus = dbus.SessionBus()
        self.db_locked = False
        self.callbacks_locked = []
        self.callbacks_unlocked = []
        self.notify = None      # Canola puts here wait notify window
        self._connect_to_canola_daemon()

    def _connect_to_canola_daemon(self):
        dbus_activate(self.bus, "br.org.indt.canola.Daemon")
        self.obj = self.bus.get_object("br.org.indt.canola.Daemon",
                                       "/br/org/indt/canola/Daemon",
                                       introspect=False)
        self.iface = dbus.Interface(self.obj, "br.org.indt.canola.Daemon")
        self.iface.connect_to_signal("db_locked", self.lock_db)
        self.iface.connect_to_signal("db_unlocked", self.unlock_db)
        self.iface.connect_to_signal("dev_added", self.dev_added)
        self.iface.connect_to_signal("dev_removed", self.dev_removed)

    def lock_db(self):
        log.debug("Received database lock signal")
        self.db_locked = True
        if self.callbacks_locked:
            if not self.callbacks_locked[0]():
                self.callbacks_locked.pop(0)

    def unlock_db(self):
        log.debug("Received database unlock signal")
        self.db_locked = False
        if self.callbacks_unlocked:
            if not self.callbacks_unlocked[0]():
                self.callbacks_unlocked.pop(0)

    def dev_added(self, mount_point):
        mount_point = self.fix_path(mount_point)
        prefs = PluginPrefs("settings")
        for media in ("audio", "video", "photo"):
            if not mount_point in prefs[media]:
                prefs[media].append(mount_point)
        prefs.save()

    def dev_removed(self, mount_point):
        mount_point = self.fix_path(mount_point)
        prefs = PluginPrefs("settings")
        for media in ("audio", "video", "photo"):
            if mount_point in prefs[media]:
                prefs[media].remove(mount_point)
        prefs.save()

    def db_is_locked(self):
        return self.iface.db_is_locked()

    def _reply_handler(self, *args):
        pass

    def _error_handler(self, error):
        log.error("CanolaDaemon error: %s", error)

    def fix_path(self, arg):
        path_array = array.array("B")
        path_array.fromlist(arg)
        ret = path_array.tostring()
        return ret

    def fix_list(self, scanlist):
        if not scanlist or len(scanlist) == 0 or \
           not isinstance(scanlist, list):
            scanlist = [""]
            return scanlist
        new_scanlist = self.create_dbus_array(scanlist)
        return new_scanlist

    def create_dbus_array(self, scanlist):
        new_scanlist = []
        for path in scanlist:
            path = dbus.Array(path, signature=dbus.Signature("y"))
            new_scanlist.append(path)
        return new_scanlist

    def start_monitoring(self):
        self.iface.start_monitoring(reply_handler=self._reply_handler,
                                    error_handler=self._error_handler)

    def scan_all_paths(self, clean_database=False):
        self.iface.scan_all_paths(clean_database,
                                  reply_handler=self._reply_handler,
                                  error_handler=self._error_handler)

    def check_all_paths(self):
        self.iface.check_all_paths(reply_handler=self._reply_handler,
                                  error_handler=self._error_handler)

    def rescan_by_type(self, scanlist=None, type="audio"):
        scanlist = self.fix_list(scanlist)
        self.iface.scan_paths_by_type(scanlist, type,
                                      reply_handler=self._reply_handler,
                                      error_handler=self._error_handler)

    def rescan(self):
        clean_database = True
        self.iface.scan_all_paths(clean_database,
                                  reply_handler=self._reply_handler,
                                  error_handler=self._error_handler)

    def scan_status(self, callback):
        def reply(*args):
            if not args:
                return
            callback(args[0])
        return self.iface.scan_status(reply_handler=reply,
                                      error_handler=self._error_handler)

    def shutdown(self):
        # shutdown canola-d
        self.iface.stop()


class NowPlayingHook(PlayerHook):
    def __init__(self, window):
        PlayerHook.__init__(self)
        self._window = window

    def setup(self, player):
        if player.player_state.is_playing:
            self.playing()
        else:
            self.paused()

    def delete(self):
        self._window.multitask_state_set(self._window.MULTITASK_NONE)

    def playing(self):
        self._window.multitask_state_set(self._window.MULTITASK_PLAYING)

    def paused(self):
        self._window.multitask_state_set(self._window.MULTITASK_PAUSED)


class MainController(TerraObject):
    terra_type = "MainController"

    def __init__(self, ecore_evas, theme):
        TerraObject.__init__(self)
        self.ecore_evas = ecore_evas
        self._theme = theme
        self.animating = False
        self._setup_view()
        self._setup_remote_player()
        self.startup_notify = WaitNotify("Loading Canola...", 120.0,
                                         self._startup_notify_done)
        self.show_notify(self.startup_notify)
        self.view.controls_hide(transition=False)
        ecore.timer_add(0.5, self._lazy_init)

    def _startup_notify_done(self, *ignored):
        del self.startup_notify

    def _lazy_init(self):
        verbose = logging.getLogger().getEffectiveLevel() < logging.WARNING
        def progress(msg):
            if not verbose:
                return
            self.startup_notify.message_set(msg)
            for i in xrange(20):
                edje.message_signal_process()
                ecore.main_loop_iterate()
                self.view.evas.render()

        self._mger = Manager()

        progress("Setting up media library...")
        self._mger.canola_db = CanolaDB()

        progress("Starting components...")
        self._mger.canola_daemon = CanolaDaemon()

        progress("Loading notifying modules...")
        self._setup_status()

        progress("Loading data...")
        self._load_model()

        progress("Loading applications...")
        self._setup_sub_controllers()

        progress("Loading power save module...")
        self._setup_screen_powersave()

        progress("Setting up environment...")
        self._setup_multitask()
        self._load_autostart_plugins()

        progress("Checking media library...")

        self._setup_db_listener()
        self.hide_notify(self._notify_controller, None)
        self.view.controls_show()

        return False

    def _load_model(self):
        cls = self._mger.get_class("Model/Folder/MainMenu")
        self.menu_model = cls()
        self.menu_model.load()

    def _setup_view(self):
        self.view = MainWindow(self.ecore_evas.evas, self._theme)
        self.view.callback_back = self.cb_on_back
        self.view.callback_toggle_fullscreen = self.cb_on_toggle_fullscreen
        self.view.callback_multitask_clicked = self.cb_on_multitask_clicked
        self.view.callback_go_home = self.cb_on_go_home
        self.view.callback_panel_back = self.cb_on_panel_back
        self._notify_controller = None
        self._notify_models = []

        self.view.title_set("Canola")

        kdb = self.view.key_down_bindings
        kdb['Escape'] = lambda: self.cb_on_back(self.view)
        kdb['F4'] = lambda: self.cb_on_go_home(self.view)
        # XXX: debug functions that should just dump data in stdin/err
        kdb['d'] = lambda: dump_object_summary(True)
        kdb['c'] = lambda: dump_object_summary(False)

    def _setup_remote_player(self):
        self._remote_player = RemotePlayer()
        kdb = self.view.key_down_bindings

        def raise_volume():
            if self._player_hook:
                self._remote_player.raise_volume()
        kdb['F7'] = raise_volume

        def lower_volume():
            if self._player_hook:
                self._remote_player.lower_volume()
        kdb['F8'] = lower_volume

        #def show_remote():
        #    if self._player_hook and not self.menu_controller.animating:
        #        self.remote_player = RemotePlayerModel("remote_player")
        #        self.show_notify(self.remote_player)
        #kdb['Return'] = show_remote

    def _setup_status(self):
        self.status_controllers = []
        for type_name in ("Model/Status/Network", "Model/Status/Battery"):
            try:
                model_cls = self._mger.get_class(type_name)
            except ValueError:
                continue

            try:
                model = model_cls()
            except:
                log.debug_warning("couldn't create status model %r",
                                  type_name, exc_info=True)
                continue

            cont = get_controller_for_model(model, self.ecore_evas.evas, self)
            if cont is not None:
                self._mger.add_status_notifier(model)
                self.status_controllers.append(cont)

    def _adjust_screen_powersave(self, batt):
        if batt.status[0] == batt.BATTERY_CHARGING_OFF:
            self._mger.screen_powersave.enabled = True
        else:
            settings = PluginPrefs("settings")
            value = settings.get("display_lit_when_charging")
            if value is None:
                value = True
            self._mger.screen_powersave.enabled = not value

    def _setup_screen_powersave(self):
        try:
            scrps_cls = self._mger.get_class("ScreenPowerSave")
            self._mger.screen_powersave = scrps_cls()
        except ValueError:
            scrps_cls = None
            self._mger.screen_powersave = None

        batt = self._mger.get_status_notifier("Battery")
        if scrps_cls is None or batt is None:
            return

        batt.add_listener(self._adjust_screen_powersave)

    def _setup_sub_controllers(self):
        self.menu_controller = get_controller_for_model(self.menu_model,
                                                        self.ecore_evas.evas,
                                                        self)
        self.menu_controller.ref()

        self._sub_controllers = [self.menu_controller]
        self._sub_controller = self.menu_controller
        self._options_controller = None
        self.view.use(self._sub_controller.view)

    def _reset_multitask(self):
        self._multitask.reset()
        self.view.multitask_state_set(self.view.MULTITASK_NONE)

    def _remove_nowplaying(self, task_cont):
        task_cont.remove_nowplaying()
        self.view.multitask_state_set(self.view.MULTITASK_NONE)

    def _player_added(self, player):
        assert self._player_hook is None
        hook = NowPlayingHook(self.view)
        player.add_hook(hook)
        self._player_hook = hook

    def _player_removed(self, player):
        assert self._player_hook is not None
        player.remove_hook(self._player_hook)
        self._player_hook = None

    def _setup_multitask(self):
        self._taskcont = None    # controller of foreground task

        try:
            cls = self._mger.get_class("Model/Multitask")
        except ValueError:
            log.error("no multitask plugin??")
            return

        self._multitask = cls()
        self._player_hook = None
        self._multitask.callback_stop_player = self._remove_nowplaying
        self._multitask.callback_player_added = self._player_added
        self._multitask.callback_player_removed = self._player_removed

    def _task_running(self):
        return self._taskcont is not None

    def _setup_db_listener(self):
        def db_unlocked():
            notify = self._mger.canola_daemon.notify
            if notify is None:
                return True
            self._mger.canola_daemon.notify = None
            notify.stop()
            return True

        def notify_answered(*ignored):
            if self._mger.canola_daemon.notify is not None:
                self._mger.canola_daemon.notify = None
                log.debug_error("wait notify answered before receiving "
                                "unlocked DB signal")

        def db_locked():
            if self._mger.canola_daemon.notify is not None:
                log.debug_warning("received DB locked signal but already "
                                  "notifying user, ignoring it ...")
                return True
            notify = WaitNotify("Updating media library...",
                                120.0, notify_answered)
            self._mger.canola_daemon.notify = notify
            self._killtree(notify)
            return True

        self._mger.canola_daemon.callbacks_locked.insert(0, db_locked)
        self._mger.canola_daemon.callbacks_unlocked.insert(0, db_unlocked)
        self._mger.canola_daemon.start_monitoring()

        pref = PluginPrefs("settings")
        scan_startup = pref.get("scan_startup")
        if scan_startup is None:
            scan_startup = True

        if scan_startup:
            self._mger.canola_daemon.scan_all_paths()
        else:
            self._mger.canola_daemon.check_all_paths()

        return self._mger.canola_daemon.db_is_locked()

    def _killtree(self, notify):
        tasks = self._multitask.tasks.values()
        for tsk in tasks:
            if tsk is not self._taskcont: # not foreground task, please!
                tsk.killtree(notify)
                self._multitask.remove(tsk)

        if self._task_running():
            self._set_state_to_home()
            self._taskcont.killtree(notify)
            # XXX: make menu controller reload/reset whatever is
            # necessary. This is here because of our Menu and SubMenu
            # "always loaded" model optimizations.
            self.menu_controller.reset()
        else:
            old_cb = notify.answer_callback
            def answer_callback(*ignored):
                if old_cb is not None:
                    old_cb(*ignored)

                def cb(*ignored):
                    # XXX: make menu controller reload/reset whatever is
                    # necessary. This is here because of our Menu and SubMenu
                    # "always loaded" model optimizations.
                    self.menu_controller.reset()

                self.cb_on_go_home(self.view, end_callback=cb)

            notify.answer_callback = answer_callback
            self.show_notify(notify)

        self.view.multitask_state_set(self.view.MULTITASK_NONE)

    def _load_autostart_plugins(self):
        self._autostart = []

        try:
            plgs = self._mger.get_classes("AutoStart")
        except:
            log.info("no autostart plugins, continuing ...")
            return

        for cls in plgs:
            try:
                p = cls()
            except:
                log.debug_error("error when loading autostart plugin %r",
                                p, exc_info=True)
                continue
            self._autostart.append(p)

    def _view_transition_to(self, controller, end_callback):
        def cb(view, sub_view_old, sub_view_new):
            self.animating = False
            if end_callback is not None:
                end_callback()

        controller.resume()
        self.animating = True
        self.view.transition_to(controller.view, cb)

    def change_view_to(self, controller, end_callback=None):
        if isinstance(controller.view, Screen):
            self._view_transition_to(controller, end_callback)
        elif end_callback is not None:
            end_callback()

    def _back_change_view_to(self, last, controller, end_callback=None):
        if isinstance(last.view, Screen):
            self._view_transition_to(controller, end_callback)
        elif end_callback is not None:
            end_callback()

    def show_notify(self, model):
        if self._notify_controller is not None:
            self._notify_models.insert(0, model)
            return

        controller = get_controller_for_model(model, self.ecore_evas.evas, self)
        self._notify_controller = controller
        self.animating = True
        self.view.show_notify(controller.view)

    def hide_notify(self, controller, answer):
        assert controller == self._notify_controller

        self.view.hide_notify(controller.view)
        self.animating = False
        self._notify_controller = None

        cb = controller.model.answer_callback
        if cb is not None:
            cb(controller.model, answer)
        controller.delete()

        while self._notify_models:
            model = self._notify_models.pop(0)
            if not model._delete_me:
                self.show_notify(model)
                break
            del model

    def can_use(self, obj):
        if not isinstance(obj, Model):
            return True

        model = obj
        if not model.is_valid:
            return False

        cls = get_controller_class_for_model(model)
        if cls is not None:
            return True

        model.state_reason = PluginNotFoundError()
        return False

    def use_task(self, task, end_callback=None):
        task_cont = self._multitask.task_controller_for(task)
        if task_cont is None:
            task_cont = get_task_controller_for_task(task, self,
                                                     self._multitask)
            self._multitask.add(task_cont)

            run_task = task_cont.run
        else:
            run_task = task_cont.resume

        def end(*ignored):
            if end_callback is not None:
                end_callback(self, task)

        self._taskcont = task_cont
        run_task(end)

    def use_options(self, obj, end_callback=None):
        if self.animating:
            log.info("ignored use_options: was animating")
            return

        if self._options_controller is not None:
            log.error("there is an options controller being used already")
            return

        if not isinstance(obj, Controller):
            log.error("you must pass a controller")
            return

        self._options_controller = obj
        if end_callback is not None:
            end_callback(self, self._options_controller.model)

    def use(self, obj, end_callback=None):
        if self.animating:
            log.info("ignored: was animating")
            return

        if isinstance(obj, Task):
            last_controller = self._sub_controller
            def end(*ignored):
                last_controller.suspend()
                if end_callback is not None:
                    end_callback(self, self._sub_controller.model)
            self.use_task(obj, end)
            return
        elif isinstance(obj, Model):
            controller = get_controller_for_model(obj, self.ecore_evas.evas,
                                                  self)
        elif isinstance(obj, Controller):
            controller = obj
        else:
            log.error("unknown object type %r (expected Task, Model "
                      "or Controller)", obj.__class__.__name__)
            return

        last_controller = self._sub_controller
        self._sub_controllers.append(controller)
        self._sub_controller = controller
        self._sub_controller.ref()

        def end():
            last_controller.suspend()
            if end_callback is not None:
                end_callback(self, self._sub_controller.model)

        self.change_view_to(self._sub_controller, end)

    def back_options(self, end_callback=None):
        if self._options_controller is None:
            log.error("trying to back from options without having an "
                      "options controller in place.")
            return

        model = self._options_controller.model
        self._options_controller = None
        if end_callback is not None:
            end_callback(self, model)

    def _back_from_task(self, end_callback):
        def end():
            if end_callback is not None:
                end_callback(self)

            if not self._taskcont.must_hold():
                self._multitask.remove(self._taskcont)

            self._taskcont = None

        self.change_view_to(self._sub_controller, end)

    def _back_from_cont(self, end_callback):
        last = self._sub_controllers.pop()
        self._sub_controller = self._sub_controllers[-1]

        def end():
            last.suspend()
            last.unref()
            if end_callback is not None:
                end_callback(self)

        self._back_change_view_to(last, self._sub_controller, end)

    def back(self, end_callback=None):
        if self.animating:
            log.info("ignored: was animating")
            return

        if self._task_running():
            self._back_from_task(end_callback)
        else:
            self._back_from_cont(end_callback)

    def _set_state_to_home(self):
        del self._sub_controllers[0] # remove menu controller
        while self._sub_controllers:
            c = self._sub_controllers.pop()
            c.unref()

        self._sub_controllers = [self.menu_controller]
        self._sub_controller = self.menu_controller

    def _go_home_from_task(self, end_callback):
        def end():
            if end_callback is not None:
                end_callback(self)

            if not self._taskcont.must_hold():
                self._multitask.remove(self._taskcont)

            self._taskcont = None
            self._set_state_to_home()

        self.change_view_to(self.menu_controller, end)

    def _go_home_from_cont(self, end_callback):
        last = self._sub_controllers.pop()

        def end():
            last.unref()
            if end_callback is not None:
                end_callback(self)

            self._set_state_to_home()

        self._back_change_view_to(last, self.menu_controller, end)

    def go_home(self, end_callback=None):
        if self.animating:
            log.info("ignored: was animating")
            return

        if self._task_running():
            self._go_home_from_task(end_callback)
        else:
            self._go_home_from_cont(end_callback)

    def killall(self, reset_menu=True):
        assert not self._task_running()
        assert self._sub_controller is self.menu_controller

        notify = Notify("killall", "Killing everything ...")
        for tsk in self._multitask.tasks.itervalues():
            tsk.killtree(notify)

        self._reset_multitask()

        # Make menu controller reload/reset whatever is
        # necessary. This is here because of our Menu and SubMenu
        # "always loaded" model optimizations. This is also the
        # default behaviour for killall()
        if reset_menu:
            self.menu_controller.reset()
        else:
            self.menu_controller.unref()

    def cleanup(self):
        self.killall(reset_menu=False)
        self._mger.canola_daemon.callbacks_locked = []
        self._mger.canola_daemon.callbacks_unlocked = []
        self._mger.canola_daemon.shutdown()

    def cb_on_multitask_clicked(self, view):
        if self.animating:
            log.info("ignored: was animating")
            return

        if self.view._multitask_state == self.view.MULTITASK_NONE:
            self.cb_on_go_home(self.view)
            return

        new_taskcont = self._multitask.get_player_task_controller()
        if new_taskcont is None:
            log.debug("no player active, ignoring ...")
            return

        end = None
        if self._task_running() and \
           self._taskcont is not new_taskcont:
            old_task = self._taskcont
            def end(*ignored):
                old_task.suspend()
        elif not self._task_running():
            def end(*ignored):
                self._sub_controller.suspend()

        self._taskcont = new_taskcont
        self._taskcont.go_to_nowplaying(end)

    def cb_on_go_home(self, view, end_callback=None):
        if self.animating:
            log.info("ignored: was animating")
            return

        if self._task_running():
            self._taskcont.do_go_home()
            return

        if self._sub_controller is not self.menu_controller: # FIXME
            self._sub_controller.go_home(end_callback=end_callback)

    def cb_on_back(self, view):
        if self.animating:
            log.info("ignored: was animating")
            return

        if self._task_running():
            self._taskcont.do_back()
            return

        self._sub_controller.back()

    def cb_on_toggle_fullscreen(self, view):
        if self.animating:
            log.info("ignored: was animating")
            return
        self.ecore_evas.fullscreen = not self.ecore_evas.fullscreen

    def cb_on_panel_back(self):
        if self._options_controller is not None:
            self._options_controller.back_area_clicked()

    def _change_theme_link(self, theme):
        app_user_dir = mger.terra_config.app_user_dir
        theme_link = os.path.join(os.path.expanduser("~"), app_user_dir,
                                  "theme", "theme.edj")
        self._theme = theme
        if os.path.islink(theme_link):
            os.unlink(theme_link)
            os.symlink(theme, theme_link)

    def change_theme(self, theme):
        if self.animating:
            log.info("ignored: was animating")
            return

        self._change_theme_link(theme)

        def end_callback(view):
            for c in self._sub_controllers:
                if c.view:
                    c.view.theme_changed()
                    c.view.hide()
            for t in self._multitask.tasks.values():
                t.theme_changed()
            self._sub_controller.view.show()
            self.animating = False
        self.animating = True
        self.view.theme_set(theme, end_callback)

    def panel_stack(self, panel, end_callback=None):
        if self.animating:
            log.info("ignored: was animating")
            return

        def cb(view, panel):
            self.animating = False
            if end_callback:
                end_callback(self, panel)
        self.animating = True
        self.view.panel_stack(panel, cb)

    def panel_unstack(self, end_callback=None):
        if self.animating:
            log.info("ignored: was animating")
            return

        def cb(view, panel):
            self.animating = False
            if end_callback:
                end_callback(self, panel)
        self.animating = True
        self.view.panel_unstack(cb)


# XXX: debug function
_last_object_count = None
def dump_object_summary(verbose):
    global _last_object_count
    import gc
    gc.collect()
    objects = gc.get_objects()

    without_class = 0
    modules = { "etk": [], "settings": [], "edje": [], "panel": [] }

    # Put "fancydebug = True" to mark a class, and "fancydebug = False" to
    # unmark a subclass from this class.
    marked_classes = {}

    import terra.ui.panel
    import terra.ui.base
    classes = { Controller: [], Model: [], Screen: [],
                terra.ui.base.Widget: [],
                terra.ui.panel.Panel: [], terra.ui.panel.PanelContent: [] }

    for o in objects:
        try:
            o.__class__
        except:
            without_class += 1
            continue

        if isinstance(o, type):
            continue

        for m in modules.iterkeys():
            if o.__class__.__module__.find(m) >= 0:
                modules[m].append(o)
        for c in classes.iterkeys():
            if isinstance(o, c):
                classes[c].append(o)
                continue

        try:
            if hasattr(o, "fancydebug") and o.fancydebug \
                    and not isinstance(o, type):
                m = o.__class__.__module__
                if m.startswith("sqlobject") or m.startswith("dbus.proxies"):
                    continue
                c = marked_classes.setdefault(o.__class__, [])
                c.append(o)
        except:
            pass

    print "\n\n\n", "=" * 20, "Your fancy live objects summary", "=" * 20

    def dump_class(c):
        return "%s / %s" % (c, c.__name__)
    def dump_id(s):
        return s
    def dump_objects(title, objdict, dump_key=dump_class):
        if not objdict:
            return
        print "===== %s ======" % title
        for c, l in objdict.iteritems():
            print "- %s (%d)"  % (dump_key(c), len(l))
            if verbose:
                for o in l: print "\t", o
        print "\n"

    dump_objects("Objects from watched modules", modules, dump_id)
    dump_objects("Objects from watched classes", classes)
    dump_objects("Objects from MARKED classes", marked_classes)

    obj_count = len(objects)
    print "=" * 73
    print "Objects without class: %d" % without_class
    print "Total object count: %d" % obj_count
    if _last_object_count is not None:
        print "Last count: %d" % _last_object_count
        print "Difference: %d" % (obj_count - _last_object_count)
    _last_object_count = obj_count
    print "=" * 73, "\n\n"
