// Copyright (c) 2011 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef WEBKIT_QUOTA_QUOTA_TYPES_H_
#define WEBKIT_QUOTA_QUOTA_TYPES_H_

#include <deque>
#include <map>
#include <set>
#include <string>
#include <vector>

#include "base/basictypes.h"
#include "base/callback.h"
#include "webkit/quota/quota_status_code.h"

class GURL;

namespace quota {

enum StorageType {
  kStorageTypeTemporary,
  kStorageTypePersistent,
  kStorageTypeSyncable,
  kStorageTypeUnknown,
};

struct UsageInfo;
typedef std::vector<UsageInfo> UsageInfoEntries;

// Common callback types that are used throughout in the quota module.
typedef base::Callback<void(StorageType status,
                            int64 usage,
                            int64 unlimited_usage)> GlobalUsageCallback;
typedef base::Callback<void(QuotaStatusCode status, int64 quota)> QuotaCallback;
typedef base::Callback<void(int64 usage)> UsageCallback;
typedef base::Callback<void(QuotaStatusCode, int64)> AvailableSpaceCallback;
typedef base::Callback<void(QuotaStatusCode)> StatusCallback;
typedef base::Callback<void(const std::set<GURL>& origins,
                            StorageType type)> GetOriginsCallback;
typedef base::Callback<void(const UsageInfoEntries&)> GetUsageInfoCallback;

// Simple template wrapper for a callback queue.
template <typename CallbackType>
class CallbackQueueBase {
 public:
  typedef typename std::deque<CallbackType> Queue;
  typedef typename Queue::iterator iterator;

  virtual ~CallbackQueueBase() {}

  // Returns true if the given |callback| is the first one added to the queue.
  bool Add(const CallbackType& callback) {
    callbacks_.push_back(callback);
    return (callbacks_.size() == 1);
  }

  bool HasCallbacks() const {
    return !callbacks_.empty();
  }

 protected:
  std::deque<CallbackType> callbacks_;
};

template <typename CallbackType1, typename A1>
class CallbackQueue1 : public CallbackQueueBase<CallbackType1> {
 public:
  typedef typename CallbackQueueBase<CallbackType1>::Queue Queue;
  // Runs the callbacks added to the queue and clears the queue.
  void Run(A1 arg) {
    // Note: template-derived class needs 'this->' to access its base class.
    for (typename Queue::iterator iter = this->callbacks_.begin();
         iter != this->callbacks_.end(); ++iter) {
      iter->Run(arg);
    }
    this->callbacks_.clear();
  }
};

template <typename CallbackType2, typename A1, typename A2>
class CallbackQueue2 : public CallbackQueueBase<CallbackType2> {
 public:
  typedef typename CallbackQueueBase<CallbackType2>::Queue Queue;
  // Runs the callbacks added to the queue and clears the queue.
  void Run(A1 arg1, A2 arg2) {
    for (typename Queue::iterator iter = this->callbacks_.begin();
         iter != this->callbacks_.end(); ++iter) {
      iter->Run(arg1, arg2);
    }
    this->callbacks_.clear();
  }
};

template <typename CallbackType3, typename A1, typename A2, typename A3>
class CallbackQueue3 : public CallbackQueueBase<CallbackType3> {
 public:
  typedef typename CallbackQueueBase<CallbackType3>::Queue Queue;
  // Runs the callbacks added to the queue and clears the queue.
  void Run(A1 arg1, A2 arg2, A3 arg3) {
    for (typename Queue::iterator iter = this->callbacks_.begin();
         iter != this->callbacks_.end(); ++iter) {
      iter->Run(arg1, arg2, arg3);
    }
    this->callbacks_.clear();
  }
};

typedef CallbackQueue3<GlobalUsageCallback,
                       StorageType, int64, int64> GlobalUsageCallbackQueue;

template <typename CallbackType, typename CallbackQueueType, typename KEY>
class CallbackQueueMapBase {
 public:
  typedef std::map<KEY, CallbackQueueType> CallbackMap;
  typedef typename CallbackMap::iterator iterator;

  bool Add(const KEY& key, const CallbackType& callback) {
    return callback_map_[key].Add(callback);
  }

  bool HasCallbacks(const KEY& key) const {
    return (callback_map_.find(key) != callback_map_.end());
  }

  bool HasAnyCallbacks() const {
    return !callback_map_.empty();
  }

  iterator Begin() { return callback_map_.begin(); }
  iterator End() { return callback_map_.end(); }

  void Clear() { callback_map_.clear(); }

 protected:
  CallbackMap callback_map_;
};

template <typename CallbackType1, typename KEY, typename ARG>
class CallbackQueueMap1
    : public CallbackQueueMapBase<CallbackType1,
                                  CallbackQueue1<CallbackType1, ARG>,
                                  KEY> {
 public:
  typedef typename CallbackQueueMapBase<
      CallbackType1,
      CallbackQueue1<CallbackType1, ARG>,
      KEY>::iterator iterator;
  typedef CallbackQueue1<CallbackType1, ARG> Queue;

  // Runs the callbacks added for the given |key| and clears the key
  // from the map.
  void Run(const KEY& key, ARG arg) {
    if (!this->HasCallbacks(key))
      return;
    Queue& queue = this->callback_map_[key];
    queue.Run(arg);
    this->callback_map_.erase(key);
  }
};

typedef CallbackQueueMap1<UsageCallback, std::string, int64>
    HostUsageCallbackMap;

}  // namespace quota

#endif  // WEBKIT_QUOTA_QUOTA_TYPES_H_
