/*
 * Copyright (C) 2009 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 */

#ifndef _HAVE_DBUS_MODEL_H
#define _HAVE_DBUS_MODEL_H

#include <glib.h>
#include <glib-object.h>

#include <dbusmodel/dbus-peer.h>

G_BEGIN_DECLS

#define DBUS_TYPE_MODEL (dbus_model_get_type ())

#define DBUS_MODEL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), \
        DBUS_TYPE_MODEL, DbusModel))

#define DBUS_MODEL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), \
        DBUS_TYPE_MODEL, DbusModelClass))

#define DBUS_IS_MODEL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), \
        DBUS_TYPE_MODEL))

#define DBUS_IS_MODEL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), \
        DBUS_TYPE_MODEL))

#define DBUS_MODEL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), \
        DBUS_TYPE_MODEL, DbusModelClass))

#define DBUS_TYPE_G_VALUE_ARRAY (dbus_g_type_get_collection ("GPtrArray", G_TYPE_VALUE))

typedef struct _DbusModel DbusModel;
typedef struct _DbusModelClass DbusModelClass;
typedef struct _DbusModelPrivate DbusModelPrivate;

/**
 * DbusModelIter:
 *
 * The DbusModelIter structure is private and should only be used with the
 * provides #DbusModel API. It is owned by DbusModel and should not be freed.
 **/
typedef struct _DbusModelIter DbusModelIter;

/**
 * DbusModel:
 *
 * All fields in the DbusModel structure are private and should never be
 * accessed directly
 */
struct _DbusModel
{
  /*< private >*/
  DbusPeer          parent;

  DbusModelPrivate *priv;
};

struct _DbusModelClass
{
  /*< private >*/
  DbusPeerClass parent_class;

  /*< public >*/

  /*< signals >*/
  void (*ready)       (DbusModel *model);
  void (*row_added)   (DbusModel *model, DbusModelIter *iter);
  void (*row_removed) (DbusModel *model, DbusModelIter *iter);
  void (*row_changed) (DbusModel *model, DbusModelIter *iter);

  /*< vtable >*/

  /*< private >*/
  void (*_dbus_model_1) (void);
  void (*_dbus_model_2) (void);
  void (*_dbus_model_3) (void);
  void (*_dbus_model_4) (void);
};

/**
 * dbus_model_get_type:
 *
 * The GType of #DbusModel
 *
 * Return value: the #GType of #DbusModel
 **/
GType           dbus_model_get_type        (void) G_GNUC_CONST;

DbusModel *     dbus_model_new             (const gchar *path,
                                            guint        n_columns,
                                            ...);

DbusModel *     dbus_model_new_from_path   (const gchar *path);

void            dbus_model_set_n_columns   (DbusModel *self,
                                            guint      n_columns);

guint           dbus_model_get_n_columns   (DbusModel *self);

guint           dbus_model_get_n_rows      (DbusModel *self);

GType           dbus_model_get_column_type (DbusModel *self,
                                            guint      column);

void            dbus_model_clear           (DbusModel *self);

void            dbus_model_add             (DbusModel *self,
                                            ...);

void            dbus_model_remove          (DbusModel     *self,
                                            DbusModelIter *iter);

void            dbus_model_set             (DbusModel     *self,
                                            DbusModelIter *iter,
                                            ...);

void            dbus_model_set_valist      (DbusModel       *self,
                                            DbusModelIter   *iter,
                                            va_list          args);

void            dbus_model_set_value       (DbusModel       *self,
                                            DbusModelIter   *iter,
                                            guint            column,
                                            const GValue    *value);

void            dbus_model_get             (DbusModel     *self,
                                            DbusModelIter *iter,
                                            ...);

void            dbus_model_get_valist      (DbusModel     *self,
                                            DbusModelIter *iter,
                                            va_list        args);

void            dbus_model_get_value       (DbusModel     *self,
                                            DbusModelIter *iter,
                                            guint          column,
                                            GValue        *value);

DbusModelIter * dbus_model_get_first_iter  (DbusModel     *self);

DbusModelIter * dbus_model_get_last_iter   (DbusModel     *self);

DbusModelIter * dbus_model_get_iter_at_row (DbusModel     *self,
                                            guint          row);

guint64         dbus_model_get_uid         (DbusModel     *self,
                                            DbusModelIter *iter);

gboolean        dbus_model_get_bool        (DbusModel     *self,
                                            DbusModelIter *iter,
                                            guint          column);

guchar          dbus_model_get_uchar       (DbusModel     *self,
                                            DbusModelIter *iter,
                                            guint          column);

gint            dbus_model_get_int         (DbusModel     *self,
                                            DbusModelIter *iter,
                                            guint           column);

guint           dbus_model_get_uint        (DbusModel     *self,
                                            DbusModelIter *iter,
                                            guint          column);

gint64          dbus_model_get_int64       (DbusModel     *self,
                                            DbusModelIter *iter,
                                            guint          column);

guint64         dbus_model_get_uint64      (DbusModel     *self,
                                            DbusModelIter *iter,
                                            guint          column);

gdouble         dbus_model_get_double      (DbusModel     *self,
                                            DbusModelIter *iter,
                                            guint          column);

const gchar *   dbus_model_get_string      (DbusModel     *self,
                                            DbusModelIter *iter,
                                            guint          column);

DbusModelIter * dbus_model_next            (DbusModel     *self,
                                            DbusModelIter *iter);

DbusModelIter * dbus_model_prev            (DbusModel     *self,
                                            DbusModelIter *iter);

gboolean        dbus_model_is_first        (DbusModel     *self,
                                            DbusModelIter *iter);

gboolean        dbus_model_is_last         (DbusModel     *self,
                                            DbusModelIter *iter);

G_END_DECLS

#endif /* _HAVE_DBUS_MODEL_H */
