/*
 * Copyright (C) 2008  Intel Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, version 2, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * In addition, as a special exception, Intel gives permission to link
 * the code of portions of this program with the OpenSSL project's
 * "OpenSSL" library (or with modified versions of it that use the same
 * license as the "OpenSSL" library), and distribute the linked
 * executables.  You must obey the GNU General Public License in all
 * respects for all of the code used other than "OpenSSL".  If you modify
 * this file, you may extend this exception to your version of the file,
 * but you are not obligated to do so.  If you do not wish to do so,
 * delete this exception statement from your version.
 */


/**
 * @file AccelHelper.h
 *
 * Definition of class AccelHelper
 */

#ifndef ACCEL_HELPER_H
#define ACCEL_HELPER_H

#ifdef WIN32
#include <windows.h>
#else
#include <pthread.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#endif /* WIN32 */

#include <set>
#include "plugins/DcsPlugin.h"
#include "plugins/accel/AccelDef.h"
#include "plugins/accel/AccelPlugin.h"
#include "accelio.h"

/**
 * @class AccelHelper
 *
 * Provides help on operating Accel driver. It is implemented as a singleton.
 */
class AccelHelper
{
public:
	/**
	 * Get the single instance of AccelHelper.
	 *
	 * @return instance of AccelHelper
	 */
	static AccelHelper &GetInstance()
	{
		static AccelHelper helper;
		return helper;
	};

	/* public functions which really implement DCS Plugin interface. */
	int Initialize(DCS_Funcs *pDispatcher);
	DCS_Return_Code HandleRequest(DCS_RequestData *pRequest);
	int DestroyClient(int cliId);
	int Cleanup();
    bool IsEventEnabled() { return m_bIsEventEnabled; };

private:

#ifdef WIN32	
	AccelHelper(void)
		: m_pFuncs(NULL), m_hAccelEvent(NULL), m_hExitEvent(NULL),
		m_hThread(NULL), m_hDevice(NULL), m_GSelect(G_SELECT_1P5G),
		m_AccelSense(ACCEL_SENSE_1P5G),
		m_GravityDivSqrt2(GRAVITY_DIV_SQRT2_1P5G),
		m_SqrGravityDiv2(SQR_GRAVITY_DIV2_1P5G),
		m_GravityMeasurement(GRAVITY_MEASUREMENT_1P5G),
		m_Orientation(DCS_SYSTEM_ORIENTATION_NORMAL),
        m_bIsEventEnabled(false)
	{
		InitializeCriticalSection(&m_UserIdLock);
	};
	
	~AccelHelper(void)
	{
		if (m_hDevice) {
			Cleanup();
		}
		DeleteCriticalSection(&m_UserIdLock);
	};
#else
	AccelHelper(void)
		: m_Thread(0), m_Device(-1),
		m_Orientation(DCS_SYSTEM_ORIENTATION_NORMAL),
		m_GSelect(G_SELECT_1P5G), m_AccelSense(ACCEL_SENSE_1P5G),
		m_GravityDivSqrt2(GRAVITY_DIV_SQRT2_1P5G),
		m_SqrGravityDiv2(SQR_GRAVITY_DIV2_1P5G),
		m_GravityMeasurement(GRAVITY_MEASUREMENT_1P5G),
        m_bIsEventEnabled(false)
	{
		pthread_mutex_init(&m_UserIdLock, NULL);
	};

	~AccelHelper(void)
	{
		if (m_Device >= 0) {
			Cleanup();
		}
		pthread_mutex_destroy(&m_UserIdLock);
	};
#endif /* WIN32 */

	/* accel related functions */
	BOOL StartAccel();
	BOOL StopAccel();
	BOOL GetAccelData(DCS_Accel_Data &data);
	BOOL SetAccelSense(int sense);
	BOOL SetAccelGSelect(int select);
	BOOL GetAccelGSelect(int *select);

	/* get current system orientation */
	DCS_SystemOrientation GetSystemOrientation(const DCS_Accel_Data &data);

	/* check if system orientation changed and then notify DCS */
	void CheckAndNotifySystemOrientation(const DCS_Accel_Data &data);

	/* user ID related functions to record which user disabled
	 * system orientation notifications
	 */
	void AddUserId(const int &userId)
	{
		LockHelper helper(m_UserIdLock);
		m_ClientIdSet.insert(userId);
	};

	void EmptyUserId()
	{
		LockHelper helper(m_UserIdLock);
		m_ClientIdSet.clear();
	};

	void RemoveUserId(const int &userId)
	{
		LockHelper helper(m_UserIdLock);
		m_ClientIdSet.erase(userId);
	};

	BOOL IsUserIdEmpty()
	{
		LockHelper helper(m_UserIdLock);
		return m_ClientIdSet.empty();
	};

	/* declare copy constructor and equal operator to prevent
	 * copied instance
	 */
	AccelHelper(const AccelHelper &);
	AccelHelper &operator =(const AccelHelper &);

#ifdef WIN32
	/* WIN32 working thread function */
	friend DWORD WINAPI AccelEventHandler(LPVOID lpParameter);

	void CloseHandles();

	static const wchar_t * const EVENT_NAME; /**< event shared with driver */
	static const wchar_t * const DEVICE_PATH; /**< path to open driver */

	HANDLE m_hAccelEvent; /**< event to get notification from driver */
	HANDLE m_hExitEvent; /**< event to exit working thread */
	HANDLE m_hThread; /**< working thread */
	HANDLE m_hDevice; /**< accel device */

	CRITICAL_SECTION m_UserIdLock; /**< lock to protect user ID set */
#else
	/** Linux working thread function */
	friend void *AccelEventHandler(void *arg);	

	static const char * const DEVICE_PATH; /** path to open driver */
	pthread_mutex_t m_UserIdLock; /** lock to protect user ID set */

	pthread_t m_Thread;
	int m_Device;
#endif /* WIN32 */

	DCS_SystemOrientation m_Orientation; /**< current system orientation */
	
	DCS_Funcs *m_pFuncs; /**< pointer to DCS dispatch function */

    bool m_bIsEventEnabled; /**< the accel event is enabled or not */

	/* set to record the user IDs which disabled system oritentation
	 * notification
	 */
	std::set<int> m_ClientIdSet;

	/* acceleration related data */
	int m_GSelect; /**< current g-select */
	char m_AccelSense; /**< sense for current g-select */
	int m_GravityDivSqrt2; /**< gravity/sqrt(2) for current g-select */
	int m_SqrGravityDiv2; /**< sqr(gravity)/2 for current g-select */
	int m_GravityMeasurement; /**< 1g measurement for current g-select */

	/* const values */
	static const int ORIENTATION_SHRESHOLD; /**< shreshold for rotation */
	static const char * const MODULE_NAME; /**< accel chip module name */
	static const int G_SELECT_1P5G; /**< g-select for 1.5g */
	static const int G_SELECT_6G; /**< g-select for 6g */
	
	static const char GRAVITY_MEASUREMENT_1P5G; /**< measurement for 1.5g */
	static const int ACCEL_SENSE_1P5G; /**< default sense for 1.5g */	
	static const int GRAVITY_DIV_SQRT2_1P5G; /**< gravity/sqrt(2) for 1.5g */
	static const int SQR_GRAVITY_DIV2_1P5G; /**< sqr(gravity)/2 for 1.5g */

	static const char GRAVITY_MEASUREMENT_6G; /**< 1g measurement for 6g */
	static const int ACCEL_SENSE_6G; /**< default sense for 6g */	
	static const int GRAVITY_DIV_SQRT2_6G; /**< Gravity/sqrt(2) for 6g */
	static const int SQR_GRAVITY_DIV2_6G; /**< sqr(Gravity)/2 for 6g */

	
	/**
	 * @class LockHelper
	 *
	 * Provides auto lock and unlock approach by constructor and destructor.
	 */
#ifdef WIN32
	class LockHelper
	{
	public:
		LockHelper(CRITICAL_SECTION &lock)
			: m_Lock(lock)
		{
			EnterCriticalSection(&m_Lock);
		};

		~LockHelper()
		{
			LeaveCriticalSection(&m_Lock);
		};

	private:
		CRITICAL_SECTION &m_Lock;
	};
#else
	class LockHelper
	{
	public:
		LockHelper(pthread_mutex_t &lock)
			:m_Lock(lock)
		{
			pthread_mutex_lock(&m_Lock);
		};

		~LockHelper()
		{
			pthread_mutex_unlock(&m_Lock);
		};

	private:
		pthread_mutex_t m_Lock;
	};
#endif /* WIN32 */
};
#endif /* ACCEL_HELPER_H */

