/*
* Copyright (C) 2008  Intel Corporation
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License, version 2, as
* published by the Free Software Foundation.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License along
* with this program; if not, write to the Free Software Foundation, Inc.,
* 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*
* In addition, as a special exception, Intel gives permission to link
* the code of portions of this program with the OpenSSL project's
* "OpenSSL" library (or with modified versions of it that use the same
* license as the "OpenSSL" library), and distribute the linked
* executables.  You must obey the GNU General Public License in all
* respects for all of the code used other than "OpenSSL".  If you modify
* this file, you may extend this exception to your version of the file,
* but you are not obligated to do so.  If you do not wish to do so,
* delete this exception statement from your version.
*/

#include <stdlib.h>
#include <string.h>
#include <X11/XF86keysym.h>
#include <X11/keysym.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/soundcard.h>
#include <sys/types.h>
#include <pwd.h>
#include <pthread.h>
#include "FnKeyMon.h"
#include "UserSession.h"

const struct FnKeyMon::HotKey FnKeyMon::HOT_KEYS[HTK_NUM] = {
	{ADU_MUT, XF86XK_AudioMute}, {ADU_DCR, XF86XK_AudioLowerVolume},
	{ADU_INC, XF86XK_AudioRaiseVolume}, {LCK_CPS, XK_Caps_Lock},
	{LCK_NUM, XK_Num_Lock}, {LCK_SCR, XK_Scroll_Lock}	
};

int FnKeyMon::MAX_BRG = 1;
char FnKeyMon::SNG_CLC_CMD[MAX_PATH] = {0};
char FnKeyMon::DBL_CLC_CMD[MAX_PATH] = {0};
char FnKeyMon::SNG_CLC_HOME[MAX_PATH] = {0};
char FnKeyMon::DBL_CLC_HOME[MAX_PATH] = {0};

const int FnKeyMon::MAX_VLM = 100;
const int FnKeyMon::MAX_LVL = 14;
const float FnKeyMon::VLM_PER_LVL = FnKeyMon::MAX_VLM / FnKeyMon::MAX_LVL;
const char *FnKeyMon::SNG_CLC_ETC =
	"/etc/intel/bezel_button/single_click_launch";
const char *FnKeyMon::DBL_CLC_ETC =
	"/etc/intel/bezel_button/double_click_launch";
const char *FnKeyMon::MIXER_DEV = "/dev/mixer";
const char *FnKeyMon::DSP_TGG = "/usr/bin/dcs/DisplayToggle Toggle";
const char *FnKeyMon::USR_STT = "/usr/bin/dcs/UserSessionStatus";
const char *FnKeyMon::DSK_SWT = "desktop-switcher --set-mode=netbook";
const char *FnKeyMon::SHW_DSK = "wmctrl -k on";
const char *FnKeyMon::SHW_QCP = "/usr/share/QuickController/QuickController&";

/**
 * The working thread to install XKB keys for several times.
 *
 * @param arg	a pointer to FnKeyMon
 */
void *InsWrk(void *arg)
{
	FnKeyMon *mon = (FnKeyMon *)arg;
	while (1) {
		mon->InsXkbKey();
		sleep(8);
	}
	return 0;
}

/**
 * The working thread to receive key event/notification from X Server.
 *
 * @param arg	a pointer to FnKeyMon
 */
void *XkbWrk(void *arg)
{
	XkbEvent evn;
	FnKeyMon *mon = (FnKeyMon *)arg;
	Display *dsp = NULL;
	int vlm = 0, rgh = 0, lft = 0, lvl = 0;
	int mut_vlm = 0; /* volume when mute key was pressed */
	int msk, stt;
	pthread_t thr;

	mon->OpnDsp();
	dsp = mon->GetDsp();

	/* start a new thread to install XKB key for several times */
	pthread_create(&thr, NULL, InsWrk, mon);
	
	while (1) {
		XNextEvent(dsp, &evn.core);
		DbgPrint("key event received\n");

		if (XkbActionMessage == evn.any.xkb_type) {
			if (!mon->IsMixerValid()) {
				OnScrDsp::GetIns().ShowWnd(OnScrDsp::WND_PRV, lvl);
				continue;
			}

			switch (evn.message.keycode) {
			case ADU_MUT: /* mute was pressed */
				if (0 == mut_vlm) {
				   mut_vlm = mon->GetVlm();
				}
				vlm = rgh = lft = lvl = 0;
				mon->SetVlm(vlm);
				OnScrDsp::GetIns().ShowWnd(OnScrDsp::WND_MUT, lvl);
				break;

			case ADU_DCR: /* volume- was pressed */
				if (0 != mut_vlm) {
					vlm = mut_vlm;
					mut_vlm = 0;
				} else {
					vlm = mon->GetVlm();
				}
				lft = vlm & 0xFF;
				lft -= FnKeyMon::VLM_PER_LVL;
				if (lft < 0) {
					lft = 0;
				}
				rgh = lft;
				vlm = (rgh << 8) + lft;
				mon->SetVlm(vlm);
				lvl = lft / FnKeyMon::VLM_PER_LVL;
				OnScrDsp::GetIns().ShowWnd(OnScrDsp::WND_VLM, lvl);
				break;

			case ADU_INC: /* volume+ was pressed */
				if (0 != mut_vlm) {
					vlm = mut_vlm;
					mut_vlm = 0;
				} else {
					vlm = mon->GetVlm();
				}

				lft = vlm & 0xFF;
				lft += FnKeyMon::VLM_PER_LVL;
				if (lft > FnKeyMon::MAX_VLM) {
					lft = FnKeyMon::MAX_VLM;
				}
				rgh = lft;
				vlm = (rgh << 8) + lft;
				mon->SetVlm(vlm);
				lvl = lft / FnKeyMon::VLM_PER_LVL;
				OnScrDsp::GetIns().ShowWnd(OnScrDsp::WND_VLM, lvl);
				break;

			default:
				DbgPrint("get wrong keycode\n");
			}
		} else if (XkbIndicatorStateNotify == evn.any.xkb_type) {
			msk = evn.indicators.changed;
			stt = evn.indicators.state;
			
			switch (msk) {
			case 1: /* Caps Lock mask */
				OnScrDsp::GetIns().ShowWnd(OnScrDsp::WND_CPS, msk & stt);
				break;
			
			case 2: /* Num Lock mask */
				OnScrDsp::GetIns().ShowWnd(OnScrDsp::WND_NUM, msk & stt);
				break;
			
			case 3: /* Scroll Lock mask */
				OnScrDsp::GetIns().ShowWnd(OnScrDsp::WND_SCR, msk & stt);
				break;
			
			default:
				DbgPrint("get wrong status\n");
			}
		} else {
			DbgPrint("get wrong type\n");
		}
	}
	return 0;
}

/**
 * An empty function for Xlib after function.
 */
int AftFnc(Display *)
{
	return 0;
}

/**
 * Error handler to prevent application crash from X11 BadValue error
 */
int ErrHnd(Display *, XErrorEvent *)
{
	DbgPrint("FnKeyMon had X11 BadValue Error!!!\n");
//	DbgPrint("serial: %ld, error_code: %d, request_code: %d, minor_code: %d\n",
//		evn->serial, evn->error_code, evn->request_code, evn->minor_code);

	return 0;
}

/**
 * Constructor of FnKeyMon
 */
FnKeyMon::FnKeyMon() : m_pDsp(NULL), m_pXkb(NULL), m_Mixer(-1)
{
}

/**
 * Destructor of FnKeyMon
 */
FnKeyMon::~FnKeyMon()
{
	if (NULL != m_pDsp) {
		XCloseDisplay(m_pDsp);
	}

	if (-1 != m_Mixer) {
		close(m_Mixer);
	}
}

/**
 * Called by DCS when a function key was pressed.
 *
 * @param pData	a function key value
 */
void CALLBACK FnKeyMon::FnKeyCb(DCS_VKbd_Data *pData)
{
	DCS_Return_Code rsl = DCS_SUCCESS;
	BOOL stt = TRUE;
	int lvl = 5;
	int usr;

	if (NULL == pData) {
		DbgPrint("receive NULL data in FnKeyCb");
		assert(FALSE);
		return;
	}
	
	switch (*pData) {
	case FN_WLAN_SWITCH:
		DbgPrint("FN_WLAN_SWITCH\n");
		rsl = DCS_GetWirelessStatus(&stt);
		if (DCS_SUCCESS == rsl) {
			rsl = DCS_SetWirelessStatus(!stt);
			/* after set WLAN status, WrlChnCb should be callbacked
			 * and show OSD there
			 */
			if (DCS_REQUEST_DENIED == rsl) {
				OnScrDsp::GetIns().ShowWnd(OnScrDsp::WND_PRV, 0);
			}
		} else if (DCS_REQUEST_DENIED == rsl) {
			OnScrDsp::GetIns().ShowWnd(OnScrDsp::WND_PRV, 0);
		}
		break;

	case FN_DISPLAY_SWITCH:
		DbgPrint("FN_DISPLAY_SWITCH\n");
		/* check if user session is active */
		usr = system(USR_STT);
		if ((-1 != usr) && (127 != usr)) {
			usr = WEXITSTATUS(usr);
			if (USS_ActiveSession != usr) {
				return;
			}
		} else {
			DbgPrint("execute UserSessionStatus error: %d\n", usr);
			return;
		}

		/* run "DisplayToggle Toggle" to toggle display */
		lvl = system(DSP_TGG);
		/* 127 means /bin/sh can't be executed */
		if ((-1 != lvl) && (127 != lvl)) {
			lvl = WEXITSTATUS(lvl);
			DbgPrint("display status is %d\n", lvl);
			if ((1 == lvl) || (2 == lvl) || (3 == lvl)) {
				OnScrDsp::GetIns().ShowWnd(OnScrDsp::WND_DSP, lvl);
			} else {
				DbgPrint("unknown display status: %d\n", lvl);
			}
		} else {
			DbgPrint("execute DisplayToggle error: %d\n", lvl);
		}
		break;

	case FN_BRIGHTNESS_DOWN:
		DbgPrint("FN_BRIGHTNESS_DOWN\n");
		rsl = DCS_GetLCDBrightness(&lvl);
		if (DCS_SUCCESS == rsl) {
			if (lvl > 0) {
				lvl -= 1;
			}
			rsl = DCS_SetLCDBrightness(lvl);
			/* after set brightness, BrgChnCb should be callbacked
			 * and show OSD there
			 */
			if (DCS_REQUEST_DENIED == rsl) {
				OnScrDsp::GetIns().ShowWnd(OnScrDsp::WND_PRV, 0);
			}
		} else if (DCS_REQUEST_DENIED == rsl) {
			OnScrDsp::GetIns().ShowWnd(OnScrDsp::WND_PRV, 0);
		}
		break;

	case FN_BRIGHTNESS_UP:
		DbgPrint("FN_BRIGHTNESS_UP\n");
		rsl = DCS_GetLCDBrightness(&lvl);
		if (DCS_SUCCESS == rsl) {
			if (lvl < MAX_BRG) {
				lvl += 1;
			}
			rsl = DCS_SetLCDBrightness(lvl);
				/* after set brightness, BrgChnCb should be callbacked
				 * and show OSD there
				 */
			if (DCS_REQUEST_DENIED == rsl) {
				OnScrDsp::GetIns().ShowWnd(OnScrDsp::WND_PRV, 0);
			}
		} else if (DCS_REQUEST_DENIED == rsl) {
			OnScrDsp::GetIns().ShowWnd(OnScrDsp::WND_PRV, 0);
		}		
		break;

	case BEZELBUTTON_SINGLE_CLICK:
		DbgPrint("BEZELBUTTON_SINGLE_CLICK\n");
		/* run single clock command */
		system(SNG_CLC_CMD);
		system(SHW_DSK);
		break;

	case BEZELBUTTON_DOUBLE_CLICK:
		DbgPrint("BEZELBUTTON_DOUBLE_CLICK\n");
		/* run double click command */
		system(DBL_CLC_CMD);
		break;

	default:	/* should never hit this */
		DbgPrint("known function key\n");
	}
}

/**
 * Called by DCS when brightness changed. 
 *
 * @param pData	brightness value
 */
void CALLBACK FnKeyMon::BrgChnCb(int *pData)
{
	DbgPrint("currenly brightness is %d\n", *pData);
	OnScrDsp::GetIns().ShowWnd(OnScrDsp::WND_BRG, *pData);
}

/**
 * Called by DCS when WLAN status changed. 
 *
 * @param pData	WLAN on/off status
 */
void CALLBACK FnKeyMon::WrlChnCb(BOOL *pData)
{
	OnScrDsp::GetIns().ShowWnd(OnScrDsp::WND_WRL, *pData);
}

/**
 * Initialize FnKeyMon. It must be called after DCS_Initialize.
 */
void FnKeyMon::Init()
{
	int i = 0;

	if (0 == GetUsrHome(SNG_CLC_HOME)) {
		strcat(SNG_CLC_HOME, "/.intel/bezel_button/single_click_launch");
	}
		
	for (i = 0; i < MAX_PATH; ++i) {
		SNG_CLC_CMD[i] = 0;
	}

	if (0 != GetCmd(SNG_CLC_HOME, SNG_CLC_CMD)) {
		if (0 != GetCmd(SNG_CLC_ETC, SNG_CLC_CMD)) {
			DbgPrint("can't get single click command\n");
			strcpy(SNG_CLC_CMD, DSK_SWT);
		}
	}
	DbgPrint("single click command is: %s\n", SNG_CLC_CMD);

	if (0 == GetUsrHome(DBL_CLC_HOME)) {
		strcat(DBL_CLC_HOME, "/.intel/bezel_button/double_click_launch");
	}

	for (i = 0; i < MAX_PATH; ++i) {
		DBL_CLC_CMD[i] = 0;
	}

	if (0 != GetCmd(DBL_CLC_HOME, DBL_CLC_CMD)) {
		if (0 != GetCmd(DBL_CLC_ETC, DBL_CLC_CMD)) {
			DbgPrint("can't get double click command\n");
			strcpy(DBL_CLC_CMD, SHW_QCP);
		}
	}
	DbgPrint("double click command is: %s\n", DBL_CLC_CMD);

	if (DCS_SUCCESS != DCS_GetLCDMaxBrightness(&MAX_BRG)) {
		DbgPrint("DCS_GetLCDMaxBrightness error\n");
		DbgPrint("Max brightness will set to 1\n");
		MAX_BRG = 1;
	}

	m_Mixer = open(MIXER_DEV, O_RDWR | O_NONBLOCK);
	if (-1 == m_Mixer) {
		DbgPrint("Open mixer failed!\n");
	}
}

/**
 * Get current user's home folder. 
 *
 * @param buf	buffer to return the home folder
 * @return		0 if success or other values for errors
 */
int FnKeyMon::GetUsrHome(char *buf)
{
	uid_t uid;
	struct passwd *user;

	/* get user's home folder */
	uid = getuid();
	user = getpwuid(uid);
	if (NULL == user) {
		DbgPrint("can't find the current user!\n");
		return -1;
	} else {
		strcpy(buf, user->pw_dir);
		return 0;
	}
}

/**
 * Get command in configure file. 
 *
 * @param cfg	name of configure file
 * @param buf	buffer to return the home folder
 * @return		0 if success or other values for errors
 */
int FnKeyMon::GetCmd(const char *cfg, char *buf)
{
	FILE *fil = NULL;
	int len;

	fil = fopen(cfg, "r");
	if (NULL == fil) {
		DbgPrint("configure file of %s is not exists\n", cfg);
		return -1;
	}

	fgets(buf, MAX_PATH, fil);
	len = strnlen(buf, MAX_PATH);
	while (buf[len] <= 32) {
		buf[len] = 0;
		len--;
	}
	buf[len + 1] = '&';
	
	return 0;
}

/**
 * Install key pressed related events to X Server. The key events include
 * mute, volume+/- using XKB ActionMessage and Caps/Num/Scroll Lock using
 * XKB IndicatorStateNotify.
 */
void FnKeyMon::InsXkbKey()
{
	int i, cod;
	
	InsLckKey();

	for (i = 0; i < HTK_NUM; ++i) {
		cod = HOT_KEYS[i].cod;
		if ((ADU_MUT == cod) || (ADU_DCR == cod) || (ADU_INC == cod)) {
			InsVlmKey(cod, HOT_KEYS[i].sym);
			InsVlmKey(cod, HOT_KEYS[i].sym);
		}
	}
}

/**
 * Open connection to X Server and XKB.
 */
void FnKeyMon::OpnDsp()
{
	int rtr, err, mjr, mnr, rsn;
	
	mjr = XkbMajorVersion;
	mnr = XkbMinorVersion;
	m_pDsp = XkbOpenDisplay(NULL, &rtr, &err, &mjr, &mnr, &rsn);
	if (NULL == m_pDsp) {
		DbgPrint("XOpenDisplay failed\n");
		switch (err) {
		case XkbOD_BadLibraryVersion:
			DbgPrint("X suppm_pXkborts version %d.%d\n", mjr, mnr);
				break;
			case XkbOD_ConnectionRefused:
				DbgPrint("X connection refused\n");
				break;
			case XkbOD_BadServerVersion:
				DbgPrint("Server version is %d.%d\n", mjr, mnr);
				break;
			case XkbOD_NonXkbServer:
				DbgPrint("XKB not supported");
				break;
			default:
				DbgPrint("Unknown error %d from XkbOpenDisplay\n", err);
        }
		assert(FALSE);
	}

	XSynchronize(m_pDsp, TRUE);

	/* to prevent the X11 BadValue error to crash the application */
	XSetAfterFunction(m_pDsp, AftFnc);
	XSetErrorHandler(ErrHnd);

	m_pXkb = XkbGetMap(m_pDsp, XkbAllMapComponentsMask, XkbUseCoreKbd);
	if (NULL == m_pXkb) {
		DbgPrint("XkbGetMap fialed\n");
		assert(FALSE);
	}
}

/**
 * Install key pressed events to X Server including mute, volume+/-
 * using XKB ActionMessage.
 */
void FnKeyMon::InsVlmKey(int cod, KeySym sym)
{
	XkbAction act;
	XkbMapChangesRec chn;
	int typ[4];

	/* volume keys use XKB ActionMessage */
	act.msg.type = XkbSA_ActionMessage;
	act.msg.flags = XkbSA_MessageOnPress;
	typ[0] = XkbOneLevelIndex;

	/* To select ActionMessage, need to change key type and symbol first */
	if (Success != XkbChangeTypesOfKey(m_pXkb, cod,
		1, XkbGroup1Mask, typ, NULL)) {
		DbgPrint("XkbChangeTypeOfKey failed\n");
		assert(FALSE);
	}

	if (NULL == XkbResizeKeySyms(m_pXkb, cod, 1)) {
		DbgPrint("XkbResizeKeySyms failed\n");
		assert(FALSE);
	}
	*XkbKeySymsPtr(m_pXkb, cod) = sym;

	ChnMapTypSym(cod);
	ChnMapTypSym(cod);

	/* alloc memeory for new action */
	if (NULL == XkbResizeKeyActions(m_pXkb, cod, 1)) {
		DbgPrint("XkbResizeKeyAction failed\n");
		assert(FALSE);
	}
	(&(m_pXkb->server->acts[m_pXkb->server->key_acts[cod]]))[0] = act;

	memset(&chn, 0, sizeof(chn));
	chn.changed = XkbKeyActionsMask;
	chn.first_key_act = cod;
	chn.num_key_acts = 1;
	if (!XkbChangeMap(m_pDsp, m_pXkb, &chn)) {
		DbgPrint("XkbChangeMap failed\n");
		assert(FALSE);
	}

	if (!XkbSelectEvents(m_pDsp, XkbUseCoreKbd, XkbActionMessageMask,
		XkbActionMessageMask)) {
		DbgPrint("XkbSelectEvents failed\n");
		assert(FALSE);
	}
}

/**
 * Install Caps/Num/Scroll Lock XKB IndicatorStateNotify.
 */
void FnKeyMon::InsLckKey()
{
	if (!XkbSelectEvents(m_pDsp, XkbUseCoreKbd, XkbIndicatorStateNotifyMask,
		XkbIndicatorStateNotifyMask)) {
		DbgPrint("XkbSelectEvents failed\n");
		assert(FALSE);
	}
}

/**
 * Change key's type and symbol.
 *
 * @param cod	a key code
 */
void FnKeyMon::ChnMapTypSym(int cod)
{
	XkbMapChangesRec chn;

	memset(&chn, 0, sizeof(chn));
	chn.changed = XkbKeySymsMask | XkbKeyTypesMask;
	chn.first_key_sym = cod;
	chn.num_key_syms = 1;
	chn.first_type = 0;
	chn.num_types = m_pXkb->map->num_types;

	if (!XkbChangeMap(m_pDsp, m_pXkb, &chn)) {
		DbgPrint("XkbChangeMap failed\n");
		assert(FALSE);
	}
}

/**
 * Check if the audio mixer is valid.
 *
 * @return	TRUE if the mixer is valid otherwise FALSE
 */
BOOL FnKeyMon::IsMixerValid()
{
	if (-1 != m_Mixer) {
		return TRUE;
	} else {
		return FALSE;
	}
}

/**
 * Get current system volume.
 *
 * @return	current system volume
 */
int FnKeyMon::GetVlm()
{
	int vlm = 0, rtr;

	rtr = ioctl(m_Mixer, SOUND_MIXER_READ_VOLUME, &vlm);
	if (-1 == rtr) {
		DbgPrint("SOUND_MIXER_READ_VOLUME failed\n");
//		assert(FALSE);
	}

	return vlm;
}

/**
 * Set system volume.
 *
 * @param vlm	the volume value to set
 */
void FnKeyMon::SetVlm(int vlm)
{
	int rtr;

	rtr = ioctl(m_Mixer, SOUND_MIXER_WRITE_VOLUME, &vlm);
	if (-1 == rtr) {
		DbgPrint("SOUND_MIXER_WRITE_VOLUME failed\n");
//		assert(FALSE);
	}
}
