# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

import pgm
from elisa.plugins.pigment.widgets.list_horizontal import List
from elisa.plugins.pigment.widgets.const import *

from pgm.utils import maths
from pgm.timing import implicit

import math
import time as mod_time

import logging


class GridHorizontal(List):

    preloaded = 1

    def __init__(self, widget_class, rows=3, columns=5):
        self.rows = rows
        self.columns = columns

        super(GridHorizontal, self).__init__(widget_class, visible_range_size=columns)

    def _render_widgets(self, start, end):
        if self._renderer is None:
            return

        for i, widget in enumerate(self._widgets[start:end+1]):
            item_index = self._item_index_from_widget_index(i+start)
            if item_index >= 0 and item_index < len(self.model):
                item = self.model[item_index]
                self._renderer(item, widget)
                widget.visible = True
            else:
                widget.visible = False

    def visible_range_start__set(self, visible_range_start):
        old_start = self._visible_range_start
        self._visible_range_start = visible_range_start
        delta = int(math.modf(visible_range_start)[1]-math.modf(old_start)[1])

        total_columns = self.preloaded + self.columns
        if abs(delta) >= total_columns:
            self._render_widgets(0, len(self._widgets)-1)
        elif delta >= 1:
            self._move_widgets_from_beginning_to_end(delta * self.rows)
            self._render_widgets(len(self._widgets)-(delta * self.rows),
                                 len(self._widgets)-1)
        elif delta <= -1:
            self._move_widgets_from_end_to_beginning(-delta * self.rows)
            self._render_widgets(0, -delta * self.rows - 1)

        self._layout_all_widgets()

    visible_range_start = property(List.visible_range_start__get,
                                   visible_range_start__set)

    def visible_range_size__set(self, visible_range_size):
        self.columns = visible_range_size
        self._item_width = 1.0/self.columns
        self._item_height = 1.0/self.rows
        super(GridHorizontal, self).visible_range_size__set(visible_range_size)

    visible_range_size = property(List.visible_range_size__get,
                                  visible_range_size__set)

    def _create_widgets(self):
         # FIXME: do not destroy and reload everything; support floats
        for widget in self._widgets:
            widget.clean()
        self._widgets[:] = []

        nb_widgets = self._visible_range_size * self.rows + 2 * self.preloaded * self.rows
        for i in xrange(nb_widgets):
            widget = self._widget_class()
            self._widgets.append(widget)
            self.add(widget, forward_signals=False)
            self._prepare_widget(widget)
            self._connect_widget(widget)

    def _selected_to_range_start(self, selected):
        # FIXME: check me!
        half_size = (self.visible_range_size-1.0)/2.0

        total_columns = int(math.ceil(len(self.model) / self.rows))
        selected_column = int(selected / self.rows)

        if selected_column <= half_size or total_columns <= self.visible_range_size:
            visible_range_start = 0.0
        elif selected_column >= total_columns - half_size:
            visible_range_start = total_columns - self.visible_range_size + 1
        else:
            visible_range_start = selected_column - half_size

        return visible_range_start

    def _range_start_to_selected(self, range_start):
        # FIXME: check me!
        half_size = (self.visible_range_size-1.0)/2.0
        selected = (range_start + half_size) * self.rows
        selected = int(round(selected))
        selected = maths.clamp(selected, 0, len(self.model)-1)
        return selected

    def _item_index_from_widget_index(self, widget_index):
        toto = int(self._visible_range_start) * self.rows
        nb_preloaded_widgets = self.preloaded * self.rows
        item_index = widget_index + toto - nb_preloaded_widgets
        return item_index

    def _widget_index_from_item_index(self, item_index):
        toto = int(self._visible_range_start) * self.rows
        nb_preloaded_widgets = self.preloaded * self.rows
        widget_index = item_index - toto + nb_preloaded_widgets
        return widget_index

    def _prepare_widget(self, widget):
        widget.size = (self._item_width, self._item_height)

    def _prepare_selector(self):
        if self._selector != None:
            self._selector.size = (self._item_width, self._item_height)

    def _layout_widget(self, widget, row, column):
        x = self.compute_x(row, column)
        y = self.compute_y(row, column)
        widget.opacity = self.compute_opacity(row, column)
        widget.position = (x, y, 0.0)

    def _layout_all_widgets(self):
        if not self.visible:
            return

        # we assume that the widgets are created and loaded with their
        # corresponding data from the model

        for i, widget in enumerate(self._widgets):
            position = i - self.preloaded * self.rows
            row = int(position) % self.rows
            column = position/self.rows - math.modf(self._visible_range_start)[0]
            self._layout_widget(widget, row, column)

    def _layout_selector(self):
        if not self._selector:
            return

        visible_range_start = self._selected_to_range_start(self.selected_item_index)
        position = self.selected_item_index - int(visible_range_start) * self.rows
        row = int(position) % self.rows
        column = position/self.rows - math.modf(visible_range_start)[0]

        if self._selector.visible and self._selector.opacity != 0:
            selector = self._animated_selector
        else:
            selector = self._selector
            selector.z = -1.0

        selector.x = self.compute_x(row, column)
        selector.y = self.compute_y(row, column)

    def compute_x(self, row, column):
        return column*self._item_width

    def compute_y(self, row, column):
        return row*self._item_height

    # FIXME: the opacity of the last column should be full
    def compute_opacity(self, row, column):
        full_opacity = 255
        shaded_opacity = 150
        invisible_opacity = -80

        # make the transformation symmetrical
        if column <= self._visible_range_size/2.0:
            # beginning of the visible items case
            column = column
            start = self._visible_range_start
        else:
            # end of the visible items case
            column = self._visible_range_size - column - 1
            start = len(self.model) - self._visible_range_size -\
                    self._visible_range_start

        if start <= 0 or column >= 1 or self._visible_range_size <= 1.0:
            opacity = full_opacity
        elif column >= 0:
            opacity = maths.lerp(shaded_opacity, full_opacity, column)
        elif column < 0:
            if start >= 1:
                opacity = maths.lerp(invisible_opacity, shaded_opacity, column+1)
            else:
                opacity = maths.lerp(invisible_opacity, full_opacity, column+1)

        opacity = max(min(255, opacity), 0)

        return opacity

    # Signals support methods

    def do_scrolled(self, x, y, z, direction, time):
        if direction == pgm.SCROLL_UP:
            index = self.selected_item_index - self.rows
            if index >= 0:
                self.selected_item_index = index
        else:
            index = self.selected_item_index + self.rows
            if index < len(self.model):
                self.selected_item_index = index
        return True

    def do_key_press_event(self, viewport, event, widget):
        if event.keyval == pgm.keysyms.Left:
            if self.selected_item_index > 0:
                self.selected_item_index -= self.rows
        elif event.keyval == pgm.keysyms.Right:
            if self.selected_item_index < (len(self.model) - 1):
                self.selected_item_index += self.rows
        if event.keyval == pgm.keysyms.Up:
            if self.selected_item_index > 0:
                self.selected_item_index -= 1
        elif event.keyval == pgm.keysyms.Down:
            if self.selected_item_index < (len(self.model) - 1):
                self.selected_item_index += 1
        elif event.keyval == pgm.keysyms.Home:
            self.selected_item_index = 0
        elif event.keyval == pgm.keysyms.End:
            self.selected_item_index = len(self.model) - 1
        elif event.keyval == pgm.keysyms.Page_Up:
            self.selected_item_index -= (self.visible_range_size - 1) * self.rows
        elif event.keyval == pgm.keysyms.Page_Down:
            self.selected_item_index += (self.visible_range_size - 1) * self.rows

    def _visible_selected_from_position(self, x, y):
        relative_x = (x-self.absolute_x)/self.absolute_width
        visible_selected_x = int(relative_x*self.columns)
        relative_y = (y-self.absolute_y)/self.absolute_height
        visible_selected_y = int(relative_y*self.rows)

        selected_item_index = (visible_selected_x * self.rows) + visible_selected_y

        return selected_item_index

    def do_drag_motion(self, x, y, z, button, time, pressure):
        time_since_last = time - self._last_drag_motion
        if time_since_last > self.drag_motion_resolution:
            self._last_drag_motion = time
        else:
            return True

        absolute_item_width = self._item_width*self.absolute_width
        motion = (x-self._initial[0])/absolute_item_width
        self.visible_range_start -= motion

        time_delta = time - self._initial[2]
        if time_delta != 0:
            self.speed = motion/time_delta*1000.0

        self._initial = (x, y, time)
        self._drag_accum += abs(motion)

        return True

if __name__ == "__main__":
    logger = logging.getLogger()
    logger.setLevel(logging.DEBUG)

    grid = GridHorizontal.demo()
    grid.position = (0.0, 0.0, 0.0)
    grid.size = (4.0, 3.0)

    def renderer(item, widget):
        widget.label = str(item)
        widget.font_height = 0.5
        widget.alignment = pgm.TEXT_ALIGN_CENTER
        widget.bg_a = 0
    grid.set_renderer(renderer)

    try:
        __IPYTHON__
    except NameError:
        pgm.main()

