# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.


__maintainer__ = 'Florian Boucault <florian@fluendo.com>'


from elisa.core import common
from elisa.core import manager
from elisa.core import input_event
from elisa.core.utils import classinit
from elisa.core.components.input_provider import PushInputProvider, PollInputProvider
from twisted.internet import reactor, task

import gobject

class InputManager(manager.Manager,gobject.GObject):
    """
    InputManager provides a common place to retrieve input events
    coming from a GUI toolkit, additional input sources or even a
    network. InputEvents can be pushed by
    L{elisa.core.components.input_provider.InputProvider}s
    or be polled by the InputManager, depending on the implementation
    chosen: L{elisa.core.components.input_provider.PollInputProvider}
    or L{elisa.core.components.input_provider.PushInputProvider}.

    Other objects can connect to the manager's signals
    that are emitted when L{elisa.core.input_event.InputEvent}s coming from
    L{elisa.core.components.input_provider.InputProvider}s are received.
    """

    __gsignals__ = {
                    'input_event' : (gobject.SIGNAL_RUN_LAST,
                                     gobject.TYPE_NONE,
                                     (object,))
                    }

    def __init__(self):
        manager.Manager.__init__(self)

        # 35Hz polling rate for the PollInputProvider
        self._poll_rate = 1 / 35.
        self._poll_call = task.LoopingCall(self._poll_events)
        self._running = False

    def start(self):
        """ Start a polling loop that will regularly check the registered
        L{elisa.core.components.input_provider.PollInputProvider}s for new
        L{elisa.core.input_event.InputEvent}s.
        """
        self.info("Starting")
        self._running = True
        self._check_polling()

    def stop(self):
        """ Clean all the registered
        L{elisa.core.components.input_provider.InputProvider}s and stop
        polling for new L{elisa.core.input_event.InputEvent}s.
        """
        self.info("Stopping")
        self._running = False

        if self._poll_call.running:
            self._poll_call.stop()
        
        return manager.Manager.stop(self)

    def process_event(self, event):
        """ Fire the signal corresponding to the event.

        Each event type is mapped to a signal instance to which other
        elisa components can connect (e.g to monitor user key presses).

        This method can be called by
        L{elisa.core.components.input_provider.PushInputProvider}
        components when they receive input data from the input device.

        @param event:         the event to process
        @type event:          L{elisa.core.input_event.InputEvent}
        """
        self.debug("Event received: %r", event)

        if event == None:
            return

        self.emit('input_event', event)

    def register_component(self, component):
        """ Register a new InputProvider in the InputManager so that the
        events collected by the former are propagated by the latter.

        @param component: the InputProvider instance to register
        @type component:  L{elisa.core.components.input_provider.InputProvider}
        """
        manager.Manager.register_component(self, component)
        # register myself to push_input_providers
        if isinstance(component, PushInputProvider):
            component.input_manager = self
            component.bind()

        self._check_polling()

    def unregister_component(self, component):
        """ Clean the InputProvider and unregister it from the InputManager;
        no events from the InputProvider will be propagated anymore.

        @param component: the InputProvider instance to unregister
        @type component:  L{elisa.core.components.input_provider.InputProvider}
        """
        manager.Manager.unregister_component(self, component)
        if isinstance(component, PushInputProvider):
            component.unbind()
        self._check_polling()

    def _check_polling(self):
        if not self._running:
            return

        have_poll = False
        for component in self._components:
            if isinstance(component, PollInputProvider):
                have_poll = True
                break

        if have_poll and not self._poll_call.running:
            self._poll_call.start(self._poll_rate)
        elif self._poll_call.running:
            self._poll_call.stop()

    def _poll_events(self):
        """ Poll each registered PollInputProvider for InputEvents to
        process.
        """
        for component in self._components:
            if not isinstance(component, PollInputProvider):
                continue

            events = component.get_input_events()
            for event in events:
                self.process_event(event)

gobject.type_register(InputManager)

