/*
 * Enhanced Easy Network
 * Copyright (c) <2009>, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "ProfileManagerDlg.h"
#include <wx/msgdlg.h>

#include <wx/bitmap.h>
#include <wx/imaglist.h>
#include <wx/filedlg.h>
#include <wx/file.h>
#include <wx/filename.h>
#include "ENMApp.h"
#include "ENMUtil.h"

#include "pic/NetManager.xpm"
#include "pic/checked.xpm"
#include "pic/unchecked.xpm"
#include "pic/checked_dis.xpm"
#include "pic/unchecked_dis.xpm"
#include "pic/blank.xpm"

#include "EditProfileDialog.h"
#include "PolicyManager.h"
#include "ProfileVersion.h"
const long ProfileManagerDlg::ID_PROFILELISTCTRL = wxNewId();
const long ProfileManagerDlg::ID_NEWBUTTON = wxNewId();
const long ProfileManagerDlg::ID_EDITBUTTON = wxNewId();
const long ProfileManagerDlg::ID_DELETEBUTTON = wxNewId();
const long ProfileManagerDlg::ID_OKBUTTON = wxNewId();
const long ProfileManagerDlg::ID_CANCLEBUTTON = wxNewId();
const long ProfileManagerDlg::ID_UPBITMAPBUTTON = wxNewId();
const long ProfileManagerDlg::ID_DOWNBITMAPBUTTON = wxNewId();
const long ProfileManagerDlg::ID_PROFILEPANEL = wxNewId();
const long ProfileManagerDlg::ID_IMPORT = wxNewId();
const long ProfileManagerDlg::ID_EXPORT = wxNewId();
const long ProfileManagerDlg::ID_EXIT = wxNewId();
const long ProfileManagerDlg::ID_NEWPROFILE = wxNewId();
const long ProfileManagerDlg::ID_EDITPROFILE = wxNewId();
const long ProfileManagerDlg::ID_DELETEPROFILE = wxNewId();
const long ProfileManagerDlg::ID_HELP = wxNewId();
const long ProfileManagerDlg::ID_SELECTCHECKBOX = wxNewId();
const int  IMAGE_INIT_POS = 5 ;

BEGIN_EVENT_TABLE(ProfileManagerDlg, wxFrame)
    EVT_BUTTON(ID_NEWBUTTON, ProfileManagerDlg::OnButtonNewClick)
    EVT_BUTTON(ID_EDITBUTTON, ProfileManagerDlg::OnButtonEditClick)
    EVT_BUTTON(ID_DELETEBUTTON, ProfileManagerDlg::OnButtonDeleteClick)
    EVT_BUTTON(ID_OKBUTTON, ProfileManagerDlg::OnButtonOKClick)
    EVT_BUTTON(ID_CANCLEBUTTON, ProfileManagerDlg::OnButtonCancelClick)
    EVT_BUTTON(ID_UPBITMAPBUTTON, ProfileManagerDlg::OnButtonUpClick)
    EVT_BUTTON(ID_DOWNBITMAPBUTTON, ProfileManagerDlg::OnButtonDownClick)
    EVT_MENU(ID_NEWPROFILE, ProfileManagerDlg::Onm_miNewSelected)
    EVT_MENU(ID_EDITPROFILE, ProfileManagerDlg::Onm_miEditSelected)
    EVT_MENU(ID_DELETEPROFILE, ProfileManagerDlg::Onm_miDeleteSelected)
    EVT_MENU(ID_IMPORT, ProfileManagerDlg::Onm_miImportSelected)
    EVT_MENU(ID_EXPORT, ProfileManagerDlg::Onm_miExportSelected)
    EVT_MENU(ID_EXIT, ProfileManagerDlg::Onm_miExitSelected)
    EVT_MENU(ID_HELP, ProfileManagerDlg::MenuHelpContent)
    EVT_CHECKBOX(ID_SELECTCHECKBOX, ProfileManagerDlg::OnCheckboxSelectClick)
    EVT_CLOSE(ProfileManagerDlg::OnCloseClicked)
    EVT_PROFILE(ID_PROFILE_CHANGED, ProfileManagerDlg::OnProfileChanged)
END_EVENT_TABLE()

ProfileManagerDlg::ProfileManagerDlg(wxWindow* parent,wxWindowID id)
        : wxFrame(), m_imageList(16, 16)
{
    Create(parent, wxID_ANY, _("Network Profile Manager"), wxDefaultPosition, wxDefaultSize, wxMINIMIZE_BOX | wxSYSTEM_MENU | wxCLOSE_BOX | wxCAPTION | wxCLIP_CHILDREN);
    int width = 500, height = 375;
    int margin = 10;
    SetClientSize(wxSize(width, height));
    wxSize listSize(385, 290);
    m_lcProfile = new wxCheckedListCtrl(this, ID_PROFILELISTCTRL, wxPoint(margin, margin), listSize, wxLC_REPORT | wxLC_SINGLE_SEL);

    wxSize btnSize(85, 30);
    int x = width - btnSize.GetWidth() - margin;
    int y = margin;
    m_bbUp = new wxBitmapButton(this, ID_UPBITMAPBUTTON, wxBitmap(wxImage(::wxGetApp().strImagePath + _T("up.png"))), wxPoint(x, y), btnSize, wxBU_AUTODRAW);
    y += btnSize.GetHeight() + margin;
    m_bbDown = new wxBitmapButton(this, ID_DOWNBITMAPBUTTON, wxBitmap(wxImage(::wxGetApp().strImagePath + _T("down.png"))), wxPoint(x, y), btnSize, wxBU_AUTODRAW);
    y = listSize.GetHeight() / 2;
    m_buttonNew = new wxButton(this, ID_NEWBUTTON, _("New..."), wxPoint(x, y),btnSize);
    y += btnSize.GetHeight() + margin;
    m_buttonEdit = new wxButton(this, ID_EDITBUTTON, _("Edit..."), wxPoint(x, y), btnSize);
    y += btnSize.GetHeight() + margin;
    m_buttonDelete = new wxButton(this, ID_DELETEBUTTON, _("Delete"), wxPoint(x, y), btnSize);

    x = margin;
    y = listSize.GetHeight() + margin * 2;
    m_checkboxSelect = new wxCheckBox(this, ID_SELECTCHECKBOX, _("&Show All"), wxPoint(x, y), wxSize(120, 30), wxALIGN_CENTER_VERTICAL);
    x = width - (btnSize.GetWidth() + margin) * 2;
    m_buttonCancel = new wxButton(this, ID_CANCLEBUTTON, _("&Cancel"), wxPoint(x, y), btnSize);
    x += btnSize.GetWidth() + margin;
    m_buttonOK = new wxButton(this, ID_OKBUTTON, _("&OK"), wxPoint(x, y), btnSize);

    m_menuProfile = new wxMenuBar();
    m_menuFile = new wxMenu();
    m_miImport = new wxMenuItem(m_menuFile, ID_IMPORT, _("&Import Profile..."), wxEmptyString, wxITEM_NORMAL);
    m_menuFile->Append(m_miImport);
    m_miExport = new wxMenuItem(m_menuFile, ID_EXPORT, _("E&xport Profile..."), wxEmptyString, wxITEM_NORMAL);
    m_menuFile->Append(m_miExport);
    m_menuFile->Enable(ID_EXPORT, false);
    m_menuFile->AppendSeparator();

    m_miNew = new wxMenuItem(m_menuFile, ID_NEWPROFILE, _("&New..."), wxEmptyString, wxITEM_NORMAL);
    m_menuFile->Append(m_miNew);
    m_miEdit = new wxMenuItem(m_menuFile, ID_EDITPROFILE, _("&Edit..."), wxEmptyString, wxITEM_NORMAL);
    m_menuFile->Append(m_miEdit);
    m_menuFile->Enable(ID_EDITPROFILE, false);
    m_miDelete = new wxMenuItem(m_menuFile, ID_DELETEPROFILE, _("&Delete"), wxEmptyString, wxITEM_NORMAL);
    m_menuFile->Append(m_miDelete);
    m_menuFile->Enable(ID_DELETEPROFILE, false);

    m_menuFile->AppendSeparator();

    m_miExit = new wxMenuItem(m_menuFile, ID_EXIT, _("&Exit"), wxEmptyString, wxITEM_NORMAL);
    m_menuFile->Append(m_miExit);
    m_menuProfile->Append(m_menuFile, _("&File"));

    m_menuHelp = new wxMenu();
    m_miHelp = new wxMenuItem(m_menuHelp, ID_HELP, _("Help &Contents  \tF1"), wxEmptyString, wxITEM_NORMAL);
    m_menuHelp->Append(m_miHelp);
    m_menuProfile->Append(m_menuHelp, _("&Help"));
    SetMenuBar(m_menuProfile);
    Center();

    Connect(ID_PROFILELISTCTRL,wxEVT_COMMAND_LIST_ITEM_SELECTED,(wxObjectEventFunction)&ProfileManagerDlg::Onm_lcProfileSelected);
    Connect(ID_PROFILELISTCTRL,wxEVT_COMMAND_LIST_ITEM_DESELECTED,(wxObjectEventFunction)&ProfileManagerDlg::Onm_lcProfileDeselected);
    Connect(ID_PROFILELISTCTRL,wxEVT_COMMAND_LIST_ITEM_ACTIVATED,(wxObjectEventFunction)&ProfileManagerDlg::OnDoubleClick);

    wxListItem itemCol;
    itemCol.SetText(_("Show"));
    m_lcProfile->InsertColumn(0, itemCol);
    m_lcProfile->SetColumnWidth(0, 80);

    itemCol.SetText(_("Priority"));
    m_lcProfile->InsertColumn(1, itemCol);
    m_lcProfile->SetColumnWidth(1, 80);

    itemCol.SetText(_("Profile Name"));
    m_lcProfile->InsertColumn(2, itemCol);
    m_lcProfile->SetColumnWidth(2, 225);

    itemCol.SetAlign(wxLIST_FORMAT_CENTRE);


    m_lcProfile->SetImageList(&m_imageList, wxIMAGE_LIST_SMALL);

#if defined(__UNIX__)
    wxImage m_imageGeneral(::wxGetApp().strAppPath + _T("pic/general.png"), wxBITMAP_TYPE_PNG);
    wxImage m_imageHome(::wxGetApp().strAppPath + _T("pic/home.png"), wxBITMAP_TYPE_PNG);
    wxImage m_imagePersonal(::wxGetApp().strAppPath + _T("pic/personal.png"), wxBITMAP_TYPE_PNG);
    wxImage m_imageHouse(::wxGetApp().strAppPath + _T("pic/house.png"), wxBITMAP_TYPE_PNG);
    wxImage m_imageHospital(::wxGetApp().strAppPath + _T("pic/hospital.png"), wxBITMAP_TYPE_PNG);
    wxImage m_imageBuilding(::wxGetApp().strAppPath + _T("pic/building.png"), wxBITMAP_TYPE_PNG);
    wxImage m_imageBucket(::wxGetApp().strAppPath + _T("pic/bucket.png"), wxBITMAP_TYPE_PNG);
    wxImage m_imageAirport(::wxGetApp().strAppPath + _T("pic/airport.png"), wxBITMAP_TYPE_PNG);
#endif

    wxBitmap m_bmGeneral(m_imageGeneral);
    wxBitmap m_bmHome(m_imageHome);
    wxBitmap m_bmPersonal(m_imagePersonal);
    wxBitmap m_bmHouse(m_imageHouse);
    wxBitmap m_bmHospital(m_imageHospital);
    wxBitmap m_bmBuilding(m_imageBuilding);
    wxBitmap m_bmBucket(m_imageBucket);
    wxBitmap m_bmAirport(m_imageAirport);

    //four icon for checkbox
    m_imageList.Add(wxIcon(unchecked_xpm));
    m_imageList.Add(wxIcon(checked_xpm));
    m_imageList.Add(wxIcon(unchecked_dis_xpm));
    m_imageList.Add(wxIcon(checked_dis_xpm));
    m_imageList.Add(wxIcon(blank_xpm));

    m_imageList.Add(m_bmGeneral);
    m_imageList.Add(m_bmHome);
    m_imageList.Add(m_bmPersonal);
    m_imageList.Add(m_bmHouse);
    m_imageList.Add(m_bmHospital);
    m_imageList.Add(m_bmBuilding);
    m_imageList.Add(m_bmBucket);
    m_imageList.Add(m_bmAirport);

    m_bbUp->Enable(false);
    m_bbDown->Enable(false);
    m_buttonEdit->Enable(false);
    m_buttonDelete->Enable(false);

//    wxIcon icon_netmanager(netmanager_xpm);
//    SetIcon(icon_netmanager);
    wxIcon icon;
    wxBitmap bitmap(wxImage(::wxGetApp().strAppPath + _T("pic/NetManager.png")));
    icon.CopyFromBitmap(bitmap);
    SetIcon(icon);
    m_isDeselected = true;
}

ProfileManagerDlg::~ProfileManagerDlg()
{
}

bool ProfileManagerDlg::IsProfileExist(const wxString& strNewName, const wxString& strOldName) const
{
    bool result = false;
    size_t iSize = m_vCurrentUserProfileList.size();
    for ( size_t i = 0; i < iSize; i++ )
    {
        wxString name = m_vCurrentUserProfileList[i].sProfileName;
        if ( strOldName.Length() > 0 )
            result |= ENMUtil::FilenameEquals(strNewName, name) && !ENMUtil::FilenameEquals(strNewName, strOldName);
        else
            result |= ENMUtil::FilenameEquals(name, strNewName);
    }
    return result;
}

void ProfileManagerDlg::InsertNewProfile(const ProfileList_t& profile)
{
    if ( !IsShown() )
        return;
    if ( -1 != IsGuestExist() )
    {
        m_vProfileList.insert( m_vProfileList.end()-1, profile);
        m_vOldProfileList.insert(m_vOldProfileList.end()-1, profile);
    }
    else
    {
    	m_vOldProfileList.push_back(profile);
    	m_vProfileList.push_back(profile);
    m_vCurrentUserProfileList.push_back(profile);
    }
    drawListCtrl(-1);
}

int ProfileManagerDlg::Confirm()
{
    wxMessageDialog dlg(this, _("One or more profiles may have changed. \n\nAre you sure you want to save them?"), _("Confirm"), wxYES_NO | wxYES_DEFAULT | wxICON_QUESTION);
    return dlg.ShowModal();
}

int ProfileManagerDlg::ConfirmOverwrite()
{
    wxMessageDialog dlg(this, _("This profile already exists.\n\nAre you sure you want to overwrite it?"), _("Confirm"), wxYES_NO | wxNO_DEFAULT | wxICON_EXCLAMATION);
    return dlg.ShowModal();
}

bool ProfileManagerDlg::Save()
{
    int listCount = m_lcProfile->GetItemCount();
    for( int n = 0 ; n < listCount; n++ ) // modify
    {
        bool bCheck = m_lcProfile->IsChecked(n);
        ModifyCheckFlag(n, bCheck);
    }

    wxArrayString array;
    if ( !DataCenter::GetInstance()->GetProfileController()->ConfirmModify(&m_vProfileList, array) )
    {
        wxString msg(_("Failed to save the following profiles:\n"));
        for ( size_t i = 0; i < array.Count(); i++ )
        {
            msg.Append(array[i] + _T("\n"));
        }
        wxMessageDialog dlg(this, msg, wxEmptyString, wxOK | wxICON_EXCLAMATION);
        dlg.ShowModal();
    }
    m_vProfileList.clear();
    m_lcProfile->DeleteAllItems();
    return true;
}

bool ProfileManagerDlg::Show(bool ifShow)
{
    m_isModified = false;
    if ( 0 == m_lcProfile->GetItemCount() && ifShow )
    {
        m_bIfAdmin = ENMUtil::IsAdministrator();
        DataCenter::GetInstance()->GetProfileController()->GetUserProfiles(m_vProfileList, m_vCurrentUserProfileList, m_bIfAdmin, true);
        m_vOldProfileList = m_vProfileList;
        m_iAdminCount = DataCenter::GetInstance()->GetProfileController()->GetAdminProfileCount();

        drawListCtrl(-1);

        m_bbUp->Enable(false);
        m_bbDown->Enable(false);
        m_buttonEdit->Enable(false);
        m_buttonDelete->Enable(false);
        m_menuFile->Enable(ID_EDITPROFILE, false);
        m_menuFile->Enable(ID_DELETEPROFILE, false);
        m_menuFile->Enable(ID_EXPORT, false);
    }
    return wxFrame::Show(ifShow);
}

bool ProfileManagerDlg::Destroy()
{
    Show(false);
    return false;
}

void ProfileManagerDlg::drawListCtrl(long selectedRow)
{
    m_critsectDrawlist.Enter();
    wxListItem newItem;
    int  listCount = m_lcProfile->GetItemCount();
    int  vecSize = m_vProfileList.size();

    if( listCount <= vecSize )
    {
        for(int n = 0 ; n < listCount; n++) // modify
        {
            wxString buf;
   //         int bShow = atol(m_vProfileList[n].sShow.c_str());
            long tmp = n ;

            if( m_vProfileList[n].sShow == _T("1") )
                m_lcProfile->Check(tmp,true);
            else
                m_lcProfile->Check(tmp,false);

            buf.Printf(_T("%d"), n+1);
            m_lcProfile->SetItem(tmp, 1, buf);

            buf = m_vProfileList[n].sProfileName;
            m_lcProfile->SetItem(tmp, 2, buf);

            long imgNum = 0;
			m_vProfileList[n].sProfileIcon.ToLong(&imgNum);
            if(imgNum >=0)
                m_lcProfile->SetItemColumnImage(tmp, 2, IMAGE_INIT_POS + imgNum);
            else
                m_lcProfile->SetItemColumnImage(tmp, 2,4);
        }
        for( int m = listCount ; m < vecSize ; m++) //add
        {
            wxString buf;
            int bShow = atol(m_vProfileList[m].sShow.mb_str());
            long tmp = m_lcProfile->InsertItem(m, buf, 0);
            if(bShow)
                m_lcProfile->Check(tmp,true);
            else
                m_lcProfile->Check(tmp,false);

            m_lcProfile->SetItemData(tmp, m);

            buf.Printf(_T("%d"), m+1);
            m_lcProfile->SetItem(tmp, 1, buf);

            buf = m_vProfileList[m].sProfileName;
            m_lcProfile->SetItem(tmp, 2, buf);

            int imgNum = atol(m_vProfileList[m].sProfileIcon.mb_str());
            if( imgNum >= 0 )
                m_lcProfile->SetItemColumnImage(tmp, 2, IMAGE_INIT_POS + imgNum);
            else
                m_lcProfile->SetItemColumnImage(tmp, 2,4);
        }
    }
    else
    {
        for(int n = 0 ; n < vecSize; n++) // modify
        {
            wxString buf;
            int bShow = atol(m_vProfileList[n].sShow.mb_str());
            long tmp = n ;

            if(bShow)
                m_lcProfile->Check(tmp,true);
            else
                m_lcProfile->Check(tmp,false);

            buf.Printf(_T("%d"), n+1);
            m_lcProfile->SetItem(tmp, 1, buf);

        //    buf.Printf(m_vProfileList[n].sProfileName.c_str()));
            m_lcProfile->SetItem(tmp, 2, m_vProfileList[n].sProfileName);

            int imgNum = atol(m_vProfileList[n].sProfileIcon.mb_str());
            if(imgNum >=0)
                m_lcProfile->SetItemColumnImage(tmp, 2, IMAGE_INIT_POS + imgNum);
            else
                m_lcProfile->SetItemColumnImage(tmp, 2,4);
        }
        for(int m = 0 ; m < listCount - vecSize ; m++)
        {
            m_lcProfile->DeleteItem(vecSize);
        }
    }

    if ( !m_bIfAdmin )
    {
        listCount = m_lcProfile->GetItemCount();
        if(m_iAdminCount <= listCount )
        {
            for(int n = 0 ; n < m_iAdminCount ; n++)
            {
               wxColour  bgcolor = m_lcProfile->GetItemBackgroundColour(n);
               m_lcProfile->Enable(n,false);
               m_lcProfile->SetItemBackgroundColour(n, wxColour(255, 255, 255) );
               m_lcProfile->SetItemTextColour(n, wxColour(220, 220, 220));
            }
        }
    }

    int nGuestProfile = IsGuestExist();
    if ( -1 != nGuestProfile )
    {
        m_lcProfile->Enable(nGuestProfile,false);
        m_lcProfile->SetItemBackgroundColour(nGuestProfile, wxColour(255, 255, 255) );
    }
    if( selectedRow >= 0 && selectedRow < listCount )
    {
        m_lcProfile->SetItemStateOri(selectedRow, wxLIST_STATE_SELECTED, wxLIST_STATE_SELECTED);
    }
    m_lcProfile->EnableCheck(-1);
    for ( vector<ProfileList_t>::size_type i = 0; i < m_vProfileList.size(); i++)
    {
        if ( m_vProfileList[i].sProfileName == DataCenter::GetInstance()->GetProfileController()->GetCurrentProfile() )
        {
            m_lcProfile->EnableCheck(i);
        }
    }

    // set status of "Select All" checkbox
    bool bAllSelected = m_lcProfile->GetItemCount() > 0;
    for ( int i = 0; i < m_lcProfile->GetItemCount(); i++ )
    {
        bAllSelected &= m_lcProfile->IsChecked(i);
    }
    m_checkboxSelect->SetValue(bAllSelected);
    m_critsectDrawlist.Leave();
}

void ProfileManagerDlg::OnButtonUpClick(wxCommandEvent& event)
{
    m_isDeselected = false;
    long item = -1;
    long nowId = m_lcProfile->GetNextItem(item, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);

    int  listCount = m_lcProfile->GetItemCount();
    for(int n = 0 ; n < listCount; n++) // modify
    {
        bool bCheck = m_lcProfile->IsChecked(n);
        ModifyCheckFlag(n, bCheck);
    }

    if(nowId > 0)
    {
//        m_isModified = true;
        ChangePriority( nowId , nowId + 1 );
        m_lcProfile->SetItemStateOri(nowId, 0, wxLIST_STATE_SELECTED);
        drawListCtrl( nowId - 1 );
    }
    m_isDeselected = true;
}

void ProfileManagerDlg::OnButtonDownClick(wxCommandEvent& event)
{
    m_isDeselected = false;
    long item = -1;
    long nowId = m_lcProfile->GetNextItem(item, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
    int  listCount = m_lcProfile->GetItemCount();
    for(int n = 0 ; n < listCount; n++) // modify
    {
        bool bCheck = m_lcProfile->IsChecked(n);
        ModifyCheckFlag(n, bCheck);
    }

    if(nowId >= 0 && nowId < m_lcProfile->GetItemCount())
    {
//        m_isModified = true;
        if ( -1 != IsGuestExist() && nowId == m_lcProfile->GetItemCount()-1)
            return;
        ChangePriority( nowId+1 , nowId + 2 );
        m_lcProfile->SetItemStateOri(nowId, 0, wxLIST_STATE_SELECTED);
        drawListCtrl( nowId + 1 );
    }
    m_isDeselected = true;
}

void ProfileManagerDlg::OnButtonNewClick(wxCommandEvent& event)
{
    PolicyManager* p = DataCenter::GetInstance()->GetPolicyManager();
    if ( !p->IsAllowOwnerProfile() )
    {
        wxMessageDialog dlg(this, _("The current policy forbids the creation of new profiles."), wxEmptyString, wxOK | wxICON_INFORMATION);
        dlg.ShowModal();
        return;
    }

    ScanNetworkDlg* pScanNetworkDlg = DataCenter::GetInstance()->GetScanNetworkDlgHandle();
    if ( pScanNetworkDlg != NULL && pScanNetworkDlg->IsShown() )
    {
        pScanNetworkDlg->CloseDialog(wxID_CANCEL);
    }

    ProfileGeneralSettings wizard(this);
    if ( wizard.ShowWizard() )
    {
        ProfileList_t newProfile = wizard.GetProfile();

        newProfile.sShow = _T("1");

        if ( -1 != IsGuestExist() )
        {
            m_vProfileList.insert( m_vProfileList.end()-1, newProfile);
        }
        else
            m_vProfileList.push_back(newProfile);
        m_vCurrentUserProfileList.push_back(newProfile);
        SaveProfile();
//        CheckButtonStatus(count);
        drawListCtrl(-1);
        m_isDeselected = true;
        ModifyAdminCount();
    }
//    wizard.RunWizard(wizard.GetFirstPage());
}

void ProfileManagerDlg::Onm_miNewSelected(wxCommandEvent& event)
{
    OnButtonNewClick(event);
}

void ProfileManagerDlg::OnDoubleClick(wxListEvent& event)
{
    OnButtonEditClick(event);
}

void ProfileManagerDlg::OnButtonEditClick(wxCommandEvent& event)
{
    long item = -1;
    long nowId = m_lcProfile->GetNextItem(item, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
    if ( nowId < 0 || nowId >= (long)m_lcProfile->GetItemCount() )
        return;
    if ( !ENMUtil::IsAdministrator() && nowId < m_iAdminCount )
        return;
    if ( nowId == IsGuestExist() )
        return;
    ScanNetworkDlg* pScanNetworkDlg = DataCenter::GetInstance()->GetScanNetworkDlgHandle();
    if ( pScanNetworkDlg != NULL && pScanNetworkDlg->IsShown() )
    {
        pScanNetworkDlg->CloseDialog(wxID_CANCEL);
    }

    EditProfileDialog editDlg(this, m_vProfileList[nowId]);
    if ( editDlg.ShowModal() == wxID_OK )
    {
        m_vProfileList[nowId] = editDlg.GetProfile();
        drawListCtrl(nowId);
    }
}

void ProfileManagerDlg::Onm_miEditSelected(wxCommandEvent& event)
{
    OnButtonEditClick(event);
}

void ProfileManagerDlg::OnButtonDeleteClick(wxCommandEvent& event)
{
    if ( m_lcProfile->GetSelectedItemCount() > 1 )
    {
        wxMessageDialog dlg(this, _("Are you sure you want to delete these profiles?"), wxEmptyString, wxYES_NO | wxNO_DEFAULT | wxICON_QUESTION);
        if ( dlg.ShowModal() == wxID_YES )
        {
            long item = -1;
            m_isModified = true;
            for (int i = 0; i < m_lcProfile->GetSelectedItemCount(); i++)
            {
                long nowId = m_lcProfile->GetNextItem(item, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
                RemoveProfile(nowId - i);
                item = nowId;
            }
            m_lcProfile->DeleteAllItems();
            drawListCtrl(-1);
            m_bbUp->Enable(false);
            m_bbDown->Enable(false);
            m_buttonEdit->Enable(false);
            m_buttonDelete->Enable(false);
            m_menuFile->Enable(ID_EDITPROFILE, false);
            m_menuFile->Enable(ID_DELETEPROFILE, false);
            m_menuFile->Enable(ID_EXPORT, false);
        }
    }
    else
    {
        wxMessageDialog dlg(this, _("Are you sure you want to delete this profile?"), wxEmptyString, wxYES_NO | wxNO_DEFAULT | wxICON_QUESTION);
        if ( dlg.ShowModal() == wxID_YES )
        {
            m_isModified = true;
            long item = -1;
            wxListItem Litem;
            long nowId = m_lcProfile->GetNextItem(item, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
            RemoveProfile(nowId);
            m_lcProfile->DeleteAllItems();
            drawListCtrl(-1);
            m_bbUp->Enable(false);
            m_bbDown->Enable(false);
            m_buttonEdit->Enable(false);
            m_buttonDelete->Enable(false);
            m_menuFile->Enable(ID_EDITPROFILE, false);
            m_menuFile->Enable(ID_DELETEPROFILE, false);
            m_menuFile->Enable(ID_EXPORT, false);
        }
    }

}
void ProfileManagerDlg::Onm_miDeleteSelected(wxCommandEvent& event)
{
    OnButtonDeleteClick(event);
}

void ProfileManagerDlg::Onm_lcProfileSelected(wxCommandEvent& event)
{
    long item = -1;
    long nowId = m_lcProfile->GetNextItem(item, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
    CheckButtonStatus(nowId);
    bool bCheck = m_lcProfile->IsChecked(nowId);
    ModifyCheckFlag(nowId,bCheck);
    if ( m_lcProfile->GetSelectedItemCount() > 1 )
    {
        m_bbUp->Enable(false);
        m_bbDown->Enable(false);
        m_buttonEdit->Enable(false);
        m_buttonDelete->Enable(true);
        m_menuFile->Enable(ID_EDITPROFILE, false);
        m_menuFile->Enable(ID_DELETEPROFILE, true);
        m_menuFile->Enable(ID_EXPORT, false);
    }
    else if ( nowId == IsGuestExist() )
    {
        m_bbUp->Enable(false);
        m_bbDown->Enable(false);
        m_buttonEdit->Enable(false);
        m_buttonDelete->Enable(false);
        m_menuFile->Enable(ID_EDITPROFILE, false);
        m_menuFile->Enable(ID_DELETEPROFILE, false);
        m_menuFile->Enable(ID_EXPORT, false);
    }
    else if ( nowId == IsGuestExist()-1)
    {
        m_bbDown->Enable(false);
    }
}

void ProfileManagerDlg::ModifyCheckFlag(int pos, bool bCheck)
{
    // modify check flag of data list
    if( pos >=0 && pos < (int)m_vProfileList.size() )
    {
        if( bCheck )
        {
            m_vProfileList[pos].sShow = _T("1");
        }
        else
        {
            m_vProfileList[pos].sShow = _T("0");
        }
    }
}

void ProfileManagerDlg::ModifyAdminCount()
{
    m_isModified = true;
    if ( m_bIfAdmin )
    {
        m_iAdminCount += 1;
    }
}

void ProfileManagerDlg::CheckButtonStatus(long nowId)
{
    long totalCount = m_lcProfile->GetItemCount();

    if ( m_iAdminCount < nowId + 1 )
    {
        m_bbUp->Enable(nowId > m_iAdminCount);
        m_bbDown->Enable(nowId < totalCount - 1);
        m_buttonEdit->Enable(true);
        m_buttonDelete->Enable(true);
        m_menuFile->Enable(ID_EDITPROFILE, true);
        m_menuFile->Enable(ID_DELETEPROFILE, true);
        m_menuFile->Enable(ID_EXPORT, true);
    }
    else
    {
        if ( m_bIfAdmin )
        {
            m_bbUp->Enable(true);
            m_bbDown->Enable(true);
            m_buttonEdit->Enable(true);
            m_buttonDelete->Enable(true);
            m_menuFile->Enable(ID_EDITPROFILE, true);
            m_menuFile->Enable(ID_DELETEPROFILE, true);
            m_menuFile->Enable(ID_EXPORT, true);
        }
        else
        {
            m_bbUp->Enable(false);
            m_bbDown->Enable(false);
            m_buttonEdit->Enable(false);
            m_buttonDelete->Enable(false);
            m_menuFile->Enable(ID_EDITPROFILE, false);
            m_menuFile->Enable(ID_DELETEPROFILE, false);
            m_menuFile->Enable(ID_EXPORT, false);
        }
    }

    if ( nowId == totalCount - 1 )
    {
        m_bbDown->Enable(false);
    }
    if ( 0 == nowId )
    {
        m_bbUp->Enable(false);
    }
    if ( nowId == -1 )
    {
        return;
    }

    if ( m_vProfileList[nowId].sProfileName == DataCenter::GetInstance()->GetProfileController()->GetCurrentProfile() )
    {
        m_buttonDelete->Enable(false);
        m_menuFile->Enable(ID_DELETEPROFILE, false);
    }
}

void ProfileManagerDlg::OnProfileChanged(ProfileEvent& event)
{
    if ( IsShown() )
    {
        switch ( event.GetEventCode() )
        {
        case PEVT_PROFILE_ADD:
            break;
        case PEVT_CURRENT_PROFILE_CHANGED:
            break;
        default:
            break;
        }
    }
}

void ProfileManagerDlg::Onm_miExitSelected(wxCommandEvent& event)
{
    OnButtonCancelClick(event);
}

void ProfileManagerDlg::OnCheckboxSelectClick(wxCommandEvent& event)
{
    int  listCount = m_lcProfile->GetItemCount();
    int n = 0;
    if ( !ENMUtil::IsAdministrator() )
    {
        n = m_iAdminCount;
    }

    for(; n < listCount; n++)
    {
        if ( n == IsGuestExist() )
        {
            continue;
        }
    //    if ( wxString::FromUTF8(m_vProfileList[n].sProfileName.c_str()).Cmp(DataCenter::GetInstance()->GetProfileController()->GetCurrentProfile()) )
		if ( m_vProfileList[n].sProfileName != DataCenter::GetInstance()->GetProfileController()->GetCurrentProfile() )
        {
        bool IfChecked = m_checkboxSelect->IsChecked();
        m_lcProfile->Check(n, IfChecked);
        ModifyCheckFlag(n, IfChecked);
        }
    }
}

void ProfileManagerDlg::CheckModifyStatus()
{
    if ( !m_isModified )
        m_isModified = m_vProfileList != m_vOldProfileList;
    if ( m_isModified && Confirm() == wxID_YES)
    {
        Save();
    }
    m_vProfileList.clear();
    m_lcProfile->DeleteAllItems();
}

void ProfileManagerDlg::OnCloseClicked(wxCloseEvent& event)
{
    CheckModifyStatus();
    Show(false);
}

void ProfileManagerDlg::OnButtonCancelClick(wxCommandEvent& event)
{
    CheckModifyStatus();
    m_bbUp->Enable(false);
    m_bbDown->Enable(false);
    m_buttonEdit->Enable(false);
    m_buttonDelete->Enable(false);
    m_menuFile->Enable(ID_EDITPROFILE, false);
    m_menuFile->Enable(ID_DELETEPROFILE, false);
    m_menuFile->Enable(ID_EXPORT, false);
    Show(false);
}

void ProfileManagerDlg::Onm_miImportSelected(wxCommandEvent& event)
{
    wxFileDialog fileDlg(this, _("Select a profile"), _T(""), _T(""), _T("*.profile|*.profile"), wxFD_OPEN | wxFD_FILE_MUST_EXIST);
    if ( fileDlg.ShowModal() != wxID_OK )
        return;

    wxString fileName = fileDlg.GetPath();
    ProfileManagementControl* pController = DataCenter::GetInstance()->GetProfileController();
    ProfileList_t profile;
    if ( !pController->ReadProfile(profile, fileName) )
    {
        wxMessageDialog msgdlg(this, _("Failed to read profile."), wxEmptyString, wxOK | wxICON_EXCLAMATION);
        msgdlg.ShowModal();
        return;
    }

    ProfileList_t current = pController->GetCurrentUsedProfile();
    if ( profile.sProfileName == current.sProfileName )
    {
        wxString strMsg = _("Failed to import profile.\n");
        strMsg += current.sProfileName + _(" is currently in use.");
        wxMessageDialog msgdlg(this, strMsg, wxEmptyString, wxOK | wxICON_EXCLAMATION);
        msgdlg.ShowModal();
        return;
    }

    size_t firstPos = m_iAdminCount;
    if ( ENMUtil::IsAdministrator() )
    {
        profile.sPermission = _T("admin");
        firstPos = 0;
    }
    else
    {
        profile.sPermission = _T("user");
    }
    unsigned int overwrite = firstPos - 1;
    for ( vector<ProfileList_t>::size_type i = firstPos; i < m_vProfileList.size(); i++ )
    {
        ProfileList_t p = m_vProfileList.at(i);
        if ( p.sProfileName == profile.sProfileName )
        {
            overwrite = i;
            break;
        }
    }
    if ( overwrite != firstPos -1 && ConfirmOverwrite() != wxID_YES )
    {
        return;
    }

    if ( overwrite != firstPos -1 )
    {
        vector<ProfileList_t>::iterator iter = m_vProfileList.begin();
        for ( size_t n = 0; n < overwrite; n++ )
            iter++;
        m_vProfileList.erase(iter);
    }
    else if ( ENMUtil::IsAdministrator() )
    {
        m_iAdminCount ++;
    }
    if ( -1 != IsGuestExist() )
    {
        m_vProfileList.insert( m_vProfileList.end()-1, profile);
    }
    else
        m_vProfileList.push_back(profile);

    SaveProfile();

    wxMessageDialog msg(this, _("Import successful."), wxEmptyString, wxOK | wxICON_INFORMATION);
    msg.ShowModal();

    m_isModified = true;
}

void ProfileManagerDlg::Onm_miExportSelected(wxCommandEvent& event)
{
    wxLogNull logNo;
    long selected = m_lcProfile->GetNextItem(-1, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
    if ( selected < 0 || selected >= m_lcProfile->GetItemCount() )
        return;
    ProfileList_t profile = m_vProfileList.at(selected);
    wxString strSelectedProfileName = profile.sProfileName + _T(".profile");
    wxString strSrcProfile;
    if ( selected < m_iAdminCount )
    {
        strSrcProfile = ProfileManagementControl::SYSTEM_PROFOLE_DIR + wxFileName::GetPathSeparator() + strSelectedProfileName;
    }
    else
    {
        strSrcProfile = ProfileManagementControl::USER_PROFILE_DIR + wxFileName::GetPathSeparator() + strSelectedProfileName;
    }
    wxFile file;
    if ( !wxFile::Exists(strSrcProfile) || !file.Open(strSrcProfile) )
    {
        wxMessageDialog msgdlg(this, _("Cannot open selected file."), _("Message"), wxOK | wxICON_EXCLAMATION);
        msgdlg.ShowModal();
        return;
    }
    wxFileOffset length = file.Length();
    file.Close();

//    wxDirDialog dialog(this, _("Select a path"), _T(""), wxDD_DIR_MUST_EXIST);
//    if ( dialog.ShowModal() != wxID_OK )
//        return;

//    wxString strDstPath = dialog.GetPath();
    wxString strDstPath = ::wxDirSelector(_("Select a path"), _T(""), 0, wxDefaultPosition, this);
    if ( strDstPath.IsEmpty() )
        return;

    wxString strDstProfile = strDstPath + wxFileName::GetPathSeparator() + strSelectedProfileName;
    if ( wxFileExists(strDstProfile) && ConfirmOverwrite() != wxID_YES )
    {
        return;
    }

    wxLongLong diskFree = 0;
    if ( ::wxGetDiskSpace(strDstPath, NULL, &diskFree) && (diskFree <= length) )
    {
        wxMessageDialog msgdlg(this, _("Not enough disk space available."), wxEmptyString, wxOK | wxICON_EXCLAMATION);
        msgdlg.ShowModal();
        return;
    }
    ProfileManagementControl* pController = DataCenter::GetInstance()->GetProfileController();
    if ( pController->ExportProfile(profile, strDstPath) )
    {
        SaveProfile();

        wxMessageDialog msg(this, _("Export successful."), wxEmptyString, wxOK | wxICON_INFORMATION);
        msg.ShowModal();
    }
    else
    {
        wxMessageDialog msgdlg(this, _("Failed to export profile."), wxEmptyString, wxOK | wxICON_EXCLAMATION);
        msgdlg.ShowModal();
    }
}

void ProfileManagerDlg::OnButtonOKClick(wxCommandEvent& event)
{
    Save();
    Show(false);
}

void ProfileManagerDlg::Onm_lcProfileDeselected(wxCommandEvent& event)
{
    if( m_isDeselected )
    {
        m_bbUp->Enable(false);
        m_bbDown->Enable(false);
        m_buttonEdit->Enable(false);
        m_buttonDelete->Enable(false);
        m_menuFile->Enable(ID_EDITPROFILE, false);
        m_menuFile->Enable(ID_DELETEPROFILE, false);
        m_menuFile->Enable(ID_EXPORT, false);
    }
}

void ProfileManagerDlg::SaveProfile()
{
    m_lcProfile->DeleteAllItems();
    drawListCtrl(-1);
    m_bbUp->Enable(false);
    m_bbDown->Enable(false);
    m_buttonEdit->Enable(false);
    m_buttonDelete->Enable(false);
    m_menuFile->Enable(ID_EDITPROFILE, false);
    m_menuFile->Enable(ID_DELETEPROFILE, false);
    m_menuFile->Enable(ID_EXPORT, false);
}

bool ProfileManagerDlg::ChangePriority(int firstPriority, int secondPriority)
{
    bool bRet = true;

    if ( secondPriority > (int)m_vProfileList.size() )
    {
        bRet = false;
    }
    else
    {
        ProfileList_t vTemp = m_vProfileList[firstPriority - 1];
        m_vProfileList[firstPriority - 1] = m_vProfileList[secondPriority - 1];
        m_vProfileList[secondPriority - 1] = vTemp;
    }
    return bRet;
}

bool ProfileManagerDlg::RemoveProfile(int index)
{
    bool bRet = true;
    if ( m_vProfileList.size() <= 0 )
    {
        bRet = false;
    }
    else
    {
        vector<ProfileList_t>::iterator iter = m_vProfileList.begin();
        while ( index > 0 )
        {
            iter++;
            index--;
        }
        m_vProfileList.erase(iter);
    }
    return bRet;
}

void ProfileManagerDlg::MenuHelpContent(wxCommandEvent& event)
{
//	ENMUtil::ShowContextHelp(this, _T("ProfileManager"));

    #if defined(__UNIX__)
    if(wxFileExists(wxT("/usr/share/doc/enhanced-easy-network/Intel_CMPC2.0_Enhanced_Easy_Network_User_Manual_Ubuntu.pdf.gz")))
    {
        wxString cmd = wxT("evince /usr/share/doc/enhanced-easy-network/Intel_CMPC2.0_Enhanced_Easy_Network_User_Manual_Ubuntu.pdf.gz --page-label=1");
        wxExecute(cmd);
    }
    #endif
}
int ProfileManagerDlg::IsGuestExist()
{
    wxString GuestProfileName(wxT("Guest"));
    wxString ProfileName;
    int  vecSize = m_vProfileList.size();
    for ( int n = 0; n < vecSize; n++ )
    {
        ProfileName = m_vProfileList[n].sProfileName;
        if(GuestProfileName.Cmp(ProfileName) == 0)
        {
            return n;
        }
    }
    return -1;
}

void ProfileManagerDlg::RefreshProfile()
{
    m_lcProfile->DeleteAllItems();

    m_bIfAdmin = ENMUtil::IsAdministrator();
    DataCenter::GetInstance()->GetProfileController()->GetUserProfiles(m_vProfileList, m_bIfAdmin, true);
    m_vOldProfileList = m_vProfileList;
    m_iAdminCount = DataCenter::GetInstance()->GetProfileController()->GetAdminProfileCount();

    drawListCtrl(-1);
}
void ProfileManagerDlg::ModifyProfileVersionInfo()
{
    ProfileVersion profileversion;
    wxArrayString VersionInfo;
    bool bReturn = profileversion.GetProfileVersion(VersionInfo);
    vector<ProfileList_t> ModifiedProfileList;
    wxArrayString FinalModified;
    if ( bReturn )
    {
        for ( size_t i = 2; i < VersionInfo.Count(); i++ )
        {
            wxString ProfileName = VersionInfo.Item(i);
            int iIterOld = SearchProfileList(m_vOldProfileList, ProfileName);
            int iIterNew = SearchProfileList(m_vProfileList, ProfileName);
            if ( iIterOld != -1 && iIterNew != -1 )
            {
                bool bChange = m_vOldProfileList[iIterOld] != m_vProfileList[iIterNew];
                if ( bChange )
                {
                    FinalModified.Add(ProfileName);
                }
            }
        }
        if ( 0 != FinalModified.Count() )
            profileversion.ModifyProfileVersion(FinalModified);
    }
}

int ProfileManagerDlg::SearchProfileList(const vector<ProfileList_t> &ProfileList, const wxString& ProfileName)
{
    for ( vector<ProfileList_t>::size_type iter = 0; iter < ProfileList.size(); iter++ )
    {
        wxString wxProfileName = ProfileList[iter].sProfileName + wxT(".profile");
        if ( 0 == wxProfileName.Cmp(ProfileName) )
        {
            return iter;
        }
    }
    return -1;
}

