/* e-book-backend-android.c - Android contact backend.
 *
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 * Authors:
 *      Renato Araujo Oliveira Filho <renato@canonical.com>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <time.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/time.h>

#include <glib/gstdio.h>
#include <gio/gio.h>

#include <libebook/e-contact.h>
#include <libedata-book/e-book-backend-sexp.h>
#include <libedata-book/e-data-book.h>
#include <libedata-book/e-data-book-view.h>
#include <libedata-book/e-book-backend-cache.h>

#include <sqlite3.h>

#include "e-book-backend-android.h"

#define UBUNTU_ANDROID_SERVICE_NAME             "com.canonical.Android"
#define UBUNTU_ANDROID_OBJECT_PATH              "/com/canonical/android/contacts/Contacts"
#define UBUNTU_ANDROID_INTERFACE_NAME           "com.canonical.android.contacts.Contacts"
#define UBUNTU_ANDROID_CONTACT_CHANGED_SIGNAL   "contactsChanged"
#define UBUNTU_ANDROID_CONTACT_REMOVED_SIGNAL   "contactsRemoved"

#define SYNC_TIMEOUT                            5

#define EDB_ERROR(_code) e_data_book_create_error (E_DATA_BOOK_STATUS_ ## _code, NULL)
#define EDB_ERROR_MSG(_code, _msg) e_data_book_create_error (E_DATA_BOOK_STATUS_ ## _code, _msg)

G_DEFINE_TYPE (EBookBackendAndroid, e_book_backend_android, E_TYPE_BOOK_BACKEND)

struct _EBookBackendAndroidPrivate {
    sqlite3 *cache;
    GList *bookviews;
    GDBusProxy *proxy;
    guint watch_id;
    GMutex *sync_lock;
    GCancellable *sync_cancel;
    guint pending_sync;
    GRecMutex *db_lock;
};


typedef struct _EBookBackendAndroidSyncData  EBookBackendAndroidSyncData;
struct  _EBookBackendAndroidSyncData {
    EBookBackendAndroid *self;
    GList *cached_contacts;
    int current_contact_index;
    gchar *token;
    gboolean notify;
};

static void     e_book_backend_android_ready_cb     (GObject *source_object,
                                                     GAsyncResult *res,
                                                     gpointer user_data);
static GError*  e_book_backend_android_sync_contact (EBookBackend *backend,
                                                     gboolean notify);
static void     e_book_backend_android_get_contacts (EBookBackendAndroidSyncData *data);

static sqlite3 *
e_book_sqlite_cache_new (EBookBackendAndroidPrivate *priv, const gchar *filename)
{
    sqlite3 *handle = NULL;
    g_rec_mutex_lock(priv->db_lock);
    int ret = sqlite3_open(filename, &handle);
    if (handle && ret == SQLITE_OK) {
        sqlite3_exec(handle,"CREATE TABLE IF NOT EXISTS contacts (id TEXT PRIMARY KEY,vcard TEXT NOT NULL)",0,0,0);
    }
    g_rec_mutex_unlock(priv->db_lock);
    return handle;
}

static gboolean 
e_book_sqlite_cache_is_populated (EBookBackendAndroidPrivate *priv)
{
    int count = 0;
    sqlite3_stmt *stmt;
    g_rec_mutex_lock(priv->db_lock);
    if (!priv->cache) {
        goto out;
    }

    if (sqlite3_prepare_v2(priv->cache, "SELECT count(*) FROM contacts", -1, &stmt, NULL) == SQLITE_OK) {
         sqlite3_step(stmt);
         count = sqlite3_column_int(stmt, 0);
         sqlite3_finalize(stmt);
    }

out:
    g_rec_mutex_unlock(priv->db_lock);
    return count != 0;
}

static gboolean
e_book_sqlite_cache_add_contact (EBookBackendAndroidPrivate *priv, const gchar *id, const gchar *vcard)
{
    int ret = SQLITE_ERROR;
    g_rec_mutex_lock(priv->db_lock);
    if (!priv->cache) {
        goto out;
    }

    char* queryDelete = sqlite3_mprintf("delete from contacts where id = '%q';", id);
    char* queryInsert = sqlite3_mprintf("insert into contacts values ('%q', '%q');", id, vcard);
    sqlite3_exec(priv->cache, queryDelete, 0, 0, 0);
    sqlite3_free(queryDelete);
    ret = sqlite3_exec(priv->cache, queryInsert, 0, 0, 0);
    sqlite3_free(queryInsert);
out:
    g_rec_mutex_unlock(priv->db_lock);
    return (ret == SQLITE_OK);
}

static gboolean
e_book_sqlite_cache_remove_contact (EBookBackendAndroidPrivate *priv, const gchar *uid)
{
    int ret = SQLITE_ERROR;
    g_rec_mutex_lock(priv->db_lock);
    if (!priv->cache) {
        goto out;
    }

    char* queryDelete = sqlite3_mprintf("delete from contacts where id = '%q';", uid);
    ret = sqlite3_exec(priv->cache, queryDelete, 0, 0, 0) ;
    sqlite3_free(queryDelete);
out:
    g_rec_mutex_unlock(priv->db_lock);
    return (ret == SQLITE_OK);
}

static GList*
e_book_sqlite_cache_get_contacts (EBookBackendAndroidPrivate *priv, const gchar *query)
{
    sqlite3_stmt *stmt;
    GList *contacts = NULL;
    g_rec_mutex_lock(priv->db_lock);

    if (!priv->cache) {
        goto out;
    }

    if (sqlite3_prepare_v2(priv->cache, "SELECT vcard FROM contacts", -1, &stmt, NULL) != SQLITE_OK) {
        goto out;
    }
    while (sqlite3_step(stmt) == SQLITE_ROW) {
         EContact *contact = e_contact_new_from_vcard (sqlite3_column_text(stmt, 0));
         contacts = g_list_append(contacts, contact);
    }
    sqlite3_finalize(stmt);
out:
    g_rec_mutex_unlock(priv->db_lock);
    return contacts;
}

static EContact*
e_book_sqlite_cache_get_contact (EBookBackendAndroidPrivate *priv, const gchar *id)
{
    int count = 0;
    sqlite3_stmt *stmt;
    EContact *contact = NULL;

    g_rec_mutex_lock(priv->db_lock);

    if (!priv->cache) {
        goto out;
    }
    
    char* querySelect = sqlite3_mprintf("SELECT vcard FROM contacts where id = '%q'", id);

    if (sqlite3_prepare_v2(priv->cache, querySelect, -1, &stmt, NULL) == SQLITE_OK) {
         sqlite3_step(stmt);
         contact = e_contact_new_from_vcard (sqlite3_column_text(stmt, 0));
         sqlite3_finalize(stmt);
    }
    sqlite3_free(querySelect);

out:    
    g_rec_mutex_unlock(priv->db_lock);
    return contact;
}

static GPtrArray *
e_book_sqlite_cache_search (EBookBackendAndroidPrivate *priv, const gchar *query)
{
    return NULL;
}

static void
e_book_backend_android_set_fullname (EContact *e_contact)
{
    const gchar *fullname = e_contact_get_const (e_contact, E_CONTACT_FULL_NAME);
    if (!fullname) {
        EContactName *name = e_contact_get (e_contact, E_CONTACT_NAME);
        if (name) {
            gchar *formatted_name = e_contact_name_to_string(name);
            e_contact_set(e_contact, E_CONTACT_FULL_NAME, formatted_name);
            g_free(formatted_name);
            g_free(name);
        }
    }
}

static void
e_book_backend_android_dbus_service_appear (GDBusConnection *connection,
                                            const gchar *name,
                                            const gchar *name_owner,
                                            gpointer user_data)
{
    g_debug ("FUNCTION: %s\n", __FUNCTION__);
    g_dbus_proxy_new_for_bus (G_BUS_TYPE_SESSION,
                              G_DBUS_PROXY_FLAGS_NONE,
                              NULL,
                              UBUNTU_ANDROID_SERVICE_NAME,
                              UBUNTU_ANDROID_OBJECT_PATH,
                              UBUNTU_ANDROID_INTERFACE_NAME,
                              NULL,
                              e_book_backend_android_ready_cb,
                              user_data);
}

static void
e_book_backend_android_dbus_service_disapear (GDBusConnection *connection,
                                              const gchar *name,
                                              gpointer user_data)
{
    g_debug ("FUNCTION: %s\n", __FUNCTION__);
    EBookBackendAndroid *b = E_BOOK_BACKEND_ANDROID (user_data);

    e_book_backend_notify_error (E_BOOK_BACKEND (user_data), "Discconected from Android");
    g_clear_object (&b->priv->proxy);
}

static gboolean
e_book_backend_android_start_pending_sync (EBookBackend *backend)
{
    g_debug ("FUNCTION: %s\n", __FUNCTION__);
    EBookBackendAndroid *b = E_BOOK_BACKEND_ANDROID (backend);
    EBookBackendAndroidPrivate *priv = b->priv;

    priv->pending_sync = 0;
    GError *error = e_book_backend_android_sync_contact (backend, TRUE);
    if (error) {
        g_warning ("Fail to sync contacts: %s", error->message);
        g_error_free (error);
    }

    return FALSE;
}

static void
e_book_backend_android_append_sync (EBookBackendAndroid *self)
{
    g_debug ("FUNCTION: %s\n", __FUNCTION__);
    EBookBackendAndroidPrivate *priv = self->priv;
    if (priv->pending_sync != 0) {
        g_source_remove (priv->pending_sync);
    }
    priv->pending_sync = g_timeout_add_seconds (SYNC_TIMEOUT,
                                                (GSourceFunc) e_book_backend_android_start_pending_sync,
                                                self);
}

static void
e_book_backend_android_on_dbus_signal (GDBusProxy *proxy,
                                       gchar *sender_name,
                                       gchar *signal_name,
                                       GVariant *parameters,
                                       gpointer user_data)
{
    g_debug ("FUNCTION: %s:%s\n", __FUNCTION__, signal_name);
    EBookBackendAndroid *b = E_BOOK_BACKEND_ANDROID (user_data);
    EBookBackendAndroidPrivate *priv = b->priv;
    gchar *id = NULL;

    if (g_str_equal(signal_name, UBUNTU_ANDROID_CONTACT_CHANGED_SIGNAL)) {
        GVariantIter *iter;
        g_variant_get (parameters, "(ass)", &iter, &id);
        gchar *vCard;
        while (g_variant_iter_loop (iter, "s", &vCard)) {
            EContact *contact = e_contact_new_from_vcard (vCard);
            if (contact) {
                // add/update old contact
                const gchar *id = e_contact_get_const (contact, E_CONTACT_UID);
                e_book_sqlite_cache_add_contact (priv, id, vCard);
                GList *book_iter;

                for (book_iter = priv->bookviews; book_iter; book_iter = book_iter->next) {
                    e_data_book_view_notify_update (E_DATA_BOOK_VIEW (book_iter->data), contact);
                }
            }
        }
        g_variant_iter_free (iter);
    } else if (g_str_equal(signal_name, UBUNTU_ANDROID_CONTACT_REMOVED_SIGNAL)) {
        GVariantIter *iter;
        gchar *contact_id;
        g_variant_get (parameters, "(ass)", &iter, &id);
        while (g_variant_iter_loop (iter, "s", &contact_id)) {
            GList *book_iter;
            for (book_iter = priv->bookviews; book_iter; book_iter = book_iter->next) {
                e_data_book_view_notify_remove (E_DATA_BOOK_VIEW (book_iter->data), contact_id);
            }
            e_book_sqlite_cache_remove_contact (priv, contact_id);
        }
        g_variant_iter_free (iter);
    }

    if (id) {
        g_dbus_proxy_call_sync (priv->proxy, "signalProcessed",
                                         g_variant_new ("(s)", id),
                                         G_DBUS_CALL_FLAGS_NONE,
                                         -1,
                                         NULL,
                                         NULL);
        
    }
}

static void
e_book_backend_android_ready_cb (GObject *source_object,
                                 GAsyncResult *res,
                                 gpointer user_data)
{
    g_debug ("FUNCTION: %s\n", __FUNCTION__);
    EBookBackendAndroid *b = E_BOOK_BACKEND_ANDROID (user_data);
    GError *error = NULL;

    g_return_if_fail(b != NULL);

    GDBusProxy *proxy = g_dbus_proxy_new_for_bus_finish (res, &error);
    if (error != NULL) {
        e_book_backend_notify_error (E_BOOK_BACKEND (user_data), error->message);
        g_error_free (error);
        return;
    }

    g_signal_connect (proxy, "g-signal",
                      G_CALLBACK (e_book_backend_android_on_dbus_signal), b);

    b->priv->proxy = proxy;
    e_book_backend_notify_online (E_BOOK_BACKEND (user_data), TRUE);
    e_book_backend_notify_readonly (E_BOOK_BACKEND (user_data), FALSE);

    if (!b->priv->cache || !e_book_sqlite_cache_is_populated(b->priv)) {
        error = e_book_backend_android_sync_contact (E_BOOK_BACKEND (user_data), TRUE);
        if (error) {
            e_book_backend_notify_error (E_BOOK_BACKEND (user_data), error->message);
            g_error_free (error);
            return;
        }
    } else {
        // if we have cache already, it means we need to start the observer
        g_dbus_proxy_call_sync (b->priv->proxy, "startContactObserver",
                                     NULL,
                                     G_DBUS_CALL_FLAGS_NONE,
                                     -1,
                                     NULL,
                                     &error);
        if (error) {
            e_book_backend_notify_error (E_BOOK_BACKEND (user_data), error->message);
            g_error_free (error);
            return;
        }
    }
}

static void
e_book_backend_android_sync_contact_finish (EBookBackendAndroidSyncData *data)
{
    g_debug ("FUNCTION: %s\n", __FUNCTION__);
    EBookBackendAndroidPrivate *priv = data->self->priv;

    g_list_free_full (data->cached_contacts, g_free);
    g_free (data->token);
    g_free (data);
    g_mutex_unlock (priv->sync_lock);
}

void cache_iterator(gchar *data, EBookBackendAndroidPrivate* priv) 
{
    g_debug ("FUNCTION: %s\n", __FUNCTION__);
    EContact *contact = e_contact_new_from_vcard (data);
    if (contact) {
        const gchar *id = e_contact_get_const (contact, E_CONTACT_UID);
        e_book_sqlite_cache_add_contact (priv, id, data);
        g_object_unref (contact);
    }
}

static void
e_book_backend_android_get_contacts_finish (GDBusProxy *proxy,
                                            GAsyncResult *res,
                                            EBookBackendAndroidSyncData *data)
{
    g_debug ("FUNCTION: %s\n", __FUNCTION__);
    EBookBackendAndroidPrivate *priv = data->self->priv;
    GError *error = NULL;
    GVariant *result = g_dbus_proxy_call_finish (proxy, res, &error);
    if (error) {
        g_warning ("Sync contacts error: %s", error->message);
        e_book_backend_android_sync_contact_finish (data);
        return;
    }

    if (g_cancellable_is_cancelled (priv->sync_cancel)) {
        e_book_backend_android_sync_contact_finish (data);
    }

    GVariantIter *iter;
    g_variant_get (result, "(as)", &iter);

    if (g_variant_iter_n_children (iter) == 0) {
        // End of pages
        g_list_foreach(data->cached_contacts, (GFunc)cache_iterator, (gpointer)priv);

        if (g_list_length(data->cached_contacts) == 0) {
            g_dbus_proxy_call_sync (priv->proxy, "startContactObserver",
                                     NULL,
                                     G_DBUS_CALL_FLAGS_NONE,
                                     -1,
                                     NULL,
                                     NULL);
        }
        e_book_backend_android_sync_contact_finish (data);
        return;
    }

    gchar *vCard;
    while (g_variant_iter_loop (iter, "s", &vCard)) {
        if ((vCard != NULL) && (strlen (vCard) > 0)){
            EContact *contact = e_contact_new_from_vcard (vCard);
            if (contact) {
                data->cached_contacts = g_list_append (data->cached_contacts,
                                         g_strdup (vCard));
                if (data->notify) {
                    GList *book_iter;
                    for (book_iter = priv->bookviews; book_iter; book_iter = book_iter->next) {
                        e_data_book_view_notify_update (E_DATA_BOOK_VIEW (book_iter->data), contact);
                    }
                }
            }
        }
        data->current_contact_index++;
    }

    g_variant_iter_free (iter);
    g_variant_unref (result);

    if (g_cancellable_is_cancelled (priv->sync_cancel)) {
        e_book_backend_android_sync_contact_finish (data);
    } else {
        e_book_backend_android_get_contacts (data);
    }
}

static void
e_book_backend_android_get_contacts (EBookBackendAndroidSyncData *data)
{
    g_debug ("FUNCTION: %s\n", __FUNCTION__);
    EBookBackendAndroidPrivate *priv = data->self->priv;

    // Request vcards
    g_dbus_proxy_call (priv->proxy,
                       "listContactsChangesAsyncGetPage",
                       g_variant_new ("(s)", data->token),
                       G_DBUS_CALL_FLAGS_NONE,
                       -1,
                       priv->sync_cancel,
                       (GAsyncReadyCallback) e_book_backend_android_get_contacts_finish,
                       data);
}

static void
e_book_backend_android_cancel_sync (EBookBackendAndroid *b)
{
    EBookBackendAndroidPrivate *priv = b->priv;
    if (!g_mutex_trylock (priv->sync_lock)) {
        g_cancellable_cancel (priv->sync_cancel);

        // Wait for dbus functions finish
        while (!g_mutex_trylock (priv->sync_lock)) {
            g_main_context_iteration (NULL, TRUE);
        }
    }
    g_cancellable_reset (priv->sync_cancel);
}

static GError*
e_book_backend_android_sync_contact (EBookBackend *backend, gboolean notify)
{
    g_debug ("FUNCTION: %s\n", __FUNCTION__);
    EBookBackendAndroid *b = E_BOOK_BACKEND_ANDROID (backend);
    EBookBackendAndroidPrivate *priv = b->priv;

    // Cancel current sync and lock mutex
    e_book_backend_android_cancel_sync (b);

    EBookBackendAndroidSyncData *data = g_new0(EBookBackendAndroidSyncData, 1);
    data->self = b;
    data->notify = notify;

    if (!priv->cache) {
        const gchar *cache_dir = e_book_backend_get_cache_dir (E_BOOK_BACKEND(backend));
        gchar *filename = g_build_filename (cache_dir, "android-cache.db", NULL);
        g_mkdir_with_parents(cache_dir, 0700);
        priv->cache = e_book_sqlite_cache_new (priv, filename);
        g_free(filename);
    }

    if (e_book_sqlite_cache_is_populated(priv)) {
        g_dbus_proxy_call_sync (b->priv->proxy, "startContactObserver",
                                     NULL,
                                     G_DBUS_CALL_FLAGS_NONE,
                                     -1,
                                     NULL,
                                     NULL);
        return NULL;
    }

    // Prepare current cached contacts id
    GVariantBuilder *cached_ids_builder;
    cached_ids_builder = g_variant_builder_new (G_VARIANT_TYPE ("as"));

    // Request token
    GError *error = NULL;
    GVariant *result = g_dbus_proxy_call_sync (priv->proxy,
                                               "listContactsChangesAsync",
                                               g_variant_new ("(as)", cached_ids_builder),
                                               G_DBUS_CALL_FLAGS_NONE,
                                               -1,
                                               priv->sync_cancel,
                                               &error);

    g_variant_builder_unref (cached_ids_builder);
    if (!result) {
        g_warning ("Fail to call listContactsChangesAsync: %s", error->message);
        return error;
    }

    g_variant_get (result, "(s)", &(data->token));
    e_book_backend_android_get_contacts (data);
    return NULL;
}

static void
e_book_backend_android_create_contact (EBookBackend *backend,
                                       EDataBook *book,
                                       guint32 opid,
                                       GCancellable *cancellable,
                                       const gchar *vcard)
{
    g_debug ("FUNCTION: %s\n", __FUNCTION__);
    GError *error = NULL;
    EContact *e_contact = NULL;

    EBookBackendAndroid *b = E_BOOK_BACKEND_ANDROID (backend);
    EBookBackendAndroidPrivate *priv = b->priv;
    GVariant *result;

    if (priv->proxy == NULL) {
        GError *offlineError = e_data_book_create_error (E_DATA_BOOK_STATUS_OTHER_ERROR, "Android service not found");
        e_data_book_respond_create (book, opid, offlineError, NULL);
        g_error_free (offlineError);
        return;
    }
    
    e_contact = e_contact_new_from_vcard (vcard);
    if (!e_contact) {
        GError *vcardError = e_data_book_create_error (E_DATA_BOOK_STATUS_OTHER_ERROR, "Invalid vcard");
        e_data_book_respond_create (book, opid, vcardError, NULL);
        return;
    }

    e_book_backend_android_set_fullname(e_contact);

    gchar *new_vcard = e_vcard_to_string (E_VCARD(e_contact), EVC_FORMAT_VCARD_30);
    result = g_dbus_proxy_call_sync (priv->proxy,
                                     "createContact",
                                      g_variant_new ("(s)", new_vcard),
                                      G_DBUS_CALL_FLAGS_NONE,
                                      -1,
                                     cancellable,
                                     &error);


    if (error) {
        g_warning ("Fail to call createContact");
    } else {
        char *contact_key = NULL;
        g_variant_get (result, "(s)", &contact_key);
        if (contact_key) {
            if (e_contact) {
                e_contact_set (e_contact, E_CONTACT_UID, contact_key);
                e_book_sqlite_cache_add_contact (priv, contact_key, e_vcard_to_string (E_VCARD (e_contact), EVC_FORMAT_VCARD_30));
            }
            g_free (contact_key);
        } else {
            error = e_data_book_create_error (E_DATA_BOOK_STATUS_OTHER_ERROR, "Invalid reply from the server");
        }
        g_variant_unref (result);
    }
    g_free (new_vcard);
    e_data_book_respond_create (book, opid, error, e_contact);
}

static void
e_book_backend_android_remove_contacts (EBookBackend *backend,
                                        EDataBook *book,
                                        guint32 opid,
                                        GCancellable *cancellable,
                                        const GSList *id_list)
{
    g_debug ("FUNCTION: %s\n", __FUNCTION__);
    GError *error = NULL;
    EBookBackendAndroid *b = E_BOOK_BACKEND_ANDROID (backend);
    EBookBackendAndroidPrivate *priv = b->priv;

    if (priv->proxy == NULL) {
        GError *offlineError = e_data_book_create_error (E_DATA_BOOK_STATUS_OTHER_ERROR, "Android service not found");
        e_data_book_respond_remove_contacts (book, opid, offlineError, NULL);
        return;
    }

    const GSList *node;
    GVariantBuilder *builder = g_variant_builder_new (G_VARIANT_TYPE ("as"));
    for (node = id_list; node; node = node->next) {
        g_variant_builder_add (builder, "s", node->data);
    }

    GVariant *result = g_dbus_proxy_call_sync (priv->proxy,
                                               "deleteContacts",
                                               g_variant_new ("(as)", builder),
                                               G_DBUS_CALL_FLAGS_NONE,
                                               -1,
                                               cancellable,
                                               &error);
    const GSList *id_list_removed = NULL;

    if (!result) {
        g_warning ("Fail to call deleteContacts");
    } else {
        gint32 removed_count = 0;
        g_variant_get (result, "(i)", &removed_count);
        // If the number of removed elements is the same as id_list size then all contacts were removed successfully,
        // otherwise we need to wait for contact update signal to make sure which contact was removed.
        if (removed_count == g_slist_length ((GSList*) id_list)) {
            const GSList *node = id_list;
            for (node = id_list; node; node = node->next) {
                e_book_sqlite_cache_remove_contact (priv, node->data);
            }
            id_list_removed = id_list;
        }
        g_variant_unref (result);
    }

    g_variant_builder_unref (builder);

    e_data_book_respond_remove_contacts (book, opid, error, id_list_removed);
}

static void
e_book_backend_android_modify_contact (EBookBackend *backend,
                                       EDataBook *book,
                                       guint32 opid,
                                       GCancellable *cancellable,
                                       const gchar *vcard)
{
    g_debug ("FUNCTION: %s\n", __FUNCTION__);
    GError *error = NULL;
    EContact *e_contact = NULL;
    EBookBackendAndroid *b = E_BOOK_BACKEND_ANDROID (backend);
    EBookBackendAndroidPrivate *priv = b->priv;

    if (priv->proxy == NULL) {
        GError *offlineError = e_data_book_create_error (E_DATA_BOOK_STATUS_OTHER_ERROR, "Android service not found");
        e_data_book_respond_modify (book, opid, offlineError, NULL);
        return;
    }

    // Make sure this is valid evolution contact
    e_contact = e_contact_new_from_vcard (vcard);
    if (!e_contact) {
        GError *vcardError = e_data_book_create_error (E_DATA_BOOK_STATUS_OTHER_ERROR, "Invalid vcard");
        e_data_book_respond_modify (book, opid, vcardError, NULL);
        return;
    }

    const gchar *local_id = e_contact_get_const (e_contact, E_CONTACT_UID);

    e_book_backend_android_set_fullname(e_contact);

    // Get Evolution vcard information (its safe)
    gchar *new_vcard = e_vcard_to_string (E_VCARD(e_contact), EVC_FORMAT_VCARD_30);
    GVariant *result = g_dbus_proxy_call_sync (priv->proxy,
                                               "updateContact",
                                               g_variant_new ("(s)", new_vcard),
                                               G_DBUS_CALL_FLAGS_NONE,
                                               -1,
                                               cancellable,
                                               &error);
    if (error) {
        g_warning ("Fail to call updateContact");
    } else {
        g_variant_unref (result);
    }

    g_free (new_vcard);

    e_contact_set (e_contact, E_CONTACT_UID, local_id);
    e_data_book_respond_modify (book, opid, error, e_contact);
}

static void
e_book_backend_android_get_contact (EBookBackend *backend,
                                    EDataBook *book,
                                    guint32 opid,
                                    GCancellable *cancellable,
                                    const gchar *id)
{
    g_debug ("FUNCTION: %s\n", __FUNCTION__);
    EBookBackendAndroid *b = E_BOOK_BACKEND_ANDROID (backend);

    EContact *contact = e_book_sqlite_cache_get_contact (b->priv, id);
    if (contact) {
        gchar *vcard = e_vcard_to_string (E_VCARD (contact), EVC_FORMAT_VCARD_30);
        e_data_book_respond_get_contact (book, opid, NULL, vcard);
        g_free(vcard);
        g_object_unref (contact);
    } else {
        e_data_book_respond_get_contact (book, opid, EDB_ERROR (NOT_SUPPORTED), NULL);
    }
}

static void
e_book_backend_android_get_contact_list (EBookBackend *backend,
                                         EDataBook *book,
                                         guint32 opid,
                                         GCancellable *cancellable,
                                         const gchar *query)
{
    g_debug ("FUNCTION: %s\n", __FUNCTION__);

    EBookBackendAndroid *b = E_BOOK_BACKEND_ANDROID (backend);
    GList *contact_list = e_book_sqlite_cache_get_contacts (b->priv, query);
    GList *iter;

    GSList *vcard_list = NULL;

    for(iter = contact_list; iter; iter = iter->next) {
        EContact *contact = E_CONTACT(iter->data);
        gchar *vcard = e_vcard_to_string (E_VCARD (contact), EVC_FORMAT_VCARD_30);

        if (vcard) {
            vcard_list = g_slist_append (vcard_list, vcard);
        }

        g_object_unref (contact);
    }

    e_data_book_respond_get_contact_list (book, opid, NULL, vcard_list);

    g_slist_free_full (vcard_list, (GDestroyNotify) g_free);
    g_list_free (contact_list);
}

static void
e_book_backend_android_get_contact_list_uids (EBookBackend *backend,
                                              EDataBook *book,
                                              guint32 opid,
                                              GCancellable *cancellable,
                                              const gchar *query)
{
    g_debug ("FUNCTION: %s\n", __FUNCTION__);

    EBookBackendAndroid *b = E_BOOK_BACKEND_ANDROID (backend);
    GPtrArray *contact_list = e_book_sqlite_cache_search (b->priv, query);
    GSList *uids_list = NULL;

    if (contact_list) {
        int size = contact_list->len;
        int i = 0;
        for(; i < size; i++) {
            uids_list = g_slist_append (uids_list, g_ptr_array_index (contact_list, i));
        }

        g_ptr_array_free (contact_list, TRUE);
    }

    e_data_book_respond_get_contact_list_uids (book, opid, NULL, uids_list);
    g_slist_foreach (uids_list, (GFunc) g_free, NULL);
}

static void
e_book_backend_android_start_book_view (EBookBackend *backend,
                                        EDataBookView *book_view)
{
    g_debug ("FUNCTION: %s\n", __FUNCTION__);

    EBookBackendAndroid *b = E_BOOK_BACKEND_ANDROID (backend);
    EBookBackendAndroidPrivate *priv = b->priv;
    GError *error = NULL;

    if (priv->proxy) {
        g_dbus_proxy_call_sync (priv->proxy, "stopContactObserver",
                                 NULL,
                                 G_DBUS_CALL_FLAGS_NONE,
                                 -1,
                                 NULL,
                                 &error);
        if (error) {
            e_book_backend_notify_error (E_BOOK_BACKEND (backend), error->message);
            g_error_free (error);
            return;
        }
    }

    priv->bookviews = g_list_append (priv->bookviews, book_view);
    e_data_book_view_ref (book_view);

    e_data_book_view_notify_progress (book_view, -1, "Loading...");

    if (!priv->cache) {
        // load cache if we have one
        const gchar *cache_dir = e_book_backend_get_cache_dir (E_BOOK_BACKEND(backend));
        gchar *filename = g_build_filename (cache_dir, "android-cache.db", NULL);
        g_mkdir_with_parents(cache_dir, 0700);
        priv->cache = e_book_sqlite_cache_new (priv, filename);
        g_free(filename);
        if (e_book_sqlite_cache_is_populated(priv)) {
            GList *cached_contacts = e_book_sqlite_cache_get_contacts (priv,
                                                                    "(contains \"x-evolution-any-field\" \"\")");
            if (cached_contacts) {
                GList *iter;
                for(iter = cached_contacts; iter; iter = iter->next) {
                    EContact *contact = E_CONTACT(iter->data);
                    if (contact) {
                        e_data_book_view_notify_update (E_DATA_BOOK_VIEW (book_view), contact);
                    }
                    g_object_unref (contact);
                }
                g_list_free (cached_contacts);
            }
            if (priv->proxy) {
                g_dbus_proxy_call_sync (priv->proxy, "startContactObserver",
                                         NULL,
                                         G_DBUS_CALL_FLAGS_NONE,
                                         -1,
                                         NULL,
                                         NULL);
            }
        } else if (priv->proxy) {
            GError *error = e_book_backend_android_sync_contact (backend, TRUE);
            if (error) {
                e_data_book_view_notify_complete (book_view, error);
                return;
            }
        }
    } else {
        const gchar *query = e_data_book_view_get_card_query (book_view);
        GList *cached_contacts = e_book_sqlite_cache_get_contacts (priv, query);
        for (; cached_contacts; cached_contacts = g_list_delete_link (cached_contacts, cached_contacts)) {
            EContact *contact = cached_contacts->data;
            e_data_book_view_notify_update (book_view, contact);
            g_object_unref (contact);
        }

        g_list_free (cached_contacts);
        GError *error = NULL;
        if (priv->proxy) {
            g_dbus_proxy_call_sync (priv->proxy, "startContactObserver",
                                         NULL,
                                         G_DBUS_CALL_FLAGS_NONE,
                                         -1,
                                         NULL,
                                         &error);
            if (error) {
                e_book_backend_notify_error (E_BOOK_BACKEND (backend), error->message);
                g_error_free (error);
                return;
            }
        }
    }

    e_data_book_view_notify_complete (book_view, NULL /* Success */);
}

static void
e_book_backend_android_stop_book_view (EBookBackend *backend,
                                       EDataBookView *book_view)
{
    g_debug ("FUNCTION: %s\n", __FUNCTION__);
    EBookBackendAndroid *b = E_BOOK_BACKEND_ANDROID (backend);
    EBookBackendAndroidPrivate *priv = b->priv;
    GList *view;

    if ((view = g_list_find (priv->bookviews, book_view)) != NULL) {
        priv->bookviews = g_list_delete_link (priv->bookviews, view);
        e_data_book_view_unref (book_view);
    }
}

static void
e_book_backend_android_authenticate_user (EBookBackend *backend,
                                          GCancellable *cancellable,
                                          ECredentials *credentials)
{
    /* Success */
}

static void
e_book_backend_android_set_online (EBookBackend *backend,
                                   gboolean is_online)
{
    e_book_backend_notify_online (backend, is_online);
}

static void
e_book_backend_android_open (EBookBackend *backend,
                             EDataBook *book,
                             guint opid,
                             GCancellable *cancellable,
                             gboolean only_if_exists)
{
    g_debug ("FUNCTION: %s\n", __FUNCTION__);
    EBookBackendAndroid *b = E_BOOK_BACKEND_ANDROID (backend);


    if (b->priv->watch_id > 0) {
        return;
    }

    // wait for a new connection
    b->priv->watch_id = g_bus_watch_name (G_BUS_TYPE_SESSION,
                                          UBUNTU_ANDROID_SERVICE_NAME,
                                          G_BUS_NAME_WATCHER_FLAGS_NONE,
                                          e_book_backend_android_dbus_service_appear,
                                          e_book_backend_android_dbus_service_disapear,
                                          backend,
                                          NULL);

    e_book_backend_notify_opened (backend, NULL);
    e_data_book_respond_open (book, opid, NULL /* Success */);
}


static void
e_book_backend_android_remove (EBookBackend *backend,
                               EDataBook *book,
                               guint32 opid,
                               GCancellable *cancellable)
{
    g_debug ("FUNCTION: %s\n", __FUNCTION__);
    EBookBackendAndroid *b = E_BOOK_BACKEND_ANDROID (backend);

    if (b->priv->watch_id > 0) {
        g_bus_unwatch_name (b->priv->watch_id);
        b->priv->watch_id = 0;
    }

    g_clear_object (&b->priv->proxy);

    e_data_book_respond_remove (book, opid, NULL);
}

static void
e_book_backend_android_get_backend_property (EBookBackend *backend,
                                             EDataBook *book,
                                             guint32 opid,
                                             GCancellable *cancellable,
                                             const gchar *prop_name)
{
    if (g_str_equal (prop_name, CLIENT_BACKEND_PROPERTY_CAPABILITIES)) {
        e_data_book_respond_get_backend_property (book, opid, NULL, "local,contact-lists,do-initial-query,bulk-removes");
    } else if (g_str_equal (prop_name, BOOK_BACKEND_PROPERTY_REQUIRED_FIELDS)) {
        e_data_book_respond_get_backend_property (book, opid, NULL, "");
    } else if (g_str_equal (prop_name, BOOK_BACKEND_PROPERTY_SUPPORTED_AUTH_METHODS)) {
        e_data_book_respond_get_backend_property (book, opid, NULL, "");
    } else if (g_str_equal (prop_name, BOOK_BACKEND_PROPERTY_SUPPORTED_FIELDS)) {
        GSList *fields = NULL;
        gint i;

        const gint supported_fields[] = {
            // E_CONTACT_FILE_AS,
            // E_CONTACT_BOOK_URI,
            E_CONTACT_FULL_NAME,
            E_CONTACT_GIVEN_NAME,
            E_CONTACT_FAMILY_NAME,
            E_CONTACT_NICKNAME,
            E_CONTACT_EMAIL_1,
            E_CONTACT_EMAIL_2,
            E_CONTACT_EMAIL_3,
            E_CONTACT_EMAIL_4,
            // E_CONTACT_MAILER,
            E_CONTACT_ADDRESS_LABEL_HOME,
            E_CONTACT_ADDRESS_LABEL_WORK,
            E_CONTACT_ADDRESS_LABEL_OTHER,
            // E_CONTACT_PHONE_ASSISTANT,
            E_CONTACT_PHONE_BUSINESS,
            E_CONTACT_PHONE_BUSINESS_2,
            E_CONTACT_PHONE_BUSINESS_FAX,
            // E_CONTACT_PHONE_CALLBACK,
            // E_CONTACT_PHONE_CAR,
            // E_CONTACT_PHONE_COMPANY,
            E_CONTACT_PHONE_HOME,
            E_CONTACT_PHONE_HOME_2,
            E_CONTACT_PHONE_HOME_FAX,
            // E_CONTACT_PHONE_ISDN,
            E_CONTACT_PHONE_MOBILE,
            E_CONTACT_PHONE_OTHER,
            // E_CONTACT_PHONE_OTHER_FAX,
            // E_CONTACT_PHONE_PRIMARY,
            // E_CONTACT_PHONE_RADIO,
            // E_CONTACT_PHONE_TELEX,
            // E_CONTACT_PHONE_TTYTDD,
            E_CONTACT_PHONE_PAGER,
            E_CONTACT_ORG,
            E_CONTACT_ORG_UNIT,
            E_CONTACT_OFFICE,
            E_CONTACT_TITLE,
            E_CONTACT_ROLE,
            // E_CONTACT_MANAGER,
            // E_CONTACT_ASSISTANT,
            E_CONTACT_HOMEPAGE_URL,
            // E_CONTACT_BLOG_URL,
            E_CONTACT_CATEGORIES,
            // E_CONTACT_CALENDAR_URI,
            // E_CONTACT_FREEBUSY_URL,
            // E_CONTACT_ICS_CALENDAR,
            // E_CONTACT_VIDEO_URL,
            // E_CONTACT_SPOUSE,
            E_CONTACT_NOTE,
            E_CONTACT_IM_AIM_HOME_1,
            E_CONTACT_IM_AIM_HOME_2,
            E_CONTACT_IM_AIM_HOME_3,
            E_CONTACT_IM_AIM_WORK_1,
            E_CONTACT_IM_AIM_WORK_2,
            E_CONTACT_IM_AIM_WORK_3,
            // E_CONTACT_IM_GROUPWISE_HOME_1,
            // E_CONTACT_IM_GROUPWISE_HOME_2,
            // E_CONTACT_IM_GROUPWISE_HOME_3,
            // E_CONTACT_IM_GROUPWISE_WORK_1,
            // E_CONTACT_IM_GROUPWISE_WORK_2,
            // E_CONTACT_IM_GROUPWISE_WORK_3,
            E_CONTACT_IM_JABBER_HOME_1,
            E_CONTACT_IM_JABBER_HOME_2,
            E_CONTACT_IM_JABBER_HOME_3,
            E_CONTACT_IM_JABBER_WORK_1,
            E_CONTACT_IM_JABBER_WORK_2,
            E_CONTACT_IM_JABBER_WORK_3,
            E_CONTACT_IM_YAHOO_HOME_1,
            E_CONTACT_IM_YAHOO_HOME_2,
            E_CONTACT_IM_YAHOO_HOME_3,
            E_CONTACT_IM_YAHOO_WORK_1,
            E_CONTACT_IM_YAHOO_WORK_2,
            E_CONTACT_IM_YAHOO_WORK_3,
            E_CONTACT_IM_MSN_HOME_1,
            E_CONTACT_IM_MSN_HOME_2,
            E_CONTACT_IM_MSN_HOME_3,
            E_CONTACT_IM_MSN_WORK_1,
            E_CONTACT_IM_MSN_WORK_2,
            E_CONTACT_IM_MSN_WORK_3,
            E_CONTACT_IM_ICQ_HOME_1,
            E_CONTACT_IM_ICQ_HOME_2,
            E_CONTACT_IM_ICQ_HOME_3,
            E_CONTACT_IM_ICQ_WORK_1,
            E_CONTACT_IM_ICQ_WORK_2,
            E_CONTACT_IM_ICQ_WORK_3,
            // E_CONTACT_IM_GADUGADU_HOME_1,
            // E_CONTACT_IM_GADUGADU_HOME_2,
            // E_CONTACT_IM_GADUGADU_HOME_3,
            // E_CONTACT_IM_GADUGADU_WORK_1,
            // E_CONTACT_IM_GADUGADU_WORK_2,
            // E_CONTACT_IM_GADUGADU_WORK_3,
            E_CONTACT_IM_SKYPE_HOME_1,
            E_CONTACT_IM_SKYPE_HOME_2,
            E_CONTACT_IM_SKYPE_HOME_3,
            E_CONTACT_IM_SKYPE_WORK_1,
            E_CONTACT_IM_SKYPE_WORK_2,
            E_CONTACT_IM_SKYPE_WORK_3,
            E_CONTACT_IM_GOOGLE_TALK_HOME_1,
            E_CONTACT_IM_GOOGLE_TALK_HOME_2,
            E_CONTACT_IM_GOOGLE_TALK_HOME_3,
            E_CONTACT_IM_GOOGLE_TALK_WORK_1,
            E_CONTACT_IM_GOOGLE_TALK_WORK_2,
            E_CONTACT_IM_GOOGLE_TALK_WORK_3,
            // E_CONTACT_REV,
            // E_CONTACT_NAME_OR_ORG,
            E_CONTACT_ADDRESS,
            E_CONTACT_ADDRESS_HOME,
            E_CONTACT_ADDRESS_WORK,
            E_CONTACT_ADDRESS_OTHER,
            // E_CONTACT_CATEGORY_LIST,
            E_CONTACT_PHOTO,
            // E_CONTACT_LOGO,
            E_CONTACT_NAME,
            E_CONTACT_EMAIL,
            E_CONTACT_IM_AIM,
            // E_CONTACT_IM_GROUPWISE,
            E_CONTACT_IM_JABBER,
            E_CONTACT_IM_YAHOO,
            E_CONTACT_IM_MSN,
            E_CONTACT_IM_ICQ,
            // E_CONTACT_IM_GADUGADU,
            E_CONTACT_IM_SKYPE,
            E_CONTACT_IM_GOOGLE_TALK,
            // E_CONTACT_WANTS_HTML,
            // E_CONTACT_IS_LIST,
            // E_CONTACT_LIST_SHOW_ADDRESSES,
            // E_CONTACT_BIRTH_DATE,
            // E_CONTACT_ANNIVERSARY,
            // E_CONTACT_X509_CERT,
            // E_CONTACT_GEO,
            E_CONTACT_TEL,
            E_CONTACT_SIP,
        };

        /* Add all the fields above to the list */
        for (i = 0; i < G_N_ELEMENTS (supported_fields); i++) {
            const gchar *field_name = e_contact_field_name (supported_fields[i]);
            fields = g_slist_prepend (fields, (gpointer) field_name);
        }

        gchar *fields_str= e_data_book_string_slist_to_comma_string (fields);
        g_slist_free (fields);

        e_data_book_respond_get_backend_property (book, opid, NULL, fields_str);

        g_free (fields_str);
    } else {
        g_debug ("PROPERTY NOT SUPPORTED: %s\n", prop_name);
        E_BOOK_BACKEND_CLASS (e_book_backend_android_parent_class)->get_backend_property (backend, book, opid, cancellable, prop_name);
    }
}

EBookBackend *
e_book_backend_android_new (void)
{
    return g_object_new (E_TYPE_BOOK_BACKEND_ANDROID, NULL);
}

static void
e_book_backend_android_dispose (GObject *object)
{
    EBookBackendAndroid *b = E_BOOK_BACKEND_ANDROID (object);
    EBookBackendAndroidPrivate *priv = b->priv;

    if (priv->pending_sync != 0) {
        g_source_remove (priv->pending_sync);
    }

    e_book_backend_android_cancel_sync (b);

    if (priv) {
        while (priv->bookviews) {
            e_data_book_view_unref (priv->bookviews->data);
            priv->bookviews = g_list_delete_link (priv->bookviews, priv->bookviews);
        }

        if (priv->proxy) {
            g_object_unref(priv->proxy);
        }

        if (priv->cache) {
            sqlite3_close(priv->cache);
        }
    }

    g_object_unref (priv->sync_cancel);
    g_mutex_unlock (priv->sync_lock);
    g_free (priv->sync_lock);
    g_rec_mutex_unlock (priv->db_lock);
    g_free (priv->db_lock);
    G_OBJECT_CLASS (e_book_backend_android_parent_class)->dispose (object);
}

static void
e_book_backend_android_finalize (GObject *object)
{
    G_OBJECT_CLASS (e_book_backend_android_parent_class)->finalize (object);
}

static void
e_book_backend_android_class_init (EBookBackendAndroidClass *klass)
{
    GObjectClass    *object_class = G_OBJECT_CLASS (klass);
    EBookBackendClass *backend_class = E_BOOK_BACKEND_CLASS (klass);

    g_type_class_add_private (klass, sizeof (EBookBackendAndroidPrivate));

    /* Set the virtual methods. */
    backend_class->start_book_view      = e_book_backend_android_start_book_view;
    backend_class->stop_book_view       = e_book_backend_android_stop_book_view;
    backend_class->set_online           = e_book_backend_android_set_online;
    backend_class->open                 = e_book_backend_android_open;
    backend_class->get_backend_property = e_book_backend_android_get_backend_property;
    backend_class->start_book_view      = e_book_backend_android_start_book_view;
    backend_class->stop_book_view       = e_book_backend_android_stop_book_view;
    backend_class->remove               = e_book_backend_android_remove;
    backend_class->create_contact       = e_book_backend_android_create_contact;
    backend_class->remove_contacts      = e_book_backend_android_remove_contacts;
    backend_class->modify_contact       = e_book_backend_android_modify_contact;
    backend_class->get_contact          = e_book_backend_android_get_contact;
    backend_class->get_contact_list     = e_book_backend_android_get_contact_list;
    backend_class->get_contact_list_uids= e_book_backend_android_get_contact_list_uids;
    backend_class->authenticate_user    = e_book_backend_android_authenticate_user;

    object_class->dispose   = e_book_backend_android_dispose;
    object_class->finalize  = e_book_backend_android_finalize;
}

static void
e_book_backend_android_init (EBookBackendAndroid *backend)
{
    backend->priv = G_TYPE_INSTANCE_GET_PRIVATE (backend,
                                                 E_TYPE_BOOK_BACKEND_ANDROID,
                                                 EBookBackendAndroidPrivate);
    backend->priv->sync_lock = g_new (GMutex, 1);
    backend->priv->db_lock = g_new (GRecMutex, 1);
    backend->priv->sync_cancel = g_cancellable_new ();
    backend->priv->cache = NULL;
    g_mutex_init (backend->priv->sync_lock);
    g_rec_mutex_init (backend->priv->db_lock);
}

