# This file is part of the Falcon repository manager
# Copyright (C) 2005-2008 Dennis Kaarsemaker
# See the file named COPYING in the root of the source tree for license details
#
# config.py(.in) - Configuration system

import falcon
import falcon.util
import shelve, snack, os, re, sys
import cPickle as pickle
from gettext import gettext as _

# Try and detect the rootdir here as well so we can check the database file
# and see if it says we need to use mysql/postgres
rootdir = None
uri = None
try:
    os.readlink(os.path.join(os.environ['HOME'], '.falcon', 'rootdir'))
except OSError:
    pass
if '-r' in sys.argv:
    i = sys.argv.index('-r') + 1
    if i < len(sys.argv):
        rootdir = sys.argv[i]
if rootdir:
    if os.path.exists(os.path.join(rootdir,'.falcon','data')):
        fd = open(os.path.join(rootdir,'.falcon','data'))
        inp = fd.read(200)
        fd.close()
        if not inp.startswith('SQLite format 3'):
            # OK, so we need a database connection
            uri = falcon.util.parse_uri(uri)

import django.conf
if uri:
    django.conf.settings.configure(
        DATABASE_ENGINE = uri.engine,
        DATABASE_HOST = uri.hostname,
        DATABASE_PORT = uri.port,
        DATABASE_NAME = uri.database,
        DATABASE_USER = uri.username,
        DAPABASE_PASSWORD = uri.password,
        INSTALLED_APPS  = ('falcon',)
    )
else:
    django.conf.settings.configure(
        DATABASE_ENGINE = 'sqlite3',
        INSTALLED_APPS  = ('falcon',)
    )
del rootdir
del uri

from django.db import models

falcon_version = '2.0.5'
prefix         = '/usr'
falcon_author  = 'Dennis Kaarsemaker <dennis@kaarsemaker.net>'
falcon_url     = 'http://www.kaarsemaker.net/software/falcon'

mcre    = re.compile(r'[a-z0-9]+([-.+][a-z0-9]+)*')

class ConfigurationKey(models.Model):
    """An automatically pickled key/value pair"""
    key = models.CharField(maxlength=50, unique=True)
    value = models.TextField()
    comment = models.TextField(blank=True)

    def __str__(self):
        return str(pickle.loads(self.value))
    def __repr__(self):
        return "<Configuration Key %s: %s>" % (self.key, pickle.loads(self.value))

class Configuration(object):
    """A dict-like configuration interface using auto-pickled values"""
    def __init__(self, c_prefix='', options = {}):
        self.__dict__['c_prefix'] = c_prefix
        self.__dict__['values'] = {}
        self.__dict__['questions'] = []

    def __repr__(self):
        return '<Falcon configuration>'

    def __str__(self):
        return "\n".join(["%s: %s" % (x, self.__dict__[x]) for x in self.__dict__ \
                          if x[0] != '_' and x not in ('options', 'values', 'questions')]) + \
               "\n" + "\n".join(["%s: %s" % (x, pickle.loads(self.values[x].value)) for x in self.values])

    def configure(self):
        pass
    
    def set_options(self, options):
        """Initialize volatile elements"""
        for o in options.__dict__:
            if o[0] == '_':
                continue
            self.__dict__[o] = getattr(options,o)

    def register(self, name, default, question=None):
        """Register a key/value pair with question"""
        p_name = self.c_prefix + name
        try:
            key = ConfigurationKey.objects.get(key=p_name)
        except ConfigurationKey.DoesNotExist:
            key = ConfigurationKey(key = p_name, value = pickle.dumps(default))
            key.save()
        self.values[p_name] = key
        if question:
            question.config = self
            question.key = name
            self.questions.append(question)

    def __getattr__(self, key):
        key = self.c_prefix + key
        if key in self.__dict__:
            return self.__dict__[key]
        else:
            v = self.__dict__['values'].get(key)
            if v:
                return pickle.loads(v.value)

    def __getitem__(self, key): 
        key = self.c_prefix + key
        return pickle.loads(self.values[key].value)

    def __setitem__(self, key, value):
        key = self.c_prefix + key
        key = self.__dict__['values'][key]
        key.value = pickle.dumps(value)
        key.save()

    def __setattr__(self, key, value):
        key = self.c_prefix + key
        if key in self.values:
            key = self.values[key]
            key.value = pickle.dumps(value)
            key.save()
        else:
            self.__dict__[key] = value

class SystemConfiguration(Configuration):
    """Configuration subclass with custom configure function"""
    def set_options(self, options, needs_rootdir=True, action=None):
        """Main initialization function"""
        super(SystemConfiguration, self).set_options(options)
        # Option checking
        if options.application == 'falcon':
            if options.force_yes and options.force_no:
                falcon.util.error(_("Can't force both yes and no"))
            if not options.iso_architecture:
                self.iso_architecture = self.architecture
            try:
                self.isosize = int(self.isosize)
            except:
                falcon.util.error(_("Iso size should be a number"))
        if options.application == 'falcon-import':
            if options.delete and not options.extract:
                falcon.util.error(_("Deleting files without extracting them is silly"))
        if action and action != 'build':
            if bool(options.component) ^ bool(options.pocket):
                falcon.util.error(_("You need to specify both pocket and component (or neither)"))

        # Check rootdir
        if os.path.islink(os.path.join(os.getenv('HOME'), '.falcon', 'rootdir')):
            self.rootdir = os.readlink(os.path.join(os.getenv('HOME'), '.falcon', 'rootdir')) 
        if options.rootdir:
            self.rootdir = options.rootdir

        if self.rootdir:
            if not os.path.exists(self.rootdir):
                if needs_rootdir:
                    falcon.util.error(_("Repository root %s does not exist") % self.rootdir)
                falcon.util.warning(_("Repository root %s does not exist") % self.rootdir)
            else:
                if 'pocket' in dir(options) and options.pocket:
                    if not os.path.exists(os.path.join(self.rootdir, 'pool', options.pocket)):
                        falcon.util.error(_("Pocket %s does not exist") % options.pocket)
                    if options.component:
                        if not os.path.exists(os.path.join(self.rootdir, 'pool', options.pocket, options.component)):
                            falcon.util.error(_("Pocket %s does not exist") % os.path.join(options.pocket, options.component))
                falcon.init_rootdir()
        elif needs_rootdir:
            falcon.util.error(_("No repository root sepcified"))

    def configure(self):
        """Python-newt based interactive config editor"""
        screen = falcon.screen = snack.SnackScreen()
        screen.drawRootText(1,1, _("Falcon %s interactive configuration editor") % self.falcon_version)
        screen.drawRootText(1,2, "(C)2005-2008 %s" % (self.falcon_author))
        self.force_yes = False
        self.force_no = False

        EXIT, GENERAL, POCKETS, MIRRORS, PLUGINS, BUILDING = range(6)
        mainmenu = falcon.questions.SelectionForm(_("Configuration wizard"), _("Please choose from the menu"), 
                                                  ((_("General configuration"), GENERAL),
                                                   (_("Pockets and components"), POCKETS),
                                                   (_("Mirrors"), MIRRORS),
                                                   (_("Plugins"), PLUGINS),
                                                   (_("Building"), BUILDING)),
                                                   _("Exit configuration"))
        while True:
            ret = mainmenu.run()
            if not ret:
                screen.suspend()
                self.configured = True
                return

            if ret == GENERAL:
                current = dict([(x.key, self[x.key]) for x in self.questions])
                form = falcon.questions.QuestionForm(_("General configuration"), self.questions)
                answers = form.run(current)
                if answers:
                    for k in answers:
                        self[k] = answers[k]
                                
            elif ret == POCKETS:
                pocketmenu = falcon.questions.SelectionForm(_("Pocket configuration"), _("Select a pocket to configure"),
                                                            tuple([(p.name,p) for p in falcon.pockets]),
                                                            _("Back to main menu"))
                while True:
                    pocket = pocketmenu.run()
                    if not pocket:
                        break
                    
                    # Now configure the pocket
                    questions = [
                     falcon.questions.Entry(_("Version number for this pocket"),[falcon.validators.version]),
                     falcon.questions.Text(_("Description of this pocket"),[])
                    ]
                    questions[0].key = 'version'
                    questions[1].key = 'description'

                    current =  {'version': pocket.version, 'description': pocket.description}
                    for c in pocket.realcomponents.all():
                        q = falcon.questions.Text(_("Description for component %s") % c.name)
                        q.key = c.id
                        current[c.id] = c.description
                        questions.append(q)

                    form = falcon.questions.QuestionForm(_("Pocket %s") % pocket.name, questions)
                    answers = form.run(current)

                    if not answers:
                        continue

                    pocket.version = answers['version']
                    pocket.description = answers['description']
                    pocket.save()

                    for c in pocket.realcomponents.all():
                        c.description = answers[c.id]
                        c.save()

                    # Metacomponents
                    while True:
                        questions = []
                        current = {}
                        for c in pocket.metacomponents.all():
                            q1 = falcon.questions.Text(_("Description for metacomponent %s") % c.name, [])
                            q1.key = (c.id,'description')
                            current[(c.id,'description')] = c.description
                            q2 = falcon.questions.SelectMulti(_("Components"), sorted([y.name for y in pocket.realcomponents.all()]), [])
                            q2.key = (c.id, 'components')
                            current[(c.id,'components')] = [x.name for x in c.components.all()]
                            questions += [q1, q2]
                        questions.append(falcon.questions.Entry(_("Add new metacomponent"), [falcon.validators.nospaces]))
                        questions[-1].key = 'newcomponent'
                        current['newcomponent'] = ''

                        metacompform = falcon.questions.QuestionForm(_("Pocket %s, metacomponents") % pocket.name, questions)
                        answers = metacompform.run(current)
                        if not answers:
                            break

                        for c in pocket.metacomponents.all():
                            c.description = answers[(c.id, 'description')]
                            current_components = [x.id for x in c.components.all()]
                            if not len(answers[(c.id, 'components')]):
                                c.delete()
                                continue
                            for _c in answers[(c.id, 'components')]:
                                _c = falcon.pocket.Component.objects.get(name=_c, pocket=pocket)
                                if _c.id not in current_components:
                                    c.components.add(_c)
                                else:
                                    current_components.remove(_c.id)
                            for _c in c.components.all():
                                if c.id in current_components:
                                    c.components.remove(_c)
                            c.save()
                        if answers['newcomponent']:
                            c = falcon.pocket.MetaComponent(pocket=pocket, name=answers['newcomponent'])
                            c.save()
                        else:
                            break

            elif ret == MIRRORS:
                while True:
                    mirrormenu = falcon.questions.SelectionForm(_("Mirror configuration"), _("Select a mirror to configure"),
                                                                tuple([(m.name, m) for m in falcon.mirror.Mirror.objects.all()] +
                                                                      [(_("New mirror"), -1)]),
                                                                _("Back to main menu"))

                    mirror = mirrormenu.run()
                    if not mirror:
                        break
                    if mirror == -1:
                        mirror = falcon.mirror.Mirror()
                        comps = []
                    else:
                        comps = list(mirror.realcomponents.all()) + list(mirror.metacomponents.all())

                    questions = [
                     falcon.questions.Entry(_("Name of this mirror"), [falcon.validators.shellsafe, falcon.validators.nospaces]),
                     falcon.questions.Entry(_("URL of this mirror's root")),
                     falcon.questions.Entry(_("Rsync upload path"), [falcon.validators.shellsafe]),
                     falcon.questions.QuotedEntry(_("Extra rsync arguments")),
                     falcon.questions.Entry(_("Who sponsors this mirror")),
                     falcon.questions.ComponentTree(_("Which components should be mirrored"))
                    ]
                    questions[0].key = 'name'; questions[1].key = 'webbase'; questions[2].key = 'rsync'
                    questions[3].key = 'rsync_opts'; questions[4].key = 'sponsor'; questions[5].key = 'components'
                    current = {
                     'name': mirror.name, 'webbase': mirror.webbase, 'rsync': mirror.rsync,
                     'rsync_opts': mirror.rsync_opts, 'sponsor': mirror.sponsor, 'components': comps
                    }
                    form = falcon.questions.QuestionForm(_("Mirror %s") % mirror.name, questions)
                    answers = form.run(current)
                    if not answers:
                        continue

                    mirror.name = answers['name']
                    mirror.webbase = answers['webbase']
                    mirror.rsync = answers['rsync']
                    mirror.rsync_opts = answers['rsync_opts']
                    mirror.sponsor = answers['sponsor']

                    if not len(mirror.name):
                        if mirror.id:
                            mirror.delete()
                    else:
                        mirror.save()
                        mirror = falcon.mirror.Mirror.objects.get(name=mirror.name)
                        mirror.realcomponents = [x for x in answers['components'] if type(x) == falcon.pocket.Component]
                        mirror.metacomponents = [x for x in answers['components'] if type(x) == falcon.pocket.MetaComponent]
                        mirror.save()

            elif ret == BUILDING:
                while True:
                    buildmenu = falcon.questions.SelectionForm(_("Builder configuration"), _("Select a builder to configure"),
                                                                tuple([(b.name, b) for b in falcon.build.Builder.objects.all()] +
                                                                      [(_("New builder"), -1)]),
                                                                _("Back to main menu"))

                    builder = buildmenu.run()
                    if not builder:
                        break
                    if builder == -1:
                        builder = falcon.build.Builder()

                    all_architectures = ('i386','amd64','powerpc','sparc','arm','ia64','alpha','hppa','m68k','mips','mipsel','s390')
                    questions = [
                     falcon.questions.Entry(_("Name of this buildd"), [falcon.validators.shellsafe, falcon.validators.nospaces]),
                     falcon.questions.Entry(_("Hostname of this buildd")),
                     falcon.questions.Select(_("Architecture"), all_architectures),
                     falcon.questions.Boolean(_("Build Arch: all packages")),
                     falcon.questions.Entry(_("Rsync upload path (with hostname)"), [falcon.validators.shellsafe]),
                     falcon.questions.Entry(_("Rsync download path (with hostname)"), [falcon.validators.shellsafe]),
                     falcon.questions.QuotedEntry(_("Build command")),
                     falcon.questions.Text(_("Comment"))
                    ]
                    questions[0].key = 'name'; questions[1].key = 'hostname'; questions[2].key = 'arch'; questions[3].key = 'arch_all'
                    questions[4].key = 'uploadpath'; questions[5].key = 'downloadpath'; questions[6].key = 'buildcommand'; questions[7].key = 'comment'
                    current = {
                      'name': builder.name, 'hostname': builder.hostname, 'arch': builder.arch, 
                      'arch_all': builder.arch_all, 'uploadpath': builder.uploadpath, 'downloadpath': builder.downloadpath, 
                      'buildcommand': builder.buildcommand, 'comment': builder.comment
                    }

                    form = falcon.questions.QuestionForm(_("Builder %s") % builder.name, questions)
                    answers = form.run(current)
                    if not answers:
                        continue

                    builder.name = answers['name']
                    builder.hostname = answers['hostname']
                    builder.arch = answers['arch']
                    builder.arch_all = answers['arch_all']
                    builder.uploadpath = answers['uploadpath']
                    builder.downloadpath = answers['downloadpath']
                    builder.buildcommand = answers['buildcommand']
                    builder.comment = answers['comment']

                    if not len(builder.name):
                        if builder.id:
                            builder.delete()
                    else:
                        builder.save()

            elif ret == PLUGINS:
                while True:
                    pluginmenu = falcon.questions.SelectionForm(_("Plugin configuration"), _("Select a plugin to configure"),
                                                                tuple([(p.name, p) for p in falcon.plugin.loaded_plugins.values()]),
                                                                _("Back to main menu"))
                    plugin = pluginmenu.run()
                    if not plugin:
                        break
                    plugin.configure()

# Initial config
conf = SystemConfiguration()
conf.falcon_version = falcon_version
conf.falcon_author  = falcon_author
conf.falcon_url     = falcon_url
conf.prefix         = prefix
conf.template_api   = (1,0)
conf.plugin_api     = (1,1)
conf.database_abi   = (3,0)
conf.architecture   = falcon.util.run(['dpkg', '--print-architecture']).strip()
