/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  AFILE *AFwrWVhead (FILE *fp, struct AF_write *AF)

Purpose:
  Write header information to a WAVE file

Description:
  This routine opens and writes header information to a WAVE format audio
  file.

  WAVE file:
   Offset Length Type    Contents
      0     4    char   "RIFF" file identifier
      4     4    int    Chunk length
      8     4    char   "WAVE" file identifier
     12     4    char   "fmt " chunk identifier
     16     4    int    Chunk length
     20     2    int      Audio data type
     22     2    int      Number of interleaved channels
     24     4    int      Sample rate
     28     4    int      Average bytes/sec
     32     2    int      Block align
     34     2    int      Data word length (bits)
     36     2    int      Extra info size
      C     4    int    "fact" chunk identifier (only for non-PCM data)
    C+4     4    int    Chunk length
    C+8     4    int      Number of samples (per channel)
    ...   ...    ...    ...
      D     4    char   "data" chunk identifier
    D+4     4    int    Chunk length
    D+8    ...   ...      Audio data

Parameters:
  <-  AFILE *AFwrWVhead
      Audio file pointer for the audio file.  In case of error, the audio file
      pointer is set to NULL.
   -> FILE *fp
      File pointer for the audio file
  <-> struct AF_write *AFw
      Structure containing file parameters

Author / revision:
  P. Kabal  Copyright (C) 2003
  $Revision: 1.69 $  $Date: 2003/05/13 01:57:49 $

-------------------------------------------------------------------------*/
 /************************************************************************
  * Copyright 2005-2010 by Freescale Semiconductor, Inc.
  * All modifications are confidential and proprietary information
  * of Freescale Semiconductor, Inc.
  ************************************************************************/
#include <assert.h>
#include <setjmp.h>
#include <string.h>
#include <errno.h>
#include <math.h>	/* floor */
#include <stdio.h>
#include <stdlib.h>		/* EXIT_FAILURE */
#include <stdarg.h>		/* ANSI C variable-length argument list */
#include <time.h>
#include <ctype.h>




#define AF_DATA_LENGTHS
#define AF_DATA_TYPE_NAMES
#define AF_SPKR_NAMES
#define WV_CHANNEL_MAP
#define WAVEFORMATEX_SUBTYPE
#include "AFoperation.h"
#include "libtsp.h"
#include "nucleus.h"



#define ICEILV(n,m)	(((n) + ((m) - 1)) / (m))	/* int n,m >= 0 */
#define RNDUPV(n,m)	((m) * ICEILV (n, m))		/* Round up */

#define MCOPY(src,dest)		memcpy ((void *) (dest), \
					(const void *) (src), sizeof (dest))
#define WRPAD(fp,size,align) \
     AFwriteHead (fp, NULL, 1, (int) (RNDUPV(size, align) - (size)), \
		  DS_NATIVE);

#define ALIGN		2	/* Chunks padded out to a multiple of ALIGN */

/* setjmp / longjmp environment */
jmp_buf AFW_JMPENV;		/* Defining point */


static void
AF_setFMT (struct WV_CKfmt *CKfmt, const struct AF_write *AFw);
static void
AF_wrRIFF (FILE *fp, const struct WV_CKRIFF *CKRIFF);
static int
AF_wrFMT (FILE *fp, const struct WV_CKfmt *CKfmt);
static uint4_t
AF_encChannelConfig (const unsigned char *SpkrConfig);


AFILE *
AFwrWVhead (FILE *fp, struct AF_write *AFw)

{
  AFILE *AFp;
  int Lw;
  long int size, Ldata;
  struct WV_CKRIFF CKRIFF;

/* Set the long jump environment; on error return a NULL */
  if (setjmp (AFW_JMPENV))
    return NULL;	/* Return from a header write error */

/* Set up the encoding parameters */
  Lw = AF_DL[AFw->DFormat.Format];
  if (AFw->Nframe != AF_NFRAME_UNDEF)
    Ldata = AFw->Nframe * AFw->Nchan * Lw;
  else if (FLseekable (fp))
    Ldata = 0L;
  else {
    UTwarn ("AFwrWVhead - %s", AFM_WV_WRAccess);
    return NULL;
  }

  /* RIFF chunk */
  MCOPY ("RIFF", CKRIFF.ckID);
  /* defer filling in the chunk size */
  MCOPY ("WAVE", CKRIFF.WAVEID);

  /* fmt chunk */
  AF_setFMT (&CKRIFF.CKfmt, AFw);

  /* fact chunk */
  if (CKRIFF.CKfmt.wFormatTag != WAVE_FORMAT_PCM) {
    MCOPY ("fact", CKRIFF.CKfact.ckID);
    CKRIFF.CKfact.ckSize = (uint4_t) sizeof (CKRIFF.CKfact.dwSampleLength);
    CKRIFF.CKfact.dwSampleLength = (uint4_t) 0;
  }

  /* data chunk */
  MCOPY ("data", CKRIFF.CKdata.ckID);
  CKRIFF.CKdata.ckSize = (uint4_t) Ldata;

  /* Fill in the RIFF chunk size */
  size = 4 + 8 + RNDUPV(CKRIFF.CKfmt.ckSize, ALIGN) +
             8 + RNDUPV(CKRIFF.CKdata.ckSize, ALIGN);
  if (CKRIFF.CKfmt.wFormatTag != WAVE_FORMAT_PCM)
    size += 8 + RNDUPV(CKRIFF.CKfact.ckSize, ALIGN);
  CKRIFF.ckSize = (uint4_t) size;

/* Write out the header (error return via longjmp) */
  AFw->DFormat.Swapb = DS_EL;
  AF_wrRIFF (fp, &CKRIFF);

/* Set the parameters for file access */
  AFp = AFsetWrite (fp, FT_WAVE, AFw);

  return AFp;
}

/* Fill in the fmt chunk */


static void
AF_setFMT (struct WV_CKfmt *CKfmt, const struct AF_write *AFw)

{
  int Lw, Ext, Res, NbS;
  uint2_t FormatTag;

  Lw = AF_DL[AFw->DFormat.Format];
  Res = 8 * Lw;
  NbS = AFw->DFormat.NbS;

  /* Determine whether to use an extensible header */
  Ext = 0;
  if (AFw->Ftype == FTW_WAVE)
    Ext = (AFw->Nchan > 2 || NbS != Res ||
	   AFw->DFormat.Format == FD_INT24 ||
	   AFw->DFormat.Format == FD_INT32 ||
	   (AFw->SpkrConfig[0] != AF_X_SPKR && AFw->SpkrConfig[0] != '\0'));

  switch (AFw->DFormat.Format) {
  case FD_UINT8:
  case FD_INT16:
  case FD_INT24:
  case FD_INT32:
    FormatTag = WAVE_FORMAT_PCM;
    break;
  case FD_MULAW8:
    FormatTag = WAVE_FORMAT_MULAW;
    break;
  case FD_ALAW8:
    FormatTag = WAVE_FORMAT_ALAW;
    break;
  case FD_FLOAT32:
  case FD_FLOAT64:
    FormatTag = WAVE_FORMAT_IEEE_FLOAT;
    break;
  default:
    FormatTag = WAVE_FORMAT_UNKNOWN;
    assert (0);
  }

  /* Set the chunk size (minimum size for now) */
  MCOPY ("fmt ", CKfmt->ckID);
  CKfmt->ckSize = sizeof (CKfmt->wFormatTag) + sizeof (CKfmt->nChannels)
    + sizeof (CKfmt->nSamplesPerSec) + sizeof (CKfmt->nAvgBytesPerSec)
    + sizeof (CKfmt->nBlockAlign) + sizeof (CKfmt->wBitsPerSample);

  CKfmt->nChannels = (uint2_t) AFw->Nchan;
  CKfmt->nSamplesPerSec = (uint4_t) (AFw->Sfreq + 0.5);	/* Rounding */
  CKfmt->nAvgBytesPerSec = (uint4_t) (CKfmt->nSamplesPerSec * AFw->Nchan * Lw);
  CKfmt->nBlockAlign = (uint2_t) (Lw * AFw->Nchan);

  if (Ext) {
    CKfmt->wBitsPerSample = (uint2_t) Res;
    CKfmt->wFormatTag = WAVE_FORMAT_EXTENSIBLE;
    CKfmt->cbSize = 22;
    if (FormatTag == WAVE_FORMAT_PCM || FormatTag == WAVE_FORMAT_IEEE_FLOAT)
      CKfmt->wValidBitsPerSample = (uint2_t) NbS;
    else
      CKfmt->wValidBitsPerSample = (uint2_t) Res;
    CKfmt->dwChannelMask = AF_encChannelConfig (AFw->SpkrConfig);
    CKfmt->SubFormat.wFormatTag = FormatTag;
    MCOPY (WAVEFORMATEX_TEMPLATE.guidx, CKfmt->SubFormat.guidx);
  }
  else {
    /* Use wBitsPerSample to specify the "valid bits per sample" */
    if (RNDUPV(NbS, 8) != Res) {
      UTwarn (AFMF_WV_InvNbS, "AFwrWVhead -", NbS, Res);
      NbS = Res;
    }
    CKfmt->wBitsPerSample = (uint2_t) NbS;
    CKfmt->wFormatTag = FormatTag;
    CKfmt->cbSize = 0;
  }

  /* Update the chunk size with the size of the chunk extension */
  if (CKfmt->wFormatTag != WAVE_FORMAT_PCM)
    CKfmt->ckSize += (sizeof (CKfmt->cbSize) + CKfmt->cbSize);

  return;
}

/* Encode channel/speaker information */


static uint4_t
AF_encChannelConfig (const unsigned char *SpkrConfig)

{
  int i, n, Nspkr;
  uint4_t ChannelMask;

  if (SpkrConfig[0] == AF_X_SPKR || SpkrConfig[0] == '\0')
    Nspkr = 0;
  else
    Nspkr = strlen ((const char *) SpkrConfig);

  ChannelMask = 0;
  for (i = 0; i < Nspkr; ++i) {
    n = SpkrConfig[i];
    if (i > 0 && n < SpkrConfig[i-1]) {
      UTwarn ("AFwrWVhead - %s", AFM_WV_BadSpkr);
      ChannelMask = 0;
      break;
    }
    if (n != AF_SPKR_X)
      ChannelMask = ChannelMask | WV_ChannelMap[n-1];
  }

  return ChannelMask;
}

/* Write out the header */


static void
AF_wrRIFF (FILE *fp, const struct WV_CKRIFF *CKRIFF)

{
  WHEAD_S (fp, CKRIFF->ckID);
  WHEAD_V (fp, CKRIFF->ckSize, DS_EL);
  WHEAD_S (fp, CKRIFF->WAVEID);

  AF_wrFMT (fp, &CKRIFF->CKfmt);

  /* Write the Fact chunk if the data format is not PCM */
  if (!(CKRIFF->CKfmt.wFormatTag == WAVE_FORMAT_PCM ||
      (CKRIFF->CKfmt.wFormatTag == WAVE_FORMAT_EXTENSIBLE &&
       CKRIFF->CKfmt.SubFormat.wFormatTag == WAVE_FORMAT_PCM))) {
    WHEAD_S (fp, CKRIFF->CKfact.ckID);
    WHEAD_V (fp, CKRIFF->CKfact.ckSize, DS_EL);
    WHEAD_V (fp, CKRIFF->CKfact.dwSampleLength, DS_EL);
  }

  WHEAD_S (fp, CKRIFF->CKdata.ckID);
  WHEAD_V (fp, CKRIFF->CKdata.ckSize, DS_EL);

  return;
}

/* Write the format chunk */


static int
AF_wrFMT (FILE *fp, const struct WV_CKfmt *CKfmt)

{
  long int offs, LFMT;

  offs  = WHEAD_S (fp, CKfmt->ckID);
  offs += WHEAD_V (fp, CKfmt->ckSize, DS_EL);
  LFMT = CKfmt->ckSize + 8;
  offs += WHEAD_V (fp, CKfmt->wFormatTag, DS_EL);
  offs += WHEAD_V (fp, CKfmt->nChannels, DS_EL);
  offs += WHEAD_V (fp, CKfmt->nSamplesPerSec, DS_EL);
  offs += WHEAD_V (fp, CKfmt->nAvgBytesPerSec, DS_EL);
  offs += WHEAD_V (fp, CKfmt->nBlockAlign, DS_EL);
  offs += WHEAD_V (fp, CKfmt->wBitsPerSample, DS_EL);
  if (offs < LFMT)
    offs += WHEAD_V (fp, CKfmt->cbSize, DS_EL);
  if (offs < LFMT) {
    offs += WHEAD_V (fp, CKfmt->wValidBitsPerSample, DS_EL);
    offs += WHEAD_V (fp, CKfmt->dwChannelMask, DS_EL);
    offs += WHEAD_V (fp, CKfmt->SubFormat.wFormatTag, DS_EL);
    offs += WHEAD_S (fp, CKfmt->SubFormat.guidx);
  }

  assert (offs == LFMT);

  return offs;
}

/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  int AFwriteHead (FILE *fp, const void *Buf, int Size, int Nv, int Swapb)

Purpose:
  Write (with optionally swap) audio file header values

Description:
  This routine writes data to an audio file header.  The information to be
  written is considered to be organized as Nv elements each of Size bytes.
  Optionally each of the Nv elements is byte swapped.  The data (Nv * Size
  bytes) is then written to the file.  If input buffer is buffer is a NULL
  pointer, this routine writes zero-valued bytes to the file.

  If an error occurs, this routine issues a longjmp to the AFW_JMPENV
  environment set up by the calling routine.

Parameters:
  <-  int AFwriteHead
      Number of bytes written
   -> FILE *fp
      File pointer associated with the file
   -> const void *Buf
      Pointer to a buffer of size Nelem * size or a NULL pointer
   -> int Size
      Size of each element in bytes
   -> int Nv
      Number of elements to be written
   -> int Swapb
      Byte swap flag.  This parameter is not used for size = 1.  If the bytes
      are to be swapped, size must be 2, 4 or 8 bytes.
      DS_EB     - File data is in big-endian byte order.  The data will be
      	          swapped if the current host uses little-endian byte order.
      DS_EL     - File data is in little-endian byte order data.  The data will
                  be swapped if the current host uses big-endian
		  byte order.
      DS_NATIVE - File data is in native byte order
      DS_SWAP   - File data is byte-swapped

Author / revision:
  P. Kabal  Copyright (C) 2003
  $Revision: 1.25 $  $Date: 2003/05/09 01:21:36 $

-------------------------------------------------------------------------*/


#define BSIZE(x)	((int)(sizeof (x)))
#define MINV(a, b)	(((a) < (b)) ? (a) : (b))

#define FWRITE(buf,size,nv,fp)	(int) fwrite ((const char *) buf, \
					      (size_t) size, (size_t) nv, fp)

#define NBUF	256	/* Number of doubles */
#define NPAD	16	/* Number of bytes */

/* setjmp / longjmp environment */
jmp_buf AFW_JMPENV;		/* Defining point */


int
AFwriteHead (FILE * fp, const void *Buf, int Size, int Nv, int Swapb)

{
  int n, Nreq, Nw;
  const char *p;
  double Lbuf[NBUF];
  static const char Pad[NPAD] = "";		/* Initialized to zero */

  n = 0;
  if (Buf == NULL) {

    /* Write zeros to the file */
    Nv = Nv * Size;	/* Change to units of bytes */
    Size = 1;
    while (n < Nv) {
      Nreq = MINV (Nv - n, NPAD);
      Nw = FWRITE (Pad, 1, Nreq, fp);
      n += Nw;
      if (Nw < Nreq)
	break;
    }
  }

  else if (Size != 1 && UTswapCode (Swapb) == DS_SWAP) {

    /* Swapped data passes through a temporary buffer */
    p = (const char *) Buf;
    while (n < Nv) {

      Nreq = MINV (Nv - n, BSIZE (Lbuf) / Size);
      VRswapBytes (p, Lbuf, Size, Nreq);
      Nw = FWRITE (Lbuf, Size, Nreq, fp);
      p += Nw * Size;
      n += Nw;
      if (Nw < Nreq)
	break;
    }
  }

  else

    /* Non-swapped data is written directly from the input buffer */
    n += FWRITE (Buf, Size, Nv, fp);

  if (n < Nv) {
/*    UTsysMsg ("AFwriteHead - %s", AFM_WriteErr);*/
    longjmp (AFW_JMPENV, 1);
  }

  return (Size * n);
}

/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  int AFwriteData (AFILE *AFp, const float Dbuff[], int Nval)

Purpose:
  Write data to an audio file (float input values)

Description:
  Use AFfWriteData instead.

Parameters:
  <-  int AFwriteData
      Number of samples written
   -> AFILE *AFp
      Audio file pointer for an audio file opened by AFopnWrite
   -> const float Dbuff[]
      Array of floats with Nval samples
   -> int Nval
      Number of samples to be written

Author / revision:
  P. Kabal  Copyright (C) 2003
  $Revision: 1.28 $  $Date: 2003/05/09 01:03:46 $

-------------------------------------------------------------------------*/


int
AFwriteData (AFILE *AFp, const float Dbuff[], int Nval)

{
  int Nw;

  Nw = AFfWriteData (AFp, Dbuff, Nval);

  return Nw;
}


/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  int AFupdWVhead (AFILE *AFp)

Purpose:
  Update header information in a RIFF WAVE file

Description:
  This routine updates the data length fields of a RIFF WAVE file.  The file
  is assumed to have been opened with routine AFopnWrite.  This routine also
  writes information chunks at the end of the file.

Parameters:
  <-  int AFupdWVhead
      Error code, zero for no error
   -> AFILE *AFp
      Audio file pointer for an audio file opened by AFopnWrite

Author / revision:
  P. Kabal  Copyright (C) 2003
  $Revision: 1.34 $  $Date: 2003/11/03 18:48:31 $

-------------------------------------------------------------------------*/

#define ICEILV(n,m)	(((n) + ((m) - 1)) / (m))	/* int n,m >= 0 */
#define NELEM(array)	((int) ((sizeof array) / (sizeof array[0])))
#define RNDUPV(n,m)	((m) * ICEILV (n, m))		/* Round up */

#define MCOPY(src,dest)		memcpy ((void *) (dest), \
					(const void *) (src), sizeof (dest))
#define WRPAD(fp,size,align) \
     AFwriteHead (fp, NULL, 1, (int) (RNDUPV(size, align) - (size)), \
		  DS_NATIVE);
#define SAME_CSTR(str,ref) 	(memcmp (str, ref, sizeof (str)) == 0)


#define ALIGN		2	/* Chunks padded out to a multiple of ALIGN */

/* setjmp / longjmp environment */
extern jmp_buf AFW_JMPENV;

static const char *
AF_findRec (const char *Id[], const struct AF_info *InfoS);
static void
AF_setAFSP (struct WV_AFSP *afsp, const struct AF_info *InfoS);
static void
AF_setDISP (struct WV_DISP *disp, const struct AF_info *InfoS);
static void
AF_setLIST_INFO (struct WV_LIST_INFO *info, const struct AF_info *InfoS);
static int
AF_wrAFSP (FILE *fp, const struct WV_AFSP *afsp);
static int
AF_wrDISP (FILE *fp, const struct WV_DISP *disp);
static int
AF_wrLIST_INFO (FILE *fp, const struct WV_LIST_INFO *info);


int
AFupdWVhead (AFILE *AFp)

{
  long int Nbytes, Ldata;
  uint4_t val;
  struct WV_AFSP afsp;
  struct WV_DISP disp;
  struct WV_LIST_INFO info;

/* Set the long jump environment; on error return a 1 */
  if (setjmp (AFW_JMPENV))
    return 1;		/* Return from a header write error */

/* This routine assumes that the header has the following layout
     offset  contents
        4     "RIFF" chunk
       12      "fmt" chunk
      ...     other data
      D-20    "fact" chunk (if present)
      D-16     Chunk size (4)
      D-12     Number of samples
      D-8     "data" chunk
      D-4       Chunk size
        D       Audio data

   Header size:
               RIFF  fmt   fact   Data   Total
               WAVE             preamble
   PCM          12    24     -     8       44
   non-PCM      12    26    12     8       58
   PCM-Ext      12    48     -     8       68
   non-PCM-Ext  12    48    12     8       80
*/

/* Add a padding byte to the sound data; this padding byte is not included
   in the data chunk cksize field, but is included in the RIFF chunk
   cksize field
*/
  Ldata = AF_DL[AFp->Format] * AFp->Nsamp;
  Nbytes = AFp->Start + Ldata;
  Nbytes += WRPAD (AFp->fp, Nbytes, ALIGN);	/* Write pad byte */

/* Generate the information records */
  AF_setAFSP (&afsp, &AFp->InfoS);
  AF_setDISP (&disp, &AFp->InfoS);
  AF_setLIST_INFO (&info, &AFp->InfoS);

/* Write the information records (at the end of the file) */
  if (afsp.cksize > 4) {
    AF_wrAFSP (AFp->fp, &afsp);
    Nbytes += 8 + RNDUPV (afsp.cksize, ALIGN);
  }
  if (disp.cksize > 4) {
    AF_wrDISP (AFp->fp, &disp);
    Nbytes += 8 + RNDUPV (disp.cksize, ALIGN);
  }
  if (info.cksize > 4) {
    Nbytes += 8 + RNDUPV (info.cksize, ALIGN);
    AF_wrLIST_INFO (AFp->fp, &info);
  }

/* Update the "RIFF" chunk cksize field (at the beginnning of the file) */
  val = (uint4_t) (Nbytes - 8);
  if (AFseek (AFp->fp, 4L, NULL))	/* Back at the beginning of the file */
    return 1;
  WHEAD_V (AFp->fp, val, DS_EL);

/* Update the "fact" chunk SampleLength field */
  assert (AFp->Start == 44 || AFp->Start == 58 ||
	  AFp->Start == 68 || AFp->Start == 80);
  if (AFp->Start == 58 || AFp->Start == 80) {
    if (AFseek (AFp->fp, AFp->Start - 12, NULL))
      return 1;
    val = (uint4_t) (AFp->Nsamp / AFp->Nchan);
    WHEAD_V (AFp->fp, val, DS_EL);
  }

/* Update the "data" chunk cksize field */
  if (AFseek (AFp->fp, AFp->Start - 4, NULL))
    return 1;
  val = (uint4_t) Ldata;
  WHEAD_V (AFp->fp, val, DS_EL); /* Number of data bytes */

  return 0;
}

/* Fill in the afsp chunk */


static void
AF_setAFSP (struct WV_AFSP *afsp, const struct AF_info *InfoS)

{
  MCOPY ("afsp", afsp->ckid);
  if (InfoS->N > 0)
    afsp->cksize = (uint4_t) (4 + InfoS->N);
  else
    afsp->cksize = (uint4_t) 4;
  MCOPY (FM_AFSP, afsp->AFspID);
  afsp->text = InfoS->Info;

  return;
}

/* Write the afsp chunk */


static int
AF_wrAFSP (FILE *fp, const struct WV_AFSP *afsp)

{
  int offs;

  offs = WHEAD_S (fp, afsp->ckid);
  offs += WHEAD_V (fp, afsp->cksize, DS_EL);
  offs += WHEAD_S (fp, afsp->AFspID);
  offs += WHEAD_SN (fp, afsp->text, afsp->cksize - 4);

  assert (8 + (int) afsp->cksize == offs);
  offs += WRPAD (fp, offs, ALIGN);

  return offs;
}

/* Fill in the DISP/text chunk */


static void
AF_setDISP (struct WV_DISP *disp, const struct AF_info *InfoS)

{
  const char *title;
  static const char *TitleID[] = {"title:", "display_name:", "name:",
				  "user_comment:", NULL};

  MCOPY ("DISP", disp->ckid);
  title = AF_findRec (TitleID, InfoS);
  if (title == NULL)
    disp->cksize = (uint4_t) 4;
  else
    disp->cksize = (uint4_t) (4 + strlen (title) + 1);
  disp->type = (uint4_t) (CF_TEXT);
  disp->text = title;

  return;
}

/* Write the DISP/text chunk */


static int
AF_wrDISP (FILE *fp, const struct WV_DISP *disp)

{
  int offs;

  offs = WHEAD_S (fp, disp->ckid);
  offs += WHEAD_V (fp, disp->cksize, DS_EL);
  offs += WHEAD_V (fp, disp->type, DS_EL);
  offs += WHEAD_SN (fp, disp->text, disp->cksize - 4);

  assert (8 + (int) disp->cksize == offs);
  offs += WRPAD (fp, offs, ALIGN);

  return offs;
}

/* Fill in the LIST/INFO chunk */


static void
AF_setLIST_INFO (struct WV_LIST_INFO *info, const struct AF_info *InfoS)

{
  int i, k, Nk, size;
  const char *text;
  static const char *IID[] = {
    "IARL", "archival_location:", NULL,
    "IART", "artist:", NULL,
    "ICMS", "commissioned:", NULL,
    "ICRD", "creation_date:", "date:", "recording_date:", NULL,
    "ICMT", "comments:", NULL,
    "ICOP", "copyright:", NULL,
    "IENG", "engineer:", NULL,
    "IGNR", "genre:", NULL,
    "IKEY", "keywords:", NULL,
    "IMED", "medium:", NULL,
    "INAM", "name:", NULL,
    "IPRD", "product:", NULL,
    "ISBJ", "subject:", NULL,
    "ISFT", "software:", "program:", NULL,
    "ISRC", "source:", NULL,
    "ISRF", "source_form:", NULL,
    "ITCH", "technician:", NULL,
    "ICMT", "user:", NULL	/* user: in comments field */
  };

  MCOPY ("LIST", info->ckid);
  MCOPY ("INFO", info->listid);

  Nk = (int) NELEM (IID);
  size = 4;
  k = 0;
  i = 0;
  while (k < Nk && i < WV_N_LIST_INFO) {

    /* Look for a record with the given keywords */
    text = AF_findRec (&IID[k+1], InfoS);	/* Keywords */
    if (text != NULL) {
      MCOPY (IID[k], info->listitem[i].ckid);	/* Item ID */
      info->listitem[i].cksize = strlen (text) + 1;
      info->listitem[i].text = text;
      size += 8 + RNDUPV (info->listitem[i].cksize, ALIGN);
      ++i;
    }

    /* Skip over the item ID and keywords */
    for (; IID[k] != NULL; ++k)
      ;
    ++k;
  }

  /* INFO chunk size */
  info->cksize = size;
  info->N = i;

  return;
}

/* Write the LIST/INFO chunk */


static int
AF_wrLIST_INFO (FILE *fp, const struct WV_LIST_INFO *info)

{
  int offs, i;

  offs = WHEAD_S (fp, info->ckid);
  offs += WHEAD_V (fp, info->cksize, DS_EL);
  offs += WHEAD_S (fp, info->listid);
  for (i = 0; i < info->N; ++i) {
    offs += WHEAD_S (fp, info->listitem[i].ckid);
    offs += WHEAD_V (fp, info->listitem[i].cksize, DS_EL);
    offs += WHEAD_SN (fp, info->listitem[i].text, info->listitem[i].cksize);
    offs += WRPAD (fp, info->listitem[i].cksize, ALIGN);
  }
  assert (8 + (int) info->cksize == offs);

  return offs;
}

/* Search for named records in an AFsp information structure */


static const char *
AF_findRec (const char *Id[], const struct AF_info *InfoS)

{
  int i;
  const char *p;

  p = NULL;
  for (i = 0; Id[i] != NULL; ++i) {
    p = AFgetInfoRec (Id[i], InfoS);
    if (p != NULL)
      break;
  }

  if (p != NULL)
    p = STtrimIws (p);

  return p;
}


/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  int AFupdHead (AFILE *AFp)

Purpose:
  Update the header for an audio file

Description:
  This routine updates the header for an audio file opened for write.

Parameters:
  <-  int AFupdHead
      Error code, 0 for no error
   -> AFILE *AFp
      Audio file pointer for the audio file

Author / revision:
  P. Kabal  Copyright (C) 2003
  $Revision: 1.8 $  $Date: 2003/05/09 01:21:36 $

-------------------------------------------------------------------------*/


int
AFupdHead (AFILE *AFp)

{
  long int pos;
  int (*AF_upd)(AFILE *AFp);

  assert (AFp->Op == FO_WO);
  assert (! AFp->Error);

  /* Set up the appropriate routine */
  AF_upd = NULL;
  switch (AFp->Ftype) {
  case FT_AU:
/*    AF_upd = AFupdAUhead;*/
    break;
  case FT_WAVE:
    AF_upd = AFupdWVhead;
    break;
  case FT_AIFF:
  case FT_AIFF_C:
/*    AF_upd = AFupdAIhead;*/
    break;
  case FT_NH:
    break;
  default:
    assert (0);
  }

  /* Save the file position, update the header, restore the position */
  if (AF_upd != NULL && FLseekable (AFp->fp)) {
    pos = AFtell (AFp->fp, &AFp->Error);
    if (! AFp->Error) {
      AFp->Error = (*AF_upd) (AFp);
      AFseek (AFp->fp, pos, &AFp->Error);
    }
  }

  if (AFp->Error)
    UTwarn ("AFupdHead: %s", AFM_UpdHeadErr);

  return AFp->Error;
}


/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  long int AFtell (FILE *fp, int *ErrCode)

Purpose:
  Determine the position in a file

Description:
  This routine returns the position in a file.  For binary files this will be
  the number of bytes from the beginning of the file.  For non-binary files,
  the returned value does not necessarily have this interpretation.  However,
  in all cases the returned value can be used as an argument to AFseek to
  return to the same position.

Parameters:
  <-  long int AFtell
      Position in the file
   -> FILE *fp
      File pointer associated with the file
  <-> int ErrCode
      Error code.  This value is set if an error is detected, but otherwise
      remains unchanged.

Author / revision:
  P. Kabal  Copyright (C) 2003
  $Revision: 1.9 $  $Date: 2003/05/09 01:11:35 $

-------------------------------------------------------------------------*/

long int
AFtell (FILE *fp, int *ErrCode)

{
  long int pos;

  pos = -1L;
  if (! *ErrCode) {
    errno = 0;
    pos = ftell (fp);
    if (pos == -1L && errno) {
/*      UTsysMsg ("AFtell: %s", AFM_NoFilePos);*/
      *ErrCode = AF_IOERR;
    }
  }

  return pos;
}


/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  AFILE *AFsetWrite (FILE *fp, int Ftype, const struct AF_write *AFw)

Purpose:
  Set up the parameters in an audio file structure for writing

Description:
  This routine checks the input parameters and puts them into a new audio file
  structure.

  This routine checks for a "sample_rate:" record in the information string.
  This record can specify a non-integer sampling frequency.

Parameters:
  <-  AFILE *AFsetWrite
      Audio file pointer for the audio file.  This routine allocates the space
      for this structure.
   -> FILE *fp
      File pointer for the audio file
   -> int Ftype
      File type: FT_NH, FT_AU, FT_WAVE, or FT_AIFF_C
   -> const struct AF_write *AFw
      Structure with data and file parameters

Author / revision:
  P. Kabal  Copyright (C) 2003
  $Revision: 1.46 $  $Date: 2003/11/03 18:49:05 $

-------------------------------------------------------------------------*/

static void
AF_setInfo (const struct AF_infoX *InfoX, struct AF_info *InfoS);
static int
AF_setWNbS (int NbS, int Format);


AFILE *
AFsetWrite (FILE *fp, int Ftype, const struct AF_write *AFw)

{
  AFILE *AFp;
  long int Start;
  int ErrCode, Nspkr;

  assert (Ftype == FT_WAVE || Ftype == FT_AU ||
	  Ftype == FT_AIFF || Ftype == FT_AIFF_C ||
	  Ftype == FT_NH);
  assert (AFw->DFormat.Format > 0 && AFw->DFormat.Format < NFD);
  assert (AFw->Nchan > 0);

/* Warnings */
  if (AFw->Sfreq <= 0.0)
    UTwarn ("AFsetWrite - %s", AFM_NPSFreq);

/* Set the start of data */
  if (FLseekable (fp)) {
    ErrCode = 0;
    Start = AFtell (fp, &ErrCode);
    if (ErrCode)
      return NULL;
  }
  else
    Start = 0;

/* Set the parameters for file access */
  AFp = (AFILE *) UTmalloc (sizeof (AFILE));
  AFp->fp = fp;
  AFp->Op = FO_WO;
  AFp->Error = AF_NOERR;
  AFp->Novld = 0;
  AFp->Sfreq = AFw->Sfreq;
  AFp->Ftype = Ftype;
  AFp->Format = AFw->DFormat.Format;
  if (AF_DL[AFw->DFormat.Format] <= 1)
    AFp->Swapb = DS_NATIVE;
  else
    AFp->Swapb = UTswapCode (AFw->DFormat.Swapb);
  AFp->NbS = AF_setWNbS (AFw->DFormat.NbS, AFp->Format);
  AFp->ScaleF = AFw->DFormat.ScaleF;
  if (AFp->ScaleF == AF_SF_DEFAULT)
    AFp->ScaleF = 1. / AF_SF[AFp->Format];
  AFp->Nchan = AFw->Nchan;

  AFp->SpkrConfig = NULL;
  Nspkr = strlen ((const char *) AFw->SpkrConfig);
  if (Nspkr > 0) {
    AFp->SpkrConfig = (unsigned char *) UTmalloc (Nspkr + 1);
    strcpy ((char *) AFp->SpkrConfig, (const char *) AFw->SpkrConfig);
  }

  AFp->Start = Start;
  AFp->Isamp = 0;
  AFp->Nsamp = 0;
  AF_setInfo (&AFw->InfoX, &AFp->InfoS);

  return AFp;
}

/* Fill in the AFsp Information structure */


static void
AF_setInfo (const struct AF_infoX *InfoX, struct AF_info *InfoS)

{
  int N;

  if (InfoX == NULL || InfoX->N <= 0) {
    InfoS->Info = NULL;
    InfoS->N = 0;
  }
  else {
    N = InfoX->N;
    if (InfoX->Info[N-1] != '\0')
      ++N;
    InfoS->Info = (char *) UTmalloc (N);
    memcpy (InfoS->Info, InfoX->Info, InfoX->N);
    InfoS->Info[N-1] = '\0';	/* Make sure there is a terminating null */
    InfoS->N = N;
  }

  return;
}

/* Set the bits/sample value */


static int
AF_setWNbS (int NbS, int Format)

{
  int NbSx;

  NbSx = 8 * AF_DL[Format];
  if (NbS < 0 || NbS > NbSx)
    UTwarn (AFMF_InvNbS, "AFsetWrite -", NbS, NbSx);
  else if (NbS != 0)
    NbSx = NbS;

  return NbSx;
}


/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  int AFseek (FILE *fp, long int pos, int *ErrCode)

Purpose:
  Seek to a position in a file

Description:
  This routine seeks to a given position in a file.

Parameters:
  <-  int AFseek
      Error status
   -> FILE *fp
      File pointer associated with the file
   -> long int pos
      Position in the file relative to the start of the file.  If pos is
      equal to AF_SEEK_END, the position is set to the end-of-file.
  <-> int ErrCode
      Error code.  This value is set if an error is detected, but otherwise
      remains unchanged.  If ErrCode is set on input, this routine is a no-op.
      ErrCode set to a NULL pointer is equivalent to ErrCode not being set.
      In this case, the error status is in the function return value.

Author / revision:
  P. Kabal  Copyright (C) 2003
  $Revision: 1.8 $  $Date: 2003/05/09 01:11:35 $

-------------------------------------------------------------------------*/


int
AFseek (FILE *fp, long int pos, int *ErrCode)

{
  int status;

  if (ErrCode == NULL)
    status = 0;
  else
    status = *ErrCode;

  if (! status) {
    if (pos == AF_SEEK_END)
      status = fseek (fp, 0L, SEEK_END);
    else
      status = fseek (fp, pos, SEEK_SET);
    if (status) {
      UTwarn ("AFseek: %s", AFM_FilePosErr);
    }

    if (status)
      status = AF_IOERR;
  }
  if (ErrCode != NULL)
    *ErrCode = status;

  return status;
}

/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  int AF_write AFpreSetWPar (int Ftype, int Dformat, long int Nchan,
                             double Sfreq, struct AF_write *AFw)

Purpose:
  Create a parameter structure for writing an audio file

Description:
  This routine creates a parameter structure for writing to an audio file.  The
  The structure is filled in with default values, values supplied as input, or
  user supplied options.

  Fields set from the input values:
    AFw->DFormat.Format
    AFw->Sfreq
    AFw->Nchan
    AFw->Ftype
  Fields set from options:
    AFw->DFormat.NbS
    AFw->SpkrConfig
    AFw->InfoX
    AFw->Nframe
  Fields preset to tentative values
    AFw->DFormat.ScaleF
    AFw->DFormat.Swapb

Parameters:
  <-  int AFpreSetWPar
      Error flag, 0 for no error
   -> int Ftype
      File type code
   -> int Dformat
      Data format code
   -> long int Nchan
      Number of channels
   -> double Sfreq
      Sampling frequency
  <-  struct AF_write *AFw
      Pointer to an audio write structure

Author / revision:
  P. Kabal  Copyright (C) 2006
  $Revision: 1.20 $  $Date: 2006/06/06 13:55:37 $

-------------------------------------------------------------------------*/





#define ICEILV(n,m)	(((n) + ((m) - 1)) / (m))	/* int n,m >= 0 */

static int
AF_checkDT (int Ftype, int Format);
static int
AF_checkIEEE (int Ftype, int Format);
static int
AF_setNbS (int Format, int NbS);
static void
AF_setSpeaker (const unsigned char *SpkrX, int Nchan,
	       unsigned char *SpkrConfig);
static struct AF_infoX
AF_genInfo (const char *Uinfo, const struct AF_write *AFw);
static int
AF_stdInfo (const struct AF_write *AFw, char *Sinfo);
static int
AF_nlNull (const char *InfoX, char *Info);


int
AFpreSetWPar (int Ftype, int Dformat, long int Nchan, double Sfreq,
	      struct AF_write *AFw)

{
  struct AF_opt *AFopt;

  AFopt = AFoptions ();

/* File type and data format */
  AFw->Ftype = Ftype;
  AFw->DFormat.Format = Dformat;

/* Set file data parameters */
  AFw->Sfreq = Sfreq;
  AFw->DFormat.ScaleF = AF_SF_DEFAULT;
  AFw->DFormat.Swapb = DS_NATIVE;		    /* Tentative */
  AFw->DFormat.NbS = AF_setNbS (AFw->DFormat.Format, AFopt->NbS);
  AFw->Nchan = Nchan;
  AFw->Nframe = AFopt->Nframe;	/* From the options structure */
  AF_setSpeaker (AFopt->SpkrConfig, Nchan, AFw->SpkrConfig);

/* Info string:
     The space for the info string is allocated within AF_genInfo.  It is
     reclaimed each time AF_genInfo is called.
  */
  AFw->InfoX = AF_genInfo (AFopt->Uinfo, AFw);

/* Error Checks */
  if (AF_checkDT (AFw->Ftype, AFw->DFormat.Format))
    return 1;
  if (AF_checkIEEE (AFw->Ftype, AFw->DFormat.Format))
    return 1;

  if (AFw->Nchan <= 0L) {
    UTwarn ("AFopnWrite - %s: %ld", AFM_BadNChan, AFw->Nchan);
    return 1;
  }
  if (AFw->Nframe != AF_NFRAME_UNDEF && AFw->Nframe < 0L) {
    UTwarn ("AFopnWrite - %s: %ld", AFM_BadNFrame, AFw->Nframe);
    return 1;
  }

  return 0;
}

/* Check for file type / data type compatibility */


static int
AF_checkDT (int Ftype, int Format)

{
  int ErrCode;

/* undef mulaw  Alaw uint8  int8 int16 int24 int32 flt32 flt64 text */
  static const int DT_AU[NFD] = {
     0,    1,    1,    0,    1,    1,    1,    1,    1,    1,    0};
  static const int DT_WAVE[NFD] = {
     0,    1,    1,    1,    0,    1,    1,    1,    1,    1,    0};
  static const int DT_AIFF_C[NFD] = {
     0,    1,    1,    0,    1,    1,    1,    1,    1,    1,    0};
  static const int DT_AIFF[NFD] = {
     0,    0,    0,    0,    1,    1,    1,    1,    0,    0,    0};
  static const int DT_NH[NFD] = {
     0,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1};

  ErrCode = 0;
  if (Format < 1 || Format >= NFD) {
    UTwarn ("AFopnWrite - %s: \"%d\"", AFM_BadDataC, Format);
    ErrCode = 1;
    return ErrCode;
  }

  switch (Ftype) {
  case FTW_AU:
    if (DT_AU[Format] == 0) {
      UTwarn ("AFopnWrite - %s: \"%s\"", AFM_AU_UnsData, AF_DTN[Format]);
      ErrCode = 1;
    }
    break;
  case FTW_WAVE:
  case FTW_WAVE_NOEX:
    if (DT_WAVE[Format] == 0) {
      UTwarn ("AFopnWrite - %s: \"%s\"", AFM_WV_UnsData, AF_DTN[Format]);
      ErrCode = 1;
    }
    break;
  case FTW_AIFF_C:
    if (DT_AIFF_C[Format] == 0) {
      UTwarn ("AFopnWrite - %s: \"%s\"", AFM_AIFF_UnsData, AF_DTN[Format]);
      ErrCode = 1;
    }
    break;
  case FTW_AIFF:
    if (DT_AIFF[Format] == 0) {
      UTwarn ("AFopnWrite - %s: \"%s\"", AFM_AIFF_UnsData, AF_DTN[Format]);
      ErrCode = 1;
    }
    break;
  case FTW_NH_EB:
  case FTW_NH_EL:
  case FTW_NH_NATIVE:
  case FTW_NH_SWAP:
    if (DT_NH[Format] == 0) {
      UTwarn ("AFopnWrite - %s: \"%s\"", AFM_NH_UnsData, AF_DTN[Format]);
      ErrCode = 1;
    }
    break;
  default:
    UTwarn ("AFopnWrite - %s: %d", AFM_BadFTypeC, Ftype);
    ErrCode = 1;
    break;
  }
  return ErrCode;
}

/* Check for IEEE data format */

static int
AF_checkIEEE (int Ftype, int Format)

{
  int ErrCode;

  ErrCode = 0;
  switch (Ftype) {
  case FTW_AU:
    if ((Format == FD_FLOAT32 || Format == FD_FLOAT64) && ! UTcheckIEEE ()) {
      UTwarn ("AFwrAUhead - %s", AFM_NoIEEE);
      ErrCode = 1;
    }
    break;
  case FTW_WAVE:
  case FTW_WAVE_NOEX:
    if ((Format == FD_FLOAT32 || Format == FD_FLOAT64) && ! UTcheckIEEE ()) {
      UTwarn ("AFwrWVhead - %s", AFM_NoIEEE);
      ErrCode = 1;
    }
    break;
  case FTW_AIFF_C:
  case FTW_AIFF:
    if ((Format == FD_FLOAT32 || Format == FD_FLOAT64) && ! UTcheckIEEE ()) {
      UTwarn ("AFwrAIhead - %s", AFM_NoIEEE);
      ErrCode = 1;
    }
    break;
  default:
    break;
  }

  return ErrCode;
}

/* Set the number of bits per sample */


static int
AF_setNbS (int Format, int NbS)

{
  int Res;

  Res = 8 * AF_DL[Format];
  if (NbS == 0)
    NbS = Res;

  /* Res == 0 for text files */
  if (Res > 0 && (NbS <= 0 || NbS > Res)) {
    UTwarn (AFMF_InvNbS, "AFopnWrite -", NbS, Res);
    NbS = Res;
  }
  else if (Res == 0 && NbS != 0) {
    UTwarn ("AFopnWrite - %s", AFM_BadNbS);
    NbS = Res;
  }

  /* Warn and reset NbS for inappropriate formats */
  if (NbS != Res &&
      ! (Format == FD_UINT8 || Format == FD_INT8 || Format == FD_INT16 ||
	 Format == FD_INT24 || Format == FD_INT32)) {
    UTwarn ("AFopnWrite - %s", AFM_InaNbS);
    NbS = Res;
  }

  return NbS;
}

/* Set the speaker configuration */


static void
AF_setSpeaker (const unsigned char *SpkrX, int Nchan,
	       unsigned char *SpkrConfig)

{
  int Nspkr;

  SpkrConfig[0] = AF_X_SPKR;
  if (SpkrX != NULL) {
    SpkrConfig[0] = '\0';
    Nspkr = strlen ((const char *) SpkrX);
    if (Nspkr > Nchan)
      Nspkr = Nchan;
    if (Nspkr > AF_MAXN_SPKR)
      UTwarn ("AFopnWrite - %s", AFM_XSpkr);
    else {
      strncpy ((char *) SpkrConfig, (const char *) SpkrX, Nspkr);
      SpkrConfig[Nspkr] = '\0';
    }
  }

  return;
}

#define MAX_SINFO	320


static struct AF_infoX
AF_genInfo (const char *Uinfo, const struct AF_write *AFw)

{
  char Sinfo[MAX_SINFO];
  int Nc, ns, nu;
  static struct AF_infoX InfoX = {NULL, 0, 0};

  if (Uinfo == NULL || Uinfo[0] == '\n' ||
      (Uinfo[0] == '\\' && Uinfo[1] == 'n'))
    ns = AF_stdInfo (AFw, Sinfo);
  else {
    ns = 0;
    Sinfo[0] = '\0';
  }

/* Size of the user specified information */
  if (Uinfo != NULL)
    nu = strlen (Uinfo);
  else
    nu = 0;

/* Allocate storage */
  Nc = nu + ns + 1;	/* Leave room for a null */
  UTfree ((void *) InfoX.Info);
  InfoX.Info = (char *) UTmalloc (Nc);
  InfoX.Nmax = Nc;

/* Form the output string.
   - Uinfo == NULL
       Info <= Sinfo
   - Uinfo empty ("")
       Info <= empty
   - Uinfo starts with '\n'
       Info <= Sinfo + Uinfo
       In this case, the leading '\n' in Uinfo overlays the trailing null
       in Sinfo.
   - Uinfo does not start with '\n'
       Info <= Uinfo
*/
  strcpy (InfoX.Info, Sinfo);
  if (Uinfo != NULL)
    strcpy (&InfoX.Info[ns], Uinfo);

/* Change newlines to nulls */
  InfoX.N = AF_nlNull (InfoX.Info, InfoX.Info);

/* Return a pointer to the information records */
  return InfoX;
}

/* Generate the standard information string */
/* "date: xxx"
   + "\n" + "sample_rate: xxx" (only if non-integer)
   + "\n" + "user: xxx"
   + "\n" + "bits_per_sample: xx" (only if not equal to full precision)
   + "\n" + "loudspeakers: xxx" (only if specified)

   Lines are separated by newline characters.  The overall string is null
   terminated.  The returned function value is the number of characters not
   including the terminating null.
*/

#define NSPKR_EXTRA 0
#define NC_SPKR ((AF_NC_SPKR + 1) * (AF_MAXN_SPKR + NSPKR_EXTRA) - 1)

static int
AF_stdInfo (const struct AF_write *AFw, char *Sinfo)

{
  int N, NbS, Res;
  char *p;
  double Sfreq;
  const unsigned char *SpkrConfig;
  char SpkrNames[NC_SPKR+1];

/* String lengths
  Record          Fixed    Variable
  Name            Length   Max       Total
   Date             6        40       46
   Program Name    10        40       50
   Sampling Freq.  14        15       29
   Number of Bits  19         4       23
   Loudspeakers    15        76       91
   null             1                  1
                                     ---
                                     240
*/
  N = sprintf (Sinfo, "date: %.40s", UTdate(3));

  p = UTgetProg ();
  if (*p != '\0')
    N += sprintf (&Sinfo[N], "\nprogram: %.40s", p);

  Sfreq = AFw->Sfreq;
  if (Sfreq > 0.0 && Sfreq != floor (Sfreq))
    N += sprintf (&Sinfo[N], "\nsample_rate: %.7g", Sfreq);

  Res = 8 * AF_DL[AFw->DFormat.Format];
  NbS = AFw->DFormat.NbS;
  if (NbS != 0 && NbS != Res)
    N += sprintf (&Sinfo[N], "\nbits_per_sample: %d/%d", NbS, Res);

  SpkrConfig = AFw->SpkrConfig;
  if (SpkrConfig[0] != AF_X_SPKR) {
    AFspeakerNames (AFw->Nchan, SpkrConfig, NSPKR_EXTRA, SpkrNames);
    N += sprintf (&Sinfo[N], "\nloudspeakers: %s", SpkrNames);
  }

  assert (N < MAX_SINFO);

  return N;
}

/* Change newlines to nulls (unless escaped) */
/* - Escaped characters.  In Unix it is relatively easy to get a newline
     into a command line string.  In Windows, one can use the escape
     sequences instead.
     - <nl>  => <nul>  newline
     - \n    => <nul>  newline
     - \r    => <nl>   escaped newline
     - <cr>  => <nl>   escaped newline
     - \<nl> => <nl>   escaped newline
   - If the length of the input string is not zero and the last character in
     the input string was not a newline (changed to a null), a null character
     is added to the string.
   - The count of the number of characters includes the null character at the
     end of the string.
   - If the number of input characters is zero, the returned value is zero.
*/


static int
AF_nlNull (const char *InfoX, char *Info)

{
  int i, n;

  for (i = 0, n = 0; InfoX[i] != '\0'; ++i, ++n) {
    if (InfoX[i] == '\n')
      Info[n] = '\0';
    else if (InfoX[i] == '\\' && InfoX[i+1] == 'n') {
      Info[n] = '\0';
      ++i;
    }
    else if (InfoX[i] == '\\' && InfoX[i+1] == 'r') {
      Info[n] = '\n';
      ++i;
    }
    else if (InfoX[i] == '\r')
      Info[n] = '\n';
    else if (InfoX[i] == '\\' && InfoX[i+1] == '\n') {
      Info[n] = '\n';
      ++i;
    }
    else
      Info[n] = InfoX[i];
  }

  /* Add a terminating null (included in the character count) if one is not
     present */
  if (n > 0 && Info[n-1] != '\0') {
    Info[n] = '\0';
    ++n;
  }

  return n;
}

/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  AFILE *AFopenWrite (const char Fname[], int Fformat, long int Nchan,
                      double Sfreq, FILE *fpinfo)

Purpose:
  Open an audio file for writing

Description:
  Use AFopnWrite instead. The difference between the routines is the scaling
  of the data.

Parameters:
  <-  AFILE *AFopenWrite
      Audio file pointer for the audio file
   -> const char Fname[]
      Character string specifying the file name.  The file name "-" means
      standard output.
   -> int Fformat
      Audio file format code, evaluated as the sum of a data format code and a
      file type,
        Fformat = Dformat + 256 * Ftype
      Dformat: data format
        FD_MULAW8  = 1,  mu-law 8-bit data
        FD_ALAW8   = 2,  A-law 8-bit data
        FD_UINT8   = 3,  offset-binary 8-bit integer data
        FD_INT8    = 4,  two's-complement 8-bit integer data
        FD_INT16   = 5,  two's-complement 16-bit integer data
        FD_INT24   = 6,  two's-complement 24-bit integer data
        FD_INT32   = 7,  two's-complement 32-bit integer data
        FD_FLOAT32 = 8,  32-bit floating-point data
        FD_FLOAT64 = 9,  64-bit floating-point data
        FD_TEXT    = 10,  text data
      Ftype: output file type
        FTW_AU        = 1,      AU audio file
        FTW_WAVE      = 2,      WAVE file (auto extensible format)
        FTW_WAVE_NOEX = 2 + 16, WAVE file (no extensible format)
        FTW_AIFF_C    = 3,      AIFF-C sound file
        FTW_NH_EB     = 4 +  0, Headerless file (big-endian byte order)
        FTW_NH_EL     = 4 + 16, Headerless file (little-endian byte order)
        FTW_NH_NATIVE = 4 + 32, Headerless file (native byte order)
        FTW_NH_SWAP   = 4 + 48, Headerless file (byte-swapped byte order)
	FTW_AIFF      = 5,      AIFF sound file
   -> long int Nchan
      Number of channels
   -> double Sfreq
      Sampling frequency
   -> FILE *fpinfo
      File pointer for printing audio file information.  If fpinfo is not NULL,
      information about the audio file is printed on the stream selected by
      fpinfo.

Author / revision:
  P. Kabal  Copyright (C) 2003
  $Revision: 1.79 $  $Date: 2003/05/13 01:51:30 $

-------------------------------------------------------------------------*/

#define FTW_FTYPE_MOD	256
#define FTW_dformat(code)	((code) % FTW_FTYPE_MOD)
#define FTW_ftype(code)		((code) / FTW_FTYPE_MOD)


AFILE *
AFopenWrite (const char Fname[], int Fformat, long int Nchan, double Sfreq,
	     FILE *fpinfo)

{
  AFILE *AFp;
  int Ftype, Dformat;

  Ftype = FTW_ftype (Fformat);
  Dformat = FTW_dformat (Fformat);
  AFp = AFopnWrite (Fname, Ftype, Dformat, Nchan, Sfreq, fpinfo);
  AFp->ScaleF = AFp->ScaleF / 32768.;

  return AFp;
}


/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  AFILE *AFopnWrite (const char Fname[], int Ftype, int Dtype, long int Nchan,
                     double Sfreq, FILE *fpinfo)

Purpose:
  Open an audio file for writing

Description:
  This routine opens an audio file for writing.  This routine sets up the audio
  file parameters to write data of the given format to the audio file.  After
  writing data to the file, the routine AFclose should be called to update the
  file header information and close the file.

  By default, information consisting of the date and the program creating the
  file (see UTsetProg) is written to the audio file header or trailer.  The
  routine AFsetInfo can be called before calling this routine to specify
  additional information to be written to the file.

  This routine can write AU audio files, AIFF-C files, WAVE files, and
  headerless audio files.

  AU audio file:
    8-bit mu-law, 8-bit A-law, 8/16/24/32-bit integer, and 32/64-bit IEEE
   floating-point data formats are supported.
  WAVE file:
    8-bit mu-law, 8-bit A-law, offset-binary 8-bit integer, 16/24/32-bit
    integer, and 32/64-bit IEEE floating-point, data formats are supported.
  AIFF sound file:
    8/16/24/32-bit integer data formats are supported.
  AIFF-C sound file:
    8-bit mu-law, 8-bit A-law, 8/16/24/32-bit integer, and 32/64-bit IEEE
    floating-point data formats are supported.
  Headerless file:
    8-bit mu-law, 8-bit A-law, offset-binary 8-bit integer, 8/16/24/32-bit
    integer, 32/64-bit floating-point, and text data formats are supported.
    A text format file has the data in character form, one value to a line.

  For the fixed point file data representations, input values in the range
  [-1,  +1) ([-.5, +0.5) for 8-bit data) will be converted without clipping.
  The scaling factor shown below is applied to the data before they are
  written to the file.
     data format   scaling factor   file data values
    8-bit mu-law    32,768         [-32,124, +32,124]
    8-bit A-law     32,768         [-32,256, +32,256]
    8-bit integer   256            [-128, 127]
    16-bit integer  32,768         [-32,768, +32,767]
    24-bit integer  8,388,608      [-8,388,608, +8,388,607]
    32-bit integer  2,147,483,648  [-2,147,483,648, 2,147,483,647]
  For files containing floating-point data, values are scaled by unity before
  being written to the file.

Parameters:
  <-  AFILE *AFopnWrite
      Audio file pointer for the audio file
   -> const char Fname[]
      Character string specifying the file name.  The file name "-" means
      standard output.
   -> int Ftype
      Output file type code
        FTW_AU        = 1,      AU audio file
        FTW_WAVE      = 2,      WAVE file (auto extensible format)
        FTW_WAVE_NOEX = 2 + 16, WAVE file (no extensible format)
        FTW_AIFF_C    = 3,      AIFF-C sound file
        FTW_NH_EB     = 4 +  0, Headerless file (big-endian byte order)
        FTW_NH_EL     = 4 + 16, Headerless file (little-endian byte order)
        FTW_NH_NATIVE = 4 + 32, Headerless file (native byte order)
        FTW_NH_SWAP   = 4 + 48, Headerless file (byte-swapped byte order)
	FTW_AIFF      = 5,      AIFF sound file
   -> int Dformat
      Data format code
        FD_MULAW8  = 1,  mu-law 8-bit data
        FD_ALAW8   = 2,  A-law 8-bit data
        FD_UINT8   = 3,  offset-binary 8-bit integer data
        FD_INT8    = 4,  two's-complement 8-bit integer data
        FD_INT16   = 5,  two's-complement 16-bit integer data
        FD_INT24   = 6,  two's-complement 24-bit integer data
        FD_INT32   = 7,  two's-complement 32-bit integer data
        FD_FLOAT32 = 8,  32-bit floating-point data
        FD_FLOAT64 = 9,  64-bit floating-point data
        FD_TEXT    = 10,  text data
   -> long int Nchan
      Number of channels
   -> double Sfreq
      Sampling frequency
   -> FILE *fpinfo
      File pointer for printing audio file information.  If fpinfo is not NULL,
      information about the audio file is printed on the stream selected by
      fpinfo.

Author / revision:
  P. Kabal  Copyright (C) 2006
  $Revision: 1.7 $  $Date: 2006/06/06 13:53:15 $

-------------------------------------------------------------------------*/
#define ERR_MSG	0
#define SYS_MSG	1

static void
AF_error (const char Fname[], int sysFlag);
static FILE*
AF_open_write (const char Fname[], int Dformat);

/* If ErrorHalt is clear, execution continues after an error */


AFILE *
AFopnWrite (const char Fname[], int Ftype, int Dformat, long int Nchan,
	    double Sfreq,  FILE *fpinfo)

{
    AFILE *AFp=NULL;
    FILE *fp;
    struct AF_write AFw;

    /* Set up and check the audio file parameters */
    if (AFpreSetWPar (Ftype, Dformat, Nchan, Sfreq, &AFw)) {
	AF_error (Fname, ERR_MSG);
	return NULL;
    }

    /* Open the file for writing */
    fp = AF_open_write (Fname, AFw.DFormat.Format);
    if (fp == NULL)
	return NULL;

    switch (AFw.Ftype) {
    case FTW_AU:
	/*    AFp = AFwrAUhead (fp, &AFw);*/
	break;
    case FTW_WAVE:
    case FTW_WAVE_NOEX:
	AFp = AFwrWVhead (fp, &AFw);
	break;
    case FTW_AIFF:
    case FTW_AIFF_C:
	/*    AFp = AFwrAIhead (fp, &AFw);*/
	break;
    case FTW_NH_EB:
    case FTW_NH_EL:
    case FTW_NH_NATIVE:
    case FTW_NH_SWAP:
	/*    AFp = AFsetNHwrite (fp, &AFw);*/
	break;
    default:
	AFp = NULL;
	break;
    }

    /* Error messages */
    if (AFp == NULL) {
	fclose (fp);
	AF_error (Fname, ERR_MSG);
	return NULL;
    }

/* Reset options */
/*  (void) AFresetOptions (AF_OPT_OUTPUT);*/

/* Print the header information */
/*  AFprAFpar (AFp, Fname, fpinfo);*/

    return AFp;
}

/* Open a file for writing */


#if (SY_OS == SY_OS_WINDOWS || SY_OS == SY_OS_CYGWIN)
#  include <io.h>	/* setmode */
#  include <fcntl.h>	/* IO_BINARY */
#endif

static FILE *
AF_open_write (const char Fname[], int Dformat)

{
  FILE *fp;

  if (strcmp (Fname, "-") == 0) {

    /* Output to standard output */
    fp = stdout;
#if (SY_OS == SY_OS_WINDOWS || SY_OS == SY_OS_CYGWIN)
    if (Dformat == FD_TEXT) {

	if (setmode (fileno(fp), O_TEXT) == -1)
	UTwarn ("AFopnWrite - %s", AFM_NoWTextMode);

    }
    else {

	if (setmode (fileno(fp), O_BINARY) == -1)
	UTwarn ("AFopnWrite - %s", AFM_NoWBinMode);

  }
#endif

  }
  else {

    /* Output to a file */
    if (Dformat == FD_TEXT)
      fp = fopen (Fname, "w");
    else
      fp = fopen (Fname, "wb");
  }

  if (fp == NULL) {
    AF_error (Fname, SYS_MSG);
    return NULL;
  }

  return fp;
}

/* Print an error message with the file name */


static void
AF_error (const char Fname[], int sysFlag)

{
  const char *fn;
  struct AF_opt *AFopt;

  if (strcmp (Fname, "-") == 0)
    fn = "<stdout>";
  else
    fn = Fname;


  if (sysFlag)
    /*UTsysMsg ("AFopnWrite: %s \"%s\"", AFM_OpenWErr, fn);*/;
  else
    UTwarn ("AFopnWrite: %s \"%s\"", AFM_OpenWErr, fn);

  AFopt = AFoptions ();
  if (AFopt->ErrorHalt)
    exit (EXIT_FAILURE);

  /* Reset options */
  (void) AFresetOptions (AF_OPT_OUTPUT);

  return;
}


/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  struct AF_opt *AFoptions (void)
  struct AF_opt *AFresetOptions (unsigned int Cat)

Purpose:
  Get a pointer to the audio file options stucture

Description:
  This routine returns a pointer to the audio file options structure.  The
  options structure is defined in the header file AFpar.h.  Routines are
  provided to set and access some of the fields.

  Program Behaviour:
  Error Control (ErrorHalt):
    The top level AF routines (AFopnRead, AFopnWrite, AFdReadData, etc.)
    normally print an error message and halt when an error is passed from a
    lower level routine.  Optionally, errors can also be signalled by a return
    value such as NULL file pointer or other error flag.  The error options are
    as follows.
      0 - Continue on error, returning an error code
      1 - Stop on error (default behaviour)

  Options for Input Files:
  Number of samples (NsampND):
    The number of samples in a speech file opened for read is normally
    returned by AFopnRead.  In some cases, the number of samples can only
    be determined from the actual number of records in the file.  This
    determination is not possible for input streams which are not random
    access.  For such cases, the number of samples value can be set to a value
    which indicates that this value is not available.  The NsampND options are
    as follows.
      0 - Always return the number of samples in the file.
	  An error condition exists if the number of samples cannot be
	  determined (default behaviour).
      1 - Return the number of samples if the file is seekable or the
	  number of samples is specified in the file header.  Otherwise
	  return a value indicating that this value is undefined
	  (AF_NSAMP_UNDEF).
  Random access requirement (RAccess):
    Some programs require the ability to reposition the file to an earlier
    position (to reread data, for instance).  This parameter controls whether
    a file will be opened if it does not support random access (viz. file
    streams from a pipe are not random access).
      0 - Allow input files which are not random access (default behaviour).
	  If an attempt is made to reposition such a file during a read
	  operation, an error condition will occur.
      1 - Do not open input audio files which are not random access.
  Input File Type (FtypeI):
    Input audio file type (default FT_AUTO).  This parameter can be set with
    symbolic parameters using the routine AFsetFileType.
  File parameters for headerless files (struct AF_NHpar):
    Parameters for noheader audio files.  These parameters can be set with
    symbolic values using the routine AFsetNHpar.

  Options for Output Files:
  Number of frames (long int Nframe):
    This value specifies the number of frames in an output audio file.  This
    value is used to set the number of frames information in a file header
    before opening the file.  It is used for non-random access output files
    which cannot update the header after writing the audio data.  If this value
    is AF_NFRAME_UNDEF, the number of frames is not pre-defined.  For some
    types of output files, this setting means that the output file must be
    random acesss.
  Number of bits per sample (int NbS):
    This value affects a corresponding value in some file headers.  Each
    sample is in a container with a size which is a multiple of 8.  The
    container size is determined from the data format.  The number of bits per
    sample indicates that some of the bits are not significant.  The number of
    bits per sample rounded up to a multiple of eight must equal the container
    size.  The default value of zero indicates that the number of bits is the
    same as the container size for the data.
  Speaker configuration (char *SpkrConfig):
    This null-terminated string contains the speaker location codes.  This
    string can be set with symbolic values using the routine AFsetSpkr.  This
    string is allocated as needed.  It can be deallocated by invoking
    AFsetSpeaker ("").
  Header information string (char *Uinfo):
    This string is the information string supplied by the user via the routine
    AFsetInfo.  This string is allocated as needed.  It can be deallocated by
    invoking AFsetInfo (NULL).

Parameters:
  <-  struct AF_opt *
      Pointer to the option structure
   -> unsigned int Cat
      For AFresetOptions: Categories of options to be reset.  Cat is the sum
      of the following sub-categories:
        AF_OPT_GENERAL - General options (ErrorHalt)
        AF_OPT_INPUT   - Input file options
        AF_OPT_OUTPUT  - Output file options

Author / revision:
  P. Kabal  Copyright (C) 2003
  $Revision: 1.19 $  $Date: 2003/11/03 18:52:00 $

-------------------------------------------------------------------------*/

/* Default options */
#define AF_NHPAR_DEFAULT \
	{ FD_UNDEF, 0L, AF_SFREQ_DEFAULT, DS_NATIVE, 1L, AF_SF_DEFAULT }
#define AF_OPT_DEFAULT \
	{ 1, \
	  0, 0, FT_AUTO , AF_NHPAR_DEFAULT, \
	  AF_NFRAME_UNDEF, 0, NULL, NULL }

/* Initialize to default values */
static struct AF_opt AFopt = AF_OPT_DEFAULT;
static struct AF_NHpar NHpar_default = AF_NHPAR_DEFAULT;


struct AF_opt *
AFoptions (void)

{
  return &AFopt;
}

/* Reset the options */


struct AF_opt *
AFresetOptions (unsigned int Cat)

{
  if (Cat & AF_OPT_GENERAL) {
    AFopt.ErrorHalt = 1;
  }

  if (Cat & AF_OPT_INPUT) {
    AFopt.NsampND = 0;
    AFopt.RAccess = 0;
    AFopt.FtypeI = FT_AUTO;
    AFopt.NHpar = NHpar_default;
  }

  if (Cat & AF_OPT_OUTPUT) {
    AFopt.Nframe = AF_NFRAME_UNDEF;
    AFopt.NbS = 0;
    UTfree (AFopt.SpkrConfig);
    AFopt.SpkrConfig = NULL;
    UTfree (AFopt.Uinfo);
    AFopt.Uinfo = NULL;
  }

  return &AFopt;
}


/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  const char *AFgetInfoRec (const char name[], const struct AF_info *InfoS)

Purpose:
  Find a named record in an AFsp information structure

Description:
  This routine searches through records in an AFsp information structure,
  looking for a match to a record with a given name.  A pointer to the data
  value for that record is returned.

  The records are separated by nulls.  If the information string does not have
  a terminating null, the last unterminated record will not be checked.

  AFsp Information Records:
    The header information records are separated by null characters.  The name
    field is the first part of the record and the value field is the remainder.
    Standard names are terminated by a ':' character, which effectively
    delimits the name field from the value field.

Parameters:
  <-  const char *AFgetInfoRec
      Pointer to a null terminated value string in the information string
   -> const char name[]
      Record name to be matched.  An empty name will match the first record.
   -> const struct AF_info *InfoS
      AFsp information structure

Author / revision:
  P. Kabal  Copyright (C) 2003
  $Revision: 1.26 $  $Date: 2003/11/04 12:42:08 $

-------------------------------------------------------------------------*/



const char *
AFgetInfoRec (const char name[], const struct AF_info *InfoS)

{
  const char *pst;
  const char *h_end;
  const char *r_term;
  int lenn;

  if (InfoS == NULL || InfoS->Info == NULL || InfoS->N <= 0)
    return NULL;

  h_end = &InfoS->Info[InfoS->N];
  pst = InfoS->Info;
  lenn = strlen (name);
  while (pst < h_end) {

    /* Find the record terminator */
    r_term = (char *) memchr (pst, '\0', h_end - pst);
    if (r_term == NULL)
      break;

    /* Check for a name match */
    if (r_term - pst >= lenn && memcmp (pst, name, lenn) == 0 ) {
      return (pst + lenn);
    }
    pst = r_term + 1;
  }

/* No match to the name */
  return NULL;
}

/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  int AFfWrI1 (AFILE *AFp, const float Dbuff[], int Nval)
  int AFfWrI2 (AFILE *AFp, const float Dbuff[], int Nval)
  int AFfWrI3 (AFILE *AFp, const float Dbuff[], int Nval)
  int AFfWrI4 (AFILE *AFp, const float Dbuff[], int Nval)

Purpose:
  Write 8-bit integer data to an audio file (float input values)
  Write 16-bit integer data to an audio file (float input values)
  Write 24-bit integer data to an audio file (float input values)
  Write 32-bit integer data to an audio file (float input values)

Description:
  This routine writes a specified number of integer samples to an audio file.
  The input to this routine is a buffer of float values.

Parameters:
  <-  int AFfWrIx
      Number of samples written.  If this value is less than Nval, an error
      has occurred.
   -> AFILE *AFp
      Audio file pointer for an audio file opened by AFopnWrite
   -> const float Dbuff[]
      Array of floats with the samples to be written
   -> int Nval
      Number of samples to be written

Author / revision:
  P. Kabal  Copyright (C) 2003
  $Revision: 1.2 $  $Date: 2003/05/09 01:11:34 $

-------------------------------------------------------------------------*/


#define LW1		FDL_INT8
#define LW2		FDL_INT16
#define LW3		FDL_INT24
#define LW4		FDL_INT32
#define MINV(a, b)	(((a) < (b)) ? (a) : (b))
#define NBBUF		8192
#define INT3_MAX	8388607
#define INT3_MIN	-8388608

#define FWRITE(buf,size,nv,fp)	(int) fwrite ((const char *) buf, \
					      (size_t) size, (size_t) nv, fp)


int
AFfWrI1 (AFILE *AFp, const float Dbuff[], int Nval)

{
  int is, N, Nw, i;
  int1_t Buf[NBBUF/LW1];
  double g, Dv;

/* Write data to the audio file */
  is = 0;
  g = AFp->ScaleF;
  while (is < Nval) {
    N = MINV (NBBUF / LW1, Nval - is);
    for (i = 0; i < N; ++i) {
      Dv = g * Dbuff[i+is];
      if (Dv >= 0.0) {
	Dv += 0.5;
	if (Dv >= INT1_MAX + 1) {
	  ++AFp->Novld;
	  Dv = INT1_MAX;
	}
      }
      else {
	Dv += -0.5;
	if (Dv <= INT1_MIN - 1) {
	  ++AFp->Novld;
	  Dv = INT1_MIN;
	}
      }
      Buf[i] = (int1_t) Dv;
    }
    Nw = FWRITE (Buf, LW1, N, AFp->fp);
    is += Nw;
    if (Nw < N)
      break;
  }

  return is;
}

int
AFfWrI2 (AFILE *AFp, const float Dbuff[], int Nval)

{
  int is, N, Nw, i;
  int2_t Buf[NBBUF/LW2];
  double g, Dv;
  unsigned char *cp;
  unsigned char t;

/* Write data to the audio file */
  is = 0;
  g = AFp->ScaleF;
  while (is < Nval) {
    N = MINV (NBBUF / LW2, Nval - is);
    for (i = 0; i < N; ++i) {
      Dv = g * Dbuff[i+is];
      if (Dv >= 0.0) {
	Dv += 0.5;
	if (Dv >= INT2_MAX + 1) {
	  ++AFp->Novld;
	  Dv = INT2_MAX;
	}
      }
      else {
	Dv += -0.5;
	if (Dv <= INT2_MIN - 1) {
	  ++AFp->Novld;
	  Dv = INT2_MIN;
	}
      }
      Buf[i] = (int2_t) Dv;
      if (AFp->Swapb == DS_SWAP) {
	cp = (unsigned char *) &Buf[i];
	t = cp[1]; cp[1] = cp[0]; cp[0] = t;
      }
    }
    Nw = FWRITE (Buf, LW2, N, AFp->fp);
    is += Nw;
    if (Nw < N)
      break;
  }

  return is;
}

int
AFfWrI3 (AFILE *AFp, const float Dbuff[], int Nval)

{
  int is, N, Nw, i, j, Hbo;
  int4_t Iv;
  unsigned char Buf[NBBUF];
  double g, Dv;
  unsigned char *cp;

/* Write data to the audio file */
  Hbo = UTbyteOrder ();
  cp = (unsigned char *) &Iv;
  is = 0;
  g = AFp->ScaleF;
  while (is < Nval) {
    N = MINV (NBBUF / LW3, Nval - is);
    for (i = 0, j = is; i < LW3*N; i += LW3, ++j) {
      Dv = g * Dbuff[j];
      if (Dv >= 0.0) {
	Dv += 0.5;
	if (Dv >= INT3_MAX + 1) {
	  ++AFp->Novld;
	  Dv = INT3_MAX;
	}
      }
      else {
	Dv += -0.5;
	if (Dv <= INT3_MIN - 1) {
	  ++AFp->Novld;
	  Dv = INT3_MIN;
	}
      }
      if (Hbo == DS_EL)
	Iv = (int4_t) Dv;		/* DS_EL:  X  2  1  0  */
      else				/*        MSB      LSB */
	Iv = 256 * ((int4_t) Dv);       /* DS_EB:  0  1  2  X  */
      if (AFp->Swapb == DS_SWAP) {
	Buf[i] = cp[2];
	Buf[i+1] = cp[1];
	Buf[i+2] = cp[0];
      }
      else {
	Buf[i] = cp[0];
	Buf[i+1] = cp[1];
	Buf[i+2] = cp[2];
      }
    }

    Nw = FWRITE (Buf, LW3, N, AFp->fp);
    is += Nw;
    if (Nw < N)
      break;
  }

  return is;
}

int
AFfWrI4 (AFILE *AFp, const float Dbuff[], int Nval)

{
  int is, N, Nw, i;
  int4_t Buf[NBBUF/LW4];
  double g, Dv;
  unsigned char *cp;
  unsigned char t;

/* Write data to the audio file */
  is = 0;
  g = AFp->ScaleF;
  while (is < Nval) {
    N = MINV (NBBUF / LW4, Nval - is);
    for (i = 0; i < N; ++i) {
      Dv = g * Dbuff[i+is];
      if (Dv >= 0.0) {
	Dv += 0.5;
	if (Dv >= (double) INT4_MAX + 1.) {
	  ++AFp->Novld;
	  Dv = INT4_MAX;
	}
      }
      else {
	Dv += -0.5;
	if (Dv <= (double) (INT4_MIN) - 1.) {
	  ++AFp->Novld;
	  Dv = INT4_MIN;
	}
      }
      Buf[i] = (int4_t) Dv;
      if (AFp->Swapb == DS_SWAP) {
	cp = (unsigned char *) &Buf[i];
	t = cp[3]; cp[3] = cp[0]; cp[0] = t;
	t = cp[2]; cp[2] = cp[1]; cp[1] = t;
      }
    }
    Nw = FWRITE (Buf, LW4, N, AFp->fp);
    is += Nw;
    if (Nw < N)
      break;
  }

  return is;
}

/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  int AFfWriteData (AFILE *AFp, const float Dbuff[], int Nval)

Purpose:
  Write data to an audio file (float input values)

Description:
  This routine writes a specified number of samples to an audio file.  The
  float input data is converted to the audio file data representation.  The
  file data representation is set on opening the file with routine AFopnWrite.
  This routine writes data sequentially to the file.  A warning message is
  printed if the input values exceed the dynamic range of the file data
  representation.

Parameters:
  <-  int AFfWriteData
      Number of samples written
   -> AFILE *AFp
      Audio file pointer for an audio file opened by AFopnWrite
   -> const float Dbuff[]
      Array of floats with Nval samples
   -> int Nval
      Number of samples to be written

Author / revision:
  P. Kabal  Copyright (C) 2003
  $Revision: 1.3 $  $Date: 2003/05/13 01:50:31 $

-------------------------------------------------------------------------*/


#define MAXV(a, b)	(((a) > (b)) ? (a) : (b))

/* Writing routines */
static int
(*AF_Write[NFD])(AFILE *AFp, const float Dbuff[], int Nval) =
  { NULL,    	NULL,	NULL,	NULL,       	NULL,
    AFfWrI2,	AFfWrI3,	NULL,	NULL,	NULL,
    NULL };


int
AFfWriteData (AFILE *AFp, const float Dbuff[], int Nval)

{
  int Nw;
  long int Novld;
  struct AF_opt *AFopt;

  assert (AFp->Op == FO_WO);

/* The file writing routines write scaled data to the file.  They write to the
   current file position.  They use the following AFp fields:
     AFp->fp - file pointer
     AFp->Swapb - data swap flag
     AFp->ScaleF - data scaling factor
     AFp->Ovld - overload counter

  This routine updates the following AFp values
    AFp->Error - error flag
    AFp->Isamp - current data sample.  This value is incremented by the
      number of samples written.
    AFp->Nsamp - last sample (updated if AFp->Isamp is beyond it)
*/

/* Transfer data to the audio file */
  Novld = AFp->Novld;	/* Save the value before writing */
  Nw = (*AF_Write[AFp->Format]) (AFp, Dbuff, Nval);
  AFp->Isamp += Nw;
  AFp->Nsamp = MAXV (AFp->Isamp, AFp->Nsamp);

/* Check for an error */
  if (Nw < Nval) {
    UTsysMsg ("AFfWriteData: %s", AFM_WriteErr);
    AFopt = AFoptions ();
    if (AFopt->ErrorHalt)
      exit (EXIT_FAILURE);
    AFp->Error = AF_IOERR;
  }

/* Check for overloads (print a message the first time only) */
  if (Novld == 0L && AFp->Novld != 0L)
    UTwarn ("AFfWriteData - %s", AFM_OClip);

  return Nw;
}

/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  int AFdecSpeaker (const char String[], unsigned char *SpkrConfig, int MaxN)
  int AFcheckSpeakers (const unsigned char *SpkrConfig);
  int AFspeakerNames (const unsigned char *SpkrConfig, char *SpkrNames,
                      int MaxNc)

Purpose:
  Decode, check, list loudspeaker spacial positions

Description:
  This routine sets the default mapping of output channels to speaker
  positions.  The spacial positions of the loudspeakers are specified as a
  list of comma and/or white-space separated locations from the list below.
    "FL"  - Front Left
    "FR"  - Front Right
    "FC"  - Front Center
    "LF"  - Low Frequency
    "BL"  - Back Left
    "BR"  - Back Right
    "FLC" - Front Left of Center
    "FRC" - Front Right of Center
    "BC"  - Back Center
    "SL"  - Side Left
    "SR"  - Side Right
    "TC"  - Top Center
    "TFL" - Top Front Left
    "TFC" - Top Front Center
    "TFR" - Top Front Right
    "TBL" - Top Back Lefty
    "TBC" - Top Back Center
    "TBR" - Top Back Right
    "-"   - none
  A speaker position can be associated with only one channel.  Only WAVE files
  store the speaker locations in the file header.  In the case of WAVE files,
  the subset of spacial positions must appear in the order given above.

Parameters:
  <-  int AFdecSpeaker
      Error flag, zero for no error
   -> const char String[]
      String containing the list of speaker positions
  <-  unsigned char *SpkrConfig
      Null-terminated string containing the speaker location codes
   -> int MaxN
      Maximum number of speaker locations

  <-  int AFspeakerNames
      Error flag, zero for no error
   -> int Nchan
      Number of channels
   -> const unsigned char *SpkrConfig
      Null-terminated string containing the speaker location codes
   -> int Nextra
      Maximum number of extra (not-specified) speaker positions
  <-  char *SpkrNames
      String containing the list of speaker positions

  <-  int AFcheckSpeakers
      Error flag, zero for no error
   -> const unsigned char *SpkrConfig
      Null-terminated string containing the speaker location codes

Author / revision:
  P. Kabal  Copyright (C) 2004
  $Revision: 1.8 $  $Date: 2004/03/31 13:25:41 $

-------------------------------------------------------------------------*/

#define NCBUF           512



int
AFcheckSpeakers (const unsigned char *SpkrConfig)

{
  int ErrCode, Nspkr, i, n;
  unsigned char SpkrCode[AF_N_SPKR_NAMES];

  ErrCode = 0;
  if (SpkrConfig == NULL || SpkrConfig[0] == AF_X_SPKR)
    Nspkr = 0;
  else
    Nspkr = strlen ((const char *) SpkrConfig);

  for (i = 0; i < AF_N_SPKR_NAMES; ++i)
    SpkrCode[i] = 0;

  for (i = 0; i < Nspkr; ++i) {
    n = SpkrConfig[i];
    assert (n >= 1 && n <= AF_N_SPKR_NAMES);
    if (n != AF_SPKR_X) {
      if (SpkrCode[n-1] > 0) {
	ErrCode = 1;
	UTwarn ("AFcheckSpeakers - %s: \"%.10s\"", AFM_DupSpkr,
		AF_Spkr_Names[n-1]);
	break;
      }
    }
    ++SpkrCode[n-1];
  }

  return ErrCode;
}

/* Speaker configuration names */
/* Let Nspkr be the number of locations specified in SpkrConfig. The number of
   of speaker locations returned in SpkrNames is
     if (Nchan <= Nspkr+Nextra)
       Ns = Nchan
     else
       Ns = Nspkr
   The "extra" speaker locations are designated "-".
   SpkrNames must provides space for Ns * AF_NC_SPKR - 1 characters.
*/


int
AFspeakerNames (int Nchan, const unsigned char *SpkrConfig, int Nextra,
		char *SpkrNames)

{
  int ErrCode;
  int i, Nspkr, Ns, nc, n;

  ErrCode = 0;

  SpkrNames[0] = '\0';
  if (SpkrConfig == NULL || SpkrConfig[0] == AF_X_SPKR ||
      SpkrConfig[0] == '\0')
    return ErrCode;		/* Quick bypass */

  /* Check for duplicate speakers */
  ErrCode = AFcheckSpeakers (SpkrConfig);
  if (ErrCode)
    return ErrCode;

  Nspkr = strlen ((const char *) SpkrConfig);
  if (Nchan <= Nspkr + Nextra)
    Ns = Nchan;
  else
    Ns = Nspkr;

  SpkrNames[0] = '\0';
  for (i = 0; i < Ns; ++i) {

    /* Append the speaker name */
    nc = strlen (SpkrNames);
    if (nc > 0) {
      SpkrNames[nc] = ' ';
      nc = nc + 1;
    }
    if (i >= Nspkr)
      n = AF_SPKR_X;
    else
      n = SpkrConfig[i];
    strcpy (&SpkrNames[nc], AF_Spkr_Names[n-1]);
  }

  if (ErrCode)
    SpkrNames[0] = '\0';

  return ErrCode;
}

/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  void AFclose (AFILE *AFp)

Purpose:
  Close an audio file

Description:
  This routine closes an audio file opened with AFopnRead or AFopnWrite.
  If the file was opened for write, the file header is updated with the number
  of samples in the file.  For both read and write operations, the audio file
  parameter structure associated with the file pointer is deallocated and the
  file is closed.

  If the file was opened for write, the number of overloads detected during
  write operations is reported.

Parameters:
   -> AFILE *AFp
      Audio file pointer for the audio file.  This structure is deallocated
      by this routine.

Author / revision:
  P. Kabal  Copyright (C) 2003
  $Revision: 1.45 $  $Date: 2003/11/03 18:44:43 $

-------------------------------------------------------------------------*/



void
AFclose (AFILE *AFp)

{
  /* Quiet return if the audio file structure is NULL */
  if (AFp == NULL)
    return;

/* Update the header for output files */
  if (AFp->Op == FO_WO) {

    assert (AFp->Format > 0 && AFp->Format < NFD);

    if (! AFp->Error) {	/* If an error has occurred, skip the updates */

      /* Check for Nsamp / Nchan / sample size consistency */
      if (AFp->Nsamp % AFp->Nchan != 0) {
	UTwarn ("AFclose - %s:", AFM_NSampNChan);
	UTwarn (AFMF_NSampNChan, "         ", AFp->Nsamp, AFp->Nchan);
      }

      /* Update the header */
      if (AFupdHead (AFp) && (AFoptions ())->ErrorHalt)
	exit (EXIT_FAILURE);

      /* Report the number of overloads */
      if (AFp->Novld > 0L)
	UTwarn (AFMF_NClip, "AFclose -", AFp->Novld);
    }
  }

  else
    assert (AFp->Op == FO_RO);

/* Close the file */
  fclose (AFp->fp);

/* Reset some AFILE structure values */
  AFp->fp = NULL;
  AFp->Op = FO_NONE;
  UTfree ((void *) AFp->SpkrConfig);
  AFp->InfoS.N = 0;
  UTfree ((void *) AFp->InfoS.Info);

/* Deallocate the AFILE structure */
  UTfree ((void *) AFp);

  return;
}


/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  void VRswapBytes (const void *BuffI, void *BuffO, int Size, int Nelem)

Purpose:
  Swap bytes in an array of values

Description:
  This routine does a swap of the bytes making up each word in an array of
  values.  Swapping bytes involves reversing the order of the constituent bytes
  in each word.

Parameters:
   -> const void *BuffI
      Array of values, each element having wordsize Size bytes
  <-  void *BuffO
      Output array of byte swapped values.  The output buffer can occupy the
      same memory as the input buffer.
   -> int Size
      Size of each element in bytes.  This value must be 1 (no-op), 2, 4 or 8.
   -> int Nelem
      Number of values to be byte-swapped

Author / revision:
  P. Kabal  Copyright (C) 2003
  $Revision: 1.21 $  $Date: 2003/05/09 03:33:52 $

-------------------------------------------------------------------------*/
void
VRswapBytes (const void *BuffI, void *BuffO, int Size, int Nelem)

{
  const unsigned char *cp;
  unsigned char *sp;
  const int2_t *B2;
  int2_t *S2;
  const int4_t *B4;
  int4_t *S4;
  const double8_t *B8;
  double8_t *S8;
  unsigned char t;

  switch (Size) {
  case (1):
    memcpy (BuffO, BuffI, (size_t) ((long int) Size * Nelem));
    break;

  case (2):
    B2 = (const int2_t *) BuffI;
    S2 = (int2_t *) BuffO;
    while (Nelem-- > 0) {
      cp = (const unsigned char *) B2;
      sp = (unsigned char *) S2;
      t = cp[1];  sp[1] = cp[0];  sp[0] = t;
      ++B2;
      ++S2;
    }
    break;

  case (4):
    B4 = (const int4_t *) BuffI;
    S4 = (int4_t *) BuffO;
    while (Nelem-- > 0) {
      cp = (const unsigned char *) B4;
      sp = (unsigned char *) S4;
      t = cp[3];  sp[3] = cp[0];  sp[0] = t;
      t = cp[2];  sp[2] = cp[1];  sp[1] = t;
      ++B4;
      ++S4;
    }
    break;

  case (8):
    B8 = (const double8_t *) BuffI;
    S8 = (double8_t *) BuffO;
    while (Nelem-- > 0) {
      cp = (const unsigned char *) B8;
      sp = (unsigned char *) S8;
      t = cp[7];  sp[7] = cp[0];  sp[0] = t;
      t = cp[6];  sp[6] = cp[1];  sp[1] = t;
      t = cp[5];  sp[5] = cp[2];  sp[2] = t;
      t = cp[4];  sp[4] = cp[3];  sp[3] = t;
      ++B8;
      ++S8;
    }
    break;

  default:
/*    UThalt ("VRswapBytes: %s", VRM_BadSize);*/
    break;
  }
  return;
}


/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  void UTwarn (const char Warnmsg[], ...)

Purpose:
  Print a warning message

Description:
  This routine prints a warning message on stderr (standard error).  An example
  of the use of this routine is as follows.
    UTwarn ("XXProc - Output data clipped (%d values(s))", N);

Parameters:
   -> const char Warnmsg[]
      Character string to be printed.  This string can contain optional
      formatting codes.  The arguments corresponding to the formatting codes
      appear at the end of the argument list.  The input string should not
      normally have a terminating newline character, since this routine
      supplies a newline.
  ->  <args...>
      Arguments corresponding to the formatting codes.  The format string and
      the variable number of arguments is passed on to the system routine
      vprintf.

Author / revision:
  P. Kabal  Copyright (C) 2003
  $Revision: 1.22 $  $Date: 2003/05/09 03:20:37 $

-------------------------------------------------------------------------*/





void
UTwarn (const char Warnmsg[], ...)

{
  va_list ap;

  va_start (ap, Warnmsg);

/* Print the warning message */
  vfprintf (stderr, Warnmsg, ap);
  fprintf (stderr, "\n");

  va_end (ap);

  return;
}


/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  void UTsysMsg (const char Warnmsg[], ...)

Purpose:
  Print user and system warning messages

Description:
  This routine prints the system error message corresponding to the last system
  error encountered.  It first prints a user supplied text, and then prints
  the system error message.  Messages are written to stderr (standard error).

  The system error message is printed using perror.  The preamble to the system
  error message is either the name of this routine or the string supplied to
  UTsetProg.  An example of the use of this routine is as follows.
    UTsetProg ("XXProg");
    ...
    fp = fopen (Fname, ...);
    if (fp == NULL)
      UTsysMsg ("XXProc - Cannot open file \"%s\"", Fname);
  If fopen fails, a typical output to stderr would be:
    XXProc - Cannot open file "abc"
    XXProg: No such file or directory

Parameters:
   -> const char Warnmsg[]
      Character string to be printed.  This string can contain optional
      formatting codes.  The arguments corresponding to the formatting codes
      appear at the end of the argument list.  The input string should not
      normally have a terminating newline character, since this routine
      supplies a newline.
   -> <args...>
      Arguments corresponding to the formatting codes.  The format string and
      the variable number of arguments is passed on to the system routine
      vprintf.

Author / revision:
  P. Kabal  Copyright (C) 2003
  $Revision: 1.8 $  $Date: 2003/05/09 03:20:37 $

-------------------------------------------------------------------------*/



void
UTsysMsg (const char Warnmsg[], ...)

{
  va_list ap;
  char *p;

  va_start (ap, Warnmsg);

/* Print the warning message */
  vfprintf (stderr, Warnmsg, ap);
  fprintf (stderr, "\n");

  va_end (ap);

/* Print the text for the system error message */
  if (errno) {
    p = UTgetProg ();
    if (*p != '\0')
      perror (p);
    else
      perror ("UTsysMsg");
  }

  return;
}
/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  int UTswapCode (int Dbo)

Purpose:
  Determine a swap code for decoding data

Description:
  This routine determines a swap code given a data byte ordering code.  Given
  the swap code for the data, the returned swap code is determined from the
  host byte order.

Parameters:
  <-  int UTswapCode
      Swap code, DS_NATIVE or DS_SWAP
   -> int Dbo
      Data byte swap code.
      DS_EB     - Data is in big-endian byte order.  The output swap code
                  will be DS_SWAP if the current host uses little-endian
		  byte order.
      DS_EL     - Data is in little-endian byte order data.  The output swap
                  code will be DS_SWAP if the current host uses big-endian
		  byte order.
      DS_NATIVE - Data is in native byte order
      DS_SWAP   - Data is byte-swapped

Author / revision:
  P. Kabal  Copyright (C) 2003
  $Revision: 1.5 $  $Date: 2003/05/09 03:23:41 $

-------------------------------------------------------------------------*/



int
UTswapCode (int Dbo)

{
  static int Hbo = DS_UNDEF;

  switch (Dbo) {
  case DS_EB:
  case DS_EL:
    if (Hbo == DS_UNDEF)	/* Cache the host byte order */
      Hbo = UTbyteOrder ();
    if (Hbo == Dbo)
      Dbo = DS_NATIVE;
    else
      Dbo = DS_SWAP;
    break;
  case DS_SWAP:
  case DS_NATIVE:
    break;
  default:
/*    UThalt ("UTswapCode: %s", UTM_BadSwap);*/
      break;
  }

  return Dbo;
}

/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  void UTsetProg (const char Program[]);
  char *UTgetProg (void)

Purpose:
  Set the program name for messages
  Get the program name for messages

Description:
  This routine sets a program name string.  This string is used by error
  reporting routines as an identifier in error message strings.

Parameters:
  UTsetProg:
   -> const char Program[]
      Program name, normally without a path
  UTgetProg:
  <-  char *UTgetProg
      Pointer to the program name.  If the program name has not been set, this
      is a pointer to an empty string.

Author / revision:
  P. Kabal  Copyright (C) 2003
  $Revision: 1.12 $  $Date: 2003/05/09 03:20:37 $

-------------------------------------------------------------------------*/


static char *Pgm = NULL;


void
UTsetProg (const char Program[])

{
  int nc;

  UTfree ((void *) Pgm);
  nc = strlen (Program);
  if (nc > 0) {
    Pgm = (char *) UTmalloc (nc + 1);
    strcpy (Pgm, Program);
  }
  else
    Pgm = NULL;
}



char *
UTgetProg (void)

{
  static char Empty[] = "";

  if (Pgm == NULL)
    return Empty;
  else
    return Pgm;
}
/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  void *UTmalloc (int size)

Purpose:
  Allocate a block of memory

Description:
  This routine invokes the system routine malloc to allocate a block of memory.
  Unlike the system routine malloc, this routine uses a signed value for
  the size of the block of memory requested.  If the size requested is less
  than or equal to zero, a NULL pointer is returned.  This is not an error
  condition.  However, if the system routine malloc returns an error, an
  error message is printed and execution is terminated.  The block of memory
  allocated can be freed up using UTfree.

Parameters:
  <-  void *UTmalloc
      Pointer to the allocated memory.  This should be cast to the appropriate
      pointer type.
   -> int size
      Size of the memory block to be allocated.  This is a signed value.  If
      size is less than or equal to zero, a NULL pointer is returned.

Author / revision:
  P. Kabal  Copyright (C) 2003
  $Revision: 1.10 $  $Date: 2003/05/09 03:20:37 $

-------------------------------------------------------------------------*/



void *
UTmalloc (int size)

{
  void *p;

  if (size <= 0)
    p = NULL;
  else {
    p = malloc ((size_t) size);
    if (p == NULL)
      UTerror ("UTmalloc: Error return from malloc");
  }
  return p;
}
/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  void UTfree (void *ptr)

Purpose:
  Deallocate a block of memory

Description:
  This routine deallocates a block of memory allocated by UTmalloc (or malloc).

Parameters:
   -> void *ptr
      Pointer to the memory to be deallocated.  If ptr is NULL, no action is
      taken.

Author / revision:
  P. Kabal  Copyright (C) 2003
  $Revision: 1.15 $  $Date: 2003/05/09 03:20:37 $

-------------------------------------------------------------------------*/



void
UTfree (void *ptr)

{
  if (ptr != NULL) {		/* For non-ANSI compliant versions of free() */
    errno = 0;
    free (ptr);
    if (errno != 0)
      UTerror ("UTfree: Error detected in free()");
  }
  return;
}
/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  void UTerror (const char Errmsg[], ...)

Purpose:
  Print user and system error messages, stop with error status set

Description:
  This routine first prints a user supplied text.  It then prints the system
  error message corresponding to the last system error encountered.  Messages
  are written to stderr (standard error).  Execution is terminated and the
  exit status is set to EXIT_FAILURE.  Normally this routine is called after a
  system routine has returned an error condition.

  The system error message is printed using perror.  The preamble to the system
  error message is either the name of this routine or the string supplied to
  UTsetProg.  An example of the use of this routine is as follows.
    UTsetProg ("XXProg");
    ...
    fp = fopen (Fname, ...);
    if (fp == NULL)
      UTerror ("XXProc: Cannot open file \"%s\"", Fname);
  If fopen fails, a typical output to stderr would be:
    XXProc: Cannot open file "abc"
    XXProg: No such file or directory

Parameters:
   -> const char Errmsg[]
      Character string to be printed.  This string can contain optional
      formatting codes.  The arguments corresponding to the formatting codes
      appear at the end of the argument list.  The input string should not
      normally have a terminating newline character, since this routine
      supplies a newline.
   -> <args...>
      Arguments corresponding to the formatting codes.  The format string and
      the variable number of arguments is passed on to the system routine
      vprintf.

Author / revision:
  P. Kabal  Copyright (C) 2003
  $Revision: 1.27 $  $Date: 2003/05/09 03:20:37 $

-------------------------------------------------------------------------*/



void
UTerror (const char Errmsg[], ...)

{
  va_list ap;
  char *p;

  va_start (ap, Errmsg);

/* Print the user error message */
  vfprintf (stderr, Errmsg, ap);
  fprintf (stderr, "\n");

  va_end (ap);

/* Print the text for the system error message */
  if (errno) {
    p = UTgetProg ();
    if (*p != '\0')
      perror (p);
    else
      perror ("UTerror");
  }

  exit (EXIT_FAILURE);
}
/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  char *UTdate (int format)

Purpose:
  Return a date/time string for the current time

Description:
  This routine returns a date/time string in a number of standard formats.
  Formats 0, 1 and 2 are in local time.  Format 3 gives the date and time in
  GMT (Universal Coordinated Time).

  Format 0 is the standard C-language format (without the trailing newline
  character).  Format 1 includes the time zone abbreviation.  Formats 0 and 1
  use abbreviations for the day of the week and the month.  Formats 2 and 3
  avoid language dependent names (except for the time-zone code).

   Format  Example                      time zone    typical length
     0    Sun Sep 16 01:03:52 1973      local time   24 + null
     1    Sun Sep 16 01:03:52 EST 1973  local time   28* + null
     2    1994-01-23 09:59:53 EST       local time   23* + null
     3    1994-01-23 14:59:53 UTC       GMT          23 + null
               (*) the time zone length can vary

Parameters:
  <-  char *UTdate
      Pointer to a character string for the date and time.  This is a pointer
      to an internal static storage area; each call to this routine overlays
      this storage.
   -> int format
      Date / time format code, taking on values from 0 to 3

Author / revision:
  P. Kabal  Copyright (C) 2003
  $Revision: 1.17 $  $Date: 2003/05/09 03:20:37 $

-------------------------------------------------------------------------*/




#define MAXDATE	64


char *
UTdate (int format)

{
  static char Datetime[MAXDATE+1];
  time_t tnow;

  tnow = time ((time_t *) NULL);
  STcopyMax (UTctime (&tnow, format), Datetime, MAXDATE);

  return Datetime;
}
/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  char *UTctime (time_t *timer, int format)

Purpose:
  Convert a time value to a date/time string

Description:
  This routine converts a time value to date/time string in a number of
  standard formats.  Formats 0, 1 and 2 are in local time.  Format 3 gives
  the date and time in GMT (Universal Coordinated Time).

  Format 0 is the standard C-language format (without the trailing newline
  character).  Format 1 includes the time zone abbreviation.  Formats 0 and 1
  use abbreviations for the day of the week and the month.  Formats 2 and 3
  avoid language dependent names (except for the time-zone code).

   Format  Example                      time zone    typical length
     0    Sun Sep 16 01:03:52 1973      local time   24 + null
     1    Sun Sep 16 01:03:52 EST 1973  local time   28* + null
     2    1994-01-23 09:59:53 EST       local time   23* + null
     3    1994-01-23 14:59:53 UTC       GMT          23 + null
               (*) the time zone length can vary

Parameters:
  <-  char UTctime[]
      Pointer to a character string for the date and time.  This is a pointer
      to an internal static storage area; each call to this routine overlays
      this storage.
   -> time_t *timer
      Input time value
   -> int format
      Date / time format code, taking on values from 0 to 3

Author / revision:
  P. Kabal  Copyright (C) 2003
  $Revision: 1.22 $  $Date: 2003/05/09 03:49:05 $

-------------------------------------------------------------------------*/


#define MAXDATE	64


char *
UTctime (time_t *timer, int format)

{
  int nc;
  char *stdtime;
  static char Datetime[MAXDATE+1];
  static int InitTZ = 0;

  /* Initialize the time zone setting */
  if (! InitTZ) {
/*    tzset ();*/
    InitTZ = 1;
  }

/* Notes:
   - On some systems, the time zone is verbose.  For insance, on Windows,
     with the MSVC compiler, the time zone is of the form
     "Eastern Daylight Time".
   - Setting the TZ environment variable will force the time zone names,
     for instance TZ=EST5EDT with force the use of the 3 letter short forms
     for the time zones.
*/

  switch (format) {

  case (0):
  default:
    stdtime = asctime (localtime (timer));
    STcopyNMax (stdtime, Datetime, 24, MAXDATE);
    break;

  case (1):
    stdtime = asctime (localtime (timer));
    nc = STcopyNMax (stdtime, Datetime, 20, MAXDATE);
    strftime (&Datetime[nc], MAXDATE - nc, "%Z %Y", localtime (timer));
    STtrim (&Datetime[nc], &Datetime[nc]); /* In case time zone is empty */
    break;

  case (2):
    strftime (Datetime, MAXDATE, "%Y-%m-%d %H:%M:%S %Z", localtime (timer));
    STtrim (Datetime, Datetime);	/* In case time zone is empty */
    break;

  case (3):
    strftime (Datetime, MAXDATE, "%Y-%m-%d %H:%M:%S UTC", gmtime (timer));
    break;
  }

  return Datetime;
}
/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  int UTcheckIEEE (void)

Purpose:
  Check if host uses IEEE standard 754 format for float values

Description:
  This routine determines if the current host uses IEEE standard 754 format to
  represent float values.

Parameters:
  <-  int UTcheckIEEE
      Flag, 1 if host uses IEEE float format, 0 otherwise

Author / revision:
  P. Kabal  Copyright (C) 2003
  $Revision: 1.14 $  $Date: 2003/05/09 03:23:41 $

-------------------------------------------------------------------------*/


/* Note:
   On many Unix machines, this routine could instead check the setting of
   _IEEE in values.h.
*/

/* IEEE float value check values */
static const uint4_t IEEEiv = 0xc3268000;
static const float4_t IEEEfv = -166.5;


int
UTcheckIEEE (void)

{
  union {
    float4_t fv;
    uint4_t iv;
  } Floatv;

/* Store a float value and check the bit pattern of the result */
  Floatv.fv = IEEEfv;
  if (Floatv.iv == IEEEiv)
    return 1;
  else
    return 0;
}
/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  int UTbyteOrder (void)

Purpose:
  Determine the byte order for data storage

Description:
  This function returns a code indicating whether the storage order the current
  host is little-endian or big-endian.  For big-endian machines the constituent
  bytes in a multi-byte entity are stored starting with the most significant
  byte, ending with the least significant byte.  For little-endian machines,
  bytes are stored from least significant to most significant.

Parameters:
  <-  int UTbyteOrder
      Returned integer code
        0 - Big-endian storage order
        1 - Little-endian storage order

Author / revision:
  P. Kabal  Copyright (C) 2003
  $Revision: 1.24 $  $Date: 2003/05/09 03:23:41 $

-------------------------------------------------------------------------*/


#define U4		((int) sizeof (uint4_t))

static union { uint4_t U; unsigned char C[U4]; } u;
static const unsigned char C[4] = { 1, 2, 4, 8 };
static const uint4_t I4L = 0x08040201;
static const uint4_t I4B = 0x01020408;


int
UTbyteOrder (void)

{
  static int Hbo = DS_UNDEF;

  if (Hbo == DS_UNDEF) {	/* Cache the byte order */
    u.C[0] = C[0];
    u.C[1] = C[1];
    u.C[2] = C[2];
    u.C[U4-1] = C[3];	/* Implicit check that sizeof (uint4_t) is 4 */
    if (u.U == I4L)
      Hbo = DS_EL;
    else if (u.U == I4B)
      Hbo = DS_EB;
    else
      assert (0);	/* Funny byte order */
  }

  return Hbo;
}
/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  char *STtrimIws (const char Si[])

Purpose:
  Trim leading white-space

Description:
  This routine returns a pointer to the first character in a string that is not
  white-space (as defined by isspace).  If the input string consists entirely
  of white-space, this routine returns a pointer to the terminating null
  character.

Parameters:
  <-  char *STrimIws
      Pointer to the first non-white-space character
   -> const char Si[]
      Input character string

Author / revision:
  P. Kabal  Copyright (C) 2003
  $Revision: 1.8 $  $Date: 2003/05/09 03:06:42 $

-------------------------------------------------------------------------*/






char *
STtrimIws (const char Si[])

{
  /* Find the first non-white-space character */
  for (; isspace ((int) *Si); ++Si)
    ;

  return ((char *) Si);
}
/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  int STtrim (const char Si[], char So[])

Purpose:
  Copy a string, trimming leading and trailing white-space

Description:
  This routine copies characters from the input string to the output string.
  Leading white space characters and trailing white space characters are
  omitted.

Parameters:
  <-  int STtrim
      Number of characters in the output string
   -> const char Si[]
      Input character string
  <-  char So[]
      Output character string.  The output string pointer can be the same as
      the input string pointer.

Author / revision:
  P. Kabal  Copyright (C) 2003
  $Revision: 1.7 $  $Date: 2003/05/09 03:02:44 $

-------------------------------------------------------------------------*/



int
STtrim (const char Si[], char So[])

{
  const char *q;
  int n, nc;

  /* Find the first non-white-space character */
  for (; isspace ((int) *Si); ++Si)
    ;

  /* Find the end of the string */
  for (q = Si; *q != '\0'; ++q)
    ;
  nc = q - Si;			/* q points to the null */

  /* Trim trailing white-space */
  for (; nc > 0; --nc) {	/* nc is the number of characters */
    if (! isspace ((int) *(--q)))
      break;
  }

  /* Copy the trimmed string to the output string */
  for (n = 0; n < nc; ++n)
    *So++ = *Si++;
  *So = '\0';			/* Add a trailing null */

  return nc;
}
/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  int STcopyNMax (const char Si[], char So[], int N, int Maxchar)

Purpose:
  Copy N characters characters to a string

Description:
  This routine is used to copy min(N, strlen(Si)) characters from the input
  input string to the output string.  If min(N, strlen(Si) is greater than
  Maxchar, a string is truncated at that point and a warning message is
  printed.

Parameters:
 <-   int STcopyNMax
      Number of characters in the output string
   -> const char Si[]
      Input character string
  <-  char So[]
      Output character string.  This string is always null terminated, with
      at most Maxchar characters not including the terminating null character.
   -> int N
      Number of characters to be transferred
   -> int Maxchar
      Maximum number of characters (not including the terminating null
      character) to be placed in So.

Author / revision:
  P. Kabal  Copyright (C) 2003
  $Revision: 1.20 $  $Date: 2003/05/09 03:02:44 $

-------------------------------------------------------------------------*/


#define MINV(a, b)	(((a) < (b)) ? (a) : (b))


int
STcopyNMax (const char Si[], char So[], int N, int Maxchar)

{
  const char *si;
  int n;
  int NM;

  /* Save the initial input pointer */
  si = Si;

  /* Copy at most max(N, Maxchar) characters */
  NM = MINV (Maxchar, N);
  for (n = 0; n < NM && *si != '\0'; n++)
    *So++ = *si++;

  /* Add a trailing null */
  *So = '\0';

  /* Check for truncation */
  if (N > Maxchar && n == NM && Si[NM-1] != '\0')
    UTwarn ("STcopyNMax - %s: \"%.*s...\"", STM_StrTrunc, MINV (30, n), Si);

  /* Return the number of characters in So */
  return n;
}
/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  int STcopyMax (const char Si[], char So[], int Maxchar)

Purpose:
  Copy at most Maxchar characters to a string

Description:
  This routine copies characters from the input string to the output string.
  Characters are copied until a null is seen in the input string or the number
  of characters copied is Maxchar.  Then a trailing null character is appended
  to the output string.  If the input string is longer than Maxchar (not
  including the null character), a string truncated warning message is printed.

Parameters:
 <-   int STcopyMax
      Number of characters in the output string
   -> const char Si[]
      Input character string
  <-  char So[]
      Output character string.  This string is always null terminated, with
      at most Maxchar characters not including the terminating null character.
      If the input string is longer than Maxchar, only the first Maxchar
      characters are copied and a warning message is printed.
   -> int Maxchar
      Maximum number of characters (not including the trailing null character)
      to be placed in So.

Author / revision:
  P. Kabal  Copyright (C) 2003
  $Revision: 1.17 $  $Date: 2003/05/09 03:02:44 $

-------------------------------------------------------------------------*/
#define MINV(a, b)	(((a) < (b)) ? (a) : (b))


int
STcopyMax (const char Si[], char So[], int Maxchar)

{
  const char *si;
  int n;

  /* Save the initial input pointer */
  si = Si;

  /* Copy at most Maxchar characters */
  for (n = 0; n < Maxchar && *si != '\0'; n++)
    *So++ = *si++;

  /* Add a trailing null */
  *So = '\0';

  /* Check for truncation */

  if (*si != '\0')
    UTwarn ("STcopyMax - %s: \"%.*s...\"", STM_StrTrunc, MINV (30, n), Si);

  /* Return the number of characters in So */
  return n;
}
/*------------ Telecommunications & Signal Processing Lab -------------
                         McGill University

Routine:
  int FLseekable (FILE *fp)

Purpose:
  Determine if an I/O stream is random access.

Description:
  This routine determines if an I/O stream associated with a given file pointer
  is random access.

Parameters:
  <-  int FLseekable
      Return value, 1 if the file is random access, 0 otherwise
   -> FILE *fp
      File pointer

Author / revision:
  P. Kabal  Copyright (C) 2003
  $Revision: 1.11 $  $Date: 2003/05/09 01:39:26 $

----------------------------------------------------------------------*/




#ifndef S_ISREG		/* Defined by POSIX */
#  define S_ISREG(m)	(((m) & S_IFMT) == S_IFREG)
#endif


int
FLseekable (FILE *fp)

{
    return 1;
}

