// GENPO - the GENeral Purpose Organ
// Copyright (C) 2003,2004 - Steve Merrony 

// This program is free software; you may redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the Free 
// Software Foundation; either version 2 of the License, or (at your option) 
// any later version.

// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for 
// more details.

// You should have received a copy of the GNU General Public License along with 
// this program; if not, write to the Free Software Foundation, Inc., 59 Temple 
// Place, Suite 330, Boston, MA 02111-1307 USA

#include "application.h"
#include "customEvents.h"
#include "midiConfigForm.h"
#include "configParser.h"
#include "console.h"
#include "passingButton.h"
#include "passingCheckBox.h"
#include "pistonEventData.h"

#include <alsa/asoundlib.h>

#include <qaccel.h>
#include <qbuttongroup.h>
#include <qapplication.h>
#include <qcombobox.h>
#include <qfile.h>
#include <qfiledialog.h>
#include <qgroupbox.h>
#include <qimage.h>
#include <qlayout.h>
#include <qmenubar.h>
#include <qmessagebox.h>
#include <qnamespace.h>
#include <qpaintdevicemetrics.h>
#include <qpainter.h>
#include <qpixmap.h>
#include <qpopupmenu.h>
#include <qstatusbar.h>
#include <qtextedit.h>
#include <qtextstream.h>
#include <qtoolbutton.h>
#include <qtooltip.h>
    
#include <iostream>
using namespace std;

#include "fileopen.xpm"

// constructor
ApplicationWindow::ApplicationWindow( volatile Console *shared_console )
  : QMainWindow( 0, 0, WDestructiveClose )
{
  console = (Console *)shared_console;

  /////  First, sort out the menu /////

  QPixmap openIcon; // , saveIcon, printIcon;
  openIcon = QPixmap( fileopen );

  QMimeSourceFactory::defaultFactory()->setPixmap( "fileopen", openIcon );

  QPopupMenu * file = new QPopupMenu( this );
  menuBar()->insertItem( "&File", file );

  file->insertItem( openIcon, "&Open...", this, SLOT( choose() ), CTRL+Key_O );
  file->insertItem( openIcon, "&Reload", this, SLOT( reload() ), CTRL+Key_R );
  //  file->setItemEnabled( rl, FALSE );  // we can't reload anything at startup
  file->insertSeparator();
  //    file->insertItem( "&Close", this, SLOT(close()), CTRL+Key_W );
  file->insertItem( "&Quit", qApp, SLOT( closeAllWindows() ), CTRL+Key_Q );

  // MIDI menu...
  QPopupMenu * midi = new QPopupMenu( this );
  menuBar()->insertItem( "&MIDI", midi );
  //  midi->insertItem( "&Configuration", this, SLOT(midiConfig()), CTRL+Key_M );
  midi->insertItem( "&Panic", this, SLOT(midiPanic()), CTRL+Key_P );

  // Help menu...
  menuBar()->insertSeparator();

  QPopupMenu * help = new QPopupMenu( this );
  menuBar()->insertItem( "&Help", help );

  help->insertItem( "&About", this, SLOT(about()), Key_F1 );

  central = new QWidget( this );
  setCentralWidget( central );

  QHBoxLayout *topLayout = new QHBoxLayout( central, 10 );

  grid = new QGridLayout( 0,0, 5  );  
  topLayout->addLayout( grid, 1 );
  
  // create widgets we'll need later
  desc = new QLabel( "", central,  "descLab" );
  stopGroup = new QButtonGroup( "", central );  // a single invisible group for ALL the stops
  
  gcButton = new QPushButton( "G.C. ( )", central );

  seq = &(shared_console->seq);
  op_a = &(shared_console->out_port_a);
  op_b = &(shared_console->out_port_b);
  
  resize( 750, 700 );

  if (console->orgFileName) {
    // an organ filename was passed into the program - load it
    load( console->orgFileName );
  }
}


ApplicationWindow::~ApplicationWindow()
{

}


void ApplicationWindow::choose()
{
  QString fn = QFileDialog::getOpenFileName( QString::null,
					     "Organs (*.org)",
					     this,
					     "Load an organ configuration",
					     "Choose an organ configuration" );
  if ( !fn.isEmpty() )
    load( fn );
  else
    statusBar()->message( "Loading aborted", 2000 );
}


void ApplicationWindow::load( const QString &fileName )
{
  // clean out anything that might already be there
  generalCancel();
  for (int m = 0; m<console->num_divisions; m++) {
    delete divisionLabels[m];
    for (int s = 0; s<console->divisions[m].num_stops; s++)   delete stopButtons[m][s];
    for (int p = 0; p<console->divisions[m].num_pistons; p++) {
      delete pistonButtons[m][p];
    }
    if (console->divisions[m].num_pistons > 0) { delete pvbox[m]; delete pgrp[m]; }
    if (console->divisions[m].num_couplers > 0) { delete cvbox[m]; delete cgrp[m]; }
  }
  if (console->num_toePistons > 0) {
    delete tphbox;
    delete tpgrp;
  }
  
  QFile f( fileName );
  if ( !f.open( IO_ReadOnly ) ) {
    return;
  }
  else {
    QTextStream ts( &f );
    ConfigParser handler;
    QXmlInputSource source( &f );
    QXmlSimpleReader reader;
    reader.setContentHandler( &handler );
    reader.parse( source );

    const char accel_keys[4][10] = {
      { 'z','x','c','v','b','n','m',',','.','/' },
      { 'a','s','d','f','g','h','j','k','l',';' },
      { 'q','w','e','r','t','y','u','i','o','p' },
      { '1','2','3','4','5','6','7','8','9','0' }
    };
    handler.copyConfig( console );
      
    desc->setText( console->organName + " (" + console->description + ")" );
    grid->addMultiCellWidget( desc, 0, 0, 0, grid->numCols() - 1 , AlignCenter );

    QString midiChan;
    int num_pistons = 0;
    int num_couples = 0;	
    int num_toe_pistons = 0;

    // console names
    for (int m = 0; m<console->num_divisions; m++) {
      divisionLabels[m] = new QLabel( "X", central );
      divisionLabels[m]->setText(console->divisions[m].name );
      divisionLabels[m]->setAlignment( AlignHCenter );
      grid->addWidget( divisionLabels[m], 1, m );
      divisionLabels[m]->show();
    }
      
    // stops
    
    for (int m = 0; m<console->num_divisions; m++) {
      for (int s = 0; s<console->divisions[m].num_stops; s++) {
      	stopButtons[m][s] = new QPushButton( "", central );
      	stopButtons[m][s]->setMinimumHeight( 14 );
      	// insert into the invisible group with a hashed ID
      	stopGroup->insert( stopButtons[m][s], m * MAX_STOPS + s );
      	stopGroup->setFrameStyle( QFrame::NoFrame );

	//stopButtons[m][s]->setText( console->divisions[m].stops[s].name );
	stopButtons[m][s]->setName( console->divisions[m].stops[s].label );
	stopButtons[m][s]->setToggleButton( TRUE );
	if ( ((console->acceleratorStyle == "QWERTY") || (!console->acceleratorStyle)) && 
	     (m < 4) && 
	     (s < 10) ) {  // default accelerators for 1st 10 stops on 1st 4 divisions
	  stopButtons[m][s]->setText( console->divisions[m].stops[s].name + "   (&" + accel_keys[m][s] + ")" );
	  stopButtons[m][s]->setAccel( (const QKeySequence) (const QCString) UNICODE_ACCEL + accel_keys[m][s] );
	}
	else {
	  stopButtons[m][s]->setText( console->divisions[m].stops[s].name );
	}
        // colouring - stop colour overrides manual colour
        if (console->divisions[m].stops[s].colour != "DEFAULT") {
	  stopButtons[m][s]->setPaletteBackgroundColor( console->divisions[m].stops[s].colour );
	} else {
	  if (console->divisions[m].colour != "DEFAULT") {
	    stopButtons[m][s]->setPaletteBackgroundColor( console->divisions[m].colour );
	  }
	}
	grid->addWidget( stopButtons[m][s], s+3, m );
	stopButtons[m][s]->show();
      }
    }

    // one Slot-Handler for all stops
    stopGroup->disconnect( SIGNAL( clicked( int ) ), this , SLOT( changeStop( int ) ) );
    connect( stopGroup, SIGNAL( clicked( int ) ), this, SLOT( changeStop( int ) ) ); 

    // pistons
    for (int m = 0; m<console->num_divisions; m++) {
      if (console->divisions[m].num_pistons > 0) {
 	pgrp[m] = new QButtonGroup( central );
    	pvbox[m] = new QGridLayout( pgrp[m], 0, 3 );
	pgrp[m]->setFrameStyle( QFrame::NoFrame );
	for (int p = 0; p<console->divisions[m].num_pistons; p++) {
 	  pistons[num_pistons].manual = m;
      	  pistons[num_pistons].piston = p;
      	  pistonButtons[m][p] = new PassingButton( (void *)&pistons[num_pistons], "", pgrp[m], "Piston" );
      	  num_pistons++;

	  pistonButtons[m][p]->setText( console->divisions[m].pistons[p].label );
	  pistonButtons[m][p]->setToggleButton( TRUE );
	  pistonButtons[m][p]->setMinimumHeight( 16 );
	  pistonButtons[m][p]->show();
	  //QToolTip::add( pistonButtons[m][p], "(Huh?)" );
	  pvbox[m]->addWidget( (QWidget *)pistonButtons[m][p], p / 3, p % 3 );
	  pistonButtons[m][p]->disconnect( SIGNAL( clicked( void * ) ), this, SLOT( pistonChanged( void * ) ) );
	  connect( pistonButtons[m][p], SIGNAL( clicked( void * ) ), this, SLOT( pistonChanged( void * ) ) );
	  // store the widget ID in the shared structure
	  console->divisions[m].pistons[p].widgetID = pistonButtons[m][p];
	}
	grid->addWidget( pgrp[m], 2+MAX_STOPS, m );
	pgrp[m]->show();
      } 
    }
   
    // couplers
    for (int m = 0; m<console->num_divisions; m++) {
      if (console->verboseMode) cout << "Division " << m << ", couplers: " << console->divisions[m].num_couplers << "\n"; 
      if (console->divisions[m].num_couplers > 0) {
	cgrp[m] = new QButtonGroup( central );
        cvbox[m] = new QVBoxLayout( cgrp[m], 5 );
	cgrp[m]->setFrameStyle( QFrame::StyledPanel );
	for (int c = 0; c<console->divisions[m].num_couplers; c++) {
	  couples[num_couples].manual = m;
	  couples[num_couples].coupler = c;
	  couplerCheckBoxes[m][c] = new PassingCheckBox( (void *)&couples[num_couples], "", cgrp[m] , "Coupler" );
	  couplerCheckBoxes[m][c]->setMinimumHeight( 14 );
	  num_couples++;
	  couplerCheckBoxes[m][c]->setText( console->divisions[m].couplers[c].name );
	  couplerCheckBoxes[m][c]->setName( console->divisions[m].couplers[c].label );
	  // don't add if already there
	  if (cvbox[m]->findWidget((QWidget *)couplerCheckBoxes[m][c] ) == -1) {
	    cvbox[m]->addWidget( (QWidget *)couplerCheckBoxes[m][c] ); 
	  }
	  couplerCheckBoxes[m][c]->show();
	  couplerCheckBoxes[m][c]->disconnect( SIGNAL( clicked( void * ) ), this, SLOT( couplerChanged( void * ) ) );
	  connect( couplerCheckBoxes[m][c], SIGNAL( clicked( void * ) ), this, SLOT( couplerChanged( void * ) ) );
	}
	grid->addWidget( cgrp[m], MAX_STOPS+MAX_PISTONS, m );
	cgrp[m]->show();
      }
    }

    // Toe Pistons
    if (console->num_toePistons > 0) {
      tpgrp = new QButtonGroup( central );
      tphbox = new QHBoxLayout( tpgrp, 0, 5 );
      //tphbox->setAutoAdd( TRUE );
      tpgrp->setFrameStyle( QFrame::NoFrame );
      for (int tp = 0; tp < console->num_toePistons; tp++) {
	toePistons[tp].toePiston_ix = tp;
	toePistonButtons[tp] = new PassingButton( (void *)&toePistons[num_toe_pistons], "", tpgrp, "Toe Piston" );
	num_toe_pistons++;
	
	toePistonButtons[tp]->setText( console->toePistons[tp].label );
	toePistonButtons[tp]->setMinimumHeight( 16 );
	toePistonButtons[tp]->setToggleButton( TRUE );
	toePistonButtons[tp]->show();
	tphbox->addWidget( (QWidget *)toePistonButtons[tp] );
	toePistonButtons[tp]->disconnect( SIGNAL( clicked( void * ) ), this, SLOT( toePistonChanged( void * ) ) );
	connect(toePistonButtons[tp], SIGNAL( clicked( void * ) ), this, SLOT( toePistonChanged( void * ) ) );
	console->toePistons[tp].widgetID = toePistonButtons[tp];
      }
      grid->addMultiCellWidget( tpgrp, MAX_STOPS+MAX_PISTONS+MAX_COUPLERS,MAX_STOPS+MAX_PISTONS+MAX_COUPLERS, 0, console->num_divisions - 1 );
      tpgrp->show();
    }
    
    // General Cancel
    grid->addWidget( gcButton, MAX_STOPS+MAX_PISTONS+MAX_COUPLERS + 1, 0 );
    gcButton->setAccel( Key_Space );
    gcButton->disconnect( SIGNAL( clicked() ), this, SLOT( generalCancel() ) );
    connect( gcButton, SIGNAL( clicked() ), this, SLOT( generalCancel() ) );
    gcButton->show();

    console->orgFileName = fileName;

    //    setCaption( APP_NAME + ": " + console.organName );
    statusBar()->message( "Loaded configuration: " + console->organName, 4000 );
  }
}


void ApplicationWindow::closeEvent( QCloseEvent* ce ) {
  ce->accept();
  if (console->verboseMode) cout << "Normal user exit\n";
}

void ApplicationWindow::reload() {

  if (console->orgFileName) {
    // remove child widgets	
    for (int m = 0; m<console->num_divisions; m++) {
      for (int s = 0; s<console->divisions[m].num_stops; s++) {
        stopButtons[m][s]->hide();
      }
    }
    load( console->orgFileName );
  }
  else {
    statusBar()->message( "No configuration (.org) file loaded - cannot reload!" );
  }
}

void ApplicationWindow::midiPanic() {

  for (int mc = 0; mc < console->num_out_ports * MIDI_CHANNELS_PER_PORT ; mc++) {
    silenceMidiChannel( mc );
  }
}

void ApplicationWindow::silenceMidiChannel( int mc ) {

  snd_seq_event_t ano_event;
  int rc;

  snd_seq_ev_clear( &ano_event );
  snd_seq_ev_set_controller( &ano_event, mc, MIDI_CTL_ALL_NOTES_OFF,0 );
  
  snd_seq_ev_set_subs( &ano_event );
  snd_seq_ev_set_direct( &ano_event );
  if (mc <MIDI_CHANNELS_PER_PORT) {
    snd_seq_ev_set_source( &ano_event, op_a->my_port );
  } 
  else {
    snd_seq_ev_set_source( &ano_event, op_b->my_port );
  }

  rc = snd_seq_event_output_direct( seq->seq_handle, &ano_event );
  if (rc < 0) {
    cout << "Error: sending sequencer controller event (" << snd_strerror( rc ) << ")\n";
  }

}

void ApplicationWindow::about()
{
  QMessageBox::about( this, APP_NAME,
		      "Version: "
		      APP_VERSION 
		      " (c)2003 - 2007 S.Merrony" );
}

void ApplicationWindow::changeStop( int stopIDhash ) {

  int m, s;

  m = stopIDhash / MAX_STOPS;  // manual number
  s = stopIDhash % MAX_STOPS;  // stop number in that manual

  if (stopButtons[m][s]->isOn()) {
    // handle single-stop mode
    if (console->singleStopMode) {
      generalCancel();
      this->update();
    }
    pullStop( m, s );
  }
  else {
    pushStop( m, s );
  }

} 

void ApplicationWindow::customEvent( QCustomEvent * e ) {
  if ( e->type() == PRESET_CHANGE_EVENT ) { 
    if (console->verboseMode) cout << "Caught custom-event\n";
    PistonEventData * pe = (PistonEventData *)e->data();
    if (console->verboseMode) cout << "Division " << pe->pe_manual << ", piston" << pe->pe_piston << endl;
    if (pe->pe_manual == TOE_PISTON) {
      changeToePiston( pe->pe_piston );
    }
    else {
      if (console->verboseMode) cout << "ToePiston " << pe->pe_piston << endl;
      changePiston( pe->pe_manual, pe->pe_piston );
    }
    delete e;
    this->update(); // I'm not quite sure why this is required - but if fixes some odd behaviour
  }
}
    

void ApplicationWindow::changePiston( int m, int pc ) {

  // do nothing if there are no pistons for this manual
  if ( (m<=console->num_divisions) && (console->divisions[m].num_pistons > 0) && (pc <=console->divisions[m].num_pistons)) {

    //dumpVoices();
    // clear all stops on this manual
    for (int s = 0; s < console->divisions[m].num_stops; s++) {
      if (stopButtons[m][s]->isOn()) {
	stopButtons[m][s]->setOn( FALSE );
	pushStop( m, s );
      }
    }
    //printf( "Post-clear:\t" );
    //dumpVoices();

  
    // pull the stops for this piston
    for (int p = 0; p < console->divisions[m].pistons[pc].numPistonStops; p++) {
      for (int s = 0; s < console->divisions[m].num_stops; s++) {
	if  ( QString::compare( console->divisions[m].pistons[pc].pistonStops[p].stopLabel,
				console->divisions[m].stops[s].label
				) == 0 ) {
	  stopButtons[m][s]->setOn( TRUE );
	  pullStop( m, s );
	}
      }
    }
  }

} 

void ApplicationWindow::pistonChanged( void *piston ) {

  piston_t *p = (piston_t *)piston;
  int m = p->manual;
  int pc = p->piston;

  changePiston( m, pc );

  // clear the button
  pistonButtons[m][pc]->setOn(FALSE);

  //printf( "Done\t" );
  //dumpVoices();

}

void ApplicationWindow::changeToePiston( int tp_ix ) {
  
  generalCancel();
  
  // pull the stops for this piston
  for (int p = 0; p < console->toePistons[tp_ix].numPistonStops; p++) {
    for (int m = 0; m < console->num_divisions; m++) {
      for (int s = 0; s < console->divisions[m].num_stops; s++) {
	if  ( QString::compare( console->toePistons[tp_ix].pistonStops[p].stopLabel,
	    console->divisions[m].stops[s].label
			    ) == 0 ) {
	      stopButtons[m][s]->setOn( TRUE );
	      pullStop( m, s );
			    }
      }
    }
  }

}

void ApplicationWindow::toePistonChanged( void *toePiston ) {

  ToePiston *tp = (ToePiston *)toePiston;
  int tpix = tp->toePiston_ix;

  changeToePiston( tpix );

  // clear the button
  toePistonButtons[tpix]->setOn(FALSE);

  //printf( "Done\t" );
  //dumpVoices();

}
void ApplicationWindow::couplerChanged( void *coupler ) {

  couple_t *c = (couple_t *)coupler;

  if (console->verboseMode) cout << "Coupler " << c->coupler <<" on manual " << c->manual << " changed\n";

  console->divisions[c->manual].couplers[c->coupler].active = !console->divisions[c->manual].couplers[c->coupler].active;

}

void ApplicationWindow::generalCancel() {

  // stops and coupler boxes
  for (int m = 0; m<console->num_divisions; m++) {
    for (int s = 0; s<console->divisions[m].num_stops; s++) {
      //      if (stopButtons[m][s]->isOn()) {
      if (console->divisions[m].stops[s].pulled) {
	stopButtons[m][s]->setOn( FALSE );
	pushStop( m, s );
      }
    }
    for (int c = 0; c<console->divisions[m].num_couplers; c++) {
      couplerCheckBoxes[m][c]->setChecked( FALSE );
      console->divisions[m].couplers[c].active = FALSE;
    }
  }
}

void ApplicationWindow::pullStop( int m, int s ) { // activate a stop

  int rc, man_channel, new_channel;
  snd_seq_event_t pc_event;
  QString msg;
  
  console->divisions[m].stops[s].pulled = TRUE;

  man_channel = console->divisions[m].midiChannel;
  //  dumpVoices();
  // find 1st free slot for a new voice
  new_channel = 0;
  while (new_channel<=console->num_out_ports*MIDI_CHANNELS_PER_PORT && console->voices[new_channel].active) new_channel++;

  //printf( "Candidate channel for new voice: %d\n", new_channel );
  if (new_channel >= console->num_out_ports*MIDI_CHANNELS_PER_PORT) {
    stopButtons[m][s]->setOn( FALSE );
    if (console->num_out_ports == 0)
      msg = "Warning: Too many stops pulled - GENPO thinks no synthesizers are connected!";
    else
      msg = "Warning: Too many stops pulled (16 per connected synthesizer are allowed).";
    statusBar()->message( msg );
    cout << msg << endl;
  }
  else {
    statusBar()->clear();
    console->voices[new_channel].active = TRUE;
    console->voices[new_channel].manual_channel = man_channel;
    console->voices[new_channel].velocity = console->divisions[m].stops[s].velocity;
    console->voices[new_channel].synth = 0;  // *** JUST using 0 for now ***<<<<<<<<<
    console->voices[new_channel].bank = console->divisions[m].stops[s].midiBank;
    console->voices[new_channel].voice = console->divisions[m].stops[s].midiVoice;

    // store the assigned voice in the stop
    console->divisions[m].stops[s].kbdMidiChannel = man_channel;
    console->divisions[m].stops[s].voiceIndex = new_channel;

    // send a MIDI event to change the program for this voice (channel)
    snd_seq_ev_clear( &pc_event );

    if ( new_channel < MIDI_CHANNELS_PER_PORT) {
      snd_seq_ev_set_controller( &pc_event, new_channel, 0, console->divisions[m].stops[s].midiBank );
      snd_seq_ev_set_subs( &pc_event );
      snd_seq_ev_set_direct( &pc_event );
      snd_seq_ev_set_source( &pc_event, op_a->my_port );
    }
    else {
      snd_seq_ev_set_controller( &pc_event, new_channel - MIDI_CHANNELS_PER_PORT, 0, console->divisions[m].stops[s].midiBank );
      snd_seq_ev_set_subs( &pc_event );
      snd_seq_ev_set_direct( &pc_event );
      snd_seq_ev_set_source( &pc_event, op_b->my_port );
    }

    rc = snd_seq_event_output_direct( seq->seq_handle, &pc_event );
    if (rc < 0) {
      cout << "Error: sending sequencer controller event (" << snd_strerror( rc ) << ")\n";
    }   

    snd_seq_ev_set_pgmchange( &pc_event, new_channel, console->divisions[m].stops[s].midiVoice );

    rc = snd_seq_event_output_direct( seq->seq_handle, &pc_event );
    if (rc < 0) {
      cout << "Error: sending sequencer pgm change event (" << snd_strerror( rc ) << ")\n";
    }

    if (console->verboseMode) cout << "Voice " << console->divisions[m].stops[s].midiVoice << " set on channel " << new_channel << "\n";

  }
}

void ApplicationWindow::pushStop( int m, int s ) {  // turn off a stop

  int old_voice = console->divisions[m].stops[s].voiceIndex;

  // this if should not be required it stops a crash that shouldn't happen <===========================
  if (old_voice >= 0 && old_voice < console->num_out_ports*MIDI_CHANNELS_PER_PORT) {
    console->divisions[m].stops[s].pulled = FALSE;

    console->voices[old_voice].active = FALSE;
    console->voices[old_voice].manual_channel = FREE_CHANNEL;
    console->divisions[m].stops[s].kbdMidiChannel = FREE_CHANNEL;
    console->divisions[m].stops[s].kbdMidiChannel = FREE_CHANNEL;
  
    // turn all notes off on the Voice
    silenceMidiChannel( old_voice );
  }
  if (console->verboseMode) cout << "PushStop( " << m << ", " << s <<" )\n";

}

void ApplicationWindow::dumpVoices() {

  for (int c = 0; c < console->num_out_ports*MIDI_CHANNELS_PER_PORT; c++) {
    cout << c << ":" << console->voices[c].active << "/" << console->voices[c].manual_channel << "/" << console->voices[c].voice << ", ";
  }
  cout << "\n";
}

