/*
 * Copyright (C) 2008-2009 Freescale Semiconductor, Inc. All rights reserved.
 *
 */
 
/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
 
/*
 * Module Name:    mfw_gst_rvdecoder.c
 *
 * Description:    RealMedia video decoder plugin for Gstreamer.
 *
 * Portability:    This code is written for Linux OS and Gstreamer
 */  
 
/*
 * Changelog: 
 * Aug 22 2008 Sario HU <b01138@freescale.com>
 * - Initial version
 *
 */


/*===============================================================================
                            INCLUDE FILES
=============================================================================*/

#include <gst/gst.h>

/* Realvideo core head files */
#include "realvideo_dec/rv89_dec_api.h"  


#include "mfw_gst_utils.h"
#ifdef MEMORY_DEBUG
#include "mfw_gst_debug.h"
#endif

#include "mfw_gst_rvdecoder.h"
#include "rv_def.h"


/*=============================================================================
                                        LOCAL MACROS
=============================================================================*/
#define RV_DECODER_FATAL_ERROR(...) g_print(RED_STR(__VA_ARGS__))

#define RV_DECODER_FLOW(...) g_print(PURPLE_STR(__VA_ARGS__))

#ifndef RV_DECODER_FLOW
#define RV_DECODER_FLOW(...)
#endif


#define CROP_LEFT_LENGTH  16
#define CROP_TOP_LENGTH    16

#define MFW_GST_RV_VIDEO_CAPS \
    "video/x-pn-realvideo"\
    ", moftag = (int)" STR(HX_MEDIA_VIDEO)\
    ", submoftag = (int){"  STR(HX_RV30VIDEO_ID) ", "\
                            STR(HX_RV40VIDEO_ID) ", "\
                            STR(HX_RV89COMBO_ID) "}"

/* used	for	debugging */
#define	GST_CAT_DEFAULT    mfw_gst_rv_decoder_debug

#define RECEIVE_TIMESTAMP(demuxer, timestamp)\
    do{\
        guint index = (demuxer)->timestamprx;\
        (demuxer)->timestampBuf[index] = (timestamp);\
        (demuxer)->timestamprx = (index+1) & TIMESTAMP_INDEX_MASK;\
    }while (0)

#define GET_TIMESTAMP(demuxer, timestamp)\
    do{\
        guint index = (demuxer)->timestamptx;\
        (timestamp) = (demuxer)->timestampBuf[index];\
        (demuxer)->timestampBuf[index] = GST_CLOCK_TIME_NONE;\
        (demuxer)->timestamptx = (index+1) & TIMESTAMP_INDEX_MASK;\
    }while (0)

#define GST_CLEAN_TIMESTAMP_POOL(demuxer)\
    do{\
        guint index = (demuxer)->timestamptx;\
        while (index!=(demuxer)->timestamprx){\
            (demuxer)->timestampBuf[index] = GST_CLOCK_TIME_NONE;\
            (index) = (index+1) & TIMESTAMP_INDEX_MASK;\
        }\
    }while (0)


#define GST_INIT_TIMESTAMP_POOL(demuxer)\
    do{\
        guint index;\
        for (index=0;index<MAX_TIMESTAMP;index++){\
            (demuxer)->timestampBuf[index] = GST_CLOCK_TIME_NONE;\
        }\
        (demuxer)->timestamprx = (demuxer)->timestamptx = 0;\
    }while (0)
        

#define RV_DISABLE_DBLOCK 0x4
#define RV_DROP_BFRAME 0x8

#ifdef MX35
#define RV_RESOLUTION_SMALL (320*240)  
#define RV_RESOLUTION_MEDIUM (600*300) 
#else
#define RV_RESOLUTION_SMALL (352*288)  
#define RV_RESOLUTION_MEDIUM (800*600) 
#endif

/*=============================================================================
                            LOCAL CONSTANTS
=============================================================================*/


/*=============================================================================
                LOCAL TYPEDEFS (STRUCTURES, UNIONS, ENUMS)
=============================================================================*/
enum {
    MFW_RVDEC_FRAMERATE = 1,
	
};

static GstStaticPadTemplate sink_factory = GST_STATIC_PAD_TEMPLATE("sink",
								   GST_PAD_SINK,
								   GST_PAD_ALWAYS,
								   GST_STATIC_CAPS
								   (MFW_GST_RV_VIDEO_CAPS)
    );
/*=============================================================================
                                LOCAL MACROS
=============================================================================*/
#ifdef MEMORY_DEBUG
#define RV_MALLOC(rv_dec, size)\
    dbg_malloc((&((rv_dec)->memmgr)),(size), "line" STR(__LINE__) "of" STR(__FUNCTION__) )
    
#define RV_FREE(rv_dec, ptr)\
    dbg_free(&((rv_dec)->memmgr), (ptr), "line" STR(__LINE__) "of" STR(__FUNCTION__) )

#else
#define RV_MALLOC(rv_dec, size)\
    g_malloc((size))

#define RV_FREE(rv_dec, ptr)\
    g_free((ptr))
#endif

#define RM_DEPACK16_VALUE(value16, buf)         \
    do {                                        \
        *((guint16 *)(&(value16))) = (((guint16)(((unsigned char *)(buf))[0]))<<8)   \
                   |(guint16)(((unsigned char *)(buf))[1]);        \
    }while(0)   

#define RM_DEPACK32_VALUE(value32, buf)         \
    do {                                        \
        *((guint32 *)(&(value32))) = (((guint32)(((unsigned char *)(buf))[0]))<<24)   \
                   |(((guint32)(((unsigned char *)(buf))[1]))<<16) \
                   |(((guint32)(((unsigned char *)(buf))[2]))<<8)  \
                   |(guint32)(((unsigned char *)(buf))[3]);        \
    }while(0)

#define RM_DEPACK16(value16, buf, length)       \
    do {                                        \
        RM_DEPACK16_VALUE(value16, buf);        \
        (buf)+=2;                               \
        (length)-=2;                            \
    }while(0) 
    
#define RM_DEPACK32(value32, buf, length)       \
    do {                                        \
        RM_DEPACK32_VALUE(value32, buf);        \
        (buf)+=4;                               \
        (length)-=4;                            \
    }while(0)
    
#define RM_HTOL32_BUF(buf32, buf8, length, loop)\
    do {\
        int i;\
        for (i=0;i<(loop);i++){\
            RM_DEPACK32((buf32)[i], buf8, length);\
        }\
    }while (0)

#define RM_IGNORE_BUF(buf, length, numofignore)\
    do {\
        (buf)+=(numofignore);\
        (length)-=(numofignore);\
    }while(0)

/*=============================================================================
                               STATIC VARIABLES
=============================================================================*/
static GstElementClass *parent_class = NULL;


/*=============================================================================
                        STATIC FUNCTION PROTOTYPES
=============================================================================*/
GST_DEBUG_CATEGORY_STATIC(mfw_gst_rv_decoder_debug);
static void mfw_gst_rv_decoder_class_init(MFW_GST_RV_DECODER_CLASS_T
					     * klass);
static void mfw_gst_rv_decoder_base_init(MFW_GST_RV_DECODER_CLASS_T *
					    klass);
static void mfw_gst_rv_decoder_init(MFW_GST_RV_DECODER_INFO_T *
				       filter);

static void mfw_gst_rv_decoder_set_property(GObject * object,
					       guint prop_id,
					       const GValue * value,
					       GParamSpec * pspec);

static void mfw_gst_rv_decoder_get_property(GObject * object,
					       guint prop_id,
					       GValue * value,
					       GParamSpec * pspec);

static gboolean mfw_gst_rv_decoder_sink_event(GstPad *, GstEvent *);
static gboolean mfw_gst_rv_decoder_src_event(GstPad *, GstEvent *);

static gboolean mfw_gst_rv_decoder_set_caps(GstPad * pad,
					       GstCaps * caps);
static GstFlowReturn mfw_gst_rv_decoder_chain(GstPad * pad,
						 GstBuffer * buf);

static GstStateChangeReturn
mfw_gst_rv_decoder_change_state(GstElement *, GstStateChange);

/* Call back function used for direct render v2 */
static void * mfw_gst_rv_getbuffer(void* pvAppContext);
static void mfw_gst_rv_rejectbuffer(void* pbuffer, void* pvAppContext);
static void mfw_gst_rv_releasebuffer(void* pbuffer, void* pvAppContext);

    
/*=============================================================================
                            LOCAL FUNCTIONS
=============================================================================*/


/*=============================================================================
FUNCTION:               mfw_gst_rv_getbuffer

DESCRIPTION:            Callback function for decoder. The call is issued when 
                        decoder need a new frame buffer.

ARGUMENTS PASSED:       pvAppContext -> Pointer to the context variable.

RETURN VALUE:           Pointer to a frame buffer.  -> On success.
                        Null.                       -> On fail.

PRE-CONDITIONS:         None.

POST-CONDITIONS:  	    None.
=============================================================================*/
static void * 
mfw_gst_rv_getbuffer(void* pvAppContext)
{

	MFW_GST_RV_DECODER_INFO_T * rv_dec = (MFW_GST_RV_DECODER_INFO_T *)pvAppContext;
	void * pbuffer;
	GstCaps *caps = NULL;
	int output_size ;
    if (rv_dec->caps_set == FALSE) {
        gint64 start = 0;	/*  proper timestamp has to set here */
        GstCaps *caps;
        gint fourcc = GST_STR_FOURCC("I420");
        guint framerate_n, framerate_d;
        guint crop_right_len = 0, crop_bottom_len = 0;
        crop_right_len = rv_dec->frame_width_padded 
                         - rv_dec->width - CROP_LEFT_LENGTH;

        crop_bottom_len = rv_dec->frame_height_padded
                         - rv_dec->height - CROP_TOP_LENGTH;

        GST_DEBUG("right crop=%d,bottom crop=%d\n", crop_right_len,
                    crop_bottom_len);

        caps =  gst_caps_new_simple("video/x-raw-yuv", "format",
                    GST_TYPE_FOURCC, fourcc, "width",
                    G_TYPE_INT,
                    rv_dec->frame_width_padded,
                    "height", G_TYPE_INT,
                    rv_dec->frame_height_padded,
                    "pixel-aspect-ratio", GST_TYPE_FRACTION, 1, 1,
                    "crop-left-by-pixel", G_TYPE_INT, CROP_LEFT_LENGTH,
                    "crop-top-by-pixel", G_TYPE_INT, CROP_TOP_LENGTH,
                    "crop-right-by-pixel", G_TYPE_INT,
                    (crop_right_len + 7) / 8 * 8,
                    "crop-bottom-by-pixel", G_TYPE_INT,
                    (crop_bottom_len + 7) / 8 * 8, 
                    "num-buffers-required", G_TYPE_INT,
                    BM_GET_BUFFERNUM,
                    NULL); 
        if (rv_dec->frame_rate != 0) {
            framerate_n = (rv_dec->frame_rate * 1000);
            framerate_d = 1000;
            gst_caps_set_simple(caps, "framerate", GST_TYPE_FRACTION,
                                framerate_n, framerate_d, NULL);
        }
        if (!(gst_pad_set_caps(rv_dec->srcpad, caps))) {
            GST_ERROR("\nCould not set the caps for the rvdecoder src pad\n");
        }
        
        rv_dec->caps_set = TRUE;
        gst_caps_unref(caps);
    }

    output_size = rv_dec->frame_width_padded * rv_dec->frame_height_padded *3 /2;

	BM_GET_BUFFER(rv_dec->srcpad, output_size, pbuffer);
    
	return pbuffer;
}

/*=============================================================================
FUNCTION:               mfw_gst_rv_rejectbuffer

DESCRIPTION:            Callback function for decoder. The call is issued when 
                        decoder want to indicate a frame buffer would not be 
                        used as a output.

ARGUMENTS PASSED:       pbuffer      -> Pointer to the frame buffer for reject
                        pvAppContext -> Pointer to the context variable.

RETURN VALUE:           None

PRE-CONDITIONS:         None.

POST-CONDITIONS:  	    None.
=============================================================================*/
static void 
mfw_gst_rv_rejectbuffer(void* pbuffer, void* pvAppContext)
{
    BM_REJECT_BUFFER(pbuffer);
}

/*=============================================================================
FUNCTION:               mfw_gst_rv_releasebuffer

DESCRIPTION:            Callback function for decoder. The call is issued when 
                        decoder want to indicate a frame buffer would never used
                        as a reference.

ARGUMENTS PASSED:       pbuffer      -> Pointer to the frame buffer for release
                        pvAppContext -> Pointer to the context variable.

RETURN VALUE:           None

PRE-CONDITIONS:         None.

POST-CONDITIONS:  	    None.
=============================================================================*/
static void 
mfw_gst_rv_releasebuffer(void* pbuffer, void* pvAppContext)
{
    BM_RELEASE_BUFFER(pbuffer);
}

/*=============================================================================
FUNCTION: mfw_gst_rv_decoder_set_property

DESCRIPTION: sets the property of the element

ARGUMENTS PASSED:
        object     - pointer to the elements object
        prop_id    - ID of the property;
        value      - value of the property set by the application
        pspec      - pointer to the attributes of the property

RETURN VALUE:
        None

PRE-CONDITIONS:
        None

POST-CONDITIONS:


IMPORTANT NOTES:
        None
=============================================================================*/
static void
mfw_gst_rv_decoder_set_property(GObject * object, guint prop_id,
				   const GValue * value,
				   GParamSpec * pspec)
{

    MFW_GST_RV_DECODER_INFO_T *rv_dec =
	MFW_GST_RV_DECODER(object);
    switch (prop_id) {
    case MFW_RVDEC_FRAMERATE:
    	rv_dec->frame_rate = g_value_get_float(value);
    	GST_DEBUG("framerate=%f\n", rv_dec->frame_rate);
    	break;
    default:
    	G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
    	break;
    }
}

/*=============================================================================
FUNCTION: mfw_gst_rv_decoder_get_property

DESCRIPTION: gets the property of the element

ARGUMENTS PASSED:
        object     - pointer to the elements object
        prop_id    - ID of the property;
        value      - value of the property to be set for the next element
        pspec      - pointer to the attributes of the property

RETURN VALUE:
        None

PRE-CONDITIONS:
        None

POST-CONDITIONS:


IMPORTANT NOTES:
        None
=============================================================================*/
static void
mfw_gst_rv_decoder_get_property(GObject * object, guint prop_id,
				   GValue * value, GParamSpec * pspec)
{

    MFW_GST_RV_DECODER_INFO_T *rv_dec =
	MFW_GST_RV_DECODER(object);
    switch (prop_id) {
    case MFW_RVDEC_FRAMERATE:
    	g_value_set_float(value, rv_dec->frame_rate);
    	break;

    default:
    	G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
    	break;
    }
}

static gboolean
mfw_gst_rv_allocatememory(MFW_GST_RV_DECODER_INFO_T * rvdec)
{
    int i, j;
    RV_Dec_Mem_Alloc_Info_Sub * mem_info = &(rvdec->rvConfig->mem_info_sub);
    for(i=0;i<RV_DEC_MAX_NUM_MEM_REQS;i++){
		if(mem_info->mem_desc != RV_DEC_SCRATCH_MEM)
		{   
			mem_info->app_base_ptr = (void *)RV_MALLOC(rvdec, mem_info->size);
            if (mem_info->app_base_ptr==NULL)
            {
				return FALSE;
			}
		}else{
            mem_info->app_base_ptr = 0x1;
		}
        mem_info++;
    }
    return TRUE;
}

static void
mfw_gst_rv_freememory(MFW_GST_RV_DECODER_INFO_T * rvdec)
{
    int i;
    RV_Dec_Mem_Alloc_Info_Sub * mem_info = &(rvdec->rvConfig->mem_info_sub);
    for(i=0;i<RV_DEC_MAX_NUM_MEM_REQS;i++){
		if((mem_info->mem_desc != RV_DEC_SCRATCH_MEM) 
            && (mem_info->app_base_ptr))
		{
			RV_FREE(rvdec, mem_info->app_base_ptr);
            mem_info->app_base_ptr = NULL;
		}
    }
}

static void mfw_gst_rv_decframe(MFW_GST_RV_DECODER_INFO_T * rvdec, GstBuffer * buffer)
{
    RV_Status eDecRetVal;
    RV_Decoder_Decompression_Flags flags = 0;
    RV_Decoder_Decompression_Notes notes;
    struct RVDecoder * rvdecoder = rvdec->rvDecoder;
    gint32 temporal_offset = 0;
    struct RV_Image * img_src = &rvdec->rv_img_src;
    guint32 segment = 0;
    char * buf = NULL;
    guint32 bufsize = 0, datalen;
    GstFlowReturn result = GST_FLOW_OK;
    static guint64 ts = 0;

    if (G_LIKELY(buffer)) {

        RECEIVE_TIMESTAMP(rvdec, GST_BUFFER_TIMESTAMP(buffer)); 
        
        buf = GST_BUFFER_DATA(buffer);
        bufsize = GST_BUFFER_SIZE(buffer);

        RM_DEPACK32(datalen, buf, bufsize);

        RM_IGNORE_BUF(buf, bufsize, 12);
        RM_DEPACK32(segment, buf, bufsize);
        
        if (segment)
        {
            RV_Segment_Info_MSG seg_msg;
            guint32 * tmp = buf;
            seg_msg.message_id = RV_MSG_ID_Set_Decode_Segment_Info;
            seg_msg.number_of_segments = segment - 1;
            
            seg_msg.segment_info = (RV_Segment_Info *)buf;

            RM_HTOL32_BUF(tmp, buf, bufsize, segment<<1);//software decoder accept little-endian
            
            eDecRetVal = Decoder_Custom_Message(rvdecoder->m_coredecoder, &seg_msg.message_id);
            if (eDecRetVal != RV_S_OK)
            {
                GST_ERROR("decoder set segment info message fail \r\n ");
            }
        }

    }else{
        if(rvdec->rvopt.latency_mode){
            flags |= RV_DDF_LAST_FRAME;
        }
    }

    img_src->yuv_info.y_plane = buf;
    img_src->size = bufsize;

    do {
        notes = 0;

        eDecRetVal = Decoder_Decode(rvdecoder->m_coredecoder, img_src, &rvdec->rv_img_des,
    			                    flags, &notes);
    	if (eDecRetVal == RV_S_OK) {
            if (!(notes & RV_DDN_DONT_DRAW)){
                eDecRetVal = Decoder_GetOutputFrame(rvdecoder->m_coredecoder,&rvdec->rv_img_des);
                if(eDecRetVal == RV_S_NULL_PARAM){
                	GST_ERROR("ERROR  --  Null display pointer \r\n");
                	return;
                }

                GET_TIMESTAMP(rvdec, ts);
                
                if (GST_CLOCK_TIME_IS_VALID(ts)){
                    rvdec->next_ts = ts;
        			rvdec->decoded_frames = 0;		               
                }else{
                    ts = rvdec->next_ts+
                            gst_util_uint64_scale(rvdec->decoded_frames, GST_SECOND,
        					              rvdec->frame_rate);
                }
                
                BM_RENDER_BUFFER(rvdec->rv_img_des.yuv_info.y_plane, rvdec->srcpad, result, ts, 0);
                rvdec->decoded_frames++;
            }else{
            }
    	}else if (eDecRetVal==RV_S_DROP_BFRAME){
	        GET_TIMESTAMP(rvdec, ts);
	        rvdec->decoded_frames++;
                
    	}else{
    	    GST_ERROR("Decoder_Decode return error with %d\n", eDecRetVal);
            break;
        }
        
     }while (notes & RV_DDN_MORE_FRAMES);

}

static gint mfw_read_rv_format_info(MFW_GST_RV_DECODER_INFO_T * rv_dec, GstBuffer * gstbuf)
{
    
    char * buf = GST_BUFFER_DATA(gstbuf);
    int formatinfolen;
    
    RM_DEPACK32(formatinfolen, buf, formatinfolen);
    RM_DEPACK32(rv_dec->moftag, buf, formatinfolen);
    RM_DEPACK32(rv_dec->submoftag, buf, formatinfolen);
    RM_DEPACK16(rv_dec->width, buf, formatinfolen);
    RM_DEPACK16(rv_dec->height, buf, formatinfolen);
    RM_IGNORE_BUF(buf,formatinfolen,10);

    GST_BUFFER_DATA(gstbuf)+=26;
    GST_BUFFER_SIZE(gstbuf)-=26;
    return formatinfolen;
}

/*=============================================================================
FUNCTION:               mfw_init_decoder_front

DESCRIPTION:            The function RV_Read_RMFF_Decoder_FrontEnd, decodes the 
                        opaque data header information present in the output of
                        Helix RealFormat SDK's RM parser and populates the front
                        end structure.

ARGUMENTS PASSED:       rv_dec      -> Pointer to the decoder context
                        pFrontEnd   -> Pointer to the decoder frontend.
                        odatabuf    -> Pointer to a gstbuffer contain opaque data
                                       header.

RETURN VALUE:           0 for successfully. non-zero for error

PRE-CONDITIONS:         None.

POST-CONDITIONS:  	    None.
=============================================================================*/
static rv_decoder_front * 
mfw_new_decoder_front(MFW_GST_RV_DECODER_INFO_T * rv_dec, GstBuffer * odatabuf, gint odatalen)
{
	char * baseptr;
    rv_decoder_front * pFrontEnd;
    char ucTmp;
    guint32 i,tmp32;
    int w = 0xFF; /* For run-time endianness detection */

    pFrontEnd = RV_MALLOC(rv_dec, sizeof(rv_decoder_front));

    if (pFrontEnd==NULL){
        goto ErrorFront;
    }

    memset(pFrontEnd, 0, sizeof(rv_decoder_front)); 
    
    if (rv_dec->moftag!= HX_MEDIA_VIDEO)
        goto ErrorFront;

    pFrontEnd->m_SPOExtra = 0;
    pFrontEnd->m_EncodeSize = 0;
    pFrontEnd->m_NumResampledImageSizes = 0;

	/*Only RV30, RV40 and RV89 combo are being supported.*/
	switch (rv_dec->submoftag)
    {
    case HX_RV30VIDEO_ID:
    case HX_RV40VIDEO_ID:
    case HX_RV89COMBO_ID:
        if (rv_dec->submoftag == HX_RV30VIDEO_ID)
            pFrontEnd->m_ECCMask = 0x20;
        else if (rv_dec->submoftag == HX_RV40VIDEO_ID)
            pFrontEnd->m_ECCMask = 0x80;
        else
            pFrontEnd->m_ECCMask = 0;
        break;
    default:
        /* unknown format */
        goto ErrorFront;
    }

	if (odatalen)
    {
        /* Decode opaque data */
        baseptr = GST_BUFFER_DATA(odatabuf);

        /* m_SPOExtra contains CODEC options */
		GET_DATA_BYTES(baseptr,pFrontEnd->m_SPOExtra,tmp32,4);

        pFrontEnd->m_SPOExtra = BYTE_SWAP_U32(pFrontEnd->m_SPOExtra);

        /* m_StreamVersion */
		GET_DATA_BYTES(baseptr,pFrontEnd->m_StreamVersion,tmp32,4);

        pFrontEnd->m_StreamVersion = BYTE_SWAP_U32(pFrontEnd->m_StreamVersion);

        pFrontEnd->m_MajorBitstreamVersion = HX_GET_MAJOR_VERSION(pFrontEnd->m_StreamVersion);
        pFrontEnd->m_MinorBitstreamVersion = HX_GET_MINOR_VERSION(pFrontEnd->m_StreamVersion);

        /* Decode extra opaque data */
        if (!(pFrontEnd->m_MinorBitstreamVersion & RAW_BITSTREAM_MINOR_VERSION))
        {
            if (pFrontEnd->m_MajorBitstreamVersion == RV30_MAJOR_BITSTREAM_VERSION)
            {
                /* RPR (Reference Picture Resampling) sizes */
                pFrontEnd->m_NumResampledImageSizes = (pFrontEnd->m_SPOExtra & RV40_SPO_BITS_NUMRESAMPLE_IMAGES)
                                                       >> RV40_SPO_BITS_NUMRESAMPLE_IMAGES_SHIFT;
                /* loop over dimensions of possible resampled images sizes              */
                /* This byzantine method of extracting bytes is required to solve       */
                /* misaligned write problems in UNIX                                    */
                /* note 2 byte offset in m_pDimensions buffer for resampled sizes         */
                /* these 2 bytes are later filled with the native pels and lines sizes. */
                for(i = 0; i < pFrontEnd->m_NumResampledImageSizes; i++)
                {
					GET_DATA_BYTES(baseptr,ucTmp,tmp32,1);
                    *(pFrontEnd->m_pDimensions + 2*i + 2) = (guint32)(ucTmp<<2); /* width */
					GET_DATA_BYTES(baseptr,ucTmp,tmp32,1);
                    *(pFrontEnd->m_pDimensions + 2*i + 3) = (guint32)(ucTmp<<2); /* height */
                }
            }
            else if (pFrontEnd->m_MajorBitstreamVersion == RV40_MAJOR_BITSTREAM_VERSION)
            {
                /* RV9 largest encoded dimensions */
                if (odatalen>= 12)
                {
					GET_DATA_BYTES(baseptr,pFrontEnd->m_EncodeSize,tmp32,4);

                    pFrontEnd->m_EncodeSize = BYTE_SWAP_U32(pFrontEnd->m_EncodeSize);
                }
            }
        }

        GST_BUFFER_DATA(odatabuf)+=odatalen;
        GST_BUFFER_SIZE(odatabuf)-=odatalen;
    }


    /*Get the fid depending on major and minor version*/
    if ((pFrontEnd->m_MajorBitstreamVersion == RV_BITSTREAM_MAJOR_VERSION)
        &&(pFrontEnd->m_MinorBitstreamVersion == RV_BITSTREAM_MINOR_VERSION)){
        rv_dec->fid = RV_FID_REALVIDEO30;
    }else if (( pFrontEnd->m_MajorBitstreamVersion == RV_BITSTREAM_MAJOR_VERSION_RV8)
              &&( pFrontEnd->m_MinorBitstreamVersion == RV_BITSTREAM_MINOR_VERSION_RV8)){
        rv_dec->fid = RV_FID_REALVIDEO30;
        /* tell the decoder to decode RV8 */
        rv_dec->is_rv8 = TRUE;
    }
    else if ( pFrontEnd->m_MinorBitstreamVersion == RAW_BITSTREAM_MINOR_VERSION){
        rv_dec->fid = RV_FID_RV89COMBO;
        if (pFrontEnd->m_MajorBitstreamVersion == RV_BITSTREAM_MAJOR_VERSION_RV8){
            rv_dec->is_rv8 = TRUE;
        }
    }else{
        GST_ERROR("Failed in parser version %d.%d\n", pFrontEnd->m_MajorBitstreamVersion, pFrontEnd->m_MinorBitstreamVersion);
    }
    
	/* Set largest encoded dimensions */
    pFrontEnd->m_LargestPels = ((pFrontEnd->m_EncodeSize >> 14) & 0x3FFFC);
    if (pFrontEnd->m_LargestPels == 0){
        pFrontEnd->m_LargestPels = rv_dec->width;
    }
    pFrontEnd->m_LargestLines = ((pFrontEnd->m_EncodeSize << 2) & 0x3FFFC);
    if (pFrontEnd->m_LargestLines == 0){
        pFrontEnd->m_LargestLines = rv_dec->height;
    }

    return pFrontEnd;

ErrorFront:
    if (pFrontEnd){
        RV_FREE(rv_dec,pFrontEnd);
        pFrontEnd = NULL;
    }

	return pFrontEnd;
}

static GstFlowReturn
mfw_gst_rv_core_cleanup(MFW_GST_RV_DECODER_INFO_T *rv_dec)
{
    if (rv_dec->rvDecoder){
        RVDecoder_Delete(rv_dec->rvDecoder);
        mfw_gst_rv_freememory(rv_dec);
        RVDecoder_Unload();
        RV_FREE(rv_dec, rv_dec->rvDecoder);
        rv_dec->rvDecoder = NULL;
    }
    if (rv_dec->rvConfig){
        RV_FREE(rv_dec, rv_dec->rvConfig);
        rv_dec->rvConfig = NULL;
    }

    if (rv_dec->rvfront){
        RV_FREE(rv_dec, rv_dec->rvfront);
        rv_dec->rvfront = NULL;
    }
}

static guint
mfw_gst_rv_auto_cpu_scalability(MFW_GST_RV_DECODER_INFO_T *rv_dec)
{
    guint cpu_sca = 0;
    if ((rv_dec->width*rv_dec->height)>RV_RESOLUTION_SMALL){
        RV_DECODER_FLOW("RealVideo deblock disabled!\n", 0);
        cpu_sca |= RV_DISABLE_DBLOCK;
    }
    if ((rv_dec->width*rv_dec->height)>RV_RESOLUTION_MEDIUM){
        RV_DECODER_FLOW("RealVideo drop B frame enabled!\n", 0);
        cpu_sca |= RV_DROP_BFRAME;
    }
    return cpu_sca;
    
}


static GstFlowReturn
mfw_gst_rv_core_init(MFW_GST_RV_DECODER_INFO_T *rv_dec, GstBuffer * buffer)
{
    RV_Decoder_config * rv_config;
    struct RVDecoder * rv_decoder;
    RV_Status eDecRetVal;
    gint odatalen;
    GstFlowReturn ret = GST_FLOW_OK;

    /* malloc decoder relative context and configuration. */
	rv_config =
	    (RV_Decoder_config *) RV_MALLOC(rv_dec, sizeof(RV_Decoder_config));

	if (rv_config == NULL) {
	    GST_ERROR
		("\nUnable to allocate memory for Rv Decoder config structure\n");
	    goto initFail;
	} else {
	    memset(rv_config, 0, sizeof(RV_Decoder_config));
	    rv_dec->rvConfig = rv_config;
	}

    rv_decoder =
        (struct RVDecoder *) RV_MALLOC(rv_dec, sizeof(struct RVDecoder));

    if (rv_decoder == NULL) {
	    GST_ERROR
		("\nUnable to allocate memory for Rv Decoder structure\n");
	    goto initFail;
	} else {
	    memset(rv_decoder, 0, sizeof(struct RVDecoder));
	    rv_dec->rvDecoder = rv_decoder;
	}

    rv_decoder->pvAppContext = rv_dec;

    if (!RVDecoder_Load()){
        GST_ERROR("Can not load Rv decoder\n");
        RVDecoder_Unload();
        goto initFail;
    }else{
    }

    /* populate front end parameters by parse opaque data header. */

    odatalen = mfw_read_rv_format_info(rv_dec, buffer);     
    
    rv_dec->rvfront = mfw_new_decoder_front(rv_dec, buffer, odatalen);

    if (rv_dec->rvfront==NULL){
        goto initFail;
    }
    
    if (rv_dec->rvfront->m_LargestPels!=0){
        rv_dec->width = rv_dec->rvfront->m_LargestPels;
    }
    if (rv_dec->rvfront->m_LargestLines!=0){
        rv_dec->height = rv_dec->rvfront->m_LargestLines;
    }

    RV_DECODER_FLOW("Video resolution %dx%d\n", rv_dec->width, rv_dec->height);

    rv_dec->frame_width_padded = (rv_dec->width+15)/16*16+32; 
    rv_dec->frame_height_padded = (rv_dec->height+15)/16*16+32;

    /* initialize rv image structure for decoder video data. */
    struct RV_Image * imgsrc = &rv_dec->rv_img_src;
    struct RV_Image * imgdes = &rv_dec->rv_img_des;
    
    memset(imgsrc, 0, sizeof(struct RV_Image));
    memset(imgdes, 0, sizeof(struct RV_Image));
	imgsrc->format.fid = rv_dec->fid;
    
    imgdes->format.fid = RV_FID_YUV12;
    
	imgsrc->format.dimensions.height = rv_dec->height;
    imgsrc->format.dimensions.width = rv_dec->width;
    imgdes->format.dimensions = imgsrc->format.dimensions;
   
    eDecRetVal = RVDecoderQueryDecMem(rv_config,imgsrc, rv_dec->is_rv8);

    if (eDecRetVal!=RV_S_OK){
        GST_ERROR("RVDecoderQueryDecMem failed, return %d\n", eDecRetVal);
        RVDecoder_Unload();
        goto initFail;
    }else{
    }

    if (mfw_gst_rv_allocatememory(rv_dec)){
    }else{
        GST_ERROR("Failed to allocate memory for decoder\n");
        mfw_gst_rv_freememory(rv_dec);
        RVDecoder_Unload();
        goto initFail;
    }

    RVDecoder_Init(rv_config, rv_decoder, rv_dec->fid, &eDecRetVal);

    if (eDecRetVal!=RV_S_OK){
        GST_ERROR("RVDecoder_Init failed, return %d\n", eDecRetVal);
        
        RVDecoder_Unload();
        mfw_gst_rv_freememory(rv_dec);
        goto initFail;
    }else{
    }

#define BUFFER_NUM 3
    /* initialize direct render mechanism */
    BM_INIT(0, BUFFER_NUM+2, 0);

    RVDecoder_SetAdditionalCallbackFunction(rv_decoder, E_GET_FRAME, mfw_gst_rv_getbuffer);
	RVDecoder_SetAdditionalCallbackFunction(rv_decoder, E_REJECT_FRAME, mfw_gst_rv_rejectbuffer);
	RVDecoder_SetAdditionalCallbackFunction(rv_decoder, E_RELEASE_FRAME, mfw_gst_rv_releasebuffer);
    
    rv_dec->rvopt.smoothing_postfilter = FALSE;
    rv_dec->rvopt.latency_mode = FALSE;
    rv_dec->rvopt.seek_key_frame_mode = 0;
    rv_dec->rvopt.frames_to_decode_bef_seek = 0;
    rv_dec->rvopt.keyFrameFound = FALSE;
    rv_dec->rvopt.enableSkip = 0;
    rv_dec->rvopt.verbose = FALSE;
    rv_dec->rvopt.start_image = 0;
    rv_dec->rvopt.cpu_scalability_val = mfw_gst_rv_auto_cpu_scalability(rv_dec);
    rv_dec->rvopt.stop_image = (guint32)(0xffffffffUL-1);
    rv_dec->rvopt.cpu_scalable = 1;
    
    if(!( rv_dec->rvfront->m_MinorBitstreamVersion  & RAW_BITSTREAM_MINOR_VERSION)){
        if (rv_dec->rvfront->m_SPOExtra & RV40_SPO_FLAG_BFRAMES){
            rv_dec->rvopt.latency_mode = TRUE;
        }
    }
    
    /* Set any requested decoder options. */
    if (rv_dec->is_rv8)
    {
        RV_MSG_Simple              simple_msg;
        simple_msg.message_id = RV_MSG_ID_RealVideo8;
        simple_msg.value1 = RV_MSG_ENABLE;
        simple_msg.value2 = 0; /* not used */
        eDecRetVal = Decoder_Custom_Message(rv_decoder->m_coredecoder, &simple_msg.message_id);
        if (eDecRetVal != RV_S_OK)
        {
            printf("ERROR -- failed to enable RealVideo 8 decoding \r\n");
            RVDecoder_Delete(rv_decoder);
            RVDecoder_Unload();
            goto initFail;
        }
    }

    if (rv_dec->rvopt.latency_mode)
    {
        RV_MSG_Simple              simple_msg;
        simple_msg.message_id = RV_MSG_ID_Latency_Display_Mode;
        simple_msg.value1 = RV_MSG_ENABLE;
        simple_msg.value2 = 0; /* not used */
        eDecRetVal = Decoder_Custom_Message(rv_decoder->m_coredecoder, &simple_msg.message_id);
        if (eDecRetVal != RV_S_OK)
        {
            printf("ERROR -- failed to enable latency mode \r\n");
            RVDecoder_Delete(rv_decoder);
            RVDecoder_Unload();
            goto initFail;
        }
    }

    if (rv_dec->rvopt.cpu_scalable)
    {
        RV_MSG_Simple              simple_msg;
        simple_msg.message_id = RV_MSG_ID_CPU_Scalability;
        simple_msg.value1 = RV_MSG_SET;
        simple_msg.value2 = rv_dec->rvopt.cpu_scalability_val;
        eDecRetVal = Decoder_Custom_Message(rv_decoder->m_coredecoder, &simple_msg.message_id);
        if (eDecRetVal != RV_S_OK)
        {
            printf("ERROR -- failed to enable cpu scalable \r\n");
            RVDecoder_Delete(rv_decoder);
            RVDecoder_Unload();
            goto initFail;
        }
    }
    
    if((rv_dec->rvfront->m_MajorBitstreamVersion == RV30_MAJOR_BITSTREAM_VERSION) &&
        (rv_dec->rvfront->m_NumResampledImageSizes !=0))
    {
        RV_MSG_RVDecoder_RPR_Sizes rrs;
        
        rrs.message_id = RV_MSG_ID_Set_RVDecoder_RPR_Sizes;
        rrs.num_sizes = rv_dec->rvfront->m_NumResampledImageSizes + 1;   /* includes native size */

        rv_dec->rvfront->m_pDimensions[0] = rv_dec->rvfront->m_LargestPels; /* native width */
        rv_dec->rvfront->m_pDimensions[1] = rv_dec->rvfront->m_LargestLines; /* native height */

        rrs.sizes= (guint32 *)(rv_dec->rvfront->m_pDimensions);
        
        eDecRetVal = Decoder_Custom_Message(rv_decoder->m_coredecoder, &rrs.message_id);
        if (eDecRetVal != RV_S_OK)
        {
            printf("ERROR -- failed to enable RPR sizes \r\n");
            RVDecoder_Delete(rv_decoder);
            RVDecoder_Unload();
            goto initFail;
        }
    }

    eDecRetVal = Decoder_Start_Sequence(rv_config, rv_decoder->m_coredecoder, &(imgsrc->format), &(imgdes->format));

    if (eDecRetVal!=RV_S_OK){
        GST_ERROR("Decoder_Start_Sequence failed, return %d\n", eDecRetVal);
        RVDecoder_Delete(rv_decoder);
        mfw_gst_rv_freememory(rv_dec);
        RVDecoder_Unload();
        goto initFail;
    }else{
    }
    return ret; 

initFail:
    mfw_gst_rv_core_cleanup(rv_dec);
    
    ret = GST_FLOW_ERROR;
    return ret;
}


/*=============================================================================
FUNCTION: mfw_gst_rv_decoder_chain

DESCRIPTION: Initializing the decoder and calling the actual decoding function

ARGUMENTS PASSED:
        pad     - pointer to pad
        buffer  - pointer to received buffer

RETURN VALUE:
        GST_FLOW_OK		- Frame decoded successfully
		GST_FLOW_ERROR	- Failure

PRE-CONDITIONS:
        None

POST-CONDITIONS:


IMPORTANT NOTES:
        None
=============================================================================*/

static GstFlowReturn
mfw_gst_rv_decoder_chain(GstPad * pad, GstBuffer * buffer)
{

    MFW_GST_RV_DECODER_INFO_T *rv_dec;
    GstFlowReturn ret = GST_FLOW_OK;
    unsigned short row, col;
    gint cur_buf = 0;
    
    rv_dec = MFW_GST_RV_DECODER(GST_PAD_PARENT(pad));

    if (G_UNLIKELY(rv_dec->init_done==FALSE)) {
        
    	ret = mfw_gst_rv_core_init(rv_dec, buffer);
        if (ret!=GST_FLOW_OK){
            GST_ERROR("mfw_gst_rv_core_init failed with return %d\n", ret);
            return ret;
        }

        rv_dec->init_done = TRUE;
    }
    
    mfw_gst_rv_decframe(rv_dec,  buffer);
    gst_buffer_unref(buffer);
    
    return ret;    
}

/*=============================================================================
FUNCTION:   mfw_gst_rv_decoder_change_state

DESCRIPTION: this function keeps track of different states of pipeline.

ARGUMENTS PASSED:
        element     -   pointer to element
        transition  -   state of the pipeline

RETURN VALUE:
        GST_STATE_CHANGE_FAILURE    - the state change failed
        GST_STATE_CHANGE_SUCCESS    - the state change succeeded
        GST_STATE_CHANGE_ASYNC      - the state change will happen
                                        asynchronously
        GST_STATE_CHANGE_NO_PREROLL - the state change cannot be prerolled

PRE-CONDITIONS:
        None

POST-CONDITIONS:
        None

IMPORTANT NOTES:
        None
=============================================================================*/
static GstStateChangeReturn
mfw_gst_rv_decoder_change_state(GstElement * element,
				   GstStateChange transition)
{

    GstStateChangeReturn ret = 0;
    MFW_GST_RV_DECODER_INFO_T *rv_dec;
    rv_dec = MFW_GST_RV_DECODER(element);
    switch (transition) {
    case GST_STATE_CHANGE_NULL_TO_READY:
	{
	    rv_dec->eos = 0;
	    rv_dec->caps_set = FALSE;
 	    rv_dec->Time = 0;
	    rv_dec->chain_Time = 0;
	    rv_dec->no_of_frames = 0;
	    rv_dec->avg_fps_decoding = 0.0;
	    rv_dec->no_of_frames_dropped = 0;
        rv_dec->rvConfig = NULL;
        rv_dec->rvDecoder = NULL;
        rv_dec->rvfront = NULL;
        rv_dec->init_done = FALSE;
        rv_dec->flushLastFrameOutput = FALSE;
        GST_INIT_TIMESTAMP_POOL(rv_dec);
#ifdef MEMORY_DEBUG
        init_memmanager(&rv_dec->memmgr, "RV");
#endif         
	}
	break;

    case GST_STATE_CHANGE_READY_TO_PAUSED:
	{
	    rv_dec->decoded_frames = 0;
	    rv_dec->cur_buf = 0;
	    rv_dec->send_newseg = FALSE;
	    rv_dec->next_ts = 0;
	}
	break;

    case GST_STATE_CHANGE_PAUSED_TO_PLAYING:
	break;
    default:
	break;
    }

    ret = parent_class->change_state(element, transition);

    switch (transition) {
	float avg_mcps = 0, avg_plugin_time = 0, avg_dec_time = 0;
    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
	break;
    case GST_STATE_CHANGE_PAUSED_TO_READY:

	rv_dec->send_newseg = FALSE;
	rv_dec->cur_buf = 0;

    mfw_gst_rv_core_cleanup(rv_dec);

	rv_dec->eos = 0;
	rv_dec->caps_set = FALSE;
 	rv_dec->caps_set = FALSE;
	rv_dec->init_done = 0;
	break;
    case GST_STATE_CHANGE_READY_TO_NULL:
        BM_CLEAN_LIST;
#ifdef MEMORY_DEBUG
        deinit_memmanager(&rv_dec->memmgr);
#endif 
	break;
    default:
	break;
    }

    return ret;

}

/*=============================================================================
FUNCTION:		mfw_gst_rv_decoder_sink_event

DESCRIPTION:	This functions handles the events that triggers the
				sink pad of the rv decoder element.

ARGUMENTS PASSED:
        pad        -    pointer to pad
        event      -    pointer to event
RETURN VALUE:
        TRUE       -	if event is sent to sink properly
	    FALSE	   -	if event is not sent to sink properly

PRE-CONDITIONS:
        None

POST-CONDITIONS:
        None

IMPORTANT NOTES:
        None
=============================================================================*/
static gboolean
mfw_gst_rv_decoder_sink_event(GstPad * pad, GstEvent * event)
{


    GstFlowReturn result = GST_FLOW_OK;
    gboolean ret = TRUE;
    MFW_GST_RV_DECODER_INFO_T *rv_dec;
    GstBuffer *outbuffer;
    guint8 *outdata;
    //eRvDecRetType eDecRetVal = E_RVD_FAILURE;
    guint size;
    GstCaps *src_caps = NULL;
    //sRvDecObject *psRvDecObject = NULL;
    gchar *hw_qp;
    //gint8 *quant;
    //sRvDecoderParams *pDecPar;
    unsigned short row, col;
    gint cur_buf = 0;
    GstFormat format;
    rv_dec = MFW_GST_RV_DECODER(GST_PAD_PARENT(pad));
    switch (GST_EVENT_TYPE(event)) {
    case GST_EVENT_NEWSEGMENT:
	{


	    GstFormat format;
	    gint64 start, stop, position;
	    gdouble rate;

	    gst_event_parse_new_segment(event, NULL, &rate, &format,
					&start, &stop, &position);

	    GST_DEBUG(" start = %" GST_TIME_FORMAT, GST_TIME_ARGS(start));
	    GST_DEBUG(" stop = %" GST_TIME_FORMAT, GST_TIME_ARGS(stop));


	    GST_DEBUG(" position in rv  =%" GST_TIME_FORMAT,
		      GST_TIME_ARGS(position));


	    if (GST_FORMAT_TIME == format) {
		rv_dec->decoded_frames = 0;
        rv_dec->next_ts = start;
        GST_CLEAN_TIMESTAMP_POOL(rv_dec);

		result = gst_pad_push_event(rv_dec->srcpad, event);
	    } else {
		GST_DEBUG("dropping newsegment	event in format	%s",
			  gst_format_get_name(format));
		gst_event_unref(event);
		rv_dec->send_newseg = TRUE;
	    }
	    break;
	}
     case GST_EVENT_EOS:
	{
	    RV_DECODER_FLOW("\nRVDecoder: Got an EOS from Demuxer\n", 0);
        
	    rv_dec->eos = 1;
        mfw_gst_rv_decframe(rv_dec, NULL);
	    result = gst_pad_push_event(rv_dec->srcpad, event);
	    if (!result) {
		GST_ERROR("\n Error in pushing the event,result is %d\n",
			  result);
	    } else {
		GST_DEBUG("\n EOS event sent to the peer element\n");
	    }
	    break;
	}
     case GST_EVENT_FLUSH_STOP:
	{

	    result = gst_pad_push_event(rv_dec->srcpad, event);

	    if (TRUE != result) {
		GST_ERROR("\n Error in pushing the event,result	is %d\n",
			  result);
	    }
	    break;
	}
	/* not handling flush start */
    case GST_EVENT_FLUSH_START:
    default:
	{
	    result = gst_pad_event_default(pad, event);
	    break;
	}

    }

    if (result == GST_FLOW_OK)
	ret = TRUE;
    else
	ret = FALSE;
    return ret;
}


/*=============================================================================
FUNCTION:   mfw_gst_rv_decoder_src_event

DESCRIPTION: This functions handles the events that triggers the
			 source pad of the rv decoder element.

ARGUMENTS PASSED:
        pad        -    pointer to pad
        event      -    pointer to event
RETURN VALUE:
	    FALSE	   -	if event is not sent to src properly
        TRUE       -	if event is sent to src properly
PRE-CONDITIONS:
        None

POST-CONDITIONS:
        None

IMPORTANT NOTES:
        None
=============================================================================*/
static gboolean
mfw_gst_rv_decoder_src_event(GstPad * pad, GstEvent * event)
{
    gboolean res;

    MFW_GST_RV_DECODER_INFO_T *rv_dec =
	MFW_GST_RV_DECODER(gst_pad_get_parent(pad));

#if 1
    switch (GST_EVENT_TYPE(event)) {
    case GST_EVENT_SEEK:
	res = gst_pad_push_event(rv_dec->sinkpad, event);
	break;
#ifdef SFD_ENABLED    
	/* judge the timestamp from system time */
    case GST_EVENT_QOS:
    {
    if (rv_dec->is_sfd) { //
        struct sfd_frames_info *pSfd_info = &rv_dec->sfd_info;
		gdouble proportion;
		GstClockTimeDiff diff;
		GstClockTime timestamp;

		gst_event_parse_qos(event, &proportion, &diff, &timestamp);

		if (diff >= 0) {
	        GST_QOS_EVENT_HANDLE(pSfd_info,diff,rv_dec->frame_rate);
		} else {
		    GST_DEBUG
			("the time of decoding is before the system, it is OK\n");
		}
		res = gst_pad_push_event(rv_dec->sinkpad, event);
    }
	break;
    }
#endif
    case GST_EVENT_NAVIGATION:
	/* Forward a navigation event unchanged */
    default:
	res = gst_pad_push_event(rv_dec->sinkpad, event);
	break;
    }

    gst_object_unref(rv_dec);
    return res;
#endif
}
/*=============================================================================
FUNCTION:               src_templ

DESCRIPTION:            Template to create a srcpad for the decoder.

ARGUMENTS PASSED:       None.


RETURN VALUE:           a GstPadTemplate


PRE-CONDITIONS:  	    None

POST-CONDITIONS:   	    None

IMPORTANT NOTES:   	    None
=============================================================================*/
static GstPadTemplate *src_templ(void)
{
    static GstPadTemplate *templ = NULL;
    if (!templ) {
	GstCaps *caps;
	GstStructure *structure;
	GValue list = { 0 }
	, fps = {
	0}
	, fmt = {
	0};
	gchar *fmts[] = { "YV12", "I420", "Y42B", NULL };
	guint n;

	caps = gst_caps_new_simple("video/x-raw-yuv",
				   "format", GST_TYPE_FOURCC,
				   GST_MAKE_FOURCC('I', '4', '2', '0'),
				   "width", GST_TYPE_INT_RANGE, 16, 4096,
				   "height", GST_TYPE_INT_RANGE, 16, 4096,
				   NULL);
	templ =
	    gst_pad_template_new("src", GST_PAD_SRC, GST_PAD_ALWAYS, caps);
    }

    return templ;
}
/*=============================================================================
FUNCTION:   mfw_gst_rv_decoder_set_caps

DESCRIPTION:    this function handles the link with other plug-ins and used for
                capability negotiation  between pads

ARGUMENTS PASSED:
        pad        -    pointer to GstPad
        caps       -    pointer to GstCaps

RETURN VALUE:
        TRUE       -    if capabilities are set properly
        FALSE      -    if capabilities are not set properly
PRE-CONDITIONS:
        None

POST-CONDITIONS:
        None

IMPORTANT NOTES:
        None
=============================================================================*/
static gboolean
mfw_gst_rv_decoder_set_caps(GstPad * pad, GstCaps * caps)
{
    MFW_GST_RV_DECODER_INFO_T *rv_dec;
    GstStructure *structure = gst_caps_get_structure(caps, 0);
    rv_dec = MFW_GST_RV_DECODER(GST_OBJECT_PARENT(pad));

    const gchar *mime;
    gint32 frame_rate_de = 0;
    gint32 frame_rate_nu = 0;
    gint32 value;

    
    
    mime = gst_structure_get_name(structure);

    gst_structure_get_fraction(structure, "framerate", &frame_rate_nu,
			       &frame_rate_de);

    if (frame_rate_de != 0) {
	    rv_dec->frame_rate = (gfloat) (frame_rate_nu) / frame_rate_de;
    }

    if (rv_dec->frame_rate==0){
        rv_dec->frame_rate = 30;//default
    }
    


    if (!gst_pad_set_caps(pad, caps)) {
	return FALSE;
    }
    return TRUE;
}


/*=============================================================================
FUNCTION:   mfw_gst_rv_decoder_init

DESCRIPTION:This function creates the pads on the elements and register the
			function pointers which operate on these pads.

ARGUMENTS PASSED:
        pointer the rv_decoder element handle.

RETURN VALUE:
        None

PRE-CONDITIONS:
        _base_init and _class_init are called

POST-CONDITIONS:
        None

IMPORTANT NOTES:
        None
=============================================================================*/
static void
mfw_gst_rv_decoder_init(MFW_GST_RV_DECODER_INFO_T * rv_dec)
{
    GstElementClass *klass = GST_ELEMENT_GET_CLASS(rv_dec);
    rv_dec->init_done = 0;

    rv_dec->sinkpad =
	gst_pad_new_from_template(gst_element_class_get_pad_template
				  (klass, "sink"), "sink");
    gst_pad_set_setcaps_function(rv_dec->sinkpad,
				 mfw_gst_rv_decoder_set_caps);
    
    gst_pad_set_chain_function(rv_dec->sinkpad,
			       mfw_gst_rv_decoder_chain);
    gst_pad_set_event_function(rv_dec->sinkpad,
			       GST_DEBUG_FUNCPTR
			       (mfw_gst_rv_decoder_sink_event));

    gst_element_add_pad(GST_ELEMENT(rv_dec), rv_dec->sinkpad);

    rv_dec->srcpad = gst_pad_new_from_template(src_templ(), "src");
    gst_pad_set_event_function(rv_dec->srcpad,
			       GST_DEBUG_FUNCPTR
			       (mfw_gst_rv_decoder_src_event));
    
    gst_element_add_pad(GST_ELEMENT(rv_dec), rv_dec->srcpad);
    rv_dec->frame_rate = 0.0;
#ifdef SFD_ENABLED    
    INIT_SFD_INFO(&rv_dec->sfd_info);
    rv_dec->is_sfd = TRUE;
#endif

#define MFW_GST_RV_DECODER_PLUGIN VERSION
    PRINT_CORE_VERSION(RVDecoder_CodecVersionInfo());
    PRINT_PLUGIN_VERSION(MFW_GST_RV_DECODER_PLUGIN);    
}

/*=============================================================================
FUNCTION:   mfw_gst_rv_decoder_class_init

DESCRIPTION:Initialise the class only once (specifying what signals,
            arguments and virtual functions the class has and setting up
            global state)
ARGUMENTS PASSED:
       	klass   - pointer to rv element class

RETURN VALUE:
        None

PRE-CONDITIONS:
        None

POST-CONDITIONS:
        None

IMPORTANT NOTES:
        None
=============================================================================*/

static void
mfw_gst_rv_decoder_class_init(MFW_GST_RV_DECODER_CLASS_T * klass)
{
    GObjectClass *gobject_class = NULL;
    GstElementClass *gstelement_class = NULL;
    gobject_class = (GObjectClass *) klass;
    gstelement_class = (GstElementClass *) klass;
    parent_class = g_type_class_ref(GST_TYPE_ELEMENT);
    gobject_class->set_property = mfw_gst_rv_decoder_set_property;
    gobject_class->get_property = mfw_gst_rv_decoder_get_property;
    gstelement_class->change_state = mfw_gst_rv_decoder_change_state;

     g_object_class_install_property(gobject_class, MFW_RVDEC_FRAMERATE,
				    g_param_spec_float("framerate",
						       "FrameRate",
						       "gets the framerate at which the input stream is to be displayed",
						       -G_MAXFLOAT,
						       G_MAXFLOAT, 0.0,
						       G_PARAM_READWRITE));
 
 	
}

/*=============================================================================
FUNCTION:  mfw_gst_rv_decoder_base_init

DESCRIPTION:
            rvdecoder element details are registered with the plugin during
            _base_init ,This function will initialise the class and child
            class properties during each new child class creation


ARGUMENTS PASSED:
        Klass   -   pointer to rv decoder plug-in class
        g_param_spec_float("framerate", "FrameRate", 
        "gets the framerate at which the input stream is to be displayed",
        None

PRE-CONDITIONS:
        None

POST-CONDITIONS:
        None

IMPORTANT NOTES:
        None
=============================================================================*/
static void
mfw_gst_rv_decoder_base_init(MFW_GST_RV_DECODER_CLASS_T * klass)
{
    static GstElementDetails element_details = {
	"Freescale RealMedia video decoder",
	"Codec/Decoder/Video",
	"Decodes RealVideo Bitstreams",
	FSL_GST_MM_PLUGIN_AUTHOR
    };
    GstElementClass *element_class = GST_ELEMENT_CLASS(klass);
    gst_element_class_add_pad_template(element_class, src_templ());
    gst_element_class_add_pad_template(element_class,
				       gst_static_pad_template_get
				       (&sink_factory));
    gst_element_class_set_details(element_class, &element_details);
}

/*=============================================================================
FUNCTION: mfw_gst_rv_decoder_get_type

DESCRIPTION:    intefaces are initiated in this function.you can register one
                or more interfaces  after having registered the type itself.

ARGUMENTS PASSED:
            None

RETURN VALUE:
                 A numerical value ,which represents the unique identifier of this
            element(rvdecoder)

PRE-CONDITIONS:
            None

POST-CONDITIONS:
            None

IMPORTANT NOTES:
            None
=============================================================================*/

GType mfw_gst_rv_decoder_get_type(void)
{
    static GType rv_decoder_type = 0;

    if (!rv_decoder_type) {
        static const GTypeInfo rv_decoder_info = {
            sizeof(MFW_GST_RV_DECODER_CLASS_T),
            (GBaseInitFunc) mfw_gst_rv_decoder_base_init,
            NULL,
            (GClassInitFunc) mfw_gst_rv_decoder_class_init,
            NULL,
            NULL,
            sizeof(MFW_GST_RV_DECODER_INFO_T),
            0,
            (GInstanceInitFunc) mfw_gst_rv_decoder_init,
	};
        
	rv_decoder_type = g_type_register_static(GST_TYPE_ELEMENT,
						    "MFW_GST_RV_DECODER_INFO_T",
						    &rv_decoder_info,
						    0);
    }
    
    GST_DEBUG_CATEGORY_INIT(mfw_gst_rv_decoder_debug,
			    "mfw_rvdecoder", 0,
			    "FreeScale's RealVideo Decoder's Log");
    return rv_decoder_type;
}

/*=============================================================================
FUNCTION:   plugin_init

DESCRIPTION:    special function , which is called as soon as the plugin or
                element is loaded and information returned by this function
                will be cached in central registry

ARGUMENTS PASSED:
        plugin     -    pointer to container that contains features loaded
                        from shared object module

RETURN VALUE:
        return TRUE or FALSE depending on whether it loaded initialized any
        dependency correctly

PRE-CONDITIONS:
        None

POST-CONDITIONS:
        None

IMPORTANT NOTES:
        None
=============================================================================*/
static gboolean plugin_init(GstPlugin * plugin)
{
    return gst_element_register(plugin, "mfw_rvdecoder",
				GST_RANK_PRIMARY,
				MFW_GST_TYPE_RV_DECODER);
}

/*****************************************************************************/
/*    This is used to define the entry point and meta data of plugin         */
/*****************************************************************************/
GST_PLUGIN_DEFINE(GST_VERSION_MAJOR,	/* major version of Gstreamer    */
		  GST_VERSION_MINOR,	/* minor version of Gstreamer    */
		  "mfw_rvdecoder",	/* name of the plugin            */
		  "decodes the RealVideo bitstreams",	/* what plugin actually does     */
		  plugin_init,	/* first function to be called   */
		  VERSION,
		  GST_LICENSE_UNKNOWN,
		  FSL_GST_MM_PLUGIN_PACKAGE_NAME, FSL_GST_MM_PLUGIN_PACKAGE_ORIG)
