#include <bognor/br-iface-player.h>
#include <clutter-gst/clutter-gst.h>

#include "hrn.h"
#include "hrn-state-manager.h"
#include "hrn-theatre.h"
#include "hrn-audio-viewer.h"
#include "hrn-iface-player.h"
#include "hrn-image-player.h"
#include "hrn-video-player.h"

enum
{
  PROP_0,
};

struct _HrnTheatrePrivate
{
  HrnStateManager *state_manager;
  ClutterActor *backdrop;

  ClutterActor *player;
  BklItemType   player_type;

  guint32 progress_id;

  gboolean active;   /* Is the theatre being shown? */
  gboolean playing;

  gboolean slideshow; /* We can override playing for images by setting
                         slideshow */

  guint32 no;
};

#define NUMBER_OF_BUTTONS    9 /* Keep in sync with hrn-thumb-bar.c */

#define GET_PRIVATE(obj)    (G_TYPE_INSTANCE_GET_PRIVATE ((obj), \
                                                          HRN_TYPE_THEATRE, \
                                                          HrnTheatrePrivate))
G_DEFINE_TYPE (HrnTheatre, hrn_theatre, CLUTTER_TYPE_GROUP);

static void
hrn_theatre_finalize (GObject *object)
{
  G_OBJECT_CLASS (hrn_theatre_parent_class)->finalize (object);
}

static void
hrn_theatre_dispose (GObject *object)
{
  G_OBJECT_CLASS (hrn_theatre_parent_class)->dispose (object);
}

static void
hrn_theatre_set_property (GObject *object, guint prop_id, const GValue *value,
                          GParamSpec   *pspec)
{
  switch (prop_id)
    {
      default:
        break;
    }
}

static void
hrn_theatre_get_property (GObject *object, guint prop_id, GValue     *value,
                          GParamSpec *pspec)
{
  switch (prop_id)
    {
      default:
        break;
    }
}

static void
hrn_theatre_allocate (ClutterActor *actor, const ClutterActorBox *box,
                      ClutterAllocationFlags flags)
{
  HrnTheatre        *theatre = (HrnTheatre *) actor;
  HrnTheatrePrivate *priv    = theatre->priv;

  CLUTTER_ACTOR_CLASS (hrn_theatre_parent_class)->allocate (actor, box,
                                                            flags);

  if (priv->player)
    {
      clutter_actor_allocate_preferred_size (priv->player,
                                             flags);
    }
}

static gboolean
hrn_theatre_key_press (ClutterActor *actor, ClutterKeyEvent *event)
{
  HrnTheatre        *theatre = (HrnTheatre *) actor;
  HrnTheatrePrivate *priv    = theatre->priv;
  BklItem           *item;

  switch (clutter_event_get_key_symbol ((ClutterEvent *) event))
    {
      case CLUTTER_Left:
        if (priv->no > 0)
          {
            item = hrn_view_item_to_bkl_item (priv->no - 1);
            if (item)
              {
                hrn_play_now (item);
              }
          }
        break;

      case CLUTTER_Right:
        item = hrn_view_item_to_bkl_item (priv->no + 1);
        if (item)
          {
            hrn_play_now (item);
          }
        break;

      default:
        break;
    }

  return FALSE;
}

static void
hrn_theatre_class_init (HrnTheatreClass *klass)
{
  GObjectClass      *o_class = (GObjectClass *) klass;
  ClutterActorClass *a_class = (ClutterActorClass *) klass;

  o_class->dispose      = hrn_theatre_dispose;
  o_class->finalize     = hrn_theatre_finalize;
  o_class->set_property = hrn_theatre_set_property;
  o_class->get_property = hrn_theatre_get_property;

  a_class->allocate        = hrn_theatre_allocate;
  a_class->key_press_event = hrn_theatre_key_press;

  g_type_class_add_private (klass, sizeof (HrnTheatrePrivate));
}

static void
player_position_changed (HrnVideoPlayer *player,
                         double          position,
                         HrnTheatre     *theatre)
{
  HrnTheatrePrivate *priv = theatre->priv;

  br_iface_player_emit_position_changed
    ((BrIfacePlayer *) priv->state_manager, position);
}

static void
player_eos (HrnVideoPlayer *player,
            HrnTheatre     *theatre)
{
  HrnTheatrePrivate *priv = theatre->priv;

  br_iface_player_emit_uri_completed
    ((BrIfacePlayer *) priv->state_manager, "");
}

static void
hrn_theatre_init (HrnTheatre *self)
{
  HrnTheatrePrivate *priv  = GET_PRIVATE (self);
  ClutterColor       black = { 0, 0, 0, 0xff };

  self->priv = priv;

  priv->active  = FALSE;
  priv->playing = FALSE;

  clutter_actor_set_reactive ((ClutterActor *) self, TRUE);
  clutter_actor_set_opacity ((ClutterActor *) self, 0x00);

  priv->backdrop = clutter_rectangle_new_with_color (&black);
  clutter_group_add (CLUTTER_GROUP (self), priv->backdrop);
  clutter_actor_set_reactive (priv->backdrop, TRUE);
  clutter_actor_hide (priv->backdrop);
  clutter_actor_set_opacity (priv->backdrop, 0);

  /* hard coded size that should guarantee the rectangle
     covering up even with huge stages */
  clutter_actor_set_position (priv->backdrop, -400, -400);
  clutter_actor_set_size (priv->backdrop, 4096, 4096);
}

static void
theatre_play (HrnTheatre *theatre)
{
  HrnTheatrePrivate *priv = theatre->priv;

  priv->playing = TRUE;

  if (priv->player)
    {
      gboolean playing = priv->playing;

      if (priv->player_type == BKL_ITEM_TYPE_IMAGE) {
        playing = priv->slideshow;
      }

      hrn_iface_player_set_playing ((HrnIfacePlayer *) priv->player, playing);
    }
}

static void
theatre_stop (HrnTheatre *theatre)
{
  HrnTheatrePrivate *priv = theatre->priv;

  priv->playing = FALSE;

  if (priv->player)
    {
      hrn_iface_player_set_playing ((HrnIfacePlayer *) priv->player, FALSE);
    }
}

static void
theatre_playing (HrnStateManager *state,
                 gboolean         playing,
                 HrnTheatre      *theatre)
{
  if (playing) {
    theatre_play (theatre);
  } else {
    theatre_stop (theatre);
  }
}

static void
theatre_progress_changed (HrnStateManager *state,
                          double           progress,
                          HrnTheatre      *theatre)
{
  HrnTheatrePrivate *priv = theatre->priv;

  if (priv->player && priv->progress_id)
    {
      g_signal_handler_block (priv->player, priv->progress_id);
      hrn_iface_player_set_position ((HrnIfacePlayer *) priv->player,
                                     progress);
      g_signal_handler_unblock (priv->player, priv->progress_id);
    }
}

static void
make_video_player (HrnTheatre *theatre)
{
  HrnTheatrePrivate *priv = theatre->priv;

  priv->player      = g_object_new (HRN_TYPE_VIDEO_PLAYER, NULL);
  priv->progress_id = g_signal_connect (priv->player, "position-changed",
                                        G_CALLBACK (player_position_changed),
                                        theatre);
  g_signal_connect (priv->player, "eos",
                    G_CALLBACK (player_eos), theatre);

  clutter_group_add (CLUTTER_ACTOR (theatre), priv->player);
  clutter_actor_show (priv->player);
}

static void
make_image_player (HrnTheatre *theatre)
{
  HrnTheatrePrivate *priv = theatre->priv;

  priv->player = g_object_new (HRN_TYPE_IMAGE_PLAYER, NULL);
  g_signal_connect (priv->player, "eos",
                    G_CALLBACK (player_eos), theatre);

  clutter_group_add (CLUTTER_GROUP (theatre), priv->player);
  clutter_actor_show (priv->player);
}

static void
make_audio_viewer (HrnTheatre *theatre)
{
  HrnTheatrePrivate *priv = theatre->priv;

  priv->player = g_object_new (HRN_TYPE_AUDIO_VIEWER, NULL);

  clutter_group_add (CLUTTER_GROUP (theatre), priv->player);
  clutter_actor_set_position (priv->player, 112, 140);
  clutter_actor_set_size (priv->player, 800, 100);
  clutter_actor_show (priv->player);
}

static void
play_uri (HrnStateManager *state,
          const char      *uri,
          HrnTheatre      *theatre)
{
  HrnTheatrePrivate *priv = theatre->priv;
  gboolean playing = priv->playing;
  char *mimetype;


  if (uri == NULL || *uri == '\0')
    {
      return;
    }

  if (g_str_has_prefix (uri, "file:"))
    {
      GFile *file = g_file_new_for_uri (uri);

      if (!g_file_query_exists (file, NULL))
        {
          g_debug ("%s: Bailing local file %s doesnt exist", G_STRLOC, uri);
          g_object_unref (file);
          br_iface_player_emit_uri_completed ((BrIfacePlayer*)state, "");
          return;
        }
      g_object_unref (file);
    }


  mimetype = hrn_resolve_mimetype (uri);
  if (g_str_has_prefix (mimetype, "video/"))
    {
      g_debug ("Setting video: %s", uri);

      if (priv->player && priv->player_type != BKL_ITEM_TYPE_VIDEO)
        {
          clutter_actor_destroy (priv->player);
          priv->player = NULL;
        }

      if (priv->player == NULL)
        {
          make_video_player (theatre);
          priv->player_type = BKL_ITEM_TYPE_VIDEO;
        }
    }
  else if (g_str_has_prefix (mimetype, "image/"))
    {
      if (priv->player && priv->player_type != BKL_ITEM_TYPE_IMAGE)
        {
          clutter_actor_destroy (priv->player);
          priv->player = NULL;
        }

      if (priv->player == NULL)
        {
          make_image_player (theatre);
          priv->player_type = BKL_ITEM_TYPE_IMAGE;
        }
    }
  else if (g_str_has_prefix (mimetype, "audio/"))
    {
      if (priv->player && priv->player_type != BKL_ITEM_TYPE_AUDIO)
        {
          clutter_actor_destroy (priv->player);
          priv->player = NULL;
        }

      if (priv->player == NULL)
        {
          make_audio_viewer (theatre);
          priv->player_type = BKL_ITEM_TYPE_AUDIO;
        }
    }
  g_free (mimetype);

  if (priv->player)
    {
      hrn_iface_player_set_uri ((HrnIfacePlayer *) priv->player, uri);
    }

  if (priv->player_type == BKL_ITEM_TYPE_IMAGE) {
      playing = priv->slideshow;
  }

  if (priv->player)
    {
      hrn_iface_player_set_playing ((HrnIfacePlayer *) priv->player,
                                    playing);
    }

  /* FIXME: This is conceptually broken because the uri may not actually
     be in the current view. We need to separate all this into model/view */
  priv->no = hrn_view_find_uri (uri);
}

static void
hrn_theatre_got_focus_cb (ClutterStage *stage,
                          HrnTheatre   *theatre)
{
  HrnTheatrePrivate *priv = theatre->priv;

  if (priv->active)
    {
      /* Tell Bognor that we're now active */
      hrn_state_manager_set_can_show_visual (priv->state_manager, TRUE);
    }
}

static void
hrn_theatre_lost_focus_cb (ClutterStage *stage,
                           HrnTheatre   *theatre)
{
  HrnTheatrePrivate *priv = theatre->priv;
  if (priv->active)
    {
      /* Tell Bognor that we're no longer active */
      hrn_state_manager_set_can_show_visual (priv->state_manager, FALSE);
    }
}

static void
state_ready_cb (HrnStateManager *state,
                HrnTheatre      *theatre)
{
  play_uri (state, hrn_state_manager_get_uri (state), theatre);
  theatre_playing (state, hrn_state_manager_get_playing (state), theatre);
  theatre_progress_changed (state, hrn_state_manager_get_progress (state),
                            theatre);
}

HrnTheatre *
hrn_theatre_new (HrnStateManager *state)
{
  HrnTheatre *theatre = g_object_new (HRN_TYPE_THEATRE, NULL);
  HrnTheatrePrivate *priv = theatre->priv;

  priv->state_manager = state;
  g_signal_connect (priv->state_manager, "ready",
                    G_CALLBACK (state_ready_cb), theatre);
  g_signal_connect (priv->state_manager, "play-uri",
                    G_CALLBACK (play_uri), theatre);
  g_signal_connect (priv->state_manager, "playing",
                    G_CALLBACK (theatre_playing), theatre);
  g_signal_connect (priv->state_manager, "progress-changed",
                    G_CALLBACK (theatre_progress_changed), theatre);

  /* We register the focus handler with the stage, inside the handlers
   * we make sure that the event we're in theatre mode by checking the
   * ->active state.
   */
  g_signal_connect (clutter_stage_get_default (), "activate",
                    G_CALLBACK (hrn_theatre_got_focus_cb), theatre);
  g_signal_connect (clutter_stage_get_default (), "deactivate",
                    G_CALLBACK (hrn_theatre_lost_focus_cb), theatre);


  return theatre;
}

gboolean
hrn_theatre_get_active (HrnTheatre *theatre)
{
  HrnTheatrePrivate *priv = theatre->priv;

  return priv->active;
}

void
hrn_theatre_show (HrnTheatre *theatre)
{
  HrnTheatrePrivate *priv = theatre->priv;

  if (priv->active)
    {
      return;
    }

  priv->active = TRUE;

  clutter_actor_show ((ClutterActor *)hrn_theatre_parent);
  clutter_actor_hide ((ClutterActor *)hrn_controls);

  clutter_actor_show ((ClutterActor *) theatre);
  clutter_actor_animate ((ClutterActor *) theatre, CLUTTER_LINEAR,
                         HRN_TO_THEATRE_DURATION,
                         "opacity", 0xff,
                         NULL);

  clutter_actor_show (priv->backdrop);
  clutter_actor_animate (priv->backdrop, CLUTTER_LINEAR,
                         HRN_TO_THEATRE_DURATION,
                         "opacity", 255,
                         NULL);

  /* FIXME: This needs to be somewhere sane */
  clutter_actor_animate (hrn_controls_frame, CLUTTER_LINEAR,
                         HRN_TO_THEATRE_DURATION,
                         "opacity", 0x00,
                         NULL);
  clutter_actor_animate (hrn_scrollbar, CLUTTER_EASE_IN_OUT_CUBIC,
                         HRN_TO_THEATRE_DURATION,
                         "x", 30.0,
                         NULL);

  hrn_controls_can_has_fade = TRUE;
  hrn_controls_wake ();   /* wake it so that it can do away .. */

  clutter_actor_animate (hrn_content_area, CLUTTER_EASE_IN_OUT_CUBIC,
                         SIDEBAR_OUT_DURATION,
                         "depth", 0.0,
                         "x", 0.0,
                         NULL);

  clutter_actor_animate (hrn_sidebar, CLUTTER_EASE_IN_OUT_CUBIC,
                         SIDEBAR_OUT_DURATION,
                         "opacity", 190,
                         NULL);
  clutter_stage_set_key_focus (CLUTTER_STAGE (clutter_stage_get_default ()),
                               (ClutterActor *) theatre);

  /* Tell Bognor that we're now active */
  hrn_state_manager_set_can_show_visual (priv->state_manager, TRUE);
}

static void
hide_on_complete (ClutterAnimation *anim,
                  HrnTheatre       *theatre)
{
  HrnTheatrePrivate *priv = theatre->priv;

  clutter_actor_hide (priv->backdrop);
  clutter_actor_hide ((ClutterActor *) theatre);
}

void
hrn_theatre_hide (HrnTheatre *theatre)
{
  HrnTheatrePrivate *priv = theatre->priv;
  ClutterAnimation  *anim;

  if (priv->active == FALSE)
    {
      return;
    }
  clutter_actor_hide ((ClutterActor *)hrn_theatre_parent);
  clutter_actor_show ((ClutterActor *)hrn_controls);

  priv->active = FALSE;

  anim = clutter_actor_animate (priv->backdrop, CLUTTER_EASE_IN_OUT_CUBIC,
                                HRN_FROM_THEATRE_DURATION * 1.5,
                                "opacity", 0,
                                NULL);
  g_signal_connect (anim, "completed",
                    G_CALLBACK (hide_on_complete), theatre);
  clutter_actor_animate ((ClutterActor *) theatre, CLUTTER_EASE_IN_OUT_CUBIC,
                         HRN_FROM_THEATRE_DURATION * 1.5,
                         "opacity", 0,
                         NULL);

  /* Tell Bognor...*/
  hrn_state_manager_set_can_show_visual (priv->state_manager, FALSE);

  /* FIXME: This needs to be somewhere sane too */
  clutter_actor_show (hrn_view);

  clutter_actor_animate (hrn_scrollbar, CLUTTER_EASE_IN_OUT_CUBIC,
                         HRN_FROM_THEATRE_DURATION,
                         "x", 0.0,
                         NULL);
  clutter_actor_animate (hrn_controls_frame, CLUTTER_LINEAR,
                         HRN_FROM_THEATRE_DURATION,
                         "opacity", 0xFF,
                         NULL);
  clutter_actor_animate (hrn_view, CLUTTER_LINEAR,
                         HRN_FROM_THEATRE_DURATION,
                         "opacity", 0xFF,
                         NULL);
  clutter_actor_animate (hrn_sidebar, CLUTTER_EASE_IN_OUT_CUBIC,
                         SIDEBAR_IN_DURATION,
                         "opacity", 255,
                         NULL);
  if (hrn_sidebar_visible ())
    {
      clutter_actor_animate (hrn_content_area, CLUTTER_EASE_IN_OUT_QUINT,
                             SIDEBAR_IN_DURATION,
                             "depth", -200.0,
                             "x", 110.0,
                             NULL);
    }

  hrn_controls_can_has_fade = FALSE;
  hrn_controls_wake ();   /* wake it so that it notices it shouldn't fade */
  hrn_toolbar_focused (HRN_TOOLBAR (hrn_toolbar));

  /* FIXME: fake_out (); */
}

void
hrn_theatre_set_image_slideshow (HrnTheatre *theatre,
                                 gboolean    enabled)
{
  HrnTheatrePrivate *priv = theatre->priv;

  if (priv->player_type == BKL_ITEM_TYPE_IMAGE && priv->active) {
    hrn_iface_player_set_playing ((HrnIfacePlayer *) priv->player, enabled);
  }

  priv->slideshow = enabled;
}
