#include "hrn.h"
#include "hrn-toolbar.h"

enum
{
  PROP_0,
};

struct _HrnToolbarPrivate
{
  NbtkWidget *zoom_bar;
  NbtkWidget *zoom_bar_label;

  NbtkWidget *view_table;

  NbtkWidget *close_button;

  /* Things in view_group */
  NbtkWidget *filter_bar;
  NbtkWidget *sort_by_button;
  NbtkWidget *show_button;
  NbtkWidget *settings_button;

  int filter_mode;

  guint32        search_handler;
  HrnToolbarMode mode;
};

#define GET_PRIVATE(obj)    (G_TYPE_INSTANCE_GET_PRIVATE ((obj), \
                                                          HRN_TYPE_TOOLBAR, \
                                                          HrnToolbarPrivate))
G_DEFINE_TYPE (HrnToolbar, hrn_toolbar, NBTK_TYPE_TABLE);

static void
hrn_toolbar_finalize (GObject *object)
{
  G_OBJECT_CLASS (hrn_toolbar_parent_class)->finalize (object);
}

static void
hrn_toolbar_dispose (GObject *object)
{
  HrnToolbar        *self = (HrnToolbar *) object;
  HrnToolbarPrivate *priv = self->priv;

  if (priv->search_handler > 0)
    {
      g_source_remove (priv->search_handler);
      priv->search_handler = 0;
    }

  G_OBJECT_CLASS (hrn_toolbar_parent_class)->dispose (object);
}

static void
hrn_toolbar_set_property (GObject *object, guint prop_id, const GValue *value,
                          GParamSpec   *pspec)
{
  switch (prop_id)
    {
      default:
        break;
    }
}

static void
hrn_toolbar_get_property (GObject *object, guint prop_id, GValue     *value,
                          GParamSpec *pspec)
{
  switch (prop_id)
    {
      default:
        break;
    }
}

static void
hrn_toolbar_allocate (ClutterActor *self, const ClutterActorBox *box,
                      ClutterAllocationFlags flags)
{
  HrnToolbar        *toolbar = (HrnToolbar *) self;
  HrnToolbarPrivate *priv    = toolbar->priv;
  float              x, y;

  CLUTTER_ACTOR_CLASS (hrn_toolbar_parent_class)->allocate (self, box, flags);

  clutter_actor_get_transformed_position ((ClutterActor *) priv->filter_bar,
                                          &x, &y);
}

static void
hrn_toolbar_map (ClutterActor *self)
{
  HrnToolbar        *toolbar = (HrnToolbar *) self;
  HrnToolbarPrivate *priv    = toolbar->priv;

  CLUTTER_ACTOR_CLASS (hrn_toolbar_parent_class)->map (self);

  clutter_actor_map ((ClutterActor *) priv->zoom_bar_label);
}

static void
hrn_toolbar_unmap (ClutterActor *self)
{
  HrnToolbar        *toolbar = (HrnToolbar *) self;
  HrnToolbarPrivate *priv    = toolbar->priv;

  CLUTTER_ACTOR_CLASS (hrn_toolbar_parent_class)->unmap (self);

  clutter_actor_unmap ((ClutterActor *) priv->zoom_bar_label);
}

static void
hrn_toolbar_paint (ClutterActor *actor)
{
  CLUTTER_ACTOR_CLASS (hrn_toolbar_parent_class)->paint (actor);
}

static void
hrn_toolbar_pick (ClutterActor *actor, const ClutterColor *color)
{
  HrnToolbar        *toolbar = (HrnToolbar *) actor;
  HrnToolbarPrivate *priv    = toolbar->priv;

  clutter_actor_paint ((ClutterActor *) priv->zoom_bar);
  if (CLUTTER_ACTOR_IS_VISIBLE (priv->view_table))
    {
      clutter_actor_paint ((ClutterActor *) priv->view_table);
    }

  clutter_actor_paint ((ClutterActor *) priv->close_button);
}

static void
hrn_toolbar_class_init (HrnToolbarClass *klass)
{
  GObjectClass      *o_class = (GObjectClass *) klass;
  ClutterActorClass *a_class = (ClutterActorClass *) klass;

  o_class->dispose      = hrn_toolbar_dispose;
  o_class->finalize     = hrn_toolbar_finalize;
  o_class->set_property = hrn_toolbar_set_property;
  o_class->get_property = hrn_toolbar_get_property;

  a_class->map      = hrn_toolbar_map;
  a_class->unmap    = hrn_toolbar_unmap;
  a_class->allocate = hrn_toolbar_allocate;
  a_class->paint    = hrn_toolbar_paint;
  a_class->pick     = hrn_toolbar_pick;

  g_type_class_add_private (klass, sizeof (HrnToolbarPrivate));
}

static gboolean
deferred_search (gpointer data)
{
  HrnToolbar        *toolbar = (HrnToolbar *) data;
  HrnToolbarPrivate *priv    = toolbar->priv;
  ClutterActor      *text;
  const char        *str;

  priv->search_handler = 0;

  text = nbtk_entry_get_clutter_text (NBTK_ENTRY (priv->filter_bar));
  str  = clutter_text_get_text ((ClutterText *) text);
  if (str && *str != '\0')
    {
      hrn_view_set_search (HRN_VIEW (hrn_view), str);
    }
  else
    {
      hrn_view_set_search (HRN_VIEW (hrn_view), NULL);
    }

  return FALSE;
}

gboolean hrn_search_direct = FALSE;
static gboolean
search_updated (ClutterActor *entry, HrnToolbar   *toolbar)
{
  HrnToolbarPrivate *priv = toolbar->priv;

  if (priv->search_handler)
    {
      g_source_remove (priv->search_handler);
    }
  priv->search_handler = 0;
  if (hrn_search_direct)
    {
      deferred_search (toolbar);
      return TRUE;
    }

  priv->search_handler = g_timeout_add (250, deferred_search, toolbar);
  return TRUE;
}

static void
hrn_toolbar_quit_clicked_cb (ClutterActor *actor,
                             HrnToolbar   *toolbar)
{
  /* FIXME: Really this should be a signal, and then the program
     would handle it as appropriate */
  hrn_quit ();
}

static void
set_filter (HrnToolbar *toolbar, int mode)
{
  HrnToolbarPrivate *priv = toolbar->priv;

  hrn_toolbar_set_filter_mode (toolbar, mode);
  hrn_view_set_filter (HRN_VIEW (hrn_view), priv->filter_mode);

#if HRN_CLEAR_QUERY_ON_FILTER_CHANGE
  hrn_toolbar_set_query (toolbar, "");
#endif
  hrn_view_center (hrn_view, 0);

  hrn_popup_close ();
}

static void
all_cb (HrnToolbar *toolbar)
{
  set_filter (toolbar, BKL_ITEM_TYPE_AUDIO |
              BKL_ITEM_TYPE_VIDEO |
              BKL_ITEM_TYPE_IMAGE);
}

static void
audio_cb (HrnToolbar *toolbar)
{
  set_filter (toolbar, BKL_ITEM_TYPE_AUDIO);
}

static void
video_cb (HrnToolbar *toolbar)
{
  set_filter (toolbar, BKL_ITEM_TYPE_VIDEO);
}

static void
image_cb (HrnToolbar *toolbar)
{
  set_filter (toolbar, BKL_ITEM_TYPE_IMAGE);
}

/* FIXME: This should be an array of structs */
static gpointer filter_actions[] = { N_ ("All"), all_cb,
                                     N_ ("Audio"),    audio_cb,
                                     N_ ("Video"),    video_cb,
                                     N_ ("Images"),   image_cb,
                                     NULL };

static gboolean
show_button_release (ClutterActor *actor, ClutterButtonEvent *event,
                     HrnToolbar         *toolbar)
{
  if (event->button == 1)
    {
      float x, y;
      int   selected;

      clutter_actor_get_transformed_position (actor, &x, &y);

      /* FIXME: hardcoded offset to handle padding */
      x -= 14;
      y -= 12;

      switch (hrn_view_get_filter (HRN_VIEW (hrn_view)))
        {
          case BKL_ITEM_TYPE_AUDIO:
            selected = 1;
            break;

          case BKL_ITEM_TYPE_VIDEO:
            selected = 2;
            break;

          case BKL_ITEM_TYPE_IMAGE:
            selected = 3;
            break;

          default:
            selected = 0;
        }

      hrn_popup_actor_fixed (x, y,
                             hrn_popup_actions_bolded ((void *) filter_actions,
                                                       toolbar, selected));
      return TRUE;
    }

  return FALSE;
}

static void
zoom_paint_label (ClutterActor *slider, HrnToolbar   *toolbar)
{
  HrnToolbarPrivate *priv  = toolbar->priv;
  ClutterActor      *label = (ClutterActor *) priv->zoom_bar_label;

  cogl_push_matrix ();
  cogl_translate
                             ((clutter_actor_get_width (slider) -
                               clutter_actor_get_width (label)) / 2,
                             28, 0);
  clutter_actor_paint (label);
  cogl_pop_matrix ();
}

static void
zoom_change_label (NbtkAdjustment *adjustment, GParamSpec     *pspec,
                   HrnToolbar     *toolbar)
{
  HrnToolbarPrivate *priv            = toolbar->priv;
  double             current         = nbtk_adjustment_get_value (adjustment);
  static char       *descriptions[6] = { N_ ("Overview"),
                                         N_ ("Collections"),
                                         N_ ("Small Items"),
                                         N_ ("Large Items"),
                                         N_ ("Theatre"),     };

  nbtk_label_set_text ((NbtkLabel *) priv->zoom_bar_label,
                       _ (descriptions[(int)((current * 4) + 0.5)]));
}

static void
hrn_toolbar_init (HrnToolbar *self)
{
  HrnToolbarPrivate *priv;
  ClutterActor      *stage;
  ClutterActor      *separator;
  ClutterActor      *text;
  float              width;

  self->priv = GET_PRIVATE (self);
  priv       = self->priv;

  priv->mode = HRN_TOOLBAR_MODE_VIEW;

  clutter_actor_set_name ((ClutterActor *) self, "toolbar");

  stage = clutter_stage_get_default ();
  width = clutter_actor_get_width (stage);
  clutter_actor_set_size ((ClutterActor *) self, width, 62);

  /* FIXME: Move zoom adjustment private to HrnToolbar */
  priv->zoom_bar = nbtk_scroll_bar_new (zoom_adjustment);
  clutter_actor_set_name ((ClutterActor *) priv->zoom_bar, "nbtk-zoom-bar");
  clutter_actor_set_size ((ClutterActor *) priv->zoom_bar,
                          clutter_actor_get_width (stage) / 3.0, 40.0);
  nbtk_table_add_actor_with_properties (NBTK_TABLE (self),
                                        (ClutterActor *) priv->zoom_bar, 0, 0,
                                        "row-span", 1,
                                        "col-span", 1,
                                        "x-fill", FALSE,
                                        "y-fill", FALSE,
                                        "x-expand", FALSE,
                                        "y-expand", FALSE,
                                        "x-align", 0.5,
                                        "y-align", 0.5,
                                        NULL);
  priv->zoom_bar_label = nbtk_label_new (_ ("Overview"));
  clutter_actor_set_name ((ClutterActor *) priv->zoom_bar_label,
                          "hrn-zoom-bar-label");
  clutter_actor_set_parent ((ClutterActor *) priv->zoom_bar_label,
                            (ClutterActor *) self);
  g_signal_connect (zoom_adjustment, "notify::value",
                    G_CALLBACK (zoom_change_label), self);
  g_signal_connect (priv->zoom_bar, "paint",
                    G_CALLBACK (zoom_paint_label), self);

  priv->view_table = nbtk_table_new ();
  nbtk_table_add_actor_with_properties (NBTK_TABLE (self),
                                        (ClutterActor *) priv->view_table,
                                        0, 1,
                                        "x-align", 0.0,
                                        "y-align", 0.0,
                                        NULL);

  priv->filter_bar = g_object_new (HRN_TYPE_SEARCH_BAR, NULL);
  clutter_actor_set_name ((ClutterActor *) priv->filter_bar, "filter-bar");

  text = nbtk_entry_get_clutter_text (NBTK_ENTRY (priv->filter_bar));
  g_signal_connect (text, "text-changed", G_CALLBACK (search_updated), self);
  clutter_stage_set_key_focus (CLUTTER_STAGE (clutter_stage_get_default ()),
                               CLUTTER_ACTOR (text));
  nbtk_table_add_actor_with_properties (NBTK_TABLE (priv->view_table),
                                        (ClutterActor *) priv->filter_bar,
                                        0, 0,
                                        "y-fill", FALSE,
                                        NULL);

  separator = clutter_texture_new_from_file (
    PKGDATADIR "nbtk-toolbar-separator.png", NULL);
  nbtk_table_add_actor_with_properties (NBTK_TABLE (priv->view_table),
                                        (ClutterActor *) separator, 0, 1,
                                        "x-expand", FALSE,
                                        "y-expand", FALSE,
                                        "x-fill", FALSE,
                                        "y-fill", FALSE,
                                        NULL);

  priv->show_button = nbtk_button_new_with_label (_ ("Show"));
  clutter_actor_set_name ((ClutterActor *) priv->show_button, "show-button");
  clutter_actor_set_size ((ClutterActor *) priv->show_button, 70, 40);
  g_signal_connect (priv->show_button, "button-release-event",
                    G_CALLBACK (show_button_release), self);
  nbtk_table_add_actor_with_properties (NBTK_TABLE (priv->view_table),
                                        (ClutterActor *) priv->show_button,
                                        0, 2,
                                        "x-expand", FALSE,
                                        "y-expand", FALSE,
                                        "x-fill", FALSE,
                                        "y-fill", FALSE,
                                        NULL);

  separator = clutter_clone_new (separator);
  nbtk_table_add_actor_with_properties (NBTK_TABLE (self),
                                        (ClutterActor *) separator, 0, 2,
                                        "x-expand", FALSE,
                                        "y-expand", FALSE,
                                        "x-fill", FALSE,
                                        "y-fill", FALSE,
                                        NULL);

  priv->close_button = nbtk_fade_button_new ();
  clutter_actor_set_name ((ClutterActor *) priv->close_button,
                          "close-button");
  clutter_actor_set_size ((ClutterActor *) priv->close_button, 57, 51);
  g_object_set (G_OBJECT(priv->close_button), "transition-duration", 100, NULL);

  g_signal_connect (priv->close_button, "clicked",
                    G_CALLBACK (hrn_toolbar_quit_clicked_cb), self);
  nbtk_table_add_actor_with_properties (NBTK_TABLE (self),
                                        (ClutterActor *) priv->close_button,
                                        0, 3,
                                        "x-expand", FALSE,
                                        "y-expand", FALSE,
                                        "x-fill", FALSE,
                                        "y-fill", FALSE,
                                        NULL);
}

NbtkWidget *
hrn_toolbar_new (void)
{
  return g_object_new (HRN_TYPE_TOOLBAR, NULL);
}

void
hrn_toolbar_set_pinned (HrnToolbar *toolbar, gboolean pinned)
{
  HrnToolbarPrivate *priv = toolbar->priv;

  /* FIXME: API urgh */
  if (pinned)
    {
      hrn_search_bar_set_pinned (HRN_SEARCH_BAR (priv->filter_bar));
    }
  else
    {
      hrn_search_bar_unpin (HRN_SEARCH_BAR (priv->filter_bar));
    }
}

void
hrn_toolbar_focused (HrnToolbar *toolbar)
{
  HrnToolbarPrivate *priv = toolbar->priv;
  ClutterActor      *stage, *text;

  stage = clutter_stage_get_default ();

  text = nbtk_entry_get_clutter_text ((NbtkEntry *) priv->filter_bar);
  clutter_stage_set_key_focus ((ClutterStage *) stage, text);
}

void
hrn_toolbar_set_filter_mode (HrnToolbar *toolbar, int mode)
{
  HrnToolbarPrivate *priv   = toolbar->priv;
  NbtkButton        *button = (NbtkButton *) priv->show_button;

  if (priv->filter_mode == mode)
    {
      return;
    }

  hrn_toolbar_set_pinned (toolbar, FALSE);
  priv->filter_mode = mode;

  switch (mode)
    {
      case BKL_ITEM_TYPE_IMAGE:
        nbtk_button_set_label (button, _ ("Images"));
        break;

      case BKL_ITEM_TYPE_AUDIO:
        nbtk_button_set_label (button, _ ("Audio"));
        break;

      case BKL_ITEM_TYPE_VIDEO:
        nbtk_button_set_label (button, _ ("Video"));
        break;

      default:
        nbtk_button_set_label (button, _ ("Show"));
        break;
    }
}

guint
hrn_toolbar_get_mode (HrnToolbar *toolbar)
{
  HrnToolbarPrivate *priv = toolbar->priv;

  return priv->filter_mode;
}

void
hrn_toolbar_set_query (HrnToolbar *toolbar, const char *query)
{
  HrnToolbarPrivate *priv = toolbar->priv;

  nbtk_entry_set_text (NBTK_ENTRY (priv->filter_bar), query);
}

void
hrn_toolbar_set_mode (HrnToolbar *toolbar, HrnToolbarMode mode)
{
  HrnToolbarPrivate *priv = toolbar->priv;

  if (mode == HRN_TOOLBAR_MODE_VIEW)
    {
      clutter_actor_show ((ClutterActor *) priv->view_table);
    }
  else
    {
      clutter_actor_hide ((ClutterActor *) priv->view_table);
    }
  priv->mode = mode;
}
