#include "config.h"
#include <bognor/br-iface-player.h>
#ifdef USE_HELIX
#include <clutter-helix/clutter-helix.h>
#else
#include <clutter-gst/clutter-gst.h>
#endif
#include "hrn.h"
#include "hrn-state-manager.h"
#include "hrn-theatre.h"
#include "hrn-iface-player.h"
#include "hrn-image-player.h"
#include "hrn-video-player.h"

enum {
  PROP_0,
};

enum {
    THEATRE_FINISHED,
    POSITION_CHANGED,
    DURATION,
    LAST_SIGNAL
};

struct _HrnTheatrePrivate
{
  HrnStateManager *state_manager;
  ClutterActor *backdrop;

  ClutterActor *player;
  BklItemType   player_type;

  guint32 progress_id;

  gboolean active;   /* Is the theatre being shown? */
  gboolean playing;

  gboolean slideshow; /* We can override playing for images by setting
                         slideshow */

  guint32 no;
};

#define NUMBER_OF_BUTTONS    9 /* Keep in sync with hrn-thumb-bar.c */

#define GET_PRIVATE(obj)    (G_TYPE_INSTANCE_GET_PRIVATE ((obj), \
                                                          HRN_TYPE_THEATRE, \
                                                          HrnTheatrePrivate))
G_DEFINE_TYPE (HrnTheatre, hrn_theatre, CLUTTER_TYPE_GROUP);
static guint32 signals[LAST_SIGNAL] = {0, };

static void
hrn_theatre_finalize (GObject *object)
{
  G_OBJECT_CLASS (hrn_theatre_parent_class)->finalize (object);
}

static void
hrn_theatre_dispose (GObject *object)
{
  G_OBJECT_CLASS (hrn_theatre_parent_class)->dispose (object);
}

static void
hrn_theatre_set_property (GObject *object, guint prop_id, const GValue *value,
                          GParamSpec   *pspec)
{
  switch (prop_id)
    {
      default:
        break;
    }
}

static void
hrn_theatre_get_property (GObject *object, guint prop_id, GValue     *value,
                          GParamSpec *pspec)
{
  switch (prop_id)
    {
      default:
        break;
    }
}

static void
hrn_theatre_allocate (ClutterActor          *actor,
                      const ClutterActorBox *box,
                      ClutterAllocationFlags flags)
{
    HrnTheatre        *theatre = (HrnTheatre *) actor;
    HrnTheatrePrivate *priv    = theatre->priv;

    CLUTTER_ACTOR_CLASS (hrn_theatre_parent_class)->allocate (actor, box, flags);

    if (priv->player) {
        clutter_actor_allocate_preferred_size (priv->player, flags);
    }

    clutter_actor_set_size (priv->backdrop, box->x2 - box->x1,
                            box->y2 - box->y1);
}

static gboolean
hrn_theatre_key_press (ClutterActor *actor, ClutterKeyEvent *event)
{
#if 0
  HrnTheatre        *theatre = (HrnTheatre *) actor;
  HrnTheatrePrivate *priv    = theatre->priv;
  BklItem           *item;

  switch (clutter_event_get_key_symbol ((ClutterEvent *) event))
    {
      case CLUTTER_Left:
        if (priv->no > 0)
          {
            item = hrn_view_item_to_bkl_item (priv->no - 1);
            if (item)
              {
                hrn_play_now (item);
              }
          }
        break;

      case CLUTTER_Right:
        item = hrn_view_item_to_bkl_item (priv->no + 1);
        if (item)
          {
            hrn_play_now (item);
          }
        break;

      default:
        break;
    }
#endif
  return FALSE;
}

static void
hrn_theatre_class_init (HrnTheatreClass *klass)
{
  GObjectClass      *o_class = (GObjectClass *) klass;
  ClutterActorClass *a_class = (ClutterActorClass *) klass;

  o_class->dispose      = hrn_theatre_dispose;
  o_class->finalize     = hrn_theatre_finalize;
  o_class->set_property = hrn_theatre_set_property;
  o_class->get_property = hrn_theatre_get_property;

  a_class->allocate        = hrn_theatre_allocate;
  a_class->key_press_event = hrn_theatre_key_press;

  g_type_class_add_private (klass, sizeof (HrnTheatrePrivate));

  signals[THEATRE_FINISHED] = g_signal_new ("finished",
                                            G_TYPE_FROM_CLASS (o_class),
                                            G_SIGNAL_RUN_FIRST |
                                            G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                            g_cclosure_marshal_VOID__VOID,
                                            G_TYPE_NONE, 0);
  signals[POSITION_CHANGED] = g_signal_new ("position-changed",
                                            G_TYPE_FROM_CLASS (o_class),
                                            G_SIGNAL_RUN_FIRST |
                                            G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                            g_cclosure_marshal_VOID__DOUBLE,
                                            G_TYPE_NONE, 1, G_TYPE_DOUBLE);
  signals[DURATION] = g_signal_new ("duration-changed",
                                    G_TYPE_FROM_CLASS (o_class),
                                    G_SIGNAL_RUN_FIRST |
                                    G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                    g_cclosure_marshal_VOID__DOUBLE,
                                    G_TYPE_NONE, 1, G_TYPE_DOUBLE);
}

static void
hrn_theatre_init (HrnTheatre *self)
{
  HrnTheatrePrivate *priv  = GET_PRIVATE (self);
  ClutterColor       black = { 0, 0, 0, 0xff };

  self->priv = priv;

  priv->active  = FALSE;
  priv->playing = FALSE;

  clutter_actor_set_reactive ((ClutterActor *) self, TRUE);
  clutter_actor_set_opacity ((ClutterActor *) self, 0xff);

  priv->backdrop = clutter_rectangle_new_with_color (&black);
  clutter_group_add (CLUTTER_GROUP (self), priv->backdrop);
  clutter_actor_set_reactive (priv->backdrop, TRUE);
  clutter_actor_set_position (priv->backdrop, 0, 0);
}

static void
player_position_changed (HrnIfacePlayer *player,
                         double          position,
                         HrnTheatre     *theatre)
{
    g_signal_emit (theatre, signals[POSITION_CHANGED], 0, position);
}

static void
player_eos (HrnIfacePlayer *player,
            HrnTheatre     *theatre)
{
    g_signal_emit (theatre, signals[THEATRE_FINISHED], 0);
}

static void
player_duration (HrnIfacePlayer *player,
                 double          duration,
                 HrnTheatre     *theatre)
{
    g_signal_emit (theatre, signals[DURATION], 0, duration);
}

static void
make_video_player (HrnTheatre *theatre)
{
    HrnTheatrePrivate *priv = theatre->priv;

    priv->player      = g_object_new (HRN_TYPE_VIDEO_PLAYER, NULL);
    priv->player_type = BKL_ITEM_TYPE_VIDEO;
    priv->progress_id = g_signal_connect (priv->player, "position-changed",
                                          G_CALLBACK (player_position_changed),
                                          theatre);
    g_signal_connect (priv->player, "eos",
                      G_CALLBACK (player_eos), theatre);
    g_signal_connect (priv->player, "duration-changed",
                      G_CALLBACK (player_duration), theatre);

    clutter_group_add (CLUTTER_ACTOR (theatre), priv->player);
    clutter_actor_show (priv->player);
}

static void
make_image_player (HrnTheatre *theatre)
{
    HrnTheatrePrivate *priv = theatre->priv;

    priv->player = g_object_new (HRN_TYPE_IMAGE_PLAYER, NULL);
    priv->player_type = BKL_ITEM_TYPE_IMAGE;
    priv->progress_id = 0;
    g_signal_connect (priv->player, "eos",
                      G_CALLBACK (player_eos), theatre);

    clutter_group_add (CLUTTER_GROUP (theatre), priv->player);
    clutter_actor_show (priv->player);
}

HrnTheatre *
hrn_theatre_new (HrnStateManager *state)
{
  HrnTheatre *theatre = g_object_new (HRN_TYPE_THEATRE, NULL);

  return theatre;
}

static void
theatre_show (HrnTheatre *theatre,
              const char *uri,
              BklItemType type)
{
    HrnTheatrePrivate *priv = theatre->priv;

    switch (type) {
    case BKL_ITEM_TYPE_AUDIO:
        g_warning ("HrnTheatre does not play audio file %s", uri);
        return;

    case BKL_ITEM_TYPE_IMAGE:
        if (priv->player == NULL || priv->player_type != BKL_ITEM_TYPE_IMAGE) {
            if (priv->player) {
                clutter_actor_destroy (priv->player);
            }
            make_image_player (theatre);
        }

        break;

    case BKL_ITEM_TYPE_VIDEO:
        if (priv->player == NULL || priv->player_type != BKL_ITEM_TYPE_VIDEO) {
            if (priv->player) {
                clutter_actor_destroy (priv->player);
            }
            make_video_player (theatre);
        }

        break;

    default:
        break;
    }

    hrn_iface_player_set_uri ((HrnIfacePlayer *) priv->player, uri);
}

void
hrn_theatre_show_item (HrnTheatre *theatre,
                       BklItem    *item)
{
    theatre_show (theatre, bkl_item_get_uri (item),
                  bkl_item_get_item_type (item));

    hrn_add_recent_uri (bkl_item_get_uri (item), bkl_item_get_mimetype (item));
}

void
hrn_theatre_show_uri (HrnTheatre *theatre,
                      const char *uri,
                      const char *mimetype)
{
    HrnTheatrePrivate *priv = theatre->priv;
    BklItemType type;

    if (uri == NULL) {
        if (priv->player) {
            clutter_container_remove_actor ((ClutterContainer *) theatre,
                                            (ClutterActor *) priv->player);
            priv->player = NULL;
        }
        return;
    }

    if (g_str_has_prefix (mimetype, "audio/")) {
        type = BKL_ITEM_TYPE_AUDIO;
    } else if (g_str_has_prefix (mimetype, "image/")) {
        type = BKL_ITEM_TYPE_IMAGE;
    } else if (g_str_has_prefix (mimetype, "video/")) {
        type = BKL_ITEM_TYPE_VIDEO;
    } else {
        return;
    }

    theatre_show (theatre, uri, type);

    hrn_add_recent_uri (uri, mimetype);
}

void
hrn_theatre_set_playing (HrnTheatre *theatre,
                         gboolean    playing)
{
    HrnTheatrePrivate *priv = theatre->priv;

    if (priv->player == NULL) {
        return;
    }

    hrn_iface_player_set_playing ((HrnIfacePlayer *) priv->player, playing);
}

void
hrn_theatre_set_position (HrnTheatre *theatre,
                          double      position)
{
    HrnTheatrePrivate *priv = theatre->priv;

    if (priv->player == NULL) {
        return;
    }

    if (priv->progress_id > 0) {
        g_signal_handler_block (priv->player, priv->progress_id);
    }

    hrn_iface_player_set_position ((HrnIfacePlayer *) priv->player, position);

    if (priv->progress_id > 0) {
        g_signal_handler_unblock (priv->player, priv->progress_id);
    }
}

guint64
hrn_theatre_get_duration (HrnTheatre *theatre)
{
    HrnTheatrePrivate *priv = theatre->priv;

    if (priv->player == NULL) {
        return 0;
    }

    return hrn_iface_player_get_duration (HRN_IFACE_PLAYER (priv->player));
}
