import logging
import tempfile
import commands
import os

from imagestore.lib import bpickle
from imagestore.lib.service import (
    ThreadedService, ServiceError, ServiceTask, taskHandlerInThread)


class SignatureServiceError(ServiceError):
    pass

class SignatureError(SignatureServiceError):
    pass


class CheckImageSignatureTask(ServiceTask):

    def __init__(self, image):
        self.image = image


class SignatureService(ThreadedService):

    def __init__(self, reactor, keyringPath):
        self._keyringPath = keyringPath
        ThreadedService.__init__(self, reactor)

    @taskHandlerInThread(CheckImageSignatureTask)
    def _checkImageSignature(self, task):
        image = task.image
        imageDict = image.toDict()

        for key in ("signature-fields", "files"):
            if key not in image["signature-fields"]:
                raise SignatureError("%s must be part of the signature" % key)

        signatureDict = {}
        for key in image["signature-fields"]:
            signatureDict[key] = imageDict[key]

        signature = image["signature"]
        signatureFD, signaturePath = tempfile.mkstemp()
        try:
            os.write(signatureFD, signature)
            os.fsync(signatureFD)

            signaturePayload = bpickle.dumps(signatureDict)
            signaturePayloadFD, signaturePayloadPath = tempfile.mkstemp()
            try:
                os.write(signaturePayloadFD, signaturePayload)
                os.fsync(signaturePayloadFD)

                homeDir = os.path.dirname(self._keyringPath)

                command = ("gpg --batch --no-secmem-warning --status-fd 1 "
                           "--no-options --no-default-keyring "
                           "--keyring %s --homedir %s --verify %s %s" %
                           (self._keyringPath, homeDir,
                            signaturePath, signaturePayloadPath))

                logging.debug("Checking image signature: %s" % (command,))

                status, output = commands.getstatusoutput(command)

                if status != 0:
                    logging.error("Image signature is invalid. GPG output:\n%s"
                                  % (output,))
                    raise SignatureError("Image signature is invalid")
                else:
                    logging.debug("GPG output:\n%s" % (output,))
                    logging.info("Image has valid signature.")
            finally:
                os.close(signaturePayloadFD)
                os.unlink(signaturePayloadPath)

        finally:
            os.close(signatureFD)
            os.unlink(signaturePath)

        return image
