/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "nl-gadget-icon.h"

#include <glib.h>
#include <glib/gi18n-lib.h>
#include <clutk/clutk.h>

#include <launcher/launcher.h>

#include "nl-pixbuf-loader.h"

G_DEFINE_TYPE (NlGadgetIcon, nl_gadget_icon, CTK_TYPE_BUTTON);

#define NL_GADGETS_ICON_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE \
        ((obj), NL_TYPE_GADGETS_ICON, NlGadgetIconPrivate))

#define DEFAULT_ICON "/usr/share/icons/gnome/scalable/status/weather-few-clouds.svg"

struct _NlGadgetIconPrivate
{
  NlPixbufLoader *loader;
  NlShell        *shell;
  GadgetInfo     *info;

  gchar          *icon_name;
  gchar          *path;
  ClutterActor   *add;
  CtkImage       *image;
};

enum
{
  PROP_0,

  PROP_SHELL,
  PROP_INFO
};

enum
{
  BEGIN_DRAG_MOVE,

  LAST_SIGNAL
};
static guint32 _actor_signals[LAST_SIGNAL] = { 0, };

/* Globals */
static ClutterActor *bg_texture = NULL;

/* Forwards */
static void show_menu       (CtkButton *self, guint32 event_time);
static void on_icon_clicked (CtkButton *self);
static gboolean on_motion   (ClutterActor *actor, ClutterMotionEvent *event);
static gboolean on_enter    (ClutterActor *actor, ClutterCrossingEvent *event);
static gboolean on_leave    (ClutterActor *actor, ClutterCrossingEvent *event);

/* GObject stuff */
static void
set_property (GObject      *object,
              guint         prop_id,
              const GValue *value,
              GParamSpec   *pspec)
{
  NlGadgetIconPrivate *priv;

  g_return_if_fail (NL_IS_GADGETS_ICON (object));
  priv = NL_GADGETS_ICON_GET_PRIVATE (object);

  switch (prop_id)
    {
    case PROP_SHELL:
      priv->shell = g_value_get_pointer (value);
      break;

    case PROP_INFO:
      priv->info = g_value_get_pointer (value);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
get_property (GObject      *object,
              guint         prop_id,
              GValue       *value,
              GParamSpec   *pspec)
{
  NlGadgetIconPrivate *priv;

  g_return_if_fail (NL_IS_GADGETS_ICON (object));
  priv = NL_GADGETS_ICON_GET_PRIVATE (object);

  switch (prop_id)
    {
    case PROP_SHELL:
      g_value_set_pointer (value, priv->shell);
      break;

    case PROP_INFO:
      g_value_set_pointer (value, NULL);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
nl_gadget_icon_finalize (GObject *object)
{
  NlGadgetIconPrivate *priv;

  priv = NL_GADGETS_ICON_GET_PRIVATE (object);

  if (priv->path)
    {
      g_free (priv->path);
      priv->path = NULL;
    }

  if (priv->add)
    {
      clutter_actor_unparent (priv->add);
      priv->add = NULL;
    }

  if (priv->loader)
    {
      g_object_unref (priv->loader);
      priv->loader = NULL;
    }

  if (priv->icon_name)
    {
      g_free (priv->icon_name);
      priv->icon_name = NULL;
    }

  G_OBJECT_CLASS (nl_gadget_icon_parent_class)->finalize (object);
}

static gchar *
build_icon_filename (const gchar *icon_name, gboolean *needs_caching)
{
  gchar *escaped;
  gchar *filename;

  if (needs_caching != NULL)
    *needs_caching = FALSE;

  escaped = g_uri_escape_string (icon_name, NULL, TRUE);

  /* Test first in /usr/share/netbook-launcher/icons.. */
  filename = g_build_filename (DATADIR,
                               "netbook-launcher",
                               "icons",
                               escaped,
                               NULL);

  if (!g_file_test (filename, G_FILE_TEST_EXISTS))
    {
      g_free (filename);

      filename = g_build_filename (g_get_user_config_dir (),
                                   "netbook-launcher",
                                   "icons",
                                   escaped,
                                   NULL);

      if (needs_caching != NULL)
        *needs_caching = TRUE;
    }

  g_free (escaped);

  return filename;
}

static void
on_pixbuf_load_completed (NlPixbufLoader *loader,
                          GdkPixbuf      *pixbuf,
                          NlGadgetIcon   *self)
{
  NlGadgetIconPrivate *priv;
  gboolean needs_caching = TRUE;

  if (!NL_IS_GADGETS_ICON (self)) return;
  priv = self->priv;

  /* FIXME: Do cache and fallback here */
  if (GDK_IS_PIXBUF (pixbuf))
    {
      gchar *filename;

      ctk_image_set_from_pixbuf (CTK_IMAGE (priv->image), pixbuf);

      filename = build_icon_filename (priv->icon_name, &needs_caching);

      /* Cache the icon for next time */
      if (needs_caching) {
        gdk_pixbuf_save (pixbuf, filename, "png", NULL, NULL);
      }
      g_free (filename);
    }
  else
    {
      GdkPixbuf *temp = gdk_pixbuf_new_from_file (DEFAULT_ICON, NULL);
      ctk_image_set_from_pixbuf (CTK_IMAGE (priv->image), temp);
      g_object_unref (temp);
    }
}

static void
nl_gadget_icon_constructed (GObject *object)
{
  NlGadgetIconPrivate *priv;
  GadgetInfo   *info;
  CtkImage     *image;
  CtkText      *text;
  gchar        *str;
  CtkPadding    padding = { 30, 20, 30, 20 };
  ClutterActor *bg;

  priv = NL_GADGETS_ICON (object)->priv;
  info = priv->info;

  priv->path = g_strdup (info->path);

  ctk_actor_set_padding (CTK_ACTOR (object), &padding);
  bg = nl_texture_frame_new (CLUTTER_TEXTURE (bg_texture), 25, 25, 25, 25);
  ctk_actor_set_background (CTK_ACTOR (object), bg);

  image = priv->image = ctk_button_get_image (CTK_BUTTON (object));
  ctk_image_set_size (image, 96);
  if (info->icon)
    {
      ctk_image_set_from_pixbuf (image, info->icon);
    }
  else if (info->icon_name)
    {
      GdkPixbuf *pixbuf;
      gchar     *filename;

      filename = build_icon_filename (info->icon_name, NULL);
      pixbuf = gdk_pixbuf_new_from_file (filename, NULL);
      if (pixbuf)
        {
          ctk_image_set_from_pixbuf (image, pixbuf);
          g_object_unref (pixbuf);
        }
      else
        {
          if (g_str_has_prefix (info->icon_name, "http")
              || g_str_has_prefix (info->icon_name, "ftp"))
            {
              priv->icon_name = g_strdup (info->icon_name);
              priv->loader = nl_pixbuf_loader_new (info->icon_name);
              g_signal_connect (priv->loader, "completed",
                                G_CALLBACK (on_pixbuf_load_completed), object);
            }
          else
            {
              NlPixbufCache *cache = nl_pixbuf_cache_get_default ();

              pixbuf = nl_pixbuf_cache_icon_for_name (cache,
                                                      info->icon_name,
                                                      96);

              if (pixbuf)
                {
                  ctk_image_set_from_pixbuf (image, pixbuf);
                  g_object_unref (pixbuf);
                }
            }
        }
      g_free (filename);
    }
  else
    {
      GdkPixbuf *temp = gdk_pixbuf_new_from_file (DEFAULT_ICON, NULL);
      ctk_image_set_from_pixbuf (image, temp);
      g_object_unref (temp);
    }

  str = g_strdup_printf ("<big><b>%s</b></big>\n\n%s",
                         info->name, info->description);
  text = ctk_button_get_text (CTK_BUTTON (object));
  clutter_text_set_markup (CLUTTER_TEXT (text), str);
  clutter_text_set_line_wrap (CLUTTER_TEXT (text), TRUE);
  clutter_text_set_line_wrap_mode (CLUTTER_TEXT (text),
                                   PANGO_WRAP_WORD_CHAR);
  clutter_text_set_line_alignment (CLUTTER_TEXT (text),
                                   PANGO_ALIGN_CENTER);

  priv->add = clutter_texture_new_from_file (DATADIR"/netbook-launcher/add.png",
              NULL);
  clutter_actor_set_parent (priv->add, CLUTTER_ACTOR (object));
  clutter_actor_set_opacity (priv->add, 0);
  clutter_actor_show (priv->add);

  g_free (str);
  priv->info = NULL;
}

static void
nl_gadget_icon_get_preferred_width (ClutterActor *self,
                                    gfloat        for_height,
                                    gfloat       *min_width,
                                    gfloat       *nat_width)
{
  if (min_width)
    *min_width = 170.0;
  if (nat_width)
    *nat_width = 170.0;
}

static void
allocate (ClutterActor          *self,
          const ClutterActorBox *box,
          ClutterAllocationFlags flags)
{
  NlGadgetIconPrivate *priv = NL_GADGETS_ICON (self)->priv;
  ClutterActorBox child_box;

  CLUTTER_ACTOR_CLASS (nl_gadget_icon_parent_class)->allocate (self,box,flags);

  child_box.x1 = box->x2-box->x1-clutter_actor_get_width (priv->add) - 8;
  child_box.x2 = child_box.x1 + clutter_actor_get_width (priv->add);
  child_box.y1 = 10;
  child_box.y2 = child_box.y1 + clutter_actor_get_height (priv->add);

  clutter_actor_allocate (priv->add, &child_box, flags);
}

static void
paint (ClutterActor *self)
{
  CLUTTER_ACTOR_CLASS (nl_gadget_icon_parent_class)->paint (self);

  clutter_actor_paint (NL_GADGETS_ICON (self)->priv->add);
}

static void
map (ClutterActor *self)
{
  CLUTTER_ACTOR_CLASS (nl_gadget_icon_parent_class)->map (self);

  clutter_actor_map (NL_GADGETS_ICON (self)->priv->add);
}

static void
unmap (ClutterActor *self)
{
  CLUTTER_ACTOR_CLASS (nl_gadget_icon_parent_class)->unmap (self);

  clutter_actor_unmap (NL_GADGETS_ICON (self)->priv->add);
}

static void
nl_gadget_icon_class_init (NlGadgetIconClass *klass)
{
  GObjectClass      *obj_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *act_class = CLUTTER_ACTOR_CLASS (klass);
  CtkButtonClass    *but_class = CTK_BUTTON_CLASS (klass);
  GParamSpec        *pspec;

  obj_class->finalize     = nl_gadget_icon_finalize;
  obj_class->constructed  = nl_gadget_icon_constructed;
  obj_class->set_property = set_property;
  obj_class->get_property = get_property;

  act_class->get_preferred_width = nl_gadget_icon_get_preferred_width;
  act_class->motion_event        = on_motion;
  act_class->map                 = map;
  act_class->unmap               = unmap;
  act_class->paint               = paint;
  act_class->allocate            = allocate;

  but_class->show_context_menu = show_menu;
  but_class->clicked           = on_icon_clicked;

  /* Install properties */
  pspec = g_param_spec_pointer ("shell", "shell", "shell",
                                G_PARAM_READWRITE | G_PARAM_CONSTRUCT);
  g_object_class_install_property (obj_class, PROP_SHELL, pspec);

  pspec = g_param_spec_pointer ("info", "info", "info",
                                G_PARAM_READWRITE | G_PARAM_CONSTRUCT);
  g_object_class_install_property (obj_class, PROP_INFO, pspec);


  /* Signal */
  _actor_signals[BEGIN_DRAG_MOVE] =
    g_signal_new ("begin-drag-move",
                  G_OBJECT_CLASS_TYPE (obj_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (NlGadgetIconClass, begin_drag_move),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);

  g_type_class_add_private (obj_class, sizeof (NlGadgetIconPrivate));
}

static void
nl_gadget_icon_init (NlGadgetIcon *self)
{
#define BG_PATH DATADIR"/netbook-launcher/iconview_normal.png"
  NlGadgetIconPrivate *priv;

  priv = self->priv = NL_GADGETS_ICON_GET_PRIVATE (self);

  if (!CLUTTER_IS_ACTOR (bg_texture))
    {
      bg_texture = clutter_texture_new_from_file (BG_PATH, NULL);
    }

  g_signal_connect (self, "enter-event", G_CALLBACK (on_enter), NULL);
  g_signal_connect (self, "leave-event", G_CALLBACK (on_leave), NULL);
}

static gchar *
drag_data_func (GObject *self)
{
  NlGadgetIconPrivate *priv;
  gchar               *uid = NULL;
  GSList              *favs, *f;
  gint                 i;
  gchar               *filename;
  LauncherFavorites   *favorites;

  g_return_val_if_fail (NL_IS_GADGETS_ICON (self), NULL);
  priv = NL_GADGETS_ICON (self)->priv;

  favorites = launcher_favorites_get_default ();

  filename = g_path_get_basename (priv->path);

  /* Get list of favs and try and create a new one with a normal name */
  favs = launcher_favorites_get_favorites (favorites);
  for (i = 0; i < 50; i++)
    {
      gchar *temp_uid = g_strdup_printf ("gadget-%d",
                                         i);
      gboolean existing = FALSE;

      for (f = favs; f; f = f->next)
        {
          if (g_strcmp0 (f->data, temp_uid) == 0)
            {
              existing = TRUE;
              break;
            }
        }
      if (!existing)
        {
          uid = temp_uid;
          break;
        }
      else
        g_free (temp_uid);
    }
  g_slist_foreach (favs, (GFunc)g_free, NULL);
  g_slist_free (favs);
  if (!uid)
    uid = g_strdup_printf ("gadget-%d", (gint)time (NULL));


  /* Set the properties of the favorite */
  launcher_favorites_set_string (favorites, uid, "type", "libgadget-gadget");
  launcher_favorites_set_string (favorites, uid, "path", priv->path);
  launcher_favorites_add_favorite (favorites, uid);

  g_free (filename);
  g_object_unref (favorites);

  return uid;
}

static gboolean
on_motion (ClutterActor *actor, ClutterMotionEvent *event)
{
  g_return_val_if_fail (NL_IS_GADGETS_ICON (actor), FALSE);

  if (event->modifier_state & CLUTTER_BUTTON1_MASK)
    {
      NlGadgetIconPrivate *priv = NL_GADGETS_ICON (actor)->priv;
      ClutterActor        *stage, *clone;
      CtkImage            *image;
      gfloat               x=0, y=0;

      clutter_actor_get_transformed_position (CLUTTER_ACTOR (actor), &x, &y);

      stage = nl_shell_get_stage (priv->shell);
      image = ctk_button_get_image (CTK_BUTTON (actor));

      clone = ctk_image_new_from_pixbuf (64, ctk_image_get_pixbuf (image));
      clutter_container_add_actor (CLUTTER_CONTAINER (stage), clone);
      clutter_actor_set_position (clone, x, y);
      clutter_actor_raise_top (clone);
      clutter_actor_show (clone);

      nl_shell_begin_favorite_drag (priv->shell,
                                    G_OBJECT (actor),
                                    clone,
                                    drag_data_func,
                                    NULL);

      return TRUE;
    }
  return FALSE;
}


/*
 * Public methods
 */
ClutterActor *
nl_gadget_icon_new (NlShell *shell, GadgetInfo *info)

{
  ClutterActor *gadget_icon = NULL;

  gadget_icon = g_object_new (NL_TYPE_GADGETS_ICON,
                              "shell", shell,
                              "info", info,
                              "orientation", CTK_ORIENTATION_VERTICAL,
                              NULL);

  return gadget_icon;
}

static void
fav_app (GtkMenuItem *item, NlGadgetIcon *self)
{
  NlGadgetIconPrivate *priv;
  gchar               *uid = NULL;
  GSList              *favs, *f;
  gint                 i;
  gchar               *filename;
  LauncherFavorites   *favorites;

  g_return_if_fail (NL_IS_GADGETS_ICON (self));
  priv = self->priv;

  favorites = launcher_favorites_get_default ();

  filename = g_path_get_basename (priv->path);

  /* Get list of favs and try and create a new one with a normal name */
  favs = launcher_favorites_get_favorites (favorites);
  for (i = 0; i < 50; i++)
    {
      gchar *temp_uid = g_strdup_printf ("gadget-%d",
                                         i);
      gboolean existing = FALSE;

      for (f = favs; f; f = f->next)
        {
          if (g_strcmp0 (f->data, temp_uid) == 0)
            {
              existing = TRUE;
              break;
            }
        }
      if (!existing)
        {
          uid = temp_uid;
          break;
        }
      else
        g_free (temp_uid);
    }
  g_slist_foreach (favs, (GFunc)g_free, NULL);
  g_slist_free (favs);
  if (!uid)
    uid = g_strdup_printf ("gadget-%d", (gint)time (NULL));


  /* Set the properties of the favorite */
  launcher_favorites_set_string (favorites, uid, "type", "libgadget-gadget");
  launcher_favorites_set_string (favorites, uid, "path", priv->path);
  launcher_favorites_add_favorite (favorites, uid);

  nl_shell_add_favorite (priv->shell, uid);

  g_free (uid);
  g_free (filename);
  g_object_unref (favorites);
}

static void
show_menu (CtkButton *self, guint32 event_time)
{
  GtkWidget *menu, *item;

  g_return_if_fail (NL_IS_GADGETS_ICON (self));

  menu = gtk_menu_new ();

  item = gtk_image_menu_item_new_with_label (_("Add to Favorites"));
  g_object_set (item,
                "image", gtk_image_new_from_stock (GTK_STOCK_ADD,
                                                   GTK_ICON_SIZE_MENU),
                NULL);
  gtk_widget_show (item);
  g_signal_connect (item, "activate", G_CALLBACK (fav_app), self);
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);

  gtk_menu_popup (GTK_MENU (menu),
                  NULL, NULL,
                  NULL, NULL,
                  3, event_time);
}

static void
on_icon_clicked (CtkButton *self)
{
  fav_app (NULL, NL_GADGETS_ICON (self));
}

static gboolean
on_enter  (ClutterActor *actor, ClutterCrossingEvent *event)
{
  g_return_val_if_fail (NL_IS_GADGETS_ICON (actor), FALSE);

  clutter_actor_animate (NL_GADGETS_ICON (actor)->priv->add,
                         CLUTTER_EASE_OUT_SINE, 200, "opacity", 255, NULL);
  return FALSE;
}

static gboolean
on_leave  (ClutterActor *actor, ClutterCrossingEvent *event)
{
  g_return_val_if_fail (NL_IS_GADGETS_ICON (actor), FALSE);

  clutter_actor_animate (NL_GADGETS_ICON (actor)->priv->add,
                         CLUTTER_EASE_OUT_SINE, 200, "opacity", 0, NULL);
  return FALSE;
}
