/* Quanta Composite Keys Driver
 *
 * Copyright (C) 2011 Quanta Computer Inc.
 * Author: Wayne Lin <wayne.lin@quantatw.com>
 *
 * This software is licensed under the terms of the GNU General Public
 * License version 2, as published by the Free Software Foundation, and
 * may be copied, distributed, and modified under those terms.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

/*-----------------------------------------------------------------------------
 * Global Include files
 *---------------------------------------------------------------------------*/
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/mutex.h>
#include <linux/interrupt.h>
#include <linux/input.h>
#include <linux/gpio.h>
#include <linux/delay.h>
#include <linux/poll.h>
#include <linux/platform_device.h>
#include <linux/pmic_external.h>
#include <linux/on9_comp_keys.h>

/*-----------------------------------------------------------------------------
 * Local Include files
 *---------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
 * Constants
 *---------------------------------------------------------------------------*/
#define DEBUG_MSG
#define ON9_COMP_KEYS_DRV_NAME	"on9_comp_keys"
#define ON9_COMP_KEYS_DEV_NAME	"on9_comp_keys"
#define KEYBOARD_NAME		"Quanta ON9 composite keys driver"
#define ON9_COMP_KEYS_DEV_MINOR MISC_DYNAMIC_MINOR
#define ON9_COMP_KEYS_MTP_EN	_IO(0xA1,0)
#define ON9_COMP_KEYS_MTP_DEN	_IO(0xA1,1)
/*-----------------------------------------------------------------------------
 * Marcos
 *---------------------------------------------------------------------------*/
#if defined(DEBUG_MSG)
#define pr_info_comp_key(func, line, type, value)		pr_info("[ON9 COMP. KEYS]%s %d: %s key value = %d \n",\
									(func), (line), (type), (value))
#else
#define pr_info_comp_key(func, line, type, value)
#endif
/*-----------------------------------------------------------------------------
 * Global variables
 *---------------------------------------------------------------------------*/
static int __devinit on9_comp_keys_probe(struct platform_device *pdev);
static int __devexit on9_comp_keys_remove(struct platform_device *pdev);
struct on9_comp_keys_drv_data *on9_comp_keys_drv;

/*-----------------------------------------------------------------------------
 * Local functions
 *---------------------------------------------------------------------------*/
static irqreturn_t on9_comp_keys_interrupt_handler(int irq, void *dev)
{
	struct on9_comp_keys_drv_data *pdata = (struct on9_comp_keys_drv_data *)dev;

	disable_irq_nosync(pdata->irq_hall);
	schedule_delayed_work(&pdata->workqueue, 20);
	return IRQ_HANDLED;
}

static void on9_comp_keys_work_handler(struct work_struct *work)
{
	struct on9_comp_keys_drv_data *pdata =
		container_of(work, struct on9_comp_keys_drv_data, workqueue.work);
	int rc;

	mutex_lock(&pdata->lock);
	rc = gpio_get_value(pdata->gpio_hall);
        if ( rc == 0 )
	{
		input_report_switch(pdata->on9kbd_dev, 
				(pdata->mtp_mode)?SW_TABLET_MODE:SW_LID, 1);
		pr_info_comp_key(__func__, __LINE__,
				(pdata->mtp_mode)?"SW_TABLET_MODE":"SW_LID", rc);
        }else if ( rc == 1 )
	{
		input_report_switch(pdata->on9kbd_dev, 
				(pdata->mtp_mode)?SW_TABLET_MODE:SW_LID, 0);
		pr_info_comp_key(__func__, __LINE__,
				(pdata->mtp_mode)?"SW_TABLET_MODE":"SW_LID", rc);
	}else
	{
		pr_err("[ON9 COMP. KEYS] failed to report hall sensor event\n"); 
	}

	input_sync(pdata->on9kbd_dev);
	mutex_unlock(&pdata->lock);
	enable_irq(pdata->irq_hall);
}

static void power_on_evt_handler(void *data)
{
        unsigned int value;
	struct on9_comp_keys_drv_data *pdata = data;
	/* Read this register to determine which POWERON# event was trigger */
        pmic_read_reg(REG_INT_SENSE1, &value, 0xffffff);

	/* PWRON1 evnet */
        if (value & 0x000008){
		input_report_key(pdata->on9kbd_dev, KEY_POWER, 0);
		pr_info_comp_key(__func__, __LINE__, "KEY_POWER", 0);
	}
        else{
		input_report_key(pdata->on9kbd_dev, KEY_POWER, 1);      // Power key down
		pr_info_comp_key(__func__, __LINE__, "KEY_POWER", 1);
	}
	input_sync(pdata->on9kbd_dev);
}

/*-----------------------------------------------------------------------------
 * Misc device file operations
 *---------------------------------------------------------------------------*/

static int on9_comp_keys_open(struct inode *inode, struct file *filp)
{
	filp->private_data = (void *)on9_comp_keys_drv;
	return 0;
}

static int on9_comp_keys_ioctl(struct inode *inode, struct file *filp, unsigned int cmd, unsigned long arg)
{
	struct on9_comp_keys_drv_data *pdata = (struct on9_comp_keys_drv_data *)filp->private_data;
	switch (cmd)
	{
		case ON9_COMP_KEYS_MTP_EN:
		 //Enable MTP testing mode
		 pdata->mtp_mode = 1;
		pr_info("MTP 1\n");
		break;
		case ON9_COMP_KEYS_MTP_DEN:
		 //Disable MTP testing mode
		 pdata->mtp_mode = 0;
		pr_info("MTP 0\n");
		break;
		default:
			return -EINVAL;
		break;
	}
	return 0;
}

static int on9_comp_keys_close(struct inode *inode, struct file  *filp)
{
	return 0;
}

static const struct file_operations on9_comp_keys_ops = {
        .owner = THIS_MODULE,
	.open = on9_comp_keys_open,
	.ioctl = on9_comp_keys_ioctl,
	.release = on9_comp_keys_close,
};

static int __devinit on9_comp_keys_probe(struct platform_device *pdev)
{
	int ret;
	struct on9_comp_keys_drv_data *pdata = pdev->dev.platform_data;
	
	pdata->pdev = pdev;

	/* Misc device initial */
	pdata->mdev.minor = ON9_COMP_KEYS_DEV_MINOR;
	pdata->mdev.name  = ON9_COMP_KEYS_DEV_NAME;
	pdata->mdev.fops  = &on9_comp_keys_ops;
	on9_comp_keys_drv = pdata;	//Used to store filp->private_data
	pdata->mtp_mode = 0;

	INIT_DELAYED_WORK(&pdata->workqueue, on9_comp_keys_work_handler);
	mutex_init(&pdata->lock);

	ret = request_irq(pdata->irq_hall,
			  on9_comp_keys_interrupt_handler,
			  IRQF_TRIGGER_FALLING | IRQF_TRIGGER_RISING,
			  ON9_COMP_KEYS_DRV_NAME,
			  pdata);
	if (ret) {
		pr_err("[ON9 COMP. KEYS] Get hall sensor irq failed\n");
		goto request_irq_fail;
	}

	pdata->on9kbd_dev = input_allocate_device();
        if (!pdata->on9kbd_dev) {
                pr_err("[ON9 COMP. KEYS] Allocting memory err\n");
                ret = -ENOMEM;
                goto allocate_fail;
        }

	/* Input device initializtion */
	pdata->on9kbd_dev->name       = KEYBOARD_NAME;
	pdata->on9kbd_dev->phys       = ON9_COMP_KEYS_DEV_NAME;
	pdata->on9kbd_dev->id.bustype = BUS_HOST;
	pdata->on9kbd_dev->evbit[0]   = BIT_MASK(EV_KEY) | BIT_MASK(EV_SW);
	pdata->on9kbd_dev->swbit[0]   = BIT_MASK(SW_LID) | BIT_MASK(SW_TABLET_MODE);
        __set_bit(KEY_POWER, pdata->on9kbd_dev->keybit);


	ret = input_register_device(pdata->on9kbd_dev);
	if (ret) {
		pr_err("[ON9 COMP. KEYS] err input register device\n");
		goto register_input_fail;
	}
	
	/* PMIC13892 event handler initialization */
	pdata->pwr_event.event_handler.param = pdata;
        pdata->pwr_event.event_handler.func = (void *)power_on_evt_handler;
        pmic_event_subscribe(pdata->pwr_event.event_type, pdata->pwr_event.event_handler);

	ret = misc_register(&pdata->mdev);
	if (ret) {
		pr_err("[ON9 COMP. KEYS] Misc device register failed\n");
		goto register_misc_fail;
	}

	pr_info("[ON9 COMP. KEYS] Probe successful\n");
	return 0;

request_irq_fail:
	return ret;

allocate_fail:
	free_irq(pdata->irq_hall, pdata);
	return ret;

register_input_fail:
	input_free_device(pdata->on9kbd_dev);
	free_irq(pdata->irq_hall, pdata);
	return ret;

register_misc_fail:
	input_unregister_device(pdata->on9kbd_dev);
	input_free_device(pdata->on9kbd_dev);
	free_irq(pdata->irq_hall, pdata);

	return ret;
}

static int __devexit on9_comp_keys_remove(struct platform_device *pdev)
{
	struct on9_comp_keys_drv_data *pdata = pdev->dev.platform_data;

	input_free_device(pdata->on9kbd_dev);
	misc_deregister(&pdata->mdev);
	free_irq(pdata->irq_hall, pdata);
	gpio_free(pdata->gpio_hall);
	kfree(pdata);
	return 0;
}

struct platform_driver on9_comp_keys_driver = {
        .driver = {
                .owner = THIS_MODULE,
                .name  = ON9_COMP_KEYS_DRV_NAME,
		},
        .probe  = on9_comp_keys_probe,
        .remove = on9_comp_keys_remove,
};

static int __init on9_comp_keys_init(void)
{
	return platform_driver_register(&on9_comp_keys_driver);
}

static void __exit on9_comp_keys_exit(void)
{
	platform_driver_unregister(&on9_comp_keys_driver);
}

module_init(on9_comp_keys_init);
module_exit(on9_comp_keys_exit);

MODULE_AUTHOR("Quanta Computer Inc.");
MODULE_DESCRIPTION("Quanta Comp. keys Driver");
MODULE_VERSION("0.1.2");
MODULE_LICENSE("GPL v2");
