/*
 * Copyright 2006-2008 Freescale Semiconductor, Inc. All Rights Reserved.
 */

/*
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

/*!
 * @file mc521da.c
 *
 * @brief MC521DA camera driver functions
 *
 * @ingroup Camera
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/ctype.h>
#include <linux/types.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/i2c.h>
#include <linux/clk.h>
#include "mxc_v4l2_capture.h"

#define MC521DA_I2C_ADDRESS	0x22
#define MC521DA_TERM		0xFF

typedef struct {
	u16 width;
	u16 height;
} mc521da_image_format;

struct mc521da_reg {
	u8 reg;
	u8 val;
};

static sensor_interface *interface_param = NULL;

static mc521da_image_format format[2] = {
	{
	 .width = 1600,
	 .height = 1200,
	 },
	{
	 .width = 640,
	 .height = 480,
	 },
};

const static struct mc521da_reg mc521da_initial[] = {
	/*----------------------------------------------------------
	 * Sensor Setting Start
	 *----------------------------------------------------------
	 */
	{0xff, 0x01},		/* Sensor setting start */
	{0x01, 0x10},		/* Wavetable script, generated by waveman */
	{0x10, 0x64},
	{0x03, 0x00}, {0x04, 0x06}, {0x05, 0x30}, {0x06, 0x02}, {0x08, 0x00},
	{0x03, 0x01}, {0x04, 0x41}, {0x05, 0x70}, {0x06, 0x03}, {0x08, 0x00},
	{0x03, 0x02}, {0x04, 0x55}, {0x05, 0x30}, {0x06, 0x03}, {0x08, 0x00},
	{0x03, 0x03}, {0x04, 0x5A}, {0x05, 0x30}, {0x06, 0x02}, {0x08, 0x00},
	{0x03, 0x04}, {0x04, 0x7A}, {0x05, 0x30}, {0x06, 0x06}, {0x08, 0x00},
	{0x03, 0x05}, {0x04, 0x9C}, {0x05, 0x30}, {0x06, 0x0F}, {0x08, 0x00},
	{0x03, 0x06}, {0x04, 0x73}, {0x05, 0x31}, {0x06, 0x06}, {0x08, 0x00},
	{0x03, 0x07}, {0x04, 0x2D}, {0x05, 0x3B}, {0x06, 0x06}, {0x08, 0x00},
	{0x03, 0x08}, {0x04, 0x32}, {0x05, 0x33}, {0x06, 0x06}, {0x08, 0x00},
	{0x03, 0x09}, {0x04, 0x67}, {0x05, 0x63}, {0x06, 0x06}, {0x08, 0x00},
	{0x03, 0x0a}, {0x04, 0x6C}, {0x05, 0x23}, {0x06, 0x0E}, {0x08, 0x00},
	{0x03, 0x0b}, {0x04, 0x71}, {0x05, 0x23}, {0x06, 0x06}, {0x08, 0x00},
	{0x03, 0x0c}, {0x04, 0x30}, {0x05, 0x2F}, {0x06, 0x06}, {0x08, 0x00},
	{0x03, 0x0d}, {0x04, 0x00}, {0x05, 0x00}, {0x06, 0x06}, {0x08, 0x00},
	{0x07, 0x0e},

	/* Start Address */
	{0x10, 0x64}, {0x14, 0x10}, {0x15, 0x00},

	/* SYNC */
	{0x18, 0x40}, {0x19, 0x00}, {0x1A, 0x03}, {0x1B, 0x00},

	/* X-Y Mirror */
	{0x11, 0x00}, {0xda, 0x00},	/* X mirror OFF, Y Mirror OFF */

	/* Frame height */
	{0x1c, 0x13}, {0x1d, 0x04}, {0x0e, 0x4b}, {0x0f, 0x05},
	{0x9e, 0x04}, {0x9d, 0xc6}, {0xcc, 0x14}, {0xcd, 0x05},

	/* Frame width */
	{0x0c, 0x35}, {0x0d, 0x07}, {0x9b, 0x10}, {0x9c, 0x07},
	{0x93, 0x21},

	{0x01, 0x01}, {0x40, 0x00}, {0x41, 0x00}, {0x42, 0xf0},
	{0x43, 0x03}, {0x44, 0x0a}, {0x45, 0x00}, {0x3b, 0x40},
	{0x38, 0x18}, {0x3c, 0x00}, {0x20, 0x00}, {0x21, 0x01},
	{0x22, 0x00}, {0x23, 0x01}, {0x24, 0x00}, {0x25, 0x01},
	{0x26, 0x00}, {0x27, 0x01}, {0xb9, 0x04}, {0xb8, 0xc3},
	{0xbb, 0x04}, {0xba, 0xc3}, {0xbf, 0x04}, {0xbe, 0xc3},

	/* Ramp */
	{0x57, 0x07}, {0x56, 0xd6}, {0x55, 0x03}, {0x54, 0x74},
	{0x9f, 0x99}, {0x94, 0x80}, {0x91, 0x78}, {0x92, 0x8b},

	/* Output Mode */
	{0x52, 0x10}, {0x51, 0x00},

	/* Analog Gain and Output driver */
	{0x28, 0x00}, {0xdd, 0x82}, {0xdb, 0x00}, {0xdc, 0x00},

	/* Update */
	{0x00, 0x84},

	/* PLL  ADC clock = 75 MHz */
	{0xb5, 0x60}, {0xb4, 0x02}, {0xb5, 0x20},

	/*----------------------------------------------*/
	/*      ISP Setting Start                       */
	/*----------------------------------------------*/
	{0xff, 0x02},
	{0x01, 0xbd}, {0x02, 0xf8}, {0x03, 0x3a}, {0x04, 0x00}, {0x0e, 0x00},

	/* Output mode */
	{0x88, 0x00}, {0x87, 0x11},

	/* Threshold */
	{0xb6, 0x1b}, {0x0d, 0xc0}, {0x24, 0x00}, {0x25, 0x00}, {0x26, 0x00},

	/* Image Effect */
	{0x3f, 0x80}, {0x40, 0x00}, {0x41, 0x00}, {0x42, 0x80}, {0x43, 0x00},
	{0x44, 0x00}, {0x45, 0x00}, {0x46, 0x00}, {0x56, 0x80}, {0x57, 0x20},
	{0x58, 0x20}, {0x59, 0x02}, {0x5a, 0x00}, {0x5b, 0x78}, {0x5c, 0x7c},
	{0x5d, 0x84}, {0x5e, 0x85}, {0x5f, 0x78}, {0x60, 0x7e}, {0x61, 0x82},
	{0x62, 0x85}, {0x63, 0x00}, {0x64, 0x80}, {0x65, 0x00}, {0x66, 0x80},
	{0x67, 0x80}, {0x68, 0x80},

	/* Auto Focus */
	{0x6e, 0x02}, {0x6f, 0xe5}, {0x70, 0x08}, {0x71, 0x01}, {0x72, 0x00},

	/* Decimator */
	{0x78, 0xff}, {0x79, 0xff}, {0x7a, 0x70}, {0x7b, 0x00}, {0x7c, 0x00},
	{0x7d, 0x00}, {0x7e, 0xc8}, {0x7f, 0xc8}, {0x80, 0x96}, {0x81, 0x96},
	{0x82, 0x00}, {0x83, 0x00}, {0x84, 0x00}, {0x85, 0x00}, {0x86, 0x00},

	/* Luminance Info */
	{0xf9, 0x20}, {0xb7, 0x7f}, {0xb8, 0x28}, {0xb9, 0x08},
	{0xf9, 0xa0}, {0xb7, 0x10}, {0xb9, 0x00},
	{0xf9, 0x40}, {0xb7, 0x7f}, {0xb8, 0x28}, {0xb9, 0x08},
	{0xf9, 0xc0}, {0xb7, 0x08}, {0xb9, 0x00},
	{0xf9, 0x60}, {0xb7, 0x7f}, {0xb8, 0x28}, {0xb9, 0x08},
	{0xf9, 0xe0}, {0xb7, 0x05}, {0xb9, 0x00},
	{0xf9, 0x00}, {0xb7, 0x03}, {0xb8, 0x2d}, {0xb9, 0xcd},
	{0xf9, 0x80}, {0xb7, 0x02}, {0xb9, 0x00},

	/* AE */
	{0x8a, 0x00}, {0x89, 0xc0}, {0x8c, 0x32}, {0x8d, 0x96}, {0x8e, 0x25},
	{0x8f, 0x70}, {0x90, 0x12}, {0x91, 0x41}, {0x9e, 0x2e}, {0x9f, 0x2e},
	{0xa0, 0x0b}, {0xa1, 0x71}, {0xa2, 0xb0}, {0xa3, 0x09}, {0xa4, 0x89},
	{0xa5, 0x68}, {0xa6, 0x1a}, {0xa7, 0xb3}, {0xa8, 0xf0}, {0xa9, 0x19},
	{0xaa, 0x6a}, {0xab, 0x6b}, {0xac, 0x01}, {0xad, 0xe8}, {0xae, 0x48},
	{0xaf, 0x01}, {0xb0, 0x96}, {0xb1, 0xe6}, {0xb2, 0x03}, {0xb3, 0x00},
	{0xb4, 0x10}, {0xb5, 0x00}, {0xb6, 0x04}, {0xba, 0x44}, {0xbb, 0x3a},
	{0xbc, 0x01}, {0xbd, 0x08}, {0xbe, 0xa0}, {0xbf, 0x01}, {0xc0, 0x82},
	{0x8a, 0xe1}, {0x8b, 0x8c},

	/* AWB */
	{0xc8, 0x00}, {0xc9, 0x00}, {0xca, 0x40}, {0xcb, 0xB0}, {0xcc, 0x40},
	{0xcd, 0xff}, {0xce, 0x19}, {0xcf, 0x40}, {0xd0, 0x01}, {0xd1, 0x43},
	{0xd2, 0x80}, {0xd3, 0x80}, {0xd4, 0xf1}, {0xdf, 0x00}, {0xe0, 0x8f},
	{0xe1, 0x8f}, {0xe2, 0x53}, {0xe3, 0x97}, {0xe4, 0x1f}, {0xe5, 0x3b},
	{0xe6, 0x9c}, {0xe7, 0x2e}, {0xe8, 0x03}, {0xe9, 0x02},

	/* Neutral CCM */
	{0xfa, 0x00}, {0xd5, 0x3f}, {0xd6, 0x8c}, {0xd7, 0x43}, {0xd8, 0x08},
	{0xd9, 0x27}, {0xda, 0x7e}, {0xdb, 0x17}, {0xdc, 0x1a}, {0xdd, 0x47},
	{0xde, 0xa1},

	/* Blue CCM */
	{0xfa, 0x01}, {0xd5, 0x3f}, {0xd6, 0x77}, {0xd7, 0x34}, {0xd8, 0x03},
	{0xd9, 0x18}, {0xda, 0x6e}, {0xdb, 0x16}, {0xdc, 0x0f}, {0xdd, 0x29},
	{0xde, 0x77},

	/* Red CCM */
	{0xfa, 0x02}, {0xd5, 0x3f}, {0xd6, 0x7d}, {0xd7, 0x2f}, {0xd8, 0x0e},
	{0xd9, 0x1e}, {0xda, 0x76}, {0xdb, 0x18}, {0xdc, 0x29}, {0xdd, 0x51},
	{0xde, 0xba},

	/* AWB */
	{0xea, 0x00}, {0xeb, 0x1a}, {0xc8, 0x33}, {0xc9, 0xc2},

	{0xed, 0x02}, {0xee, 0x02},

	/* AFD */
	{0xf0, 0x11}, {0xf1, 0x03}, {0xf2, 0x05}, {0xf5, 0x05}, {0xf6, 0x32},
	{0xf7, 0x32},

	/* Lens Shading */
	{0xf9, 0x00}, {0x05, 0x04}, {0x06, 0xff}, {0x07, 0xf2}, {0x08, 0x00},
	{0x09, 0x00}, {0x0a, 0xf2}, {0x0b, 0xff}, {0x0c, 0xff},
	{0xf9, 0x01}, {0x05, 0x04}, {0x06, 0xff}, {0x07, 0x8b}, {0x08, 0x16},
	{0x09, 0x16}, {0x0a, 0x8b}, {0x0b, 0xff}, {0x0c, 0xe0},
	{0xf9, 0x02}, {0x05, 0x04}, {0x06, 0xff}, {0x07, 0x8b}, {0x08, 0x16},
	{0x09, 0x16}, {0x0a, 0x8b}, {0x0b, 0xff}, {0x0c, 0xe0},
	{0xf9, 0x03}, {0x05, 0x04}, {0x06, 0xff}, {0x07, 0x7c}, {0x08, 0x26},
	{0x09, 0x26}, {0x0a, 0x7c}, {0x0b, 0xd0}, {0x0c, 0xe0},
	{0xf9, 0x04}, {0x05, 0x0d}, {0x06, 0x40}, {0x07, 0xa0}, {0x08, 0x00},
	{0x09, 0x00}, {0x0a, 0xa0}, {0x0b, 0x40}, {0x0c, 0xe0},
	{0xf9, 0x05}, {0x05, 0x0d}, {0x06, 0x40}, {0x07, 0xa0}, {0x08, 0x00},
	{0x09, 0x00}, {0x0a, 0xa0}, {0x0b, 0x40}, {0x0c, 0xa0},
	{0xf9, 0x06}, {0x05, 0x0d}, {0x06, 0x40}, {0x07, 0xa0}, {0x08, 0x00},
	{0x09, 0x00}, {0x0a, 0xa0}, {0x0b, 0x40}, {0x0c, 0xa0},
	{0xf9, 0x07}, {0x05, 0x0d}, {0x06, 0x40}, {0x07, 0xa0}, {0x08, 0x00},
	{0x09, 0x00}, {0x0a, 0xa0}, {0x0b, 0x40}, {0x0c, 0xa0},

	/* Edge setting */
	{0x73, 0x68}, {0x74, 0x40}, {0x75, 0x00}, {0x76, 0xff}, {0x77, 0x80},
	{0x4f, 0x80}, {0x50, 0x82}, {0x51, 0x82}, {0x52, 0x08},

	/* Interpolation Setting */
	{0x23, 0x7f}, {0x22, 0x08}, {0x18, 0xff}, {0x19, 0x00},
	{0x40, 0x00}, {0x53, 0xff}, {0x54, 0x0a}, {0x55, 0xc2},
	{0x1b, 0x18},

	{0xfa, 0x00}, {0x15, 0x0c}, {0x22, 0x00}, {0x0e, 0xef}, {0x1f, 0x1d},
	{0x20, 0x2d}, {0x1c, 0x01}, {0x1d, 0x02}, {0x1e, 0x03}, {0x0e, 0xee},
	{0x12, 0x10}, {0x16, 0x10}, {0x17, 0x02}, {0x1a, 0x01},
	{0xfa, 0x04}, {0x0e, 0xef}, {0x1c, 0x01}, {0x1d, 0x02}, {0x1e, 0x03},
	{0x1f, 0x11}, {0x20, 0x11}, {0x0e, 0xee}, {0x12, 0x03}, {0x16, 0x10},
	{0x17, 0x02}, {0x1a, 0xee},
	{0xfa, 0x08}, {0x0e, 0xef}, {0x1c, 0x01}, {0x1d, 0x02}, {0x1e, 0x03},
	{0x1f, 0x00}, {0x20, 0x00}, {0x0e, 0xee}, {0x12, 0x03}, {0x16, 0x10},
	{0x17, 0x02}, {0x1a, 0x22},

	/* Gamma Correction */
	{0x27, 0x62}, {0x28, 0x00}, {0x27, 0x62}, {0x28, 0x00}, {0x29, 0x00},
	{0x2a, 0x00}, {0x2f, 0x03}, {0x30, 0x10}, {0x31, 0x2b}, {0x32, 0x50},
	{0x33, 0x70}, {0x34, 0x90}, {0x35, 0xB0}, {0x36, 0xD0}, {0x37, 0x00},
	{0x38, 0x18}, {0x39, 0x57}, {0x3a, 0x89}, {0x3b, 0xac}, {0x3c, 0xc9},
	{0x3d, 0xde}, {0x3e, 0xef}, {0x2b, 0x00}, {0x2c, 0x00}, {0x2d, 0x40},
	{0x2e, 0xab},

	/* Contrast */
	{0x47, 0x10}, {0x48, 0x1f}, {0x49, 0xe3}, {0x4a, 0xf0}, {0x4b, 0x08},
	{0x4c, 0x14}, {0x4d, 0xe9}, {0x4e, 0xf5}, {0x98, 0x8a},

	{0xfa, 0x00},
	{MC521DA_TERM, MC521DA_TERM}
};

static int mc521da_attach(struct i2c_adapter *adapter);
static int mc521da_detach(struct i2c_client *client);

static struct i2c_driver mc521da_i2c_driver = {
	.driver = {
		   .owner = THIS_MODULE,
		   .name = "MC521DA Client",
		   },
	.attach_adapter = mc521da_attach,
	.detach_client = mc521da_detach,
};

static struct i2c_client mc521da_i2c_client = {
	.name = "MC521DA I2C dev",
	.addr = MC521DA_I2C_ADDRESS,
	.driver = &mc521da_i2c_driver,
};

static inline int mc521da_read_reg(u8 reg)
{
	return i2c_smbus_read_byte_data(&mc521da_i2c_client, reg);
}

static inline int mc521da_write_reg(u8 reg, u8 val)
{
	return i2c_smbus_write_byte_data(&mc521da_i2c_client, reg, val);
}

static int mc521da_write_regs(const struct mc521da_reg reglist[])
{
	int err;
	const struct mc521da_reg *next = reglist;

	while (!((next->reg == MC521DA_TERM) && (next->val == MC521DA_TERM))) {
		err = mc521da_write_reg(next->reg, next->val);
		if (err) {
			return err;
		}
		next++;
	}
	return 0;
}

/*!
 * mc521da sensor downscale function
 * @param downscale            bool
 * @return  Error code indicating success or failure
 */
static u8 mc521da_sensor_downscale(bool downscale)
{
	u8 data;
	u32 i = 0;

	if (downscale == true) {
		// VGA
		mc521da_write_reg(0xff, 0x01);

		mc521da_write_reg(0x52, 0x30);
		mc521da_write_reg(0x51, 0x00);

		mc521da_write_reg(0xda, 0x01);
		mc521da_write_reg(0x00, 0x8C);

		/* Wait for changes to take effect */
		while (i < 256) {
			i++;
			data = mc521da_read_reg(0x00);
			if ((data & 0x80) == 0)
				break;
			msleep(5);
		}

		/* ISP */
		mc521da_write_reg(0xff, 0x02);

		mc521da_write_reg(0x03, 0x3b);	/* Enable Decimator */

		mc521da_write_reg(0x7a, 0x74);
		mc521da_write_reg(0x7b, 0x01);
		mc521da_write_reg(0x7e, 0x50);
		mc521da_write_reg(0x7f, 0x50);
		mc521da_write_reg(0x80, 0x3c);
		mc521da_write_reg(0x81, 0x3c);
	} else {
		//UXGA
		mc521da_write_reg(0xff, 0x01);
		mc521da_write_reg(0x52, 0x10);
		mc521da_write_reg(0x51, 0x00);
		mc521da_write_reg(0xda, 0x00);

		/* update */
		mc521da_write_reg(0x00, 0x84);

		/* Wait for changes to take effect */
		while (i < 256) {
			i++;
			data = mc521da_read_reg(0x00);
			if ((data & 0x80) == 0)
				break;
			msleep(5);
		}

		/* ISP */
		mc521da_write_reg(0xff, 0x02);

		mc521da_write_reg(0x03, 0x3a);

		mc521da_write_reg(0x7a, 0x70);
		mc521da_write_reg(0x7b, 0x00);
		mc521da_write_reg(0x7e, 0xc8);
		mc521da_write_reg(0x7f, 0xc8);
		mc521da_write_reg(0x80, 0x96);
		mc521da_write_reg(0x81, 0x96);
	}

	return 0;
}

/*!
 * mc521da sensor interface Initialization
 * @param param            sensor_interface *
 * @param width            u32
 * @param height           u32
 * @return  None
 */
static void mc521da_interface(sensor_interface * param, u32 width, u32 height)
{
	param->clk_mode = 0x0;	//gated
	param->pixclk_pol = 0x0;
	param->data_width = 0x1;
	param->data_pol = 0x0;
	param->ext_vsync = 0x0;
	param->Vsync_pol = 0x0;
	param->Hsync_pol = 0x0;
	param->width = width - 1;
	param->height = height - 1;
	param->active_width = width;
	param->active_height = height;
	param->pixel_fmt = IPU_PIX_FMT_UYVY;
}

extern void gpio_sensor_reset(bool flag);

/*!
 * mc521da Reset function
 *
 * @return  None
 */
static sensor_interface *mc521da_reset(void)
{
	if (interface_param == NULL)
		return NULL;

	mc521da_interface(interface_param, format[1].width, format[1].height);
	set_mclk_rate(&interface_param->mclk);

	gpio_sensor_reset(true);
	msleep(10);
	gpio_sensor_reset(false);
	msleep(50);

	return interface_param;
}

/*!
 * mc521da sensor configuration
 *
 * @param frame_rate       int 	*
 * @param high_quality     int
 * @return  sensor_interface *
 */
static sensor_interface *mc521da_config(int *frame_rate, int high_quality)
{
	int num_clock_per_row, err;
	int max_rate = 0;
	int index = 1;
	u16 frame_height;

	if (high_quality == 1)
		index = 0;

	err = mc521da_write_regs(mc521da_initial);
	if (err) {
		/* Reduce the MCLK */
		interface_param->mclk = 20000000;
		mc521da_reset();

		printk(KERN_INFO "mc521da: mclk reduced\n");
		mc521da_write_regs(mc521da_initial);
	}

	mc521da_interface(interface_param, format[index].width,
			  format[index].height);

	if (index == 0) {
		mc521da_sensor_downscale(false);
	} else {
		mc521da_sensor_downscale(true);
	}

	num_clock_per_row = 1845;
	max_rate = interface_param->mclk * 3 * (index + 1)
	    / (2 * num_clock_per_row * 1300);

	if ((*frame_rate > max_rate) || (*frame_rate == 0)) {
		*frame_rate = max_rate;
	}

	frame_height = 1300 * max_rate / (*frame_rate);

	*frame_rate = interface_param->mclk * 3 * (index + 1)
	    / (2 * num_clock_per_row * frame_height);

	mc521da_write_reg(0xff, 0x01);
	mc521da_write_reg(0xE, frame_height & 0xFF);
	mc521da_write_reg(0xF, (frame_height & 0xFF00) >> 8);
	mc521da_write_reg(0xCC, frame_height & 0xFF);
	mc521da_write_reg(0xCD, (frame_height & 0xFF00) >> 8);

	return interface_param;
}

/*!
 * mc521da sensor set color configuration
 *
 * @param bright       int
 * @param saturation   int
 * @param red          int
 * @param green        int
 * @param blue         int
 * @return  None
 */
static void
mc521da_set_color(int bright, int saturation, int red, int green, int blue)
{
	/* Select ISP */
	mc521da_write_reg(0xff, 0x02);

	mc521da_write_reg(0x41, bright);
	mc521da_write_reg(0xca, red);
	mc521da_write_reg(0xcb, green);
	mc521da_write_reg(0xcc, blue);
}

/*!
 * mc521da sensor get color configuration
 *
 * @param bright       int *
 * @param saturation   int *
 * @param red          int *
 * @param green        int *
 * @param blue         int *
 * @return  None
 */
static void
mc521da_get_color(int *bright, int *saturation, int *red, int *green, int *blue)
{
	*saturation = 0;

	/* Select ISP */
	mc521da_write_reg(0xff, 0x02);

	*bright = mc521da_read_reg(0x41);
	*red = mc521da_read_reg(0xCA);
	*green = mc521da_read_reg(0xCB);
	*blue = mc521da_read_reg(0xCC);
}

struct camera_sensor camera_sensor_if = {
      set_color:mc521da_set_color,
      get_color:mc521da_get_color,
      config:mc521da_config,
      reset:mc521da_reset,
};

/*!
 * mc521da I2C detect_client function
 *
 * @param adapter            struct i2c_adapter *
 * @param address            int
 * @param kind               int
 *
 * @return  Error code indicating success or failure
 */
static int mc521da_detect_client(struct i2c_adapter *adapter, int address,
				 int kind)
{
	mc521da_i2c_client.adapter = adapter;
	if (i2c_attach_client(&mc521da_i2c_client)) {
		mc521da_i2c_client.adapter = NULL;
		printk(KERN_ERR "mc521da_attach: i2c_attach_client failed\n");
		return -1;
	}

	interface_param = (sensor_interface *)
	    kmalloc(sizeof(sensor_interface), GFP_KERNEL);
	if (!interface_param) {
		printk(KERN_ERR "mc521da_attach: kmalloc failed \n");
		return -1;
	}

	interface_param->mclk = 25000000;

	printk(KERN_INFO "mc521da Detected\n");

	return 0;
}

static unsigned short normal_i2c[] = { MC521DA_I2C_ADDRESS, I2C_CLIENT_END };

/* Magic definition of all other variables and things */
I2C_CLIENT_INSMOD;

static int mc521da_attach(struct i2c_adapter *adap)
{
	uint32_t mclk = 25000000;
	struct clk *clk;
	int err;

	clk = clk_get(NULL, "csi_clk");
	clk_enable(clk);
	set_mclk_rate(&mclk);

	gpio_sensor_reset(true);
	msleep(10);
	gpio_sensor_reset(false);
	msleep(100);

	err = i2c_probe(adap, &addr_data, &mc521da_detect_client);

	clk_disable(clk);
	clk_put(clk);

	return err;
}

/*!
 * mc521da I2C detach function
 *
 * @param client            struct i2c_client *
 * @return  Error code indicating success or failure
 */
static int mc521da_detach(struct i2c_client *client)
{
	int err;

	if (!mc521da_i2c_client.adapter)
		return -1;

	err = i2c_detach_client(&mc521da_i2c_client);
	mc521da_i2c_client.adapter = NULL;

	if (interface_param)
		kfree(interface_param);
	interface_param = NULL;

	return err;
}

extern void gpio_sensor_active(void);
extern void gpio_sensor_inactive(void);

/*!
 * mc521da init function
 *
 * @return  Error code indicating success or failure
 */
static __init int mc521da_init(void)
{
	gpio_sensor_active();
	return i2c_add_driver(&mc521da_i2c_driver);
}

/*!
 * mc521da cleanup function
 *
 * @return  Error code indicating success or failure
 */
static void __exit mc521da_clean(void)
{
	i2c_del_driver(&mc521da_i2c_driver);
	gpio_sensor_inactive();
}

module_init(mc521da_init);
module_exit(mc521da_clean);

/* Exported symbols for modules. */
EXPORT_SYMBOL(camera_sensor_if);

MODULE_AUTHOR("Freescale Semiconductor, Inc.");
MODULE_DESCRIPTION("MC521DA Camera Driver");
MODULE_LICENSE("GPL");
