/*---------------------------------------------------------------------------
 * Pegatron Modify History
 *---------------------------------------------------------------------------
 * 2009/11/03  Kevin Wei  - create vibrator driver V1.00
 *---------------------------------------------------------------------------
 *  echo 1 > /proc/vibrator/enable  // enable motor vibrator
 *  echo 0 > /proc/vibrator/enable  // disable motor vibrator
 *---------------------------------------------------------------------------
 * 2009/11/16  Eric Chou  - 1.fine-tune
 *---------------------------------------------------------------------------*/

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/proc_fs.h>             // proc
#include <asm/uaccess.h>               // copy_from_user()
#include <linux/regulator/consumer.h>  // regulator_enable() / regulator_disable()
#include <linux/delay.h>               // msleep()

#define DEBUG_TRACE        1
#if ( DEBUG_TRACE )
#define DEBUG_PRINTK(x)    printk x
#else
#define DEBUG_PRINTK(x)
#endif

static struct mxc_vibrator_platform_data *vibrator;
static struct proc_dir_entry *proc_dir_vibrator;
static struct proc_dir_entry *proc_entry_vibrator_enable;
static struct regulator *reg;

static void vibrator_power(int on)
{
   if ( reg )
   {
      if ( on )
         regulator_enable(reg);
      else
         regulator_disable(reg);
   }
}

static int vibrator_proc_write(struct file *fp, const char __user *buf, unsigned long count, void *data)
{
   int len;
   int enable;
   char str_32bitnum[10];

   if ( 0 == strcmp(data, "enable") )
   {
      printk("vibrator_proc_write enter\n");

      if ( count > 10 )
         len = 10;
      else
         len = count;

      if ( copy_from_user(str_32bitnum, buf, len) )
         return -EFAULT;

      enable = simple_strtol(str_32bitnum, NULL, 16);
      if ( enable == 1 || enable == 0 )
      {
         vibrator_power(enable);   // enable power
         msleep(10);
         vibrator->enable(enable); // gpio set high / low
      }
      return count;
    }
    return 0;
}

static int vibrator_proc_read(char *buf, char **start, off_t offset, int count, int *eof, void *data)
{
   return 0;
}

static void vibrator_create_proc(void)
{
   proc_dir_vibrator = proc_mkdir("vibrator", NULL);
   if(proc_dir_vibrator == NULL)
      printk("vibrator proc dir failed\n");

   proc_entry_vibrator_enable = create_proc_entry("enable", 0644, proc_dir_vibrator);
   if ( proc_entry_vibrator_enable == NULL )
      printk("vibrator proc entry enable failed \n");

   proc_entry_vibrator_enable->owner      = THIS_MODULE;
   proc_entry_vibrator_enable->read_proc  = vibrator_proc_read;
   proc_entry_vibrator_enable->write_proc = vibrator_proc_write;
   proc_entry_vibrator_enable->data       = (void*)"enable";
}

static void vibrator_remove_proc(void)
{
   remove_proc_entry("enable", proc_dir_vibrator);
}

static int vibrator_probe(struct platform_device *pdev)
{
   int err;

   vibrator = pdev->dev.platform_data;

   if ( !vibrator )
      return -EBUSY;

   if ( vibrator->reg_vdd )
   {
      reg = regulator_get(NULL, vibrator->reg_vdd);
      if ( !IS_ERR(reg) )
      {
         printk("set VSD 3.15V success \n");
         regulator_set_voltage(reg, 3150000, 3150000); // set power voltage
      }
      else
      {
         printk("set VSD 3.15V failure \n");
         err = -EINVAL;
         return err;
      }
   }
   vibrator_create_proc();

   return 0;
}

static int __devexit vibrator_remove(struct platform_device *pdev)
{
   vibrator_remove_proc();

   if (reg)
   {
      vibrator_power(0);
      regulator_put(reg);
   }
   return 0;
}

static struct platform_driver vibrator_driver =
{
   .probe    = vibrator_probe,
   .remove   = __devexit_p(vibrator_remove),
   .driver   =
   {
      .name  = "vibrator",
      .owner = THIS_MODULE,
   },
};

static int __init vibrator_init(void)
{
   return platform_driver_register(&vibrator_driver);
}

static void __exit vibrator_exit(void)
{
   platform_driver_unregister(&vibrator_driver);
}

module_init(vibrator_init);
module_exit(vibrator_exit);

MODULE_AUTHOR("Pegatron Kevin");
MODULE_DESCRIPTION("Motor Vibrator Driver");
MODULE_LICENSE("GPL");
