/* **********************************************************
 * Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 * driver.c --
 *
 * Filesystem part of Linux kernel module implementation of driver for
 * the VMware Host/Guest filesystem.
 */

/* Must come before any kernel header file */
#include "driver-config.h"

#include "main.h"
#include "staticEscape.h"  // for buffer escaping/unescaping
#include "escBitvector.h"  // bitvectors for escaping
#include "cpName.h"        // cross-platform name conversion
#include "hgfsUtil.h"      // for cross-platform time conversion
#include "hgfsDevLinux.h"  // for HgfsMountInfo definition
#include "hgfsProto.h"     // for hgfs protocol

#include <linux/vfs.h>

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 25)
#define KERNEL_25_FS 0
#else
#define KERNEL_25_FS 1
#endif

/*
 * Macros for accessing members that are private to this code in
 * sb/inode/file structs.
 */

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 42)
#define HGFS_SET_SB_TO_COMMON(sb, common) do { (sb)->u.generic_sbp = (common); } while (0)
#define HGFS_SB_TO_COMMON(sb)             ((HgfsSuperInfo *)(sb)->u.generic_sbp)
#else
#define HGFS_SET_SB_TO_COMMON(sb, common) do { (sb)->s_fs_info = (common); } while (0)
#define HGFS_SB_TO_COMMON(sb)             ((HgfsSuperInfo *)(sb)->s_fs_info)
#endif

#define INODE_SET_II_P(inode, info) do { (inode)->u.generic_ip = (info); } while (0)
#define INODE_GET_II_P(inode) ((HgfsInodeInfo *)(inode)->u.generic_ip)

/*
 * 2.5.x kernels support nanoseconds timestamps.
 */

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 5, 48)
#define HGFS_SET_TIME(unixtm,nttime) HgfsConvertFromNtTime(&unixtm, nttime)
/*
 * Beware! This macro returns list of two elements. Do not add braces around.
 */
#define HGFS_GET_TIME(unixtm) unixtm, 0L
#else
#define HGFS_SET_TIME(unixtm,nttime) HgfsConvertFromNtTimeNsec(&unixtm, nttime)
/*
 * Beware! This macro returns list of two elements. Do not add braces around.
 */
#define HGFS_GET_TIME(unixtm) unixtm.tv_sec, unixtm.tv_nsec
#endif

/*
 * 2.5.74 renamed struct statfs to kstatfs.
 */

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 5, 74)
#define compat_kstatfs kstatfs
#else
#define compat_kstatfs statfs
#endif


/*
 * 2.6.x kernels define FS_BINARY_MOUNTDATA which we must use.
 */

#ifndef FS_BINARY_MOUNTDATA
#define FS_BINARY_MOUNTDATA 0
#endif


/*
 * For files opened in our actual Host/Guest filesystem, the
 * file->private_data field is used for storing the HgfsHandle of the
 * opened file. This macro is for accessing the handle from the
 * file *.
 */
#define FILE_SET_HAND_P(file, handle) do { (file)->private_data = (handle); } while (0)
#define FILE_GET_HAND_P(file)         ((HgfsHandle *)(file)->private_data)
#define FILE_GET_HAND(file)           (*FILE_GET_HAND_P(file))


/* Data stored per inode in inode->u */
typedef struct HgfsInodeInfo {
   char *name;         /* Name of this file on remote server (w.r.t. the fs root) */
} HgfsInodeInfo;


/* maybe I should just put this ahead of HgfsDentryRevalidate in the file */
static int HgfsRevalidate(struct dentry *dentry);

/*
 * Operations structures
 */
static struct inode_operations HgfsFileInodeOperations;
static struct inode_operations HgfsDirInodeOperations;
static struct file_operations HgfsFileOperations;
static struct file_operations HgfsDirOperations;


/*
 * Helper functions
 */

// This symbol needs to be defined.
void
Log(const char *string, ...)
{
   // do nothing.
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsMakeFullName --
 *
 *    Makes a fully qualified pathname from a base name (i.e. the
 *    name of the parent directory) and the component name (i.e. the
 *    name of the file).
 *
 *    The new pathname is kmalloc'd, and must be freed after use.
 *
 * Results:
 *    Returns a pointer to the full name on success, or NULL if
 *    memory could not be allocated.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static char *
HgfsMakeFullName(char const *baseName,      // IN: Base dir name
                 char const *componentName) // IN: Name of file
{
   unsigned int baseNameLen;
   unsigned int componentNameLen;
   char *fullName;

   ASSERT(baseName);
   ASSERT(componentName);

   // XXX We should save this length in the private inode info --hpreg
   baseNameLen = strlen(baseName);
   componentNameLen = strlen(componentName);
   fullName = kmalloc(baseNameLen + componentNameLen + 2, GFP_KERNEL);
   if (!fullName) {
      return NULL;
   }

   memcpy(fullName, baseName, baseNameLen);
   fullName[baseNameLen] = '/';
   memcpy(fullName + baseNameLen + 1, componentName, componentNameLen);
   fullName[baseNameLen + 1 + componentNameLen] = '\0';

   return fullName;
}


/*
 *-----------------------------------------------------------------------------
 *
 * HgfsEscapeBuffer --
 *
 *    Escape any characters that are not legal in a linux filename,
 *    which is just the character "/". We also of course have to
 *    escape the escape character, which is "%".
 *
 *    sizeBufOut does NOT include space for the terminating NUL;
 *    in other words, bufOut should have sizeBufOut + 1 bytes
 *    of room. [bac]
 *
 * Results:
 *    On success, the size (excluding the NUL terminator) of the
 *    escaped, NUL terminated buffer.
 *    On failure (bufOut not big enough to hold result), negative value.
 *
 * Side effects:
 *    None
 *
 *-----------------------------------------------------------------------------
 */

static int
HgfsEscapeBuffer(char const *bufIn, // IN:  Buffer with unescaped input
                 uint32 sizeIn,     // IN:  Size of input buffer
                 uint32 sizeBufOut, // IN:  Size of output buffer
                 char *bufOut)      // OUT: Buffer for escaped output
{
   /*
    * This is just a wrapper around the more general escape
    * routine; we pass it the correct bitvector and the
    * buffer to escape. [bac]
    */
   EscBitVector bytesToEsc;

   ASSERT(bufIn);
   ASSERT(bufOut);

   /* Set up the bitvector for "/" and "%" */
   EscBitVector_Init(&bytesToEsc);
   EscBitVector_Set(&bytesToEsc, (unsigned char)'%');
   EscBitVector_Set(&bytesToEsc, (unsigned char)'/');

   return StaticEscape_Do('%',
                          &bytesToEsc,
                          bufIn,
                          sizeIn,
                          sizeBufOut,
                          bufOut);
}


/*
 *-----------------------------------------------------------------------------
 *
 * HgfsUnescapeBuffer --
 *
 *    Unescape a buffer that was escaped using HgfsEscapeBuffer.
 *
 *    The unescaping is done in place in the input buffer, and
 *    can not fail.
 *
 * Results:
 *    The size (excluding the NUL terminator) of the unescaped, NUL
 *    terminated buffer.
 *
 * Side effects:
 *    None
 *
 *-----------------------------------------------------------------------------
 */

static int
HgfsUnescapeBuffer(char *bufIn,   // IN: Buffer to be unescaped
                   uint32 sizeIn) // IN: Size of input buffer
{
   /*
    * This is just a wrapper around the more general unescape
    * routine; we pass it the correct escape characer and the
    * buffer to unescape. [bac]
    */
   ASSERT(bufIn);
   return StaticEscape_Undo('%', bufIn, sizeIn);
}


/*
 * Testing only
 */

#if 0
static void
TestCPConversion(void)
{
   char buf1[1024];
   char buf2[1024];
   char name[50];
   unsigned int i;
   int n;

   /* convert a name to CP and back */
   strcpy(name, "/home/bac/temp");
   LOG(4, (KERN_DEBUG "converting %s\n", name));
   n = CPName_ConvertTo(name, sizeof buf1, buf1);
   if (n < 0) {
      LOG(4, (KERN_DEBUG "Convert to CP failed\n"));
      return;
   }
   LOG(4, (KERN_DEBUG "Buf is %d big\n", n));
   LOG(4, (KERN_DEBUG "%s\n", CPName_Print(buf1, n)));

   LOG(4, (KERN_DEBUG "converting back...\n"));
   n = CPName_ConvertFrom(buf1, n, sizeof buf2, buf2);
   if (n < 0) {
      LOG(4, (KERN_DEBUG "Convert from CP failed\n"));
      return;
   }
   LOG(4, (KERN_DEBUG "name is \"%s\", size %d\n", buf2, n));

   LOG(4, (KERN_DEBUG "\n"));

   /*
    * Escape a name and convert it FROM CP.
    * This is what we do when we get a name from the
    * server, before we do a lookup, etc.
    */
   strcpy(name, "my/name");
   LOG(4, (KERN_DEBUG "escaping/converting \"%s\"\n", name));
   n = HgfsEscapeBuffer(name, strlen(name), (sizeof buf1) - 1, buf1);
   if (n < 0) {
      LOG(4, (KERN_DEBUG "Escape buffer failed\n"));
      return;
   }
   LOG(4, (KERN_DEBUG "escaped buffer is \"%s\"\n", buf1));
   n = CPName_ConvertFrom(buf1, n, sizeof buf2, buf2);
   if (n < 0) {
      LOG(4, (KERN_DEBUG "convert from CP failed\n"));
      return;
   }
   LOG(4, (KERN_DEBUG "converted buffer is \"%s\"\n", buf2));

   LOG(4, (KERN_DEBUG "\n"));

   /* do same for a multi-part name. */
   strcpy(name, "my name%isthis");
   n = strlen(name);
   name[2] = '\0';
   LOG(4, (KERN_DEBUG "escaping/converting: "));
   LOG(4, (KERN_DEBUG "%s\n", CPName_Print(name, n)));

   n = HgfsEscapeBuffer(name, n, (sizeof buf1) - 1, buf1);
   if (n < 0) {
      LOG(4, (KERN_DEBUG "Escape buffer failed\n"));
      return;
   }
   LOG(4, (KERN_DEBUG "escaped buffer "));
   LOG(4, (KERN_DEBUG "%s\n", CPName_Print(buf1, n)));

   n = CPName_ConvertFrom(buf1, n, sizeof buf2, buf2);
   if (n < 0) {
      LOG(4, (KERN_DEBUG "convert from CP failed\n"));
      return;
   }
   LOG(4, (KERN_DEBUG "converted buffer is \"%s\"\n", buf2));

   LOG(4, (KERN_DEBUG "\n\n"));
   /*
    * Unescape a name and convert it TO CP. This is what we do before
    * sending a name to the server.
    *
    * The conversion has to be done first, otherwise we may unescape
    * some / which will then be incorrectly interpreted by the
    * conversion as a path separator.
    */
   {
      char *tests[] = {
         "/",
         "/test/",
         "/test/.",
         "/home/bac/temp",
         "/home/b%c/temp",
         "/home/b%2F/tem%25p",
      };

      for (i = 0; i < sizeof tests / sizeof tests[0]; i++) {
         LOG(4, (KERN_DEBUG "testing \"%s\"\n", tests[i]));
         n = CPName_ConvertTo(tests[i], 1024, buf1);
         if (n < 0) {
            LOG(4, (KERN_DEBUG "conversion failed\n"));
            return;
         }
         LOG(4, (KERN_DEBUG "converted buffer "));
         LOG(4, (KERN_DEBUG "%s\n", CPName_Print(buf1, n)));

         n = HgfsUnescapeBuffer(buf1, n);
         if (n < 0) {
            LOG(4, (KERN_DEBUG "unescaping failed\n"));
            return;
         }
         LOG(4, (KERN_DEBUG "converted/unescaped buffer "));
         LOG(4, (KERN_DEBUG "%s\n", CPName_Print(buf1, n)));
         LOG(4, (KERN_DEBUG "\n"));
      }
   }
}
#endif

/*
 * This is the test code from staticEscape.c, modified for
 * the kernel (i.e. printf -> printk).
 */
#if 0
static void
TestEscape(void)
{
   uint32 NAME_SIZE = 20;
   EscBitVector toEsc;
   char buf[NAME_SIZE+1];
   unsigned int i;
   int result;

   static struct {
      const char *in;
      const char *out;
   } tests[] = {
      { "", "", },
      { "b", "b", },
      { "bac", "bac", },
      { "/", "%2F", },
      { "%", "%25", },
      { "home/bac", "home%2Fbac", },
      { "////", "%2F%2F%2F%2F", },
      { "%/%%", "%25%2F%25%25", },
      { "/bac/", "%2Fbac%2F", },
      { "%bacbac%", "%25bacbac%25", },
      /*
       * The tests from here down should fail because the result is
       * too big to fit in the buffer.
       */
      { "this one is just too big", "", }, // input just to big
      { "first chunk too big/", "", },     // first chunk too big
      { "1st chunk okay/,fail", "", },     // first chunk okay, last chunk too big
   };

   /* set up bit vector */
   EscBitVector_Init(&toEsc);
   EscBitVector_Set(&toEsc, (unsigned char)'%');
   EscBitVector_Set(&toEsc, (unsigned char)'/');

   /* test bit vector */
   LOG(4, (KERN_DEBUG "chars to escape are: "));
   for (i = 0; i < 256; i++) {
      if (EscBitVector_Test(&toEsc, i)) {
         LOG(4, (KERN_DEBUG "%c ", (unsigned char)i));
      }
   }
   LOG(4, (KERN_DEBUG "\n\n"));


   /*
    * Test buffer escaping/unescaping.
    *
    * These first 10 tests should pass.
    */
   for (i = 0; i < 10; i++) {
      LOG(4, (KERN_DEBUG "Test %u: \"%s\", ", i, tests[i].in));
      result = StaticEscape_Do('%',
                               &toEsc,
                               tests[i].in,
                               strlen(tests[i].in),
                               NAME_SIZE+1,
                               buf);

      LOG(4, (KERN_DEBUG "escaped buf is \"%s\". ", buf));
      if (result != strlen(tests[i].out) ||
         strcmp(buf, tests[i].out) != 0)
      {
         LOG(4, (KERN_DEBUG "escaping test %u failed: \"%s\"\n", i, buf));
      }

      result = StaticEscape_Undo('%', buf, result);
      if (result != strlen(tests[i].in) ||
         strcmp(buf, tests[i].in) != 0)
      {
         LOG(4, (KERN_DEBUG "unescaping test %u failed: \"%s\"\n", i, buf));
      }
      LOG(4, (KERN_DEBUG "\t\t\t...passed.\n"));
   }

   /* These remaining tests should fail. See above. */

   for (i = 10; i < sizeof tests / sizeof tests[0]; i++) {
      LOG(4, (KERN_DEBUG "Test %u: \"%s\", ", i, tests[i].in));
      result = result = StaticEscape_Do('%',
                                        &toEsc,
                                        tests[i].in,
                                        strlen(tests[i].in),
                                        NAME_SIZE+1,
                                        buf);
      if (result >= 0) {
         LOG(4, (KERN_DEBUG "test %u failed to fail\n", i));
      }

      LOG(4, (KERN_DEBUG "\t\t\t...passed.\n"));
   }

   LOG(4, (KERN_DEBUG "ALL TESTS PASSED\n"));
}
#endif


#if 0
/*
 * Testing time conversion
 */
void
TestTimeConversion(void)
{
   time_t tests[] = {0, 1, 1000, 1000000};
   time_t temp;
   unsigned int i;

   LOG(4, (KERN_DEBUG "testing time conversion\n"));
   LOG(4, (KERN_DEBUG "sizeof time_t is: %u\n", sizeof(time_t)));

   for (i = 0; i < sizeof tests / sizeof tests[0]; i++) {
      HgfsConvertFromNtTime(&temp, HgfsConvertToNtTime(tests[i], 0));
      if (tests[i] != temp) {
         LOG(4, (KERN_DEBUG "test failed: %lu != %lu\n", tests[i], temp));
         return;
      }
      LOG(4, (KERN_DEBUG "test passed: %lu = %lu\n", tests[i], temp));
   }

   LOG(4, (KERN_DEBUG "all tests passed\n"));
}

#endif


/*
 *----------------------------------------------------------------------
 *
 * HgfsGetDentryFullName --
 *
 *    Resolve a dentry name into a full pathname w.r.t. the
 *    filesystem root.
 *
 *    Takes a dentry and builds its fully qualified name by
 *    traversing its "parent" links until the root is
 *    reached.
 *
 *    XXX This is used in only one place (as a debugging check, and
 *    an unnecessary one at that). It can probably just be nuked.
 *
 * Results:
 *    Returns a pointer to a new kmalloc'd string containing the
 *    full filename on success, or NULL if there is an error.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

/*
 * XXX This has had very little testing, needs to be tested heavily.
 *
 * This is probably only needed if there is a rename; otherwise, the
 * names in the inodes should be valid. We need a way of invalidating
 * them on a rename, btw.
 */

static char *
HgfsGetDentryFullName(struct dentry *dentry) // IN
{
   struct dentry *temp;
   char *fullName = NULL;
   char *tempName = NULL;
   int len = 0;

   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsGetDentryFullName: entered\n"));

   ASSERT(dentry);

   if (!dentry) {
      return NULL;
   }

   /*
    * XXX do I need to hold some kind of lock while I'm doing this?
    * what if a d_name gets lengthened in between my kmalloc and the
    * strcats?
    */
   for (temp = dentry; temp->d_parent != temp; temp = temp->d_parent) {
      /*
       * Allocate space for the two strings, plus "/", plus the
       * trailing \0.
       */
      len = (fullName) ? strlen(fullName) : 0;
      tempName = kmalloc(((strlen(temp->d_name.name) +
                           len + 2) * sizeof(char)), GFP_KERNEL);
      if (!tempName) {
         LOG(4, (KERN_DEBUG "VMware hgfs: HgfsGetDentryFullName: "
                 "can't allocate memory\n"));
         return NULL;
      }
      memset(tempName, 0, len);
      strcat(tempName, temp->d_name.name);
      if (fullName) {
         strcat(tempName, "/");
         strcat(tempName, fullName);
         kfree(fullName);
      }
      fullName = tempName;
   }

   /*
    * Now we're at the root, so just append "/" to the beginning. If
    * we started at the root, then fullName is still NULL and we didn't
    * execute any of the for loop above, so we will just return "/".
    */
   ASSERT(!(strcmp(temp->d_name.name,"/")));
   len = (fullName) ? strlen(fullName) : 0;
   tempName = kmalloc(((len + 2) * sizeof(char)), GFP_KERNEL);
   if (!tempName) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsGetDentryFullName: "
              "can't allocate memory\n"));
      return NULL;
   }
   memset(tempName, 0, ((len + 2) * sizeof(char)));
   strcat(tempName, "/");
   if (fullName) {
      strcat(tempName, fullName);
      kfree(fullName);
   }
   fullName = tempName;

   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsGetDentryFullName: fullname is %s\n",
          fullName));
   return fullName;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsGetNextIno --
 *
 *    Get the next free inode number.
 *
 *    Grabs the hgfsBigLock, so this can't be called with the
 *    lock held.
 *
 *    XXX Note that this is seriously broken; right now I just hand
 *    inode numbers out in increasing order. I need a way of handing out
 *    unused inode numbers that won't wrap around after 2^32 files have been
 *    visited. [bac]
 *
 * Results:
 *    Returns next free inode number.
 *
 * Side effects:
 *    Increments nextIno in the HgfsSuperInfo for this sb.
 *
 *----------------------------------------------------------------------
 */

static ino_t
HgfsGetNextIno(struct super_block *sb) // IN: Superblock of this fs
{
   ino_t ino;
   HgfsSuperInfo *common;

   ASSERT(sb);

   common = HGFS_SB_TO_COMMON(sb);
   spin_lock(&hgfsBigLock);
   ino = common->nextIno;
   common->nextIno++;
   spin_unlock(&hgfsBigLock);
   if (ino == HGFS_ROOT_INO) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsGetNextIno: inode numbers wrapped around!\n"));
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsGetNextIno: XXXXXXXXXXXXXXXXXXXXXXXXXXXXX\n"));
   }

   return ino;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsSetFileType --
 *
 *    Set file type in inode according to the hgfs attributes.
 *
 * Results:
 *    None
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static void
HgfsSetFileType(struct inode *inode,  // IN/OUT: Inode to update
                HgfsAttr const *attr) // IN: Attrs to use to update
{
   ASSERT(inode);
   ASSERT(attr);

   switch (attr->type) {
   case HGFS_FILE_TYPE_DIRECTORY:
      inode->i_mode = S_IFDIR;
      inode->i_op = &HgfsDirInodeOperations;
      inode->i_fop = &HgfsDirOperations;
      break;

   case HGFS_FILE_TYPE_REGULAR:
      inode->i_mode = S_IFREG;
      inode->i_op = &HgfsFileInodeOperations;
      inode->i_fop = &HgfsFileOperations;
      break;

   default:
      /*
       * XXX Should never happen. I'd put NOT_IMPLEMENTED() here
       * but if the driver ever goes in the host it's probably not
       * a good idea for an attacker to be able to hang the host
       * simply by using a bogus file type in a reply. [bac]
       */
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsSetFileType: UNSUPPORTED inode type\n"));
      inode->i_mode = 0;
//      NOT_IMPLEMENTED();
      break;
   }
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsChangeFileAttributes --
 *
 *    Update an inode's attributes to match those of the HgfsAttr.
 *
 *    XXX not yet fully functional
 *
 * Results:
 *    None
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static void
HgfsChangeFileAttributes(struct inode *inode,  // IN/OUT: Inode to update
                         HgfsAttr const *attr) // IN: Attrs to use to update
{
   ASSERT(inode);
   ASSERT(attr);

   /*
    * XXX Should we invalidate inode pages here if an inode is a regular
    * file and its size does not match the attr size?
    */

   /*
    * Set appropriate fields in the inode.
    *
    * XXX I'm still faking a few fields here. Clean it up.
    */

   HgfsSetFileType(inode, attr);

   /* Set the access mode. */
   inode->i_mode &= ~S_IRWXUGO;
   inode->i_mode |= (attr->permissions << 6); /* Only applies to owner. */
   inode->i_nlink = 1; /* Bogus? */
   inode->i_uid = 0;   /* This is bogus, should be the mount owner. */
   inode->i_gid = 0;   /* This is bogus, should be the mount owner. */
   inode->i_rdev = 0;  /* Device nodes are not supported */
   inode->i_blksize = HGFS_BLOCKSIZE;
   inode->i_blocks = (attr->size + HGFS_BLOCKSIZE - 1) / HGFS_BLOCKSIZE;
   inode->i_size = attr->size;
   HGFS_SET_TIME(inode->i_atime, attr->accessTime);
   HGFS_SET_TIME(inode->i_mtime, attr->writeTime);
   /*
    * Windows doesn't know about ctime, and might send us something
    * bogus; if the ctime is invalid, use the mtime instead.
    */
   if (HGFS_SET_TIME(inode->i_ctime, attr->attrChangeTime)) {
      inode->i_ctime = inode->i_mtime;
   }

   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsChangeFileAttributes: size %"FMT64"u, ",
          attr->size));
   LOG(6, (KERN_DEBUG "Inode remote name: \"%s\"", INODE_GET_II_P(inode)->name));
   LOG(6, (KERN_DEBUG "mode %o, perms were %o\n", inode->i_mode, attr->permissions));
   LOG(6, (KERN_DEBUG "atime: %"FMT64"u/%ld.%09ld\n",
           attr->accessTime, HGFS_GET_TIME(inode->i_atime)));
   LOG(6, (KERN_DEBUG "mtime: %"FMT64"u/%ld.%09ld\n",
           attr->writeTime, HGFS_GET_TIME(inode->i_mtime)));
   LOG(6, (KERN_DEBUG "ctime: %"FMT64"u/%ld.%09ld\n",
           attr->attrChangeTime, HGFS_GET_TIME(inode->i_ctime)));
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsInitializeInode --
 *
 *    Set up a new inode with the correct i_op and i_fop pointers
 *    and a HgfsInodeInfo with the correct remote file name.
 *
 * Results:
 *    None
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static void
HgfsInitializeInode(struct inode *inode,    // IN/OUT: Inode to initialize
                    HgfsAttr const *attr,   // IN: Attributes of this file
                    char const *remoteName) // IN: Remote name of this file
{
   HgfsInodeInfo *iinfo;

   ASSERT(inode);
   ASSERT(attr);
   ASSERT(remoteName);

   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsInitializeInode: entered\n"));

   /*
    * The kernel copies the inode's 'i_fop' field to the 'f_op' field of any
    * struct file that it makes point (indirectly, via a dentry) to the inode.
    */

   /* Set file type. */
   HgfsSetFileType(inode, attr);

   inode->i_size = attr->size;

   iinfo = kmalloc(sizeof *iinfo, GFP_KERNEL);
   if (!iinfo) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsInitializeInode: no memory for iinfo!\n"));
      return;
   }
   iinfo->name = vmstrdup(remoteName);
   INODE_SET_II_P(inode, iinfo);
   /*
    * XXX check return value of kstrdup? it's not like there's
    * anything we can do if it fails...
    */
   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsInitializeInode: finished\n"));
}


/*
 *-----------------------------------------------------------------------------
 *
 * HgfsIget --
 *
 *    Lookup or create an inode with the given attributes and remote filename.
 *
 * Results:
 *    The inode on success
 *    NULL on failure
 *
 * Side effects:
 *    None
 *
 *-----------------------------------------------------------------------------
 */

static struct inode *
HgfsIget(struct super_block *sb, // IN: Superblock of this fs
         ino_t ino,              // IN: Inode number to lookup/create
         HgfsAttr const *attr,   // IN: Attributes to create with
         char const *remoteName) // IN: Remote name for new file
{
   struct inode *inode;

   ASSERT(sb);
   ASSERT(attr);
   ASSERT(remoteName);

   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsIget: entered\n"));

   inode = iget(sb, ino);
   if (inode) {
      /*
       * When it creates a new VFS inode, iget() sets INODE_GET_II_P(inode) to
       * NULL. --hpreg
       *
       * XXX I verified all callers of HgfsIget(): VFS inodes are currently
       *     always created, so this should be an ASSERT. --hpreg
       */
      if (!(INODE_GET_II_P(inode))) {
         HgfsInitializeInode(inode, attr, remoteName);
      }

      /*
       * XXX If this inode was already initialized,
       * do I really want to change its attributes
       * and reset its version? I think not...
       */
      HgfsChangeFileAttributes(inode, attr);
      /* XXX Needed? */
      inode->i_version = 1;
   }

   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsIget: done\n"));
   return inode;
}


/*
 * Communications functions
 */


/*
 *----------------------------------------------------------------------
 *
 * HgfsDisableSignals --
 *
 *    Disable all signals except SIGKILL (and disable even SIGKILL
 *    when exiting).
 *
 * Results:
 *    None.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static void
HgfsDisableSignals(sigset_t *old_set)  // OUT: Previously blocked signals
{
   unsigned long mask, flags;

   spin_lock_irqsave(&current->compat_sigmask_lock, flags);
   *old_set = current->blocked;
   /*
    * Disable all signals if task exits. If task is killed by SIGKILL,
    * we abort all our cleanup calls with -ERESTARTSYS, and it is not
    * exactly what we want. Maybe we should post CLOSE request
    * asynchronously...
    */
   if (current->flags & PF_EXITING) {
      mask = 0;
   } else {
      mask = sigmask(SIGKILL);
   }
   siginitsetinv(&current->blocked, mask);
   compat_recalc_sigpending();
   spin_unlock_irqrestore(&current->compat_sigmask_lock, flags);
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsRestoreSignals --
 *
 *    Restore signals blocking previously disabled by HgfsDisableSignals.
 *
 * Results:
 *    None.
 *
 * Side effects:
 *    Signal may become pending.
 *
 *----------------------------------------------------------------------
 */

static void
HgfsRestoreSignals(const sigset_t *old_set)  // IN: Previous blocking state
{
   unsigned long flags;

   spin_lock_irqsave(&current->compat_sigmask_lock, flags);
   current->blocked = *old_set;
   compat_recalc_sigpending();
   spin_unlock_irqrestore(&current->compat_sigmask_lock, flags);
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsAddClientWaitOnRequest --
 *
 *    Makes the current client wait for a reply to the request.
 *
 * Results:
 *    None
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static inline void
HgfsAddClientWaitOnRequest(HgfsReq *req) // IN/OUT: Request
{
   ASSERT(req);
   req->client = current;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsRemoveClientWaitOnRequest --
 *
 *    Removes the current process from waiting on this request.
 *
 * Results:
 *    None
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static inline void
HgfsRemoveClientWaitOnRequest(HgfsReq *req) // IN/OUT: Request
{
   ASSERT(req);
   req->client = NULL;
}



/*
 *----------------------------------------------------------------------
 *
 * HgfsTestAndFreeRequest --
 *
 *    Test to see if the request was interrupted and free it (or not)
 *    accordingly.
 *
 *    - If it was not interrupted, it is finished, so free it.
 *    - If it was interrupted BEFORE being sent, it is not in use
 *      by anyone, so free it.
 *    - If it was interrupted AFTER being sent, then it has to
 *      stay around until the reply returns, at which time HgfsDevWrite
 *      will free it, so we do NOT free it.
 *
 *    Should be called with the result returned by HgfsSendRequest and
 *    the pointer to the corresponding request. [bac]
 *
 * Results:
 *    None
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static void
HgfsTestAndFreeRequest(int result,   // IN: Result of sending request
                       HgfsReq *req) // IN: Request to check
{
   ASSERT(req);

   if (!req) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsTestAndFreeRequest: NULL request!\n"));
      return;
   }

   /*
    * If result is -ERESTARTSYS and the request state is not "inactive", it
    * means we got interrupted after the request was sent. Otherwise,
    * either we were interrupted before the request was sent or we
    * weren't interrupted at all. In either of the latter cases, we
    * free the request ourselves. This also releases the semaphore.
    */
   if ((result == -ERESTARTSYS) && (req->stateFile != HGFS_REQ_STATE_INACTIVE)) {
      LOG(8, (KERN_DEBUG "VMware hgfs: HgfsTestAndFreeRequest: NOT freeing request\n"));
   } else {
      LOG(8, (KERN_DEBUG "VMware hgfs: HgfsTestAndFreeRequest: freeing request\n"));
      spin_lock(&hgfsBigLock);
      HgfsFreeRequest(req);
      spin_unlock(&hgfsBigLock);
   }
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsGetNewRequest --
 *
 *    Get a new request structure off the free list and initialize it.
 *
 * Results:
 *    On success the new struct is returned with all fields
 *    initialized. Returns NULL on failure. Note that this should
 *    never fail, because this is only called from a process that has
 *    already acquired the requestSem.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static HgfsReq *
HgfsGetNewRequest(void)
{
   HgfsReq *req = NULL;
   sigset_t sigset;

   HgfsDisableSignals(&sigset);
   /* Acquire the request semaphore. */
   if (down_interruptible(&requestSem)) {
      HgfsRestoreSignals(&sigset);
      /* If we get interrupted while waiting, abort. */
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsGetNewRequest: interrupted while "
              "waiting for outstandingSem, returning\n"));
      return NULL;
   }
   HgfsRestoreSignals(&sigset);

   spin_lock(&hgfsBigLock);

   /*
    * Remove a req from the front of the free list.
    *
    * The requestSem protects the free list, so if we get here this
    * should never be null.
    */
   req = list_entry(reqFreeList.next, HgfsReq, list);
   ASSERT(req != list_entry(&reqFreeList, HgfsReq, list));
   list_del_init(&req->list);

   LOG(8, (KERN_DEBUG "VMware hgfs: HgfsGetNewRequest: got req #%u off the free list\n",
           req->id));

   spin_unlock(&hgfsBigLock);

   req->client = NULL;
   req->error = 0;
   req->packetSize = 0;
   memset(req->packet, 0, HGFS_PACKET_MAX);    /* XXX Is this necessary? */

   return req;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsWaitRequestReply --
 *
 *    Wait for the reply to a request that we sent. Called with the
 *    hgfsBigLock held, which gets released and then reacquired.
 *
 * Results:
 *    Returns zero when the answer has been received, -ERESTARTSYS if
 *    interrupted, or an appropriate error otherwise. It is important
 *    that -ERESTARTSYS be returned in the event of a signal getting caught,
 *    because calling functions test the return value to determine
 *    whether or not to free the request object.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static int
HgfsWaitRequestReply(HgfsReq *req)  // IN/OUT: Request object
{
   int sig; // XXX debugging only
   unsigned long s;
   sigset_t sigset;
   int err = 0;

   ASSERT(req);

   if (!req) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsWaitRequestReply: null req\n"));
      return -EINVAL;
   }

   HgfsAddClientWaitOnRequest(req);

   /*
    * Loop until the request goes into "finished" state, or until
    * we catch a signal.
    */

   HgfsDisableSignals(&sigset);
   while ((req) && (req->stateFile != HGFS_REQ_STATE_FINISHED)) {
      set_current_state(TASK_INTERRUPTIBLE);
      if (signal_pending(current)) {
         /* We caught a signal. */
         s = current->pending.signal.sig[0];
         sig = (s) ? 1 + ffz(~s) : 0;
         LOG(4, (KERN_DEBUG "VMware hgfs: HgfsWaitRequestReply: caught "
                 "signal %d while waiting!\n", sig));
	 /*
	  * Nobody in kernel should ever use -EINTR: syscall layer translates
	  * various -ERESTART* to appropriate -EINTR or restart according to
	  * the signal settings.
	  */
         err = -ERESTARTSYS;
	 break;
      }
      spin_unlock(&hgfsBigLock);
      schedule();
      spin_lock(&hgfsBigLock);
   }

   set_current_state(TASK_RUNNING);
   HgfsRemoveClientWaitOnRequest(req);
   if (err) {
      /*
       * If we get interrupted BEFORE req has been sent (i.e. it's
       * still on the outQ) we should remove it from the outQ.
       */
      if (req->stateFile == HGFS_REQ_STATE_INACTIVE) {
         LOG(4, (KERN_DEBUG "VMware hgfs: HgfsWaitRequestReply: signal caught"
                 "before sent, removing from outQ\n"));
         list_del_init(&req->list);
      }
   }
   HgfsRestoreSignals(&sigset);
   LOG(8, (KERN_DEBUG "VMware hgfs: HgfsWaitRequestReply: request finished, %s, "
           "returning\n", err ? "interrupted" : "no error"));
   return err;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsSendRequest --
 *
 *    Sends a request to the user process and waits for the response.
 *    Returns when the request is complete or if there has been an
 *    error.
 *
 * Results:
 *    Returns zero on success, -ERESTARTSYS if interrupted (this value will
 *    be returned by HgfsWaitRequestReply), negative error
 *    otherwise. Callers use the -ERESTARTSYS return value to determine
 *    whether they should free the request object before exiting.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static int
HgfsSendRequest(HgfsSuperInfo *si,  // IN/OUT: Superblock info struct
                HgfsReq *req)       // IN/OUT: Outgoing request
{
   int error = 0;

   ASSERT(si);
   ASSERT(req);

   if ((!si) || (!req)) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsSendRequest: NULL input\n"));
      return -EINVAL;
   }

   ASSERT(req->packetSize <= HGFS_PACKET_MAX);

   /*
    * Done with the lock held:
    *
    *   - Add this request to the outQ
    *   - wake up the server process
    *   - wait until we get a response
    *
    *    (HgfsWaitRequestReply will deal with the lock correctly, and
    *     HgfsDevRead will take us off the outQ when we're sent.)
    */

   spin_lock(&hgfsBigLock);
   error = -ENOTCONN;
   if (si->userFile) {
      list_add_tail(&req->list, &si->outQ);
      wake_up(&si->waitQ);
      error = HgfsWaitRequestReply(req);
   } else {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsSendRequest: userFile is NULL!!\n"));
   }
   spin_unlock(&hgfsBigLock);

   /*
    * HgfsDevWrite should have set our state to "finished", unless
    * we got interrupted before the reply came.
    */
   if (req->stateFile != HGFS_REQ_STATE_FINISHED) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsSendRequest: WRONG STATE in request: %lu\n",
             (unsigned long)req->stateFile));
   }

   return error;
}


/*
 * Dentry operations
 */


/*
 *----------------------------------------------------------------------
 *
 * HgfsDentryRevalidate --
 *
 *    Called by namei.c every time a dentry is looked up in the dcache
 *    to determine if it is still valid.
 *
 *    If the entry is found to be invalid, namei calls dput on it and
 *    returns NULL, which causes a new lookup to be done in the actual
 *    filesystem, which in our case means that HgfsLookup is called.
 *
 * Results:
 *    Positive value if the entry IS valid.
 *    Zero if the entry is NOT valid.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 5, 75)
static int
HgfsDentryRevalidate(struct dentry *dentry,  // IN: Dentry to revalidate
                     struct nameidata *nd)   // IN: Lookup flags & intent
#else
static int
HgfsDentryRevalidate(struct dentry *dentry,  // IN: Dentry to revalidate
                     int flags)              // IN: Lookup flags (e.g. LOOKUP_CONTINUE)
#endif
{
   int error;
   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsDentryRevalidate: calling HgfsRevalidate\n"));

   ASSERT(dentry);

   /* Just call HgfsRevaliate, which does the right thing. */
   error = HgfsRevalidate(dentry);
   if (error) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDentryRevalidate: invalid\n"));
      return 0;
   }

   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsDentryRevalidate: valid\n"));
   return 1;
}

static struct dentry_operations HgfsDentryOperations = {
   d_revalidate:   HgfsDentryRevalidate,
};


/*
 * Inode operations
 */


/*
 *----------------------------------------------------------------------
 *
 * HgfsPrivateGetattr --
 *
 *    Internal getattr routine. Send a getattr request to the server
 *    for the indicated remote name, and if it succeeds copy the
 *    results of the getattr into the provided HgfsAttr.
 *
 * Results:
 *    Returns zero on success, or a negative error on failure.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static int
HgfsPrivateGetattr(struct super_block *sb, // IN: Superblock of this fs
                   char const *name,       // IN: Name to lookup
                   HgfsAttr *attr)         // OUT: Attr to copy into
{
   struct HgfsSuperInfo *si;
   HgfsReq *req;
   HgfsRequestGetattr *request;
   HgfsReplyGetattr *reply;
   int result = 0;

   ASSERT(sb);
   ASSERT(name);
   ASSERT(attr);

   si = HGFS_SB_TO_COMMON(sb);

   req = HgfsGetNewRequest();
   if (!req) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsPrivateGetattr: interrupted "
              "waiting for request\n"));
      return -ERESTARTSYS;
   }

   request = (HgfsRequestGetattr *)(req->packet);

   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsPrivateGetattr: getting attrs "
           "for \"%s\"\n", name));

   /* Fill out the request packet. */
   request->header.id = req->id;
   request->header.op = HGFS_OP_GETATTR;
   /* Convert to CP name. */
   result = CPName_ConvertTo(name, PATH_MAX, request->fileName.name);
   if (result < 0) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsPrivateGetattr: CP conversion failed\n"));
      return -EINVAL;
   }
   /* Unescape the CP name. */
   result = HgfsUnescapeBuffer(request->fileName.name, result);
   request->fileName.length = result;
   req->packetSize = sizeof *request + result;
   req->error = 0;

   result = HgfsSendRequest(si, req);
   if ((!result) && (req->stateFile == HGFS_REQ_STATE_FINISHED)) {
      if (req->error) {
         LOG(4, (KERN_DEBUG "VMware hgfs: HgfsPrivateGetattr: server "
                 "returned error: %d\n", req->error));
         result = req->error;
      } else {
         LOG(6, (KERN_DEBUG "VMware hgfs: HgfsPrivateGetattr: got reply\n"));
         reply = (HgfsReplyGetattr *)(req->packet);

         /*
          * If the getattr succeeded on the server, copy the stats
          * into the HgfsAttr, otherwise return an error.
          */
         switch (reply->header.status) {
         case HGFS_STATUS_SUCCESS:
            if (req->packetSize != sizeof *reply) {
               /*
                * If status != success, the packetSize will be smaller,
                * so this test only applies in the success case. [bac]
                */
               LOG(4, (KERN_DEBUG "VMware hgfs: HgfsPrivateGetattr: wrong packet size\n"));
               result = -EPROTO;
               break;
            }
            memcpy(attr, &reply->attr, sizeof *attr);
            result = 0;
            break;

         case HGFS_STATUS_NO_SUCH_FILE_OR_DIR:
            result = -ENOENT;
            break;

         case HGFS_STATUS_OPERATION_NOT_PERMITTED:
            result = -EPERM;
            break;

         default:
            result = -EPROTO;
            break;
         }
      }
   }

   /*
    * See if request needs freeing and free it if necessary.
    */
   HgfsTestAndFreeRequest(result, req);

   return result;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsCreate --
 *
 *    Create inode for a new file. Called directly by vfs_create,
 *    which is called by open_namei (both in fs/namei.c), as a result
 *    of someone doing a creat(2) or an open(2) with O_CREAT.
 *
 *    This gets called BEFORE f_op->open is called, so the file on the
 *    remote end has not been created yet when we get here. So, we
 *    just cheat and create a reasonable looking inode and instantiate
 *    it. When this returns, our open routine will get called, which
 *    will create the actual file on the server. If that fails for
 *    some reason, dentry_open (which calls f_op->open) will cleanup
 *    things and fput the dentry.
 *
 *    XXX This shares a lot of code with HgfsLookup; could consolidate?
 *
 * Results:
 *    Returns zero on success, negative error on failure.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 5, 75)
static int
HgfsCreate(struct inode *dir,     // IN: Parent dir to create in
           struct dentry *dentry, // IN: Dentry containing name to create
           int mode,              // IN: Mode of file to be created
	   struct nameidata *nd)  // IN: Intent, vfsmount, ...
#else
static int
HgfsCreate(struct inode *dir,     // IN: Parent dir to create in
           struct dentry *dentry, // IN: Dentry containing name to create
           int mode)              // IN: Mode of file to be created
#endif
{
   HgfsAttr attr;
   char *remoteName;
   struct inode *inode;
   ino_t ino;
   int error;

   ASSERT(dir);
   ASSERT(dentry);

   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsCreate: parent name \"%s\", entry \"%s\"\n",
          INODE_GET_II_P(dir)->name, dentry->d_name.name));

   /*
    * Create the full name.
    */
   remoteName = HgfsMakeFullName(INODE_GET_II_P(dir)->name,
                                 dentry->d_name.name);
   if (!remoteName) {
      return -ENOMEM;
   }
   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsCreate: full name \"%s\"\n",
          remoteName));

   /* Create appropriate attrs for this file. */
   memset(&attr, 0, sizeof attr);
   attr.type = HGFS_FILE_TYPE_REGULAR;
   attr.size = 0; /* just to be explicit */
   attr.permissions = (mode & S_IRWXU) >> 6;

   /* Get the next available inode number. */
   ino = HgfsGetNextIno(dir->i_sb);

   /* Create the inode and check it. */
   inode = HgfsIget(dir->i_sb, ino, &attr, remoteName);
   if (!inode) {
      error = -ENOMEM;
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsCreate: no mem getting inode\n"));
      goto out;
   }

   dentry->d_time = jiffies;
   dentry->d_op = &HgfsDentryOperations;
   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsCreate: instantiating inode\n"));
   d_instantiate(dentry, inode);
   error = 0;

out:
   kfree(remoteName);
   return error;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsLookup --
 *
 *    Lookup a file in a directory.
 *
 *    We do a getattr to see if the file exists on the server, and if
 *    so we create a new inode and fill in the fields appropriately by
 *    calling HgfsIget with the results of the getattr, and then
 *    call d_add with the new dentry.
 *
 *    For the curious, the way lookup in linux works (see fs/namei.c)
 *    is roughly as follows: first a d_lookup is done to see if there
 *    is an appropriate entry in the dcache already. If there is, it
 *    is revalidated by calling d_op->d_revalidate, which calls our
 *    HgfsDentryRevalidate (see above). If there is no dentry in the
 *    cache or if the dentry is no longer valid, then namei calls
 *    i_op->lookup, which calls HgfsLookup.
 *
 * Results:
 *    Returns NULL on success, negative error on failure.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 5, 75)
static struct dentry *
HgfsLookup(struct inode * dir,      // IN: Inode of parent directory
           struct dentry * dentry,  // IN: Dentry containing name to look up
	   struct nameidata * nd)   // IN: Intent, vfsmount, ...
#else
static struct dentry *
HgfsLookup(struct inode * dir,      // IN: Inode of parent directory
           struct dentry * dentry)  // IN: Dentry containing name to look up
#endif
{
   HgfsAttr attr;
   char *remoteName;
   struct inode *inode;
   ino_t ino;
   int error = 0;

   ASSERT(dir);
   ASSERT(dentry);

   if (!dir || !dentry) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsLookup: NULL input\n"));
      return ERR_PTR(-EFAULT);
   }

   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsLookup: dir ino %lu, i_dev %u\n",
          dir->i_ino, dir->i_sb->s_dev));
   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsLookup: dir name is \"%s\", entry name \"%s\"\n",
          INODE_GET_II_P(dir)->name, dentry->d_name.name));

   /*
    * Here we create the name we want to look up: the parent inode's
    * name plus "/" plus the dentry name.
    */
   remoteName = HgfsMakeFullName(INODE_GET_II_P(dir)->name,
                                 dentry->d_name.name);
   if (!remoteName) {
      return ERR_PTR(-ENOMEM);
   }
   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsLookup: looking up \"%s\"\n",
          remoteName));

   /*
    * Do a getattr on the file to see if it exists on the server.
    */
   inode = NULL;
   error = HgfsPrivateGetattr(dir->i_sb, remoteName, &attr);
   if (!error) {
      /* File exists on the server. */

      /* Get the next available inode number. */
      ino = HgfsGetNextIno(dir->i_sb);

      /*
       * Get the inode with this inode number, remotename, and the attrs we
       * got from the server.
       */
      inode = HgfsIget(dir->i_sb, ino, &attr, remoteName);
      if (!inode) {
         error = -ENOMEM;
         LOG(4, (KERN_DEBUG "VMware hgfs: HgfsLookup: no mem getting inode\n"));
         goto error;
      }
   } else if (error != -ENOENT) {
      /*
       * Either the file doesn't exist or there was a more serious
       * error; if it's the former, it's okay, we just do nothing.
       */
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsLookup: error other than ENOENT\n"));
      goto error;
   }

   /*
    * Set the dentry's time to NOW, set its operations pointer, add it
    * and the new (possibly NULL) inode to the dcache.
    */
   dentry->d_time = jiffies;
   dentry->d_op = &HgfsDentryOperations;
   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsLookup: adding new entry\n"));
   d_add(dentry, inode);

   kfree(remoteName);
   return NULL;

error:
   kfree(remoteName);
   return ERR_PTR(error);
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsMkdir --
 *
 *    Handle a mkdir request
 *
 *    XXX details?
 *
 * Results:
 *    Returns zero on success, or a negative error on failure.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

int
HgfsMkdir(struct inode *dir,     // IN: Inode of parent directory
          struct dentry *dentry, // IN: Dentry with name to be created
          int mode)              // IN: Mode of dir to be created
{
   struct HgfsSuperInfo *si;
   HgfsReq *req;
   HgfsRequestCreateDir *request;
   HgfsReplyCreateDir *reply;
   char *remoteName = NULL;
   int result = 0;

   ASSERT(dir);
   ASSERT(dentry);

   /*
    * Create full name.
    */
   remoteName = HgfsMakeFullName(INODE_GET_II_P(dir)->name,
                                 dentry->d_name.name);
   if (!remoteName) {
      return -ENOMEM;
   }

   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsMkdir: making \"%s\", mode %o\n",
          remoteName, (mode & S_IRWXU) >> 6));

   si = HGFS_SB_TO_COMMON(dir->i_sb);

   req = HgfsGetNewRequest();
   if (!req) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsMkdir: interrupted waiting for request\n"));
      return -ERESTARTSYS;
   }

   request = (HgfsRequestCreateDir *)(req->packet);

   request->header.id = req->id;
   request->header.op = HGFS_OP_CREATE_DIR;
   request->permissions = (mode & S_IRWXU) >> 6;
   /* Convert to CP name. */
   result = CPName_ConvertTo(remoteName, PATH_MAX, request->fileName.name);
   if (result < 0) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsMkdir: CP conversion failed\n"));
      result = -EINVAL;
      goto out;
   }
   /* Unescape the CP name. */
   result = HgfsUnescapeBuffer(request->fileName.name, result);
   request->fileName.length = result;
   req->packetSize = sizeof *request + result;
   req->error = 0;

   result = HgfsSendRequest(si, req);
   if ((!result) && (req->stateFile == HGFS_REQ_STATE_FINISHED)) {
      if (req->error) {
         LOG(4, (KERN_DEBUG "VMware hgfs: HgfsMkdir: server returned error: %d\n",
                req->error));
         result = req->error;
      } else if (req->packetSize != sizeof *reply) {
         /* This packet size should never vary. */
         LOG(4, (KERN_DEBUG "VMware hgfs: HgfsMkdir: wrong packet size\n"));
         result = -EPROTO;
      } else {
         LOG(6, (KERN_DEBUG "VMware hgfs: HgfsMkdir: got reply\n"));
         reply = (HgfsReplyCreateDir *)(req->packet);

         /*
          * Return appropriate error code.
          */
         switch (reply->header.status) {
         case HGFS_STATUS_SUCCESS:
            result = 0;
            break;
         case HGFS_STATUS_OPERATION_NOT_PERMITTED:
            result = -EPERM;
            break;
         case HGFS_STATUS_FILE_EXISTS:
            result = -EEXIST;
            break;

         default:
            result = -EPROTO;
            break;
         }
      }
   }

out:
   kfree(remoteName);

   /*
    * See if request needs freeing and free it if necessary.
    */
   HgfsTestAndFreeRequest(result, req);

   return result;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsDelete --
 *
 *    Handle both unlink and rmdir requests.
 *
 *    XXX details?
 *
 * Results:
 *    Returns zero on success, or a negative error on failure.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static int
HgfsDelete(struct inode * dir,     // IN: Parent dir of file/dir to delete
           struct dentry * dentry, // IN: Dentry of file/dir to delete
           HgfsOp op)              // IN: Opcode for file type (file or dir)
{
   struct HgfsSuperInfo *si;
   HgfsReq *req;
   HgfsRequestDelete *request;
   HgfsReplyDelete *reply;
   char *remoteName = NULL;
   int result = 0;

   ASSERT(dir);
   ASSERT(dentry);

   if (!dir || !dentry) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDelete: NULL input\n"));
      return -EFAULT;
   }

   /* Check opcode. */
   if ((op != HGFS_OP_DELETE_FILE) &&
      (op != HGFS_OP_DELETE_DIR)) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDelete: Invalid opcode\n"));
      return -EINVAL;
   }

   remoteName = HgfsMakeFullName(INODE_GET_II_P(dir)->name,
                                 dentry->d_name.name);
   if (!remoteName) {
      return -ENOMEM;
   }

   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsDelete: deleting \"%s\", op %u\n",
          remoteName, op));

   si = HGFS_SB_TO_COMMON(dir->i_sb);

   req = HgfsGetNewRequest();
   if (!req) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDelete: interrupted waiting for request\n"));
      kfree(remoteName);
      return -ERESTARTSYS;
   }

   request = (HgfsRequestDelete *)(req->packet);

   request->header.id = req->id;
   request->header.op = op;
   /* Convert to CP name. */
   result = CPName_ConvertTo(remoteName, PATH_MAX, request->fileName.name);
   if (result < 0) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDelete: CP conversion failed\n"));
      result = -EINVAL;
      goto out;
   }
   /* Unescape the CP name. */
   result = HgfsUnescapeBuffer(request->fileName.name, result);
   request->fileName.length = result;
   req->packetSize = sizeof *request + result;
   req->error = 0;

   result = HgfsSendRequest(si, req);
   if ((!result) && (req->stateFile == HGFS_REQ_STATE_FINISHED)) {
      if (req->error) {
         LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDelete: server returned error: %d\n",
                req->error));
         result = req->error;
      } else if (req->packetSize != sizeof *reply) {
         /* This packet size should never vary. */
         LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDelete: wrong packet size\n"));
         result = -EPROTO;
      } else {
         LOG(6, (KERN_DEBUG "VMware hgfs: HgfsDelete: got reply\n"));
         reply = (HgfsReplyDelete *)(req->packet);

         /*
          * Return appropriate error code.
          */
         switch (reply->header.status) {
         case HGFS_STATUS_SUCCESS:
            result = 0;
            break;
         case HGFS_STATUS_OPERATION_NOT_PERMITTED:
            result = -EPERM;
            break;
         case HGFS_STATUS_NOT_DIRECTORY:
            result = -ENOTDIR;
            break;
         case HGFS_STATUS_DIR_NOT_EMPTY:
            result = -ENOTEMPTY;
            break;
         case HGFS_STATUS_NO_SUCH_FILE_OR_DIR:
            result = -ENOENT;
            break;

         default:
            result = -EPROTO;
            break;
         }
      }
   }

out:
   kfree(remoteName);

   /*
    * See if request needs freeing and free it if necessary.
    */
   HgfsTestAndFreeRequest(result, req);

   return result;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsRmdir --
 *
 *    Handle an rmdir request. Just calls HgfsDelete with the
 *    correct opcode.
 *
 * Results:
 *    Returns zero on success, or a negative error on failure.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

int
HgfsRmdir(struct inode * dir,      // IN: Parent dir of dir to remove
          struct dentry * dentry)  // IN: Dentry of dir to remove
{
   LOG(8, (KERN_DEBUG "VMware hgfs: HgfsRmdir: was called\n"));

   return HgfsDelete(dir, dentry, HGFS_OP_DELETE_DIR);

   /*
    * XXXX Uhhhh... shouldn't we iput the file's inode?
    * Also do we need to dec_count the dir?
    */
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsUnlink --
 *
 *    Handle an unlink request. Just calls HgfsDelete with the
 *    correct opcode.
 *
 * Results:
 *    Returns zero on success, or a negative error on failure.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

int
HgfsUnlink(struct inode * dir,      // IN: Parent dir of file to unlink
           struct dentry * dentry)  // IN: Dentry of file to unlink
{
   LOG(8, (KERN_DEBUG "VMware hgfs: HgfsUnlink: was called\n"));

   return HgfsDelete(dir, dentry, HGFS_OP_DELETE_FILE);

   /* XXX should we iput dentry->d_inode ? */
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsRename --
 *
 *    Handle rename requests.
 *
 *    XXX details?
 *
 * Results:
 *    Returns zero on success, or a negative error on failure.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

int
HgfsRename(struct inode *oldDir,      // IN: Inode of original directory
           struct dentry *oldDentry,  // IN: Dentry of file to rename
           struct inode *newDir,      // IN: Inode of new directory
           struct dentry *newDentry)  // IN: Dentry containing new name
{
   struct HgfsSuperInfo *si;
   HgfsReq *req;
   HgfsRequestRename *request;
   HgfsReplyRename *reply;
   char *oldRemoteName = NULL;
   char *newRemoteName = NULL;
   HgfsFileName *newNameP = NULL;
   int result = 0;

   ASSERT(oldDir);
   ASSERT(oldDentry);
   ASSERT(newDir);
   ASSERT(newDentry);

   if (!oldDir || !oldDentry || !newDir || !newDentry) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsRename: NULL input\n"));
      return -EFAULT;
   }

   oldRemoteName = HgfsMakeFullName(INODE_GET_II_P(oldDir)->name,
                                    oldDentry->d_name.name);
   newRemoteName = HgfsMakeFullName(INODE_GET_II_P(newDir)->name,
                                    newDentry->d_name.name);
   if (!oldRemoteName || !newRemoteName) {
      return -ENOMEM;
   }

   /* Make sure we can fit the filenames into a request. */
   if ((sizeof(HgfsRequestDelete) + strlen(oldRemoteName) +  strlen(newRemoteName)) >
      HGFS_PACKET_MAX) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsRename: names too big for one request\n"));
      return -EPROTO;
   }

   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsRename: renaming \"%s\" to \"%s\"\n",
          oldRemoteName, newRemoteName));

   si = HGFS_SB_TO_COMMON(oldDir->i_sb);

   req = HgfsGetNewRequest();
   if (!req) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsRename: interrupted waiting for request\n"));
      kfree(oldRemoteName);
      kfree(newRemoteName);
      return -ERESTARTSYS;
   }

   request = (HgfsRequestRename *)(req->packet);

   request->header.id = req->id;
   request->header.op = HGFS_OP_RENAME;
   /* Convert old name to CP format. */
   result = CPName_ConvertTo(oldRemoteName, PATH_MAX, request->oldName.name);
   if (result < 0) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsRename: oldName CP conversion failed\n"));
      result = -EINVAL;
      goto out;
   }
   /* Unescape the old CP name. */
   result = HgfsUnescapeBuffer(request->oldName.name, result);
   request->oldName.length = result;
   req->packetSize = sizeof *request + result;
   /* Convert new name to CP format. */
   newNameP = (HgfsFileName*)((char*)&request->oldName + sizeof(HgfsFileName) + result);
   result = CPName_ConvertTo(newRemoteName, PATH_MAX, newNameP->name);
   if (result < 0) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsRename: newName CP conversion failed\n"));
      result = -EINVAL;
      goto out;
   }
   /* Unescape the new CP name. */
   result = HgfsUnescapeBuffer(newNameP->name, result);
   newNameP->length = result;
   req->packetSize += result;
   req->error = 0;

   result = HgfsSendRequest(si, req);
   if ((!result) && (req->stateFile == HGFS_REQ_STATE_FINISHED)) {
      if (req->error) {
         LOG(4, (KERN_DEBUG "VMware hgfs: HgfsRename: server returned error: %d\n",
                req->error));
         result = req->error;
      } else if (req->packetSize != sizeof *reply) {
         /* This packet size should never vary. */
         LOG(4, (KERN_DEBUG "VMware hgfs: HgfsRename: wrong packet size\n"));
         result = -EPROTO;
      } else {
         LOG(6, (KERN_DEBUG "VMware hgfs: HgfsRename: got reply\n"));
         reply = (HgfsReplyRename *)(req->packet);

         /*
          * Return appropriate error code.
          */
         switch(reply->header.status) {
         case HGFS_STATUS_SUCCESS:
            result = 0;
            break;
         case HGFS_STATUS_OPERATION_NOT_PERMITTED:
            result = -EPERM;
            break;
         case HGFS_STATUS_NOT_DIRECTORY:
            result = -ENOTDIR;
            break;
         case HGFS_STATUS_DIR_NOT_EMPTY:
            result = -ENOTEMPTY;
            break;
         case HGFS_STATUS_NO_SUCH_FILE_OR_DIR:
            result = -ENOENT;
            break;

         default:
            result = -EPROTO;
            break;
         }
      }
   }

out:
   kfree(oldRemoteName);
   kfree(newRemoteName);

   /*
    * See if request needs freeing and free it if necessary.
    */
   HgfsTestAndFreeRequest(result, req);

   return result;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsTruncate --
 *
 *    Handle a truncate request
 *
 *    XXX details?
 *
 * Results:
 *    None
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

void
HgfsTruncate(struct inode * inode) // IN
{
   LOG(4, (KERN_DEBUG "VMware hgfs: HgfsTruncate: was called\n"));
   // XXX NOT_IMPLEMENTED();
   return;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsPermission --
 *
 *    Test access permissions on a file.
 *
 *    XXX This is unimplemented; we currently always claim that
 *    access is allowed.
 *
 * Results:
 *    Zero if access is allowed, negative error if not.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 5, 75)
static int
HgfsPermission(struct inode * inode, // IN: Inode to check perms on
               int mask,             // IN: Permissions to check
	       struct nameidata *nd) // IN: Dentry...
#else
static int
HgfsPermission(struct inode * inode, // IN: Inode to check perms on
               int mask)             // IN: Permissions to check
#endif
{
   ASSERT(inode);

   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsPermission: get perm for \"%s\", mask %o\n",
          INODE_GET_II_P(inode)->name, mask));

   /*
    * XXX we should be calling vfs_permission(inode, mask), plus
    * possibly calling getattr on failure, since the root inode will
    * start out with no permissions until we've looked them up.
    */

   return 0;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsRevalidate --
 *
 *    Called when the kernel wants to check that an inode is still
 *    valid. Called with the dentry that points to the inode we're
 *    interested in.
 *
 *    We call HgfsPrivateGetattr with the inode's remote name, and if
 *    it succeeds we update the inode's attributes and return zero
 *    (success). Otherwise, we return an error.
 *
 * Results:
 *    Returns zero if inode is valid, negative error if not.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static int
HgfsRevalidate(struct dentry *dentry)  // IN: Dentry to revalidate
{
   HgfsAttr attr;
   int error = 0;

   ASSERT(dentry);

   if (!dentry || !dentry->d_inode) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsRevalidate: null input\n"));
      return -EINVAL;
   }

   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsRevalidate: name %s, inum %lu, remotename %s\n",
          dentry->d_name.name,
          dentry->d_inode->i_ino,
          INODE_GET_II_P(dentry->d_inode)->name));

   if (dentry == dentry->d_inode->i_sb->s_root) {
      /*
       * Don't bother revalidating the root dentry; it doesn't actually
       * exist on the host anyway, and will get bogus attributes if we
       * ask the server.
       */
      LOG(6, (KERN_DEBUG "VMware hgfs: HgfsRevalidate: skipping root dentry\n"));
      /* 
       * Reason to clear S_IWUGO is to prevent processes from thinking 
       * they can remove directories from the root directory is fine. 
       * Please refer to bug 67955
       */
      dentry->d_inode->i_mode &= ~S_IWUGO;

      return 0;
   }

   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsRevalidate: calling HgfsGetattr\n"));
   error = HgfsPrivateGetattr(dentry->d_sb,
                              INODE_GET_II_P(dentry->d_inode)->name,
                              &attr);
   if (!error) {
      /* No error, so update inode's attributes. */
      HgfsChangeFileAttributes(dentry->d_inode, &attr);
   }
   return error;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsSetattrCopy --
 *
 *    See if we need to update any attributes, and if so copy them
 *    into the HgfsAttr.
 *
 * Results:
 *    TRUE if attributes need updating, FALSE if not.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

Bool
HgfsSetattrCopy(struct iattr *iattr,      // IN:  Inode attrs to update from
                HgfsAttr *attr,           // OUT: Hgfs attrs to send to server
                HgfsAttrChanges *update)  // OUT: Which attr fields to update
{
   Bool result = FALSE;
   unsigned int valid;

   ASSERT(iattr);
   ASSERT(attr);
   ASSERT(update);

   valid = iattr->ia_valid;
   /*
    * Clear all the attributes before setting some of them.
    */
   memset(attr, 0, sizeof *attr);

   /*
    * We only support changing these attributes:
    * - mode bits (i.e. permissions)
    * - size
    * - access/write times
    */

   if (valid & ATTR_MODE) {
      *update |= HGFS_ATTR_PERMISSIONS;
      attr->permissions = ((iattr->ia_mode & S_IRWXU) >> 6);
      result = TRUE;
   }

   if (valid & ATTR_SIZE) {
      *update |= HGFS_ATTR_SIZE;
      attr->size = iattr->ia_size;
      result = TRUE;
   }

   if (valid & ATTR_ATIME) {
      *update |= HGFS_ATTR_ACCESS_TIME;
      attr->accessTime = HgfsConvertToNtTime(HGFS_GET_TIME(iattr->ia_atime));
      result = TRUE;
   }

   if (valid & ATTR_MTIME) {
      *update |= HGFS_ATTR_WRITE_TIME;
      attr->writeTime = HgfsConvertToNtTime(HGFS_GET_TIME(iattr->ia_mtime));
      result = TRUE;
   }

   return result;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsSetattr --
 *
 *    Handle a setattr request. Call HgfsSetattrCopy to determine
 *    which fields need updating and convert them to the HgfsAttr
 *    format, then send the request to the server.
 *
 * Results:
 *    Returns zero on success, or a negative error on failure.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

int
HgfsSetattr(struct dentry *dentry,  // IN: File to set attributes of
            struct iattr *iattr)    // IN: Attributes to set
{
   struct HgfsSuperInfo *si;
   HgfsReq *req;
   HgfsRequestSetattr *request;
   HgfsReplySetattr *reply;
   int result = 0;

   ASSERT(dentry);
   ASSERT(iattr);

   si = HGFS_SB_TO_COMMON(dentry->d_sb);

   req = HgfsGetNewRequest();
   if (!req) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsSetattr: interrupted waiting for request\n"));
      return -ERESTARTSYS;
   }

   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsSetattr: setting attributes of \"%s\"\n",
          INODE_GET_II_P(dentry->d_inode)->name));

   request = (HgfsRequestSetattr *)(req->packet);

   /*
    * Make sure we actually need to send the request. If not, release
    * the request and return.
    */
   if (HgfsSetattrCopy(iattr, &request->attr, &request->update) == FALSE) {
      HgfsTestAndFreeRequest(0, req);
      return 0;
   }

   /* Fill in the request's fields. */
   request->header.id = req->id;
   request->header.op = HGFS_OP_SETATTR;
   /* Convert to CP name. */
   result = CPName_ConvertTo(INODE_GET_II_P(dentry->d_inode)->name,
                             PATH_MAX, request->fileName.name);
   if (result < 0) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsSetattr: CP conversion failed\n"));
      return -EINVAL;
   }
   /* Unescape the CP name. */
   result = HgfsUnescapeBuffer(request->fileName.name, result);
   request->fileName.length = result;
   req->packetSize = sizeof *request + result;
   req->error = 0;

   /* Send the request and process the reply. */
   result = HgfsSendRequest(si, req);
   if ((!result) && (req->stateFile == HGFS_REQ_STATE_FINISHED)) {
      if (req->error) {
         LOG(4, (KERN_DEBUG "VMware hgfs: HgfsSetattr: server returned error: %d\n",
                req->error));
         result = req->error;
         goto out;
      } else {
         /* Get the reply. */
         reply = (HgfsReplySetattr *)(req->packet);

         /* Return appropriate status. */
         switch (reply->header.status) {
         case HGFS_STATUS_SUCCESS:
            result = 0;
            break;
         case HGFS_STATUS_NO_SUCH_FILE_OR_DIR:
            result = -ENOENT;
            break;
         case HGFS_STATUS_OPERATION_NOT_PERMITTED:
            result = -EPERM;
            break;

         default:
            result = -EPROTO;
            break;
         }
      }
   }

out:
   /*
    * See if request needs freeing and free it if necessary.
    */
   HgfsTestAndFreeRequest(result, req);

   return result;
}


/*
 * The inode_operations structure changed in 2.5.18:
 * before:
 * . 'getattr' was defined but unused
 * . 'revalidate' was defined and used
 * after:
 * 1) 'getattr' changed and became used
 * 2) 'revalidate' was removed
 *
 * Note: Mandrake backported 1) but not 2) starting with 2.4.8-26mdk
 *
 *   --hpreg
 */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 5, 18)
#   define HGFS_GETATTR_ONLY 1
#else
#   undef HGFS_GETATTR_ONLY
#endif


#ifdef HGFS_GETATTR_ONLY
/*
 *-----------------------------------------------------------------------------
 *
 * HgfsGetattr --
 *
 *    Hgfs superblock 'getattr' method.
 *
 * Results:
 *    0 on success
 *    error < 0 on failure
 *
 * Side effects:
 *    None
 *
 *-----------------------------------------------------------------------------
 */

int
HgfsGetattr(struct vfsmount *mnt,  // Unused
            struct dentry *dentry, // IN
            struct kstat *stat)    // OUT
{
   int err;

   // XXX ASSERT(mnt); ? --hpreg
   ASSERT(dentry);
   ASSERT(stat);

   err = HgfsRevalidate(dentry);
   if (err) {
      return err;
   }

   /* Convert stats from the VFS inode format to the kernel format --hpreg */
   generic_fillattr(dentry->d_inode, stat);
   // XXX Should we set stat->blocks and stat->blksize? --hpreg

   return 0;
}
#endif


static struct inode_operations HgfsDirInodeOperations =
{
   /* Optional */
   create:       HgfsCreate,

   /* Optional */
   mkdir:        HgfsMkdir,

   lookup:       HgfsLookup,
   rmdir:        HgfsRmdir,
   unlink:       HgfsUnlink,
   rename:       HgfsRename,
//   truncate:     HgfsTruncate,  // probably not needed
   permission:   HgfsPermission,
   setattr:      HgfsSetattr,

#ifdef HGFS_GETATTR_ONLY
   /* Optional */
   getattr:      HgfsGetattr,
#else
   /* Optional */
   revalidate:   HgfsRevalidate,
#endif
};


static struct inode_operations HgfsFileInodeOperations =
{
//   truncate:     HgfsTruncate,  // probably not needed
   permission:   HgfsPermission,
   setattr:      HgfsSetattr,

#ifdef HGFS_GETATTR_ONLY
   /* Optional */
   getattr:      HgfsGetattr,
#else
   /* Optional */
   revalidate:   HgfsRevalidate,
#endif
};


/*
 *----------------------------------------------------------------------
 *
 * HgfsGetOpenMode --
 *
 *    Based on the flags requested by the process making the open()
 *    syscall, determine which open mode (access type) to request from
 *    the server.
 *
 * Results:
 *    Returns the correct HgfsOpenMode enumeration to send to the
 *    server, or -1 on failure.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static int
HgfsGetOpenMode(uint32 flags) // IN: Open flags
{
   uint32 mask = O_RDONLY|O_WRONLY|O_RDWR;
   int result = -1;

   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsGetOpenMode: entered\n"));


   /*
    * Mask the flags to only look at the access type.
    */
   flags &= mask;

   /* Pick the correct HgfsOpenMode. */
   switch (flags) {

   case O_RDONLY:
      result = HGFS_OPEN_MODE_READ_ONLY;
      break;

   case O_WRONLY:
      result = HGFS_OPEN_MODE_WRITE_ONLY;
      break;

   case O_RDWR:
      result = HGFS_OPEN_MODE_READ_WRITE;
      break;

   default:
      /*
       * This should never happen, but it could if a userlevel program
       * is behaving poorly.
       */
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsGetOpenMode: invalid open flags %o\n",
             flags));
      result = -1;
      break;
   }

   return result;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsGetOpenFlags --
 *
 *    Based on the flags requested by the process making the open()
 *    syscall, determine which flags to send to the server to open the
 *    file.
 *
 * Results:
 *    Returns the correct HgfsOpenFlags enumeration to send to the
 *    server, or -1 on failure.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static int
HgfsGetOpenFlags(uint32 flags) // IN: Open flags
{
   uint32 mask = O_CREAT | O_TRUNC | O_EXCL;
   int result = -1;

   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsGetOpenFlags: entered\n"));

   /*
    * Mask the flags to only look at O_CREAT, O_EXCL, and O_TRUNC.
    */

   flags &= mask;

   /* O_EXCL has no meaning if O_CREAT is not set. */
   if (!(flags & O_CREAT)) {
      flags &= ~O_EXCL;
   }

   /* Pick the right HgfsOpenFlags. */
   switch (flags) {

   case 0:
      /* Regular open; fails if file nonexistant. */
      result = HGFS_OPEN;
      break;

   case O_CREAT:
      /* Create file; if it exists already just open it. */
      result = HGFS_OPEN_CREATE;
      break;

   case O_TRUNC:
      /* Truncate existing file; fails if nonexistant. */
      result = HGFS_OPEN_EMPTY;
      break;

   case (O_CREAT | O_EXCL):
      /* Create file; fail if it exists already. */
      result = HGFS_OPEN_CREATE_SAFE;
      break;

   case (O_CREAT | O_TRUNC):
      /* Create file; if it exists already, truncate it. */
      result = HGFS_OPEN_CREATE_EMPTY;
      break;

   default:
      /*
       * This can only happen if all three flags are set, which
       * conceptually makes no sense because O_EXCL and O_TRUNC are
       * mutually exclusive if O_CREAT is set.
       *
       * However, the open(2) man page doesn't say you can't set all
       * three flags, and certain apps (*cough* Nautilus *cough*) do
       * so. To be friendly to those apps, we just silenty drop the
       * O_TRUNC flag on the assumption that it's safer to honor
       * O_EXCL.
       */
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsGetOpenFlags: invalid open flags %o. "
              "Ignoring the O_TRUNC flag.\n", flags));
      result = HGFS_OPEN_CREATE_SAFE;
      break;
   }

   return result;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsOpen --
 *
 *    Called whenever a process opens a file in our filesystem.
 *
 *    We send an "Open" request to the server with the name stored in
 *    this file's inode. If the Open succeeds, we store the filehandle
 *    sent by the server in the file struct so it can be accessed by
 *    read/write/close.
 *
 * Results:
 *    Returns zero if on success, error on failure.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static int
HgfsOpen(struct inode *inode,  // IN: Inode of the file to open
         struct file *file)    // IN: File pointer for this open
{
   HgfsSuperInfo *si;
   HgfsReq *req;
   HgfsRequestOpen *request;
   HgfsReplyOpen *reply;
   int result = 0;

   ASSERT(inode);
   ASSERT(file);

   si = HGFS_SB_TO_COMMON(inode->i_sb);

   req = HgfsGetNewRequest();
   if (!req) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsOpen: interrupted waiting for request\n"));
      return -ERESTARTSYS;
   }

   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsOpen: opening \"%s\", flags %o, "
           "create perms %o\n", INODE_GET_II_P(inode)->name,
           file->f_flags, file->f_mode));

   request = (HgfsRequestOpen *)(req->packet);

   /* Fill out the request's fields. */
   request->header.id = req->id;
   request->header.op = HGFS_OP_OPEN;
   result = HgfsGetOpenMode(file->f_flags);
   if (result < 0) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsOpen: failed to get open mode\n"));
      result = -EINVAL;
      goto out;
   }
   request->mode = result;
   result = HgfsGetOpenFlags(file->f_flags);
   if (result < 0) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsOpen: failed to get open flags\n"));
      result = -EINVAL;
      goto out;
   }
   request->flags = result;
   request->permissions = (inode->i_mode & S_IRWXU) >> 6;
   /* Convert to CP name. */
   result = CPName_ConvertTo(INODE_GET_II_P(inode)->name, PATH_MAX, request->fileName.name);
   if (result < 0) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsOpen: CP conversion failed\n"));
      result = -EINVAL;
      goto out;
   }
   /* Unescape the CP name. */
   result = HgfsUnescapeBuffer(request->fileName.name, result);
   request->fileName.length = result;
   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsOpen: flags %o, inode mode %o, perms %u\n",
          request->flags, inode->i_mode & S_IRWXU, request->permissions));
   req->packetSize = sizeof *request + result;
   req->error = 0;

   /* Send the request and process the reply. */
   result = HgfsSendRequest(si, req);
   if ((!result) && (req->stateFile == HGFS_REQ_STATE_FINISHED)) {
      if (req->error) {
         LOG(4, (KERN_DEBUG "VMware hgfs: HgfsOpen: server returned error: %d\n",
                req->error));
         result = req->error;
         goto out;
      } else {
         /* Get the reply and check return status. */
         reply = (HgfsReplyOpen *)(req->packet);
         switch (reply->header.status) {
         case HGFS_STATUS_SUCCESS:
            if (req->packetSize != sizeof *reply) {
               /*
                * If status != success, the packetSize will be smaller,
                * so this test only applies in the success case. [bac]
                */
               LOG(4, (KERN_DEBUG "VMware hgfs: HgfsOpen: wrong packet size\n"));
               result = -EPROTO;
               goto out;
            }
            result = 0;
            break;

         case HGFS_STATUS_NO_SUCH_FILE_OR_DIR:
            result = -ENOENT;
            goto out;

         case HGFS_STATUS_FILE_EXISTS:
            result = -EEXIST;
            goto out;

         case HGFS_STATUS_ACCESS_DENIED:
            result = -EACCES;
            goto out;

         default:
            result = -EPERM;
            goto out;
         }

         /*
          * Store the handle for this open() in the file*.  This needs
          * to be freed on a close().
          */
         FILE_SET_HAND_P(file, kmalloc(sizeof(HgfsHandle), GFP_KERNEL));
         if (!FILE_GET_HAND_P(file)) {
            result = -ENOMEM;
            goto out;
         }
         FILE_GET_HAND(file) = reply->file;
         LOG(6, (KERN_DEBUG "VMware hgfs: HgfsOpen: set handle to %u\n",
                FILE_GET_HAND(file)));
      }
   }

out:
   /*
    * See if request needs freeing and free it if necessary.
    */
   HgfsTestAndFreeRequest(result, req);

   return result;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsDoRead --
 *
 *    Do one read request. Called by HgfsRead, possibly multiple times
 *    if the size of the read is too big to be handled by one server
 *    request.
 *
 *    We send a "Read" request to the server with the given handle.
 *
 *    It is assumed that this function is never called with a larger
 *    read than what can be sent in one request.
 *
 * Results:
 *    Returns the number of bytes read on success, or an error on
 *    failure.
 *
 * Side effects:
 *    copy_to_user() inside may trigger call to HgfsFileMmapNoPage.
 *
 *----------------------------------------------------------------------
 */

static int
HgfsDoRead(HgfsSuperInfo *si,  // IN:  Super info for this mount
           HgfsHandle handle,  // IN:  Handle for this file
           char *buf,          // OUT: User buffer to copy data into
           size_t count,       // IN:  Number of bytes to read
           loff_t offset)      // IN:  Offset at which to read
{
   HgfsReq *req;
   HgfsRequestRead *request;
   HgfsReplyRead *reply;
   int result = 0;

   ASSERT(si);
   ASSERT(buf);

   req = HgfsGetNewRequest();
   if (!req) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDoRead: interrupted waiting for request\n"));
      return -ERESTARTSYS;
   }

   /* Fill out the request fields. */
   request = (HgfsRequestRead *)(req->packet);
   request->header.id = req->id;
   request->header.op = HGFS_OP_READ;
   request->file = handle;
   request->offset = offset;
   request->requiredSize = count;
   req->packetSize = sizeof *request;
   req->error = 0;

   /* Send the request and process the reply. */
   result = HgfsSendRequest(si, req);
   if ((!result) && (req->stateFile == HGFS_REQ_STATE_FINISHED)) {
      if (req->error) {
         LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDoRead: server returned error: %d\n",
                req->error));
         result = req->error;
         goto out;
      } else {
         /* Get the reply. */
         reply = (HgfsReplyRead *)(req->packet);

         if (reply->header.status != HGFS_STATUS_SUCCESS) {
            LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDoRead: read failed\n"));
            result = -EPERM;
            goto out;
         }

         /* Sanity check on read size. */
         if (reply->actualSize > count) {
            LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDoRead: read too big!\n"));
            result = -EPROTO;
            goto out;
         }

         if (!reply->actualSize) {
            /* We got no bytes, so don't need to copy to user. */
            LOG(6, (KERN_DEBUG "VMware hgfs: HgfsDoRead: server returned zero\n"));
            result = reply->actualSize;
            goto out;
         }

	 /* XXX Do this copy outside of holding requestSem lock... */
         /* Copy bytes out to user. */
         if (copy_to_user(buf, reply->payload, reply->actualSize)) {
            result = -EFAULT;
            goto out;
         } else {
            /* Return result. */
            LOG(6, (KERN_DEBUG "VMware hgfs: HgfsDoRead: copied %u bytes to user\n",
                   reply->actualSize));
            result = reply->actualSize;

            goto out;
         }
      }
   }

out:
   /*
    * See if request needs freeing and free it if necessary.
    */
   HgfsTestAndFreeRequest(result, req);

   return result;
}




/*
 *----------------------------------------------------------------------
 *
 * HgfsRead --
 *
 *    Called whenever a process reads from a file in our filesystem.
 *
 *    We send "Read" requests to the server with the handle stored in
 *    this file* corresponding to the open() system call.
 *
 *    Calls HgfsDoRead once for each server read request.
 *
 *    The code to split big reads across multiple server requests is
 *    exercised by programs such as diff, which read in chunks larger
 *    than 4k bytes.
 *
 * Results:
 *    Returns the number of bytes read on success, or an error on
 *    failure.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static ssize_t
HgfsRead(struct file *file,  // IN:  File to read from
         char *buf,          // OUT: User buffer to copy data into
         size_t count,       // IN:  Number of bytes to read
         loff_t *offset)     // IN:  Offset at which to read
{
   HgfsSuperInfo *si;
   HgfsHandle handle;
   size_t actualCount = 0;         // bytes successfully read so far
   size_t remainingCount = count;  // bytes left to read
   size_t nextCount = 0;           // size of next request to server
   loff_t curOffset = *offset;     // current read offset in the file
   int result;

   ASSERT(file);
   ASSERT(buf);
   ASSERT(offset);

   handle = FILE_GET_HAND(file);
   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsRead: read fh %u, count %Zu, offset %Lu\n",
          handle, count, *offset));

   si = HGFS_SB_TO_COMMON(file->f_dentry->d_sb);

   /*
    * Call HgfsDoRead repeatedly until either
    * - Read returns an error, or
    * - Read returns 0 (end of file), or
    * - We have read the requested number of bytes.
    */
   result = 0;
   do {
      /* Call HgfsDoRead with the request. */
      nextCount = (remainingCount > HGFS_IO_MAX) ? HGFS_IO_MAX : remainingCount;
      result = HgfsDoRead(si, handle, buf, nextCount, curOffset);
      if (result < 0) {
         LOG(4, (KERN_DEBUG "VMware hgfs: HgfsRead: read error %d\n",
                result));
         goto out;
      }
      remainingCount -= result;
      actualCount += result;
      curOffset += result;
      buf += result;
   } while ((result > 0) && (remainingCount > 0));

   /* Increment file pointer */
   *offset += actualCount;
   result = actualCount;

out:
   return result;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsDoWrite --
 *
 *    Do one write request. Called by HgfsWrite, possibly multiple
 *    times if the size of the write is too big to be handled by one
 *    server request.
 *
 *    We send a "Write" request to the server with the given handle.
 *
 *    It is assumed that this function is never called with a larger
 *    write than what can be sent in one request.
 *
 * Results:
 *    Returns the number of bytes written on success, or an error on
 *    failure.
 *
 * Side effects:
 *    copy_from_user() inside may trigger call to HgfsFileMmapNoPage.
 *
 *----------------------------------------------------------------------
 */

static int
HgfsDoWrite(HgfsSuperInfo *si,  // IN: Super info for this mount
            HgfsHandle handle,  // IN: Handle for this file
            uint32 flags,       // IN: Flags for this write
            const char *buf,    // IN: User buffer where the data is
            size_t count,       // IN: Number of bytes to write
            loff_t offset)      // IN: Offset to begin writing at
{
   HgfsReq *req;
   HgfsRequestWrite *request;
   HgfsReplyWrite *reply;
   int result = 0;

   ASSERT(si);
   ASSERT(buf);

   req = HgfsGetNewRequest();
   if (!req) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDoWrite: interrupted waiting for request\n"));
      return -ERESTARTSYS;
   }

   /* Fill out the request fields. */
   request = (HgfsRequestWrite *)(req->packet);
   request->header.id = req->id;
   request->header.op = HGFS_OP_WRITE;
   request->file = handle;
   request->flags = 0;
   request->offset = offset;
   request->requiredSize = count;
   /* Copy the bytes in from the user. */
   if (copy_from_user(request->payload, buf, request->requiredSize)) {
      result = -EFAULT;
      goto out;
   }
   req->packetSize = sizeof *request + request->requiredSize - 1;
   req->error = 0;

   /*
    * Set the write flags.
    */
   if (flags & O_APPEND) {
      LOG(6, (KERN_DEBUG "VMware hgfs: HgfsDoWrite: writing in append mode\n"));
      request->flags |= HGFS_WRITE_APPEND;
   }

   /* Send the request and process the reply. */
   result = HgfsSendRequest(si, req);
   if ((!result) && (req->stateFile == HGFS_REQ_STATE_FINISHED)) {
      if (req->error) {
         LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDoWrite: server returned error: %d\n",
                req->error));
         result = req->error;
         goto out;
      } else {
         /* Get the reply. */
         reply = (HgfsReplyWrite *)(req->packet);

         if (reply->header.status != HGFS_STATUS_SUCCESS) {
            LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDoWrite: write failed\n"));
            result = -EPERM;
            goto out;
         }

         /* Return result. */
         LOG(6, (KERN_DEBUG "VMware hgfs: HgfsDoWrite: wrote %u bytes\n",
                reply->actualSize));
         result = reply->actualSize;
         goto out;
      }
   }

out:
   /*
    * See if request needs freeing and free it if necessary.
    */
   HgfsTestAndFreeRequest(result, req);

   return result;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsWrite --
 *
 *    Called whenever a process writes to a file in our filesystem.
 *
 *    We send "Write" requests to the server with the handle stored in
 *    this file* corresponding to the open() system call.
 *
 *    Calls HgfsDoWrite once for each server write request.
 *
 *    The code to split big writes across multiple server requests is
 *    exercised by programs such as emacs, which write in chunks
 *    larger than 4k bytes.
 *
 * Results:
 *    Returns the number of bytes written on success, or an error on
 *    failure.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static ssize_t
HgfsWrite(struct file *file,  // IN: File to write to
          const char *buf,    // IN: User buffer where the data is
          size_t count,       // IN: Number of bytes to write
          loff_t *offset)     // IN: Offset to begin writing at
{
   HgfsSuperInfo *si;
   HgfsHandle handle;
   size_t actualCount = 0;
   size_t remainingCount = count;
   size_t nextCount = 0;
   loff_t curOffset = *offset;
   int result;

   ASSERT(file);
   ASSERT(buf);
   ASSERT(offset);

   /* We ignore zero byte writes. */
   if (!count) {
      return 0;
   }

   handle = FILE_GET_HAND(file);
   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsWrite: write fh %u, count %Zu, offset %Lu\n",
          handle, count, *offset));

   si = HGFS_SB_TO_COMMON(file->f_dentry->d_sb);

   /*
    * Call HgfsDoWrite repeatedly until either
    * - Write returns an error, or
    * - Write returns 0 (XXX this probably rarely happens), or
    * - We have written the requested number of bytes
    */
   result = 0;
   do {
      nextCount = (remainingCount > HGFS_IO_MAX) ? HGFS_IO_MAX : remainingCount;
      result = HgfsDoWrite(si, handle, file->f_flags, buf, nextCount, curOffset);
      if (result < 0) {
         LOG(4, (KERN_DEBUG "VMware hgfs: HgfsWrite: write error %d\n",
                result));
         goto out;
      }
      remainingCount -= result;
      actualCount += result;
      curOffset += result;
      buf += result;
   } while ((result > 0) && (remainingCount > 0));

   /* Increment file pointer. */
   *offset += actualCount;
   result = actualCount;

out:
   return result;
}




/*
 *----------------------------------------------------------------------
 *
 * HgfsRelease --
 *
 *    Called when the last user of a file closes it, i.e. when the
 *    file's f_count becomes zero.
 *
 * Results:
 *    Returns zero on success, or an error on failure.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */


static int
HgfsRelease(struct inode *inode,  // IN: Inode that this file points to
            struct file *file)    // IN: File that is getting released
{
   HgfsSuperInfo *si;
   HgfsReq *req;
   HgfsRequestClose *request;
   HgfsReplyClose *reply;
   HgfsHandle handle;
   int result = 0;

   ASSERT(inode);
   ASSERT(file);

   handle = FILE_GET_HAND(file);
   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsRelease: close fh %u\n", handle));

   /*
    * Free the HgfsHandle from the file*'s private data.
    */
   kfree(FILE_GET_HAND_P(file));
   FILE_SET_HAND_P(file, NULL);

   si = HGFS_SB_TO_COMMON(file->f_dentry->d_sb);

   req = HgfsGetNewRequest();
   if (!req) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsRelease: interrupted waiting for request\n"));
      return -ERESTARTSYS;
   }

   /* Fill in the request's fields. */
   request = (HgfsRequestClose *)(req->packet);
   request->header.id = req->id;
   request->header.op = HGFS_OP_CLOSE;
   request->file = handle;
   req->packetSize = sizeof *request;
   req->error = 0;

   /* Send the request and process the reply. */
   result = HgfsSendRequest(si, req);
   if ((!result) && (req->stateFile == HGFS_REQ_STATE_FINISHED)) {
      if (req->error) {
         LOG(4, (KERN_DEBUG "VMware hgfs: HgfsRelease: server returned error: %d\n",
                req->error));
         result = req->error;
         goto out;
      } else {
         /* Get the reply. */
         reply = (HgfsReplyClose *)(req->packet);
         if (reply->header.status != HGFS_STATUS_SUCCESS) {
            LOG(4, (KERN_DEBUG "VMware hgfs: HgfsRelease: failed on server\n"));
            result = -EPROTO;
            goto out;
         } else {
            result = 0;
            goto out;
         }
      }
   }

out:
   /*
    * See if request needs freeing and free it if necessary.
    */
   HgfsTestAndFreeRequest(result, req);

   return result;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsDirOpen --
 *
 *    Called whenever a process opens a directory in our filesystem.
 *
 *    We send a "Search Open" request to the server with the name
 *    stored in this file's inode. If the Open succeeds, we store the
 *    search handle sent by the server in the file struct so it can be
 *    accessed by readdir and close.
 *
 * Results:
 *    Returns zero if on success, error on failure.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static int
HgfsDirOpen(struct inode *inode,  // IN: Inode of the dir to open
            struct file *file)    // IN: File pointer for this open
{
   HgfsSuperInfo *si;
   HgfsReq *req;
   HgfsRequestSearchOpen *request;
   HgfsReplySearchOpen *reply;
   int result;

   ASSERT(inode);
   ASSERT(file);

   si = HGFS_SB_TO_COMMON(inode->i_sb);

   req = HgfsGetNewRequest();
   if (!req) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDirOpen: interrupted waiting for request\n"));
      return -ERESTARTSYS;
   }

   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsDirOpen: opening \"%s\"\n",
          INODE_GET_II_P(inode)->name));

   request = (HgfsRequestSearchOpen *)(req->packet);

   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsDirOpen: searching for \"%s\"\n",
          INODE_GET_II_P(inode)->name));

   /* Fill out the request's fields. */
   request->header.id = req->id;
   request->header.op = HGFS_OP_SEARCH_OPEN;
   /* Convert to CP name. */
   result = CPName_ConvertTo(INODE_GET_II_P(inode)->name,
                             HGFS_PACKET_MAX - (sizeof *request - 1),
                             request->dirName.name);
   if (result < 0) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDirOpen: CP conversion failed\n"));
      return -EINVAL;
   }
   /* Unescape the CP name. */
   result = HgfsUnescapeBuffer(request->dirName.name, result);
   request->dirName.length = result;
   req->packetSize = sizeof *request + result;
   req->error = 0;

   /* Send the request and process the reply. */
   result = HgfsSendRequest(si, req);
   if ((!result) && (req->stateFile == HGFS_REQ_STATE_FINISHED)) {
      if (req->error) {
         LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDirOpen: server returned error: %d\n",
                req->error));
         result = req->error;
         goto out;
      } else {
         /* Get the reply and check return status. */
         reply = (HgfsReplySearchOpen *)(req->packet);

         switch (reply->header.status) {
         case HGFS_STATUS_SUCCESS:
            if (req->packetSize != sizeof *reply) {
               /*
                * If status != success, the packetSize will be smaller,
                * so this test only applies in the success case. [bac]
                */
               LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDirOpen: wrong packet size\n"));
               result = -EPROTO;
               goto out;
            }

            /*
             * Store the handle for this open() in the file*.  This needs
             * to be freed on a close().
             */
            FILE_SET_HAND_P(file, kmalloc(sizeof(HgfsHandle), GFP_KERNEL));
            if (!FILE_GET_HAND_P(file)) {
               result = -ENOMEM;
               goto out;
            }
            FILE_GET_HAND(file) = reply->search;
            LOG(6, (KERN_DEBUG "VMware hgfs: HgfsDirOpen: set handle to %u\n",
                   FILE_GET_HAND(file)));
            break;

         case HGFS_STATUS_OPERATION_NOT_PERMITTED:
            result = -EPERM;
            break;

         default:
            result = -EPROTO;
            break;
         }
      }
   }

out:
   /*
    * See if request needs freeing and free it if necessary.
    */
   HgfsTestAndFreeRequest(result, req);

   return result;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsGetNextDirEntry --
 *
 *    Get the directory entry with the given offset from the server.
 *    nameOut gets allocated and must be freed by the caller after
 *    each call.
 *
 * Results:
 *    Returns zero on success, negative error on failure. If the
 *    dentry's name is too long, -ENAMETOOLONG is returned.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static int
HgfsGetNextDirEntry(HgfsSuperInfo *si,       // IN: Superinfo for this SB
                    HgfsHandle searchHandle, // IN: Handle of dir
                    uint32 offset,           // IN: Offset of next dentry to get
                    char *nameOut,           // OUT: Buffer to store new name
                    HgfsAttr *attr,          // OUT: File attributes of dentry
                    Bool *done)              /* OUT: Set true when there are
                                              * no more dentries
                                              */
{
   HgfsReq *req;
   HgfsRequestSearchRead *request;
   HgfsReplySearchRead *reply;
   int result = 0;

   ASSERT(si);
   ASSERT(nameOut);
   ASSERT(attr);
   ASSERT(done);

   req = HgfsGetNewRequest();
   if (!req) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsGetNextDirEntry: interrupted "
              "waiting for request\n"));
      return -ERESTARTSYS;
   }

   request = (HgfsRequestSearchRead *)(req->packet);

   /* Fill out the request's fields. */
   request->header.id = req->id;
   request->header.op = HGFS_OP_SEARCH_READ;
   request->search = searchHandle;
   request->offset = offset;
   req->packetSize = sizeof *request;
   req->error = 0;

   /* Send the request and process the reply. */
   result = HgfsSendRequest(si, req);
   if ((!result) && (req->stateFile == HGFS_REQ_STATE_FINISHED)) {
      if (req->error) {
         LOG(4, (KERN_DEBUG "VMware hgfs: HgfsGetNextDirEntry: server "
                 "returned error: %d\n", req->error));
         result = req->error;
         goto out;
      } else {
         /* Get the reply and check return status. */
         reply = (HgfsReplySearchRead *)(req->packet);
         if (reply->header.status != HGFS_STATUS_SUCCESS) {
            LOG(4, (KERN_DEBUG "VMware hgfs: HgfsGetNextDirEntry: server "
                    "returned non-success\n"));
            result = -EPROTO;
            goto out;
         }

         /*
          * Make sure name length is legal.
          */
         if (reply->fileName.length > NAME_MAX ||
             reply->fileName.length > HGFS_PACKET_MAX - sizeof *reply) {
            result = -ENAMETOOLONG;
            goto out;
         }

         /*
          * If the size of the name is valid (meaning the end
          * of the directory has not yet been reached), copy
          * the name to the out pointer and copy the attrs.
          */
         if (reply->fileName.length > 0) {
            /* Sanity check on name length. */
            if (reply->fileName.length != strlen(reply->fileName.name)) {
               LOG(4, (KERN_DEBUG "VMware hgfs: HgfsGetNextDirEntry: name length "
                       "mismatch %u/%Zu, name \"%s\"\n",
                      reply->fileName.length,
                      strlen(reply->fileName.name),
                      reply->fileName.name));
               result = -EPROTO;
               goto out;
            }

            strcpy(nameOut, reply->fileName.name);
            memcpy(attr, &reply->attr, sizeof *attr);
            LOG(6, (KERN_DEBUG "VMware hgfs: HgfsGetNextDirEntry: Got \"%s\"\n", nameOut));
         } else {
            /* We're at the end of the directory. */
            LOG(6, (KERN_DEBUG "VMware hgfs: HgfsGetNextDirEntry: end of dir\n"));
            *done = TRUE;
         }

         result = 0;
         goto out;
      }
   }

out:
   /*
    * See if request needs freeing and free it if necessary.
    */
   HgfsTestAndFreeRequest(result, req);

   return result;
}


/*
 *     Readdir is a bit complicated, and is best understood by reading
 *     the code. For the impatient, here is an overview of the major
 *     moving parts [bac]:
 *
 *     - Getdents syscall calls readdir, which is supposed to call
 *       filldir some number of times.
 *     - Each time it's called, filldir updates a struct with the
 *       number of bytes copied thus far, and sets an error code if
 *       appropriate.
 *     - When readdir returns, getdents checks the struct to see if
 *       any dentries were copied, and if so returns the byte count.
 *       Otherwise, it returns the error from the struct (which should
 *       still be zero if filldir was never called).
 *
 *       A consequence of this last fact is that if there are no more
 *       dentries, then readdir should NOT call filldir, and should
 *       return from readdir with a non-error.
 *
 *     Other notes:
 *
 *     - Passing an inum of zero to filldir doesn't work. At a minimum,
 *       you have to make up a bogus inum for each dentry.
 *     - Passing the correct d_type to filldir seems to be non-critical;
 *       apparently most programs (such as ls) stat each file if they
 *       really want to know what type it is. However, passing the
 *       correct type means that ls doesn't bother calling stat on
 *       directories, and that saves an entire round trip per dirctory
 *       dentry.
 */

/*
 *----------------------------------------------------------------------
 *
 * HgfsReaddir --
 *
 *    Handle a readdir request. See details above if interested.
 *
 * Results:
 *    Returns zero if on success, negative error on failure.
 *    (According to /fs/readdir.c, any non-negative return value
 *    means it succeeded).
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static int
HgfsReaddir(struct file *file, // IN:  Directory to read from
            void *dirent,      // OUT: Buffer to copy dentries into
            filldir_t filldir) // IN:  Filler function
{
   HgfsSuperInfo *si;
   HgfsAttr attr;
   uint32 d_type;    // type of dirent
   char nameBuf[NAME_MAX + 1]; // buf for name of next dentry
   char escName[NAME_MAX + 1]; // buf for escaped version of name
   int nameLength = 0;
   int result = 0;
   Bool done = FALSE;
   int ino = 100;    // XXX temporary (hopefully) [bac]

   ASSERT(file);
   ASSERT(dirent);

   if (!file ||
      !(file->f_dentry) ||
      !(file->f_dentry->d_inode)) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsReaddir: null input\n"));
      return -EFAULT;
   }

   si = HGFS_SB_TO_COMMON(file->f_dentry->d_inode->i_sb);

   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsReaddir: dir with name %s, "
           "inum %lu, f_pos %Lu\n",
          file->f_dentry->d_name.name,
          file->f_dentry->d_inode->i_ino,
          file->f_pos));

   while (1) {
      /*
       * Nonzero result = we failed to get valid reply from server.
       * Zero result:
       *     - done == TRUE means we hit the end of the directory
       *     - Otherwise, nameBuf has the name of the next dirent
       */
      result = HgfsGetNextDirEntry(si,
                                   FILE_GET_HAND(file),
                                   (uint32)file->f_pos,
                                   nameBuf,
                                   &attr,
                                   &done);
      if (result == -ENAMETOOLONG) {
         /* Skip dentry if its name is too long (see below). */
         file->f_pos++;
         continue;
      } else if (result) {
         /* Error  */
         LOG(4, (KERN_DEBUG "VMware hgfs: HgfsReaddir: error getting dentry\n"));
         return result;
      }
      if (done == TRUE) {
         LOG(6, (KERN_DEBUG "VMware hgfs: HgfsReaddir: end of dir reached\n"));
         break;
      }

      /*
       * Escape all non-printable characters (which for linux is just
       * "/").
       *
       * Note that normally we would first need to convert from the
       * CP name format, but that is done implicitely here since we
       * are guaranteed to have just one path component per dentry.
       */
      result = HgfsEscapeBuffer(nameBuf,
                                strlen(nameBuf),
                                NAME_MAX,
                                escName);

      /*
       * Check the filename length.
       *
       * If the name is too long to be represented in linux, we simply
       * skip it (i.e., that file is not visible to our filesystem) by
       * incrementing file->f_pos and repeating the loop to get the
       * next dentry.
       *
       * HgfsEscapeBuffer returns a negative value if the escaped
       * output didn't fit in the specified output size, so we can
       * just check its return value.
       */
      if (result < 0) {
         file->f_pos++;
         continue;
      }

      nameLength = result;

      /* Assign the correct dentry type. */
      switch (attr.type) {
      case HGFS_FILE_TYPE_REGULAR:
         d_type = DT_REG;
         break;

      case HGFS_FILE_TYPE_DIRECTORY:
         d_type = DT_DIR;
         break;

      default:
         /*
          * XXX Should never happen. I'd put NOT_IMPLEMENTED() here
          * but if the driver ever goes in the host it's probably not
          * a good idea for an attacker to be able to hang the host
          * simply by using a bogus file type in a reply. [bac]
          */
         d_type = DT_UNKNOWN;
         break;
      }

      /*
       * Call filldir for this dentry.
       *
       * XXX For now I just pass a bogus (made-up) inode number.
       * Eventually, we could do a lookup to get inode numbers for these
       * files. But, that would mean doing an extra round trip per
       * dentry, which isn't so great. If fake inode umbers in the output
       * of getdents(2) don't break any applications, we can probably
       * just not bother.
       */
      LOG(6, (KERN_DEBUG "VMware hgfs: HgfsReaddir: calling filldir "
              "with \"%s\", %u, %Lu\n",
             escName, nameLength, file->f_pos));
      result = filldir(dirent,                /* destination buffer (actually, filldir callback struct) */
                       escName,               /* name of dirent */
                       nameLength,            /* length of name */
                       file->f_pos,           /* offset of dirent */
                       ino,                   /* ino. If you use 0 as the value, the entries don't
                                               * get printed, so I'm making up numbers for now. [bac] */
                       d_type);               /* type of dirent */
      if (result) {
         /*
          * This means that filldir ran out of room in the user buffer
          * it was copying into; we just break out and return, but
          * don't increment f_pos. So the next time the user calls
          * getdents, this dentry will be requested again, will get
          * retrieved again, and get copied properly to the user.
          */
         break;
      }
      ino++;         // XXX temporary
      file->f_pos++;
   }

   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsReaddir: finished\n"));
   return 0;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsDirRelease --
 *
 *    Called when the last reader of a directory closes it, i.e. when
 *    the directory's file f_count field becomes zero.
 *
 * Results:
 *    Returns zero on success, or an error on failure.
 *
 * Side effects:
 *    None
 *
 *----------------------------------------------------------------------
 */

static int
HgfsDirRelease(struct inode *inode,  // IN: Inode that the file* points to
               struct file *file)    // IN: File for the dir getting released
{
   HgfsSuperInfo *si;
   HgfsReq *req;
   HgfsRequestSearchClose *request;
   HgfsReplySearchClose *reply;
   HgfsHandle handle;
   int result = 0;

   ASSERT(inode);
   ASSERT(file);

   handle = FILE_GET_HAND(file);
   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsDirRelease: close fh %u\n", handle));

   /*
    * Free the HgfsHandle from the file*'s private data.
    */
   kfree(FILE_GET_HAND_P(file));
   FILE_SET_HAND_P(file, NULL);

   si = HGFS_SB_TO_COMMON(file->f_dentry->d_sb);

   req = HgfsGetNewRequest();
   if (!req) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDirRelease: interrupted waiting for request\n"));
      return -ERESTARTSYS;
   }

   /* Fill in the request's fields. */
   request = (HgfsRequestSearchClose *)(req->packet);
   request->header.id = req->id;
   request->header.op = HGFS_OP_SEARCH_CLOSE;
   request->search = handle;
   req->packetSize = sizeof *request;
   req->error = 0;

   /* Send the request and process the reply. */
   result = HgfsSendRequest(si, req);
   if ((!result) && (req->stateFile == HGFS_REQ_STATE_FINISHED)) {
      if (req->error) {
         LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDirRelease: server returned error: %d\n",
                req->error));
         result = req->error;
         goto out;
      } else {
         /* Get the reply. */
         reply = (HgfsReplySearchClose *)(req->packet);
         if (reply->header.status != HGFS_STATUS_SUCCESS) {
            LOG(4, (KERN_DEBUG "VMware hgfs: HgfsDirRelease: failed on server\n"));
            result = -EPROTO;
            goto out;
         } else {
            result = 0;
            goto out;
         }
      }
   }

out:
   /*
    * See if request needs freeing and free it if necessary.
    */
   HgfsTestAndFreeRequest(result, req);

   return result;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsFileMmapNoPage --
 *
 *    Called when user process needs some page from mmapped area.
 *
 * Results:
 *    Returns page filled with file's data on success.
 *    Returns NULL on failure, which is converted to SIGBUS signal
 *                 by kernel
 *
 * Side effects:
 *    None.
 *
 *----------------------------------------------------------------------
 */

#if defined(VMW_NOPAGE_261)
static struct page *
HgfsFileMmapNoPage(struct vm_area_struct *area, // IN: VM area
                   unsigned long address,	// IN: Page fault address
		   int *type)			// IN: Fault type
#else
static struct page*
HgfsFileMmapNoPage(struct vm_area_struct *area,	// IN: VM area
		   unsigned long address,	// IN: Page fault address
		   int write_access)		// IN: Dummy in later kernels
#endif
{
   loff_t pos;
   struct page* page;
   char *pg_addr;
   int ret;
   mm_segment_t old_fs;
   sigset_t sigset;

   page = alloc_page(GFP_HIGHUSER);
   if (!page) {
      return page;
   }
   pg_addr = kmap(page);
   address &= PAGE_MASK;
   pos = address - area->vm_start + ((loff_t)area->vm_pgoff << PAGE_SHIFT);

   /*
    * We must disable signals: otherwise HgfsRead may return
    * -EINTR/-ERESTARTSYS and we'll return NULL in such case,
    * immediately causing SIGBUS.
    */
   HgfsDisableSignals(&sigset);
   old_fs = get_fs();
   set_fs(get_ds());
   ret = HgfsRead(area->vm_file, pg_addr, PAGE_SIZE, &pos);
   set_fs(old_fs);
   HgfsRestoreSignals(&sigset);

   if (ret < 0) {
      kunmap(page);
      put_page(page);
      return NULL;
   } else if (ret >= PAGE_SIZE) {
      /* Not that ret > PAGE_SIZE should happen... */
   } else {
      memset(pg_addr + ret, 0, PAGE_SIZE - ret);
   }
   flush_dcache_page(page);
   kunmap(page);
#ifdef VMW_NOPAGE_261
   *type = VM_FAULT_MAJOR;
#endif
   return page;
}

static struct vm_operations_struct HgfsFileMmap =
{
   .nopage	= HgfsFileMmapNoPage,
};


/*
 *----------------------------------------------------------------------
 *
 * HgfsFsync --
 *
 *    Called when user process calls fsync() on hgfs file.
 *
 *    The hgfs protocol doesn't support fsync yet, so for now, we punt
 *    and just return success. This is a little less sketchy than it
 *    might sound, because hgfs skips the buffer cache in the guest
 *    anyway (we always write to the host immediately).
 *
 *    In the future we might want to try harder though, since
 *    presumably the intent of an app calling fsync() is to get the
 *    data onto persistent storage, and as things stand now we're at
 *    the whim of the hgfs server code running on the host to fsync or
 *    not if and when it pleases.
 *
 * Results:
 *    Returns zero on success. (Currently always succeeds).
 *
 * Side effects:
 *    None.
 *
 *----------------------------------------------------------------------
 */

static int
HgfsFsync(struct file *file,		// IN: File we operate on
          struct dentry *dentry,        // IN: Dentry for this file
          int datasync)	                // IN: fdatasync or fsync
{
   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsFsync: was called\n"));

   return 0;
}


/*
 *----------------------------------------------------------------------
 *
 * HgfsMmap --
 *
 *    Called when user process calls mmap() on hgfs file.
 *
 * Results:
 *    Returns zero on success.
 *    Returns negative error value on failure (shared mappings not supported)
 *
 * Side effects:
 *    None.
 *
 *----------------------------------------------------------------------
 */

static int
HgfsMmap(struct file *file,		// IN: File we operate on
         struct vm_area_struct *vma)	// IN/OUT: VM area information
{
   struct inode *inode;

   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsMmap: was called\n"));

   /* only PAGE_COW or read-only supported now */
   if (vma->vm_flags & VM_SHARED) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsMmap: operation not supported\n"));
      return -EINVAL;
   }
   inode = file->f_dentry->d_inode;
   if (!IS_RDONLY(inode)) {
      inode->i_atime = CURRENT_TIME;
   }
   vma->vm_ops = &HgfsFileMmap;
   return 0;
}

static struct file_operations HgfsFileOperations = {
   owner:    THIS_MODULE,
   open:     HgfsOpen,
   read:     HgfsRead,
   write:    HgfsWrite,
   fsync:    HgfsFsync,
   mmap:     HgfsMmap,
   release:  HgfsRelease,
};

static struct file_operations HgfsDirOperations = {
   owner:    THIS_MODULE,
   open:     HgfsDirOpen,
   read:     generic_read_dir,
   readdir:  HgfsReaddir,
   release:  HgfsDirRelease,
};


/*
 * Superblock operations
 */


/*
 *-----------------------------------------------------------------------------
 *
 * HgfsClearInode --
 *
 *    Hgfs superblock 'clear_inode' method. Called by the kernel when it is
 *    about to destroy a VFS inode.
 *
 * Results:
 *    None
 *
 * Side effects:
 *    None
 *
 *-----------------------------------------------------------------------------
 */

void
HgfsClearInode(struct inode *inode) // IN: The VFS inode
{
   HgfsInodeInfo *iinfo;

   ASSERT(inode);

   /* The HGFS inode information may be partially constructed --hpreg */

   iinfo = INODE_GET_II_P(inode);
   if (iinfo) {
      kfree(iinfo->name);
      kfree(iinfo);
   }
}


/*
 *-----------------------------------------------------------------------------
 *
 * HgfsReadInode --
 *
 *    Hgfs superblock 'read_inode' method. Called by the kernel to fill in a
 *    VFS inode, given its hgfs inode number.
 *
 * Results:
 *    None
 *
 * Side effects:
 *    None
 *
 *-----------------------------------------------------------------------------
 */

void
HgfsReadInode(struct inode *inode) // IN/OUT: VFS inode to fill in
{
   /*
    * The server does not store any inode information, so there is nothing to
    * retrieve from it, all information should already be in the inode.
    *
    * XXX What about other fields like 'i_blksize'? Nobody seems to initialize
    *     them --hpreg
    */
}


/*
 *-----------------------------------------------------------------------------
 *
 * HgfsPutSuper --
 *
 *    Hgfs superblock 'put_super' method. Called after a umount(2) of the
 *    filesystem succeeds.
 *
 * Results:
 *    None
 *
 * Side effects:
 *    None
 *
 *-----------------------------------------------------------------------------
 */

void
HgfsPutSuper(struct super_block *sb) // IN: The superblock
{
   HgfsSuperInfo *si;

   ASSERT(sb);

   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsPutSuper: was called\n"));

   si = HGFS_SB_TO_COMMON(sb);
   ASSERT(si);

   spin_lock(&hgfsBigLock);

   ASSERT(si->sb);
   /* Tell the proc side of the driver that the filesystem side is gone. */
   si->sb = NULL;

   if (si->userFile) {
      /* The whole driver is in state 1. Transition to the initial state. */
      spin_unlock(&hgfsBigLock);
      return;
   }

   /* The whole driver is in state 4. Transition to the final state. */

   spin_unlock(&hgfsBigLock);

   kfree(si);
}


/*
 *-----------------------------------------------------------------------------
 *
 * HgfsStatfs --
 *
 *    Hgfs superblock 'statfs' method. Called when statfs(2) is invoked on the
 *    filesystem.
 *
 * Results:
 *    0 on success
 *    error < 0 on failure
 *
 * Side effects:
 *    None
 *
 *-----------------------------------------------------------------------------
 */

int
HgfsStatfs(struct super_block *sb,	    // IN : The superblock
           struct compat_kstatfs *stat) // OUT: Stat to fill in
{
   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsStatfs: was called\n"));

   memset(stat, 0, sizeof *stat);
   stat->f_type = HGFS_SUPER_MAGIC;
   stat->f_bsize = 4096;
   stat->f_blocks = 1000000;
   stat->f_bfree = 900000;
   stat->f_bavail = 800000;
   stat->f_files = 100000;
   stat->f_ffree = 90000;
   stat->f_namelen = 255;

   return 0;
}


/* Hgfs filesystem superblock operations */
static struct super_operations HgfsSuperOperations =
{
   /* write_inode is optional */

   /* Optional */
   clear_inode:  HgfsClearInode,

   /* Mandatory */
   read_inode:   HgfsReadInode,

   /* Optional */
   put_super:    HgfsPutSuper,

   /* Optional */
   statfs:       HgfsStatfs,
};


/*
 *-----------------------------------------------------------------------------
 *
 * HgfsReadSuper --
 *
 *    The main entry point of the filesystem side of the driver. Called when
 *    a userland process does a mount(2) of an hgfs filesystem. This makes the
 *    whole driver transition from its initial state to state 1. Fill the
 *    content of the uninitialized superblock provided by the kernel.
 *
 *    'rawData' is a pointer (that can be NULL) to a kernel buffer (whose
 *    size is <= PAGE_SIZE) that corresponds to the filesystem-specific 'data'
 *    argument passed to mount(2).
 *
 * Results:
 *    zero and initialized superblock on success
 *    negative value on failure
 *
 * Side effects:
 *    None
 *
 *-----------------------------------------------------------------------------
 */

static int
HgfsReadSuper(struct super_block *sb, // OUT: Superblock object
              void *rawData,          // IN: Fs-specific mount data
              int flags)              // IN: Mount flags
{
   HgfsMountInfo *mntInfo;
   struct file *userFile;
   HgfsSuperInfo *si;
   struct inode *rootInode;
   struct dentry *rootDentry;
   uint64 curTime = HgfsConvertToNtTime(HGFS_GET_TIME(CURRENT_TIME));
   HgfsAttr rootAttr = {
      HGFS_FILE_TYPE_DIRECTORY,
      4096, /* That's right, hard coded. */
      curTime,
      curTime,
      curTime,
      curTime,
      HGFS_PERM_READ | HGFS_PERM_WRITE | HGFS_PERM_EXEC,
   };

   ASSERT(sb);

   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsReadSuper: entered\n"));

   /*
    * Run some sanity checks on the mount data passed in by
    * the user process.
    */

   mntInfo = (HgfsMountInfo *)rawData;
   if (!mntInfo) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsReadSuper: null mntInfo pointer\n"));
      return -EFAULT;
   }

   if (mntInfo->version != HGFS_PROTOCOL_VERSION) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsReadSuper: mount data version %u, "
              "kernel is %u\n",
             mntInfo->version,
             HGFS_PROTOCOL_VERSION));
      return -ECHRNG;
   }

   if (mntInfo->magicNumber != HGFS_SUPER_MAGIC) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsReadSuper: magic number mismatch "
              "(mount data %u, kernel is %u)\n",
             mntInfo->magicNumber,
             HGFS_SUPER_MAGIC));
      return -ECHRNG;
   }

   LOG(8, (KERN_DEBUG "VMware hgfs: HgfsReadSuper: sanity tests passed, "
           "about to get file\n"));

   /* Check that the fd is valid. */

   userFile = fget(mntInfo->fd);
   if (userFile == NULL) {
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsReadSuper: non-existing fd %d\n",
             mntInfo->fd));
      return -EBADF;
   }
   /* We now hold a reference on file 'userFile', so it cannot be destroyed. */

   if (userFile->f_op != &HgfsProcDevOperations) {
      fput(userFile);
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsReadSuper: wrong kind of fd %d\n",
             mntInfo->fd));
      return -EBADFD;
   }

   si = HGFS_DEV_TO_COMMON(userFile);
   ASSERT(si);

   spin_lock(&hgfsBigLock);

   if (si->sb) {
      spin_unlock(&hgfsBigLock);
      fput(userFile);
      LOG(4, (KERN_DEBUG "VMware hgfs: HgfsReadSuper: fd %d already mounted\n",
             mntInfo->fd));
      return -EBUSY;
   }

   /*
    * Tell the proc side of the driver that the filesystem side is here. This
    * makes sure the common state won't be destroyed if 'userFile' is destroyed
    * in the future.
    */
   si->sb = sb;

   /* XXX Should not be in the common state --hpreg */
   si->uid = current->uid;

   spin_unlock(&hgfsBigLock);

   /*
    * We absolutely want to know if the pserver dies. The only way to know is
    * to make sure that when it happens, 'userFile' is destroyed (i.e. its
    * 'release' method is called), which we do by releasing our reference.
    */
   fput(userFile);

   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsReadSuper: got file\n"));

   HGFS_SET_SB_TO_COMMON(sb, si);
   sb->s_magic = HGFS_SUPER_MAGIC;
   sb->s_blocksize = HGFS_BLOCKSIZE;
   sb->s_op = &HgfsSuperOperations;

   /*
    * Make root inode and dentry.
    */
   rootInode = HgfsIget(sb, HGFS_ROOT_INO, &rootAttr, "/");
   rootDentry = d_alloc_root(rootInode);
   sb->s_root = rootDentry;

   /*
    * Check that the name got set correctly.
    */

   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsReadSuper: root ino %lu, i_dev %u\n",
          rootInode->i_ino, rootInode->i_sb->s_dev));
   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsReadSuper: root remote name is %s\n",
          INODE_GET_II_P(rootInode)->name));

   {
      /* XXX testing only */
      char *test = NULL;

      /* Do some quick testing. */
      test = HgfsGetDentryFullName(sb->s_root);
      if (!test) {
         LOG(4, (KERN_DEBUG "VMware hgfs: HgfsReadSuper: failed to get dentry name\n"));
      } else {
         LOG(6, (KERN_DEBUG "VMware hgfs: HgfsReadSuper: root dentry full name is %s\n",
                test));
         kfree(test);
      }
   }

   LOG(6, (KERN_DEBUG "VMware hgfs: HgfsReadSuper: finished\n"));

   return 0;
}


#if KERNEL_25_FS
/*
 *-----------------------------------------------------------------------------
 *
 * HgfsGetSb --
 *
 *    Invokes generic kernel code to prepare superblock for
 *    deviceless filesystem.
 *
 * Results:
 *    The initialized superblock on success
 *    NULL on failure
 *
 * Side effects:
 *    None
 *
 *-----------------------------------------------------------------------------
 */

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 5, 70)
static struct super_block *
HgfsGetSb(struct file_system_type *fs_type,
	  int flags,
	  const char *dev_name,
	  void *rawData)
#else
static struct super_block *
HgfsGetSb(struct file_system_type *fs_type,
	  int flags,
	  char *dev_name,
	  void *rawData)
#endif
{
   return get_sb_nodev(fs_type, flags, rawData, HgfsReadSuper);
}
#else


/*
 *-----------------------------------------------------------------------------
 *
 * HgfsReadSuper24 --
 *
 *    Compatibility wrapper for 2.4.x kernels read_super.
 *    Converts success to sb, and failure to NULL.
 *
 * Results:
 *    The initialized superblock on success
 *    NULL on failure
 *
 * Side effects:
 *    None
 *
 *-----------------------------------------------------------------------------
 */

static struct super_block *
HgfsReadSuper24(struct super_block *sb,
		void *rawData,
		int flags) {
   return HgfsReadSuper(sb, rawData, flags) ? NULL : sb;
}
#endif

/*
 * Filesystem type for HGFS, the "Host/Guest File System".
 */
struct file_system_type hgfsType = {
   /* Filesystem name */
   name: HGFS_NAME,

   /*
    * Filesystem flags.
    * No need for a device, we are like a network filesystem.
    * Our mountdata are binary structure, not a comma delimited string.
    */
   fs_flags: FS_BINARY_MOUNTDATA,

#if KERNEL_25_FS
   .get_sb =  HgfsGetSb,
   .kill_sb = kill_anon_super,
#else
   /* Function to read the superblock. */
   read_super: HgfsReadSuper24,
#endif

   /*
    * Module whose refcount needs to be incremented when this object is in use
    * by the rest of the kernel.
    */
   owner: THIS_MODULE,
};
