/******************************************************************************
 *
 * Driver for Option High Speed Mobile Devices.
 *
 *  Copyright (C) 2008  Option International
 *  Copyright (C) 2007  Andrew Bird (Sphere Systems Ltd)	ajb@spheresystems.co.uk
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
*
 *****************************************************************************/

/******************************************************************************
 *
 * Description of the device:
 *
 * Interface 0:	Contains the IP network interface on the bulk end points.
 *		The multiplexed serial ports are using the interrupt and control endpoints.
 *		Interrupt contains a bitmap telling which multiplexed serialport needs servicing.
 *
 * Interface 1:	Diagnostics port, uses bulk only, do not submit urbs until the port is opened, as
 *		this have a huge impact on the network port throughput.
 *
 * Interface 2:	Standard modem interface - circuit switched interface, should not be used.
 *
 *****************************************************************************/
#include "hso.h"

#define SIOCSETSUSPEND (SIOCDEVPRIVATE+3)
#define SIOCSETRADIO   (SIOCDEVPRIVATE+4)

/* #define DEBUG */

#define dev2net(x) (x->port_data.dev_net)
#define dev2ser(x) (x->port_data.dev_serial)

#ifdef DEBUG
#define sDEBUG "DEBUG "
#else
#define sDEBUG ""
#endif
#ifdef USE_SUSPEND
#define sUSE_SUSPEND "USE_SUSPEND "
#else
#define sUSE_SUSPEND ""
#endif
#ifdef SYSFS_NET
#define sSYSFS_NET "SYSFS_NET "
#else
#define sSYSFS_NET ""
#endif

//*****************************************************************************
// Debugging functions
//*****************************************************************************
#ifdef DEBUG
static void dbg_dump(int line_count, const char *func_name, unsigned char *buf, unsigned int len)
{
	u8 i = 0;

	printk("[%d:%s]: len %d", line_count, func_name, len);

	for (i = 0; i < len; i++) {
		if (!(i % 16))
			printk("\n    0x%03x:  ", i);
		printk("%02x ", (unsigned char)buf[i]);
	}
	printk("\n");
}
#define DUMP(buf_, len_)	dbg_dump(__LINE__, __FUNCTION__, buf_, len_)
#define DUMP1(buf_, len_)	do{ if(0x01 & debug) DUMP(buf_, len_);}while(0)
#else
#define DUMP(buf_, len_)
#define DUMP1(buf_, len_)
#endif

// >2.6.23 compilation
#if !defined(SET_MODULE_OWNER)
#define SET_MODULE_OWNER(x) do {} while(0)
#endif

//*****************************************************************************
// Globals
//*****************************************************************************
/* module parameters */
static int			debug = 0;
static int			procfs = 1;
static int			tty_major = 0;
static int			disable_net = 0;
static int			enable_autopm = 0;
/* driver info */
static const char		driver_name[] = "hso";
static const char		tty_filename[] = "ttyHS";
static const char		*version = __FILE__ ": " DRIVER_VERSION " " MOD_AUTHOR;
/* the usb driver itself (registered in hso_init) */
static struct usb_driver	hso_driver;
/* the procfs vars (if procfs enabled as parameter) */
static struct proc_dir_entry    *hso_proc_dir = NULL;
static struct proc_dir_entry    *hso_proc_dir_devices = NULL;
/* serial structures */
static struct tty_driver	*tty_drv = NULL;
static struct hso_device	*serial_table[HSO_SERIAL_TTY_MINORS];
static struct hso_device	*network_table[HSO_MAX_NET_DEVICES];
static spinlock_t		serial_table_lock;
static struct TERMIOS		*hso_serial_termios[HSO_SERIAL_TTY_MINORS];
static struct TERMIOS		*hso_serial_termios_locked[HSO_SERIAL_TTY_MINORS];

const s32	default_port_spec[] = {
		HSO_INTF_MUX|HSO_PORT_NETWORK,
		HSO_INTF_BULK|HSO_PORT_DIAG,
		HSO_INTF_BULK|HSO_PORT_MODEM,
		0
};

const s32	Icon321_port_spec[] = {
		HSO_INTF_MUX|HSO_PORT_NETWORK,
		HSO_INTF_BULK|HSO_PORT_DIAG2,
		HSO_INTF_BULK|HSO_PORT_MODEM,
		HSO_INTF_BULK|HSO_PORT_DIAG,
		0
};


/* list of devices we support */
static struct			usb_device_id hso_ids[] = {
	{USB_DEVICE(0x0af0, 0x6711), .driver_info = ( kernel_ulong_t ) default_port_spec },
	{USB_DEVICE(0x0af0, 0x6731), .driver_info = ( kernel_ulong_t ) default_port_spec },
	{USB_DEVICE(0x0af0, 0x6751), .driver_info = ( kernel_ulong_t ) default_port_spec },
	{USB_DEVICE(0x0af0, 0x6771), .driver_info = ( kernel_ulong_t ) default_port_spec },
	{USB_DEVICE(0x0af0, 0x6791), .driver_info = ( kernel_ulong_t ) default_port_spec }, 
	{USB_DEVICE(0x0af0, 0x6811), .driver_info = ( kernel_ulong_t ) default_port_spec },
	{USB_DEVICE(0x0af0, 0x6911), .driver_info = ( kernel_ulong_t ) default_port_spec },
	{USB_DEVICE(0x0af0, 0x6951), .driver_info = ( kernel_ulong_t ) default_port_spec },
	{USB_DEVICE(0x0af0, 0x6971), .driver_info = ( kernel_ulong_t ) default_port_spec },
	{USB_DEVICE(0x0af0, 0x7011), .driver_info = ( kernel_ulong_t ) default_port_spec },
	{USB_DEVICE(0x0af0, 0x7031), .driver_info = ( kernel_ulong_t ) default_port_spec },
	{USB_DEVICE(0x0af0, 0x7051), .driver_info = ( kernel_ulong_t ) default_port_spec },
	{USB_DEVICE(0x0af0, 0x7071), .driver_info = ( kernel_ulong_t ) default_port_spec },
	{USB_DEVICE(0x0af0, 0x7111), .driver_info = ( kernel_ulong_t ) default_port_spec },
	{USB_DEVICE(0x0af0, 0x7211), .driver_info = ( kernel_ulong_t ) default_port_spec },
	{USB_DEVICE(0x0af0, 0x7251), .driver_info = ( kernel_ulong_t ) default_port_spec },
	{USB_DEVICE(0x0af0, 0x7271), .driver_info = ( kernel_ulong_t ) default_port_spec },
	{USB_DEVICE(0x0af0, 0x7311), .driver_info = ( kernel_ulong_t ) default_port_spec },
	{USB_DEVICE(0x0af0, 0xc031), .driver_info = ( kernel_ulong_t ) default_port_spec }, /* Icon-Edge */
	{USB_DEVICE(0x0af0, 0xd013), .driver_info = ( kernel_ulong_t ) Icon321_port_spec }, /* Module HSxPA */
	{USB_DEVICE(0x0af0, 0xd031), .driver_info = ( kernel_ulong_t ) Icon321_port_spec }, /* Icon-321 */
	{USB_DEVICE(0x0af0, 0xd033), .driver_info = ( kernel_ulong_t ) default_port_spec }, /* Icon-322 */
	{USB_DEVICE(0x0af0, 0x7301), .driver_info = 0 }, /* GE40x */
	{USB_DEVICE(0x0af0, 0x7361), .driver_info = 0 }, /* GE40x */
	{USB_DEVICE(0x0af0, 0x7401), .driver_info = 0 }, /* GI 0401 */
	{USB_DEVICE(0x0af0, 0x7501), .driver_info = 0 }, /* GTM 382 */
	{USB_DEVICE(0x0af0, 0x7601), .driver_info = 0 }, /* GE40x */
	{}
};
/* operations setup of the serial interface */
static struct 			tty_operations hso_serial_ops = {
	.open =                 hso_serial_open,
	.close =                hso_serial_close,
	.write =                hso_serial_write,
	.write_room =           hso_serial_write_room,
	.ioctl =                hso_serial_ioctl,
	.set_termios =          hso_serial_set_termios,
	.throttle =             hso_serial_throttle,
	.unthrottle =           hso_serial_unthrottle,
	.break_ctl =            hso_serial_break,
	.chars_in_buffer =      hso_serial_chars_in_buffer,
	.read_proc =            hso_serial_read_proc,
	.hangup =		hso_serial_hangup,
	.tiocmget =		hso_serial_tiocmget,
	.tiocmset =		hso_serial_tiocmset,
};
/* driver setup */
static struct 			usb_driver hso_driver = {
	.name =			driver_name,
	.probe =		hso_probe,
	.disconnect =		hso_disconnect,
	.id_table =		hso_ids,
#ifdef USE_SUSPEND
	.suspend =		hso_suspend,
	.resume = 		hso_resume,
	.supports_autosuspend = 1,
#endif
};

// Sysfs attribute & function declaration

#if ( LINUX_VERSION_CODE > KERNEL_VERSION(2,6,19) )
static ssize_t hso_sysfs_show_porttype( struct device *dev, struct device_attribute *attr, char *buf );
DEVICE_ATTR( hsotype, S_IRUGO, hso_sysfs_show_porttype, NULL );

ssize_t hso_sysfs_show_porttype( struct device *dev, struct device_attribute *attr, char *buf )
{
	struct hso_device *hso_dev = dev->driver_data;
	char *port_name;

	if( !hso_dev ) return 0;

	switch( hso_dev->port_spec & HSO_PORT_MASK ) {
		case HSO_PORT_CONTROL:  	port_name = "Control";          break;
		case HSO_PORT_APP:      	port_name = "Application";      break;
		case HSO_PORT_APP2:      	port_name = "Application2";     break;
		case HSO_PORT_GPS:      	port_name = "GPS";              break;
		case HSO_PORT_GPS_CONTROL:      port_name = "GPS Control";      break;
		case HSO_PORT_PCSC:     	port_name = "PCSC";             break;
		case HSO_PORT_DIAG:     	port_name = "Diagnostic";       break;
		case HSO_PORT_DIAG2:     	port_name = "Diagnostic2";       	break;
		case HSO_PORT_MODEM:   		port_name = "Modem";            break;
		case HSO_PORT_NETWORK:  	port_name = "Network";		break;
		default:                	port_name = "Unknown";          break;
	}

	return sprintf( buf, "%s\n", port_name );


}

#endif
//*****************************************************************************
//*****************************************************************************
// Procfs functions
//*****************************************************************************
//*****************************************************************************

static int hso_proc_options(char *buf, char **start, off_t offset, int count, int *eof, void *data)
{
        int len = 0;
        /* get the module parameters */
	len += snprintf(buf + len, count - len, "Version: %s\n", DRIVER_VERSION );
        len += snprintf(buf + len, count - len, "debug: 0x%02x\n", debug);
        len += snprintf(buf + len, count - len, "procfs: 0x%02x\n", procfs);
        len += snprintf(buf + len, count - len, "tty_major: 0x%02x\n", tty_major);
        len += snprintf(buf + len, count - len, "enable_autopm: 0x%02x\n", enable_autopm);
	len += snprintf(buf + len, count - len, "disable_net: 0x%02x\n", disable_net );
        return len;
}

static int hso_proc_port_info(char *buf, char **start, off_t offset, int count, int *eof, void *data)
{
        int len = 0;
        struct hso_device *hso_dev = (struct hso_device *)data;
        char *port_name = NULL;

        D1("count: %d", count);

	switch( hso_dev->port_spec & HSO_PORT_MASK ) {
                        case HSO_PORT_CONTROL:  	port_name = "Control";          break;
                        case HSO_PORT_APP:      	port_name = "Application";      break;
                        case HSO_PORT_GPS:      	port_name = "GPS";              break;
                        case HSO_PORT_GPS_CONTROL:      port_name = "GPS Control";      break;
                        case HSO_PORT_APP2:     	port_name = "Application2";     break;
                        case HSO_PORT_PCSC:     	port_name = "PCSC";             break;
                        case HSO_PORT_DIAG:     	port_name = "Diagnostic";       break;
                        case HSO_PORT_DIAG2:     	port_name = "Diagnostic2";     	break;
                        case HSO_PORT_MODEM:   		port_name = "Modem";            break;
			case HSO_PORT_NETWORK:  	port_name = "Network";		break;
                        default:                	port_name = "Unknown";          break;
	}

	len += snprintf(buf + len, count - len, "%s\n", port_name);

        /* return the device id to the user, for use in scripts for autopm */
        len += snprintf(buf + len, count - len, "USB bus ID:\t%s\n", hso_dev->usb->dev.bus_id);
        return len;
}



//*****************************************************************************
//*****************************************************************************
// Network interface functions
//*****************************************************************************
//*****************************************************************************

//=============================================================================
//	function:	hso_net_open
//	purpose:	called when net interface is brought up by ifconfig
//=============================================================================
static int hso_net_open(struct net_device *net)
{
	struct hso_net *odev = netdev_priv(net);
	unsigned long flags = 0;

	//printk("%s called\n", __FUNCTION__);
	if( !odev ) {
		printk("No net device !\n");
		return -ENODEV;
	}
	/* reset read counter. */
	odev->stats.rx_packets = 0;
	odev->skb_tx_buf = NULL;

	/* setup environment */
	spin_lock_irqsave(&odev->net_lock,flags);
	odev->rx_parse_state = WAIT_IP;
	odev->rx_buf_size = 0;
	odev->rx_buf_missing = sizeof(struct iphdr);
	spin_unlock_irqrestore(&odev->net_lock,flags);

	hso_start_net_device( odev->parent );

	/* Tell the kernel we are ready to start receiving from it */
	netif_start_queue(net);

	/* We are up and running. */
	set_bit(HSO_NET_RUNNING, &odev->flags);

	return 0;
}

//=============================================================================
//	function:	hso_net_close
//	purpose:	called when interface is brought down by ifconfig
//=============================================================================
static int hso_net_close(struct net_device *net)
{
	struct hso_net *odev = netdev_priv(net);
	//printk("%s called\n", __FUNCTION__);


	/* no longer running */
	clear_bit(HSO_NET_RUNNING, &odev->flags);
	/* we don't need the queue anymore */
	netif_stop_queue(net);

	hso_stop_net_device( odev->parent);

	/* done */
	return 0;
}

//=============================================================================
//	function:	hso_net_start_xmit
//	purpose:	called by kernel when we need to transmit a packet
//=============================================================================
static int hso_net_start_xmit(struct sk_buff *skb, struct net_device *net)
{
	struct hso_net *odev = netdev_priv(net);
	int result = 0;

	//printk("%s called\n", __FUNCTION__);

	/* Tell the kernel, "No more frames 'til we are done with this one." */
	netif_stop_queue(net);
#ifdef USE_SUSPEND
	if( hso_get_activity( odev->parent ) == -EAGAIN ) {
		odev->skb_tx_buf = skb;
		return 0;
	}
#endif


	/* fetch the packet */
	skb_pull(skb, ETH_HLEN);
	/* log if asked */
	DUMP1(skb->data, skb->len);
	/* Copy it from kernel memory to OUR memory */
	memcpy(odev->mux_bulk_tx_buf, skb->data, skb->len);
	D1("len: %d/%d", skb->len, MUX_BULK_TX_BUF_SIZE);

	/* Fill in the URB for shipping it out. */
	usb_fill_bulk_urb(	odev->mux_bulk_tx_urb,
				odev->parent->usb,
				usb_sndbulkpipe(odev->parent->usb, odev->out_endp->bEndpointAddress & 0x7F ),
				odev->mux_bulk_tx_buf,
				skb->len,
				write_bulk_callback,
				odev);

	/* Deal with the Zero Length packet problem, I hope */
	odev->mux_bulk_tx_urb->transfer_flags |= URB_ZERO_PACKET;

	/* Send the URB on its merry way. */
	if ((result = usb_submit_urb(odev->mux_bulk_tx_urb, GFP_ATOMIC))) {
		WARN("failed mux_bulk_tx_urb %d", result);
		odev->stats.tx_errors++;
		netif_start_queue(net);
	} else {
		odev->stats.tx_packets++;
		odev->stats.tx_bytes += skb->len;
		net->trans_start = jiffies; /* And tell the kernel when the last transmit started. */
	}
	dev_kfree_skb(skb);
	/* we're done */
	return result;
}

//=============================================================================
//	function:	hso_net_ioctl
//	purpose:	
//=============================================================================
static int hso_net_ioctl(struct net_device *net, struct ifreq *rq, int cmd)
{
	struct hso_net *odev = netdev_priv(net);
	u32 usercmd = 0;
	char tmp[40];


	//printk("%s called\n", __FUNCTION__);
	switch (cmd) {
		case SIOCDEVPRIVATE + 1: /* Chose this one because SIOCDEVPRIVATE used somewhere else in this code */
			/* return the number of sent bytes */
			D5("Transmitted: %lu", odev->stats.tx_bytes);
			rq->ifr_ifru.ifru_ivalue = odev->stats.tx_bytes;
			return 0;
	
		case SIOCETHTOOL:
			/* net specific */
			if (copy_from_user(&usercmd, rq->ifr_data,sizeof(usercmd)))
				return -EFAULT;
		
			switch (usercmd) {
				case ETHTOOL_GDRVINFO: {
					/* get driver info */
					struct ethtool_drvinfo info = { ETHTOOL_GDRVINFO };
					strncpy(info.driver, driver_name, ETHTOOL_BUSINFO_LEN);
					strncpy(info.version, DRIVER_VERSION, ETHTOOL_BUSINFO_LEN);
					sprintf(tmp, "usb%d:%d", odev->parent->usb->bus->busnum, odev->parent->usb->devnum);
					strncpy(info.bus_info, tmp, ETHTOOL_BUSINFO_LEN);
					sprintf(tmp, "%s %x.%x", driver_name, ((odev->bcdCDC & 0xff00) >> 8), (odev->bcdCDC & 0x00ff));
					strncpy(info.fw_version, tmp, ETHTOOL_BUSINFO_LEN);
					if (copy_to_user(rq->ifr_data, &info, sizeof(info)))
						return -EFAULT;
		
					return 0;
				}
				case ETHTOOL_GLINK: {
					/* get link status */
					struct ethtool_value edata = { ETHTOOL_GLINK };

					edata.data = netif_carrier_ok(net);
					if (copy_to_user(rq->ifr_data, &edata, sizeof(edata)))
						return -EFAULT;
		
					return 0;
				}
				default: {
					WARN("Got unsupported ioctl: %x", usercmd);
					return -EOPNOTSUPP;	/* the ethtool user space tool relies on this */
				}
			}
		default:
			return -ENOTTY;	/* per ioctl man page */
	}
}

//=============================================================================
//	function:	hso_net_get_stats
//	purpose:	
//=============================================================================
static struct net_device_stats *hso_net_get_stats(struct net_device *net)
{

	//printk("%s called\n", __FUNCTION__);
	return &((struct hso_net *) netdev_priv(net))->stats;
}

//=============================================================================
//	function:	hso_net_tx_timeout
//	purpose:	called when a packet did not ack after watchdogtimeout
//=============================================================================
static void hso_net_tx_timeout(struct net_device *net)
{
	struct hso_net *odev = netdev_priv(net);

	//printk("%s called\n", __FUNCTION__);
	if (!odev) {
		return;
	}

	/* Tell syslog we are hosed. */
	WARN("%s: Tx timed out.", net->name);

	/* Tear the waiting frame off the list */
	if( odev->mux_bulk_tx_urb && (odev->mux_bulk_tx_urb->status == -EINPROGRESS) ) 
		usb_unlink_urb(odev->mux_bulk_tx_urb);

	/* Update statistics */
	odev->stats.tx_errors++;
}

//=============================================================================
//	function:	hso_net_set_multicast
//	purpose:	setup the multicast filters
//=============================================================================
static void hso_net_set_multicast(struct net_device *net)
{
	struct hso_net *odev = netdev_priv(net);
	int i = 0;
	__u8 *buf = NULL;

	//printk("%s called\n", __FUNCTION__);
	/* Tell the kernel to stop sending us frames while we get this all set up. */
	netif_stop_queue(net);

	/* Note: do not reorder, GCC is clever about common statements. */
	if (net->flags & IFF_PROMISC) {
		/* Unconditionally log net taps. */
		D1("%s: Promiscuous mode enabled", net->name);
		odev->mode_flags = 	MODE_FLAG_ALL_MULTICAST | MODE_FLAG_DIRECTED |
					MODE_FLAG_BROADCAST | MODE_FLAG_MULTICAST | 
					MODE_FLAG_PROMISCUOUS;
	} else if (net->mc_count > odev->wNumberMCFilters) {
		/* Too many to filter perfectly -- accept all multicasts. */
		D1("%s: too many MC filters for hardware, using allmulti", net->name);
		odev->mode_flags =	MODE_FLAG_ALL_MULTICAST | MODE_FLAG_DIRECTED |
					MODE_FLAG_BROADCAST | MODE_FLAG_MULTICAST;
	} else if (net->flags & IFF_ALLMULTI) {
		/* Filter in software */
		D1("%s: using allmulti", net->name);
		odev->mode_flags =	MODE_FLAG_ALL_MULTICAST | MODE_FLAG_DIRECTED |
					MODE_FLAG_BROADCAST | MODE_FLAG_MULTICAST;
	} else {
		/* do multicast filtering in hardware */
		struct dev_mc_list *mclist;
		D1("%s: set multicast filters", net->name);
		odev->mode_flags =	MODE_FLAG_ALL_MULTICAST | MODE_FLAG_DIRECTED |
					MODE_FLAG_BROADCAST | MODE_FLAG_MULTICAST;
		if (!(buf = kmalloc(6 * net->mc_count, GFP_ATOMIC))) {
			ERR("No memory to allocate?");
			goto exit;
		}
		for (i = 0, mclist = net->mc_list; mclist && i < net->mc_count; i++, mclist = mclist->next) {
			memcpy(&mclist->dmi_addr, &buf[i * 6], 6);
		}
		kfree(buf);
	}

exit:
	/* Tell the kernel to start giving frames to us again. */
	netif_wake_queue(net);
}

//=============================================================================
//	function:	read_bulk_callback
//	purpose:	Moving data from usb to kernel (in interrupt state)
//=============================================================================
static void read_bulk_callback(CALLBACK_ARGS)	/* struct urb *urb */
{
	struct hso_net *odev = urb->context;
	struct net_device *net = NULL;
	int result = 0;
	unsigned long flags = 0;
	int status = urb->status;

	//printk("%s called\n", __FUNCTION__);
	/* is al ok?  (Filip: Who's Al ?) */
	if (status) {
		log_usb_status(status, __FUNCTION__);
		return;
	}

	/* Sanity check */
	if (!odev || !test_bit(HSO_NET_RUNNING, &odev->flags)) {
		D1("BULK IN callback but driver is not active!");
		return;
	}

#ifdef USE_SUSPEND
	usb_mark_last_busy( urb->dev );
#endif
	
	net = odev->net;

	if (!netif_device_present(net)) {
		/* Somebody killed our network interface... */
		return;
	}

	if( odev->parent->port_spec & HSO_INFO_CRC_BUG ) {
		u32 rest;
		u8 crc_check[4] = { 0xDE, 0xAD, 0xBE, 0xEF };
		rest = urb->actual_length % odev->in_endp->wMaxPacketSize;
		if( ((rest == 5) || (rest == 6)) && !memcmp( ((u8*)urb->transfer_buffer) + urb->actual_length -4, 
												crc_check, 4)) {
			urb->actual_length -= 4;
		}
	}
	

	/* do we even have a packet? */
	if (urb->actual_length) {
		/* Handle the IP stream, add header and push it onto network stack if the packet is complete. */
		spin_lock_irqsave(&odev->net_lock,flags);
		packetizeRx(odev, urb->transfer_buffer, urb->actual_length, 
				(urb->transfer_buffer_length > urb->actual_length)?1:0 );
		spin_unlock_irqrestore(&odev->net_lock,flags);
	}

	/* We are done with this URB, resubmit it. Prep the USB to wait for another frame. Reuse same as received. */
	usb_fill_bulk_urb(	urb,
				odev->parent->usb,
				usb_rcvbulkpipe(odev->parent->usb, odev->in_endp->bEndpointAddress & 0x7F ),
				urb->transfer_buffer,
				MUX_BULK_RX_BUF_SIZE,
				read_bulk_callback,
				odev);

	/* Give this to the USB subsystem so it can tell us when more data arrives. */
	if ((result = usb_submit_urb(urb, GFP_ATOMIC))) {
		WARN("%s failed submit mux_bulk_rx_urb %d", __FUNCTION__, result);
	}
}

//=============================================================================
//	function:	packetizeRx
//	purpose:	make a real packet from the received USB buffer
//=============================================================================
static void packetizeRx(struct hso_net *odev, unsigned char *ip_pkt, unsigned int count, unsigned char is_eop)
{
	unsigned short temp_bytes = 0;
	unsigned short buffer_offset = 0;
	unsigned short frame_len = 0;
	unsigned char *tmp_rx_buf = NULL;
	struct ethhdr *eth_head = NULL;

	//printk("%s called\n", __FUNCTION__);
	/* log if needed */
	D("Rx %d bytes", count);
	DUMP(ip_pkt, min(128, count));

	while (count) {
		switch (odev->rx_parse_state) {
			case WAIT_IP:	/* waiting for IP header. */
				/* wanted bytes - size of ip header */
				temp_bytes = (count < odev->rx_buf_missing) ? count : odev->rx_buf_missing;
	
				memcpy(((unsigned char *)(&odev->rx_ip_hdr)) + odev->rx_buf_size, ip_pkt + buffer_offset, temp_bytes);
	
				odev->rx_buf_size += temp_bytes;
				buffer_offset += temp_bytes;
				odev->rx_buf_missing -= temp_bytes;
				count -= temp_bytes;
	
				if (!odev->rx_buf_missing) {	/* header is complete allocate an sk_buffer and continue to WAIT_DATA */
					frame_len = ntohs(odev->rx_ip_hdr.tot_len);
	
					if ( (frame_len > DEFAULT_MRU ) || ( frame_len < sizeof(struct iphdr)) ) {
						printk("Invalid frame (%d) length", frame_len);
						odev->rx_parse_state = WAIT_SYNC;
						continue;
					}

					/* Allocate an sk_buff */
					if (!(odev->skb_rx_buf = dev_alloc_skb(frame_len + 2 + odev->net->hard_header_len))) {
						/* We got no receive buffer. */
						D("could not allocate memory");
						odev->rx_parse_state = WAIT_SYNC;
						return;
					}
					/* Here's where it came from */
					odev->skb_rx_buf->dev = odev->net;
	
					/* Make some headroom: standard alignment + the ethernet header. */
					skb_reserve(odev->skb_rx_buf, 2 + odev->net->hard_header_len);
	
					/* Copy what we got so far. make room for iphdr after tail. */
					tmp_rx_buf = skb_put(odev->skb_rx_buf, sizeof(struct iphdr));
					memcpy(tmp_rx_buf, (char *)&(odev->rx_ip_hdr), sizeof(struct iphdr));
	
					/* ETH_HLEN */
					odev->rx_buf_size = odev->net->hard_header_len + sizeof(struct iphdr);
	
					/* Filip actually use .tot_len */
					odev->rx_buf_missing = frame_len - sizeof(struct iphdr);
					odev->rx_parse_state = WAIT_DATA;
				}
				break;
	
			case WAIT_DATA:
				temp_bytes = (count < odev->rx_buf_missing) ? count : odev->rx_buf_missing;
	
				/* Copy the rest of the bytes that are left in the buffer into the waiting sk_buf. */
				/* Make room for temp_bytes after tail. */
				tmp_rx_buf = skb_put(odev->skb_rx_buf, temp_bytes);
				memcpy(tmp_rx_buf, ip_pkt + buffer_offset, temp_bytes);
	
				odev->rx_buf_missing -= temp_bytes;
				count -= temp_bytes;
				buffer_offset += temp_bytes;
				odev->rx_buf_size += temp_bytes;
				if (!odev->rx_buf_missing) { /* Packet is complete. Inject into stack. */
					{
						/* Add fake ethernet header. */
						eth_head = (struct ethhdr *)skb_push(odev->skb_rx_buf, odev->net->hard_header_len);
						memcpy(eth_head, &odev->dummy_eth_head, sizeof(struct ethhdr));
	
						/* Not sure here either */
						odev->skb_rx_buf->protocol = eth_type_trans(odev->skb_rx_buf, odev->net);
						odev->skb_rx_buf->ip_summed = CHECKSUM_UNNECESSARY;	/* don't check it */
						/* Ship it off to the kernel */
						netif_rx(odev->skb_rx_buf);
						/* No longer our buffer. */
						odev->skb_rx_buf = NULL;
	
						/* update out statistics */
						odev->stats.rx_packets++;
	
						/* Hmmm, wonder if we have received the IP len or the ETH len. */
						odev->stats.rx_bytes += odev->rx_buf_size;
					}
					odev->rx_buf_size = 0;
					odev->rx_buf_missing = sizeof(struct iphdr);
					odev->rx_parse_state = WAIT_IP;
				}
				break;
	
			case WAIT_SYNC:
				D(" W_S");
				count = 0;
				break;
			default:
				D(" ");
				count--;
				break;
		}
	}
	//Recovery mechanism for WAIT_SYNC state.
	if( is_eop ) {
		if( odev->rx_parse_state == WAIT_SYNC ) {
			odev->rx_parse_state = WAIT_IP; odev->rx_buf_size = 0;
			odev->rx_buf_missing = sizeof(struct iphdr);
		} 
	}
}

//=============================================================================
//	function:	write_bulk_callback
//	purpose:	USB tells is xmit done, we should start the netqueue again
//=============================================================================
static void write_bulk_callback(CALLBACK_ARGS)	/* struct urb *urb */
{
	struct hso_net *odev = urb->context;
	int status = urb->status;

	//printk("%s called\n", __FUNCTION__);
	/* Sanity check */
	if (!odev || !test_bit(HSO_NET_RUNNING, &odev->flags)) {
		ERR("write_bulk_callback: device not running");
		return;
	}

	/* Do we still have a valid kernel network device? */
	if (!netif_device_present(odev->net)) {
		ERR("write_bulk_callback: net device not present");
		return;
	}

	/* log status, but don't act on it, we don't need to resubmit anything anyhow */
	if (status) {
		log_usb_status(status, __FUNCTION__);
	}

#ifdef USE_SUSPEND
	hso_put_activity( odev->parent );
#endif
	/* Tell the network interface we are ready for another frame */
	netif_wake_queue(odev->net);
}


//*****************************************************************************
//*****************************************************************************
// Serial driver functions
//*****************************************************************************
//*****************************************************************************

//=============================================================================
//	function:	hso_serial_open
//	purpose:	open the requested serial port
//=============================================================================
static int hso_serial_open(struct tty_struct *tty, struct file *filp)
{
	struct hso_serial *serial = get_serial_by_index(tty->index);
	int result = 0;
	unsigned long flags;

	/* sanity check */
	if (serial == NULL || serial->magic != HSO_SERIAL_MAGIC ) {
		tty->driver_data = NULL;
		D("Failed to open port");
		return -ENODEV;
	}


#ifdef USE_SUSPEND
	usb_autopm_get_interface( serial->parent->interface );
#endif

	D1("Opening %d", serial->minor);



	/* lock it down */
	spin_lock_irqsave( &serial->serial_lock, flags );

	kref_get( &serial->parent->ref );
	
	/* setup */
	tty->driver_data = serial;
	serial->tty = tty;

	/* check for port allready opened, if not set the termios */
	serial->open_count++;
	if (serial->open_count == 1) {
		tty->low_latency = 1;
		serial->flags = 0;
		/* Force default termio settings */
		_hso_serial_set_termios(tty, NULL);
		if( (result = hso_start_serial_device( serial->parent)) ) {
			hso_stop_serial_device( serial->parent );
			serial->open_count--;
			kref_put( &serial->parent->ref, hso_serial_ref_free );
		}
	} else {
		D1("Port was already open");
	}

	spin_unlock_irqrestore( &serial->serial_lock, flags );

#ifdef USE_SUSPEND
	usb_autopm_put_interface( serial->parent->interface );
#endif

	if( !result ) hso_serial_tiocmset( tty, NULL , TIOCM_RTS|TIOCM_DTR, 0);
	/* done */
	return result;
}

//=============================================================================
//	function:	hso_serial_close
//	purpose:	close the requested serial port
//=============================================================================
static void hso_serial_close(struct tty_struct *tty, struct file *filp)
{
	struct hso_serial *serial = tty->driver_data;
	unsigned long flags;
	u8 usb_gone; 
	D1("Closing serial port");

	/* sanity check */
	if (tty == NULL || serial == NULL) {
		D1("(tty == NULL || tty->driver_data == NULL)");
		return;
	}

	usb_gone = test_bit( HSO_SERIAL_USB_GONE, &serial->flags );

#ifdef USE_SUSPEND
	if( !usb_gone) usb_autopm_get_interface( serial->parent->interface );
#endif

	/* reset the rts and dtr */
	/* do the actual close */
	spin_lock_irqsave( &serial->serial_lock, flags ); 
	serial->open_count--;
	if( serial->open_count == 0 ) {
		kref_put( &serial->parent->ref, hso_serial_ref_free );
		if( serial->tty ) {
			serial->tty->driver_data = NULL;
			serial->tty = NULL;
		}
		if( !usb_gone ) hso_stop_serial_device( serial->parent );
	}

	if( serial->open_count < 0 ) serial->open_count = 0;
	spin_unlock_irqrestore( &serial->serial_lock, flags );
#ifdef USE_SUSPEND
	if( !usb_gone ) usb_autopm_put_interface( serial->parent->interface );
#endif

}

//=============================================================================
//	function:	hso_serial_write
//	purpose:	close the requested serial port
//=============================================================================
static int hso_serial_write(struct tty_struct *tty, const unsigned char *buf, int count)
{
	struct hso_serial *serial = get_serial_by_tty(tty);
	int space = 0, tx_bytes = 0;
	unsigned long flags;

	/* sanity check */
	if (serial == NULL) {
		ERR("%s(%d) tty or tty->driver_data is NULL", __FUNCTION__, __LINE__);
		return -ENODEV;
	}

	spin_lock_irqsave( &serial->serial_lock, flags );

	space = serial->tx_data_length - serial->tx_buffer_count;
	tx_bytes = (count < space )? count : space;

	if( !tx_bytes ) goto out;

	memcpy( serial->tx_buffer + serial->tx_buffer_count,  buf, tx_bytes );
	serial->tx_buffer_count+= tx_bytes;

out:
	spin_unlock_irqrestore( &serial->serial_lock, flags );

	hso_kick_transmit( serial );
	/* done */
	return tx_bytes;
}

//=============================================================================
//	function:	hso_serial_write_room
//	purpose:	how much room is there for writing
//=============================================================================
static int hso_serial_write_room(struct tty_struct *tty)
{
	struct hso_serial *serial = get_serial_by_tty(tty);
	int room = 0;
	unsigned long flags;

	/* sanity check */
	if (serial == NULL) {
		return 0;
	}

	spin_lock_irqsave( &serial->serial_lock, flags );
	room = serial->tx_data_length - serial->tx_buffer_count;
	spin_unlock_irqrestore( &serial->serial_lock, flags );

	/* return free room */
	return room;
}

//=============================================================================
//	function:	hso_serial_set_termios
//	purpose:	setup the term
//=============================================================================
static void hso_serial_set_termios(struct tty_struct *tty, struct TERMIOS *old)
{
	struct hso_serial *serial = get_serial_by_tty(tty);
	unsigned long flags;

	/* sanity check */
	if ( (!serial) || (!tty->termios) ) {
		D1("no tty structures");
		return;
	}

	if(old)
		D5("Termios called with: cflags new[%d] - old[%d]", tty->termios->c_cflag, old->c_cflag);


	/* the actual setup */
	spin_lock_irqsave( &serial->serial_lock, flags );
	if (serial->open_count) {
		_hso_serial_set_termios(tty, old);
	}
	spin_unlock_irqrestore( &serial->serial_lock, flags );

	/* done */
	return;
}

//=============================================================================
//	function:	hso_serial_chars_in_buffer
//	purpose:	how many characters in the buffer
//=============================================================================
static int hso_serial_chars_in_buffer(struct tty_struct *tty)
{
	struct hso_serial *serial = get_serial_by_tty(tty);
	int chars = 0;
	unsigned long flags;

	/* sanity check */
	if(serial == NULL) {
		return 0;
	}

	spin_lock_irqsave( &serial->serial_lock, flags );
	chars = serial->tx_buffer_count;
	spin_unlock_irqrestore( &serial->serial_lock, flags );

	// D5("chars= %d ", chars);
	return chars;
}

//=============================================================================
//	function:	hso_serial_tiocmget
//	purpose:	
//=============================================================================
static int hso_serial_tiocmget(struct tty_struct *tty, struct file *file)
{
	unsigned int value = 0;
	struct hso_serial *serial = get_serial_by_tty(tty);
	unsigned long flags;

	
	/* sanity check */
	if (!serial) {
		D1("no tty structures");
		return -EINVAL;
	}


	spin_lock_irqsave( &serial->serial_lock, flags );
	value = ((serial->rts_state) ? TIOCM_RTS : 0) |
		((serial->dtr_state) ? TIOCM_DTR : 0);
	spin_unlock_irqrestore( &serial->serial_lock, flags );

	//printk("%s: rts = %d, dtr  = %d\n", __FUNCTION__, serial->rts_state, serial->dtr_state );

	return value;
}

//=============================================================================
//	function:	hso_serial_tiocmset
//	purpose:	
//=============================================================================
static int hso_serial_tiocmset(struct tty_struct *tty, struct file *file, unsigned int set, unsigned int clear)
{
	int val = 0;
	unsigned long flags;
	struct hso_serial *serial = get_serial_by_tty(tty);
	int if_num = serial->parent->interface->altsetting->desc.bInterfaceNumber;

	//printk("in %s, set = 0x%.4X, get = 0x%.4X\n", __FUNCTION__, set, clear);
	
	/* sanity check */
	if (!serial) {
		D1("no tty structures");
		return -EINVAL;
	}

	spin_lock_irqsave( &serial->serial_lock, flags );
	if (set & TIOCM_RTS)
		serial->rts_state = 1;
	if (set & TIOCM_DTR)
		serial->dtr_state = 1;

	if (clear & TIOCM_RTS)
		serial->rts_state = 0;
	if (clear & TIOCM_DTR)
		serial->dtr_state = 0;

	if (serial->dtr_state)
		val |= 0x01;
	if (serial->rts_state)
		val |= 0x02;

	spin_unlock_irqrestore( &serial->serial_lock, flags );
	
	//printk("%s: rts = %d, dtr = %d\n", __FUNCTION__, serial->rts_state, serial->dtr_state );

	return usb_control_msg(	serial->parent->usb, usb_rcvctrlpipe(serial->parent->usb, 0),
				0x22, 0x21, val, if_num, NULL, 0, USB_CTRL_SET_TIMEOUT);
}

//=============================================================================
//	function:	hso_set_suspend
//	purpose:	Toggles suspend on or off ( used by ioctl )
//=============================================================================

static int hso_set_suspend( struct hso_device *hso_dev, int enabled )
{
	if( !hso_dev ) return -ENODEV;

#ifdef USE_SUSPEND
	if( enabled ) {
		if( hso_dev->suspend_disabled ) {
			usb_autopm_put_interface( hso_dev->interface );
			hso_dev->suspend_disabled = 0;
		}
	} else {
		if( !hso_dev->suspend_disabled ) {
			usb_autopm_get_interface( hso_dev->interface );
			hso_dev->suspend_disabled = 1;
		}
	}
#else
	return -EPERM;
#endif
	return 0;
}

//=============================================================================
//	function:	hso_set_radio
//	purpose:	Toggles radioon or off ( used by ioctl )
//=============================================================================

static int hso_set_radio( struct hso_device *hso_dev, int enabled )
{
	if( !hso_dev ) return -ENODEV;

	//printk("in %s: enabled = %d\n", __FUNCTION__, enabled );

	return usb_control_msg(	hso_dev->usb, usb_rcvctrlpipe(hso_dev->usb, 0),
				enabled?0x82:0x81, 0x40, 0, 0, NULL, 0, USB_CTRL_SET_TIMEOUT);
}

//=============================================================================
//	function:	hso_serial_ioctl
//	purpose:	ioctl not supported
//=============================================================================
static int hso_serial_ioctl(struct tty_struct *tty, struct file *file, unsigned int cmd, unsigned long arg)
{
	struct hso_serial *serial =  get_serial_by_tty(tty);
	int ret = 0;
	D4("IOCTL cmd: %d, arg: %ld", cmd, arg);

	if( !serial ) return -ENODEV;
	
	switch( cmd ) {
		case SIOCSETSUSPEND:
			if( arg) hso_set_suspend( serial->parent, 1);
			else hso_set_suspend(serial->parent, 0);
			ret = 0;
		break;
		case SIOCSETRADIO:
			if( arg ) hso_set_radio( serial->parent, 1);
			else hso_set_radio( serial->parent, 0);
			ret = 0;
		break;
		default:
			ret = -ENOIOCTLCMD;
		break;
	}
	return ret;
}

//=============================================================================
//	function:	hso_serial_throttle
//	purpose:	throttle not supported
//=============================================================================
static void hso_serial_throttle(struct tty_struct *tty)
{
	D1(" ");
}

//=============================================================================
//	function:	hso_serial_unthrottle
//	purpose:	throttle not supported
//=============================================================================
static void hso_serial_unthrottle(struct tty_struct *tty)
{
	D1(" ");
}

//=============================================================================
//	function:	hso_serial_break
//	purpose:	break not supported
//=============================================================================
static void hso_serial_break(struct tty_struct *tty, int break_state)
{
	D1(" ");
}

//=============================================================================
//	function:	hso_serial_read_proc
//	purpose:	read_proc not supported
//=============================================================================
static int hso_serial_read_proc(char *page, char **start, off_t off, int count, int *eof, void *data)
{
	return 0;
}

//=============================================================================
//	function:	hso_serial_hangup
//	purpose:	hangup not supported
//=============================================================================
static void hso_serial_hangup(struct tty_struct *tty)
{
	D1("hang up");
}

//=============================================================================
//	function:	hso_kick_transmit
//	purpose:	starts a transmit
//=============================================================================
static void hso_kick_transmit( struct hso_serial *serial )
{
	u8	*temp = 0;
	unsigned long flags;

	spin_lock_irqsave( &serial->serial_lock, flags );
	if( !serial->tx_buffer_count ) {
		goto out;
	}

	if( serial->tx_urb->status == -EINPROGRESS ) {
		goto out;
	}

#ifdef USE_SUSPEND
	//Wakeup USB interface if necessary 
	if( hso_get_activity( serial->parent ) == -EAGAIN )  goto out;
#endif

	//Switch pointers around to avoid memcpy
	temp = serial->tx_buffer;
	serial->tx_buffer = serial->tx_data;
	serial->tx_data = temp;
	serial->tx_data_count = serial->tx_buffer_count;
	serial->tx_buffer_count = 0;

out:	
	//If temp is set, it means we switched buffers
	if( temp && serial->write_data ) {
		serial->write_data( serial );
	}
	spin_unlock_irqrestore( &serial->serial_lock, flags );
}

//=============================================================================
//	function:	hso_mux_to_port
//	purpose:	converts mux value to a port spec value
//=============================================================================
u32 hso_mux_to_port( int mux )
{
	u32 result;
	switch( mux ) {
		case 0x1: result = HSO_PORT_CONTROL; break;
		case 0x2: result = HSO_PORT_APP; break;
		case 0x4: result = HSO_PORT_PCSC; break;
		case 0x8: result = HSO_PORT_GPS; break;
		case 0x10: result = HSO_PORT_APP2; break;
		default: result = HSO_PORT_NO_PORT;			
	}
	return result;
}

//=============================================================================
//	function:	hso_port_to_mux
//	purpose:	converts port spec value to a mux value
//=============================================================================
u32 hso_port_to_mux( int port )
{
	u32 result;
	switch( port & HSO_PORT_MASK ) {
		case HSO_PORT_CONTROL: result = 0x0; break;
		case HSO_PORT_APP: result = 0x1; break;
		case HSO_PORT_PCSC: result = 0x2; break;
		case HSO_PORT_GPS: result = 0x3; break;
		case HSO_PORT_APP2: result = 0x4; break;
		default: result = 0x0;			
	}
	return result;
}

//=============================================================================
//	function:	intr_callback
//	purpose:	used for muxed serial port callback (muxed serial read)
//=============================================================================
static void intr_callback(CALLBACK_ARGS)	/* struct urb *urb */
{
	struct hso_shared_int *shared_int = urb->context;
	struct hso_serial *serial = NULL;
	unsigned char *port_req = NULL;
	int status = urb->status;
	int i = 0;

#ifdef USE_SUSPEND
	usb_mark_last_busy( urb->dev );
#endif
	/* sanity check */
	if (!shared_int) {
		return;
	}
	/* status check */
	if (status) {
		log_usb_status(status, __FUNCTION__);
		return;
	}
	D4("\n--- Got intr callback 0x%02X ---", status);

	/* what request? */
	port_req = urb->transfer_buffer;
	D4(" port_req = 0x%.2X\n", *port_req);
	/* loop over all muxed ports to find the one sending this */
	for (i = 0; i < 8; i++) { /* max 8 channels on MUX */
		if( *port_req & (1 << i) ) {
			serial = get_serial_by_shared_int_and_type( shared_int, (1 << i) );
			if (serial != NULL) {
				D1("Pending read interrupt on port %d\n", i);
				if (!test_and_set_bit(HSO_SERIAL_FLAG_RX_SENT, &serial->flags)) {
					/* Setup and send a ctrl req read on port i */
					hso_mux_serial_read(serial);
				} else {
					D1("Already pending a read on port %d\n", i);
				}
			}
		}
	}
	//Resubmit interrupt urb
	hso_mux_submit_intr_urb(shared_int, urb->dev, GFP_KERNEL);
}

//=============================================================================
//	function:	hso_mux_serial_read
//	purpose:	called by intr_callback when read occurs
//=============================================================================
static int hso_mux_serial_read(struct hso_serial *serial)
{
	if(NULL == serial)
		return -EINVAL;
	/* clean data */
	memset(serial->rx_data[0], 0, CTRL_URB_RX_SIZE);
	/* make the request */

	if( serial->num_rx_urbs != 1 ) {
		printk("ERROR: mux'd reads with multiple buffers not possible\n");
		return 0;
	}
	return mux_device_request(	serial,
					GET_ENCAPSULATED_RESPONSE,
					serial->parent->port_spec & HSO_PORT_MASK,
					serial->rx_urb[0],
					&serial->ctrl_req_rx,
					serial->rx_data[0], 
					serial->rx_data_length);
}

//=============================================================================
//	function:	hso_mux_serial_write_data
//	purpose:	called for writing to muxed serial port
//=============================================================================
static int hso_mux_serial_write_data(struct hso_serial *serial)
{
	if(NULL == serial)
		return -EINVAL;
	return mux_device_request(	serial,
					SEND_ENCAPSULATED_COMMAND,
					serial->parent->port_spec & HSO_PORT_MASK,
					serial->tx_urb,
					&serial->ctrl_req_tx,
					serial->tx_data, 
					serial->tx_data_count);
}

//=============================================================================
//	function:	hso_std_serial_write_data
//	purpose:	called for writing diag or CS serial port
//=============================================================================
static int hso_std_serial_write_data(struct hso_serial *serial)
{
	int count = serial->tx_data_count;
	int result = 0;

	//printk("Writing %d bytes\n", serial->tx_data_count );

	usb_fill_bulk_urb(	serial->tx_urb,
				serial->parent->usb,
				usb_sndbulkpipe(serial->parent->usb, serial->out_endp->bEndpointAddress & 0x7F),
				serial->tx_data,
				serial->tx_data_count,
				hso_std_serial_write_bulk_callback,
				serial);

	if ((result = usb_submit_urb(serial->tx_urb, GFP_KERNEL))) {
		D("Failed to submit urb - res %d ", result);
		return result;
	}

	return count;
}

//=============================================================================
//	function:	mux_device_request
//	purpose:	make a request (for reading and writing data to muxed serial port)
//=============================================================================
static int mux_device_request(struct hso_serial *serial, u8 type, u16 port, struct urb *ctrl_urb, struct usb_ctrlrequest *ctrl_req, u8 *ctrl_urb_data, u32 size)
{
	int result = 0;
	int pipe = -1;

	/* Sanity check */
	if (!serial || !ctrl_urb || !ctrl_req) {
		ERR("Wrong arguments ");
		return -EINVAL;
	}

	/* initialize */
	ctrl_req->wValue = 0;
	ctrl_req->wIndex = hso_port_to_mux( port );
	ctrl_req->wLength = size;

	if (type == GET_ENCAPSULATED_RESPONSE) {
		/* Reading command */
		ctrl_req->bRequestType = USB_DIR_IN | USB_TYPE_OPTION_VENDOR | USB_RECIP_INTERFACE;
		ctrl_req->bRequest = GET_ENCAPSULATED_RESPONSE;
		pipe = usb_rcvctrlpipe(serial->parent->usb, 0);
	} else {
		/* Writing command */
		ctrl_req->bRequestType = USB_DIR_OUT | USB_TYPE_OPTION_VENDOR | USB_RECIP_INTERFACE;
		ctrl_req->bRequest = SEND_ENCAPSULATED_COMMAND;
		pipe = usb_sndctrlpipe(serial->parent->usb, 0);
	}
	/* syslog */
	D2("%s command (%02x) len: %d, port: %d", type == GET_ENCAPSULATED_RESPONSE ? "Read" : "Write", ctrl_req->bRequestType, ctrl_req->wLength, port);

	/* Load ctrl urb */
	ctrl_urb->transfer_flags = 0;
	usb_fill_control_urb(	ctrl_urb,
				serial->parent->usb,
				pipe,
				(u8 *) ctrl_req,
				ctrl_urb_data,
				size,
				ctrl_callback,
				serial);
	/* Send it on merry way */
	if ((result = usb_submit_urb(ctrl_urb, GFP_ATOMIC))) {
		ERR("%s failed submit ctrl_urb %d type %d", __FUNCTION__, result, type);
		return result;
	}

	/* done */
	return size;
}

//=============================================================================
//	function:	ctrl_callback
//	purpose:	callback after read or write on muxed serial port
//=============================================================================
static void ctrl_callback(CALLBACK_ARGS)	/* struct urb *urb */
{
	struct hso_serial *serial = urb->context;
	struct usb_ctrlrequest *req = NULL;
	int status = urb->status;

	/* sanity check */
	if (!serial) {
		return;
	}

	if (status) {
		log_usb_status(status, __FUNCTION__);
		return;
	}

	/* what request? */
	req = (struct usb_ctrlrequest *)(urb->setup_packet);
	D4("\n--- Got muxed ctrl callback 0x%02X ---", status);
	D4("Actual length of urb = %d\n", urb->actual_length);
	DUMP1(urb->transfer_buffer, urb->actual_length);

	if (req->bRequestType == (USB_DIR_IN | USB_TYPE_OPTION_VENDOR | USB_RECIP_INTERFACE)) {
		/* response to a read command */
		if (serial->open_count > 0) {
			/* handle RX data the normal way */
			put_rxbuf_data(urb, serial);
		}

		/* Re issue a read as long as we receive data. */
		if (urb->actual_length != 0) {
			hso_mux_serial_read(serial);
		} else {
			clear_bit(HSO_SERIAL_FLAG_RX_SENT, &serial->flags);
		}
	} else {
#ifdef USE_SUSPEND
		hso_put_activity( serial->parent );
#endif	
		tty_wakeup( serial->tty );
		/* response to a write command */
		hso_kick_transmit( serial );
	}
}

//=============================================================================
//	function:	put_rxbuf_data
//	purpose:	handle RX data for serial port
//=============================================================================
static void put_rxbuf_data(struct urb *urb, struct hso_serial *serial)
{
        struct tty_struct *tty = serial->tty;

	/* Sanity check */
	if (urb == NULL || serial == NULL) {
		D1("serial = NULL");
		return;
	}

       /* Push data to tty */
        if (tty && urb->actual_length) {
		D("data to push to tty");

#if ( LINUX_VERSION_CODE < KERNEL_VERSION(2,6,16) )
		int i = 0;
		unsigned char *data = urb->transfer_buffer;

		for (i = 0; i < urb->actual_length ; ++i) {
			if (tty->flip.count >= TTY_FLIPBUF_SIZE)
				tty_flip_buffer_push(tty);
			tty_insert_flip_char(tty, data[i], 0);
		}
		tty_flip_buffer_push(tty);
#else	/* ( LINUX_VERSION_CODE < KERNEL_VERSION(2,6,16) ) */
                tty_buffer_request_room(tty, urb->actual_length);
                tty_insert_flip_string(tty, urb->transfer_buffer, urb->actual_length);
                tty_flip_buffer_push(tty);
#endif	/* ( LINUX_VERSION_CODE < KERNEL_VERSION(2,6,16) ) */
        }
}

//=============================================================================
//	function:	hso_std_serial_read_bulk_callback
//	purpose:	read callback for Diag and CS port
//=============================================================================
static void hso_std_serial_read_bulk_callback(CALLBACK_ARGS)	/* struct urb *urb */
{
	struct hso_serial *serial = urb->context;
	int result = 0;
	int status = urb->status;

	/* sanity check */
	if (!serial) {
		D("serial == NULL");
		return;
	}
	/*if(status == -EPIPE) {
		usb_unlink_urb(urb);
		usb_clear_halt(serial->odev->usb, serial->ep_bulk_in);
		// just continue, need to resubmit the urb
	}*/
        else if (status) {
		log_usb_status(status, __FUNCTION__);
                return;
        }

	D4("\n--- Got serial_read_bulk callback %02x ---", status);
	D1("Actual length = %d\n", urb->actual_length);
	DUMP1(urb->transfer_buffer, urb->actual_length);

	/* Anyone listening? */
	if (serial->open_count == 0)
		return;

	if (status == 0) {
		if( serial->parent->port_spec & HSO_INFO_CRC_BUG ) {
			u32 rest;
			u8 crc_check[4] = { 0xDE, 0xAD, 0xBE, 0xEF };
			rest = urb->actual_length % serial->in_endp->wMaxPacketSize;
			if( ((rest == 5) || (rest == 6)) && !memcmp( ((u8*)urb->transfer_buffer) + urb->actual_length -4, 
									crc_check, 4)) {
				urb->actual_length -= 4;
			}
		}
		/* Valid data, handle RX data */
		put_rxbuf_data(urb, serial);
	} else if (status == -ENOENT || status == -ECONNRESET) {
		/* Unlinked - check for throttled port. */
		D2("Port %d, successfully unlinked urb", serial->minor);
	} else {
		D2("Port %d, status = %d for read urb", serial->minor, status);
		return;
	}

#ifdef USE_SUSPEND
	usb_mark_last_busy( urb->dev );
#endif

	/* We are done with this URB, resubmit it. Prep the USB to wait for another frame */
	usb_fill_bulk_urb( urb, serial->parent->usb,
				usb_rcvbulkpipe(serial->parent->usb, serial->in_endp->bEndpointAddress & 0x7F),
				urb->transfer_buffer,
				serial->rx_data_length,
				hso_std_serial_read_bulk_callback,
				serial);
	/* Give this to the USB subsystem so it can tell us when more data arrives. */
	if ((result = usb_submit_urb( urb, GFP_ATOMIC))) {
		ERR("%s failed submit serial rx_urb %d", __FUNCTION__, result);
	}
}

//=============================================================================
//	function:	hso_std_serial_write_bulk_callback
//	purpose:	write callback for Diag and CS port
//=============================================================================
static void hso_std_serial_write_bulk_callback(CALLBACK_ARGS)	/* struct urb *urb */
{
	struct hso_serial *serial = urb->context;
	int status = urb->status;

	/* sanity check */
	if (!serial) {
		D("serial == NULL");
		return;
	}

        if (status) {
		log_usb_status(status, __FUNCTION__);
                return;
        }

#ifdef USE_SUSPEND
	hso_put_activity( serial->parent );
#endif
	tty_wakeup( serial->tty );
	hso_kick_transmit( serial );

	D1(" ");
	return;
}

//=============================================================================
//	function:	_hso_serial_set_termios
//	purpose:	
//=============================================================================
static void _hso_serial_set_termios(struct tty_struct *tty, struct TERMIOS *old)
{
	struct hso_serial *serial = get_serial_by_tty(tty);

	if ((!tty) || (!tty->termios) || (!serial)) {
		ERR("no tty structures");
		return;
	}

	D4("port %d", serial->minor);

	/*
	 * The default requirements for this device are:
	 */
	serial->tty->termios->c_iflag &= ~(	IGNBRK		/* disable ignore break */
						| BRKINT	/* disable break causes interrupt */
						| PARMRK	/* disable mark parity errors */
						| ISTRIP	/* disable clear high bit of input characters */
						| INLCR		/* disable translate NL to CR */
						| IGNCR		/* disable ignore CR */
						| ICRNL		/* disable translate CR to NL */
						| IXON);	/* disable enable XON/XOFF flow control */

	serial->tty->termios->c_oflag &= ~OPOST;		/* disable postprocess output characters */

	serial->tty->termios->c_lflag &= ~(	ECHO		/* disable echo input characters */
						| ECHONL	/* disable echo new line */
						| ICANON	/* disable erase, kill, werase, and rprnt special characters */
						| ISIG		/* disable interrupt, quit, and suspend special characters */
						| IEXTEN);	/* disable non-POSIX special characters */

	serial->tty->termios->c_cflag &= ~(	CSIZE		/* no size */
						| PARENB	/* disable parity bit */
						| CBAUD);	/* clear current baud rate */

	serial->tty->termios->c_cflag |= (	CS8		/* character size 8 bits */
						| B115200);	/* baud rate 115200 */

	/*
	 * Force low_latency on; otherwise the pushes are scheduled;
	 * this is bad as it opens up the possibility of dropping bytes
	 * on the floor.  We don't want to drop bytes on the floor. :)
	 */
	serial->tty->low_latency = 1;

	/* Notify the tty driver that the termios have changed. */
	serial->tty->ldisc.set_termios(serial->tty, NULL);
	return;
}

//*****************************************************************************
//*****************************************************************************
// Base driver functions
//*****************************************************************************
//*****************************************************************************

//=============================================================================
//	function:	hso_init
//	purpose:	called when loading the module, initializes serial
//			interface, registers usb driver and proc devices
//=============================================================================
int __init hso_init(void)
{
	int i = 0;
	int result = 0;

	/* put it in the log */
	QFO("%s", version);

        /* if procfs is enabled, initialize devices */
        if(procfs) {
                QFO("Registering procfs");
                hso_proc_dir = proc_mkdir(driver_name, NULL);
                if(hso_proc_dir)
                {
                        hso_proc_dir_devices = proc_mkdir("devices", hso_proc_dir);
                        create_proc_read_entry("options", 0, hso_proc_dir, hso_proc_options,NULL);
                }
        }

        /* Initialise the serial table semaphore and table */
        spin_lock_init(&serial_table_lock);
	for(i = 0; i < HSO_SERIAL_TTY_MINORS; i++) {
		serial_table[i] = NULL;
	}

        /* allocate our driver using the proper amount of supported minors */
        tty_drv = alloc_tty_driver(HSO_SERIAL_TTY_MINORS);
        if (!tty_drv)
                return -ENOMEM;

        /* fill in all needed values */
        tty_drv->magic = TTY_DRIVER_MAGIC;
        tty_drv->owner = THIS_MODULE;
        tty_drv->driver_name = driver_name;
        tty_drv->name = tty_filename;

        /* if major number is provided as parameter, use that one */
        if(tty_major) {
                tty_drv->major = tty_major;
        }

        tty_drv->minor_start = 0;
        tty_drv->num = HSO_SERIAL_TTY_MINORS;
        tty_drv->type = TTY_DRIVER_TYPE_SERIAL;
        tty_drv->subtype = SERIAL_TYPE_NORMAL;
#if ( LINUX_VERSION_CODE < KERNEL_VERSION(2,6,18) )
                tty_drv->flags = TTY_DRIVER_REAL_RAW | TTY_DRIVER_NO_DEVFS;
#else
                tty_drv->flags = TTY_DRIVER_REAL_RAW | TTY_DRIVER_DYNAMIC_DEV;
#endif
        tty_drv->init_termios = tty_std_termios;
        tty_drv->init_termios.c_cflag = B9600 | CS8 | CREAD | HUPCL | CLOCAL;
        tty_drv->termios = hso_serial_termios;
        tty_drv->termios_locked = hso_serial_termios_locked;
        tty_set_operations(tty_drv, &hso_serial_ops);

        /* register the tty driver */
        result = tty_register_driver(tty_drv);
        if (result) {
                err("%s - tty_register_driver failed(%d)", __FUNCTION__,result);
                return result;
        }

	/* register this module as an usb driver */
	if ((result = usb_register(&hso_driver))) {
		ERR("Could not register hso driver? error: %d", result);
		/* cleanup serial interface */
		tty_unregister_driver( tty_drv );
		return result;
	}

	/* done */
	return 0;
}

//=============================================================================
//	function:	hso_exit
//	purpose:	called when unloading the module, cleans up
//=============================================================================
void __exit hso_exit(void)
{
	QFO("Exiting hso");
	
	tty_unregister_driver( tty_drv );
	/* deregister the usb driver */
	usb_deregister(&hso_driver);
        /* if procfs is enabled, remove the entries */

        if(procfs) {
                remove_proc_entry("options", hso_proc_dir);
                remove_proc_entry("devices", hso_proc_dir);
                remove_proc_entry(driver_name, NULL);
        }


}

void hso_log_port( struct hso_device *hso_dev )
{
	char *port_type=NULL;
	char port_dev[20];

	switch( hso_dev->port_spec & HSO_PORT_MASK ) {
                        case HSO_PORT_CONTROL:  	port_type = "Control";          break;
                        case HSO_PORT_APP:      	port_type = "Application";      break;
                        case HSO_PORT_GPS:      	port_type = "GPS";              break;
                        case HSO_PORT_GPS_CONTROL:      port_type = "GPS control";      break;
                        case HSO_PORT_APP2:     	port_type = "Application2";     break;
                        case HSO_PORT_PCSC:     	port_type = "PCSC";             break;
                        case HSO_PORT_DIAG:     	port_type = "Diagnostic";       break;
                        case HSO_PORT_DIAG2:     	port_type = "Diagnostic2";     	break;
                        case HSO_PORT_MODEM:    	port_type = "Modem";            break;
			case HSO_PORT_NETWORK:  	port_type = "Network";		break;
                        default:                	port_type = "Unknown";          break;
	}
	if( (hso_dev->port_spec & HSO_PORT_MASK) == HSO_PORT_NETWORK ) {
		sprintf( port_dev, "%s", dev2net(hso_dev)->net->name );
	} else sprintf( port_dev, "/dev/%s%d", tty_filename, dev2ser(hso_dev)->minor );

	printk("HSO: Found %s port %s\n", port_type, port_dev );
}

int hso_start_net_device( struct hso_device *hso_dev )
{
	int i, result = 0;
	struct hso_net *hso_net = dev2net(hso_dev);

	if( !hso_net ) return -ENODEV;

	/* send URBs for all read buffers */
	for (i = 0; i < MUX_BULK_RX_BUF_COUNT; i++) {

		/* Prep a receive URB */
		usb_fill_bulk_urb(hso_net->mux_bulk_rx_urb_pool[i],
				  hso_dev->usb,
				  usb_rcvbulkpipe(hso_dev->usb, hso_net->in_endp->bEndpointAddress & 0x7F ),
				  hso_net->mux_bulk_rx_buf_pool[i],
				  MUX_BULK_RX_BUF_SIZE,
				  read_bulk_callback, hso_net);

		/* Put it out there so the device can send us stuff */
		if ((result = usb_submit_urb(hso_net->mux_bulk_rx_urb_pool[i], GFP_KERNEL))) {
			WARN("%s failed mux_bulk_rx_urb[%d] %d", __FUNCTION__, i, result);
		}
	}

	return result;
}

int hso_stop_net_device( struct hso_device *hso_dev )
{
	int i;
	struct hso_net * hso_net = dev2net(hso_dev);

	if( !hso_net ) return -ENODEV;

        for (i = 0; i < MUX_BULK_RX_BUF_COUNT; i++) {
		if( hso_net->mux_bulk_rx_urb_pool[i] && (hso_net->mux_bulk_rx_urb_pool[i]->status == -EINPROGRESS) )
			usb_unlink_urb( hso_net->mux_bulk_rx_urb_pool[i] );
		
        }
        if( hso_net->mux_bulk_tx_urb && (hso_net->mux_bulk_tx_urb->status == -EINPROGRESS) ) 
			usb_unlink_urb( hso_net->mux_bulk_tx_urb );
	
	return 0;
}

int hso_start_serial_device( struct hso_device *hso_dev )
{
	int i, result=0;
	struct hso_serial *serial = dev2ser(hso_dev);

	if( !serial) return -ENODEV;

	/* If it is not the MUX port fill in and submit a bulk urb (already allocated in hso_serial_start) */
	if (!(serial->parent->port_spec & HSO_INTF_MUX)) {
		for( i=0 ; i<serial->num_rx_urbs ; i++ ) {
			usb_fill_bulk_urb(	serial->rx_urb[i],
						serial->parent->usb,
						usb_rcvbulkpipe(serial->parent->usb, serial->in_endp->bEndpointAddress & 0x7F ),
						serial->rx_data[i],
						serial->rx_data_length,
						hso_std_serial_read_bulk_callback,
						serial);
			if ((result = usb_submit_urb(serial->rx_urb[i], GFP_KERNEL))) {
				D("Failed to submit urb - res %d", result);
				break;
			}
		}
	} else {
		spin_lock_bh( &serial->shared_int->shared_int_lock );	
		if( !serial->shared_int->use_count ) {
			result = hso_mux_submit_intr_urb(serial->shared_int, hso_dev->usb, GFP_KERNEL); 
		}
		serial->shared_int->use_count++;
		spin_unlock_bh( &serial->shared_int->shared_int_lock );
	}

	return result;
}

int hso_stop_serial_device( struct hso_device *hso_dev)
{
	int i;
	struct hso_serial *serial = dev2ser(hso_dev);

	if( !serial ) return -ENODEV;

	for( i=0 ; i < serial->num_rx_urbs ; i++ ) {
		if( serial->rx_urb[i] && (serial->rx_urb[i]->status == -EINPROGRESS) ) {
			if( in_interrupt() )usb_unlink_urb(serial->rx_urb[i]);
			else usb_kill_urb(serial->rx_urb[i]);
		}
	}

	if( serial->tx_urb && (serial->tx_urb->status == -EINPROGRESS) )  {
		if( in_interrupt() ) usb_unlink_urb(serial->tx_urb);
		else usb_kill_urb( serial->tx_urb );
	}
	if( serial->shared_int ) {
		spin_lock_bh( &serial->shared_int->shared_int_lock );
		if( serial->shared_int->use_count && (--serial->shared_int->use_count == 0) ) {
			if( (serial->shared_int->shared_intr_urb) && (serial->shared_int->shared_intr_urb->status == -EINPROGRESS) ) {
				if( in_interrupt() ) usb_unlink_urb( serial->shared_int->shared_intr_urb );			
				else usb_kill_urb( serial->shared_int->shared_intr_urb );			
			}
		}	
		spin_unlock_bh( &serial->shared_int->shared_int_lock );
	}

	return 0;
}

int hso_serial_common_free( struct hso_serial *serial )
{
	int i;

#if ( LINUX_VERSION_CODE > KERNEL_VERSION(2,6,19) )
	if( serial->parent->dev ) {
		device_remove_file( serial->parent->dev, &dev_attr_hsotype );		
	}
#endif

	tty_unregister_device( tty_drv, serial->minor );

	for( i=0 ; i < serial->num_rx_urbs ; i++ ) {
                /* unlink and free RX URB */
                if (serial->rx_urb[i] != NULL) {
                        usb_free_urb(serial->rx_urb[i]);
                }
                /* free the RX buffer */
                if (serial->rx_data[i] != NULL) {
                        kfree(serial->rx_data[i]);
                }
        }

        /* unlink and free TX URB */
        if (serial->tx_urb != NULL) {
                usb_free_urb(serial->tx_urb);
        }
        /* free the TX buffer */
        if (serial->tx_data != NULL) {
                kfree(serial->tx_data);
        }

	return 0;
}

int hso_serial_common_create( struct hso_serial *serial, int num_urbs, int rx_size, int tx_size) 
{
	int minor, i;
#if ( LINUX_VERSION_CODE > KERNEL_VERSION(2,6,19) )
	struct device *dev=NULL;
#endif

	if((minor = get_free_serial_index()) < 0)
		goto exit;	
	
	/* register our minor number */
#if ( LINUX_VERSION_CODE > KERNEL_VERSION(2,6,19) )
	serial->parent->dev = tty_register_device(tty_drv, minor, &serial->parent->interface->dev);
	if( (dev = serial->parent->dev) ) {
		dev->driver_data = serial->parent;
		device_create_file( dev, &dev_attr_hsotype );
	}
#else
	tty_register_device(tty_drv, minor, &serial->parent->interface->dev);
#endif

	/* fill in specific data for later use */
	serial->minor = minor;
	serial->magic = HSO_SERIAL_MAGIC;
	spin_lock_init( &serial->serial_lock );
	serial->num_rx_urbs = num_urbs;

        /* RX , allocate urb and initialize */

        /* prepare our RX buffer */
        serial->rx_data_length = rx_size;
        for( i=0 ; i<serial->num_rx_urbs; i++ ) {
                if (!(serial->rx_urb[i] = usb_alloc_urb(0, GFP_KERNEL))) {
                        ERR("Could not allocate urb?");
                        goto exit;
                }
                serial->rx_urb[i]->transfer_buffer = NULL;
                serial->rx_urb[i]->transfer_buffer_length = 0;
                if (!(serial->rx_data[i] = kzalloc(serial->rx_data_length, GFP_KERNEL))) {
                        ERR("%s - Out of memory", __FUNCTION__);
                        goto exit;
                }
        }

        /* TX , allocate urb and initialize */
        if (!(serial->tx_urb = usb_alloc_urb(0, GFP_KERNEL))) {
                ERR("Could not allocate urb?");
                goto exit;
        }
        serial->tx_urb->transfer_buffer = NULL;
        serial->tx_urb->transfer_buffer_length = 0;
        /* prepare our TX buffer */
        serial->tx_data_count = 0;
        serial->tx_buffer_count = 0;
        serial->tx_data_length = tx_size;
        if (!(serial->tx_data = kzalloc(serial->tx_data_length, GFP_KERNEL))) {
                ERR("%s - Out of memory", __FUNCTION__);
                goto exit;
        }
        if (!(serial->tx_buffer = kzalloc(serial->tx_data_length, GFP_KERNEL))) {
                ERR("%s - Out of memory", __FUNCTION__);
                goto exit;
        }

	return 0;
exit:
	hso_serial_common_free( serial );
	return -1;

}

//=============================================================================
//	function:	hso_free_device
//	purpose:	Frees a general hso device
//=============================================================================
void  hso_free_device( struct hso_device *hso_dev )
{
	kfree( hso_dev );
}

//=============================================================================
//	function:	hso_create_device
//	purpose:	Creates a general hso device
//=============================================================================
struct hso_device *hso_create_device( struct usb_interface *intf, int port_spec )
{	
	struct hso_device *hso_dev;
	hso_dev = (struct hso_device *) kmalloc( sizeof(*hso_dev), GFP_ATOMIC );
	if( !hso_dev ) return NULL;
	memset( hso_dev, 0, sizeof( *hso_dev ));

	hso_dev->port_spec = port_spec;
	hso_dev->usb = interface_to_usbdev( intf );
	hso_dev->interface = intf;
	kref_init( &hso_dev->ref );

#ifdef USE_SUSPEND
	INIT_WORK( &hso_dev->async_get_intf, async_get_intf );
	INIT_WORK( &hso_dev->async_put_intf, async_put_intf );
#endif

 
	return hso_dev;
}

//=============================================================================
//	function:	hso_free_net_device
//	purpose:	Frees our network device
//=============================================================================
void hso_free_net_device ( struct hso_device *hso_dev )
{
	int i;
	struct hso_net *hso_net = dev2net(hso_dev);

	if( !hso_net ) return;

        if(procfs) {
                remove_proc_entry(hso_net->net->name, hso_proc_dir_devices);
        }

       /* start freeing */
        for (i = 0; i < MUX_BULK_RX_BUF_COUNT; i++) {
                if (hso_net->mux_bulk_rx_urb_pool[i]) {
                        usb_free_urb(hso_net->mux_bulk_rx_urb_pool[i]);
                }
                if (hso_net->mux_bulk_rx_buf_pool[i]) {
                        kfree(hso_net->mux_bulk_rx_buf_pool[i]);
                }
        }
        if (hso_net->mux_bulk_tx_urb) {
                usb_free_urb(hso_net->mux_bulk_tx_urb);
        }
        if (hso_net->mux_bulk_tx_buf) {
                kfree(hso_net->mux_bulk_tx_buf);
        }

	remove_net_device( hso_net->parent );

        if(hso_net->net) {
		unregister_netdev(hso_net->net);
                free_netdev(hso_net->net);
                }

        hso_free_device( hso_dev );
}

//=============================================================================
//	function:	hso_create_net_device
//	purpose:	Creates our network device
//=============================================================================
struct hso_device *hso_create_net_device( struct usb_interface *interface )
{
	int result, i;
	struct net_device *net     = NULL;
	struct hso_net *hso_net    = NULL;
	struct hso_device *hso_dev = NULL;

	hso_dev = hso_create_device( interface, HSO_INTF_MUX|HSO_PORT_NETWORK);
	if( !hso_dev ) return NULL;

	/* allocate our network device, then we can put in our private data */
	/* call hso_net_init to do the basic initialization */
	net = alloc_netdev(sizeof(struct hso_net), "hso%d", hso_net_init);
	if(!net) {
		printk("Unable to create ethernet device");
		goto exit;
	}

	
	hso_net = netdev_priv( net );

	hso_dev->port_data.dev_net 	= hso_net;
	hso_net->net 			= net;
	hso_net->parent 		= hso_dev;

	if( !(hso_net->in_endp = hso_get_ep( interface, USB_ENDPOINT_XFER_BULK, USB_DIR_IN ))) {
		printk("Can't find BULK IN endpoint\n");
		goto exit;
	}
	if( !(hso_net->out_endp = hso_get_ep( interface, USB_ENDPOINT_XFER_BULK, USB_DIR_OUT ))) {
		printk("Can't find BULK OUT endpoint\n");
		goto exit;
	}

#ifdef SYSFS_NET
	SET_NETDEV_DEV( net, &interface->dev );
#endif

	/* registering our net device */
	result = register_netdev(net);
	if(result) {
		printk("Failed to register device\n");
		goto exit;
	}

	/* start allocating */
	for(i = 0; i < MUX_BULK_RX_BUF_COUNT; i++) {
		if (!(hso_net->mux_bulk_rx_urb_pool[i] = usb_alloc_urb(0, GFP_KERNEL))) {
			printk("Could not allocate rx urb?");
			goto exit;
		}
		if (!(hso_net->mux_bulk_rx_buf_pool[i] = kzalloc(MUX_BULK_RX_BUF_SIZE, GFP_KERNEL))) {
			printk("Could not allocate rx buf?");
			goto exit;
		}
	}
	if (!(hso_net->mux_bulk_tx_urb = usb_alloc_urb(0, GFP_KERNEL))) {
		printk("Could not allocate tx urb?");
		goto exit;
	}
	if (!(hso_net->mux_bulk_tx_buf = kzalloc(MUX_BULK_TX_BUF_SIZE, GFP_KERNEL))) {
		printk("Could not allocate tx buf?");
		goto exit;
	}


	/* and don't forget the MAC address. */
	set_ethernet_addr(hso_net);

	add_net_device( hso_dev );

       /* setup the proc dirs and files if needed */
        if(procfs) {
                if(!hso_proc_dir_devices)
                        QFO("Warning, creating port info under root");
                create_proc_read_entry(hso_net->net->name, 0, hso_proc_dir_devices, hso_proc_port_info, hso_dev);
        }

	hso_log_port( hso_dev );

	return hso_dev;	
exit:
	hso_free_net_device( hso_dev );
	return NULL;
}

//=============================================================================
//	function:	hso_free_serial_device
//	purpose:	Frees an AT channel ( goes for both mux and non-mux )
//=============================================================================
void hso_free_serial_device( struct hso_device *hso_dev )
{
	struct hso_serial *serial = dev2ser(hso_dev);
	char device_name[20];

	if( !serial ) return;
	set_serial_by_index( serial->minor, NULL );

        if(procfs) {
		sprintf(device_name, "%s%d", tty_filename, serial->minor );
                remove_proc_entry( device_name, hso_proc_dir_devices );
        }

	hso_serial_common_free( serial );

	if( serial->shared_int ) {
		spin_lock_bh( &serial->shared_int->shared_int_lock) ;
		if( --serial->shared_int->ref_count == 0 ) {
			hso_free_shared_int( serial->shared_int );	
		}
		spin_unlock_bh( &serial->shared_int->shared_int_lock );
	}
	kfree( serial );
	hso_free_device( hso_dev );	
}

//=============================================================================
//	function:	hso_create_bulk_serial_device
//	purpose:	Creates a bulk AT channel
//=============================================================================
struct hso_device *hso_create_bulk_serial_device( struct usb_interface *interface, int port )
{
	struct hso_device *hso_dev = NULL;	
	struct hso_serial *serial = NULL;
	char device_name[20];
	int num_urbs;

	hso_dev = hso_create_device( interface, port);
	if( !hso_dev ) return NULL;

	serial = ( struct hso_serial *) kmalloc( sizeof(*serial), GFP_KERNEL );
	if( !serial ) goto exit;
	memset( serial, 0, sizeof(*serial));

	serial->parent = hso_dev;
	hso_dev->port_data.dev_serial = serial;

	if( port & HSO_PORT_MODEM ) num_urbs = 2;
	else num_urbs = 1;

	if( hso_serial_common_create( serial, num_urbs, BULK_URB_RX_SIZE, BULK_URB_TX_SIZE ) )
		goto exit;

	if( !(serial->in_endp = hso_get_ep( interface, USB_ENDPOINT_XFER_BULK, USB_DIR_IN ))) {
		printk("Failed to find BULK IN ep\n");
		goto exit;
	}

	if( !(serial->out_endp = hso_get_ep( interface, USB_ENDPOINT_XFER_BULK, USB_DIR_OUT ))) {
		printk("Failed to find BULK IN ep\n");
		goto exit;
	}

	serial->write_data = hso_std_serial_write_data;

	/* and record this serial */
	set_serial_by_index(serial->minor,serial);

      /* setup the proc dirs and files if needed */
        if(procfs) {
                if(!hso_proc_dir_devices)
                        QFO("Warning, creating port info under root");
		sprintf( device_name,"%s%d", tty_filename, serial->minor );
		create_proc_read_entry( device_name, 0, hso_proc_dir_devices, hso_proc_port_info, hso_dev);
        }

	hso_log_port( hso_dev );

	/* done, return it */
	return hso_dev;
exit:
	if( hso_dev && serial)
		hso_serial_common_free( serial );
	if( serial ) kfree( serial );
	if( hso_dev ) hso_free_device( hso_dev );
	return NULL;
}

//=============================================================================
//	function:	hso_create_mux_serial_device
//	purpose:	Creates a multiplexed AT channel
//=============================================================================
struct hso_device *hso_create_mux_serial_device( struct usb_interface *interface, int port, struct hso_shared_int *mux )
{
	struct hso_device *hso_dev = NULL;
	struct hso_serial *serial = NULL;
	char device_name[20];
	int port_spec = 0;

	port_spec |= HSO_INTF_MUX;
	port_spec &= ~HSO_PORT_MASK;

	port_spec |= hso_mux_to_port( port );
	if( (port_spec & HSO_PORT_MASK) == HSO_PORT_NO_PORT ) return NULL;

	hso_dev = hso_create_device( interface, port_spec );
	if( !hso_dev ) return NULL;

	serial = ( struct hso_serial *) kmalloc( sizeof(*serial), GFP_KERNEL );
	if( !serial ) goto exit;
	memset( serial, 0, sizeof(*serial));
	
	hso_dev->port_data.dev_serial = serial;
	serial->parent = hso_dev;

	if( hso_serial_common_create( serial, 1, CTRL_URB_RX_SIZE, CTRL_URB_TX_SIZE) )
		goto exit;

	serial->tx_data_length--;
	serial->write_data = hso_mux_serial_write_data;

	serial->shared_int = mux;
	spin_lock_bh( &serial->shared_int->shared_int_lock);
	serial->shared_int->ref_count++; 
	spin_unlock_bh( &serial->shared_int->shared_int_lock);

	/* and record this serial */
	set_serial_by_index(serial->minor,serial);

      /* setup the proc dirs and files if needed */
        if(procfs) {
                if(!hso_proc_dir_devices)
                        QFO("Warning, creating port info under root");
		sprintf( device_name,"%s%d", tty_filename, serial->minor );
		create_proc_read_entry( device_name, 0, hso_proc_dir_devices, hso_proc_port_info, hso_dev);
        }

	hso_log_port( hso_dev );

	/* done, return it */
	return hso_dev;

exit:
	if (serial) {
		tty_unregister_device(tty_drv,serial->minor);
		kfree(serial);
	}
	if( hso_dev ) hso_free_device( hso_dev );
	return NULL;

}

void hso_free_shared_int( struct hso_shared_int *mux )
{
	if( mux->shared_intr_urb ) usb_free_urb( mux->shared_intr_urb );
	if( mux->shared_intr_buf ) kfree( mux->shared_intr_buf );

	kfree( mux );
}

struct hso_shared_int *hso_create_shared_int( struct usb_interface *interface  )
{
	struct hso_shared_int *mux = ( struct hso_shared_int *) kmalloc( sizeof(*mux), GFP_KERNEL );	

	if( !mux ) return NULL;
	memset( mux, 0 ,sizeof(*mux) );

	if( !(mux->intr_endp = hso_get_ep( interface, USB_ENDPOINT_XFER_INT, USB_DIR_IN ))) {
		printk("Can't find INT IN endpoint\n");
		goto exit;
	}

	if (!(mux->shared_intr_urb = usb_alloc_urb(0, GFP_KERNEL))) {
		printk("Could not allocate intr urb?");
		goto exit;
	}
	if (!(mux->shared_intr_buf = kzalloc( mux->intr_endp->wMaxPacketSize, GFP_KERNEL))) {
		printk("Could not allocate intr buf?");
		goto exit;
	}

	spin_lock_init( &mux->shared_int_lock );

	return mux;

exit:
	if( mux->shared_intr_buf ) kfree( mux->shared_intr_buf );
	if( mux->shared_intr_urb ) usb_free_urb( mux->shared_intr_urb );
	kfree( mux );
	return NULL;
}

//=============================================================================
//	function:	hso_get_config_data
//	purpose:	Gets the port spec for a certain interfac
//=============================================================================
static int hso_get_config_data( struct usb_interface *interface )
{
	struct usb_device *usbdev = interface_to_usbdev(interface);
	u8 config_data[17];
	u32 if_num  = interface->altsetting->desc.bInterfaceNumber;
	s32 result = 0;

	if( usb_control_msg( usbdev, usb_rcvctrlpipe(usbdev, 0),
				0x86, 0xC0, 0, 0, config_data, 17, USB_CTRL_SET_TIMEOUT) != 0x11 ) {
		return -EIO;
	}

	switch( config_data[if_num] ) {
		case 0x0: result = 0; break;
		case 0x1: result = HSO_PORT_DIAG; break;
		case 0x2: result = HSO_PORT_GPS; break;
		case 0x3: result = HSO_PORT_GPS_CONTROL; break;
		case 0x4: result = HSO_PORT_APP; break;
		case 0x5: result = HSO_PORT_APP2; break;
		case 0x6: result = HSO_PORT_CONTROL; break;
		case 0x7: result = HSO_PORT_NETWORK; break;
		case 0x8: result = HSO_PORT_MODEM; break;
		case 0x9: result = HSO_PORT_MSD; break;
		case 0xa: result = HSO_PORT_PCSC; break;
		case 0xb: result = HSO_PORT_VOICE; break;
		default: result = 0;
	}	

	if( result ) result |= HSO_INTF_BULK;

	if( config_data[16] & 0x1 ) result |= HSO_INFO_CRC_BUG;

	return result;
}


//=============================================================================
//	function:	hso_probe
//	purpose:	called once for each interface upon device insertion */
//=============================================================================
static int hso_probe(struct usb_interface *interface, const struct usb_device_id *id)
{
	int mux, i, if_num, port_spec;
	unsigned char port_mask;
	struct hso_device *hso_dev = NULL;
	struct hso_shared_int *shared_int = NULL;

	if_num = interface->altsetting->desc.bInterfaceNumber;

	// Get the interface/port specification from either driver_info or from the device itself
	if( id->driver_info )
		port_spec = ((u32 *)(id->driver_info))[if_num];
	else port_spec = hso_get_config_data( interface );

	if( interface->cur_altsetting->desc.bInterfaceClass != 0xFF ) {
		printk("Not our interface\n");
		return -ENODEV;
	}

	// Check if we need to switch to alt interfaces prior to port configuration
	if( interface->num_altsetting > 1 ) {
		usb_set_interface( interface_to_usbdev( interface ), if_num, 1 );
	}

#ifdef USE_SUSPEND
	interface->needs_remote_wakeup = 1;
#endif

	// Allocate new hso device(s)
	switch( port_spec & HSO_INTF_MASK ) {
		case HSO_INTF_MUX: 
		{
			struct hso_device *tmp_dev = NULL;

			if( (port_spec & HSO_PORT_MASK) == HSO_PORT_NETWORK ) {
				// Create the network device
				if( !disable_net ) {
					if( !(hso_dev = hso_create_net_device( interface )) ) goto exit;
					tmp_dev = hso_dev;
				}
			}

			if( hso_get_mux_ports( interface, &port_mask ) ) 
				//TODO: de-allocate everything
				goto exit;

			if( !(shared_int = hso_create_shared_int( interface )) ) goto exit;
			for (i = 1,mux = 0; i < 0x100; i = i << 1,mux++) {
				if (port_mask & i) {
					if( !(hso_dev = hso_create_mux_serial_device( interface, i, shared_int ))) goto exit;
				}
			}

			if( tmp_dev )  {
				hso_dev = tmp_dev;
			}
		}
		break;
		
		case HSO_INTF_BULK: // It's a regular bulk interface
			if( ((port_spec & HSO_PORT_MASK) == HSO_PORT_NETWORK) && !disable_net )
				hso_dev = hso_create_net_device( interface );
			else hso_dev = hso_create_bulk_serial_device( interface, port_spec );
			if( !hso_dev) goto exit;
		break;
		default: goto exit;
	}

	usb_driver_claim_interface( &hso_driver, interface, hso_dev );

	/* save our data pointer in this device */
	usb_set_intfdata(interface, hso_dev);

#ifdef USE_SUSPEND
	if( enable_autopm == 0 ) {
		usb_autopm_get_interface( hso_dev->interface );
		hso_dev->suspend_disabled = 1;
	}
#endif

	/* done */
	return 0;

exit:
	hso_free_interface( interface );
	return -ENODEV;
}

//=============================================================================
//	function:	hso_disconnect
//	purpose:	device removed, cleaning up
//=============================================================================
static void hso_disconnect(struct usb_interface *interface)
{
	//printk("Disconnect called for interface %d\n", interface->altsetting->desc.bInterfaceNumber);

	hso_free_interface( interface );

	/* remove reference of our private data */
	usb_set_intfdata(interface,NULL);


	usb_driver_release_interface(&hso_driver, interface);
}

#ifdef USE_SUSPEND

void async_get_intf( struct work_struct *data )
{
	struct hso_device *hso_dev = container_of(data, struct hso_device, async_get_intf);
	usb_autopm_get_interface( hso_dev->interface );
}

void async_put_intf( struct work_struct *data )
{
	struct hso_device *hso_dev = container_of(data, struct hso_device, async_put_intf);
	usb_autopm_put_interface( hso_dev->interface );
}

int hso_get_activity( struct hso_device *hso_dev )
{
	//printk("hso_get_activity()\n");

	if( hso_dev->usb->state == USB_STATE_SUSPENDED ) {
		if( !hso_dev->is_active ) {
			hso_dev->is_active = 1;
			schedule_work( &hso_dev->async_get_intf);
		}
	}

	if( hso_dev->usb->state != USB_STATE_CONFIGURED ) return -EAGAIN;

	usb_mark_last_busy( hso_dev->usb );

	return 0;
}

int hso_put_activity( struct hso_device *hso_dev )
{
	//printk("hso_put_activity()\n");
	if( hso_dev->usb->state != USB_STATE_SUSPENDED ) {
		if( hso_dev->is_active ) {
			hso_dev->is_active = 0;
			schedule_work( &hso_dev->async_put_intf );
			return -EAGAIN;
		}
	}
	hso_dev->is_active = 0;
	return 0;
}


//=============================================================================
//      function:       hso_suspend
//      purpose:        called by kernel when we need to suspend device
//=============================================================================
static int hso_suspend( struct usb_interface *iface, pm_message_t message )
{
	int i, result = 0;
	
	//printk("Interface %d suspending\n", iface->altsetting->desc.bInterfaceNumber);
	// Stop all serial ports
	for( i=0 ; i<HSO_SERIAL_TTY_MINORS ; i++ ) {
		if( serial_table[i] && ( serial_table[i]->interface == iface ) ) {
			result = hso_stop_serial_device( serial_table[i] );
			if( result ) goto out;
		}
	}

	// Stop all network ports
	for( i=0 ; i<HSO_MAX_NET_DEVICES; i++ ) {
		if( network_table[i] && (network_table[i]->interface == iface ) ) {
			result = hso_stop_net_device( network_table[i] );
			if( result ) goto out;
		}
	}

out:
	return 0;
}

//=============================================================================
//      function:       hso_resume
//      purpose:        called by kernel when we need to resume device
//=============================================================================
static int hso_resume( struct usb_interface *iface )
{
	int i, result=0;
	struct hso_net *hso_net;
	
	//printk("Interface %d resuming\n", iface->altsetting->desc.bInterfaceNumber);
	// Start all serial ports
	for( i=0 ; i<HSO_SERIAL_TTY_MINORS ; i++ ) {
		if( serial_table[i] && ( serial_table[i]->interface == iface ) ) {
			if( dev2ser(serial_table[i])->open_count ) {
				result = hso_start_serial_device( serial_table[i] );
				hso_kick_transmit( dev2ser(serial_table[i]));
				if( result ) goto out;
			} 
		}
	}

	// Start all network ports
	for( i=0 ; i<HSO_MAX_NET_DEVICES; i++ ) {
		if( network_table[i] && (network_table[i]->interface == iface ) ) {
			hso_net = dev2net(network_table[i]);
			//First transmit any lingering data, then restart the device.
			if( hso_net->skb_tx_buf ) {
				printk("Transmitting lingering data\n");
				hso_net_start_xmit( hso_net->skb_tx_buf, hso_net->net );	
			}
			result = hso_start_net_device( network_table[i] );
			if( result ) goto out;
		}	
	}
	
out:
	return result;
}
#endif /* USE_SUSPEND */

void hso_serial_ref_free( struct kref *ref )
{
	struct hso_device *hso_dev = container_of( ref, struct hso_device, ref );

	hso_free_serial_device( hso_dev );	
}


void hso_free_interface( struct usb_interface *interface )
{
	int i;

	for( i=0 ; i<HSO_SERIAL_TTY_MINORS ; i++ )
	{
		if( serial_table[i] && (serial_table[i]->interface == interface) ) {
			if( dev2ser(serial_table[i])->tty) {
				tty_hangup( dev2ser(serial_table[i])->tty);
			}
			set_bit( HSO_SERIAL_USB_GONE, &dev2ser(serial_table[i])->flags );
			kref_put( &serial_table[i]->ref, hso_serial_ref_free );
		}
	}

	for(i=0 ; i<HSO_MAX_NET_DEVICES ; i++ )
	{
		if( network_table[i] && (network_table[i]->interface == interface) ) {
			// hso_stop_net_device doesn't stop the net queue since traffic needs to start it again when suspended
			netif_stop_queue( dev2net(network_table[i])->net );
			hso_stop_net_device( network_table[i] );
			hso_free_net_device( network_table[i] );
		}
	}
}

//*****************************************************************************
//*****************************************************************************
// Helper functions
//*****************************************************************************
//*****************************************************************************

//=============================================================================
//	function:	hso_get_ep
//	purpose:	Get the endpoint !
//=============================================================================
inline struct usb_endpoint_descriptor *hso_get_ep( struct usb_interface *intf, int type, int dir )
{
	int i;
	struct usb_host_interface *iface = intf->cur_altsetting;
        struct usb_endpoint_descriptor *endp = NULL;

	for( i=0 ; i< iface->desc.bNumEndpoints; i++ )
	{
                endp = &iface->endpoint[i].desc;
		if( (( endp->bEndpointAddress & USB_ENDPOINT_DIR_MASK ) == dir ) &&
		    ((endp->bmAttributes & USB_ENDPOINT_XFERTYPE_MASK) == type) )
			return endp;
	}

	return NULL;
}
//=============================================================================
//	function:	hso_get_mux_ports
//	purpose:	Get the byte that describes which ports are enabled
//=============================================================================
int hso_get_mux_ports( struct usb_interface *intf , unsigned char *ports)
{
	int i;
	struct usb_host_interface *iface = intf->cur_altsetting;

	if( iface->extralen == 3 ) {
		*ports = iface->extra[2];
		return 0;
	}

        for (i = 0; i < iface->desc.bNumEndpoints; i++) {
                if (iface->endpoint[i].extralen == 3) {
			*ports = iface->endpoint[i].extra[2];
			return 0;
		}
	}
		
	return -1;	
}

//=============================================================================
//	function:	hso_mux_submit_intr_urb
//	purpose:	interrupt urb needs to be submitted, used for serial read of muxed port
//=============================================================================
static int hso_mux_submit_intr_urb(struct hso_shared_int *shared_int,struct usb_device * usb, GFP_T gfp)
{
	int result = 0;

	usb_fill_int_urb(	shared_int->shared_intr_urb,
				usb,
				usb_rcvintpipe(usb, shared_int->intr_endp->bEndpointAddress & 0x7F ),
				shared_int->shared_intr_buf,
				shared_int->intr_endp->wMaxPacketSize,
				intr_callback,
				shared_int, 
				shared_int->intr_endp->bInterval ); //macro is already converting the value

	if ((result = usb_submit_urb(shared_int->shared_intr_urb, gfp))) {
		WARN("%s failed mux_intr_urb %d", __FUNCTION__, result);
		goto exit;
	}

exit:
	return result;
}


//=============================================================================
//	function:	hso_net_init
//	purpose:	initialize the network interface
//=============================================================================
static void hso_net_init(struct net_device *net)
{
	struct hso_net *hso_net = netdev_priv(net);

	/* initialize to zero */
	memset(hso_net,0,sizeof(*hso_net));
	D("sizeof hso_net is %d",sizeof(*hso_net));

	/* most of the setup is done by standard function */
	ether_setup(net);

	/* fill in the other fields */
	SET_MODULE_OWNER(net);
	net->open		= hso_net_open;
	net->stop		= hso_net_close;
	net->hard_start_xmit	= hso_net_start_xmit;
	net->do_ioctl		= hso_net_ioctl;
	net->get_stats		= hso_net_get_stats;
	net->tx_timeout		= hso_net_tx_timeout;
	net->watchdog_timeo	= HSO_NET_TX_TIMEOUT;
	net->set_multicast_list	= hso_net_set_multicast;
	net->flags		|= IFF_NOARP;
	net->mtu		= DEFAULT_MTU - 14;
	net->tx_queue_len	= 10;
	hso_net->skb_rx_buf	= NULL;
	hso_net->rx_parse_state	= WAIT_IP;
	hso_net->wMaxSegmentSize = DEFAULT_MTU;
	hso_net->wNumberMCFilters	= 0;	/* always use all multi, no lists of multicasts */

	/* and initialize the semaphore */
        spin_lock_init(&hso_net->net_lock);
}

//=============================================================================
//	function:	set_ethernet_addr
//	purpose:	setting the mac-address of the newly created iface
//=============================================================================
static void set_ethernet_addr(struct hso_net *odev)
{
	unsigned char mac_addr[6];
	unsigned char dummy_mac[6];
	int i = 0;
	int len = 0;
	unsigned char buffer[13];
	unsigned char checkserial[7] = "Serial";

	/* we can't fail, therefor we use a default macaddress (constructed from the pointer of our struct) */
	mac_addr[0] = 0x00;
	mac_addr[1] = 0x03;
	mac_addr[2] = (unsigned char)(((unsigned long)odev) & 0xFF);
	mac_addr[3] = (unsigned char)((((unsigned long)odev) >> 8) & 0xFF);
	mac_addr[4] = (unsigned char)((((unsigned long)odev) >> 16) & 0xFF);
	mac_addr[5] = (unsigned char)((((unsigned long)odev) >> 24) & 0xFF);
	dummy_mac[0] = 0xFA;
	dummy_mac[1] = mac_addr[1];
	dummy_mac[2] = mac_addr[2];
	dummy_mac[3] = mac_addr[3];
	dummy_mac[4] = mac_addr[4];
	dummy_mac[5] = mac_addr[5];

	/* but we like consistency, that's why we try to use the serial number as a macaddress if available */
	if ((len = usb_string(odev->parent->usb, odev->parent->usb->descriptor.iSerialNumber, buffer, 13)) != 12) {
		/* some devices don't have the serial filled in, checking ... */
		for(i = 0; i < 6; i++)
		{
			if(buffer[i] != checkserial[i]) {
				/* Fill in the mac_addr */
				for (i = 2; i < 6; i++) {
					if ((16 == buffer[2 * i]) || (16 == buffer[2 * i + 1])) {
						ERR("Bad value in MAC address i:%d", i);
					} else {
						mac_addr[i] = (hex2dec(buffer[2 * i]) << 4) + hex2dec(buffer[2 * i + 1]);
					}
				}
				break;
			}
		}
	} else {
		ERR("Attempting to get MAC address failed: using default");
	}

	/* Now copy it over to our network device structure */
	memcpy(odev->net->dev_addr, mac_addr, sizeof(mac_addr));

	/* Create the default fake ethernet header. */
	memcpy(odev->dummy_eth_head.h_dest, mac_addr, ETH_ALEN);
	memcpy(odev->dummy_eth_head.h_source, dummy_mac, ETH_ALEN);
	odev->dummy_eth_head.h_proto = htons(ETH_P_IP);
}

//=============================================================================
//	function:	add_net_device	
//	purpose:	Adds a network device in the network device table
//=============================================================================
static int add_net_device( struct hso_device *hso_dev )
{
	int i;

	for( i=0 ; i<HSO_MAX_NET_DEVICES; i++ ) {
		if( network_table[i] == NULL ) {
			network_table[i] = hso_dev;
			break;
		}
	}
	if( i == HSO_MAX_NET_DEVICES ) return -1;
	return 0;
}

//=============================================================================
//	function:	remove_net_device	
//	purpose:	Removes a network device in the network device table
//=============================================================================
static int remove_net_device( struct hso_device *hso_dev )
{
	int i;

	for( i=0 ; i<HSO_MAX_NET_DEVICES; i++ ) {
		if( network_table[i] == hso_dev) {
			network_table[i] = NULL;
			break;
		}
	}
	if( i == HSO_MAX_NET_DEVICES ) return -1;
	return 0;
}

static struct hso_serial *get_serial_by_shared_int_and_type( struct hso_shared_int *shared_int, int mux )
{
	int i, port;

	port = hso_mux_to_port(mux);

	for( i=0 ; i< HSO_SERIAL_TTY_MINORS ; i++ ) {
		if( serial_table[i] && (dev2ser(serial_table[i])->shared_int == shared_int) && 
			((serial_table[i]->port_spec & HSO_PORT_MASK) == port) ) {
			return dev2ser(serial_table[i]);
		}		
	}

	return NULL;
}

//=============================================================================
//	function:	get_serial_by_index
//	purpose:	
//=============================================================================
static struct hso_serial *get_serial_by_index(unsigned index)
{
        struct hso_serial *serial = NULL;
	unsigned long flags;

	if( !serial_table[index] ) return NULL;
        spin_lock_irqsave(&serial_table_lock, flags);
        serial = dev2ser(serial_table[index]);
        spin_unlock_irqrestore(&serial_table_lock, flags);

        return serial;
}

//=============================================================================
//	function:	get_free_serial_index
//	purpose:	
//=============================================================================
static int get_free_serial_index(void)
{
	int index = 0;
	unsigned long flags;

        spin_lock_irqsave(&serial_table_lock, flags);
	for(index = 0;index < HSO_SERIAL_TTY_MINORS;index++) {
		if( serial_table[index] == NULL ) {
			spin_unlock_irqrestore(&serial_table_lock, flags);
			return index;
		}
	}
	spin_unlock_irqrestore(&serial_table_lock, flags);

	ERR("no free serial devices in table");
        return -1;
}

//=============================================================================
//	function:	set_serial_by_index
//	purpose:	
//=============================================================================
static void set_serial_by_index(unsigned index,struct hso_serial *serial)
{
	unsigned long flags;
        spin_lock_irqsave(&serial_table_lock, flags);
	if( serial )
		serial_table[index] = serial->parent;
	else
		serial_table[index] = NULL;
        spin_unlock_irqrestore(&serial_table_lock, flags);
}

//=============================================================================
//	function:	log_usb_status
//	purpose:	log a meaningfull explanation of an USB status to syslog
//=============================================================================
static void log_usb_status(int status, const char* function)
{
	char* explanation = NULL;
	switch (status) {
		case -ENODEV:
			explanation = "no device";
			break;
		case -ENOENT:
			explanation = "endpoint not enabled";
			break;
		case -EPIPE:
			explanation = "endpoint stalled";
			break;
		case -ENOSPC:
			explanation = "not enough bandwidth";
			break;
		case -ESHUTDOWN:
			explanation = "device disabled";
			break;
		case -EHOSTUNREACH:
			explanation = "device suspended";
			break;
		case -EINVAL:
		case -EAGAIN:
		case -EFBIG:
		case -EMSGSIZE:
			explanation = "internal error";
			break;
		default:
			explanation = "unknown status";
			break;
	}
	D("%s: received USB status - %s (%d)", function, explanation, status);
}


//*****************************************************************************
//*****************************************************************************
// Module definitions
//*****************************************************************************
//*****************************************************************************
module_init(hso_init);
module_exit(hso_exit);

MODULE_AUTHOR(MOD_AUTHOR);
MODULE_DESCRIPTION(MOD_DESCRIPTION);
MODULE_LICENSE(MOD_LICENSE);
MODULE_DEVICE_TABLE(usb, hso_ids);

/* Module parameter to allow procfs (eg: insmod hso.ko procfs=1) */
MODULE_PARM_DESC(procfs, "Allow procfs [0 | 1]");
module_param(procfs, int, S_IRUGO | S_IWUSR);
/* Module parameter to change the debug level (eg: insmod hso.ko debug=0x04) */
MODULE_PARM_DESC(debug, "Level of debug [0x01 | 0x02 | 0x04 | 0x08 | 0x10]");
module_param(debug, int, S_IRUGO | S_IWUSR);
/* Module parameter to set the major tty number (eg: insmod hso.ko tty_major=245) */
MODULE_PARM_DESC(tty_major, "Set the major tty number");
module_param(tty_major, int, S_IRUGO | S_IWUSR);
/* Module parameter to set the major tty number (eg: insmod hso.ko tty_major=245) */
MODULE_PARM_DESC(disable_net, "Disable the network interface");
module_param(disable_net, int, S_IRUGO | S_IWUSR);
MODULE_INFO(Version, DRIVER_VERSION);
/* Module parameter to enable autosuspend (default disabled) (eg: insmod hso.ko enable_autopm=1) */
MODULE_PARM_DESC(enable_autopm, "Enable the auto power managment (autosuspend)");
module_param(enable_autopm, int, S_IRUGO | S_IWUSR);
MODULE_INFO(Flags, sDEBUG sUSE_SUSPEND sSYSFS_NET );
