/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 2; tab-width: 2 -*- */
/*
 * Copyright (C) 2009 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 */
/**
 * SECTION:launcher-session
 * @short_description: Handles acquiring data from the current session
 * @include: launcher-session.h
 *
 * #LauncherSession objects exist to query data from the current session,
 * Examples of this type of data include the workarea size and currently running applications
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gdk/gdkx.h>
#include <X11/Xatom.h>
#include <X11/Xutil.h>
#include <libwncksync/libwncksync.h>

#include "launcher-application.h"
#include "launcher-appman.h"

#include "launcher-session.h"

G_DEFINE_TYPE (LauncherSession, launcher_session, G_TYPE_OBJECT)

#define LAUNCHER_SESSION_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE(obj, \
        LAUNCHER_TYPE_SESSION, LauncherSessionPrivate))

#define LAUNCHERAPP_ID "launcher-application-id"

struct _LauncherSessionPrivate
{
  /* Application variables */
  WnckScreen *screen;
  GSList     *new_windows;
  int         callback_count;
};

enum
{
  APP_OPENED,
  APP_LAUNCHING,

  LAST_SIGNAL
};

static guint _session_signals[LAST_SIGNAL] = { 0 };

static LauncherSession *launcher_session = NULL;

/* Forwards */
static void on_window_opened (WnckScreen      *screen,
                              WnckWindow      *window,
                              LauncherSession *session);

static void on_window_closed (WnckScreen      *screen,
                              WnckWindow      *window,
                              LauncherSession *session);

/* GObject Init */
static void
launcher_session_finalize (GObject *session)
{
  LauncherSessionPrivate *priv;

  g_return_if_fail (LAUNCHER_IS_SESSION (session));
  priv = LAUNCHER_SESSION (session)->priv;

  /* Note: We don't ref/unref priv->screen as-per-wnck-docs */
  priv->screen = NULL;

  G_OBJECT_CLASS (launcher_session_parent_class)->finalize (session);
}

static void
launcher_session_class_init (LauncherSessionClass *klass)
{
  GObjectClass *obj_class = G_OBJECT_CLASS (klass);

  obj_class->finalize = launcher_session_finalize;

  _session_signals[APP_LAUNCHING] =
    g_signal_new ("application-launching",
                  G_OBJECT_CLASS_TYPE (obj_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (LauncherSessionClass, application_launching),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__OBJECT,
                  G_TYPE_NONE, 1, LAUNCHER_TYPE_APPLICATION);

  _session_signals[APP_OPENED] =
    g_signal_new ("application-opened",
                  G_OBJECT_CLASS_TYPE (obj_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (LauncherSessionClass, application_opened),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__OBJECT,
                  G_TYPE_NONE, 1, LAUNCHER_TYPE_APPLICATION);
  g_type_class_add_private (obj_class, sizeof (LauncherSessionPrivate));
}

static void
launcher_session_init (LauncherSession *session)
{
  LauncherSessionPrivate *priv;

  priv = session->priv = LAUNCHER_SESSION_GET_PRIVATE (session);

  wnck_set_client_type (WNCK_CLIENT_TYPE_PAGER);

  /* Grab WnckScreen and connect to the important signals */
  priv->screen = wnck_screen_get_default ();
  g_signal_connect (priv->screen, "window-opened",
                    G_CALLBACK (on_window_opened), session);

  g_signal_connect (priv->screen, "window-closed",
                    G_CALLBACK (on_window_closed), session);
}

LauncherSession *
launcher_session_get_default (void)
{
  if (G_UNLIKELY (!LAUNCHER_IS_SESSION (launcher_session)))
    {
      launcher_session = g_object_new (LAUNCHER_TYPE_SESSION,
                                       NULL);
      return launcher_session;
    }

  return g_object_ref (launcher_session);
}

/*
 * Private methods
 */

static void
windows_updated_callback (LauncherApplication *application,
                          LauncherSession     *session)
{
  LauncherApplication    *new_app;
  LauncherAppman         *appman;
  LauncherSessionPrivate *priv;
  GSList                 *l, *found_windows = NULL;
  WnckWindow             *window = NULL;

  g_return_if_fail (LAUNCHER_IS_SESSION (session));
  g_return_if_fail (LAUNCHER_IS_APPLICATION (application));

  priv = session->priv;

  appman = launcher_appman_get_default ();

  for (l = priv->new_windows; l; l = l->next)
    {
      window = l->data;

      if (launcher_application_owns_window (application, window))
        {
          found_windows = g_slist_prepend (found_windows, window);
          if (g_slist_length (launcher_application_get_windows (application)) == 1)
            {
              g_signal_emit (session, _session_signals[APP_OPENED], 0, application);
            }
        }
    }

  for (l = found_windows; l; l = l->next)
    {
      priv->new_windows = g_slist_remove_all (priv->new_windows, l->data);
    }

  g_slist_free (found_windows);

  priv->callback_count--;

  if (priv->callback_count <= 0)
    {
      priv->callback_count = 0;

      for (l = priv->new_windows; l; l = l->next)
        {
          new_app = launcher_appman_get_application_for_wnck_window (appman, window);
          g_signal_emit (session, _session_signals[APP_OPENED], 0, new_app);
        }

      g_slist_free (priv->new_windows);
      priv->new_windows = NULL;
    }
}

static void
on_window_opened (WnckScreen      *screen,
                  WnckWindow      *window,
                  LauncherSession *session)
{
  g_return_if_fail (LAUNCHER_IS_SESSION (session));
  g_return_if_fail (WNCK_IS_WINDOW (window));

  launcher_session_update_windows (session, window);
}


void
launcher_session_update_windows (LauncherSession *session,
                                 WnckWindow *window)
{
  LauncherSessionPrivate   *priv;
  LauncherApplication      *app;
  LauncherAppman           *appman;
  GSequence                *applications;
  GSequenceIter            *iter;
  gchar                    *desktop_file;
  WncksyncProxy            *proxy;

  // asks our launcher applications to update their window lists
  g_return_if_fail (WNCK_IS_WINDOW (window));

  if (wnck_window_is_skip_tasklist (window))
    return;

  priv = session->priv;

  priv->new_windows = g_slist_prepend (priv->new_windows, window);

  appman = launcher_appman_get_default ();
  applications = launcher_appman_get_applications (appman);

  if (g_sequence_get_length (applications) < 1)
    {
      // we don't have a launcher yet - so we never actually get the callback
      // called - so make a new launcherapplication through the sync api.
      proxy = wncksync_proxy_get_default ();
      desktop_file = wncksync_proxy_get_desktop_file (proxy,
                                                      wnck_window_get_xid (window));
      app = launcher_appman_get_application_for_desktop_file (appman,
                                                              desktop_file);
      if (LAUNCHER_IS_APPLICATION (app))
        g_signal_emit (session, _session_signals[APP_OPENED], 0, app);
    }


  for (iter = g_sequence_get_begin_iter (applications);
       !g_sequence_iter_is_end (iter);
       iter = g_sequence_iter_next (iter))
    {
      app = g_sequence_get (iter);

      priv->callback_count++;
      launcher_application_update_windows_with_callback (app,
                                                         (LauncherApplicationNotifyFinished) windows_updated_callback,
                                                         session);
    }
}

static void
on_window_closed (WnckScreen      *screen,
                  WnckWindow      *window,
                  LauncherSession *session)
{
  LauncherSessionPrivate   *priv;
  LauncherApplication      *app;
  LauncherAppman           *appman;
  GSequence                *applications;
  GSequenceIter            *iter;

  g_return_if_fail (LAUNCHER_IS_SESSION (session));
  g_return_if_fail (WNCK_IS_WINDOW (window));

  priv = session->priv;

  appman = launcher_appman_get_default ();
  applications = launcher_appman_get_applications (appman);

  for (iter = g_sequence_get_begin_iter (applications);
       !g_sequence_iter_is_end (iter);
       iter = g_sequence_iter_next (iter))
    {
      app = g_sequence_get (iter);
      launcher_application_update_windows (app);
    }
}
