/*
 * Copyright (C) 2010 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by
 *             Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "zeitgeist-index.h"
#include "zeitgeist-simple-result-set.h"
#include "zeitgeist-eggdbusconversions.h"
#include "eggzeitgeistbindings.h"

/**
 * SECTION:zeitgeist-index
 * @short_description: Query the Zeitgeist Full Text Search Extension
 * @include: zeitgeist.h
 *
 * 
 */

G_DEFINE_TYPE (ZeitgeistIndex, zeitgeist_index, G_TYPE_OBJECT);
#define ZEITGEIST_INDEX_GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE(obj, ZEITGEIST_TYPE_INDEX, ZeitgeistIndexPrivate))

typedef struct
{
  /* The connection to the ZG daemon
   * Note: The EggZeitgeistIndex is owned by the EggDBusObjectProxy! */
  EggDBusObjectProxy *index_proxy;
  EggZeitgeistIndex  *index;

} ZeitgeistIndexPrivate;

/* Property ids */
enum
{
	PROP_0,

	LAST_PROPERTY
};


/* Signature of egg_zeitgeist variants of search() */
typedef guint (*SearchFunc) (EggZeitgeistIndex        *instance,
                             EggDBusCallFlags          call_flags,
                             const gchar              *query,
                             EggZeitgeistTimeRange    *time_range,
                             EggDBusArraySeq          *event_templates,
                             guint                     offset,
                             guint                     num_events,
                             EggZeitgeistResultType    result_type,
                             GCancellable             *cancellable,
                             GAsyncReadyCallback       callback,
                             gpointer                  user_data);

typedef gboolean (*CollectEventsAndCountFunc) (EggZeitgeistIndex  *instance,
                                               EggDBusArraySeq   **out_events,
                                               guint              *out_hit_count,
                                               GAsyncResult       *res,
                                               GError            **error);

static void
zeitgeist_index_init (ZeitgeistIndex *self)
{
  ZeitgeistIndexPrivate *priv;
  EggDBusConnection   *conn;

  priv = ZEITGEIST_INDEX_GET_PRIVATE (self);
  
  /* Set up the connection to the ZG daemon */  
  conn = egg_dbus_connection_get_for_bus (EGG_DBUS_BUS_TYPE_SESSION);
  priv->index_proxy = 
    egg_dbus_connection_get_object_proxy (conn,
                                          "org.gnome.zeitgeist.Engine",
                                          "/org/gnome/zeitgeist/index/activity");
  
  priv->index = EGG_ZEITGEIST_QUERY_INTERFACE_INDEX (priv->index_proxy);
  g_object_unref (conn);
}

static void
zeitgeist_index_finalize (GObject *object)
{
  ZeitgeistIndex *index = ZEITGEIST_INDEX (object);
  ZeitgeistIndexPrivate *priv;
  
  priv = ZEITGEIST_INDEX_GET_PRIVATE (index);

  /* Note: priv->index is owned by priv->index_proxy */
  if (priv->index_proxy)
    {
      g_object_unref (priv->index_proxy);
    }
  
  G_OBJECT_CLASS (zeitgeist_index_parent_class)->finalize (object); 
}

static void
zeitgeist_index_get_property (GObject    *object,
                              guint       prop_id,
                              GValue     *value,
                              GParamSpec *pspec)
{
  ZeitgeistIndexPrivate *priv = ZEITGEIST_INDEX_GET_PRIVATE (object);

  switch (prop_id)
    {
      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        return;
        break;
    }
}

static void
zeitgeist_index_set_property (GObject      *object,
                              guint         prop_id,
                              const GValue *value,
                              GParamSpec   *pspec)
{
  ZeitgeistIndexPrivate *priv = ZEITGEIST_INDEX_GET_PRIVATE (object);

  switch (prop_id)
    {
      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        return;
        break;
    }
}


static void
zeitgeist_index_class_init (ZeitgeistIndexClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  GParamSpec   *pspec;
  
  object_class->finalize     = zeitgeist_index_finalize;
  object_class->get_property = zeitgeist_index_get_property;
  object_class->set_property = zeitgeist_index_set_property;
  
  g_type_class_add_private (object_class, sizeof (ZeitgeistIndexPrivate));
}

/* Used to marshal the async callbacks from EggDBus into ones
 * coming from this Index instance */
static void
dispatch_async_callback (GObject      *source_object,
                         GAsyncResult *res,
                         gpointer      user_data)
{ 
  gpointer            *data = (gpointer*) user_data;
  ZeitgeistIndex      *self = ZEITGEIST_INDEX (data[0]);
  GAsyncReadyCallback  callback = (GAsyncReadyCallback) data[1];
  gpointer             _user_data = data[2];

  if (callback != NULL)
    {
      callback (G_OBJECT (self), res, _user_data);
    }
  
  g_object_unref (self);
  g_free (user_data);
}

/*
 * API BELOW HERE
 */

/**
 * zeitgeist_index_new:
 * Create a new index that interfaces with the default event index of the Zeitgeist
 * daemon.
 *
 * Index instances are not overly expensive for neither client or the Zeitgeist
 * daemon so there's no need to go to lenghts to keep singleton instances
 * around.
 *
 * Returns: A reference to a newly allocated index.
 */
ZeitgeistIndex*
zeitgeist_index_new (void)
{
  ZeitgeistIndex        *index;

  index = (ZeitgeistIndex*) g_object_new (ZEITGEIST_TYPE_INDEX, NULL);

  return index;
}

/* Helper for _finish() functions returning an array of events */
static gboolean
_zeitgeist_index_collect_events_and_count (CollectEventsAndCountFunc  collect_func,
                                           ZeitgeistIndex     *self,
                                           GPtrArray          **out_events,
                                           guint               *out_hit_count,
                                           GAsyncResult        *res,
                                           GError             **error)
{
  ZeitgeistIndexPrivate   *priv;
  EggDBusArraySeq       *events;
  gboolean               result;

  g_return_val_if_fail (ZEITGEIST_IS_INDEX (self), FALSE);
  g_return_val_if_fail (out_events != NULL, FALSE);
  g_return_val_if_fail (G_IS_ASYNC_RESULT (res), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = ZEITGEIST_INDEX_GET_PRIVATE (self);

  result = collect_func (priv->index,
                         &events,
                         out_hit_count,
                         res,
                         error);

  if (!result)
    {
      return FALSE;
    }

  *out_events = _egg_zeitgeist_events_to_zeitgeist_events (events);
  g_ptr_array_set_free_func (*out_events, (GDestroyNotify) g_object_unref);
  
  g_object_unref (events);
  return TRUE;
}

/* Shared helper for dispatching find_events() and find_event_ids() */
static void
_zeitgeist_index_search_with_func (SearchFunc             search_func,
                                   ZeitgeistIndex         *self,
                                   const gchar            *query,
                                   ZeitgeistTimeRange     *time_range,
                                   GPtrArray              *event_templates,
                                   guint32                 offset,
                                   guint32                 num_events,
                                   ZeitgeistResultType     result_type,
                                   GCancellable           *cancellable,
                                   GAsyncReadyCallback     callback,
                                   gpointer                user_data)
{
  ZeitgeistIndexPrivate *priv;
  EggZeitgeistTimeRange *_time_range;
  EggDBusArraySeq       *_event_templates;
  gpointer              *dispatch_data;

  g_return_if_fail (ZEITGEIST_IS_INDEX (self));
  g_return_if_fail (ZEITGEIST_IS_TIME_RANGE (time_range));
  g_return_if_fail (event_templates != NULL);
  g_return_if_fail (cancellable == NULL || G_IS_CANCELLABLE(cancellable));

  priv = ZEITGEIST_INDEX_GET_PRIVATE (self);

  /* Own all floating refs on the events and the time_range */
  g_ptr_array_foreach (event_templates, (GFunc) g_object_ref_sink, NULL);
  g_object_ref_sink (time_range);
  
  _time_range = _zeitgeist_time_range_to_egg_zeitgeist_time_range (time_range);
  _event_templates = _zeitgeist_events_to_egg_zeitgeist_events (event_templates);

  dispatch_data = g_new (gpointer, 3);
  dispatch_data[0] = g_object_ref (self);
  dispatch_data[1] = callback;
  dispatch_data[2] = user_data;

  search_func (priv->index,
                    EGG_DBUS_CALL_FLAGS_NONE,
                    query,
                    _time_range,
                    _event_templates,
                    offset,
                    num_events,
                    (EggZeitgeistResultType) result_type,
                    cancellable,
                    dispatch_async_callback,
                    dispatch_data);
                                 
  /* Release the event templates */
  g_ptr_array_foreach (event_templates, (GFunc) g_object_unref, NULL);
  g_ptr_array_unref (event_templates);
  g_object_unref (time_range);
  g_object_unref (_event_templates);
  g_object_unref (_time_range);
}

/**
 * zeitgeist_index_search:
 * @self: The #ZeitgeistIndex you want to query
 * @query: The search string to send to Zeitgeist
 * @time_range: Restrict matched events to ones within this time range. If you
 *              are not interested in restricting the timerange pass
 *              zeitgeist_time_range_new_anytime() as Zeitgeist will detect this
 *              and optimize the query accordingly
 * @event_templates: Restrict matches events to ones matching these
 *                   templates
 * @offset: Offset into the result set to read events from
 * @num_events: Maximal number of events to retrieve
 * @result_type: The #ZeitgeistResultType  determining the sort order.
 *               You may pass #ZEITGEIST_RESULT_TYPE_RELEVANCY to this method
 *               to have the results ordered by relevancy calculated in relation
 *               to @query
 * @cancellable: A #GCancellable used to cancel the call or %NULL
 * @callback: A #GAsyncReadyCallback to invoke when the search results are ready
 * @user_data: User data to pass back with @callback
 *
 * Perform a full text search possibly restricted to a #ZeitgeistTimeRange
 * and/or set of event templates.
 *
 * The default boolean operator is %AND. Thus the query
 * <emphasis>foo bar</emphasis> will be interpreted as
 * <emphasis>foo AND bar</emphasis>. To exclude a term from the result
 * set prepend it with a minus sign - eg <emphasis>foo -bar</emphasis>.
 * Phrase queries can be done by double quoting the string 
 * <emphasis>"foo is a bar"</emphasis>. You can truncate terms by appending
 * a *.
 *
 * There are a few keys you can prefix to a term or phrase to search within
 * a specific set of metadata. They are used like
 * <emphasis>key:value</emphasis>. The keys <emphasis>name</emphasis> and
 * <emphasis>title</emphasis> search strictly within the text field of the
 * event subjects. The key <emphasis>app</emphasis> searches within the
 * application name or description that is found in the actor attribute of
 * the events. Lastly you can use the <emphasis>site</emphasis> key to search
 * within the domain name of the subject URIs.
 *
 * You can also control the results with theboolean operators
 * <emphasis>AND</emphasis> and <emphasis>OR</emphasis> and you may
 * use brackets, ( and ), to control the operator presedence.
 */
void
zeitgeist_index_search (ZeitgeistIndex      *self,
                        const gchar         *query,
                        ZeitgeistTimeRange  *time_range,
                        GPtrArray           *event_templates,
                        guint32              offset,
                        guint32              num_events,
                        ZeitgeistResultType  result_type,
                        GCancellable        *cancellable,
                        GAsyncReadyCallback  callback,
                        gpointer             user_data)
{
  _zeitgeist_index_search_with_func (egg_zeitgeist_index_search,
                                     self,
                                     query,
                                     time_range,
                                     event_templates,
                                     offset,
                                     num_events,
                                     result_type,
                                     cancellable,
                                     callback,
                                     user_data);
}

/**
 * zeitgeist_index_search_finish:
 * @self: The #ZeitgeistIndex to retrieve results from
 * @res: The #GAsyncResult you received in the #GAsyncReadyCallback you passed
 *       to zeitgeist_index_search()
 * @error: A place to store a #GError or %NULL in case you want to ignore errors
 *
 * Retrieve the result from an asynchronous query started with
 * zeitgeist_index_search().
 *
 * The total hit count of the query will be avilable via the returned
 * result set by calling zeitgeist_result_set_estimated_matches(). This will
 * often be bigger than actual number of events held in the result set, which
 * is limited by the @num_events paramter passed to zeitgeist_index_search().
 *
 * Returns: A newly allocated #ZeitgeistResultSet containing the
 *          #ZeitgeistEvent<!-- -->s matching the query. You must free the
 *          result set with g_object_unref(). The events held in the result set
 *          will automatically be unreffed when it is finalized.
 */
ZeitgeistResultSet*
zeitgeist_index_search_finish (ZeitgeistIndex      *self,
                               GAsyncResult        *res,
                               GError             **error)
{
  GPtrArray *events;
  guint32    hit_count;
  
  // FIXME: How to return hit count?
  _zeitgeist_index_collect_events_and_count (egg_zeitgeist_index_search_finish,
                                             self, &events, &hit_count,
                                             res, error);
  return _zeitgeist_simple_result_set_new (events, hit_count);
}
