/*
 * Moblin-Web-Browser: The web browser for Moblin
 * Copyright (c) 2009, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

const MwbIPagesObserver
  = Components.interfaces.MwbIPagesObserver;
const nsISupports = Components.interfaces.nsISupports;
const nsISupportsWeakReference = Components.interfaces.nsISupportsWeakReference;

var pagesService;
var searchStr = "";

/* An idle timer used for updating the pinned pages one at a time */
var pinnedPagesIdleHandler = -1;
/* The next pinned page index to add in the idle handler */
var nextPinnedPageNum;
/* Whether we found a matching pinned page */
var foundPinnedPage;
/* A timeout to update the search results */
var updateSearchTimeout = -1;

function PinnedPagesObserver()
{
}

PinnedPagesObserver.prototype =
{
  favoritesReceived : function(favorites, nFavorites)
  {
  },

  linkVisited : function(uri, visit_time, is_pinned)
  {
    if (is_pinned)
      rebuildPinnedPagesList();
  },

  pinnedPagesChanged : function()
  {
    rebuildPinnedPagesList();
  },

  QueryInterface : function(aIID)
  {
    if (!aIID.equals(MwbIPagesObserver)
        && !aIID.equals(nsISupports)
        && !aIID.equals(nsISupportsWeakReference))
      throw Components.results.NS_ERROR_NO_INTERFACE;
    return this;
  }
};

function searchInString(haystack, needle)
{
  var i;

  if (needle.length <= haystack.length)
    for (i = 0; i <= haystack.length - needle.length; i++)
      {
        var j;
        for (j = 0; j < needle.length; j++)
          if (haystack.charAt(i + j) != needle.charAt(j))
            break;
        if (j >= needle.length)
          return i;
      }

  return -1;
}

function rebuildPinnedPagesList()
{
  var pinnedPagesDiv = document.getElementById("pinned_pages_div");

  // Remove all existing pinned pages
  pinnedPagesDiv.innerHTML = "";

  // Start adding from the beginning
  nextPinnedPageNum = 0;
  foundPinnedPage = false;

  // Start the idle timer if we haven't already
  if (pinnedPagesIdleHandler == -1)
    pinnedPagesIdleHandler = setInterval (pinnedPageIdleCb, 0);
}

function pinnedPageIdleCb()
{
  var pinnedPagesDiv = document.getElementById("pinned_pages_div");
  var haveSearchStr = searchStr && searchStr.match(/\S/);

  if (nextPinnedPageNum < pagesService.numPinnedPages)
    {
      var page = pagesService.getPinnedPageByTitle(nextPinnedPageNum);
      var lowerSearchStr;
      var titleMatch;

      if (haveSearchStr)
        lowerSearchStr = searchStr.trim().toLowerCase();

      if (!lowerSearchStr ||
          (titleMatch = searchInString(page.title.toLowerCase(),
                                       lowerSearchStr)) != -1)
        {
          var div = document.createElement("div");
          div.setAttribute("class", "pinned_page_div");

          var a = document.createElement("a");
          a.setAttribute("href", page.uri);

          var thumbnailDiv = document.createElement("div");
          thumbnailDiv.setAttribute("class", "pinned_page_thumbnail");

          // Create elements for the close button. The outer div is
          // just used to make the position for the inner element
          // relative to the top of the thumbnail
          var closeOuter = document.createElement('div');
          closeOuter.setAttribute('class', 'close-button-outer');
          var closeInner = document.createElement('a');
          closeInner.setAttribute('class', 'close-button-inner');
          closeInner.setAttribute('href', 'javascript:');
          closeOuter.appendChild(closeInner);
          thumbnailDiv.appendChild(closeOuter);

          // Create a closure with the url and use it as a callback
          // to unpin the page when the link is clicked
          let (uri = page.uri) {
            closeInner.addEventListener("click",
                                        function(e) {
                                          pagesService.unpinPage(uri);
                                          e.preventDefault();
                                          e.stopPropagation();
                                        },
                                        false);
          }

          var img = document.createElement("img");
          img.setAttribute("class", "thumbnail");
          img.setAttribute("src", pagesService.getThumbnailURL(page.uri).spec);
          thumbnailDiv.appendChild(img);

          a.appendChild(thumbnailDiv);

          img = document.createElement("img");
          img.setAttribute("class", "pinned_page_favicon");
          img.setAttribute("src", "mwb-favicon://" + page.uri);
          a.appendChild(img);

          var span = document.createElement("span");
          span.setAttribute("class", "pinned_page_title");
          if (titleMatch != undefined)
            {
              var title = page.title;
              var preMatch =
                document.createTextNode(title.slice(0, titleMatch));
              span.appendChild(preMatch);
              var match =
                document.createTextNode(title.slice(titleMatch,
                                                    lowerSearchStr.length +
                                                    titleMatch));
              var matchSpan = document.createElement("span");
              matchSpan.setAttribute("class", "search-match");
              matchSpan.appendChild(match);
              span.appendChild(matchSpan);
              var postMatch =
                document.createTextNode(title.slice(titleMatch +
                                                    lowerSearchStr.length));
              span.appendChild(postMatch);
            }
          else
            span.appendChild(document.createTextNode(page.title));
          a.appendChild(span);

          a.appendChild(document.createElement("br"));

          var timeSpan = document.createElement("span");
          timeSpan.setAttribute("class", "pinned_page_time");
          var timeStr = pagesService.formatTime(page.visitTime);
          timeSpan.appendChild(document.createTextNode(timeStr));
          a.appendChild(timeSpan);

          div.appendChild(a);

          pinnedPagesDiv.appendChild(div);

          foundPinnedPage = true;
        }

      nextPinnedPageNum++;
    }
  else
    {
      if (!foundPinnedPage)
        {
          var div = document.createElement("div");
          var note =
            haveSearchStr ?
            _("No matching pages were found") :
            _("No pinned pages. Push the pin in the automagic bar " +
              "on a page you wish to remember.");
          div.appendChild(document.createTextNode(note));
          div.setAttribute("class", "pinned_pages_note");
          pinnedPagesDiv.appendChild(div);
        }

      // The list is complete so we don't need the idle timer anymore
      clearInterval(pinnedPagesIdleHandler);
      pinnedPagesIdleHandler = -1;
    }
}

function updateSearch()
{
  // Restart the search
  searchStr = document.getElementById("search-entry").value;
  rebuildPinnedPagesList();
  // Clear the search timeout
  if (updateSearchTimeout != -1)
    {
      clearInterval(updateSearchTimeout);
      updateSearchTimeout = -1;
    }
}

function queueUpdateSearch()
{
  // Clear any existing timeout
  if (updateSearchTimeout != -1)
    {
      clearInterval(updateSearchTimeout);
      updateSearchTimeout = -1;
    }
  // Add a new one to start in one second
  updateSearchTimeout = setInterval(updateSearch, 1000);
}

function searchEntryKeyDownCb(event)
{
  // If there is a search queued then run it immediately when enter is
  // pressed
  if (event.keyCode == 13 && updateSearchTimeout != -1)
    updateSearch();
}

function updateSearchClearButton()
{
  var searchArea = document.getElementById("search-area");
  var searchEntry = document.getElementById("search-entry");

  var oldClass = searchArea.className;

  // Remove any previous mention of the 'emtpy' or 'non-empty' class
  var match = /\s*\b(non-)?empty\b/.exec(oldClass);
  if (match)
    oldClass = (oldClass.slice(0, match.index) +
                oldClass.slice(match.index + match[0].length));

  // Hide the clear button if there is only whitespace in the search box
  searchArea.className = (oldClass + " " +
                          (searchEntry.value.match(/\S/) ?
                           "non-empty" : "empty"));
}

function textChangedCb()
{
  updateSearchClearButton();
  queueUpdateSearch();
}

function clearSearchButtonClickCb()
{
  var searchEntry = document.getElementById("search-entry");
  searchEntry.value = "";
  updateSearchClearButton();
  updateSearch();
}

var observer;

// Run the initialisation once the page has finished loading
document.addEventListener("DOMContentLoaded", function(event)
{
  pagesService = (Components.classes["@moblin.org/mwb-pages-service;1"].
                  getService(Components.interfaces.MwbIPagesService));

  observer = new PinnedPagesObserver();

  // Add our observer for when the favorites become available
  pagesService.addPagesObserver(observer);

  // Set a style on the body tag if we are in private mode so that we
  // can style the page differently
  var pbService = (Components.classes["@mozilla.org/privatebrowsing;1"].
                   getService(Components.interfaces.nsIPrivateBrowsingService));
  if (pbService.privateBrowsingEnabled)
    {
      var body = document.getElementsByTagName("body")[0];
      body.setAttribute("class", "private_mode");
    }

  // Generate the pinned pages list
  updateSearch();

  var searchEntry = document.getElementById("search-entry");
  searchEntry.addEventListener("keydown", searchEntryKeyDownCb, false);
  searchEntry.addEventListener("input", textChangedCb, false);
  updateSearchClearButton();
  var clearSearchButton = document.getElementById("clear-search-button");
  clearSearchButton.addEventListener("click", clearSearchButtonClickCb, false);

  // Unregister the load event so we don't initialize more than once
  event.currentTarget.removeEventListener("DOMContentLoaded",
                                          arguments.callee,
                                          false);
}, false);
