// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// Copyright (c) 2006 Novell, Inc.
//
// Authors:
//	Rolf Bjarne Kvinge	RKvinge@novell.com

using System;
using System.ComponentModel;
using System.Drawing;
using System.Reflection;
using System.Windows.Forms;
using System.Globalization;
using System.Collections;
using System.Threading;
using System.Text;
using System.IO;
using System.Diagnostics;
using System.Runtime.InteropServices;

using NUnit.Framework;

namespace MonoTests.System.Windows.Forms {

	[TestFixture]
	public class DateTimePickerTest {
		
		[DllImport ("kernel32.dll", CallingConvention=CallingConvention.StdCall, EntryPoint="GetUserDefaultLCID")]
		private extern static int GetUserDefaultLCID ();
		
		private static void CheckCulture ()
		{
			if (!TestHelper.RunningOnUnix) {
				int LCID = GetUserDefaultLCID ();
				if ((new CultureInfo ("en-US")).LCID != LCID) {
					Assert.Ignore ("Must be called with us-english locale, current locale is: " + new CultureInfo (LCID).Name);
				}
			}
			
			if (Thread.CurrentThread.CurrentCulture.Name != "en-US") {
				Assert.Ignore ("Must be called with us-english locale, current locale is: " + Thread.CurrentThread.CurrentCulture.Name);
			}
		}

#if NET_2_0
		// Only use on Windows!
		public static string GenerateCustomFormatTests ()
		{
			StringBuilder builder = new StringBuilder ();
			ArrayList formats = new ArrayList ();
			// Get a number of different datetime formats.
			foreach (CultureInfo ci in CultureInfo.GetCultures (CultureTypes.SpecificCultures)) {
				foreach (string format in ci.DateTimeFormat.GetAllDateTimePatterns ()) {
					if (!formats.Contains(format))
						formats.Add (format);
				}
			}
			formats.AddRange (new string [] { "y", "yy", "yyy", "yyyy", "yyyyy", "yyyyyyy", "M", "MM", "MMM", "MMMM", "MMMMM", "d", "dd", "ddd", "dddd", "ddddd", "H", "HH", "HHH", "HHHHH", "h", "hh", "hhh", "hhhh", "hhhhh", "m", "mm", "mmm", "mmmm", "mmmmm", "s", "ss", "sss", "ssss", "sssss", "t", "tt", "ttt", "tttt", "ttttt" });
			formats.AddRange (new string [] { "M/y", "yy-MM", "MM-yy", "M-y-d-h-H-m-s-t", " yy-MM", "-yy-MM" });

			string tabs = "				";
			int counter = 1;

			Form frm = new Form ();
			DateTimePicker dt = new DateTimePicker ();
			dt.Format = DateTimePickerFormat.Custom;
			dt.CustomFormat = "ddd";
			dt.Value = new DateTime (2007, 2, 8, 15, 30, 45, 60);
			frm.Controls.Add (dt);
			frm.Show ();

			string file = @"C:\code.cs";
			File.WriteAllText (file, "");

			CheckCulture ();
			
			builder.AppendLine (@"
		// On Windows this test must be called with en-US locale specified in the regional settings.
		// There is no way to change this programmatically for the test to run correctly on other locales
		// (see: http://www.microsoft.com/globaldev/getWR/steps/WRG_lclmdl.mspx#EOE)
		// To regenerate this test call GenerateCustomFormatTests() and paste the result here.
		[Test]
		public void CustomFormatTestGenerated ()
		{
			CheckCulture ();
			
			using (Form frm = new Form ()) {
				DateTimePicker dt = new DateTimePicker ();
				frm.Controls.Add (dt);
				frm.Show ();
				
				dt.Format = DateTimePickerFormat.Custom;
				dt.CustomFormat = ""ddd"";
				dt.Value = new DateTime (2007, 2, 8, 15, 30, 45, 60);
		");
			foreach (string format in formats) {
				builder.AppendLine (tabs + "dt.CustomFormat = @\"" + format.Replace ("\"", "\"\"") + "\";");
				dt.CustomFormat = format;
				builder.AppendLine (tabs + "Assert.AreEqual (@\"" + dt.Text.Replace ("\"", "\"\"") + "\", dt.Text, \"#" + (counter++).ToString () + "\");");
			}
			builder.AppendLine ("");
			builder.AppendLine (@"
				}
		}");
		
			File.AppendAllText (file, builder.ToString ());
			builder.Length = 0;

			return builder.ToString ();
		}
#endif

		// On Windows this test must be called with en-US locale specified in the regional settings.
		// There is no way to change this programmatically for the test to run correctly on other locales
		// (see: http://www.microsoft.com/globaldev/getWR/steps/WRG_lclmdl.mspx#EOE)
		// To regenerate this test call GenerateCustomFormatTests() and paste the result here.
		[Test]
		public void CustomFormatTestGenerated ()
		{
			CheckCulture ();

			using (Form frm = new Form ()) {
				DateTimePicker dt = new DateTimePicker ();
				frm.Controls.Add (dt);
				frm.Show ();

				dt.Format = DateTimePickerFormat.Custom;
				dt.CustomFormat = "ddd";
				dt.Value = new DateTime (2007, 2, 8, 15, 30, 45, 60);

				dt.CustomFormat = @"dd/MM/yy";
				Assert.AreEqual (@"08/02/07", dt.Text, "#1");
				dt.CustomFormat = @"dd/MM/yyyy";
				Assert.AreEqual (@"08/02/2007", dt.Text, "#2");
				dt.CustomFormat = @"dd/MMMM/yyyy";
				Assert.AreEqual (@"08/February/2007", dt.Text, "#3");
				dt.CustomFormat = @"dddd, dd MMMM, yyyy";
				Assert.AreEqual (@"Thursday, 08 February, 2007", dt.Text, "#4");
				dt.CustomFormat = @"dd/MMMM/yyyy hh:mm tt";
				Assert.AreEqual (@"08/February/2007 03:30 PM", dt.Text, "#5");
				dt.CustomFormat = @"dd/MMMM/yyyy HH:mm";
				Assert.AreEqual (@"08/February/2007 15:30", dt.Text, "#6");
				dt.CustomFormat = @"dddd, dd MMMM, yyyy hh:mm tt";
				Assert.AreEqual (@"Thursday, 08 February, 2007 03:30 PM", dt.Text, "#7");
				dt.CustomFormat = @"dddd, dd MMMM, yyyy HH:mm";
				Assert.AreEqual (@"Thursday, 08 February, 2007 15:30", dt.Text, "#8");
				dt.CustomFormat = @"dd/MMMM/yyyy hh:mm:ss tt";
				Assert.AreEqual (@"08/February/2007 03:30:45 PM", dt.Text, "#9");
				dt.CustomFormat = @"dd/MMMM/yyyy HH:mm:ss";
				Assert.AreEqual (@"08/February/2007 15:30:45", dt.Text, "#10");
				dt.CustomFormat = @"dddd, dd MMMM, yyyy hh:mm:ss tt";
				Assert.AreEqual (@"Thursday, 08 February, 2007 03:30:45 PM", dt.Text, "#11");
				dt.CustomFormat = @"dddd, dd MMMM, yyyy HH:mm:ss";
				Assert.AreEqual (@"Thursday, 08 February, 2007 15:30:45", dt.Text, "#12");
				dt.CustomFormat = @"dd/MM/yy hh:mm tt";
				Assert.AreEqual (@"08/02/07 03:30 PM", dt.Text, "#13");
				dt.CustomFormat = @"dd/MM/yy HH:mm";
				Assert.AreEqual (@"08/02/07 15:30", dt.Text, "#14");
				dt.CustomFormat = @"dd/MM/yyyy hh:mm tt";
				Assert.AreEqual (@"08/02/2007 03:30 PM", dt.Text, "#15");
				dt.CustomFormat = @"dd/MM/yyyy HH:mm";
				Assert.AreEqual (@"08/02/2007 15:30", dt.Text, "#16");
				dt.CustomFormat = @"dd/MM/yy hh:mm:ss tt";
				Assert.AreEqual (@"08/02/07 03:30:45 PM", dt.Text, "#17");
				dt.CustomFormat = @"dd/MM/yy HH:mm:ss";
				Assert.AreEqual (@"08/02/07 15:30:45", dt.Text, "#18");
				dt.CustomFormat = @"dd/MM/yyyy hh:mm:ss tt";
				Assert.AreEqual (@"08/02/2007 03:30:45 PM", dt.Text, "#19");
				dt.CustomFormat = @"dd/MM/yyyy HH:mm:ss";
				Assert.AreEqual (@"08/02/2007 15:30:45", dt.Text, "#20");
				dt.CustomFormat = @"dd MMMM";
				Assert.AreEqual (@"08 February", dt.Text, "#21");
				dt.CustomFormat = @"yyyy'-'MM'-'dd'T'HH':'mm':'ss.fffffffK";
				Assert.AreEqual (@"2007-02-08T15:30:45.fffffffK", dt.Text, "#22");
				dt.CustomFormat = @"ddd, dd MMM yyyy HH':'mm':'ss 'GMT'";
				Assert.AreEqual (@"Thu, 08 Feb 2007 15:30:45 GMT", dt.Text, "#23");
				dt.CustomFormat = @"yyyy'-'MM'-'dd'T'HH':'mm':'ss";
				Assert.AreEqual (@"2007-02-08T15:30:45", dt.Text, "#24");
				dt.CustomFormat = @"hh:mm tt";
				Assert.AreEqual (@"03:30 PM", dt.Text, "#25");
				dt.CustomFormat = @"HH:mm";
				Assert.AreEqual (@"15:30", dt.Text, "#26");
				dt.CustomFormat = @"hh:mm:ss tt";
				Assert.AreEqual (@"03:30:45 PM", dt.Text, "#27");
				dt.CustomFormat = @"HH:mm:ss";
				Assert.AreEqual (@"15:30:45", dt.Text, "#28");
				dt.CustomFormat = @"yyyy'-'MM'-'dd HH':'mm':'ss'Z'";
				Assert.AreEqual (@"2007-02-08 15:30:45Z", dt.Text, "#29");
				dt.CustomFormat = @"MMMM, yyyy";
				Assert.AreEqual (@"February, 2007", dt.Text, "#30");
				dt.CustomFormat = @"dd.M.yyyy 'г.'";
				Assert.AreEqual (@"08.2.2007 г.", dt.Text, "#31");
				dt.CustomFormat = @"d.M.yyyy 'г.'";
				Assert.AreEqual (@"8.2.2007 г.", dt.Text, "#32");
				dt.CustomFormat = @"dd.MM.yyyy 'г.'";
				Assert.AreEqual (@"08.02.2007 г.", dt.Text, "#33");
				dt.CustomFormat = @"yyyy-MM-dd";
				Assert.AreEqual (@"2007-02-08", dt.Text, "#34");
				dt.CustomFormat = @"dd MMMM yyyy 'г.'";
				Assert.AreEqual (@"08 February 2007 г.", dt.Text, "#35");
				dt.CustomFormat = @"dddd, dd MMMM yyyy 'г.'";
				Assert.AreEqual (@"Thursday, 08 February 2007 г.", dt.Text, "#36");
				dt.CustomFormat = @"dd MMMM yyyy 'г.' HH:mm";
				Assert.AreEqual (@"08 February 2007 г. 15:30", dt.Text, "#37");
				dt.CustomFormat = @"dd MMMM yyyy 'г.' H:mm";
				Assert.AreEqual (@"08 February 2007 г. 15:30", dt.Text, "#38");
				dt.CustomFormat = @"dddd, dd MMMM yyyy 'г.' HH:mm";
				Assert.AreEqual (@"Thursday, 08 February 2007 г. 15:30", dt.Text, "#39");
				dt.CustomFormat = @"dddd, dd MMMM yyyy 'г.' H:mm";
				Assert.AreEqual (@"Thursday, 08 February 2007 г. 15:30", dt.Text, "#40");
				dt.CustomFormat = @"dd MMMM yyyy 'г.' HH:mm:ss";
				Assert.AreEqual (@"08 February 2007 г. 15:30:45", dt.Text, "#41");
				dt.CustomFormat = @"dd MMMM yyyy 'г.' H:mm:ss";
				Assert.AreEqual (@"08 February 2007 г. 15:30:45", dt.Text, "#42");
				dt.CustomFormat = @"dddd, dd MMMM yyyy 'г.' HH:mm:ss";
				Assert.AreEqual (@"Thursday, 08 February 2007 г. 15:30:45", dt.Text, "#43");
				dt.CustomFormat = @"dddd, dd MMMM yyyy 'г.' H:mm:ss";
				Assert.AreEqual (@"Thursday, 08 February 2007 г. 15:30:45", dt.Text, "#44");
				dt.CustomFormat = @"dd.M.yyyy 'г.' HH:mm";
				Assert.AreEqual (@"08.2.2007 г. 15:30", dt.Text, "#45");
				dt.CustomFormat = @"dd.M.yyyy 'г.' H:mm";
				Assert.AreEqual (@"08.2.2007 г. 15:30", dt.Text, "#46");
				dt.CustomFormat = @"d.M.yyyy 'г.' HH:mm";
				Assert.AreEqual (@"8.2.2007 г. 15:30", dt.Text, "#47");
				dt.CustomFormat = @"d.M.yyyy 'г.' H:mm";
				Assert.AreEqual (@"8.2.2007 г. 15:30", dt.Text, "#48");
				dt.CustomFormat = @"dd.MM.yyyy 'г.' HH:mm";
				Assert.AreEqual (@"08.02.2007 г. 15:30", dt.Text, "#49");
				dt.CustomFormat = @"dd.MM.yyyy 'г.' H:mm";
				Assert.AreEqual (@"08.02.2007 г. 15:30", dt.Text, "#50");
				dt.CustomFormat = @"yyyy-MM-dd HH:mm";
				Assert.AreEqual (@"2007-02-08 15:30", dt.Text, "#51");
				dt.CustomFormat = @"yyyy-MM-dd H:mm";
				Assert.AreEqual (@"2007-02-08 15:30", dt.Text, "#52");
				dt.CustomFormat = @"dd.M.yyyy 'г.' HH:mm:ss";
				Assert.AreEqual (@"08.2.2007 г. 15:30:45", dt.Text, "#53");
				dt.CustomFormat = @"dd.M.yyyy 'г.' H:mm:ss";
				Assert.AreEqual (@"08.2.2007 г. 15:30:45", dt.Text, "#54");
				dt.CustomFormat = @"d.M.yyyy 'г.' HH:mm:ss";
				Assert.AreEqual (@"8.2.2007 г. 15:30:45", dt.Text, "#55");
				dt.CustomFormat = @"d.M.yyyy 'г.' H:mm:ss";
				Assert.AreEqual (@"8.2.2007 г. 15:30:45", dt.Text, "#56");
				dt.CustomFormat = @"dd.MM.yyyy 'г.' HH:mm:ss";
				Assert.AreEqual (@"08.02.2007 г. 15:30:45", dt.Text, "#57");
				dt.CustomFormat = @"dd.MM.yyyy 'г.' H:mm:ss";
				Assert.AreEqual (@"08.02.2007 г. 15:30:45", dt.Text, "#58");
				dt.CustomFormat = @"yyyy-MM-dd HH:mm:ss";
				Assert.AreEqual (@"2007-02-08 15:30:45", dt.Text, "#59");
				dt.CustomFormat = @"yyyy-MM-dd H:mm:ss";
				Assert.AreEqual (@"2007-02-08 15:30:45", dt.Text, "#60");
				dt.CustomFormat = @"H:mm";
				Assert.AreEqual (@"15:30", dt.Text, "#61");
				dt.CustomFormat = @"H:mm:ss";
				Assert.AreEqual (@"15:30:45", dt.Text, "#62");
				dt.CustomFormat = @"MMMM yyyy 'г.'";
				Assert.AreEqual (@"February 2007 г.", dt.Text, "#63");
				dt.CustomFormat = @"d/MM/yy";
				Assert.AreEqual (@"8/02/07", dt.Text, "#64");
				dt.CustomFormat = @"d/M/yy";
				Assert.AreEqual (@"8/2/07", dt.Text, "#65");
				dt.CustomFormat = @"dd-MM-yy";
				Assert.AreEqual (@"08-02-07", dt.Text, "#66");
				dt.CustomFormat = @"dd.MM.yy";
				Assert.AreEqual (@"08.02.07", dt.Text, "#67");
				dt.CustomFormat = @"dddd, d' / 'MMMM' / 'yyyy";
				Assert.AreEqual (@"Thursday, 8 / February / 2007", dt.Text, "#68");
				dt.CustomFormat = @"d'/'MMMM'/'yyyy";
				Assert.AreEqual (@"8/February/2007", dt.Text, "#69");
				dt.CustomFormat = @"d' 'MMMM' 'yyyy";
				Assert.AreEqual (@"8 February 2007", dt.Text, "#70");
				dt.CustomFormat = @"dddd, d' / 'MMMM' / 'yyyy HH:mm";
				Assert.AreEqual (@"Thursday, 8 / February / 2007 15:30", dt.Text, "#71");
				dt.CustomFormat = @"dddd, d' / 'MMMM' / 'yyyy H:mm";
				Assert.AreEqual (@"Thursday, 8 / February / 2007 15:30", dt.Text, "#72");
				dt.CustomFormat = @"dddd, d' / 'MMMM' / 'yyyy HH'H'mm'\''";
				Assert.AreEqual (@"Thursday, 8 / February / 2007 15H30\'", dt.Text, "#73");
				dt.CustomFormat = @"d'/'MMMM'/'yyyy HH:mm";
				Assert.AreEqual (@"8/February/2007 15:30", dt.Text, "#74");
				dt.CustomFormat = @"d'/'MMMM'/'yyyy H:mm";
				Assert.AreEqual (@"8/February/2007 15:30", dt.Text, "#75");
				dt.CustomFormat = @"d'/'MMMM'/'yyyy HH'H'mm'\''";
				Assert.AreEqual (@"8/February/2007 15H30\'", dt.Text, "#76");
				dt.CustomFormat = @"d' 'MMMM' 'yyyy HH:mm";
				Assert.AreEqual (@"8 February 2007 15:30", dt.Text, "#77");
				dt.CustomFormat = @"d' 'MMMM' 'yyyy H:mm";
				Assert.AreEqual (@"8 February 2007 15:30", dt.Text, "#78");
				dt.CustomFormat = @"d' 'MMMM' 'yyyy HH'H'mm'\''";
				Assert.AreEqual (@"8 February 2007 15H30\'", dt.Text, "#79");
				dt.CustomFormat = @"dddd, d' / 'MMMM' / 'yyyy HH:mm:ss";
				Assert.AreEqual (@"Thursday, 8 / February / 2007 15:30:45", dt.Text, "#80");
				dt.CustomFormat = @"dddd, d' / 'MMMM' / 'yyyy H:mm:ss";
				Assert.AreEqual (@"Thursday, 8 / February / 2007 15:30:45", dt.Text, "#81");
				dt.CustomFormat = @"dddd, d' / 'MMMM' / 'yyyy HH'H'mm'\''ss'\'\''";
				Assert.AreEqual (@"Thursday, 8 / February / 2007 15H30\'ss\\'", dt.Text, "#82");
				dt.CustomFormat = @"d'/'MMMM'/'yyyy HH:mm:ss";
				Assert.AreEqual (@"8/February/2007 15:30:45", dt.Text, "#83");
				dt.CustomFormat = @"d'/'MMMM'/'yyyy H:mm:ss";
				Assert.AreEqual (@"8/February/2007 15:30:45", dt.Text, "#84");
				dt.CustomFormat = @"d'/'MMMM'/'yyyy HH'H'mm'\''ss'\'\''";
				Assert.AreEqual (@"8/February/2007 15H30\'ss\\'", dt.Text, "#85");
				dt.CustomFormat = @"d' 'MMMM' 'yyyy HH:mm:ss";
				Assert.AreEqual (@"8 February 2007 15:30:45", dt.Text, "#86");
				dt.CustomFormat = @"d' 'MMMM' 'yyyy H:mm:ss";
				Assert.AreEqual (@"8 February 2007 15:30:45", dt.Text, "#87");
				dt.CustomFormat = @"d' 'MMMM' 'yyyy HH'H'mm'\''ss'\'\''";
				Assert.AreEqual (@"8 February 2007 15H30\'ss\\'", dt.Text, "#88");
				dt.CustomFormat = @"dd/MM/yyyy H:mm";
				Assert.AreEqual (@"08/02/2007 15:30", dt.Text, "#89");
				dt.CustomFormat = @"dd/MM/yyyy HH'H'mm'\''";
				Assert.AreEqual (@"08/02/2007 15H30\'", dt.Text, "#90");
				dt.CustomFormat = @"dd/MM/yy H:mm";
				Assert.AreEqual (@"08/02/07 15:30", dt.Text, "#91");
				dt.CustomFormat = @"dd/MM/yy HH'H'mm'\''";
				Assert.AreEqual (@"08/02/07 15H30\'", dt.Text, "#92");
				dt.CustomFormat = @"yyyy-MM-dd HH'H'mm'\''";
				Assert.AreEqual (@"2007-02-08 15H30\'", dt.Text, "#93");
				dt.CustomFormat = @"d/MM/yy HH:mm";
				Assert.AreEqual (@"8/02/07 15:30", dt.Text, "#94");
				dt.CustomFormat = @"d/MM/yy H:mm";
				Assert.AreEqual (@"8/02/07 15:30", dt.Text, "#95");
				dt.CustomFormat = @"d/MM/yy HH'H'mm'\''";
				Assert.AreEqual (@"8/02/07 15H30\'", dt.Text, "#96");
				dt.CustomFormat = @"d/M/yy HH:mm";
				Assert.AreEqual (@"8/2/07 15:30", dt.Text, "#97");
				dt.CustomFormat = @"d/M/yy H:mm";
				Assert.AreEqual (@"8/2/07 15:30", dt.Text, "#98");
				dt.CustomFormat = @"d/M/yy HH'H'mm'\''";
				Assert.AreEqual (@"8/2/07 15H30\'", dt.Text, "#99");
				dt.CustomFormat = @"dd-MM-yy HH:mm";
				Assert.AreEqual (@"08-02-07 15:30", dt.Text, "#100");
				dt.CustomFormat = @"dd-MM-yy H:mm";
				Assert.AreEqual (@"08-02-07 15:30", dt.Text, "#101");
				dt.CustomFormat = @"dd-MM-yy HH'H'mm'\''";
				Assert.AreEqual (@"08-02-07 15H30\'", dt.Text, "#102");
				dt.CustomFormat = @"dd.MM.yy HH:mm";
				Assert.AreEqual (@"08.02.07 15:30", dt.Text, "#103");
				dt.CustomFormat = @"dd.MM.yy H:mm";
				Assert.AreEqual (@"08.02.07 15:30", dt.Text, "#104");
				dt.CustomFormat = @"dd.MM.yy HH'H'mm'\''";
				Assert.AreEqual (@"08.02.07 15H30\'", dt.Text, "#105");
				dt.CustomFormat = @"dd/MM/yyyy H:mm:ss";
				Assert.AreEqual (@"08/02/2007 15:30:45", dt.Text, "#106");
				dt.CustomFormat = @"dd/MM/yyyy HH'H'mm'\''ss'\'\''";
				Assert.AreEqual (@"08/02/2007 15H30\'ss\\'", dt.Text, "#107");
				dt.CustomFormat = @"dd/MM/yy H:mm:ss";
				Assert.AreEqual (@"08/02/07 15:30:45", dt.Text, "#108");
				dt.CustomFormat = @"dd/MM/yy HH'H'mm'\''ss'\'\''";
				Assert.AreEqual (@"08/02/07 15H30\'ss\\'", dt.Text, "#109");
				dt.CustomFormat = @"yyyy-MM-dd HH'H'mm'\''ss'\'\''";
				Assert.AreEqual (@"2007-02-08 15H30\'ss\\'", dt.Text, "#110");
				dt.CustomFormat = @"d/MM/yy HH:mm:ss";
				Assert.AreEqual (@"8/02/07 15:30:45", dt.Text, "#111");
				dt.CustomFormat = @"d/MM/yy H:mm:ss";
				Assert.AreEqual (@"8/02/07 15:30:45", dt.Text, "#112");
				dt.CustomFormat = @"d/MM/yy HH'H'mm'\''ss'\'\''";
				Assert.AreEqual (@"8/02/07 15H30\'ss\\'", dt.Text, "#113");
				dt.CustomFormat = @"d/M/yy HH:mm:ss";
				Assert.AreEqual (@"8/2/07 15:30:45", dt.Text, "#114");
				dt.CustomFormat = @"d/M/yy H:mm:ss";
				Assert.AreEqual (@"8/2/07 15:30:45", dt.Text, "#115");
				dt.CustomFormat = @"d/M/yy HH'H'mm'\''ss'\'\''";
				Assert.AreEqual (@"8/2/07 15H30\'ss\\'", dt.Text, "#116");
				dt.CustomFormat = @"dd-MM-yy HH:mm:ss";
				Assert.AreEqual (@"08-02-07 15:30:45", dt.Text, "#117");
				dt.CustomFormat = @"dd-MM-yy H:mm:ss";
				Assert.AreEqual (@"08-02-07 15:30:45", dt.Text, "#118");
				dt.CustomFormat = @"dd-MM-yy HH'H'mm'\''ss'\'\''";
				Assert.AreEqual (@"08-02-07 15H30\'ss\\'", dt.Text, "#119");
				dt.CustomFormat = @"dd.MM.yy HH:mm:ss";
				Assert.AreEqual (@"08.02.07 15:30:45", dt.Text, "#120");
				dt.CustomFormat = @"dd.MM.yy H:mm:ss";
				Assert.AreEqual (@"08.02.07 15:30:45", dt.Text, "#121");
				dt.CustomFormat = @"dd.MM.yy HH'H'mm'\''ss'\'\''";
				Assert.AreEqual (@"08.02.07 15H30\'ss\\'", dt.Text, "#122");
				dt.CustomFormat = @"HH'H'mm'\''";
				Assert.AreEqual (@"15H30\'", dt.Text, "#123");
				dt.CustomFormat = @"HH'H'mm'\''ss'\'\''";
				Assert.AreEqual (@"15H30\'ss\\'", dt.Text, "#124");
				dt.CustomFormat = @"MMMM' / 'yyyy";
				Assert.AreEqual (@"February / 2007", dt.Text, "#125");
				dt.CustomFormat = @"MMMM' de 'yyyy";
				Assert.AreEqual (@"February de 2007", dt.Text, "#126");
				dt.CustomFormat = @"yyyy/M/d";
				Assert.AreEqual (@"2007/2/8", dt.Text, "#127");
				dt.CustomFormat = @"yyyy/MM/dd";
				Assert.AreEqual (@"2007/02/08", dt.Text, "#128");
				dt.CustomFormat = @"yyyy'年'M'月'd'日'";
				Assert.AreEqual (@"2007年2月8日", dt.Text, "#129");
				dt.CustomFormat = @"yyyy'年'MM'月'dd'日'";
				Assert.AreEqual (@"2007年02月08日", dt.Text, "#130");
				dt.CustomFormat = @"yyyy'年'M'月'd'日' tt hh:mm";
				Assert.AreEqual (@"2007年2月8日 PM 03:30", dt.Text, "#131");
				dt.CustomFormat = @"yyyy'年'M'月'd'日' tt h:mm";
				Assert.AreEqual (@"2007年2月8日 PM 3:30", dt.Text, "#132");
				dt.CustomFormat = @"yyyy'年'M'月'd'日' HH:mm";
				Assert.AreEqual (@"2007年2月8日 15:30", dt.Text, "#133");
				dt.CustomFormat = @"yyyy'年'M'月'd'日' H:mm";
				Assert.AreEqual (@"2007年2月8日 15:30", dt.Text, "#134");
				dt.CustomFormat = @"yyyy'年'MM'月'dd'日' tt hh:mm";
				Assert.AreEqual (@"2007年02月08日 PM 03:30", dt.Text, "#135");
				dt.CustomFormat = @"yyyy'年'MM'月'dd'日' tt h:mm";
				Assert.AreEqual (@"2007年02月08日 PM 3:30", dt.Text, "#136");
				dt.CustomFormat = @"yyyy'年'MM'月'dd'日' HH:mm";
				Assert.AreEqual (@"2007年02月08日 15:30", dt.Text, "#137");
				dt.CustomFormat = @"yyyy'年'MM'月'dd'日' H:mm";
				Assert.AreEqual (@"2007年02月08日 15:30", dt.Text, "#138");
				dt.CustomFormat = @"yyyy'年'M'月'd'日' tt hh:mm:ss";
				Assert.AreEqual (@"2007年2月8日 PM 03:30:45", dt.Text, "#139");
				dt.CustomFormat = @"yyyy'年'M'月'd'日' tt h:mm:ss";
				Assert.AreEqual (@"2007年2月8日 PM 3:30:45", dt.Text, "#140");
				dt.CustomFormat = @"yyyy'年'M'月'd'日' HH:mm:ss";
				Assert.AreEqual (@"2007年2月8日 15:30:45", dt.Text, "#141");
				dt.CustomFormat = @"yyyy'年'M'月'd'日' H:mm:ss";
				Assert.AreEqual (@"2007年2月8日 15:30:45", dt.Text, "#142");
				dt.CustomFormat = @"yyyy'年'MM'月'dd'日' tt hh:mm:ss";
				Assert.AreEqual (@"2007年02月08日 PM 03:30:45", dt.Text, "#143");
				dt.CustomFormat = @"yyyy'年'MM'月'dd'日' tt h:mm:ss";
				Assert.AreEqual (@"2007年02月08日 PM 3:30:45", dt.Text, "#144");
				dt.CustomFormat = @"yyyy'年'MM'月'dd'日' HH:mm:ss";
				Assert.AreEqual (@"2007年02月08日 15:30:45", dt.Text, "#145");
				dt.CustomFormat = @"yyyy'年'MM'月'dd'日' H:mm:ss";
				Assert.AreEqual (@"2007年02月08日 15:30:45", dt.Text, "#146");
				dt.CustomFormat = @"yyyy/M/d tt hh:mm";
				Assert.AreEqual (@"2007/2/8 PM 03:30", dt.Text, "#147");
				dt.CustomFormat = @"yyyy/M/d tt h:mm";
				Assert.AreEqual (@"2007/2/8 PM 3:30", dt.Text, "#148");
				dt.CustomFormat = @"yyyy/M/d HH:mm";
				Assert.AreEqual (@"2007/2/8 15:30", dt.Text, "#149");
				dt.CustomFormat = @"yyyy/M/d H:mm";
				Assert.AreEqual (@"2007/2/8 15:30", dt.Text, "#150");
				dt.CustomFormat = @"yyyy/MM/dd tt hh:mm";
				Assert.AreEqual (@"2007/02/08 PM 03:30", dt.Text, "#151");
				dt.CustomFormat = @"yyyy/MM/dd tt h:mm";
				Assert.AreEqual (@"2007/02/08 PM 3:30", dt.Text, "#152");
				dt.CustomFormat = @"yyyy/MM/dd HH:mm";
				Assert.AreEqual (@"2007/02/08 15:30", dt.Text, "#153");
				dt.CustomFormat = @"yyyy/MM/dd H:mm";
				Assert.AreEqual (@"2007/02/08 15:30", dt.Text, "#154");
				dt.CustomFormat = @"yyyy-MM-dd tt hh:mm";
				Assert.AreEqual (@"2007-02-08 PM 03:30", dt.Text, "#155");
				dt.CustomFormat = @"yyyy-MM-dd tt h:mm";
				Assert.AreEqual (@"2007-02-08 PM 3:30", dt.Text, "#156");
				dt.CustomFormat = @"yyyy/M/d tt hh:mm:ss";
				Assert.AreEqual (@"2007/2/8 PM 03:30:45", dt.Text, "#157");
				dt.CustomFormat = @"yyyy/M/d tt h:mm:ss";
				Assert.AreEqual (@"2007/2/8 PM 3:30:45", dt.Text, "#158");
				dt.CustomFormat = @"yyyy/M/d HH:mm:ss";
				Assert.AreEqual (@"2007/2/8 15:30:45", dt.Text, "#159");
				dt.CustomFormat = @"yyyy/M/d H:mm:ss";
				Assert.AreEqual (@"2007/2/8 15:30:45", dt.Text, "#160");
				dt.CustomFormat = @"yyyy/MM/dd tt hh:mm:ss";
				Assert.AreEqual (@"2007/02/08 PM 03:30:45", dt.Text, "#161");
				dt.CustomFormat = @"yyyy/MM/dd tt h:mm:ss";
				Assert.AreEqual (@"2007/02/08 PM 3:30:45", dt.Text, "#162");
				dt.CustomFormat = @"yyyy/MM/dd HH:mm:ss";
				Assert.AreEqual (@"2007/02/08 15:30:45", dt.Text, "#163");
				dt.CustomFormat = @"yyyy/MM/dd H:mm:ss";
				Assert.AreEqual (@"2007/02/08 15:30:45", dt.Text, "#164");
				dt.CustomFormat = @"yyyy-MM-dd tt hh:mm:ss";
				Assert.AreEqual (@"2007-02-08 PM 03:30:45", dt.Text, "#165");
				dt.CustomFormat = @"yyyy-MM-dd tt h:mm:ss";
				Assert.AreEqual (@"2007-02-08 PM 3:30:45", dt.Text, "#166");
				dt.CustomFormat = @"M'月'd'日'";
				Assert.AreEqual (@"2月8日", dt.Text, "#167");
				dt.CustomFormat = @"tt hh:mm";
				Assert.AreEqual (@"PM 03:30", dt.Text, "#168");
				dt.CustomFormat = @"tt h:mm";
				Assert.AreEqual (@"PM 3:30", dt.Text, "#169");
				dt.CustomFormat = @"tt hh:mm:ss";
				Assert.AreEqual (@"PM 03:30:45", dt.Text, "#170");
				dt.CustomFormat = @"tt h:mm:ss";
				Assert.AreEqual (@"PM 3:30:45", dt.Text, "#171");
				dt.CustomFormat = @"yyyy'年'M'月'";
				Assert.AreEqual (@"2007年2月", dt.Text, "#172");
				dt.CustomFormat = @"d.M.yyyy";
				Assert.AreEqual (@"8.2.2007", dt.Text, "#173");
				dt.CustomFormat = @"d-M-yy";
				Assert.AreEqual (@"8-2-07", dt.Text, "#174");
				dt.CustomFormat = @"yyyy-M-d";
				Assert.AreEqual (@"2007-2-8", dt.Text, "#175");
				dt.CustomFormat = @"yy-M-d";
				Assert.AreEqual (@"07-2-8", dt.Text, "#176");
				dt.CustomFormat = @"d. MMMM yyyy";
				Assert.AreEqual (@"8. February 2007", dt.Text, "#177");
				dt.CustomFormat = @"d. MMMM yyyy H:mm";
				Assert.AreEqual (@"8. February 2007 15:30", dt.Text, "#178");
				dt.CustomFormat = @"d. MMMM yyyy HH:mm";
				Assert.AreEqual (@"8. February 2007 15:30", dt.Text, "#179");
				dt.CustomFormat = @"d. MMMM yyyy h.mm tt";
				Assert.AreEqual (@"8. February 2007 3.30 PM", dt.Text, "#180");
				dt.CustomFormat = @"d. MMMM yyyy H.mm";
				Assert.AreEqual (@"8. February 2007 15.30", dt.Text, "#181");
				dt.CustomFormat = @"d. MMMM yyyy H:mm:ss";
				Assert.AreEqual (@"8. February 2007 15:30:45", dt.Text, "#182");
				dt.CustomFormat = @"d. MMMM yyyy HH:mm:ss";
				Assert.AreEqual (@"8. February 2007 15:30:45", dt.Text, "#183");
				dt.CustomFormat = @"d. MMMM yyyy h.mm.ss tt";
				Assert.AreEqual (@"8. February 2007 3.30.45 PM", dt.Text, "#184");
				dt.CustomFormat = @"d. MMMM yyyy H.mm.ss";
				Assert.AreEqual (@"8. February 2007 15.30.45", dt.Text, "#185");
				dt.CustomFormat = @"d.M.yyyy H:mm";
				Assert.AreEqual (@"8.2.2007 15:30", dt.Text, "#186");
				dt.CustomFormat = @"d.M.yyyy HH:mm";
				Assert.AreEqual (@"8.2.2007 15:30", dt.Text, "#187");
				dt.CustomFormat = @"d.M.yyyy h.mm tt";
				Assert.AreEqual (@"8.2.2007 3.30 PM", dt.Text, "#188");
				dt.CustomFormat = @"d.M.yyyy H.mm";
				Assert.AreEqual (@"8.2.2007 15.30", dt.Text, "#189");
				dt.CustomFormat = @"d-M-yy H:mm";
				Assert.AreEqual (@"8-2-07 15:30", dt.Text, "#190");
				dt.CustomFormat = @"d-M-yy HH:mm";
				Assert.AreEqual (@"8-2-07 15:30", dt.Text, "#191");
				dt.CustomFormat = @"d-M-yy h.mm tt";
				Assert.AreEqual (@"8-2-07 3.30 PM", dt.Text, "#192");
				dt.CustomFormat = @"d-M-yy H.mm";
				Assert.AreEqual (@"8-2-07 15.30", dt.Text, "#193");
				dt.CustomFormat = @"yyyy-MM-dd h.mm tt";
				Assert.AreEqual (@"2007-02-08 3.30 PM", dt.Text, "#194");
				dt.CustomFormat = @"yyyy-MM-dd H.mm";
				Assert.AreEqual (@"2007-02-08 15.30", dt.Text, "#195");
				dt.CustomFormat = @"yyyy-M-d H:mm";
				Assert.AreEqual (@"2007-2-8 15:30", dt.Text, "#196");
				dt.CustomFormat = @"yyyy-M-d HH:mm";
				Assert.AreEqual (@"2007-2-8 15:30", dt.Text, "#197");
				dt.CustomFormat = @"yyyy-M-d h.mm tt";
				Assert.AreEqual (@"2007-2-8 3.30 PM", dt.Text, "#198");
				dt.CustomFormat = @"yyyy-M-d H.mm";
				Assert.AreEqual (@"2007-2-8 15.30", dt.Text, "#199");
				dt.CustomFormat = @"yy-M-d H:mm";
				Assert.AreEqual (@"07-2-8 15:30", dt.Text, "#200");
				dt.CustomFormat = @"yy-M-d HH:mm";
				Assert.AreEqual (@"07-2-8 15:30", dt.Text, "#201");
				dt.CustomFormat = @"yy-M-d h.mm tt";
				Assert.AreEqual (@"07-2-8 3.30 PM", dt.Text, "#202");
				dt.CustomFormat = @"yy-M-d H.mm";
				Assert.AreEqual (@"07-2-8 15.30", dt.Text, "#203");
				dt.CustomFormat = @"d.M.yyyy H:mm:ss";
				Assert.AreEqual (@"8.2.2007 15:30:45", dt.Text, "#204");
				dt.CustomFormat = @"d.M.yyyy HH:mm:ss";
				Assert.AreEqual (@"8.2.2007 15:30:45", dt.Text, "#205");
				dt.CustomFormat = @"d.M.yyyy h.mm.ss tt";
				Assert.AreEqual (@"8.2.2007 3.30.45 PM", dt.Text, "#206");
				dt.CustomFormat = @"d.M.yyyy H.mm.ss";
				Assert.AreEqual (@"8.2.2007 15.30.45", dt.Text, "#207");
				dt.CustomFormat = @"d-M-yy H:mm:ss";
				Assert.AreEqual (@"8-2-07 15:30:45", dt.Text, "#208");
				dt.CustomFormat = @"d-M-yy HH:mm:ss";
				Assert.AreEqual (@"8-2-07 15:30:45", dt.Text, "#209");
				dt.CustomFormat = @"d-M-yy h.mm.ss tt";
				Assert.AreEqual (@"8-2-07 3.30.45 PM", dt.Text, "#210");
				dt.CustomFormat = @"d-M-yy H.mm.ss";
				Assert.AreEqual (@"8-2-07 15.30.45", dt.Text, "#211");
				dt.CustomFormat = @"yyyy-MM-dd h.mm.ss tt";
				Assert.AreEqual (@"2007-02-08 3.30.45 PM", dt.Text, "#212");
				dt.CustomFormat = @"yyyy-MM-dd H.mm.ss";
				Assert.AreEqual (@"2007-02-08 15.30.45", dt.Text, "#213");
				dt.CustomFormat = @"yyyy-M-d H:mm:ss";
				Assert.AreEqual (@"2007-2-8 15:30:45", dt.Text, "#214");
				dt.CustomFormat = @"yyyy-M-d HH:mm:ss";
				Assert.AreEqual (@"2007-2-8 15:30:45", dt.Text, "#215");
				dt.CustomFormat = @"yyyy-M-d h.mm.ss tt";
				Assert.AreEqual (@"2007-2-8 3.30.45 PM", dt.Text, "#216");
				dt.CustomFormat = @"yyyy-M-d H.mm.ss";
				Assert.AreEqual (@"2007-2-8 15.30.45", dt.Text, "#217");
				dt.CustomFormat = @"yy-M-d H:mm:ss";
				Assert.AreEqual (@"07-2-8 15:30:45", dt.Text, "#218");
				dt.CustomFormat = @"yy-M-d HH:mm:ss";
				Assert.AreEqual (@"07-2-8 15:30:45", dt.Text, "#219");
				dt.CustomFormat = @"yy-M-d h.mm.ss tt";
				Assert.AreEqual (@"07-2-8 3.30.45 PM", dt.Text, "#220");
				dt.CustomFormat = @"yy-M-d H.mm.ss";
				Assert.AreEqual (@"07-2-8 15.30.45", dt.Text, "#221");
				dt.CustomFormat = @"h.mm tt";
				Assert.AreEqual (@"3.30 PM", dt.Text, "#222");
				dt.CustomFormat = @"H.mm";
				Assert.AreEqual (@"15.30", dt.Text, "#223");
				dt.CustomFormat = @"h.mm.ss tt";
				Assert.AreEqual (@"3.30.45 PM", dt.Text, "#224");
				dt.CustomFormat = @"H.mm.ss";
				Assert.AreEqual (@"15.30.45", dt.Text, "#225");
				dt.CustomFormat = @"MMMM yyyy";
				Assert.AreEqual (@"February 2007", dt.Text, "#226");
				dt.CustomFormat = @"dd-MM-yyyy";
				Assert.AreEqual (@"08-02-2007", dt.Text, "#227");
				dt.CustomFormat = @"yyyy MM dd";
				Assert.AreEqual (@"2007 02 08", dt.Text, "#228");
				dt.CustomFormat = @"dd. MMMM yyyy";
				Assert.AreEqual (@"08. February 2007", dt.Text, "#229");
				dt.CustomFormat = @"dd. MMMM yyyy HH:mm";
				Assert.AreEqual (@"08. February 2007 15:30", dt.Text, "#230");
				dt.CustomFormat = @"dd. MMMM yyyy H:mm";
				Assert.AreEqual (@"08. February 2007 15:30", dt.Text, "#231");
				dt.CustomFormat = @"dd. MMMM yyyy HH:mm:ss";
				Assert.AreEqual (@"08. February 2007 15:30:45", dt.Text, "#232");
				dt.CustomFormat = @"dd. MMMM yyyy H:mm:ss";
				Assert.AreEqual (@"08. February 2007 15:30:45", dt.Text, "#233");
				dt.CustomFormat = @"dd-MM-yyyy HH:mm";
				Assert.AreEqual (@"08-02-2007 15:30", dt.Text, "#234");
				dt.CustomFormat = @"dd-MM-yyyy H:mm";
				Assert.AreEqual (@"08-02-2007 15:30", dt.Text, "#235");
				dt.CustomFormat = @"yyyy MM dd HH:mm";
				Assert.AreEqual (@"2007 02 08 15:30", dt.Text, "#236");
				dt.CustomFormat = @"yyyy MM dd H:mm";
				Assert.AreEqual (@"2007 02 08 15:30", dt.Text, "#237");
				dt.CustomFormat = @"dd-MM-yyyy HH:mm:ss";
				Assert.AreEqual (@"08-02-2007 15:30:45", dt.Text, "#238");
				dt.CustomFormat = @"dd-MM-yyyy H:mm:ss";
				Assert.AreEqual (@"08-02-2007 15:30:45", dt.Text, "#239");
				dt.CustomFormat = @"yyyy MM dd HH:mm:ss";
				Assert.AreEqual (@"2007 02 08 15:30:45", dt.Text, "#240");
				dt.CustomFormat = @"yyyy MM dd H:mm:ss";
				Assert.AreEqual (@"2007 02 08 15:30:45", dt.Text, "#241");
				dt.CustomFormat = @"d. MMMM";
				Assert.AreEqual (@"8. February", dt.Text, "#242");
				dt.CustomFormat = @"dd.MM.yyyy";
				Assert.AreEqual (@"08.02.2007", dt.Text, "#243");
				dt.CustomFormat = @"d.MM.yy";
				Assert.AreEqual (@"8.02.07", dt.Text, "#244");
				dt.CustomFormat = @"d.M.yy";
				Assert.AreEqual (@"8.2.07", dt.Text, "#245");
				dt.CustomFormat = @"dddd, d. MMMM yyyy";
				Assert.AreEqual (@"Thursday, 8. February 2007", dt.Text, "#246");
				dt.CustomFormat = @"d. MMM yyyy";
				Assert.AreEqual (@"8. Feb 2007", dt.Text, "#247");
				dt.CustomFormat = @"dddd, d. MMMM yyyy HH:mm";
				Assert.AreEqual (@"Thursday, 8. February 2007 15:30", dt.Text, "#248");
				dt.CustomFormat = @"dddd, d. MMMM yyyy H:mm";
				Assert.AreEqual (@"Thursday, 8. February 2007 15:30", dt.Text, "#249");
				dt.CustomFormat = @"dddd, d. MMMM yyyy H.mm";
				Assert.AreEqual (@"Thursday, 8. February 2007 15.30", dt.Text, "#250");
				dt.CustomFormat = @"dddd, d. MMMM yyyy H.mm' Uhr '";
				Assert.AreEqual (@"Thursday, 8. February 2007 15.30 Uhr ", dt.Text, "#251");
				dt.CustomFormat = @"d. MMMM yyyy H.mm' Uhr '";
				Assert.AreEqual (@"8. February 2007 15.30 Uhr ", dt.Text, "#252");
				dt.CustomFormat = @"d. MMM yyyy HH:mm";
				Assert.AreEqual (@"8. Feb 2007 15:30", dt.Text, "#253");
				dt.CustomFormat = @"d. MMM yyyy H:mm";
				Assert.AreEqual (@"8. Feb 2007 15:30", dt.Text, "#254");
				dt.CustomFormat = @"d. MMM yyyy H.mm";
				Assert.AreEqual (@"8. Feb 2007 15.30", dt.Text, "#255");
				dt.CustomFormat = @"d. MMM yyyy H.mm' Uhr '";
				Assert.AreEqual (@"8. Feb 2007 15.30 Uhr ", dt.Text, "#256");
				dt.CustomFormat = @"dddd, d. MMMM yyyy HH:mm:ss";
				Assert.AreEqual (@"Thursday, 8. February 2007 15:30:45", dt.Text, "#257");
				dt.CustomFormat = @"dddd, d. MMMM yyyy H:mm:ss";
				Assert.AreEqual (@"Thursday, 8. February 2007 15:30:45", dt.Text, "#258");
				dt.CustomFormat = @"d. MMM yyyy HH:mm:ss";
				Assert.AreEqual (@"8. Feb 2007 15:30:45", dt.Text, "#259");
				dt.CustomFormat = @"d. MMM yyyy H:mm:ss";
				Assert.AreEqual (@"8. Feb 2007 15:30:45", dt.Text, "#260");
				dt.CustomFormat = @"dd.MM.yyyy HH:mm";
				Assert.AreEqual (@"08.02.2007 15:30", dt.Text, "#261");
				dt.CustomFormat = @"dd.MM.yyyy H:mm";
				Assert.AreEqual (@"08.02.2007 15:30", dt.Text, "#262");
				dt.CustomFormat = @"dd.MM.yyyy H.mm";
				Assert.AreEqual (@"08.02.2007 15.30", dt.Text, "#263");
				dt.CustomFormat = @"dd.MM.yyyy H.mm' Uhr '";
				Assert.AreEqual (@"08.02.2007 15.30 Uhr ", dt.Text, "#264");
				dt.CustomFormat = @"dd.MM.yy H.mm";
				Assert.AreEqual (@"08.02.07 15.30", dt.Text, "#265");
				dt.CustomFormat = @"dd.MM.yy H.mm' Uhr '";
				Assert.AreEqual (@"08.02.07 15.30 Uhr ", dt.Text, "#266");
				dt.CustomFormat = @"d.MM.yy HH:mm";
				Assert.AreEqual (@"8.02.07 15:30", dt.Text, "#267");
				dt.CustomFormat = @"d.MM.yy H:mm";
				Assert.AreEqual (@"8.02.07 15:30", dt.Text, "#268");
				dt.CustomFormat = @"d.MM.yy H.mm";
				Assert.AreEqual (@"8.02.07 15.30", dt.Text, "#269");
				dt.CustomFormat = @"d.MM.yy H.mm' Uhr '";
				Assert.AreEqual (@"8.02.07 15.30 Uhr ", dt.Text, "#270");
				dt.CustomFormat = @"d.M.yy HH:mm";
				Assert.AreEqual (@"8.2.07 15:30", dt.Text, "#271");
				dt.CustomFormat = @"d.M.yy H:mm";
				Assert.AreEqual (@"8.2.07 15:30", dt.Text, "#272");
				dt.CustomFormat = @"d.M.yy H.mm";
				Assert.AreEqual (@"8.2.07 15.30", dt.Text, "#273");
				dt.CustomFormat = @"d.M.yy H.mm' Uhr '";
				Assert.AreEqual (@"8.2.07 15.30 Uhr ", dt.Text, "#274");
				dt.CustomFormat = @"d.M.yyyy H.mm' Uhr '";
				Assert.AreEqual (@"8.2.2007 15.30 Uhr ", dt.Text, "#275");
				dt.CustomFormat = @"yyyy-MM-dd H.mm' Uhr '";
				Assert.AreEqual (@"2007-02-08 15.30 Uhr ", dt.Text, "#276");
				dt.CustomFormat = @"dd.MM.yyyy HH:mm:ss";
				Assert.AreEqual (@"08.02.2007 15:30:45", dt.Text, "#277");
				dt.CustomFormat = @"dd.MM.yyyy H:mm:ss";
				Assert.AreEqual (@"08.02.2007 15:30:45", dt.Text, "#278");
				dt.CustomFormat = @"d.MM.yy HH:mm:ss";
				Assert.AreEqual (@"8.02.07 15:30:45", dt.Text, "#279");
				dt.CustomFormat = @"d.MM.yy H:mm:ss";
				Assert.AreEqual (@"8.02.07 15:30:45", dt.Text, "#280");
				dt.CustomFormat = @"d.M.yy HH:mm:ss";
				Assert.AreEqual (@"8.2.07 15:30:45", dt.Text, "#281");
				dt.CustomFormat = @"d.M.yy H:mm:ss";
				Assert.AreEqual (@"8.2.07 15:30:45", dt.Text, "#282");
				dt.CustomFormat = @"H.mm' Uhr '";
				Assert.AreEqual (@"15.30 Uhr ", dt.Text, "#283");
				dt.CustomFormat = @"d/M/yyyy";
				Assert.AreEqual (@"8/2/2007", dt.Text, "#284");
				dt.CustomFormat = @"dddd, d MMMM yyyy";
				Assert.AreEqual (@"Thursday, 8 February 2007", dt.Text, "#285");
				dt.CustomFormat = @"d MMMM yyyy";
				Assert.AreEqual (@"8 February 2007", dt.Text, "#286");
				dt.CustomFormat = @"dddd, d MMMM yyyy h:mm tt";
				Assert.AreEqual (@"Thursday, 8 February 2007 3:30 PM", dt.Text, "#287");
				dt.CustomFormat = @"dddd, d MMMM yyyy hh:mm tt";
				Assert.AreEqual (@"Thursday, 8 February 2007 03:30 PM", dt.Text, "#288");
				dt.CustomFormat = @"dddd, d MMMM yyyy H:mm";
				Assert.AreEqual (@"Thursday, 8 February 2007 15:30", dt.Text, "#289");
				dt.CustomFormat = @"dddd, d MMMM yyyy HH:mm";
				Assert.AreEqual (@"Thursday, 8 February 2007 15:30", dt.Text, "#290");
				dt.CustomFormat = @"d MMMM yyyy h:mm tt";
				Assert.AreEqual (@"8 February 2007 3:30 PM", dt.Text, "#291");
				dt.CustomFormat = @"d MMMM yyyy hh:mm tt";
				Assert.AreEqual (@"8 February 2007 03:30 PM", dt.Text, "#292");
				dt.CustomFormat = @"d MMMM yyyy H:mm";
				Assert.AreEqual (@"8 February 2007 15:30", dt.Text, "#293");
				dt.CustomFormat = @"d MMMM yyyy HH:mm";
				Assert.AreEqual (@"8 February 2007 15:30", dt.Text, "#294");
				dt.CustomFormat = @"dddd, d MMMM yyyy h:mm:ss tt";
				Assert.AreEqual (@"Thursday, 8 February 2007 3:30:45 PM", dt.Text, "#295");
				dt.CustomFormat = @"dddd, d MMMM yyyy hh:mm:ss tt";
				Assert.AreEqual (@"Thursday, 8 February 2007 03:30:45 PM", dt.Text, "#296");
				dt.CustomFormat = @"dddd, d MMMM yyyy H:mm:ss";
				Assert.AreEqual (@"Thursday, 8 February 2007 15:30:45", dt.Text, "#297");
				dt.CustomFormat = @"dddd, d MMMM yyyy HH:mm:ss";
				Assert.AreEqual (@"Thursday, 8 February 2007 15:30:45", dt.Text, "#298");
				dt.CustomFormat = @"d MMMM yyyy h:mm:ss tt";
				Assert.AreEqual (@"8 February 2007 3:30:45 PM", dt.Text, "#299");
				dt.CustomFormat = @"d MMMM yyyy hh:mm:ss tt";
				Assert.AreEqual (@"8 February 2007 03:30:45 PM", dt.Text, "#300");
				dt.CustomFormat = @"d MMMM yyyy H:mm:ss";
				Assert.AreEqual (@"8 February 2007 15:30:45", dt.Text, "#301");
				dt.CustomFormat = @"d MMMM yyyy HH:mm:ss";
				Assert.AreEqual (@"8 February 2007 15:30:45", dt.Text, "#302");
				dt.CustomFormat = @"d/M/yyyy h:mm tt";
				Assert.AreEqual (@"8/2/2007 3:30 PM", dt.Text, "#303");
				dt.CustomFormat = @"d/M/yyyy hh:mm tt";
				Assert.AreEqual (@"8/2/2007 03:30 PM", dt.Text, "#304");
				dt.CustomFormat = @"d/M/yyyy H:mm";
				Assert.AreEqual (@"8/2/2007 15:30", dt.Text, "#305");
				dt.CustomFormat = @"d/M/yyyy HH:mm";
				Assert.AreEqual (@"8/2/2007 15:30", dt.Text, "#306");
				dt.CustomFormat = @"dd/MM/yyyy h:mm tt";
				Assert.AreEqual (@"08/02/2007 3:30 PM", dt.Text, "#307");
				dt.CustomFormat = @"d/M/yy h:mm tt";
				Assert.AreEqual (@"8/2/07 3:30 PM", dt.Text, "#308");
				dt.CustomFormat = @"d/M/yy hh:mm tt";
				Assert.AreEqual (@"8/2/07 03:30 PM", dt.Text, "#309");
				dt.CustomFormat = @"dd/MM/yy h:mm tt";
				Assert.AreEqual (@"08/02/07 3:30 PM", dt.Text, "#310");
				dt.CustomFormat = @"yyyy-MM-dd h:mm tt";
				Assert.AreEqual (@"2007-02-08 3:30 PM", dt.Text, "#311");
				dt.CustomFormat = @"yyyy-MM-dd hh:mm tt";
				Assert.AreEqual (@"2007-02-08 03:30 PM", dt.Text, "#312");
				dt.CustomFormat = @"d/M/yyyy h:mm:ss tt";
				Assert.AreEqual (@"8/2/2007 3:30:45 PM", dt.Text, "#313");
				dt.CustomFormat = @"d/M/yyyy hh:mm:ss tt";
				Assert.AreEqual (@"8/2/2007 03:30:45 PM", dt.Text, "#314");
				dt.CustomFormat = @"d/M/yyyy H:mm:ss";
				Assert.AreEqual (@"8/2/2007 15:30:45", dt.Text, "#315");
				dt.CustomFormat = @"d/M/yyyy HH:mm:ss";
				Assert.AreEqual (@"8/2/2007 15:30:45", dt.Text, "#316");
				dt.CustomFormat = @"dd/MM/yyyy h:mm:ss tt";
				Assert.AreEqual (@"08/02/2007 3:30:45 PM", dt.Text, "#317");
				dt.CustomFormat = @"d/M/yy h:mm:ss tt";
				Assert.AreEqual (@"8/2/07 3:30:45 PM", dt.Text, "#318");
				dt.CustomFormat = @"d/M/yy hh:mm:ss tt";
				Assert.AreEqual (@"8/2/07 03:30:45 PM", dt.Text, "#319");
				dt.CustomFormat = @"dd/MM/yy h:mm:ss tt";
				Assert.AreEqual (@"08/02/07 3:30:45 PM", dt.Text, "#320");
				dt.CustomFormat = @"yyyy-MM-dd h:mm:ss tt";
				Assert.AreEqual (@"2007-02-08 3:30:45 PM", dt.Text, "#321");
				dt.CustomFormat = @"yyyy-MM-dd hh:mm:ss tt";
				Assert.AreEqual (@"2007-02-08 03:30:45 PM", dt.Text, "#322");
				dt.CustomFormat = @"h:mm tt";
				Assert.AreEqual (@"3:30 PM", dt.Text, "#323");
				dt.CustomFormat = @"h:mm:ss tt";
				Assert.AreEqual (@"3:30:45 PM", dt.Text, "#324");
				dt.CustomFormat = @"M/d/yyyy";
				Assert.AreEqual (@"2/8/2007", dt.Text, "#325");
				dt.CustomFormat = @"M/d/yy";
				Assert.AreEqual (@"2/8/07", dt.Text, "#326");
				dt.CustomFormat = @"MM/dd/yy";
				Assert.AreEqual (@"02/08/07", dt.Text, "#327");
				dt.CustomFormat = @"MM/dd/yyyy";
				Assert.AreEqual (@"02/08/2007", dt.Text, "#328");
				dt.CustomFormat = @"yy/MM/dd";
				Assert.AreEqual (@"07/02/08", dt.Text, "#329");
				dt.CustomFormat = @"dd-MMM-yy";
				Assert.AreEqual (@"08-Feb-07", dt.Text, "#330");
				dt.CustomFormat = @"dddd, MMMM dd, yyyy";
				Assert.AreEqual (@"Thursday, February 08, 2007", dt.Text, "#331");
				dt.CustomFormat = @"MMMM dd, yyyy";
				Assert.AreEqual (@"February 08, 2007", dt.Text, "#332");
				dt.CustomFormat = @"dd MMMM, yyyy";
				Assert.AreEqual (@"08 February, 2007", dt.Text, "#333");
				dt.CustomFormat = @"dddd, MMMM dd, yyyy h:mm tt";
				Assert.AreEqual (@"Thursday, February 08, 2007 3:30 PM", dt.Text, "#334");
				dt.CustomFormat = @"dddd, MMMM dd, yyyy hh:mm tt";
				Assert.AreEqual (@"Thursday, February 08, 2007 03:30 PM", dt.Text, "#335");
				dt.CustomFormat = @"dddd, MMMM dd, yyyy H:mm";
				Assert.AreEqual (@"Thursday, February 08, 2007 15:30", dt.Text, "#336");
				dt.CustomFormat = @"dddd, MMMM dd, yyyy HH:mm";
				Assert.AreEqual (@"Thursday, February 08, 2007 15:30", dt.Text, "#337");
				dt.CustomFormat = @"MMMM dd, yyyy h:mm tt";
				Assert.AreEqual (@"February 08, 2007 3:30 PM", dt.Text, "#338");
				dt.CustomFormat = @"MMMM dd, yyyy hh:mm tt";
				Assert.AreEqual (@"February 08, 2007 03:30 PM", dt.Text, "#339");
				dt.CustomFormat = @"MMMM dd, yyyy H:mm";
				Assert.AreEqual (@"February 08, 2007 15:30", dt.Text, "#340");
				dt.CustomFormat = @"MMMM dd, yyyy HH:mm";
				Assert.AreEqual (@"February 08, 2007 15:30", dt.Text, "#341");
				dt.CustomFormat = @"dddd, dd MMMM, yyyy h:mm tt";
				Assert.AreEqual (@"Thursday, 08 February, 2007 3:30 PM", dt.Text, "#342");
				dt.CustomFormat = @"dddd, dd MMMM, yyyy H:mm";
				Assert.AreEqual (@"Thursday, 08 February, 2007 15:30", dt.Text, "#343");
				dt.CustomFormat = @"dd MMMM, yyyy h:mm tt";
				Assert.AreEqual (@"08 February, 2007 3:30 PM", dt.Text, "#344");
				dt.CustomFormat = @"dd MMMM, yyyy hh:mm tt";
				Assert.AreEqual (@"08 February, 2007 03:30 PM", dt.Text, "#345");
				dt.CustomFormat = @"dd MMMM, yyyy H:mm";
				Assert.AreEqual (@"08 February, 2007 15:30", dt.Text, "#346");
				dt.CustomFormat = @"dd MMMM, yyyy HH:mm";
				Assert.AreEqual (@"08 February, 2007 15:30", dt.Text, "#347");
				dt.CustomFormat = @"dddd, MMMM dd, yyyy h:mm:ss tt";
				Assert.AreEqual (@"Thursday, February 08, 2007 3:30:45 PM", dt.Text, "#348");
				dt.CustomFormat = @"dddd, MMMM dd, yyyy hh:mm:ss tt";
				Assert.AreEqual (@"Thursday, February 08, 2007 03:30:45 PM", dt.Text, "#349");
				dt.CustomFormat = @"dddd, MMMM dd, yyyy H:mm:ss";
				Assert.AreEqual (@"Thursday, February 08, 2007 15:30:45", dt.Text, "#350");
				dt.CustomFormat = @"dddd, MMMM dd, yyyy HH:mm:ss";
				Assert.AreEqual (@"Thursday, February 08, 2007 15:30:45", dt.Text, "#351");
				dt.CustomFormat = @"MMMM dd, yyyy h:mm:ss tt";
				Assert.AreEqual (@"February 08, 2007 3:30:45 PM", dt.Text, "#352");
				dt.CustomFormat = @"MMMM dd, yyyy hh:mm:ss tt";
				Assert.AreEqual (@"February 08, 2007 03:30:45 PM", dt.Text, "#353");
				dt.CustomFormat = @"MMMM dd, yyyy H:mm:ss";
				Assert.AreEqual (@"February 08, 2007 15:30:45", dt.Text, "#354");
				dt.CustomFormat = @"MMMM dd, yyyy HH:mm:ss";
				Assert.AreEqual (@"February 08, 2007 15:30:45", dt.Text, "#355");
				dt.CustomFormat = @"dddd, dd MMMM, yyyy h:mm:ss tt";
				Assert.AreEqual (@"Thursday, 08 February, 2007 3:30:45 PM", dt.Text, "#356");
				dt.CustomFormat = @"dddd, dd MMMM, yyyy H:mm:ss";
				Assert.AreEqual (@"Thursday, 08 February, 2007 15:30:45", dt.Text, "#357");
				dt.CustomFormat = @"dd MMMM, yyyy h:mm:ss tt";
				Assert.AreEqual (@"08 February, 2007 3:30:45 PM", dt.Text, "#358");
				dt.CustomFormat = @"dd MMMM, yyyy hh:mm:ss tt";
				Assert.AreEqual (@"08 February, 2007 03:30:45 PM", dt.Text, "#359");
				dt.CustomFormat = @"dd MMMM, yyyy H:mm:ss";
				Assert.AreEqual (@"08 February, 2007 15:30:45", dt.Text, "#360");
				dt.CustomFormat = @"dd MMMM, yyyy HH:mm:ss";
				Assert.AreEqual (@"08 February, 2007 15:30:45", dt.Text, "#361");
				dt.CustomFormat = @"M/d/yyyy h:mm tt";
				Assert.AreEqual (@"2/8/2007 3:30 PM", dt.Text, "#362");
				dt.CustomFormat = @"M/d/yyyy hh:mm tt";
				Assert.AreEqual (@"2/8/2007 03:30 PM", dt.Text, "#363");
				dt.CustomFormat = @"M/d/yyyy H:mm";
				Assert.AreEqual (@"2/8/2007 15:30", dt.Text, "#364");
				dt.CustomFormat = @"M/d/yyyy HH:mm";
				Assert.AreEqual (@"2/8/2007 15:30", dt.Text, "#365");
				dt.CustomFormat = @"M/d/yy h:mm tt";
				Assert.AreEqual (@"2/8/07 3:30 PM", dt.Text, "#366");
				dt.CustomFormat = @"M/d/yy hh:mm tt";
				Assert.AreEqual (@"2/8/07 03:30 PM", dt.Text, "#367");
				dt.CustomFormat = @"M/d/yy H:mm";
				Assert.AreEqual (@"2/8/07 15:30", dt.Text, "#368");
				dt.CustomFormat = @"M/d/yy HH:mm";
				Assert.AreEqual (@"2/8/07 15:30", dt.Text, "#369");
				dt.CustomFormat = @"MM/dd/yy h:mm tt";
				Assert.AreEqual (@"02/08/07 3:30 PM", dt.Text, "#370");
				dt.CustomFormat = @"MM/dd/yy hh:mm tt";
				Assert.AreEqual (@"02/08/07 03:30 PM", dt.Text, "#371");
				dt.CustomFormat = @"MM/dd/yy H:mm";
				Assert.AreEqual (@"02/08/07 15:30", dt.Text, "#372");
				dt.CustomFormat = @"MM/dd/yy HH:mm";
				Assert.AreEqual (@"02/08/07 15:30", dt.Text, "#373");
				dt.CustomFormat = @"MM/dd/yyyy h:mm tt";
				Assert.AreEqual (@"02/08/2007 3:30 PM", dt.Text, "#374");
				dt.CustomFormat = @"MM/dd/yyyy hh:mm tt";
				Assert.AreEqual (@"02/08/2007 03:30 PM", dt.Text, "#375");
				dt.CustomFormat = @"MM/dd/yyyy H:mm";
				Assert.AreEqual (@"02/08/2007 15:30", dt.Text, "#376");
				dt.CustomFormat = @"MM/dd/yyyy HH:mm";
				Assert.AreEqual (@"02/08/2007 15:30", dt.Text, "#377");
				dt.CustomFormat = @"yy/MM/dd h:mm tt";
				Assert.AreEqual (@"07/02/08 3:30 PM", dt.Text, "#378");
				dt.CustomFormat = @"yy/MM/dd hh:mm tt";
				Assert.AreEqual (@"07/02/08 03:30 PM", dt.Text, "#379");
				dt.CustomFormat = @"yy/MM/dd H:mm";
				Assert.AreEqual (@"07/02/08 15:30", dt.Text, "#380");
				dt.CustomFormat = @"yy/MM/dd HH:mm";
				Assert.AreEqual (@"07/02/08 15:30", dt.Text, "#381");
				dt.CustomFormat = @"dd-MMM-yy h:mm tt";
				Assert.AreEqual (@"08-Feb-07 3:30 PM", dt.Text, "#382");
				dt.CustomFormat = @"dd-MMM-yy hh:mm tt";
				Assert.AreEqual (@"08-Feb-07 03:30 PM", dt.Text, "#383");
				dt.CustomFormat = @"dd-MMM-yy H:mm";
				Assert.AreEqual (@"08-Feb-07 15:30", dt.Text, "#384");
				dt.CustomFormat = @"dd-MMM-yy HH:mm";
				Assert.AreEqual (@"08-Feb-07 15:30", dt.Text, "#385");
				dt.CustomFormat = @"M/d/yyyy h:mm:ss tt";
				Assert.AreEqual (@"2/8/2007 3:30:45 PM", dt.Text, "#386");
				dt.CustomFormat = @"M/d/yyyy hh:mm:ss tt";
				Assert.AreEqual (@"2/8/2007 03:30:45 PM", dt.Text, "#387");
				dt.CustomFormat = @"M/d/yyyy H:mm:ss";
				Assert.AreEqual (@"2/8/2007 15:30:45", dt.Text, "#388");
				dt.CustomFormat = @"M/d/yyyy HH:mm:ss";
				Assert.AreEqual (@"2/8/2007 15:30:45", dt.Text, "#389");
				dt.CustomFormat = @"M/d/yy h:mm:ss tt";
				Assert.AreEqual (@"2/8/07 3:30:45 PM", dt.Text, "#390");
				dt.CustomFormat = @"M/d/yy hh:mm:ss tt";
				Assert.AreEqual (@"2/8/07 03:30:45 PM", dt.Text, "#391");
				dt.CustomFormat = @"M/d/yy H:mm:ss";
				Assert.AreEqual (@"2/8/07 15:30:45", dt.Text, "#392");
				dt.CustomFormat = @"M/d/yy HH:mm:ss";
				Assert.AreEqual (@"2/8/07 15:30:45", dt.Text, "#393");
				dt.CustomFormat = @"MM/dd/yy h:mm:ss tt";
				Assert.AreEqual (@"02/08/07 3:30:45 PM", dt.Text, "#394");
				dt.CustomFormat = @"MM/dd/yy hh:mm:ss tt";
				Assert.AreEqual (@"02/08/07 03:30:45 PM", dt.Text, "#395");
				dt.CustomFormat = @"MM/dd/yy H:mm:ss";
				Assert.AreEqual (@"02/08/07 15:30:45", dt.Text, "#396");
				dt.CustomFormat = @"MM/dd/yy HH:mm:ss";
				Assert.AreEqual (@"02/08/07 15:30:45", dt.Text, "#397");
				dt.CustomFormat = @"MM/dd/yyyy h:mm:ss tt";
				Assert.AreEqual (@"02/08/2007 3:30:45 PM", dt.Text, "#398");
				dt.CustomFormat = @"MM/dd/yyyy hh:mm:ss tt";
				Assert.AreEqual (@"02/08/2007 03:30:45 PM", dt.Text, "#399");
				dt.CustomFormat = @"MM/dd/yyyy H:mm:ss";
				Assert.AreEqual (@"02/08/2007 15:30:45", dt.Text, "#400");
				dt.CustomFormat = @"MM/dd/yyyy HH:mm:ss";
				Assert.AreEqual (@"02/08/2007 15:30:45", dt.Text, "#401");
				dt.CustomFormat = @"yy/MM/dd h:mm:ss tt";
				Assert.AreEqual (@"07/02/08 3:30:45 PM", dt.Text, "#402");
				dt.CustomFormat = @"yy/MM/dd hh:mm:ss tt";
				Assert.AreEqual (@"07/02/08 03:30:45 PM", dt.Text, "#403");
				dt.CustomFormat = @"yy/MM/dd H:mm:ss";
				Assert.AreEqual (@"07/02/08 15:30:45", dt.Text, "#404");
				dt.CustomFormat = @"yy/MM/dd HH:mm:ss";
				Assert.AreEqual (@"07/02/08 15:30:45", dt.Text, "#405");
				dt.CustomFormat = @"dd-MMM-yy h:mm:ss tt";
				Assert.AreEqual (@"08-Feb-07 3:30:45 PM", dt.Text, "#406");
				dt.CustomFormat = @"dd-MMM-yy hh:mm:ss tt";
				Assert.AreEqual (@"08-Feb-07 03:30:45 PM", dt.Text, "#407");
				dt.CustomFormat = @"dd-MMM-yy H:mm:ss";
				Assert.AreEqual (@"08-Feb-07 15:30:45", dt.Text, "#408");
				dt.CustomFormat = @"dd-MMM-yy HH:mm:ss";
				Assert.AreEqual (@"08-Feb-07 15:30:45", dt.Text, "#409");
				dt.CustomFormat = @"MMMM dd";
				Assert.AreEqual (@"February 08", dt.Text, "#410");
				dt.CustomFormat = @"d. MMMM'ta 'yyyy";
				Assert.AreEqual (@"8. Februaryta 2007", dt.Text, "#411");
				dt.CustomFormat = @"dd. MMMM'ta 'yyyy";
				Assert.AreEqual (@"08. Februaryta 2007", dt.Text, "#412");
				dt.CustomFormat = @"d. MMMM'ta 'yyyy H:mm";
				Assert.AreEqual (@"8. Februaryta 2007 15:30", dt.Text, "#413");
				dt.CustomFormat = @"d. MMMM'ta 'yyyy HH:mm";
				Assert.AreEqual (@"8. Februaryta 2007 15:30", dt.Text, "#414");
				dt.CustomFormat = @"dd. MMMM'ta 'yyyy H:mm";
				Assert.AreEqual (@"08. Februaryta 2007 15:30", dt.Text, "#415");
				dt.CustomFormat = @"dd. MMMM'ta 'yyyy HH:mm";
				Assert.AreEqual (@"08. Februaryta 2007 15:30", dt.Text, "#416");
				dt.CustomFormat = @"d. MMMM'ta 'yyyy H:mm:ss";
				Assert.AreEqual (@"8. Februaryta 2007 15:30:45", dt.Text, "#417");
				dt.CustomFormat = @"d. MMMM'ta 'yyyy HH:mm:ss";
				Assert.AreEqual (@"8. Februaryta 2007 15:30:45", dt.Text, "#418");
				dt.CustomFormat = @"dd. MMMM'ta 'yyyy H:mm:ss";
				Assert.AreEqual (@"08. Februaryta 2007 15:30:45", dt.Text, "#419");
				dt.CustomFormat = @"dd. MMMM'ta 'yyyy HH:mm:ss";
				Assert.AreEqual (@"08. Februaryta 2007 15:30:45", dt.Text, "#420");
				dt.CustomFormat = @"d. MMMM'ta'";
				Assert.AreEqual (@"8. Februaryta", dt.Text, "#421");
				dt.CustomFormat = @"dddd d MMMM yyyy";
				Assert.AreEqual (@"Thursday 8 February 2007", dt.Text, "#422");
				dt.CustomFormat = @"d MMM yy";
				Assert.AreEqual (@"8 Feb 07", dt.Text, "#423");
				dt.CustomFormat = @"dddd d MMMM yyyy HH:mm";
				Assert.AreEqual (@"Thursday 8 February 2007 15:30", dt.Text, "#424");
				dt.CustomFormat = @"dddd d MMMM yyyy H:mm";
				Assert.AreEqual (@"Thursday 8 February 2007 15:30", dt.Text, "#425");
				dt.CustomFormat = @"dddd d MMMM yyyy HH.mm";
				Assert.AreEqual (@"Thursday 8 February 2007 15.30", dt.Text, "#426");
				dt.CustomFormat = @"dddd d MMMM yyyy HH' h 'mm";
				Assert.AreEqual (@"Thursday 8 February 2007 15 h 30", dt.Text, "#427");
				dt.CustomFormat = @"d MMM yy HH:mm";
				Assert.AreEqual (@"8 Feb 07 15:30", dt.Text, "#428");
				dt.CustomFormat = @"d MMM yy H:mm";
				Assert.AreEqual (@"8 Feb 07 15:30", dt.Text, "#429");
				dt.CustomFormat = @"d MMM yy HH.mm";
				Assert.AreEqual (@"8 Feb 07 15.30", dt.Text, "#430");
				dt.CustomFormat = @"d MMM yy HH' h 'mm";
				Assert.AreEqual (@"8 Feb 07 15 h 30", dt.Text, "#431");
				dt.CustomFormat = @"d MMMM yyyy HH.mm";
				Assert.AreEqual (@"8 February 2007 15.30", dt.Text, "#432");
				dt.CustomFormat = @"d MMMM yyyy HH' h 'mm";
				Assert.AreEqual (@"8 February 2007 15 h 30", dt.Text, "#433");
				dt.CustomFormat = @"dddd d MMMM yyyy HH:mm:ss";
				Assert.AreEqual (@"Thursday 8 February 2007 15:30:45", dt.Text, "#434");
				dt.CustomFormat = @"dddd d MMMM yyyy H:mm:ss";
				Assert.AreEqual (@"Thursday 8 February 2007 15:30:45", dt.Text, "#435");
				dt.CustomFormat = @"d MMM yy HH:mm:ss";
				Assert.AreEqual (@"8 Feb 07 15:30:45", dt.Text, "#436");
				dt.CustomFormat = @"d MMM yy H:mm:ss";
				Assert.AreEqual (@"8 Feb 07 15:30:45", dt.Text, "#437");
				dt.CustomFormat = @"dd/MM/yyyy HH.mm";
				Assert.AreEqual (@"08/02/2007 15.30", dt.Text, "#438");
				dt.CustomFormat = @"dd/MM/yyyy HH' h 'mm";
				Assert.AreEqual (@"08/02/2007 15 h 30", dt.Text, "#439");
				dt.CustomFormat = @"dd/MM/yy HH.mm";
				Assert.AreEqual (@"08/02/07 15.30", dt.Text, "#440");
				dt.CustomFormat = @"dd/MM/yy HH' h 'mm";
				Assert.AreEqual (@"08/02/07 15 h 30", dt.Text, "#441");
				dt.CustomFormat = @"dd.MM.yy HH.mm";
				Assert.AreEqual (@"08.02.07 15.30", dt.Text, "#442");
				dt.CustomFormat = @"dd.MM.yy HH' h 'mm";
				Assert.AreEqual (@"08.02.07 15 h 30", dt.Text, "#443");
				dt.CustomFormat = @"dd-MM-yy HH.mm";
				Assert.AreEqual (@"08-02-07 15.30", dt.Text, "#444");
				dt.CustomFormat = @"dd-MM-yy HH' h 'mm";
				Assert.AreEqual (@"08-02-07 15 h 30", dt.Text, "#445");
				dt.CustomFormat = @"yyyy-MM-dd HH.mm";
				Assert.AreEqual (@"2007-02-08 15.30", dt.Text, "#446");
				dt.CustomFormat = @"yyyy-MM-dd HH' h 'mm";
				Assert.AreEqual (@"2007-02-08 15 h 30", dt.Text, "#447");
				dt.CustomFormat = @"d MMMM";
				Assert.AreEqual (@"8 February", dt.Text, "#448");
				dt.CustomFormat = @"HH.mm";
				Assert.AreEqual (@"15.30", dt.Text, "#449");
				dt.CustomFormat = @"HH' h 'mm";
				Assert.AreEqual (@"15 h 30", dt.Text, "#450");
				dt.CustomFormat = @"dd-MMMM-yyyy";
				Assert.AreEqual (@"08-February-2007", dt.Text, "#451");
				dt.CustomFormat = @"dd בMMMM yyyy";
				Assert.AreEqual (@"08 בFebruary 2007", dt.Text, "#452");
				dt.CustomFormat = @"dd MMM yy";
				Assert.AreEqual (@"08 Feb 07", dt.Text, "#453");
				dt.CustomFormat = @"dddd dd MMMM yyyy";
				Assert.AreEqual (@"Thursday 08 February 2007", dt.Text, "#454");
				dt.CustomFormat = @"dd MMMM yyyy";
				Assert.AreEqual (@"08 February 2007", dt.Text, "#455");
				dt.CustomFormat = @"dddd dd בMMMM yyyy";
				Assert.AreEqual (@"Thursday 08 בFebruary 2007", dt.Text, "#456");
				dt.CustomFormat = @"ddd dd בMMMM yyyy";
				Assert.AreEqual (@"Thu 08 בFebruary 2007", dt.Text, "#457");
				dt.CustomFormat = @"dddd dd MMMM yyyy HH:mm";
				Assert.AreEqual (@"Thursday 08 February 2007 15:30", dt.Text, "#458");
				dt.CustomFormat = @"dddd dd MMMM yyyy hh:mm tt";
				Assert.AreEqual (@"Thursday 08 February 2007 03:30 PM", dt.Text, "#459");
				dt.CustomFormat = @"dd MMMM yyyy HH:mm";
				Assert.AreEqual (@"08 February 2007 15:30", dt.Text, "#460");
				dt.CustomFormat = @"dd MMMM yyyy hh:mm tt";
				Assert.AreEqual (@"08 February 2007 03:30 PM", dt.Text, "#461");
				dt.CustomFormat = @"dddd dd בMMMM yyyy HH:mm";
				Assert.AreEqual (@"Thursday 08 בFebruary 2007 15:30", dt.Text, "#462");
				dt.CustomFormat = @"dddd dd בMMMM yyyy hh:mm tt";
				Assert.AreEqual (@"Thursday 08 בFebruary 2007 03:30 PM", dt.Text, "#463");
				dt.CustomFormat = @"ddd dd בMMMM yyyy HH:mm";
				Assert.AreEqual (@"Thu 08 בFebruary 2007 15:30", dt.Text, "#464");
				dt.CustomFormat = @"ddd dd בMMMM yyyy hh:mm tt";
				Assert.AreEqual (@"Thu 08 בFebruary 2007 03:30 PM", dt.Text, "#465");
				dt.CustomFormat = @"dd בMMMM yyyy HH:mm";
				Assert.AreEqual (@"08 בFebruary 2007 15:30", dt.Text, "#466");
				dt.CustomFormat = @"dd בMMMM yyyy hh:mm tt";
				Assert.AreEqual (@"08 בFebruary 2007 03:30 PM", dt.Text, "#467");
				dt.CustomFormat = @"dddd dd MMMM yyyy HH:mm:ss";
				Assert.AreEqual (@"Thursday 08 February 2007 15:30:45", dt.Text, "#468");
				dt.CustomFormat = @"dddd dd MMMM yyyy hh:mm:ss tt";
				Assert.AreEqual (@"Thursday 08 February 2007 03:30:45 PM", dt.Text, "#469");
				dt.CustomFormat = @"dd MMMM yyyy HH:mm:ss";
				Assert.AreEqual (@"08 February 2007 15:30:45", dt.Text, "#470");
				dt.CustomFormat = @"dd MMMM yyyy hh:mm:ss tt";
				Assert.AreEqual (@"08 February 2007 03:30:45 PM", dt.Text, "#471");
				dt.CustomFormat = @"dddd dd בMMMM yyyy HH:mm:ss";
				Assert.AreEqual (@"Thursday 08 בFebruary 2007 15:30:45", dt.Text, "#472");
				dt.CustomFormat = @"dddd dd בMMMM yyyy hh:mm:ss tt";
				Assert.AreEqual (@"Thursday 08 בFebruary 2007 03:30:45 PM", dt.Text, "#473");
				dt.CustomFormat = @"ddd dd בMMMM yyyy HH:mm:ss";
				Assert.AreEqual (@"Thu 08 בFebruary 2007 15:30:45", dt.Text, "#474");
				dt.CustomFormat = @"ddd dd בMMMM yyyy hh:mm:ss tt";
				Assert.AreEqual (@"Thu 08 בFebruary 2007 03:30:45 PM", dt.Text, "#475");
				dt.CustomFormat = @"dd בMMMM yyyy HH:mm:ss";
				Assert.AreEqual (@"08 בFebruary 2007 15:30:45", dt.Text, "#476");
				dt.CustomFormat = @"dd בMMMM yyyy hh:mm:ss tt";
				Assert.AreEqual (@"08 בFebruary 2007 03:30:45 PM", dt.Text, "#477");
				dt.CustomFormat = @"dd-MM-yy hh:mm tt";
				Assert.AreEqual (@"08-02-07 03:30 PM", dt.Text, "#478");
				dt.CustomFormat = @"dd-MM-yyyy hh:mm tt";
				Assert.AreEqual (@"08-02-2007 03:30 PM", dt.Text, "#479");
				dt.CustomFormat = @"dd-MMMM-yyyy HH:mm";
				Assert.AreEqual (@"08-February-2007 15:30", dt.Text, "#480");
				dt.CustomFormat = @"dd-MMMM-yyyy hh:mm tt";
				Assert.AreEqual (@"08-February-2007 03:30 PM", dt.Text, "#481");
				dt.CustomFormat = @"dd MMM yy HH:mm";
				Assert.AreEqual (@"08 Feb 07 15:30", dt.Text, "#482");
				dt.CustomFormat = @"dd MMM yy hh:mm tt";
				Assert.AreEqual (@"08 Feb 07 03:30 PM", dt.Text, "#483");
				dt.CustomFormat = @"dd-MM-yy hh:mm:ss tt";
				Assert.AreEqual (@"08-02-07 03:30:45 PM", dt.Text, "#484");
				dt.CustomFormat = @"dd-MM-yyyy hh:mm:ss tt";
				Assert.AreEqual (@"08-02-2007 03:30:45 PM", dt.Text, "#485");
				dt.CustomFormat = @"dd-MMMM-yyyy HH:mm:ss";
				Assert.AreEqual (@"08-February-2007 15:30:45", dt.Text, "#486");
				dt.CustomFormat = @"dd-MMMM-yyyy hh:mm:ss tt";
				Assert.AreEqual (@"08-February-2007 03:30:45 PM", dt.Text, "#487");
				dt.CustomFormat = @"dd MMM yy HH:mm:ss";
				Assert.AreEqual (@"08 Feb 07 15:30:45", dt.Text, "#488");
				dt.CustomFormat = @"dd MMM yy hh:mm:ss tt";
				Assert.AreEqual (@"08 Feb 07 03:30:45 PM", dt.Text, "#489");
				dt.CustomFormat = @"yyyy. MM. dd.";
				Assert.AreEqual (@"2007. 02. 08.", dt.Text, "#490");
				dt.CustomFormat = @"yy-MM-dd";
				Assert.AreEqual (@"07-02-08", dt.Text, "#491");
				dt.CustomFormat = @"yyyy. MMMM d.";
				Assert.AreEqual (@"2007. February 8.", dt.Text, "#492");
				dt.CustomFormat = @"yyyy. MMMM d. H:mm";
				Assert.AreEqual (@"2007. February 8. 15:30", dt.Text, "#493");
				dt.CustomFormat = @"yyyy. MMMM d. HH:mm";
				Assert.AreEqual (@"2007. February 8. 15:30", dt.Text, "#494");
				dt.CustomFormat = @"yyyy. MMMM d. h:mm tt";
				Assert.AreEqual (@"2007. February 8. 3:30 PM", dt.Text, "#495");
				dt.CustomFormat = @"yyyy. MMMM d. H:mm:ss";
				Assert.AreEqual (@"2007. February 8. 15:30:45", dt.Text, "#496");
				dt.CustomFormat = @"yyyy. MMMM d. HH:mm:ss";
				Assert.AreEqual (@"2007. February 8. 15:30:45", dt.Text, "#497");
				dt.CustomFormat = @"yyyy. MMMM d. h:mm:ss tt";
				Assert.AreEqual (@"2007. February 8. 3:30:45 PM", dt.Text, "#498");
				dt.CustomFormat = @"yyyy. MM. dd. H:mm";
				Assert.AreEqual (@"2007. 02. 08. 15:30", dt.Text, "#499");
				dt.CustomFormat = @"yyyy. MM. dd. HH:mm";
				Assert.AreEqual (@"2007. 02. 08. 15:30", dt.Text, "#500");
				dt.CustomFormat = @"yyyy. MM. dd. h:mm tt";
				Assert.AreEqual (@"2007. 02. 08. 3:30 PM", dt.Text, "#501");
				dt.CustomFormat = @"yy-MM-dd H:mm";
				Assert.AreEqual (@"07-02-08 15:30", dt.Text, "#502");
				dt.CustomFormat = @"yy-MM-dd HH:mm";
				Assert.AreEqual (@"07-02-08 15:30", dt.Text, "#503");
				dt.CustomFormat = @"yy-MM-dd h:mm tt";
				Assert.AreEqual (@"07-02-08 3:30 PM", dt.Text, "#504");
				dt.CustomFormat = @"yyyy. MM. dd. H:mm:ss";
				Assert.AreEqual (@"2007. 02. 08. 15:30:45", dt.Text, "#505");
				dt.CustomFormat = @"yyyy. MM. dd. HH:mm:ss";
				Assert.AreEqual (@"2007. 02. 08. 15:30:45", dt.Text, "#506");
				dt.CustomFormat = @"yyyy. MM. dd. h:mm:ss tt";
				Assert.AreEqual (@"2007. 02. 08. 3:30:45 PM", dt.Text, "#507");
				dt.CustomFormat = @"yy-MM-dd H:mm:ss";
				Assert.AreEqual (@"07-02-08 15:30:45", dt.Text, "#508");
				dt.CustomFormat = @"yy-MM-dd HH:mm:ss";
				Assert.AreEqual (@"07-02-08 15:30:45", dt.Text, "#509");
				dt.CustomFormat = @"yy-MM-dd h:mm:ss tt";
				Assert.AreEqual (@"07-02-08 3:30:45 PM", dt.Text, "#510");
				dt.CustomFormat = @"MMMM d.";
				Assert.AreEqual (@"February 8.", dt.Text, "#511");
				dt.CustomFormat = @"yyyy. MMMM";
				Assert.AreEqual (@"2007. February", dt.Text, "#512");
				dt.CustomFormat = @"d. M. yyyy.";
				Assert.AreEqual (@"8. 2. 2007.", dt.Text, "#513");
				dt.CustomFormat = @"d. M. '\''yy.";
				Assert.AreEqual (@"8. 2. \'yy.", dt.Text, "#514");
				dt.CustomFormat = @"yy MM dd";
				Assert.AreEqual (@"07 02 08", dt.Text, "#515");
				dt.CustomFormat = @"d. M. yyyy. HH:mm";
				Assert.AreEqual (@"8. 2. 2007. 15:30", dt.Text, "#516");
				dt.CustomFormat = @"d. M. yyyy. H:mm";
				Assert.AreEqual (@"8. 2. 2007. 15:30", dt.Text, "#517");
				dt.CustomFormat = @"d. M. '\''yy. HH:mm";
				Assert.AreEqual (@"8. 2. \'yy. HH:mm", dt.Text, "#518");
				dt.CustomFormat = @"d. M. '\''yy. H:mm";
				Assert.AreEqual (@"8. 2. \'yy. H:mm", dt.Text, "#519");
				dt.CustomFormat = @"yy MM dd HH:mm";
				Assert.AreEqual (@"07 02 08 15:30", dt.Text, "#520");
				dt.CustomFormat = @"yy MM dd H:mm";
				Assert.AreEqual (@"07 02 08 15:30", dt.Text, "#521");
				dt.CustomFormat = @"d. M. yyyy. HH:mm:ss";
				Assert.AreEqual (@"8. 2. 2007. 15:30:45", dt.Text, "#522");
				dt.CustomFormat = @"d. M. yyyy. H:mm:ss";
				Assert.AreEqual (@"8. 2. 2007. 15:30:45", dt.Text, "#523");
				dt.CustomFormat = @"d. M. '\''yy. HH:mm:ss";
				Assert.AreEqual (@"8. 2. \'yy. HH:mm:ss", dt.Text, "#524");
				dt.CustomFormat = @"d. M. '\''yy. H:mm:ss";
				Assert.AreEqual (@"8. 2. \'yy. H:mm:ss", dt.Text, "#525");
				dt.CustomFormat = @"yy MM dd HH:mm:ss";
				Assert.AreEqual (@"07 02 08 15:30:45", dt.Text, "#526");
				dt.CustomFormat = @"yy MM dd H:mm:ss";
				Assert.AreEqual (@"07 02 08 15:30:45", dt.Text, "#527");
				dt.CustomFormat = @"dd.M.yy";
				Assert.AreEqual (@"08.2.07", dt.Text, "#528");
				dt.CustomFormat = @"d-MMM-yy";
				Assert.AreEqual (@"8-Feb-07", dt.Text, "#529");
				dt.CustomFormat = @"dddd d MMMM yyyy H.mm";
				Assert.AreEqual (@"Thursday 8 February 2007 15.30", dt.Text, "#530");
				dt.CustomFormat = @"d-MMM-yy H.mm";
				Assert.AreEqual (@"8-Feb-07 15.30", dt.Text, "#531");
				dt.CustomFormat = @"d-MMM-yy HH.mm";
				Assert.AreEqual (@"8-Feb-07 15.30", dt.Text, "#532");
				dt.CustomFormat = @"d-MMM-yy HH:mm";
				Assert.AreEqual (@"8-Feb-07 15:30", dt.Text, "#533");
				dt.CustomFormat = @"d-MMM-yy H:mm";
				Assert.AreEqual (@"8-Feb-07 15:30", dt.Text, "#534");
				dt.CustomFormat = @"d MMMM yyyy H.mm";
				Assert.AreEqual (@"8 February 2007 15.30", dt.Text, "#535");
				dt.CustomFormat = @"dddd d MMMM yyyy H.mm.ss";
				Assert.AreEqual (@"Thursday 8 February 2007 15.30.45", dt.Text, "#536");
				dt.CustomFormat = @"dddd d MMMM yyyy HH.mm.ss";
				Assert.AreEqual (@"Thursday 8 February 2007 15.30.45", dt.Text, "#537");
				dt.CustomFormat = @"d-MMM-yy H.mm.ss";
				Assert.AreEqual (@"8-Feb-07 15.30.45", dt.Text, "#538");
				dt.CustomFormat = @"d-MMM-yy HH.mm.ss";
				Assert.AreEqual (@"8-Feb-07 15.30.45", dt.Text, "#539");
				dt.CustomFormat = @"d-MMM-yy HH:mm:ss";
				Assert.AreEqual (@"8-Feb-07 15:30:45", dt.Text, "#540");
				dt.CustomFormat = @"d MMMM yyyy H.mm.ss";
				Assert.AreEqual (@"8 February 2007 15.30.45", dt.Text, "#541");
				dt.CustomFormat = @"d MMMM yyyy HH.mm.ss";
				Assert.AreEqual (@"8 February 2007 15.30.45", dt.Text, "#542");
				dt.CustomFormat = @"dd/MM/yyyy H.mm";
				Assert.AreEqual (@"08/02/2007 15.30", dt.Text, "#543");
				dt.CustomFormat = @"dd/MM/yy H.mm";
				Assert.AreEqual (@"08/02/07 15.30", dt.Text, "#544");
				dt.CustomFormat = @"dd.M.yy H.mm";
				Assert.AreEqual (@"08.2.07 15.30", dt.Text, "#545");
				dt.CustomFormat = @"dd.M.yy HH.mm";
				Assert.AreEqual (@"08.2.07 15.30", dt.Text, "#546");
				dt.CustomFormat = @"dd.M.yy HH:mm";
				Assert.AreEqual (@"08.2.07 15:30", dt.Text, "#547");
				dt.CustomFormat = @"dd.M.yy H:mm";
				Assert.AreEqual (@"08.2.07 15:30", dt.Text, "#548");
				dt.CustomFormat = @"d/M/yy H.mm";
				Assert.AreEqual (@"8/2/07 15.30", dt.Text, "#549");
				dt.CustomFormat = @"d/M/yy HH.mm";
				Assert.AreEqual (@"8/2/07 15.30", dt.Text, "#550");
				dt.CustomFormat = @"dd/MM/yyyy H.mm.ss";
				Assert.AreEqual (@"08/02/2007 15.30.45", dt.Text, "#551");
				dt.CustomFormat = @"dd/MM/yyyy HH.mm.ss";
				Assert.AreEqual (@"08/02/2007 15.30.45", dt.Text, "#552");
				dt.CustomFormat = @"dd/MM/yy H.mm.ss";
				Assert.AreEqual (@"08/02/07 15.30.45", dt.Text, "#553");
				dt.CustomFormat = @"dd/MM/yy HH.mm.ss";
				Assert.AreEqual (@"08/02/07 15.30.45", dt.Text, "#554");
				dt.CustomFormat = @"dd.M.yy H.mm.ss";
				Assert.AreEqual (@"08.2.07 15.30.45", dt.Text, "#555");
				dt.CustomFormat = @"dd.M.yy HH.mm.ss";
				Assert.AreEqual (@"08.2.07 15.30.45", dt.Text, "#556");
				dt.CustomFormat = @"dd.M.yy HH:mm:ss";
				Assert.AreEqual (@"08.2.07 15:30:45", dt.Text, "#557");
				dt.CustomFormat = @"d/M/yy H.mm.ss";
				Assert.AreEqual (@"8/2/07 15.30.45", dt.Text, "#558");
				dt.CustomFormat = @"d/M/yy HH.mm.ss";
				Assert.AreEqual (@"8/2/07 15.30.45", dt.Text, "#559");
				dt.CustomFormat = @"yyyy-MM-dd HH.mm.ss";
				Assert.AreEqual (@"2007-02-08 15.30.45", dt.Text, "#560");
				dt.CustomFormat = @"HH.mm.ss";
				Assert.AreEqual (@"15.30.45", dt.Text, "#561");
				dt.CustomFormat = @"yy/M/d";
				Assert.AreEqual (@"07/2/8", dt.Text, "#562");
				dt.CustomFormat = @"yy/MM/dd' ('ddd')'";
				Assert.AreEqual (@"07/02/08 (Thu)", dt.Text, "#563");
				dt.CustomFormat = @"yy/M/d' ('ddd')'";
				Assert.AreEqual (@"07/2/8 (Thu)", dt.Text, "#564");
				dt.CustomFormat = @"yyyy/MM/dd' ('ddd')'";
				Assert.AreEqual (@"2007/02/08 (Thu)", dt.Text, "#565");
				dt.CustomFormat = @"yyyy'年'M'月'd'日' dddd";
				Assert.AreEqual (@"2007年2月8日 Thursday", dt.Text, "#566");
				dt.CustomFormat = @"yyyy'年'MM'月'dd'日' dddd";
				Assert.AreEqual (@"2007年02月08日 Thursday", dt.Text, "#567");
				dt.CustomFormat = @"yyyy'年'M'月'd'日' dddd H:mm";
				Assert.AreEqual (@"2007年2月8日 Thursday 15:30", dt.Text, "#568");
				dt.CustomFormat = @"yyyy'年'M'月'd'日' dddd HH:mm";
				Assert.AreEqual (@"2007年2月8日 Thursday 15:30", dt.Text, "#569");
				dt.CustomFormat = @"yyyy'年'M'月'd'日' dddd tt h:mm";
				Assert.AreEqual (@"2007年2月8日 Thursday PM 3:30", dt.Text, "#570");
				dt.CustomFormat = @"yyyy'年'M'月'd'日' dddd tt hh:mm";
				Assert.AreEqual (@"2007年2月8日 Thursday PM 03:30", dt.Text, "#571");
				dt.CustomFormat = @"yyyy'年'MM'月'dd'日' dddd H:mm";
				Assert.AreEqual (@"2007年02月08日 Thursday 15:30", dt.Text, "#572");
				dt.CustomFormat = @"yyyy'年'MM'月'dd'日' dddd HH:mm";
				Assert.AreEqual (@"2007年02月08日 Thursday 15:30", dt.Text, "#573");
				dt.CustomFormat = @"yyyy'年'MM'月'dd'日' dddd tt h:mm";
				Assert.AreEqual (@"2007年02月08日 Thursday PM 3:30", dt.Text, "#574");
				dt.CustomFormat = @"yyyy'年'MM'月'dd'日' dddd tt hh:mm";
				Assert.AreEqual (@"2007年02月08日 Thursday PM 03:30", dt.Text, "#575");
				dt.CustomFormat = @"yyyy'年'M'月'd'日' dddd H:mm:ss";
				Assert.AreEqual (@"2007年2月8日 Thursday 15:30:45", dt.Text, "#576");
				dt.CustomFormat = @"yyyy'年'M'月'd'日' dddd HH:mm:ss";
				Assert.AreEqual (@"2007年2月8日 Thursday 15:30:45", dt.Text, "#577");
				dt.CustomFormat = @"yyyy'年'M'月'd'日' dddd tt h:mm:ss";
				Assert.AreEqual (@"2007年2月8日 Thursday PM 3:30:45", dt.Text, "#578");
				dt.CustomFormat = @"yyyy'年'M'月'd'日' dddd tt hh:mm:ss";
				Assert.AreEqual (@"2007年2月8日 Thursday PM 03:30:45", dt.Text, "#579");
				dt.CustomFormat = @"yyyy'年'MM'月'dd'日' dddd H:mm:ss";
				Assert.AreEqual (@"2007年02月08日 Thursday 15:30:45", dt.Text, "#580");
				dt.CustomFormat = @"yyyy'年'MM'月'dd'日' dddd HH:mm:ss";
				Assert.AreEqual (@"2007年02月08日 Thursday 15:30:45", dt.Text, "#581");
				dt.CustomFormat = @"yyyy'年'MM'月'dd'日' dddd tt h:mm:ss";
				Assert.AreEqual (@"2007年02月08日 Thursday PM 3:30:45", dt.Text, "#582");
				dt.CustomFormat = @"yyyy'年'MM'月'dd'日' dddd tt hh:mm:ss";
				Assert.AreEqual (@"2007年02月08日 Thursday PM 03:30:45", dt.Text, "#583");
				dt.CustomFormat = @"yy/MM/dd tt h:mm";
				Assert.AreEqual (@"07/02/08 PM 3:30", dt.Text, "#584");
				dt.CustomFormat = @"yy/MM/dd tt hh:mm";
				Assert.AreEqual (@"07/02/08 PM 03:30", dt.Text, "#585");
				dt.CustomFormat = @"yy/M/d H:mm";
				Assert.AreEqual (@"07/2/8 15:30", dt.Text, "#586");
				dt.CustomFormat = @"yy/M/d HH:mm";
				Assert.AreEqual (@"07/2/8 15:30", dt.Text, "#587");
				dt.CustomFormat = @"yy/M/d tt h:mm";
				Assert.AreEqual (@"07/2/8 PM 3:30", dt.Text, "#588");
				dt.CustomFormat = @"yy/M/d tt hh:mm";
				Assert.AreEqual (@"07/2/8 PM 03:30", dt.Text, "#589");
				dt.CustomFormat = @"yy/MM/dd' ('ddd')' H:mm";
				Assert.AreEqual (@"07/02/08 (Thu) 15:30", dt.Text, "#590");
				dt.CustomFormat = @"yy/MM/dd' ('ddd')' HH:mm";
				Assert.AreEqual (@"07/02/08 (Thu) 15:30", dt.Text, "#591");
				dt.CustomFormat = @"yy/MM/dd' ('ddd')' tt h:mm";
				Assert.AreEqual (@"07/02/08 (Thu) PM 3:30", dt.Text, "#592");
				dt.CustomFormat = @"yy/MM/dd' ('ddd')' tt hh:mm";
				Assert.AreEqual (@"07/02/08 (Thu) PM 03:30", dt.Text, "#593");
				dt.CustomFormat = @"yy/M/d' ('ddd')' H:mm";
				Assert.AreEqual (@"07/2/8 (Thu) 15:30", dt.Text, "#594");
				dt.CustomFormat = @"yy/M/d' ('ddd')' HH:mm";
				Assert.AreEqual (@"07/2/8 (Thu) 15:30", dt.Text, "#595");
				dt.CustomFormat = @"yy/M/d' ('ddd')' tt h:mm";
				Assert.AreEqual (@"07/2/8 (Thu) PM 3:30", dt.Text, "#596");
				dt.CustomFormat = @"yy/M/d' ('ddd')' tt hh:mm";
				Assert.AreEqual (@"07/2/8 (Thu) PM 03:30", dt.Text, "#597");
				dt.CustomFormat = @"yyyy/MM/dd' ('ddd')' H:mm";
				Assert.AreEqual (@"2007/02/08 (Thu) 15:30", dt.Text, "#598");
				dt.CustomFormat = @"yyyy/MM/dd' ('ddd')' HH:mm";
				Assert.AreEqual (@"2007/02/08 (Thu) 15:30", dt.Text, "#599");
				dt.CustomFormat = @"yyyy/MM/dd' ('ddd')' tt h:mm";
				Assert.AreEqual (@"2007/02/08 (Thu) PM 3:30", dt.Text, "#600");
				dt.CustomFormat = @"yyyy/MM/dd' ('ddd')' tt hh:mm";
				Assert.AreEqual (@"2007/02/08 (Thu) PM 03:30", dt.Text, "#601");
				dt.CustomFormat = @"yy/MM/dd tt h:mm:ss";
				Assert.AreEqual (@"07/02/08 PM 3:30:45", dt.Text, "#602");
				dt.CustomFormat = @"yy/MM/dd tt hh:mm:ss";
				Assert.AreEqual (@"07/02/08 PM 03:30:45", dt.Text, "#603");
				dt.CustomFormat = @"yy/M/d H:mm:ss";
				Assert.AreEqual (@"07/2/8 15:30:45", dt.Text, "#604");
				dt.CustomFormat = @"yy/M/d HH:mm:ss";
				Assert.AreEqual (@"07/2/8 15:30:45", dt.Text, "#605");
				dt.CustomFormat = @"yy/M/d tt h:mm:ss";
				Assert.AreEqual (@"07/2/8 PM 3:30:45", dt.Text, "#606");
				dt.CustomFormat = @"yy/M/d tt hh:mm:ss";
				Assert.AreEqual (@"07/2/8 PM 03:30:45", dt.Text, "#607");
				dt.CustomFormat = @"yy/MM/dd' ('ddd')' H:mm:ss";
				Assert.AreEqual (@"07/02/08 (Thu) 15:30:45", dt.Text, "#608");
				dt.CustomFormat = @"yy/MM/dd' ('ddd')' HH:mm:ss";
				Assert.AreEqual (@"07/02/08 (Thu) 15:30:45", dt.Text, "#609");
				dt.CustomFormat = @"yy/MM/dd' ('ddd')' tt h:mm:ss";
				Assert.AreEqual (@"07/02/08 (Thu) PM 3:30:45", dt.Text, "#610");
				dt.CustomFormat = @"yy/MM/dd' ('ddd')' tt hh:mm:ss";
				Assert.AreEqual (@"07/02/08 (Thu) PM 03:30:45", dt.Text, "#611");
				dt.CustomFormat = @"yy/M/d' ('ddd')' H:mm:ss";
				Assert.AreEqual (@"07/2/8 (Thu) 15:30:45", dt.Text, "#612");
				dt.CustomFormat = @"yy/M/d' ('ddd')' HH:mm:ss";
				Assert.AreEqual (@"07/2/8 (Thu) 15:30:45", dt.Text, "#613");
				dt.CustomFormat = @"yy/M/d' ('ddd')' tt h:mm:ss";
				Assert.AreEqual (@"07/2/8 (Thu) PM 3:30:45", dt.Text, "#614");
				dt.CustomFormat = @"yy/M/d' ('ddd')' tt hh:mm:ss";
				Assert.AreEqual (@"07/2/8 (Thu) PM 03:30:45", dt.Text, "#615");
				dt.CustomFormat = @"yyyy/MM/dd' ('ddd')' H:mm:ss";
				Assert.AreEqual (@"2007/02/08 (Thu) 15:30:45", dt.Text, "#616");
				dt.CustomFormat = @"yyyy/MM/dd' ('ddd')' HH:mm:ss";
				Assert.AreEqual (@"2007/02/08 (Thu) 15:30:45", dt.Text, "#617");
				dt.CustomFormat = @"yyyy/MM/dd' ('ddd')' tt h:mm:ss";
				Assert.AreEqual (@"2007/02/08 (Thu) PM 3:30:45", dt.Text, "#618");
				dt.CustomFormat = @"yyyy/MM/dd' ('ddd')' tt hh:mm:ss";
				Assert.AreEqual (@"2007/02/08 (Thu) PM 03:30:45", dt.Text, "#619");
				dt.CustomFormat = @"yyyy'년' M'월' d'일' dddd";
				Assert.AreEqual (@"2007년 2월 8일 Thursday", dt.Text, "#620");
				dt.CustomFormat = @"yyyy'년' M'월' d'일'";
				Assert.AreEqual (@"2007년 2월 8일", dt.Text, "#621");
				dt.CustomFormat = @"yy'년' M'월' d'일' dddd";
				Assert.AreEqual (@"07년 2월 8일 Thursday", dt.Text, "#622");
				dt.CustomFormat = @"yy'년' M'월' d'일'";
				Assert.AreEqual (@"07년 2월 8일", dt.Text, "#623");
				dt.CustomFormat = @"yyyy'년' MM'월' dd'일' dddd";
				Assert.AreEqual (@"2007년 02월 08일 Thursday", dt.Text, "#624");
				dt.CustomFormat = @"yyyy'년' MM'월' dd'일'";
				Assert.AreEqual (@"2007년 02월 08일", dt.Text, "#625");
				dt.CustomFormat = @"yyyy'년' M'월' d'일' dddd tt h:mm";
				Assert.AreEqual (@"2007년 2월 8일 Thursday PM 3:30", dt.Text, "#626");
				dt.CustomFormat = @"yyyy'년' M'월' d'일' dddd tt hh:mm";
				Assert.AreEqual (@"2007년 2월 8일 Thursday PM 03:30", dt.Text, "#627");
				dt.CustomFormat = @"yyyy'년' M'월' d'일' dddd H:mm";
				Assert.AreEqual (@"2007년 2월 8일 Thursday 15:30", dt.Text, "#628");
				dt.CustomFormat = @"yyyy'년' M'월' d'일' dddd HH:mm";
				Assert.AreEqual (@"2007년 2월 8일 Thursday 15:30", dt.Text, "#629");
				dt.CustomFormat = @"yyyy'년' M'월' d'일' tt h:mm";
				Assert.AreEqual (@"2007년 2월 8일 PM 3:30", dt.Text, "#630");
				dt.CustomFormat = @"yyyy'년' M'월' d'일' tt hh:mm";
				Assert.AreEqual (@"2007년 2월 8일 PM 03:30", dt.Text, "#631");
				dt.CustomFormat = @"yyyy'년' M'월' d'일' H:mm";
				Assert.AreEqual (@"2007년 2월 8일 15:30", dt.Text, "#632");
				dt.CustomFormat = @"yyyy'년' M'월' d'일' HH:mm";
				Assert.AreEqual (@"2007년 2월 8일 15:30", dt.Text, "#633");
				dt.CustomFormat = @"yy'년' M'월' d'일' dddd tt h:mm";
				Assert.AreEqual (@"07년 2월 8일 Thursday PM 3:30", dt.Text, "#634");
				dt.CustomFormat = @"yy'년' M'월' d'일' dddd tt hh:mm";
				Assert.AreEqual (@"07년 2월 8일 Thursday PM 03:30", dt.Text, "#635");
				dt.CustomFormat = @"yy'년' M'월' d'일' dddd H:mm";
				Assert.AreEqual (@"07년 2월 8일 Thursday 15:30", dt.Text, "#636");
				dt.CustomFormat = @"yy'년' M'월' d'일' dddd HH:mm";
				Assert.AreEqual (@"07년 2월 8일 Thursday 15:30", dt.Text, "#637");
				dt.CustomFormat = @"yy'년' M'월' d'일' tt h:mm";
				Assert.AreEqual (@"07년 2월 8일 PM 3:30", dt.Text, "#638");
				dt.CustomFormat = @"yy'년' M'월' d'일' tt hh:mm";
				Assert.AreEqual (@"07년 2월 8일 PM 03:30", dt.Text, "#639");
				dt.CustomFormat = @"yy'년' M'월' d'일' H:mm";
				Assert.AreEqual (@"07년 2월 8일 15:30", dt.Text, "#640");
				dt.CustomFormat = @"yy'년' M'월' d'일' HH:mm";
				Assert.AreEqual (@"07년 2월 8일 15:30", dt.Text, "#641");
				dt.CustomFormat = @"yyyy'년' MM'월' dd'일' dddd tt h:mm";
				Assert.AreEqual (@"2007년 02월 08일 Thursday PM 3:30", dt.Text, "#642");
				dt.CustomFormat = @"yyyy'년' MM'월' dd'일' dddd tt hh:mm";
				Assert.AreEqual (@"2007년 02월 08일 Thursday PM 03:30", dt.Text, "#643");
				dt.CustomFormat = @"yyyy'년' MM'월' dd'일' dddd H:mm";
				Assert.AreEqual (@"2007년 02월 08일 Thursday 15:30", dt.Text, "#644");
				dt.CustomFormat = @"yyyy'년' MM'월' dd'일' dddd HH:mm";
				Assert.AreEqual (@"2007년 02월 08일 Thursday 15:30", dt.Text, "#645");
				dt.CustomFormat = @"yyyy'년' MM'월' dd'일' tt h:mm";
				Assert.AreEqual (@"2007년 02월 08일 PM 3:30", dt.Text, "#646");
				dt.CustomFormat = @"yyyy'년' MM'월' dd'일' tt hh:mm";
				Assert.AreEqual (@"2007년 02월 08일 PM 03:30", dt.Text, "#647");
				dt.CustomFormat = @"yyyy'년' MM'월' dd'일' H:mm";
				Assert.AreEqual (@"2007년 02월 08일 15:30", dt.Text, "#648");
				dt.CustomFormat = @"yyyy'년' MM'월' dd'일' HH:mm";
				Assert.AreEqual (@"2007년 02월 08일 15:30", dt.Text, "#649");
				dt.CustomFormat = @"yyyy'년' M'월' d'일' dddd tt h:mm:ss";
				Assert.AreEqual (@"2007년 2월 8일 Thursday PM 3:30:45", dt.Text, "#650");
				dt.CustomFormat = @"yyyy'년' M'월' d'일' dddd tt hh:mm:ss";
				Assert.AreEqual (@"2007년 2월 8일 Thursday PM 03:30:45", dt.Text, "#651");
				dt.CustomFormat = @"yyyy'년' M'월' d'일' dddd H:mm:ss";
				Assert.AreEqual (@"2007년 2월 8일 Thursday 15:30:45", dt.Text, "#652");
				dt.CustomFormat = @"yyyy'년' M'월' d'일' dddd HH:mm:ss";
				Assert.AreEqual (@"2007년 2월 8일 Thursday 15:30:45", dt.Text, "#653");
				dt.CustomFormat = @"yyyy'년' M'월' d'일' tt h:mm:ss";
				Assert.AreEqual (@"2007년 2월 8일 PM 3:30:45", dt.Text, "#654");
				dt.CustomFormat = @"yyyy'년' M'월' d'일' tt hh:mm:ss";
				Assert.AreEqual (@"2007년 2월 8일 PM 03:30:45", dt.Text, "#655");
				dt.CustomFormat = @"yyyy'년' M'월' d'일' H:mm:ss";
				Assert.AreEqual (@"2007년 2월 8일 15:30:45", dt.Text, "#656");
				dt.CustomFormat = @"yyyy'년' M'월' d'일' HH:mm:ss";
				Assert.AreEqual (@"2007년 2월 8일 15:30:45", dt.Text, "#657");
				dt.CustomFormat = @"yy'년' M'월' d'일' dddd tt h:mm:ss";
				Assert.AreEqual (@"07년 2월 8일 Thursday PM 3:30:45", dt.Text, "#658");
				dt.CustomFormat = @"yy'년' M'월' d'일' dddd tt hh:mm:ss";
				Assert.AreEqual (@"07년 2월 8일 Thursday PM 03:30:45", dt.Text, "#659");
				dt.CustomFormat = @"yy'년' M'월' d'일' dddd H:mm:ss";
				Assert.AreEqual (@"07년 2월 8일 Thursday 15:30:45", dt.Text, "#660");
				dt.CustomFormat = @"yy'년' M'월' d'일' dddd HH:mm:ss";
				Assert.AreEqual (@"07년 2월 8일 Thursday 15:30:45", dt.Text, "#661");
				dt.CustomFormat = @"yy'년' M'월' d'일' tt h:mm:ss";
				Assert.AreEqual (@"07년 2월 8일 PM 3:30:45", dt.Text, "#662");
				dt.CustomFormat = @"yy'년' M'월' d'일' tt hh:mm:ss";
				Assert.AreEqual (@"07년 2월 8일 PM 03:30:45", dt.Text, "#663");
				dt.CustomFormat = @"yy'년' M'월' d'일' H:mm:ss";
				Assert.AreEqual (@"07년 2월 8일 15:30:45", dt.Text, "#664");
				dt.CustomFormat = @"yy'년' M'월' d'일' HH:mm:ss";
				Assert.AreEqual (@"07년 2월 8일 15:30:45", dt.Text, "#665");
				dt.CustomFormat = @"yyyy'년' MM'월' dd'일' dddd tt h:mm:ss";
				Assert.AreEqual (@"2007년 02월 08일 Thursday PM 3:30:45", dt.Text, "#666");
				dt.CustomFormat = @"yyyy'년' MM'월' dd'일' dddd tt hh:mm:ss";
				Assert.AreEqual (@"2007년 02월 08일 Thursday PM 03:30:45", dt.Text, "#667");
				dt.CustomFormat = @"yyyy'년' MM'월' dd'일' dddd H:mm:ss";
				Assert.AreEqual (@"2007년 02월 08일 Thursday 15:30:45", dt.Text, "#668");
				dt.CustomFormat = @"yyyy'년' MM'월' dd'일' dddd HH:mm:ss";
				Assert.AreEqual (@"2007년 02월 08일 Thursday 15:30:45", dt.Text, "#669");
				dt.CustomFormat = @"yyyy'년' MM'월' dd'일' tt h:mm:ss";
				Assert.AreEqual (@"2007년 02월 08일 PM 3:30:45", dt.Text, "#670");
				dt.CustomFormat = @"yyyy'년' MM'월' dd'일' tt hh:mm:ss";
				Assert.AreEqual (@"2007년 02월 08일 PM 03:30:45", dt.Text, "#671");
				dt.CustomFormat = @"yyyy'년' MM'월' dd'일' H:mm:ss";
				Assert.AreEqual (@"2007년 02월 08일 15:30:45", dt.Text, "#672");
				dt.CustomFormat = @"yyyy'년' MM'월' dd'일' HH:mm:ss";
				Assert.AreEqual (@"2007년 02월 08일 15:30:45", dt.Text, "#673");
				dt.CustomFormat = @"yy-MM-dd tt h:mm";
				Assert.AreEqual (@"07-02-08 PM 3:30", dt.Text, "#674");
				dt.CustomFormat = @"yy-MM-dd tt hh:mm";
				Assert.AreEqual (@"07-02-08 PM 03:30", dt.Text, "#675");
				dt.CustomFormat = @"yy-M-d tt h:mm";
				Assert.AreEqual (@"07-2-8 PM 3:30", dt.Text, "#676");
				dt.CustomFormat = @"yy-M-d tt hh:mm";
				Assert.AreEqual (@"07-2-8 PM 03:30", dt.Text, "#677");
				dt.CustomFormat = @"yyyy-M-d tt h:mm";
				Assert.AreEqual (@"2007-2-8 PM 3:30", dt.Text, "#678");
				dt.CustomFormat = @"yyyy-M-d tt hh:mm";
				Assert.AreEqual (@"2007-2-8 PM 03:30", dt.Text, "#679");
				dt.CustomFormat = @"yy-MM-dd tt h:mm:ss";
				Assert.AreEqual (@"07-02-08 PM 3:30:45", dt.Text, "#680");
				dt.CustomFormat = @"yy-MM-dd tt hh:mm:ss";
				Assert.AreEqual (@"07-02-08 PM 03:30:45", dt.Text, "#681");
				dt.CustomFormat = @"yy-M-d tt h:mm:ss";
				Assert.AreEqual (@"07-2-8 PM 3:30:45", dt.Text, "#682");
				dt.CustomFormat = @"yy-M-d tt hh:mm:ss";
				Assert.AreEqual (@"07-2-8 PM 03:30:45", dt.Text, "#683");
				dt.CustomFormat = @"yyyy-M-d tt h:mm:ss";
				Assert.AreEqual (@"2007-2-8 PM 3:30:45", dt.Text, "#684");
				dt.CustomFormat = @"yyyy-M-d tt hh:mm:ss";
				Assert.AreEqual (@"2007-2-8 PM 03:30:45", dt.Text, "#685");
				dt.CustomFormat = @"M'월' d'일'";
				Assert.AreEqual (@"2월 8일", dt.Text, "#686");
				dt.CustomFormat = @"yyyy'년' M'월'";
				Assert.AreEqual (@"2007년 2월", dt.Text, "#687");
				dt.CustomFormat = @"d-M-yyyy";
				Assert.AreEqual (@"8-2-2007", dt.Text, "#688");
				dt.CustomFormat = @"dddd d MMMM yyyy HH.mm' uur'";
				Assert.AreEqual (@"Thursday 8 February 2007 15.30 uur", dt.Text, "#689");
				dt.CustomFormat = @"dddd d MMMM yyyy HH:mm' uur'";
				Assert.AreEqual (@"Thursday 8 February 2007 15:30 uur", dt.Text, "#690");
				dt.CustomFormat = @"d-MMM-yy HH.mm' uur'";
				Assert.AreEqual (@"8-Feb-07 15.30 uur", dt.Text, "#691");
				dt.CustomFormat = @"d-MMM-yy HH:mm' uur'";
				Assert.AreEqual (@"8-Feb-07 15:30 uur", dt.Text, "#692");
				dt.CustomFormat = @"d MMMM yyyy HH.mm' uur'";
				Assert.AreEqual (@"8 February 2007 15.30 uur", dt.Text, "#693");
				dt.CustomFormat = @"d MMMM yyyy HH:mm' uur'";
				Assert.AreEqual (@"8 February 2007 15:30 uur", dt.Text, "#694");
				dt.CustomFormat = @"d MMM yy H.mm";
				Assert.AreEqual (@"8 Feb 07 15.30", dt.Text, "#695");
				dt.CustomFormat = @"d MMM yy HH.mm' uur'";
				Assert.AreEqual (@"8 Feb 07 15.30 uur", dt.Text, "#696");
				dt.CustomFormat = @"d MMM yy HH:mm' uur'";
				Assert.AreEqual (@"8 Feb 07 15:30 uur", dt.Text, "#697");
				dt.CustomFormat = @"dddd d MMMM yyyy HH.mm.ss' uur'";
				Assert.AreEqual (@"Thursday 8 February 2007 15.30.45 uur", dt.Text, "#698");
				dt.CustomFormat = @"dddd d MMMM yyyy HH:mm:ss' uur'";
				Assert.AreEqual (@"Thursday 8 February 2007 15:30:45 uur", dt.Text, "#699");
				dt.CustomFormat = @"d-MMM-yy H:mm:ss";
				Assert.AreEqual (@"8-Feb-07 15:30:45", dt.Text, "#700");
				dt.CustomFormat = @"d-MMM-yy HH.mm.ss' uur'";
				Assert.AreEqual (@"8-Feb-07 15.30.45 uur", dt.Text, "#701");
				dt.CustomFormat = @"d-MMM-yy HH:mm:ss' uur'";
				Assert.AreEqual (@"8-Feb-07 15:30:45 uur", dt.Text, "#702");
				dt.CustomFormat = @"d MMMM yyyy HH.mm.ss' uur'";
				Assert.AreEqual (@"8 February 2007 15.30.45 uur", dt.Text, "#703");
				dt.CustomFormat = @"d MMMM yyyy HH:mm:ss' uur'";
				Assert.AreEqual (@"8 February 2007 15:30:45 uur", dt.Text, "#704");
				dt.CustomFormat = @"d MMM yy HH.mm.ss' uur'";
				Assert.AreEqual (@"8 Feb 07 15.30.45 uur", dt.Text, "#705");
				dt.CustomFormat = @"d MMM yy HH:mm:ss' uur'";
				Assert.AreEqual (@"8 Feb 07 15:30:45 uur", dt.Text, "#706");
				dt.CustomFormat = @"d-M-yyyy H:mm";
				Assert.AreEqual (@"8-2-2007 15:30", dt.Text, "#707");
				dt.CustomFormat = @"d-M-yyyy HH:mm";
				Assert.AreEqual (@"8-2-2007 15:30", dt.Text, "#708");
				dt.CustomFormat = @"d-M-yyyy H.mm";
				Assert.AreEqual (@"8-2-2007 15.30", dt.Text, "#709");
				dt.CustomFormat = @"d-M-yyyy HH.mm' uur'";
				Assert.AreEqual (@"8-2-2007 15.30 uur", dt.Text, "#710");
				dt.CustomFormat = @"d-M-yyyy HH:mm' uur'";
				Assert.AreEqual (@"8-2-2007 15:30 uur", dt.Text, "#711");
				dt.CustomFormat = @"d-M-yy HH.mm' uur'";
				Assert.AreEqual (@"8-2-07 15.30 uur", dt.Text, "#712");
				dt.CustomFormat = @"d-M-yy HH:mm' uur'";
				Assert.AreEqual (@"8-2-07 15:30 uur", dt.Text, "#713");
				dt.CustomFormat = @"dd-MM-yy H.mm";
				Assert.AreEqual (@"08-02-07 15.30", dt.Text, "#714");
				dt.CustomFormat = @"dd-MM-yy HH.mm' uur'";
				Assert.AreEqual (@"08-02-07 15.30 uur", dt.Text, "#715");
				dt.CustomFormat = @"dd-MM-yy HH:mm' uur'";
				Assert.AreEqual (@"08-02-07 15:30 uur", dt.Text, "#716");
				dt.CustomFormat = @"dd/MM/yy HH.mm' uur'";
				Assert.AreEqual (@"08/02/07 15.30 uur", dt.Text, "#717");
				dt.CustomFormat = @"dd/MM/yy HH:mm' uur'";
				Assert.AreEqual (@"08/02/07 15:30 uur", dt.Text, "#718");
				dt.CustomFormat = @"dd.MM.yy HH.mm' uur'";
				Assert.AreEqual (@"08.02.07 15.30 uur", dt.Text, "#719");
				dt.CustomFormat = @"dd.MM.yy HH:mm' uur'";
				Assert.AreEqual (@"08.02.07 15:30 uur", dt.Text, "#720");
				dt.CustomFormat = @"yyyy-MM-dd HH.mm' uur'";
				Assert.AreEqual (@"2007-02-08 15.30 uur", dt.Text, "#721");
				dt.CustomFormat = @"yyyy-MM-dd HH:mm' uur'";
				Assert.AreEqual (@"2007-02-08 15:30 uur", dt.Text, "#722");
				dt.CustomFormat = @"d-M-yyyy H:mm:ss";
				Assert.AreEqual (@"8-2-2007 15:30:45", dt.Text, "#723");
				dt.CustomFormat = @"d-M-yyyy HH:mm:ss";
				Assert.AreEqual (@"8-2-2007 15:30:45", dt.Text, "#724");
				dt.CustomFormat = @"d-M-yyyy HH.mm.ss' uur'";
				Assert.AreEqual (@"8-2-2007 15.30.45 uur", dt.Text, "#725");
				dt.CustomFormat = @"d-M-yyyy HH:mm:ss' uur'";
				Assert.AreEqual (@"8-2-2007 15:30:45 uur", dt.Text, "#726");
				dt.CustomFormat = @"d-M-yy HH.mm.ss' uur'";
				Assert.AreEqual (@"8-2-07 15.30.45 uur", dt.Text, "#727");
				dt.CustomFormat = @"d-M-yy HH:mm:ss' uur'";
				Assert.AreEqual (@"8-2-07 15:30:45 uur", dt.Text, "#728");
				dt.CustomFormat = @"dd-MM-yy HH.mm.ss' uur'";
				Assert.AreEqual (@"08-02-07 15.30.45 uur", dt.Text, "#729");
				dt.CustomFormat = @"dd-MM-yy HH:mm:ss' uur'";
				Assert.AreEqual (@"08-02-07 15:30:45 uur", dt.Text, "#730");
				dt.CustomFormat = @"dd/MM/yy HH.mm.ss' uur'";
				Assert.AreEqual (@"08/02/07 15.30.45 uur", dt.Text, "#731");
				dt.CustomFormat = @"dd/MM/yy HH:mm:ss' uur'";
				Assert.AreEqual (@"08/02/07 15:30:45 uur", dt.Text, "#732");
				dt.CustomFormat = @"dd.MM.yy HH.mm.ss' uur'";
				Assert.AreEqual (@"08.02.07 15.30.45 uur", dt.Text, "#733");
				dt.CustomFormat = @"dd.MM.yy HH:mm:ss' uur'";
				Assert.AreEqual (@"08.02.07 15:30:45 uur", dt.Text, "#734");
				dt.CustomFormat = @"yyyy-MM-dd HH.mm.ss' uur'";
				Assert.AreEqual (@"2007-02-08 15.30.45 uur", dt.Text, "#735");
				dt.CustomFormat = @"yyyy-MM-dd HH:mm:ss' uur'";
				Assert.AreEqual (@"2007-02-08 15:30:45 uur", dt.Text, "#736");
				dt.CustomFormat = @"HH.mm' uur'";
				Assert.AreEqual (@"15.30 uur", dt.Text, "#737");
				dt.CustomFormat = @"HH:mm' uur'";
				Assert.AreEqual (@"15:30 uur", dt.Text, "#738");
				dt.CustomFormat = @"HH.mm.ss' uur'";
				Assert.AreEqual (@"15.30.45 uur", dt.Text, "#739");
				dt.CustomFormat = @"HH:mm:ss' uur'";
				Assert.AreEqual (@"15:30:45 uur", dt.Text, "#740");
				dt.CustomFormat = @"d. MMMM yyyy 'kl 'HH.mm";
				Assert.AreEqual (@"8. February 2007 kl 15.30", dt.Text, "#741");
				dt.CustomFormat = @"d. MMMM yyyy HH.mm";
				Assert.AreEqual (@"8. February 2007 15.30", dt.Text, "#742");
				dt.CustomFormat = @"dd. MMMM yyyy 'kl 'HH.mm";
				Assert.AreEqual (@"08. February 2007 kl 15.30", dt.Text, "#743");
				dt.CustomFormat = @"dd. MMMM yyyy HH.mm";
				Assert.AreEqual (@"08. February 2007 15.30", dt.Text, "#744");
				dt.CustomFormat = @"d. MMMM yyyy HH.mm.ss";
				Assert.AreEqual (@"8. February 2007 15.30.45", dt.Text, "#745");
				dt.CustomFormat = @"dd. MMMM yyyy HH.mm.ss";
				Assert.AreEqual (@"08. February 2007 15.30.45", dt.Text, "#746");
				dt.CustomFormat = @"dd.MM.yyyy 'kl 'HH.mm";
				Assert.AreEqual (@"08.02.2007 kl 15.30", dt.Text, "#747");
				dt.CustomFormat = @"dd.MM.yyyy HH.mm";
				Assert.AreEqual (@"08.02.2007 15.30", dt.Text, "#748");
				dt.CustomFormat = @"dd.MM.yy 'kl 'HH.mm";
				Assert.AreEqual (@"08.02.07 kl 15.30", dt.Text, "#749");
				dt.CustomFormat = @"d.M.yy 'kl 'HH.mm";
				Assert.AreEqual (@"8.2.07 kl 15.30", dt.Text, "#750");
				dt.CustomFormat = @"d.M.yy HH.mm";
				Assert.AreEqual (@"8.2.07 15.30", dt.Text, "#751");
				dt.CustomFormat = @"yyyy-MM-dd 'kl 'HH.mm";
				Assert.AreEqual (@"2007-02-08 kl 15.30", dt.Text, "#752");
				dt.CustomFormat = @"dd.MM.yyyy HH.mm.ss";
				Assert.AreEqual (@"08.02.2007 15.30.45", dt.Text, "#753");
				dt.CustomFormat = @"dd.MM.yy HH.mm.ss";
				Assert.AreEqual (@"08.02.07 15.30.45", dt.Text, "#754");
				dt.CustomFormat = @"d.M.yy HH.mm.ss";
				Assert.AreEqual (@"8.2.07 15.30.45", dt.Text, "#755");
				dt.CustomFormat = @"'kl 'HH.mm";
				Assert.AreEqual (@"kl 15.30", dt.Text, "#756");
				dt.CustomFormat = @"dd MMMM yyyy H:mm";
				Assert.AreEqual (@"08 February 2007 15:30", dt.Text, "#757");
				dt.CustomFormat = @"dd MMMM yyyy H:mm:ss";
				Assert.AreEqual (@"08 February 2007 15:30:45", dt.Text, "#758");
				dt.CustomFormat = @"yyyy.MM.dd";
				Assert.AreEqual (@"2007.02.08", dt.Text, "#759");
				dt.CustomFormat = @"dddd, d' de 'MMMM' de 'yyyy";
				Assert.AreEqual (@"Thursday, 8 de February de 2007", dt.Text, "#760");
				dt.CustomFormat = @"d' de 'MMMM' de 'yyyy";
				Assert.AreEqual (@"8 de February de 2007", dt.Text, "#761");
				dt.CustomFormat = @"dddd, d' de 'MMMM' de 'yyyy H:mm";
				Assert.AreEqual (@"Thursday, 8 de February de 2007 15:30", dt.Text, "#762");
				dt.CustomFormat = @"dddd, d' de 'MMMM' de 'yyyy HH:mm";
				Assert.AreEqual (@"Thursday, 8 de February de 2007 15:30", dt.Text, "#763");
				dt.CustomFormat = @"d' de 'MMMM' de 'yyyy H:mm";
				Assert.AreEqual (@"8 de February de 2007 15:30", dt.Text, "#764");
				dt.CustomFormat = @"d' de 'MMMM' de 'yyyy HH:mm";
				Assert.AreEqual (@"8 de February de 2007 15:30", dt.Text, "#765");
				dt.CustomFormat = @"dddd, d' de 'MMMM' de 'yyyy H:mm:ss";
				Assert.AreEqual (@"Thursday, 8 de February de 2007 15:30:45", dt.Text, "#766");
				dt.CustomFormat = @"dddd, d' de 'MMMM' de 'yyyy HH:mm:ss";
				Assert.AreEqual (@"Thursday, 8 de February de 2007 15:30:45", dt.Text, "#767");
				dt.CustomFormat = @"d' de 'MMMM' de 'yyyy H:mm:ss";
				Assert.AreEqual (@"8 de February de 2007 15:30:45", dt.Text, "#768");
				dt.CustomFormat = @"d' de 'MMMM' de 'yyyy HH:mm:ss";
				Assert.AreEqual (@"8 de February de 2007 15:30:45", dt.Text, "#769");
				dt.CustomFormat = @"yyyy.MM.dd H:mm";
				Assert.AreEqual (@"2007.02.08 15:30", dt.Text, "#770");
				dt.CustomFormat = @"yyyy.MM.dd HH:mm";
				Assert.AreEqual (@"2007.02.08 15:30", dt.Text, "#771");
				dt.CustomFormat = @"yyyy.MM.dd H:mm:ss";
				Assert.AreEqual (@"2007.02.08 15:30:45", dt.Text, "#772");
				dt.CustomFormat = @"yyyy.MM.dd HH:mm:ss";
				Assert.AreEqual (@"2007.02.08 15:30:45", dt.Text, "#773");
				dt.CustomFormat = @"dd' de 'MMMM";
				Assert.AreEqual (@"08 de February", dt.Text, "#774");
				dt.CustomFormat = @"d MMMM yyyy 'г.'";
				Assert.AreEqual (@"8 February 2007 г.", dt.Text, "#775");
				dt.CustomFormat = @"d MMMM yyyy 'г.' H:mm";
				Assert.AreEqual (@"8 February 2007 г. 15:30", dt.Text, "#776");
				dt.CustomFormat = @"d MMMM yyyy 'г.' HH:mm";
				Assert.AreEqual (@"8 February 2007 г. 15:30", dt.Text, "#777");
				dt.CustomFormat = @"d MMMM yyyy 'г.' H:mm:ss";
				Assert.AreEqual (@"8 February 2007 г. 15:30:45", dt.Text, "#778");
				dt.CustomFormat = @"d MMMM yyyy 'г.' HH:mm:ss";
				Assert.AreEqual (@"8 February 2007 г. 15:30:45", dt.Text, "#779");
				dt.CustomFormat = @"d. M. yyyy";
				Assert.AreEqual (@"8. 2. 2007", dt.Text, "#780");
				dt.CustomFormat = @"d. M. yy";
				Assert.AreEqual (@"8. 2. 07", dt.Text, "#781");
				dt.CustomFormat = @"dd. MM. yy";
				Assert.AreEqual (@"08. 02. 07", dt.Text, "#782");
				dt.CustomFormat = @"d. M. yyyy H:mm";
				Assert.AreEqual (@"8. 2. 2007 15:30", dt.Text, "#783");
				dt.CustomFormat = @"d. M. yyyy HH:mm";
				Assert.AreEqual (@"8. 2. 2007 15:30", dt.Text, "#784");
				dt.CustomFormat = @"d. M. yy H:mm";
				Assert.AreEqual (@"8. 2. 07 15:30", dt.Text, "#785");
				dt.CustomFormat = @"d. M. yy HH:mm";
				Assert.AreEqual (@"8. 2. 07 15:30", dt.Text, "#786");
				dt.CustomFormat = @"dd. MM. yy H:mm";
				Assert.AreEqual (@"08. 02. 07 15:30", dt.Text, "#787");
				dt.CustomFormat = @"dd. MM. yy HH:mm";
				Assert.AreEqual (@"08. 02. 07 15:30", dt.Text, "#788");
				dt.CustomFormat = @"d. M. yyyy H:mm:ss";
				Assert.AreEqual (@"8. 2. 2007 15:30:45", dt.Text, "#789");
				dt.CustomFormat = @"d. M. yyyy HH:mm:ss";
				Assert.AreEqual (@"8. 2. 2007 15:30:45", dt.Text, "#790");
				dt.CustomFormat = @"d. M. yy H:mm:ss";
				Assert.AreEqual (@"8. 2. 07 15:30:45", dt.Text, "#791");
				dt.CustomFormat = @"d. M. yy HH:mm:ss";
				Assert.AreEqual (@"8. 2. 07 15:30:45", dt.Text, "#792");
				dt.CustomFormat = @"dd. MM. yy H:mm:ss";
				Assert.AreEqual (@"08. 02. 07 15:30:45", dt.Text, "#793");
				dt.CustomFormat = @"dd. MM. yy HH:mm:ss";
				Assert.AreEqual (@"08. 02. 07 15:30:45", dt.Text, "#794");
				dt.CustomFormat = @"dd. MMMM yyyy H.mm";
				Assert.AreEqual (@"08. February 2007 15.30", dt.Text, "#795");
				dt.CustomFormat = @"dd. MMMM yyyy H.mm.ss";
				Assert.AreEqual (@"08. February 2007 15.30.45", dt.Text, "#796");
				dt.CustomFormat = @"d. M. yyyy H.mm";
				Assert.AreEqual (@"8. 2. 2007 15.30", dt.Text, "#797");
				dt.CustomFormat = @"d. M. yy H.mm";
				Assert.AreEqual (@"8. 2. 07 15.30", dt.Text, "#798");
				dt.CustomFormat = @"d. M. yyyy H.mm.ss";
				Assert.AreEqual (@"8. 2. 2007 15.30.45", dt.Text, "#799");
				dt.CustomFormat = @"d. M. yy H.mm.ss";
				Assert.AreEqual (@"8. 2. 07 15.30.45", dt.Text, "#800");
				dt.CustomFormat = @"d.M.yy H.mm.ss";
				Assert.AreEqual (@"8.2.07 15.30.45", dt.Text, "#801");
				dt.CustomFormat = @"dd.MM.yyyy H.mm.ss";
				Assert.AreEqual (@"08.02.2007 15.30.45", dt.Text, "#802");
				dt.CustomFormat = @"dd.MM.yy H.mm.ss";
				Assert.AreEqual (@"08.02.07 15.30.45", dt.Text, "#803");
				dt.CustomFormat = @"yyyy-MM-dd h:mm.tt";
				Assert.AreEqual (@"2007-02-08 3:30.PM", dt.Text, "#804");
				dt.CustomFormat = @"yyyy-MM-dd h:mm:ss.tt";
				Assert.AreEqual (@"2007-02-08 3:30:45.PM", dt.Text, "#805");
				dt.CustomFormat = @"yy-MM-dd h:mm.tt";
				Assert.AreEqual (@"07-02-08 3:30.PM", dt.Text, "#806");
				dt.CustomFormat = @"yy-MM-dd h:mm:ss.tt";
				Assert.AreEqual (@"07-02-08 3:30:45.PM", dt.Text, "#807");
				dt.CustomFormat = @"h:mm.tt";
				Assert.AreEqual (@"3:30.PM", dt.Text, "#808");
				dt.CustomFormat = @"h:mm:ss.tt";
				Assert.AreEqual (@"3:30:45.PM", dt.Text, "#809");
				dt.CustomFormat = @"yyyy-MM";
				Assert.AreEqual (@"2007-02", dt.Text, "#810");
				dt.CustomFormat = @"'den 'd MMMM yyyy";
				Assert.AreEqual (@"den 8 February 2007", dt.Text, "#811");
				dt.CustomFormat = @"'den 'd MMMM yyyy HH:mm";
				Assert.AreEqual (@"den 8 February 2007 15:30", dt.Text, "#812");
				dt.CustomFormat = @"'den 'd MMMM yyyy H:mm";
				Assert.AreEqual (@"den 8 February 2007 15:30", dt.Text, "#813");
				dt.CustomFormat = @"'den 'd MMMM yyyy 'kl 'H:mm";
				Assert.AreEqual (@"den 8 February 2007 kl 15:30", dt.Text, "#814");
				dt.CustomFormat = @"'den 'd MMMM yyyy HH:mm:ss";
				Assert.AreEqual (@"den 8 February 2007 15:30:45", dt.Text, "#815");
				dt.CustomFormat = @"'den 'd MMMM yyyy H:mm:ss";
				Assert.AreEqual (@"den 8 February 2007 15:30:45", dt.Text, "#816");
				dt.CustomFormat = @"yyyy-MM-dd 'kl 'H:mm";
				Assert.AreEqual (@"2007-02-08 kl 15:30", dt.Text, "#817");
				dt.CustomFormat = @"yy-MM-dd 'kl 'H:mm";
				Assert.AreEqual (@"07-02-08 kl 15:30", dt.Text, "#818");
				dt.CustomFormat = @"'den 'd MMMM";
				Assert.AreEqual (@"den 8 February", dt.Text, "#819");
				dt.CustomFormat = @"'kl 'H:mm";
				Assert.AreEqual (@"kl 15:30", dt.Text, "#820");
				dt.CustomFormat = @"d MMM yyyy";
				Assert.AreEqual (@"8 Feb 2007", dt.Text, "#821");
				dt.CustomFormat = @"dd MMM yyyy";
				Assert.AreEqual (@"08 Feb 2007", dt.Text, "#822");
				dt.CustomFormat = @"ddd d MMMM yyyy";
				Assert.AreEqual (@"Thu 8 February 2007", dt.Text, "#823");
				dt.CustomFormat = @"'วัน'dddd'ที่' d MMMM gg yyyy";
				Assert.AreEqual (@"วันThursdayที่ 8 February   2007", dt.Text, "#824");
				dt.CustomFormat = @"ddd d MMMM yyyy H:mm";
				Assert.AreEqual (@"Thu 8 February 2007 15:30", dt.Text, "#825");
				dt.CustomFormat = @"ddd d MMMM yyyy HH:mm";
				Assert.AreEqual (@"Thu 8 February 2007 15:30", dt.Text, "#826");
				dt.CustomFormat = @"ddd d MMMM yyyy h:mm tt";
				Assert.AreEqual (@"Thu 8 February 2007 3:30 PM", dt.Text, "#827");
				dt.CustomFormat = @"ddd d MMMM yyyy hh:mm tt";
				Assert.AreEqual (@"Thu 8 February 2007 03:30 PM", dt.Text, "#828");
				dt.CustomFormat = @"'วัน'dddd'ที่' d MMMM gg yyyy H:mm";
				Assert.AreEqual (@"วันThursdayที่ 8 February   2007 15:30", dt.Text, "#829");
				dt.CustomFormat = @"'วัน'dddd'ที่' d MMMM gg yyyy HH:mm";
				Assert.AreEqual (@"วันThursdayที่ 8 February   2007 15:30", dt.Text, "#830");
				dt.CustomFormat = @"'วัน'dddd'ที่' d MMMM gg yyyy h:mm tt";
				Assert.AreEqual (@"วันThursdayที่ 8 February   2007 3:30 PM", dt.Text, "#831");
				dt.CustomFormat = @"'วัน'dddd'ที่' d MMMM gg yyyy hh:mm tt";
				Assert.AreEqual (@"วันThursdayที่ 8 February   2007 03:30 PM", dt.Text, "#832");
				dt.CustomFormat = @"ddd d MMMM yyyy H:mm:ss";
				Assert.AreEqual (@"Thu 8 February 2007 15:30:45", dt.Text, "#833");
				dt.CustomFormat = @"ddd d MMMM yyyy HH:mm:ss";
				Assert.AreEqual (@"Thu 8 February 2007 15:30:45", dt.Text, "#834");
				dt.CustomFormat = @"ddd d MMMM yyyy h:mm:ss tt";
				Assert.AreEqual (@"Thu 8 February 2007 3:30:45 PM", dt.Text, "#835");
				dt.CustomFormat = @"ddd d MMMM yyyy hh:mm:ss tt";
				Assert.AreEqual (@"Thu 8 February 2007 03:30:45 PM", dt.Text, "#836");
				dt.CustomFormat = @"'วัน'dddd'ที่' d MMMM gg yyyy H:mm:ss";
				Assert.AreEqual (@"วันThursdayที่ 8 February   2007 15:30:45", dt.Text, "#837");
				dt.CustomFormat = @"'วัน'dddd'ที่' d MMMM gg yyyy HH:mm:ss";
				Assert.AreEqual (@"วันThursdayที่ 8 February   2007 15:30:45", dt.Text, "#838");
				dt.CustomFormat = @"'วัน'dddd'ที่' d MMMM gg yyyy h:mm:ss tt";
				Assert.AreEqual (@"วันThursdayที่ 8 February   2007 3:30:45 PM", dt.Text, "#839");
				dt.CustomFormat = @"'วัน'dddd'ที่' d MMMM gg yyyy hh:mm:ss tt";
				Assert.AreEqual (@"วันThursdayที่ 8 February   2007 03:30:45 PM", dt.Text, "#840");
				dt.CustomFormat = @"d MMM yyyy H:mm";
				Assert.AreEqual (@"8 Feb 2007 15:30", dt.Text, "#841");
				dt.CustomFormat = @"d MMM yyyy HH:mm";
				Assert.AreEqual (@"8 Feb 2007 15:30", dt.Text, "#842");
				dt.CustomFormat = @"d MMM yyyy h:mm tt";
				Assert.AreEqual (@"8 Feb 2007 3:30 PM", dt.Text, "#843");
				dt.CustomFormat = @"d MMM yyyy hh:mm tt";
				Assert.AreEqual (@"8 Feb 2007 03:30 PM", dt.Text, "#844");
				dt.CustomFormat = @"dd MMM yyyy H:mm";
				Assert.AreEqual (@"08 Feb 2007 15:30", dt.Text, "#845");
				dt.CustomFormat = @"dd MMM yyyy HH:mm";
				Assert.AreEqual (@"08 Feb 2007 15:30", dt.Text, "#846");
				dt.CustomFormat = @"dd MMM yyyy h:mm tt";
				Assert.AreEqual (@"08 Feb 2007 3:30 PM", dt.Text, "#847");
				dt.CustomFormat = @"dd MMM yyyy hh:mm tt";
				Assert.AreEqual (@"08 Feb 2007 03:30 PM", dt.Text, "#848");
				dt.CustomFormat = @"d MMM yyyy H:mm:ss";
				Assert.AreEqual (@"8 Feb 2007 15:30:45", dt.Text, "#849");
				dt.CustomFormat = @"d MMM yyyy HH:mm:ss";
				Assert.AreEqual (@"8 Feb 2007 15:30:45", dt.Text, "#850");
				dt.CustomFormat = @"d MMM yyyy h:mm:ss tt";
				Assert.AreEqual (@"8 Feb 2007 3:30:45 PM", dt.Text, "#851");
				dt.CustomFormat = @"d MMM yyyy hh:mm:ss tt";
				Assert.AreEqual (@"8 Feb 2007 03:30:45 PM", dt.Text, "#852");
				dt.CustomFormat = @"dd MMM yyyy H:mm:ss";
				Assert.AreEqual (@"08 Feb 2007 15:30:45", dt.Text, "#853");
				dt.CustomFormat = @"dd MMM yyyy HH:mm:ss";
				Assert.AreEqual (@"08 Feb 2007 15:30:45", dt.Text, "#854");
				dt.CustomFormat = @"dd MMM yyyy h:mm:ss tt";
				Assert.AreEqual (@"08 Feb 2007 3:30:45 PM", dt.Text, "#855");
				dt.CustomFormat = @"dd MMM yyyy hh:mm:ss tt";
				Assert.AreEqual (@"08 Feb 2007 03:30:45 PM", dt.Text, "#856");
				dt.CustomFormat = @"dd MMMM yyyy dddd";
				Assert.AreEqual (@"08 February 2007 Thursday", dt.Text, "#857");
				dt.CustomFormat = @"dd MMMM yyyy dddd HH:mm";
				Assert.AreEqual (@"08 February 2007 Thursday 15:30", dt.Text, "#858");
				dt.CustomFormat = @"dd MMMM yyyy dddd HH:mm:ss";
				Assert.AreEqual (@"08 February 2007 Thursday 15:30:45", dt.Text, "#859");
				dt.CustomFormat = @"d MMMM yyyy' р.'";
				Assert.AreEqual (@"8 February 2007 р.", dt.Text, "#860");
				dt.CustomFormat = @"d MMMM yyyy' р.' H:mm";
				Assert.AreEqual (@"8 February 2007 р. 15:30", dt.Text, "#861");
				dt.CustomFormat = @"d MMMM yyyy' р.' HH:mm";
				Assert.AreEqual (@"8 February 2007 р. 15:30", dt.Text, "#862");
				dt.CustomFormat = @"d MMMM yyyy' р.' H:mm:ss";
				Assert.AreEqual (@"8 February 2007 р. 15:30:45", dt.Text, "#863");
				dt.CustomFormat = @"d MMMM yyyy' р.' HH:mm:ss";
				Assert.AreEqual (@"8 February 2007 р. 15:30:45", dt.Text, "#864");
				dt.CustomFormat = @"MMMM yyyy' р.'";
				Assert.AreEqual (@"February 2007 р.", dt.Text, "#865");
				dt.CustomFormat = @"d.MM.yyyy";
				Assert.AreEqual (@"8.02.2007", dt.Text, "#866");
				dt.CustomFormat = @"d. MMMM yyyy'. a.'";
				Assert.AreEqual (@"8. February 2007. a.", dt.Text, "#867");
				dt.CustomFormat = @"dd. MMMM yyyy'. a.'";
				Assert.AreEqual (@"08. February 2007. a.", dt.Text, "#868");
				dt.CustomFormat = @"d. MMMM yyyy'. a.' H:mm";
				Assert.AreEqual (@"8. February 2007. a. 15:30", dt.Text, "#869");
				dt.CustomFormat = @"d. MMMM yyyy'. a.' HH:mm";
				Assert.AreEqual (@"8. February 2007. a. 15:30", dt.Text, "#870");
				dt.CustomFormat = @"dd. MMMM yyyy'. a.' H:mm";
				Assert.AreEqual (@"08. February 2007. a. 15:30", dt.Text, "#871");
				dt.CustomFormat = @"dd. MMMM yyyy'. a.' HH:mm";
				Assert.AreEqual (@"08. February 2007. a. 15:30", dt.Text, "#872");
				dt.CustomFormat = @"d. MMMM yyyy'. a.' H:mm:ss";
				Assert.AreEqual (@"8. February 2007. a. 15:30:45", dt.Text, "#873");
				dt.CustomFormat = @"d. MMMM yyyy'. a.' HH:mm:ss";
				Assert.AreEqual (@"8. February 2007. a. 15:30:45", dt.Text, "#874");
				dt.CustomFormat = @"dd. MMMM yyyy'. a.' H:mm:ss";
				Assert.AreEqual (@"08. February 2007. a. 15:30:45", dt.Text, "#875");
				dt.CustomFormat = @"dd. MMMM yyyy'. a.' HH:mm:ss";
				Assert.AreEqual (@"08. February 2007. a. 15:30:45", dt.Text, "#876");
				dt.CustomFormat = @"d.MM.yyyy H:mm";
				Assert.AreEqual (@"8.02.2007 15:30", dt.Text, "#877");
				dt.CustomFormat = @"d.MM.yyyy HH:mm";
				Assert.AreEqual (@"8.02.2007 15:30", dt.Text, "#878");
				dt.CustomFormat = @"d.MM.yyyy H:mm:ss";
				Assert.AreEqual (@"8.02.2007 15:30:45", dt.Text, "#879");
				dt.CustomFormat = @"d.MM.yyyy HH:mm:ss";
				Assert.AreEqual (@"8.02.2007 15:30:45", dt.Text, "#880");
				dt.CustomFormat = @"MMMM yyyy'. a.'";
				Assert.AreEqual (@"February 2007. a.", dt.Text, "#881");
				dt.CustomFormat = @"yyyy.MM.dd.";
				Assert.AreEqual (@"2007.02.08.", dt.Text, "#882");
				dt.CustomFormat = @"yy.MM.dd.";
				Assert.AreEqual (@"07.02.08.", dt.Text, "#883");
				dt.CustomFormat = @"dddd, yyyy'. gada 'd. MMMM";
				Assert.AreEqual (@"Thursday, 2007. gada 8. February", dt.Text, "#884");
				dt.CustomFormat = @"dddd, yyyy'. gada 'd. MMMM H:mm";
				Assert.AreEqual (@"Thursday, 2007. gada 8. February 15:30", dt.Text, "#885");
				dt.CustomFormat = @"dddd, yyyy'. gada 'd. MMMM HH:mm";
				Assert.AreEqual (@"Thursday, 2007. gada 8. February 15:30", dt.Text, "#886");
				dt.CustomFormat = @"dddd, yyyy'. gada 'd. MMMM H:mm:ss";
				Assert.AreEqual (@"Thursday, 2007. gada 8. February 15:30:45", dt.Text, "#887");
				dt.CustomFormat = @"dddd, yyyy'. gada 'd. MMMM HH:mm:ss";
				Assert.AreEqual (@"Thursday, 2007. gada 8. February 15:30:45", dt.Text, "#888");
				dt.CustomFormat = @"yyyy.MM.dd. H:mm";
				Assert.AreEqual (@"2007.02.08. 15:30", dt.Text, "#889");
				dt.CustomFormat = @"yyyy.MM.dd. HH:mm";
				Assert.AreEqual (@"2007.02.08. 15:30", dt.Text, "#890");
				dt.CustomFormat = @"yy.MM.dd. H:mm";
				Assert.AreEqual (@"07.02.08. 15:30", dt.Text, "#891");
				dt.CustomFormat = @"yy.MM.dd. HH:mm";
				Assert.AreEqual (@"07.02.08. 15:30", dt.Text, "#892");
				dt.CustomFormat = @"yyyy.MM.dd. H:mm:ss";
				Assert.AreEqual (@"2007.02.08. 15:30:45", dt.Text, "#893");
				dt.CustomFormat = @"yyyy.MM.dd. HH:mm:ss";
				Assert.AreEqual (@"2007.02.08. 15:30:45", dt.Text, "#894");
				dt.CustomFormat = @"yy.MM.dd. H:mm:ss";
				Assert.AreEqual (@"07.02.08. 15:30:45", dt.Text, "#895");
				dt.CustomFormat = @"yy.MM.dd. HH:mm:ss";
				Assert.AreEqual (@"07.02.08. 15:30:45", dt.Text, "#896");
				dt.CustomFormat = @"yyyy 'm.' MMMM d 'd.'";
				Assert.AreEqual (@"2007 m. February 8 d.", dt.Text, "#897");
				dt.CustomFormat = @"yyyy 'm.' MMMM d 'd.' HH:mm";
				Assert.AreEqual (@"2007 m. February 8 d. 15:30", dt.Text, "#898");
				dt.CustomFormat = @"yyyy 'm.' MMMM d 'd.' HH:mm:ss";
				Assert.AreEqual (@"2007 m. February 8 d. 15:30:45", dt.Text, "#899");
				dt.CustomFormat = @"MMMM d 'd.'";
				Assert.AreEqual (@"February 8 d.", dt.Text, "#900");
				dt.CustomFormat = @"yyyy 'm.' MMMM";
				Assert.AreEqual (@"2007 m. February", dt.Text, "#901");
				dt.CustomFormat = @"dd MMMM yyyy h:mm tt";
				Assert.AreEqual (@"08 February 2007 3:30 PM", dt.Text, "#902");
				dt.CustomFormat = @"dd MMMM yyyy h:mm:ss tt";
				Assert.AreEqual (@"08 February 2007 3:30:45 PM", dt.Text, "#903");
				dt.CustomFormat = @"dd-MM-yy h:mm tt";
				Assert.AreEqual (@"08-02-07 3:30 PM", dt.Text, "#904");
				dt.CustomFormat = @"dd-MM-yyyy h:mm tt";
				Assert.AreEqual (@"08-02-2007 3:30 PM", dt.Text, "#905");
				dt.CustomFormat = @"dd-MM-yy h:mm:ss tt";
				Assert.AreEqual (@"08-02-07 3:30:45 PM", dt.Text, "#906");
				dt.CustomFormat = @"dd-MM-yyyy h:mm:ss tt";
				Assert.AreEqual (@"08-02-2007 3:30:45 PM", dt.Text, "#907");
				dt.CustomFormat = @"d/MM/yyyy";
				Assert.AreEqual (@"8/02/2007", dt.Text, "#908");
				dt.CustomFormat = @"d/MMM/yyyy";
				Assert.AreEqual (@"8/Feb/2007", dt.Text, "#909");
				dt.CustomFormat = @"dd/MMM/yyyy";
				Assert.AreEqual (@"08/Feb/2007", dt.Text, "#910");
				dt.CustomFormat = @"d MMMM, yyyy";
				Assert.AreEqual (@"8 February, 2007", dt.Text, "#911");
				dt.CustomFormat = @"dddd, dd MMMM yyyy";
				Assert.AreEqual (@"Thursday, 08 February 2007", dt.Text, "#912");
				dt.CustomFormat = @"d-MMM-yyyy";
				Assert.AreEqual (@"8-Feb-2007", dt.Text, "#913");
				dt.CustomFormat = @"dd-MMM-yyyy";
				Assert.AreEqual (@"08-Feb-2007", dt.Text, "#914");
				dt.CustomFormat = @"ddd, d-MMMM-yyyy";
				Assert.AreEqual (@"Thu, 8-February-2007", dt.Text, "#915");
				dt.CustomFormat = @"ddd, dd-MMMM-yyyy";
				Assert.AreEqual (@"Thu, 08-February-2007", dt.Text, "#916");
				dt.CustomFormat = @"d MMMM, yyyy H:mm";
				Assert.AreEqual (@"8 February, 2007 15:30", dt.Text, "#917");
				dt.CustomFormat = @"d MMMM, yyyy HH:mm";
				Assert.AreEqual (@"8 February, 2007 15:30", dt.Text, "#918");
				dt.CustomFormat = @"dddd, dd MMMM yyyy H:mm";
				Assert.AreEqual (@"Thursday, 08 February 2007 15:30", dt.Text, "#919");
				dt.CustomFormat = @"dddd, dd MMMM yyyy HH:mm";
				Assert.AreEqual (@"Thursday, 08 February 2007 15:30", dt.Text, "#920");
				dt.CustomFormat = @"d-MMM-yyyy H:mm";
				Assert.AreEqual (@"8-Feb-2007 15:30", dt.Text, "#921");
				dt.CustomFormat = @"d-MMM-yyyy HH:mm";
				Assert.AreEqual (@"8-Feb-2007 15:30", dt.Text, "#922");
				dt.CustomFormat = @"dd-MMM-yyyy H:mm";
				Assert.AreEqual (@"08-Feb-2007 15:30", dt.Text, "#923");
				dt.CustomFormat = @"dd-MMM-yyyy HH:mm";
				Assert.AreEqual (@"08-Feb-2007 15:30", dt.Text, "#924");
				dt.CustomFormat = @"ddd, d-MMMM-yyyy H:mm";
				Assert.AreEqual (@"Thu, 8-February-2007 15:30", dt.Text, "#925");
				dt.CustomFormat = @"ddd, d-MMMM-yyyy HH:mm";
				Assert.AreEqual (@"Thu, 8-February-2007 15:30", dt.Text, "#926");
				dt.CustomFormat = @"ddd, dd-MMMM-yyyy H:mm";
				Assert.AreEqual (@"Thu, 08-February-2007 15:30", dt.Text, "#927");
				dt.CustomFormat = @"ddd, dd-MMMM-yyyy HH:mm";
				Assert.AreEqual (@"Thu, 08-February-2007 15:30", dt.Text, "#928");
				dt.CustomFormat = @"d MMMM, yyyy H:mm:ss";
				Assert.AreEqual (@"8 February, 2007 15:30:45", dt.Text, "#929");
				dt.CustomFormat = @"d MMMM, yyyy HH:mm:ss";
				Assert.AreEqual (@"8 February, 2007 15:30:45", dt.Text, "#930");
				dt.CustomFormat = @"dddd, dd MMMM yyyy H:mm:ss";
				Assert.AreEqual (@"Thursday, 08 February 2007 15:30:45", dt.Text, "#931");
				dt.CustomFormat = @"dddd, dd MMMM yyyy HH:mm:ss";
				Assert.AreEqual (@"Thursday, 08 February 2007 15:30:45", dt.Text, "#932");
				dt.CustomFormat = @"d-MMM-yyyy H:mm:ss";
				Assert.AreEqual (@"8-Feb-2007 15:30:45", dt.Text, "#933");
				dt.CustomFormat = @"d-MMM-yyyy HH:mm:ss";
				Assert.AreEqual (@"8-Feb-2007 15:30:45", dt.Text, "#934");
				dt.CustomFormat = @"dd-MMM-yyyy H:mm:ss";
				Assert.AreEqual (@"08-Feb-2007 15:30:45", dt.Text, "#935");
				dt.CustomFormat = @"dd-MMM-yyyy HH:mm:ss";
				Assert.AreEqual (@"08-Feb-2007 15:30:45", dt.Text, "#936");
				dt.CustomFormat = @"ddd, d-MMMM-yyyy H:mm:ss";
				Assert.AreEqual (@"Thu, 8-February-2007 15:30:45", dt.Text, "#937");
				dt.CustomFormat = @"ddd, d-MMMM-yyyy HH:mm:ss";
				Assert.AreEqual (@"Thu, 8-February-2007 15:30:45", dt.Text, "#938");
				dt.CustomFormat = @"ddd, dd-MMMM-yyyy H:mm:ss";
				Assert.AreEqual (@"Thu, 08-February-2007 15:30:45", dt.Text, "#939");
				dt.CustomFormat = @"ddd, dd-MMMM-yyyy HH:mm:ss";
				Assert.AreEqual (@"Thu, 08-February-2007 15:30:45", dt.Text, "#940");
				dt.CustomFormat = @"d/MM/yyyy H:mm";
				Assert.AreEqual (@"8/02/2007 15:30", dt.Text, "#941");
				dt.CustomFormat = @"d/MM/yyyy HH:mm";
				Assert.AreEqual (@"8/02/2007 15:30", dt.Text, "#942");
				dt.CustomFormat = @"d/MMM/yyyy H:mm";
				Assert.AreEqual (@"8/Feb/2007 15:30", dt.Text, "#943");
				dt.CustomFormat = @"d/MMM/yyyy HH:mm";
				Assert.AreEqual (@"8/Feb/2007 15:30", dt.Text, "#944");
				dt.CustomFormat = @"dd/MMM/yyyy H:mm";
				Assert.AreEqual (@"08/Feb/2007 15:30", dt.Text, "#945");
				dt.CustomFormat = @"dd/MMM/yyyy HH:mm";
				Assert.AreEqual (@"08/Feb/2007 15:30", dt.Text, "#946");
				dt.CustomFormat = @"d/MM/yyyy H:mm:ss";
				Assert.AreEqual (@"8/02/2007 15:30:45", dt.Text, "#947");
				dt.CustomFormat = @"d/MM/yyyy HH:mm:ss";
				Assert.AreEqual (@"8/02/2007 15:30:45", dt.Text, "#948");
				dt.CustomFormat = @"d/MMM/yyyy H:mm:ss";
				Assert.AreEqual (@"8/Feb/2007 15:30:45", dt.Text, "#949");
				dt.CustomFormat = @"d/MMM/yyyy HH:mm:ss";
				Assert.AreEqual (@"8/Feb/2007 15:30:45", dt.Text, "#950");
				dt.CustomFormat = @"dd/MMM/yyyy H:mm:ss";
				Assert.AreEqual (@"08/Feb/2007 15:30:45", dt.Text, "#951");
				dt.CustomFormat = @"dd/MMM/yyyy HH:mm:ss";
				Assert.AreEqual (@"08/Feb/2007 15:30:45", dt.Text, "#952");
				dt.CustomFormat = @"dddd, yyyy.'eko' MMMM'k 'd";
				Assert.AreEqual (@"Thursday, 2007.eko Februaryk 8", dt.Text, "#953");
				dt.CustomFormat = @"dddd, yyyy.'eko' MMMM'k 'd HH:mm";
				Assert.AreEqual (@"Thursday, 2007.eko Februaryk 8 15:30", dt.Text, "#954");
				dt.CustomFormat = @"dddd, yyyy.'eko' MMMM'k 'd HH:mm:ss";
				Assert.AreEqual (@"Thursday, 2007.eko Februaryk 8 15:30:45", dt.Text, "#955");
				dt.CustomFormat = @"yyyy.'eko' MMMM";
				Assert.AreEqual (@"2007.eko February", dt.Text, "#956");
				dt.CustomFormat = @"yyyy/MM/dd hh:mm tt";
				Assert.AreEqual (@"2007/02/08 03:30 PM", dt.Text, "#957");
				dt.CustomFormat = @"yyyy/MM/dd h:mm tt";
				Assert.AreEqual (@"2007/02/08 3:30 PM", dt.Text, "#958");
				dt.CustomFormat = @"yyyy/MM/dd hh:mm:ss tt";
				Assert.AreEqual (@"2007/02/08 03:30:45 PM", dt.Text, "#959");
				dt.CustomFormat = @"yyyy/MM/dd h:mm:ss tt";
				Assert.AreEqual (@"2007/02/08 3:30:45 PM", dt.Text, "#960");
				dt.CustomFormat = @"yyyy 'წლის' dd MM, dddd";
				Assert.AreEqual (@"2007 წლის 08 02, Thursday", dt.Text, "#961");
				dt.CustomFormat = @"yyyy 'წლის' dd MM, dddd H:mm";
				Assert.AreEqual (@"2007 წლის 08 02, Thursday 15:30", dt.Text, "#962");
				dt.CustomFormat = @"yyyy 'წლის' dd MM, dddd HH:mm";
				Assert.AreEqual (@"2007 წლის 08 02, Thursday 15:30", dt.Text, "#963");
				dt.CustomFormat = @"yyyy 'წლის' dd MM, dddd H:mm:ss";
				Assert.AreEqual (@"2007 წლის 08 02, Thursday 15:30:45", dt.Text, "#964");
				dt.CustomFormat = @"yyyy 'წლის' dd MM, dddd HH:mm:ss";
				Assert.AreEqual (@"2007 წლის 08 02, Thursday 15:30:45", dt.Text, "#965");
				dt.CustomFormat = @"dd MM";
				Assert.AreEqual (@"08 02", dt.Text, "#966");
				dt.CustomFormat = @"dd-MM-yyyy HH.mm";
				Assert.AreEqual (@"08-02-2007 15.30", dt.Text, "#967");
				dt.CustomFormat = @"dd-MM-yyyy HH.mm.ss";
				Assert.AreEqual (@"08-02-2007 15.30.45", dt.Text, "#968");
				dt.CustomFormat = @"dd-MM-yy HH.mm.ss";
				Assert.AreEqual (@"08-02-07 15.30.45", dt.Text, "#969");
				dt.CustomFormat = @"dd MMMM yyyy tt hh:mm";
				Assert.AreEqual (@"08 February 2007 PM 03:30", dt.Text, "#970");
				dt.CustomFormat = @"dd MMMM yyyy tt h:mm";
				Assert.AreEqual (@"08 February 2007 PM 3:30", dt.Text, "#971");
				dt.CustomFormat = @"d MMMM yyyy tt hh:mm";
				Assert.AreEqual (@"8 February 2007 PM 03:30", dt.Text, "#972");
				dt.CustomFormat = @"d MMMM yyyy tt h:mm";
				Assert.AreEqual (@"8 February 2007 PM 3:30", dt.Text, "#973");
				dt.CustomFormat = @"dd MMMM yyyy tt hh:mm:ss";
				Assert.AreEqual (@"08 February 2007 PM 03:30:45", dt.Text, "#974");
				dt.CustomFormat = @"dd MMMM yyyy tt h:mm:ss";
				Assert.AreEqual (@"08 February 2007 PM 3:30:45", dt.Text, "#975");
				dt.CustomFormat = @"d MMMM yyyy tt hh:mm:ss";
				Assert.AreEqual (@"8 February 2007 PM 03:30:45", dt.Text, "#976");
				dt.CustomFormat = @"d MMMM yyyy tt h:mm:ss";
				Assert.AreEqual (@"8 February 2007 PM 3:30:45", dt.Text, "#977");
				dt.CustomFormat = @"dd-MM-yyyy tt hh:mm";
				Assert.AreEqual (@"08-02-2007 PM 03:30", dt.Text, "#978");
				dt.CustomFormat = @"dd-MM-yyyy tt h:mm";
				Assert.AreEqual (@"08-02-2007 PM 3:30", dt.Text, "#979");
				dt.CustomFormat = @"dd-MM-yy tt hh:mm";
				Assert.AreEqual (@"08-02-07 PM 03:30", dt.Text, "#980");
				dt.CustomFormat = @"dd-MM-yy tt h:mm";
				Assert.AreEqual (@"08-02-07 PM 3:30", dt.Text, "#981");
				dt.CustomFormat = @"d-M-yy tt hh:mm";
				Assert.AreEqual (@"8-2-07 PM 03:30", dt.Text, "#982");
				dt.CustomFormat = @"d-M-yy tt h:mm";
				Assert.AreEqual (@"8-2-07 PM 3:30", dt.Text, "#983");
				dt.CustomFormat = @"d.M.yy tt hh:mm";
				Assert.AreEqual (@"8.2.07 PM 03:30", dt.Text, "#984");
				dt.CustomFormat = @"d.M.yy tt h:mm";
				Assert.AreEqual (@"8.2.07 PM 3:30", dt.Text, "#985");
				dt.CustomFormat = @"dd-MM-yyyy tt hh:mm:ss";
				Assert.AreEqual (@"08-02-2007 PM 03:30:45", dt.Text, "#986");
				dt.CustomFormat = @"dd-MM-yyyy tt h:mm:ss";
				Assert.AreEqual (@"08-02-2007 PM 3:30:45", dt.Text, "#987");
				dt.CustomFormat = @"dd-MM-yy tt hh:mm:ss";
				Assert.AreEqual (@"08-02-07 PM 03:30:45", dt.Text, "#988");
				dt.CustomFormat = @"dd-MM-yy tt h:mm:ss";
				Assert.AreEqual (@"08-02-07 PM 3:30:45", dt.Text, "#989");
				dt.CustomFormat = @"d-M-yy tt hh:mm:ss";
				Assert.AreEqual (@"8-2-07 PM 03:30:45", dt.Text, "#990");
				dt.CustomFormat = @"d-M-yy tt h:mm:ss";
				Assert.AreEqual (@"8-2-07 PM 3:30:45", dt.Text, "#991");
				dt.CustomFormat = @"d.M.yy tt hh:mm:ss";
				Assert.AreEqual (@"8.2.07 PM 03:30:45", dt.Text, "#992");
				dt.CustomFormat = @"d.M.yy tt h:mm:ss";
				Assert.AreEqual (@"8.2.07 PM 3:30:45", dt.Text, "#993");
				dt.CustomFormat = @"d MMMM yyyy 'ж.'";
				Assert.AreEqual (@"8 February 2007 ж.", dt.Text, "#994");
				dt.CustomFormat = @"dd MMMM yyyy 'ж.'";
				Assert.AreEqual (@"08 February 2007 ж.", dt.Text, "#995");
				dt.CustomFormat = @"d MMMM yyyy 'ж.' H:mm";
				Assert.AreEqual (@"8 February 2007 ж. 15:30", dt.Text, "#996");
				dt.CustomFormat = @"d MMMM yyyy 'ж.' HH:mm";
				Assert.AreEqual (@"8 February 2007 ж. 15:30", dt.Text, "#997");
				dt.CustomFormat = @"dd MMMM yyyy 'ж.' H:mm";
				Assert.AreEqual (@"08 February 2007 ж. 15:30", dt.Text, "#998");
				dt.CustomFormat = @"dd MMMM yyyy 'ж.' HH:mm";
				Assert.AreEqual (@"08 February 2007 ж. 15:30", dt.Text, "#999");
				dt.CustomFormat = @"d MMMM yyyy 'ж.' H:mm:ss";
				Assert.AreEqual (@"8 February 2007 ж. 15:30:45", dt.Text, "#1000");
				dt.CustomFormat = @"d MMMM yyyy 'ж.' HH:mm:ss";
				Assert.AreEqual (@"8 February 2007 ж. 15:30:45", dt.Text, "#1001");
				dt.CustomFormat = @"dd MMMM yyyy 'ж.' H:mm:ss";
				Assert.AreEqual (@"08 February 2007 ж. 15:30:45", dt.Text, "#1002");
				dt.CustomFormat = @"dd MMMM yyyy 'ж.' HH:mm:ss";
				Assert.AreEqual (@"08 February 2007 ж. 15:30:45", dt.Text, "#1003");
				dt.CustomFormat = @"d'-'MMMM yyyy'-ж.'";
				Assert.AreEqual (@"8-February 2007-ж.", dt.Text, "#1004");
				dt.CustomFormat = @"d'-'MMMM yyyy'-ж.' H:mm";
				Assert.AreEqual (@"8-February 2007-ж. 15:30", dt.Text, "#1005");
				dt.CustomFormat = @"d'-'MMMM yyyy'-ж.' H:mm:ss";
				Assert.AreEqual (@"8-February 2007-ж. 15:30:45", dt.Text, "#1006");
				dt.CustomFormat = @"MMMM yyyy'-ж.'";
				Assert.AreEqual (@"February 2007-ж.", dt.Text, "#1007");
				dt.CustomFormat = @"dd/MM yyyy";
				Assert.AreEqual (@"08/02 2007", dt.Text, "#1008");
				dt.CustomFormat = @"yyyy 'yil' d-MMMM";
				Assert.AreEqual (@"2007 yil 8-February", dt.Text, "#1009");
				dt.CustomFormat = @"yyyy 'yil' d-MMMM HH:mm";
				Assert.AreEqual (@"2007 yil 8-February 15:30", dt.Text, "#1010");
				dt.CustomFormat = @"yyyy 'yil' d-MMMM H:mm";
				Assert.AreEqual (@"2007 yil 8-February 15:30", dt.Text, "#1011");
				dt.CustomFormat = @"yyyy 'yil' d-MMMM HH:mm:ss";
				Assert.AreEqual (@"2007 yil 8-February 15:30:45", dt.Text, "#1012");
				dt.CustomFormat = @"yyyy 'yil' d-MMMM H:mm:ss";
				Assert.AreEqual (@"2007 yil 8-February 15:30:45", dt.Text, "#1013");
				dt.CustomFormat = @"dd/MM yyyy HH:mm";
				Assert.AreEqual (@"08/02 2007 15:30", dt.Text, "#1014");
				dt.CustomFormat = @"dd/MM yyyy H:mm";
				Assert.AreEqual (@"08/02 2007 15:30", dt.Text, "#1015");
				dt.CustomFormat = @"dd/MM yyyy HH:mm:ss";
				Assert.AreEqual (@"08/02 2007 15:30:45", dt.Text, "#1016");
				dt.CustomFormat = @"dd/MM yyyy H:mm:ss";
				Assert.AreEqual (@"08/02 2007 15:30:45", dt.Text, "#1017");
				dt.CustomFormat = @"d-MMMM";
				Assert.AreEqual (@"8-February", dt.Text, "#1018");
				dt.CustomFormat = @"dd MMMM yyyy dddd tt hh:mm";
				Assert.AreEqual (@"08 February 2007 Thursday PM 03:30", dt.Text, "#1019");
				dt.CustomFormat = @"dd MMMM yyyy dddd tt h:mm";
				Assert.AreEqual (@"08 February 2007 Thursday PM 3:30", dt.Text, "#1020");
				dt.CustomFormat = @"dd MMMM yyyy dddd H:mm";
				Assert.AreEqual (@"08 February 2007 Thursday 15:30", dt.Text, "#1021");
				dt.CustomFormat = @"dd MMMM yyyy dddd tt hh:mm:ss";
				Assert.AreEqual (@"08 February 2007 Thursday PM 03:30:45", dt.Text, "#1022");
				dt.CustomFormat = @"dd MMMM yyyy dddd tt h:mm:ss";
				Assert.AreEqual (@"08 February 2007 Thursday PM 3:30:45", dt.Text, "#1023");
				dt.CustomFormat = @"dd MMMM yyyy dddd H:mm:ss";
				Assert.AreEqual (@"08 February 2007 Thursday 15:30:45", dt.Text, "#1024");
				dt.CustomFormat = @"yy.MM.dd";
				Assert.AreEqual (@"07.02.08", dt.Text, "#1025");
				dt.CustomFormat = @"yyyy 'оны' MMMM d";
				Assert.AreEqual (@"2007 оны February 8", dt.Text, "#1026");
				dt.CustomFormat = @"yyyy 'оны' MMMM d H:mm";
				Assert.AreEqual (@"2007 оны February 8 15:30", dt.Text, "#1027");
				dt.CustomFormat = @"yyyy 'оны' MMMM d H:mm:ss";
				Assert.AreEqual (@"2007 оны February 8 15:30:45", dt.Text, "#1028");
				dt.CustomFormat = @"yy.MM.dd H:mm";
				Assert.AreEqual (@"07.02.08 15:30", dt.Text, "#1029");
				dt.CustomFormat = @"yy.MM.dd H:mm:ss";
				Assert.AreEqual (@"07.02.08 15:30:45", dt.Text, "#1030");
				dt.CustomFormat = @"yyyy 'он' MMMM";
				Assert.AreEqual (@"2007 он February", dt.Text, "#1031");
				dt.CustomFormat = @"dddd, dd' de 'MMMM' de 'yyyy";
				Assert.AreEqual (@"Thursday, 08 de February de 2007", dt.Text, "#1032");
				dt.CustomFormat = @"dddd d' de 'MMMM' de 'yyyy";
				Assert.AreEqual (@"Thursday 8 de February de 2007", dt.Text, "#1033");
				dt.CustomFormat = @"dddd, dd' de 'MMMM' de 'yyyy H:mm";
				Assert.AreEqual (@"Thursday, 08 de February de 2007 15:30", dt.Text, "#1034");
				dt.CustomFormat = @"dddd, dd' de 'MMMM' de 'yyyy HH:mm";
				Assert.AreEqual (@"Thursday, 08 de February de 2007 15:30", dt.Text, "#1035");
				dt.CustomFormat = @"dddd, dd' de 'MMMM' de 'yyyy hh:mm tt";
				Assert.AreEqual (@"Thursday, 08 de February de 2007 03:30 PM", dt.Text, "#1036");
				dt.CustomFormat = @"dddd, dd' de 'MMMM' de 'yyyy HH'H'mm'\''";
				Assert.AreEqual (@"Thursday, 08 de February de 2007 15H30\'", dt.Text, "#1037");
				dt.CustomFormat = @"dddd d' de 'MMMM' de 'yyyy H:mm";
				Assert.AreEqual (@"Thursday 8 de February de 2007 15:30", dt.Text, "#1038");
				dt.CustomFormat = @"dddd d' de 'MMMM' de 'yyyy HH:mm";
				Assert.AreEqual (@"Thursday 8 de February de 2007 15:30", dt.Text, "#1039");
				dt.CustomFormat = @"dddd d' de 'MMMM' de 'yyyy hh:mm tt";
				Assert.AreEqual (@"Thursday 8 de February de 2007 03:30 PM", dt.Text, "#1040");
				dt.CustomFormat = @"dddd d' de 'MMMM' de 'yyyy HH'H'mm'\''";
				Assert.AreEqual (@"Thursday 8 de February de 2007 15H30\'", dt.Text, "#1041");
				dt.CustomFormat = @"d' de 'MMMM' de 'yyyy hh:mm tt";
				Assert.AreEqual (@"8 de February de 2007 03:30 PM", dt.Text, "#1042");
				dt.CustomFormat = @"d' de 'MMMM' de 'yyyy HH'H'mm'\''";
				Assert.AreEqual (@"8 de February de 2007 15H30\'", dt.Text, "#1043");
				dt.CustomFormat = @"dddd, dd' de 'MMMM' de 'yyyy H:mm:ss";
				Assert.AreEqual (@"Thursday, 08 de February de 2007 15:30:45", dt.Text, "#1044");
				dt.CustomFormat = @"dddd, dd' de 'MMMM' de 'yyyy HH:mm:ss";
				Assert.AreEqual (@"Thursday, 08 de February de 2007 15:30:45", dt.Text, "#1045");
				dt.CustomFormat = @"dddd, dd' de 'MMMM' de 'yyyy hh:mm:ss tt";
				Assert.AreEqual (@"Thursday, 08 de February de 2007 03:30:45 PM", dt.Text, "#1046");
				dt.CustomFormat = @"dddd, dd' de 'MMMM' de 'yyyy HH'H'mm'\''ss'\'\''";
				Assert.AreEqual (@"Thursday, 08 de February de 2007 15H30\'ss\\'", dt.Text, "#1047");
				dt.CustomFormat = @"dddd d' de 'MMMM' de 'yyyy H:mm:ss";
				Assert.AreEqual (@"Thursday 8 de February de 2007 15:30:45", dt.Text, "#1048");
				dt.CustomFormat = @"dddd d' de 'MMMM' de 'yyyy HH:mm:ss";
				Assert.AreEqual (@"Thursday 8 de February de 2007 15:30:45", dt.Text, "#1049");
				dt.CustomFormat = @"dddd d' de 'MMMM' de 'yyyy hh:mm:ss tt";
				Assert.AreEqual (@"Thursday 8 de February de 2007 03:30:45 PM", dt.Text, "#1050");
				dt.CustomFormat = @"dddd d' de 'MMMM' de 'yyyy HH'H'mm'\''ss'\'\''";
				Assert.AreEqual (@"Thursday 8 de February de 2007 15H30\'ss\\'", dt.Text, "#1051");
				dt.CustomFormat = @"d' de 'MMMM' de 'yyyy hh:mm:ss tt";
				Assert.AreEqual (@"8 de February de 2007 03:30:45 PM", dt.Text, "#1052");
				dt.CustomFormat = @"d' de 'MMMM' de 'yyyy HH'H'mm'\''ss'\'\''";
				Assert.AreEqual (@"8 de February de 2007 15H30\'ss\\'", dt.Text, "#1053");
				dt.CustomFormat = @"d/MM/yy hh:mm tt";
				Assert.AreEqual (@"8/02/07 03:30 PM", dt.Text, "#1054");
				dt.CustomFormat = @"dd.MM.yy hh:mm tt";
				Assert.AreEqual (@"08.02.07 03:30 PM", dt.Text, "#1055");
				dt.CustomFormat = @"d/MM/yy hh:mm:ss tt";
				Assert.AreEqual (@"8/02/07 03:30:45 PM", dt.Text, "#1056");
				dt.CustomFormat = @"dd.MM.yy hh:mm:ss tt";
				Assert.AreEqual (@"08.02.07 03:30:45 PM", dt.Text, "#1057");
				dt.CustomFormat = @"yyyy.M.d";
				Assert.AreEqual (@"2007.2.8", dt.Text, "#1058");
				dt.CustomFormat = @"yy.M.d";
				Assert.AreEqual (@"07.2.8", dt.Text, "#1059");
				dt.CustomFormat = @"dddd, yyyy-MM-dd";
				Assert.AreEqual (@"Thursday, 2007-02-08", dt.Text, "#1060");
				dt.CustomFormat = @"dddd, yyyy'年'M'月'd'日'";
				Assert.AreEqual (@"Thursday, 2007年2月8日", dt.Text, "#1061");
				dt.CustomFormat = @"dddd, yyyy-MM-dd H:mm";
				Assert.AreEqual (@"Thursday, 2007-02-08 15:30", dt.Text, "#1062");
				dt.CustomFormat = @"dddd, yyyy-MM-dd HH:mm";
				Assert.AreEqual (@"Thursday, 2007-02-08 15:30", dt.Text, "#1063");
				dt.CustomFormat = @"dddd, yyyy-MM-dd tt h:mm";
				Assert.AreEqual (@"Thursday, 2007-02-08 PM 3:30", dt.Text, "#1064");
				dt.CustomFormat = @"dddd, yyyy-MM-dd tt hh:mm";
				Assert.AreEqual (@"Thursday, 2007-02-08 PM 03:30", dt.Text, "#1065");
				dt.CustomFormat = @"dddd, yyyy'年'M'月'd'日' H:mm";
				Assert.AreEqual (@"Thursday, 2007年2月8日 15:30", dt.Text, "#1066");
				dt.CustomFormat = @"dddd, yyyy'年'M'月'd'日' HH:mm";
				Assert.AreEqual (@"Thursday, 2007年2月8日 15:30", dt.Text, "#1067");
				dt.CustomFormat = @"dddd, yyyy'年'M'月'd'日' tt h:mm";
				Assert.AreEqual (@"Thursday, 2007年2月8日 PM 3:30", dt.Text, "#1068");
				dt.CustomFormat = @"dddd, yyyy'年'M'月'd'日' tt hh:mm";
				Assert.AreEqual (@"Thursday, 2007年2月8日 PM 03:30", dt.Text, "#1069");
				dt.CustomFormat = @"dddd, yyyy-MM-dd H:mm:ss";
				Assert.AreEqual (@"Thursday, 2007-02-08 15:30:45", dt.Text, "#1070");
				dt.CustomFormat = @"dddd, yyyy-MM-dd HH:mm:ss";
				Assert.AreEqual (@"Thursday, 2007-02-08 15:30:45", dt.Text, "#1071");
				dt.CustomFormat = @"dddd, yyyy-MM-dd tt h:mm:ss";
				Assert.AreEqual (@"Thursday, 2007-02-08 PM 3:30:45", dt.Text, "#1072");
				dt.CustomFormat = @"dddd, yyyy-MM-dd tt hh:mm:ss";
				Assert.AreEqual (@"Thursday, 2007-02-08 PM 03:30:45", dt.Text, "#1073");
				dt.CustomFormat = @"dddd, yyyy'年'M'月'd'日' H:mm:ss";
				Assert.AreEqual (@"Thursday, 2007年2月8日 15:30:45", dt.Text, "#1074");
				dt.CustomFormat = @"dddd, yyyy'年'M'月'd'日' HH:mm:ss";
				Assert.AreEqual (@"Thursday, 2007年2月8日 15:30:45", dt.Text, "#1075");
				dt.CustomFormat = @"dddd, yyyy'年'M'月'd'日' tt h:mm:ss";
				Assert.AreEqual (@"Thursday, 2007年2月8日 PM 3:30:45", dt.Text, "#1076");
				dt.CustomFormat = @"dddd, yyyy'年'M'月'd'日' tt hh:mm:ss";
				Assert.AreEqual (@"Thursday, 2007年2月8日 PM 03:30:45", dt.Text, "#1077");
				dt.CustomFormat = @"yyyy.M.d H:mm";
				Assert.AreEqual (@"2007.2.8 15:30", dt.Text, "#1078");
				dt.CustomFormat = @"yyyy.M.d HH:mm";
				Assert.AreEqual (@"2007.2.8 15:30", dt.Text, "#1079");
				dt.CustomFormat = @"yyyy.M.d tt h:mm";
				Assert.AreEqual (@"2007.2.8 PM 3:30", dt.Text, "#1080");
				dt.CustomFormat = @"yyyy.M.d tt hh:mm";
				Assert.AreEqual (@"2007.2.8 PM 03:30", dt.Text, "#1081");
				dt.CustomFormat = @"yyyy.MM.dd tt h:mm";
				Assert.AreEqual (@"2007.02.08 PM 3:30", dt.Text, "#1082");
				dt.CustomFormat = @"yyyy.MM.dd tt hh:mm";
				Assert.AreEqual (@"2007.02.08 PM 03:30", dt.Text, "#1083");
				dt.CustomFormat = @"yy.M.d H:mm";
				Assert.AreEqual (@"07.2.8 15:30", dt.Text, "#1084");
				dt.CustomFormat = @"yy.M.d HH:mm";
				Assert.AreEqual (@"07.2.8 15:30", dt.Text, "#1085");
				dt.CustomFormat = @"yy.M.d tt h:mm";
				Assert.AreEqual (@"07.2.8 PM 3:30", dt.Text, "#1086");
				dt.CustomFormat = @"yy.M.d tt hh:mm";
				Assert.AreEqual (@"07.2.8 PM 03:30", dt.Text, "#1087");
				dt.CustomFormat = @"yyyy.M.d H:mm:ss";
				Assert.AreEqual (@"2007.2.8 15:30:45", dt.Text, "#1088");
				dt.CustomFormat = @"yyyy.M.d HH:mm:ss";
				Assert.AreEqual (@"2007.2.8 15:30:45", dt.Text, "#1089");
				dt.CustomFormat = @"yyyy.M.d tt h:mm:ss";
				Assert.AreEqual (@"2007.2.8 PM 3:30:45", dt.Text, "#1090");
				dt.CustomFormat = @"yyyy.M.d tt hh:mm:ss";
				Assert.AreEqual (@"2007.2.8 PM 03:30:45", dt.Text, "#1091");
				dt.CustomFormat = @"yyyy.MM.dd tt h:mm:ss";
				Assert.AreEqual (@"2007.02.08 PM 3:30:45", dt.Text, "#1092");
				dt.CustomFormat = @"yyyy.MM.dd tt hh:mm:ss";
				Assert.AreEqual (@"2007.02.08 PM 03:30:45", dt.Text, "#1093");
				dt.CustomFormat = @"yy.M.d H:mm:ss";
				Assert.AreEqual (@"07.2.8 15:30:45", dt.Text, "#1094");
				dt.CustomFormat = @"yy.M.d HH:mm:ss";
				Assert.AreEqual (@"07.2.8 15:30:45", dt.Text, "#1095");
				dt.CustomFormat = @"yy.M.d tt h:mm:ss";
				Assert.AreEqual (@"07.2.8 PM 3:30:45", dt.Text, "#1096");
				dt.CustomFormat = @"yy.M.d tt hh:mm:ss";
				Assert.AreEqual (@"07.2.8 PM 03:30:45", dt.Text, "#1097");
				dt.CustomFormat = @"yyyy.M";
				Assert.AreEqual (@"2007.2", dt.Text, "#1098");
				dt.CustomFormat = @"dd. M. yy";
				Assert.AreEqual (@"08. 2. 07", dt.Text, "#1099");
				dt.CustomFormat = @"d. MMM yy";
				Assert.AreEqual (@"8. Feb 07", dt.Text, "#1100");
				dt.CustomFormat = @"dddd, d. MMMM yyyy H.mm' h'";
				Assert.AreEqual (@"Thursday, 8. February 2007 15.30 h", dt.Text, "#1101");
				dt.CustomFormat = @"dddd, d. MMMM yyyy HH.mm' h'";
				Assert.AreEqual (@"Thursday, 8. February 2007 15.30 h", dt.Text, "#1102");
				dt.CustomFormat = @"dddd, d. MMMM yyyy H.mm' Uhr'";
				Assert.AreEqual (@"Thursday, 8. February 2007 15.30 Uhr", dt.Text, "#1103");
				dt.CustomFormat = @"d. MMMM yyyy H.mm' h'";
				Assert.AreEqual (@"8. February 2007 15.30 h", dt.Text, "#1104");
				dt.CustomFormat = @"d. MMMM yyyy HH.mm' h'";
				Assert.AreEqual (@"8. February 2007 15.30 h", dt.Text, "#1105");
				dt.CustomFormat = @"d. MMMM yyyy H.mm' Uhr'";
				Assert.AreEqual (@"8. February 2007 15.30 Uhr", dt.Text, "#1106");
				dt.CustomFormat = @"d. MMM yy HH:mm";
				Assert.AreEqual (@"8. Feb 07 15:30", dt.Text, "#1107");
				dt.CustomFormat = @"d. MMM yy H:mm";
				Assert.AreEqual (@"8. Feb 07 15:30", dt.Text, "#1108");
				dt.CustomFormat = @"d. MMM yy H.mm' h'";
				Assert.AreEqual (@"8. Feb 07 15.30 h", dt.Text, "#1109");
				dt.CustomFormat = @"d. MMM yy HH.mm' h'";
				Assert.AreEqual (@"8. Feb 07 15.30 h", dt.Text, "#1110");
				dt.CustomFormat = @"d. MMM yy H.mm' Uhr'";
				Assert.AreEqual (@"8. Feb 07 15.30 Uhr", dt.Text, "#1111");
				dt.CustomFormat = @"d. MMM yy HH:mm:ss";
				Assert.AreEqual (@"8. Feb 07 15:30:45", dt.Text, "#1112");
				dt.CustomFormat = @"d. MMM yy H:mm:ss";
				Assert.AreEqual (@"8. Feb 07 15:30:45", dt.Text, "#1113");
				dt.CustomFormat = @"dd.MM.yyyy H.mm' h'";
				Assert.AreEqual (@"08.02.2007 15.30 h", dt.Text, "#1114");
				dt.CustomFormat = @"dd.MM.yyyy HH.mm' h'";
				Assert.AreEqual (@"08.02.2007 15.30 h", dt.Text, "#1115");
				dt.CustomFormat = @"dd.MM.yyyy H.mm' Uhr'";
				Assert.AreEqual (@"08.02.2007 15.30 Uhr", dt.Text, "#1116");
				dt.CustomFormat = @"dd.MM.yy H.mm' h'";
				Assert.AreEqual (@"08.02.07 15.30 h", dt.Text, "#1117");
				dt.CustomFormat = @"dd.MM.yy HH.mm' h'";
				Assert.AreEqual (@"08.02.07 15.30 h", dt.Text, "#1118");
				dt.CustomFormat = @"dd.MM.yy H.mm' Uhr'";
				Assert.AreEqual (@"08.02.07 15.30 Uhr", dt.Text, "#1119");
				dt.CustomFormat = @"d.MM.yy H.mm' h'";
				Assert.AreEqual (@"8.02.07 15.30 h", dt.Text, "#1120");
				dt.CustomFormat = @"d.MM.yy HH.mm' h'";
				Assert.AreEqual (@"8.02.07 15.30 h", dt.Text, "#1121");
				dt.CustomFormat = @"d.MM.yy H.mm' Uhr'";
				Assert.AreEqual (@"8.02.07 15.30 Uhr", dt.Text, "#1122");
				dt.CustomFormat = @"dd. M. yy HH:mm";
				Assert.AreEqual (@"08. 2. 07 15:30", dt.Text, "#1123");
				dt.CustomFormat = @"dd. M. yy H:mm";
				Assert.AreEqual (@"08. 2. 07 15:30", dt.Text, "#1124");
				dt.CustomFormat = @"dd. M. yy H.mm' h'";
				Assert.AreEqual (@"08. 2. 07 15.30 h", dt.Text, "#1125");
				dt.CustomFormat = @"dd. M. yy HH.mm' h'";
				Assert.AreEqual (@"08. 2. 07 15.30 h", dt.Text, "#1126");
				dt.CustomFormat = @"dd. M. yy H.mm' Uhr'";
				Assert.AreEqual (@"08. 2. 07 15.30 Uhr", dt.Text, "#1127");
				dt.CustomFormat = @"d.M.yy H.mm' h'";
				Assert.AreEqual (@"8.2.07 15.30 h", dt.Text, "#1128");
				dt.CustomFormat = @"d.M.yy HH.mm' h'";
				Assert.AreEqual (@"8.2.07 15.30 h", dt.Text, "#1129");
				dt.CustomFormat = @"d.M.yy H.mm' Uhr'";
				Assert.AreEqual (@"8.2.07 15.30 Uhr", dt.Text, "#1130");
				dt.CustomFormat = @"yyyy-MM-dd H.mm' h'";
				Assert.AreEqual (@"2007-02-08 15.30 h", dt.Text, "#1131");
				dt.CustomFormat = @"yyyy-MM-dd HH.mm' h'";
				Assert.AreEqual (@"2007-02-08 15.30 h", dt.Text, "#1132");
				dt.CustomFormat = @"yyyy-MM-dd H.mm' Uhr'";
				Assert.AreEqual (@"2007-02-08 15.30 Uhr", dt.Text, "#1133");
				dt.CustomFormat = @"dd. M. yy HH:mm:ss";
				Assert.AreEqual (@"08. 2. 07 15:30:45", dt.Text, "#1134");
				dt.CustomFormat = @"dd. M. yy H:mm:ss";
				Assert.AreEqual (@"08. 2. 07 15:30:45", dt.Text, "#1135");
				dt.CustomFormat = @"H.mm' h'";
				Assert.AreEqual (@"15.30 h", dt.Text, "#1136");
				dt.CustomFormat = @"HH.mm' h'";
				Assert.AreEqual (@"15.30 h", dt.Text, "#1137");
				dt.CustomFormat = @"H.mm' Uhr'";
				Assert.AreEqual (@"15.30 Uhr", dt.Text, "#1138");
				dt.CustomFormat = @"d.M.yy hh:mm tt";
				Assert.AreEqual (@"8.2.07 03:30 PM", dt.Text, "#1139");
				dt.CustomFormat = @"d.M.yy h:mm tt";
				Assert.AreEqual (@"8.2.07 3:30 PM", dt.Text, "#1140");
				dt.CustomFormat = @"d.M.yy hh:mm:ss tt";
				Assert.AreEqual (@"8.2.07 03:30:45 PM", dt.Text, "#1141");
				dt.CustomFormat = @"d.M.yy h:mm:ss tt";
				Assert.AreEqual (@"8.2.07 3:30:45 PM", dt.Text, "#1142");
				dt.CustomFormat = @"dddd, dd' de 'MMMM' de 'yyyy h:mm tt";
				Assert.AreEqual (@"Thursday, 08 de February de 2007 3:30 PM", dt.Text, "#1143");
				dt.CustomFormat = @"dddd d' de 'MMMM' de 'yyyy h:mm tt";
				Assert.AreEqual (@"Thursday 8 de February de 2007 3:30 PM", dt.Text, "#1144");
				dt.CustomFormat = @"d' de 'MMMM' de 'yyyy h:mm tt";
				Assert.AreEqual (@"8 de February de 2007 3:30 PM", dt.Text, "#1145");
				dt.CustomFormat = @"dddd, dd' de 'MMMM' de 'yyyy h:mm:ss tt";
				Assert.AreEqual (@"Thursday, 08 de February de 2007 3:30:45 PM", dt.Text, "#1146");
				dt.CustomFormat = @"dddd d' de 'MMMM' de 'yyyy h:mm:ss tt";
				Assert.AreEqual (@"Thursday 8 de February de 2007 3:30:45 PM", dt.Text, "#1147");
				dt.CustomFormat = @"d' de 'MMMM' de 'yyyy h:mm:ss tt";
				Assert.AreEqual (@"8 de February de 2007 3:30:45 PM", dt.Text, "#1148");
				dt.CustomFormat = @"d/MM/yy h:mm tt";
				Assert.AreEqual (@"8/02/07 3:30 PM", dt.Text, "#1149");
				dt.CustomFormat = @"d/MM/yy h:mm:ss tt";
				Assert.AreEqual (@"8/02/07 3:30:45 PM", dt.Text, "#1150");
				dt.CustomFormat = @"dddd d MMMM yyyy H' h 'mm";
				Assert.AreEqual (@"Thursday 8 February 2007 15 h 30", dt.Text, "#1151");
				dt.CustomFormat = @"dddd d MMMM yyyy H' h 'm' min '";
				Assert.AreEqual (@"Thursday 8 February 2007 15 h 30 min ", dt.Text, "#1152");
				dt.CustomFormat = @"d MMMM yyyy H' h 'mm";
				Assert.AreEqual (@"8 February 2007 15 h 30", dt.Text, "#1153");
				dt.CustomFormat = @"d MMMM yyyy H' h 'm' min '";
				Assert.AreEqual (@"8 February 2007 15 h 30 min ", dt.Text, "#1154");
				dt.CustomFormat = @"dd-MMM-yy H.mm";
				Assert.AreEqual (@"08-Feb-07 15.30", dt.Text, "#1155");
				dt.CustomFormat = @"dd-MMM-yy H' h 'mm";
				Assert.AreEqual (@"08-Feb-07 15 h 30", dt.Text, "#1156");
				dt.CustomFormat = @"dd-MMM-yy H' h 'm' min '";
				Assert.AreEqual (@"08-Feb-07 15 h 30 min ", dt.Text, "#1157");
				dt.CustomFormat = @"dddd d MMMM yyyy H' h 'm' min 's' s '";
				Assert.AreEqual (@"Thursday 8 February 2007 15 h 30 min 45 s ", dt.Text, "#1158");
				dt.CustomFormat = @"d MMMM yyyy H' h 'm' min 's' s '";
				Assert.AreEqual (@"8 February 2007 15 h 30 min 45 s ", dt.Text, "#1159");
				dt.CustomFormat = @"dd-MMM-yy H' h 'm' min 's' s '";
				Assert.AreEqual (@"08-Feb-07 15 h 30 min 45 s ", dt.Text, "#1160");
				dt.CustomFormat = @"d/MM/yyyy H.mm";
				Assert.AreEqual (@"8/02/2007 15.30", dt.Text, "#1161");
				dt.CustomFormat = @"d/MM/yyyy H' h 'mm";
				Assert.AreEqual (@"8/02/2007 15 h 30", dt.Text, "#1162");
				dt.CustomFormat = @"d/MM/yyyy H' h 'm' min '";
				Assert.AreEqual (@"8/02/2007 15 h 30 min ", dt.Text, "#1163");
				dt.CustomFormat = @"d/MM/yy H.mm";
				Assert.AreEqual (@"8/02/07 15.30", dt.Text, "#1164");
				dt.CustomFormat = @"d/MM/yy H' h 'mm";
				Assert.AreEqual (@"8/02/07 15 h 30", dt.Text, "#1165");
				dt.CustomFormat = @"d/MM/yy H' h 'm' min '";
				Assert.AreEqual (@"8/02/07 15 h 30 min ", dt.Text, "#1166");
				dt.CustomFormat = @"dd.MM.yy H' h 'mm";
				Assert.AreEqual (@"08.02.07 15 h 30", dt.Text, "#1167");
				dt.CustomFormat = @"dd.MM.yy H' h 'm' min '";
				Assert.AreEqual (@"08.02.07 15 h 30 min ", dt.Text, "#1168");
				dt.CustomFormat = @"yy/MM/dd H.mm";
				Assert.AreEqual (@"07/02/08 15.30", dt.Text, "#1169");
				dt.CustomFormat = @"yy/MM/dd H' h 'mm";
				Assert.AreEqual (@"07/02/08 15 h 30", dt.Text, "#1170");
				dt.CustomFormat = @"yy/MM/dd H' h 'm' min '";
				Assert.AreEqual (@"07/02/08 15 h 30 min ", dt.Text, "#1171");
				dt.CustomFormat = @"dd-MM-yy H' h 'mm";
				Assert.AreEqual (@"08-02-07 15 h 30", dt.Text, "#1172");
				dt.CustomFormat = @"dd-MM-yy H' h 'm' min '";
				Assert.AreEqual (@"08-02-07 15 h 30 min ", dt.Text, "#1173");
				dt.CustomFormat = @"dd/MM/yyyy H' h 'mm";
				Assert.AreEqual (@"08/02/2007 15 h 30", dt.Text, "#1174");
				dt.CustomFormat = @"dd/MM/yyyy H' h 'm' min '";
				Assert.AreEqual (@"08/02/2007 15 h 30 min ", dt.Text, "#1175");
				dt.CustomFormat = @"yyyy-MM-dd H' h 'mm";
				Assert.AreEqual (@"2007-02-08 15 h 30", dt.Text, "#1176");
				dt.CustomFormat = @"yyyy-MM-dd H' h 'm' min '";
				Assert.AreEqual (@"2007-02-08 15 h 30 min ", dt.Text, "#1177");
				dt.CustomFormat = @"d/MM/yyyy H' h 'm' min 's' s '";
				Assert.AreEqual (@"8/02/2007 15 h 30 min 45 s ", dt.Text, "#1178");
				dt.CustomFormat = @"d/MM/yy H' h 'm' min 's' s '";
				Assert.AreEqual (@"8/02/07 15 h 30 min 45 s ", dt.Text, "#1179");
				dt.CustomFormat = @"dd.MM.yy H' h 'm' min 's' s '";
				Assert.AreEqual (@"08.02.07 15 h 30 min 45 s ", dt.Text, "#1180");
				dt.CustomFormat = @"yy/MM/dd H' h 'm' min 's' s '";
				Assert.AreEqual (@"07/02/08 15 h 30 min 45 s ", dt.Text, "#1181");
				dt.CustomFormat = @"dd-MM-yy H' h 'm' min 's' s '";
				Assert.AreEqual (@"08-02-07 15 h 30 min 45 s ", dt.Text, "#1182");
				dt.CustomFormat = @"dd/MM/yyyy H' h 'm' min 's' s '";
				Assert.AreEqual (@"08/02/2007 15 h 30 min 45 s ", dt.Text, "#1183");
				dt.CustomFormat = @"yyyy-MM-dd H' h 'm' min 's' s '";
				Assert.AreEqual (@"2007-02-08 15 h 30 min 45 s ", dt.Text, "#1184");
				dt.CustomFormat = @"H' h 'mm";
				Assert.AreEqual (@"15 h 30", dt.Text, "#1185");
				dt.CustomFormat = @"H' h 'm' min '";
				Assert.AreEqual (@"15 h 30 min ", dt.Text, "#1186");
				dt.CustomFormat = @"H' h 'm' min 's' s '";
				Assert.AreEqual (@"15 h 30 min 45 s ", dt.Text, "#1187");
				dt.CustomFormat = @"d-MMM-yy H.mm' h'";
				Assert.AreEqual (@"8-Feb-07 15.30 h", dt.Text, "#1188");
				dt.CustomFormat = @"d MMMM yyyy H.mm' h'";
				Assert.AreEqual (@"8 February 2007 15.30 h", dt.Text, "#1189");
				dt.CustomFormat = @"dd. MM. yy H.mm' h'";
				Assert.AreEqual (@"08. 02. 07 15.30 h", dt.Text, "#1190");
				dt.CustomFormat = @"d/M/yy H.mm' h'";
				Assert.AreEqual (@"8/2/07 15.30 h", dt.Text, "#1191");
				dt.CustomFormat = @"dd.M.yy H.mm' h'";
				Assert.AreEqual (@"08.2.07 15.30 h", dt.Text, "#1192");
				dt.CustomFormat = @"dd.M.yy H:mm:ss";
				Assert.AreEqual (@"08.2.07 15:30:45", dt.Text, "#1193");
				dt.CustomFormat = @"dddd d MMMM yyyy H.mm' u.'";
				Assert.AreEqual (@"Thursday 8 February 2007 15.30 u.", dt.Text, "#1194");
				dt.CustomFormat = @"dd-MMM-yy H.mm' u.'";
				Assert.AreEqual (@"08-Feb-07 15.30 u.", dt.Text, "#1195");
				dt.CustomFormat = @"d MMMM yyyy H.mm' u.'";
				Assert.AreEqual (@"8 February 2007 15.30 u.", dt.Text, "#1196");
				dt.CustomFormat = @"dd MMM yy H:mm";
				Assert.AreEqual (@"08 Feb 07 15:30", dt.Text, "#1197");
				dt.CustomFormat = @"dd MMM yy H.mm' u.'";
				Assert.AreEqual (@"08 Feb 07 15.30 u.", dt.Text, "#1198");
				dt.CustomFormat = @"dd MMM yy H:mm:ss";
				Assert.AreEqual (@"08 Feb 07 15:30:45", dt.Text, "#1199");
				dt.CustomFormat = @"d/MM/yyyy H.mm' u.'";
				Assert.AreEqual (@"8/02/2007 15.30 u.", dt.Text, "#1200");
				dt.CustomFormat = @"d/MM/yy H.mm' u.'";
				Assert.AreEqual (@"8/02/07 15.30 u.", dt.Text, "#1201");
				dt.CustomFormat = @"dd-MM-yy H.mm' u.'";
				Assert.AreEqual (@"08-02-07 15.30 u.", dt.Text, "#1202");
				dt.CustomFormat = @"dd.MM.yy H.mm' u.'";
				Assert.AreEqual (@"08.02.07 15.30 u.", dt.Text, "#1203");
				dt.CustomFormat = @"yyyy-MM-dd H.mm' u.'";
				Assert.AreEqual (@"2007-02-08 15.30 u.", dt.Text, "#1204");
				dt.CustomFormat = @"H.mm' u.'";
				Assert.AreEqual (@"15.30 u.", dt.Text, "#1205");
				dt.CustomFormat = @"d/MMM/yy";
				Assert.AreEqual (@"8/Feb/07", dt.Text, "#1206");
				dt.CustomFormat = @"d.MMM.yy";
				Assert.AreEqual (@"8.Feb.07", dt.Text, "#1207");
				dt.CustomFormat = @"dddd, d' de 'MMMM' de 'yyyy HH'H'mm'm'";
				Assert.AreEqual (@"Thursday, 8 de February de 2007 15H30m", dt.Text, "#1208");
				dt.CustomFormat = @"d' de 'MMMM' de 'yyyy HH'H'mm'm'";
				Assert.AreEqual (@"8 de February de 2007 15H30m", dt.Text, "#1209");
				dt.CustomFormat = @"d/MMM/yy H:mm";
				Assert.AreEqual (@"8/Feb/07 15:30", dt.Text, "#1210");
				dt.CustomFormat = @"d/MMM/yy HH:mm";
				Assert.AreEqual (@"8/Feb/07 15:30", dt.Text, "#1211");
				dt.CustomFormat = @"d/MMM/yy HH'H'mm'm'";
				Assert.AreEqual (@"8/Feb/07 15H30m", dt.Text, "#1212");
				dt.CustomFormat = @"d.MMM.yy H:mm";
				Assert.AreEqual (@"8.Feb.07 15:30", dt.Text, "#1213");
				dt.CustomFormat = @"d.MMM.yy HH:mm";
				Assert.AreEqual (@"8.Feb.07 15:30", dt.Text, "#1214");
				dt.CustomFormat = @"d.MMM.yy HH'H'mm'm'";
				Assert.AreEqual (@"8.Feb.07 15H30m", dt.Text, "#1215");
				dt.CustomFormat = @"d/MMM/yy H:mm:ss";
				Assert.AreEqual (@"8/Feb/07 15:30:45", dt.Text, "#1216");
				dt.CustomFormat = @"d/MMM/yy HH:mm:ss";
				Assert.AreEqual (@"8/Feb/07 15:30:45", dt.Text, "#1217");
				dt.CustomFormat = @"d.MMM.yy H:mm:ss";
				Assert.AreEqual (@"8.Feb.07 15:30:45", dt.Text, "#1218");
				dt.CustomFormat = @"d.MMM.yy HH:mm:ss";
				Assert.AreEqual (@"8.Feb.07 15:30:45", dt.Text, "#1219");
				dt.CustomFormat = @"dd-MM-yyyy HH'H'mm'm'";
				Assert.AreEqual (@"08-02-2007 15H30m", dt.Text, "#1220");
				dt.CustomFormat = @"yy.MM.dd HH:mm";
				Assert.AreEqual (@"07.02.08 15:30", dt.Text, "#1221");
				dt.CustomFormat = @"yy.MM.dd HH'H'mm'm'";
				Assert.AreEqual (@"07.02.08 15H30m", dt.Text, "#1222");
				dt.CustomFormat = @"d.M.yy HH'H'mm'm'";
				Assert.AreEqual (@"8.2.07 15H30m", dt.Text, "#1223");
				dt.CustomFormat = @"dd/MM/yy HH'H'mm'm'";
				Assert.AreEqual (@"08/02/07 15H30m", dt.Text, "#1224");
				dt.CustomFormat = @"yyyy-MM-dd HH'H'mm'm'";
				Assert.AreEqual (@"2007-02-08 15H30m", dt.Text, "#1225");
				dt.CustomFormat = @"yy.MM.dd HH:mm:ss";
				Assert.AreEqual (@"07.02.08 15:30:45", dt.Text, "#1226");
				dt.CustomFormat = @"d/M";
				Assert.AreEqual (@"8/2", dt.Text, "#1227");
				dt.CustomFormat = @"HH'H'mm'm'";
				Assert.AreEqual (@"15H30m", dt.Text, "#1228");
				dt.CustomFormat = @"d.M.yyyy 'kl 'H:mm";
				Assert.AreEqual (@"8.2.2007 kl 15:30", dt.Text, "#1229");
				dt.CustomFormat = @"dd.MM.yyyy 'kl 'H:mm";
				Assert.AreEqual (@"08.02.2007 kl 15:30", dt.Text, "#1230");
				dt.CustomFormat = @"d.M.yy 'kl 'H:mm";
				Assert.AreEqual (@"8.2.07 kl 15:30", dt.Text, "#1231");
				dt.CustomFormat = @"yyyy 'йил' d-MMMM";
				Assert.AreEqual (@"2007 йил 8-February", dt.Text, "#1232");
				dt.CustomFormat = @"yyyy 'йил' d-MMMM HH:mm";
				Assert.AreEqual (@"2007 йил 8-February 15:30", dt.Text, "#1233");
				dt.CustomFormat = @"yyyy 'йил' d-MMMM H:mm";
				Assert.AreEqual (@"2007 йил 8-February 15:30", dt.Text, "#1234");
				dt.CustomFormat = @"yyyy 'йил' d-MMMM HH:mm:ss";
				Assert.AreEqual (@"2007 йил 8-February 15:30:45", dt.Text, "#1235");
				dt.CustomFormat = @"yyyy 'йил' d-MMMM H:mm:ss";
				Assert.AreEqual (@"2007 йил 8-February 15:30:45", dt.Text, "#1236");
				dt.CustomFormat = @"dddd, d MMMM, yyyy";
				Assert.AreEqual (@"Thursday, 8 February, 2007", dt.Text, "#1237");
				dt.CustomFormat = @"dddd yyyy MM dd";
				Assert.AreEqual (@"Thursday 2007 02 08", dt.Text, "#1238");
				dt.CustomFormat = @"dddd, d MMMM, yyyy H:mm";
				Assert.AreEqual (@"Thursday, 8 February, 2007 15:30", dt.Text, "#1239");
				dt.CustomFormat = @"dddd, d MMMM, yyyy HH:mm";
				Assert.AreEqual (@"Thursday, 8 February, 2007 15:30", dt.Text, "#1240");
				dt.CustomFormat = @"dddd yyyy MM dd H:mm";
				Assert.AreEqual (@"Thursday 2007 02 08 15:30", dt.Text, "#1241");
				dt.CustomFormat = @"dddd yyyy MM dd HH:mm";
				Assert.AreEqual (@"Thursday 2007 02 08 15:30", dt.Text, "#1242");
				dt.CustomFormat = @"dddd, d MMMM, yyyy H:mm:ss";
				Assert.AreEqual (@"Thursday, 8 February, 2007 15:30:45", dt.Text, "#1243");
				dt.CustomFormat = @"dddd, d MMMM, yyyy HH:mm:ss";
				Assert.AreEqual (@"Thursday, 8 February, 2007 15:30:45", dt.Text, "#1244");
				dt.CustomFormat = @"dddd yyyy MM dd H:mm:ss";
				Assert.AreEqual (@"Thursday 2007 02 08 15:30:45", dt.Text, "#1245");
				dt.CustomFormat = @"dddd yyyy MM dd HH:mm:ss";
				Assert.AreEqual (@"Thursday 2007 02 08 15:30:45", dt.Text, "#1246");
				dt.CustomFormat = @"dd.M.yyyy";
				Assert.AreEqual (@"08.2.2007", dt.Text, "#1247");
				dt.CustomFormat = @"dddd, dd. MMMM yyyy";
				Assert.AreEqual (@"Thursday, 08. February 2007", dt.Text, "#1248");
				dt.CustomFormat = @"d.MMMM yyyy";
				Assert.AreEqual (@"8.February 2007", dt.Text, "#1249");
				dt.CustomFormat = @"d.MMMyyyy";
				Assert.AreEqual (@"8.Feb2007", dt.Text, "#1250");
				dt.CustomFormat = @"dddd, dd. MMMM yyyy HH:mm";
				Assert.AreEqual (@"Thursday, 08. February 2007 15:30", dt.Text, "#1251");
				dt.CustomFormat = @"dddd, dd. MMMM yyyy H:mm";
				Assert.AreEqual (@"Thursday, 08. February 2007 15:30", dt.Text, "#1252");
				dt.CustomFormat = @"dddd, dd. MMMM yyyy HH:mm' Uhr'";
				Assert.AreEqual (@"Thursday, 08. February 2007 15:30 Uhr", dt.Text, "#1253");
				dt.CustomFormat = @"d.MMMM yyyy HH:mm";
				Assert.AreEqual (@"8.February 2007 15:30", dt.Text, "#1254");
				dt.CustomFormat = @"d.MMMM yyyy H:mm";
				Assert.AreEqual (@"8.February 2007 15:30", dt.Text, "#1255");
				dt.CustomFormat = @"d.MMMM yyyy HH:mm' Uhr'";
				Assert.AreEqual (@"8.February 2007 15:30 Uhr", dt.Text, "#1256");
				dt.CustomFormat = @"d.MMMyyyy HH:mm";
				Assert.AreEqual (@"8.Feb2007 15:30", dt.Text, "#1257");
				dt.CustomFormat = @"d.MMMyyyy H:mm";
				Assert.AreEqual (@"8.Feb2007 15:30", dt.Text, "#1258");
				dt.CustomFormat = @"d.MMMyyyy HH:mm' Uhr'";
				Assert.AreEqual (@"8.Feb2007 15:30 Uhr", dt.Text, "#1259");
				dt.CustomFormat = @"d MMM yyyy HH:mm' Uhr'";
				Assert.AreEqual (@"8 Feb 2007 15:30 Uhr", dt.Text, "#1260");
				dt.CustomFormat = @"dddd, dd. MMMM yyyy HH:mm:ss";
				Assert.AreEqual (@"Thursday, 08. February 2007 15:30:45", dt.Text, "#1261");
				dt.CustomFormat = @"dddd, dd. MMMM yyyy H:mm:ss";
				Assert.AreEqual (@"Thursday, 08. February 2007 15:30:45", dt.Text, "#1262");
				dt.CustomFormat = @"d.MMMM yyyy HH:mm:ss";
				Assert.AreEqual (@"8.February 2007 15:30:45", dt.Text, "#1263");
				dt.CustomFormat = @"d.MMMM yyyy H:mm:ss";
				Assert.AreEqual (@"8.February 2007 15:30:45", dt.Text, "#1264");
				dt.CustomFormat = @"d.MMMyyyy HH:mm:ss";
				Assert.AreEqual (@"8.Feb2007 15:30:45", dt.Text, "#1265");
				dt.CustomFormat = @"d.MMMyyyy H:mm:ss";
				Assert.AreEqual (@"8.Feb2007 15:30:45", dt.Text, "#1266");
				dt.CustomFormat = @"dd.MM.yyyy HH:mm' Uhr'";
				Assert.AreEqual (@"08.02.2007 15:30 Uhr", dt.Text, "#1267");
				dt.CustomFormat = @"dd.MM.yy HH:mm' Uhr'";
				Assert.AreEqual (@"08.02.07 15:30 Uhr", dt.Text, "#1268");
				dt.CustomFormat = @"dd.M.yyyy HH:mm";
				Assert.AreEqual (@"08.2.2007 15:30", dt.Text, "#1269");
				dt.CustomFormat = @"dd.M.yyyy H:mm";
				Assert.AreEqual (@"08.2.2007 15:30", dt.Text, "#1270");
				dt.CustomFormat = @"dd.M.yyyy HH:mm' Uhr'";
				Assert.AreEqual (@"08.2.2007 15:30 Uhr", dt.Text, "#1271");
				dt.CustomFormat = @"yyyy-MM-dd HH:mm' Uhr'";
				Assert.AreEqual (@"2007-02-08 15:30 Uhr", dt.Text, "#1272");
				dt.CustomFormat = @"dd.M.yyyy HH:mm:ss";
				Assert.AreEqual (@"08.2.2007 15:30:45", dt.Text, "#1273");
				dt.CustomFormat = @"dd.M.yyyy H:mm:ss";
				Assert.AreEqual (@"08.2.2007 15:30:45", dt.Text, "#1274");
				dt.CustomFormat = @"HH:mm' Uhr'";
				Assert.AreEqual (@"15:30 Uhr", dt.Text, "#1275");
				dt.CustomFormat = @"d/MM/yyyy h:mm tt";
				Assert.AreEqual (@"8/02/2007 3:30 PM", dt.Text, "#1276");
				dt.CustomFormat = @"dd-MMMM-yyyy h:mm tt";
				Assert.AreEqual (@"08-February-2007 3:30 PM", dt.Text, "#1277");
				dt.CustomFormat = @"dd-MMMM-yyyy H:mm";
				Assert.AreEqual (@"08-February-2007 15:30", dt.Text, "#1278");
				dt.CustomFormat = @"d/MM/yyyy h:mm:ss tt";
				Assert.AreEqual (@"8/02/2007 3:30:45 PM", dt.Text, "#1279");
				dt.CustomFormat = @"dd-MMMM-yyyy h:mm:ss tt";
				Assert.AreEqual (@"08-February-2007 3:30:45 PM", dt.Text, "#1280");
				dt.CustomFormat = @"dd-MMMM-yyyy H:mm:ss";
				Assert.AreEqual (@"08-February-2007 15:30:45", dt.Text, "#1281");
				dt.CustomFormat = @"d MMM yyyy H' h 'mm";
				Assert.AreEqual (@"8 Feb 2007 15 h 30", dt.Text, "#1282");
				dt.CustomFormat = @"yy-MM-dd H' h 'mm";
				Assert.AreEqual (@"07-02-08 15 h 30", dt.Text, "#1283");
				dt.CustomFormat = @"yy MM dd H' h 'mm";
				Assert.AreEqual (@"07 02 08 15 h 30", dt.Text, "#1284");
				dt.CustomFormat = @"dd/MM/yy H' h 'mm";
				Assert.AreEqual (@"08/02/07 15 h 30", dt.Text, "#1285");
				dt.CustomFormat = @"dddd, d MMMM, yyyy tt h:mm";
				Assert.AreEqual (@"Thursday, 8 February, 2007 PM 3:30", dt.Text, "#1286");
				dt.CustomFormat = @"dddd, d MMMM, yyyy tt hh:mm";
				Assert.AreEqual (@"Thursday, 8 February, 2007 PM 03:30", dt.Text, "#1287");
				dt.CustomFormat = @"d MMMM, yyyy tt h:mm";
				Assert.AreEqual (@"8 February, 2007 PM 3:30", dt.Text, "#1288");
				dt.CustomFormat = @"d MMMM, yyyy tt hh:mm";
				Assert.AreEqual (@"8 February, 2007 PM 03:30", dt.Text, "#1289");
				dt.CustomFormat = @"dddd yyyy MM dd tt h:mm";
				Assert.AreEqual (@"Thursday 2007 02 08 PM 3:30", dt.Text, "#1290");
				dt.CustomFormat = @"dddd yyyy MM dd tt hh:mm";
				Assert.AreEqual (@"Thursday 2007 02 08 PM 03:30", dt.Text, "#1291");
				dt.CustomFormat = @"yyyy MM dd tt h:mm";
				Assert.AreEqual (@"2007 02 08 PM 3:30", dt.Text, "#1292");
				dt.CustomFormat = @"yyyy MM dd tt hh:mm";
				Assert.AreEqual (@"2007 02 08 PM 03:30", dt.Text, "#1293");
				dt.CustomFormat = @"dddd, d MMMM, yyyy tt h:mm:ss";
				Assert.AreEqual (@"Thursday, 8 February, 2007 PM 3:30:45", dt.Text, "#1294");
				dt.CustomFormat = @"dddd, d MMMM, yyyy tt hh:mm:ss";
				Assert.AreEqual (@"Thursday, 8 February, 2007 PM 03:30:45", dt.Text, "#1295");
				dt.CustomFormat = @"d MMMM, yyyy tt h:mm:ss";
				Assert.AreEqual (@"8 February, 2007 PM 3:30:45", dt.Text, "#1296");
				dt.CustomFormat = @"d MMMM, yyyy tt hh:mm:ss";
				Assert.AreEqual (@"8 February, 2007 PM 03:30:45", dt.Text, "#1297");
				dt.CustomFormat = @"dddd yyyy MM dd tt h:mm:ss";
				Assert.AreEqual (@"Thursday 2007 02 08 PM 3:30:45", dt.Text, "#1298");
				dt.CustomFormat = @"dddd yyyy MM dd tt hh:mm:ss";
				Assert.AreEqual (@"Thursday 2007 02 08 PM 03:30:45", dt.Text, "#1299");
				dt.CustomFormat = @"yyyy MM dd tt h:mm:ss";
				Assert.AreEqual (@"2007 02 08 PM 3:30:45", dt.Text, "#1300");
				dt.CustomFormat = @"yyyy MM dd tt hh:mm:ss";
				Assert.AreEqual (@"2007 02 08 PM 03:30:45", dt.Text, "#1301");
				dt.CustomFormat = @"d/M/yyyy tt h:mm";
				Assert.AreEqual (@"8/2/2007 PM 3:30", dt.Text, "#1302");
				dt.CustomFormat = @"d/M/yyyy tt hh:mm";
				Assert.AreEqual (@"8/2/2007 PM 03:30", dt.Text, "#1303");
				dt.CustomFormat = @"d/M/yy tt h:mm";
				Assert.AreEqual (@"8/2/07 PM 3:30", dt.Text, "#1304");
				dt.CustomFormat = @"d/M/yy tt hh:mm";
				Assert.AreEqual (@"8/2/07 PM 03:30", dt.Text, "#1305");
				dt.CustomFormat = @"dd/MM/yy tt h:mm";
				Assert.AreEqual (@"08/02/07 PM 3:30", dt.Text, "#1306");
				dt.CustomFormat = @"dd/MM/yy tt hh:mm";
				Assert.AreEqual (@"08/02/07 PM 03:30", dt.Text, "#1307");
				dt.CustomFormat = @"d/M/yyyy tt h:mm:ss";
				Assert.AreEqual (@"8/2/2007 PM 3:30:45", dt.Text, "#1308");
				dt.CustomFormat = @"d/M/yyyy tt hh:mm:ss";
				Assert.AreEqual (@"8/2/2007 PM 03:30:45", dt.Text, "#1309");
				dt.CustomFormat = @"d/M/yy tt h:mm:ss";
				Assert.AreEqual (@"8/2/07 PM 3:30:45", dt.Text, "#1310");
				dt.CustomFormat = @"d/M/yy tt hh:mm:ss";
				Assert.AreEqual (@"8/2/07 PM 03:30:45", dt.Text, "#1311");
				dt.CustomFormat = @"dd/MM/yy tt h:mm:ss";
				Assert.AreEqual (@"08/02/07 PM 3:30:45", dt.Text, "#1312");
				dt.CustomFormat = @"dd/MM/yy tt hh:mm:ss";
				Assert.AreEqual (@"08/02/07 PM 03:30:45", dt.Text, "#1313");
				dt.CustomFormat = @"M/dd/yy";
				Assert.AreEqual (@"2/08/07", dt.Text, "#1314");
				dt.CustomFormat = @"MMMM d, yyyy";
				Assert.AreEqual (@"February 8, 2007", dt.Text, "#1315");
				dt.CustomFormat = @"MMMM d, yyyy h:mm tt";
				Assert.AreEqual (@"February 8, 2007 3:30 PM", dt.Text, "#1316");
				dt.CustomFormat = @"MMMM d, yyyy hh:mm tt";
				Assert.AreEqual (@"February 8, 2007 03:30 PM", dt.Text, "#1317");
				dt.CustomFormat = @"MMMM d, yyyy HH:mm";
				Assert.AreEqual (@"February 8, 2007 15:30", dt.Text, "#1318");
				dt.CustomFormat = @"MMMM d, yyyy H:mm";
				Assert.AreEqual (@"February 8, 2007 15:30", dt.Text, "#1319");
				dt.CustomFormat = @"d-MMM-yy h:mm tt";
				Assert.AreEqual (@"8-Feb-07 3:30 PM", dt.Text, "#1320");
				dt.CustomFormat = @"d-MMM-yy hh:mm tt";
				Assert.AreEqual (@"8-Feb-07 03:30 PM", dt.Text, "#1321");
				dt.CustomFormat = @"MMMM d, yyyy h:mm:ss tt";
				Assert.AreEqual (@"February 8, 2007 3:30:45 PM", dt.Text, "#1322");
				dt.CustomFormat = @"MMMM d, yyyy hh:mm:ss tt";
				Assert.AreEqual (@"February 8, 2007 03:30:45 PM", dt.Text, "#1323");
				dt.CustomFormat = @"MMMM d, yyyy HH:mm:ss";
				Assert.AreEqual (@"February 8, 2007 15:30:45", dt.Text, "#1324");
				dt.CustomFormat = @"MMMM d, yyyy H:mm:ss";
				Assert.AreEqual (@"February 8, 2007 15:30:45", dt.Text, "#1325");
				dt.CustomFormat = @"d-MMM-yy h:mm:ss tt";
				Assert.AreEqual (@"8-Feb-07 3:30:45 PM", dt.Text, "#1326");
				dt.CustomFormat = @"d-MMM-yy hh:mm:ss tt";
				Assert.AreEqual (@"8-Feb-07 03:30:45 PM", dt.Text, "#1327");
				dt.CustomFormat = @"yy-MM-dd hh:mm tt";
				Assert.AreEqual (@"07-02-08 03:30 PM", dt.Text, "#1328");
				dt.CustomFormat = @"M/dd/yy h:mm tt";
				Assert.AreEqual (@"2/08/07 3:30 PM", dt.Text, "#1329");
				dt.CustomFormat = @"M/dd/yy hh:mm tt";
				Assert.AreEqual (@"2/08/07 03:30 PM", dt.Text, "#1330");
				dt.CustomFormat = @"M/dd/yy HH:mm";
				Assert.AreEqual (@"2/08/07 15:30", dt.Text, "#1331");
				dt.CustomFormat = @"M/dd/yy H:mm";
				Assert.AreEqual (@"2/08/07 15:30", dt.Text, "#1332");
				dt.CustomFormat = @"yy-MM-dd hh:mm:ss tt";
				Assert.AreEqual (@"07-02-08 03:30:45 PM", dt.Text, "#1333");
				dt.CustomFormat = @"M/dd/yy h:mm:ss tt";
				Assert.AreEqual (@"2/08/07 3:30:45 PM", dt.Text, "#1334");
				dt.CustomFormat = @"M/dd/yy hh:mm:ss tt";
				Assert.AreEqual (@"2/08/07 03:30:45 PM", dt.Text, "#1335");
				dt.CustomFormat = @"M/dd/yy HH:mm:ss";
				Assert.AreEqual (@"2/08/07 15:30:45", dt.Text, "#1336");
				dt.CustomFormat = @"M/dd/yy H:mm:ss";
				Assert.AreEqual (@"2/08/07 15:30:45", dt.Text, "#1337");
				dt.CustomFormat = @"d/MM/yyyy hh:mm tt";
				Assert.AreEqual (@"8/02/2007 03:30 PM", dt.Text, "#1338");
				dt.CustomFormat = @"d/MM/yyyy hh:mm:ss tt";
				Assert.AreEqual (@"8/02/2007 03:30:45 PM", dt.Text, "#1339");
				dt.CustomFormat = @"d MMM yy HH.mm' h'";
				Assert.AreEqual (@"8 Feb 07 15.30 h", dt.Text, "#1340");
				dt.CustomFormat = @"d.MM.yy h:mm tt";
				Assert.AreEqual (@"8.02.07 3:30 PM", dt.Text, "#1341");
				dt.CustomFormat = @"d.MM.yy hh:mm tt";
				Assert.AreEqual (@"8.02.07 03:30 PM", dt.Text, "#1342");
				dt.CustomFormat = @"d.MM.yy h:mm:ss tt";
				Assert.AreEqual (@"8.02.07 3:30:45 PM", dt.Text, "#1343");
				dt.CustomFormat = @"d.MM.yy hh:mm:ss tt";
				Assert.AreEqual (@"8.02.07 03:30:45 PM", dt.Text, "#1344");
				dt.CustomFormat = @"dddd, dd MMMM yyyy hh:mm tt";
				Assert.AreEqual (@"Thursday, 08 February 2007 03:30 PM", dt.Text, "#1345");
				dt.CustomFormat = @"dddd, dd MMMM yyyy hh:mm:ss tt";
				Assert.AreEqual (@"Thursday, 08 February 2007 03:30:45 PM", dt.Text, "#1346");
				dt.CustomFormat = @"MM-dd-yyyy";
				Assert.AreEqual (@"02-08-2007", dt.Text, "#1347");
				dt.CustomFormat = @"MM-dd-yyyy hh:mm tt";
				Assert.AreEqual (@"02-08-2007 03:30 PM", dt.Text, "#1348");
				dt.CustomFormat = @"MM-dd-yyyy HH:mm";
				Assert.AreEqual (@"02-08-2007 15:30", dt.Text, "#1349");
				dt.CustomFormat = @"MM-dd-yyyy hh:mm:ss tt";
				Assert.AreEqual (@"02-08-2007 03:30:45 PM", dt.Text, "#1350");
				dt.CustomFormat = @"MM-dd-yyyy HH:mm:ss";
				Assert.AreEqual (@"02-08-2007 15:30:45", dt.Text, "#1351");
				dt.CustomFormat = @"MMMM d'. b. 'yyyy";
				Assert.AreEqual (@"February 8. b. 2007", dt.Text, "#1352");
				dt.CustomFormat = @"MMMM d'. b. 'yyyy HH:mm:ss";
				Assert.AreEqual (@"February 8. b. 2007 15:30:45", dt.Text, "#1353");
				dt.CustomFormat = @"MMMM d'. b. 'yyyy H:mm:ss";
				Assert.AreEqual (@"February 8. b. 2007 15:30:45", dt.Text, "#1354");
				dt.CustomFormat = @"MMMM d'. b. 'yyyy HH.mm";
				Assert.AreEqual (@"February 8. b. 2007 15.30", dt.Text, "#1355");
				dt.CustomFormat = @"MMMM d'. b. 'yyyy HH.mm.ss";
				Assert.AreEqual (@"February 8. b. 2007 15.30.45", dt.Text, "#1356");
				dt.CustomFormat = @"MMMM d'. b. 'yyyy H:mm";
				Assert.AreEqual (@"February 8. b. 2007 15:30", dt.Text, "#1357");
				dt.CustomFormat = @"MMMM d'. p. 'yyyy";
				Assert.AreEqual (@"February 8. p. 2007", dt.Text, "#1358");
				dt.CustomFormat = @"MMMM d'. p. 'yyyy H:mm:ss";
				Assert.AreEqual (@"February 8. p. 2007 15:30:45", dt.Text, "#1359");
				dt.CustomFormat = @"MMMM d'. p. 'yyyy HH:mm:ss";
				Assert.AreEqual (@"February 8. p. 2007 15:30:45", dt.Text, "#1360");
				dt.CustomFormat = @"MMMM d'. p. 'yyyy H:mm";
				Assert.AreEqual (@"February 8. p. 2007 15:30", dt.Text, "#1361");
				dt.CustomFormat = @"ddMMyyyy";
				Assert.AreEqual (@"08022007", dt.Text, "#1362");
				dt.CustomFormat = @"dddd, d' ta\' 'MMMM yyyy";
				Assert.AreEqual (@"Thursday, 8 ta\ MMMM yyyy", dt.Text, "#1363");
				dt.CustomFormat = @"d' ta\' 'MMMM yyyy";
				Assert.AreEqual (@"8 ta\ MMMM yyyy", dt.Text, "#1364");
				dt.CustomFormat = @"dddd, d' ta\' 'MMMM yyyy HH:mm:ss";
				Assert.AreEqual (@"Thursday, 8 ta\ MMMM yyyy HH:mm:ss", dt.Text, "#1365");
				dt.CustomFormat = @"dddd, d' ta\' 'MMMM yyyy h:mm:ss tt";
				Assert.AreEqual (@"Thursday, 8 ta\ MMMM yyyy h:mm:ss tt", dt.Text, "#1366");
				dt.CustomFormat = @"dddd, d' ta\' 'MMMM yyyy H:mm:ss";
				Assert.AreEqual (@"Thursday, 8 ta\ MMMM yyyy H:mm:ss", dt.Text, "#1367");
				dt.CustomFormat = @"dddd, d' ta\' 'MMMM yyyy hh:mm:ss tt";
				Assert.AreEqual (@"Thursday, 8 ta\ MMMM yyyy hh:mm:ss tt", dt.Text, "#1368");
				dt.CustomFormat = @"d' ta\' 'MMMM yyyy HH:mm:ss";
				Assert.AreEqual (@"8 ta\ MMMM yyyy HH:mm:ss", dt.Text, "#1369");
				dt.CustomFormat = @"d' ta\' 'MMMM yyyy h:mm:ss tt";
				Assert.AreEqual (@"8 ta\ MMMM yyyy h:mm:ss tt", dt.Text, "#1370");
				dt.CustomFormat = @"d' ta\' 'MMMM yyyy H:mm:ss";
				Assert.AreEqual (@"8 ta\ MMMM yyyy H:mm:ss", dt.Text, "#1371");
				dt.CustomFormat = @"d' ta\' 'MMMM yyyy hh:mm:ss tt";
				Assert.AreEqual (@"8 ta\ MMMM yyyy hh:mm:ss tt", dt.Text, "#1372");
				dt.CustomFormat = @"ddMMyyyy HH:mm:ss";
				Assert.AreEqual (@"08022007 15:30:45", dt.Text, "#1373");
				dt.CustomFormat = @"ddMMyyyy h:mm:ss tt";
				Assert.AreEqual (@"08022007 3:30:45 PM", dt.Text, "#1374");
				dt.CustomFormat = @"ddMMyyyy H:mm:ss";
				Assert.AreEqual (@"08022007 15:30:45", dt.Text, "#1375");
				dt.CustomFormat = @"ddMMyyyy hh:mm:ss tt";
				Assert.AreEqual (@"08022007 03:30:45 PM", dt.Text, "#1376");
				dt.CustomFormat = @"d-M-yyyy h:mm:ss tt";
				Assert.AreEqual (@"8-2-2007 3:30:45 PM", dt.Text, "#1377");
				dt.CustomFormat = @"d-M-yyyy hh:mm:ss tt";
				Assert.AreEqual (@"8-2-2007 03:30:45 PM", dt.Text, "#1378");
				dt.CustomFormat = @"y";
				Assert.AreEqual (@"7", dt.Text, "#1379");
				dt.CustomFormat = @"yy";
				Assert.AreEqual (@"07", dt.Text, "#1380");
				dt.CustomFormat = @"yyy";
				Assert.AreEqual (@"2007", dt.Text, "#1381");
				dt.CustomFormat = @"yyyy";
				Assert.AreEqual (@"2007", dt.Text, "#1382");
				dt.CustomFormat = @"yyyyy";
				Assert.AreEqual (@"2007", dt.Text, "#1383");
				dt.CustomFormat = @"yyyyyyy";
				Assert.AreEqual (@"2007", dt.Text, "#1384");
				dt.CustomFormat = @"M";
				Assert.AreEqual (@"2", dt.Text, "#1385");
				dt.CustomFormat = @"MM";
				Assert.AreEqual (@"02", dt.Text, "#1386");
				dt.CustomFormat = @"MMM";
				Assert.AreEqual (@"Feb", dt.Text, "#1387");
				dt.CustomFormat = @"MMMM";
				Assert.AreEqual (@"February", dt.Text, "#1388");
				dt.CustomFormat = @"MMMMM";
				Assert.AreEqual (@"February", dt.Text, "#1389");
				dt.CustomFormat = @"d";
				Assert.AreEqual (@"8", dt.Text, "#1390");
				dt.CustomFormat = @"dd";
				Assert.AreEqual (@"08", dt.Text, "#1391");
				dt.CustomFormat = @"ddd";
				Assert.AreEqual (@"Thu", dt.Text, "#1392");
				dt.CustomFormat = @"dddd";
				Assert.AreEqual (@"Thursday", dt.Text, "#1393");
				dt.CustomFormat = @"ddddd";
				Assert.AreEqual (@"Thursday", dt.Text, "#1394");
				dt.CustomFormat = @"H";
				Assert.AreEqual (@"15", dt.Text, "#1395");
				dt.CustomFormat = @"HH";
				Assert.AreEqual (@"15", dt.Text, "#1396");
				dt.CustomFormat = @"HHH";
				Assert.AreEqual (@"15", dt.Text, "#1397");
				dt.CustomFormat = @"HHHHH";
				Assert.AreEqual (@"15", dt.Text, "#1398");
				dt.CustomFormat = @"h";
				Assert.AreEqual (@"3", dt.Text, "#1399");
				dt.CustomFormat = @"hh";
				Assert.AreEqual (@"03", dt.Text, "#1400");
				dt.CustomFormat = @"hhh";
				Assert.AreEqual (@"03", dt.Text, "#1401");
				dt.CustomFormat = @"hhhh";
				Assert.AreEqual (@"03", dt.Text, "#1402");
				dt.CustomFormat = @"hhhhh";
				Assert.AreEqual (@"03", dt.Text, "#1403");
				dt.CustomFormat = @"m";
				Assert.AreEqual (@"30", dt.Text, "#1404");
				dt.CustomFormat = @"mm";
				Assert.AreEqual (@"30", dt.Text, "#1405");
				dt.CustomFormat = @"mmm";
				Assert.AreEqual (@"30", dt.Text, "#1406");
				dt.CustomFormat = @"mmmm";
				Assert.AreEqual (@"30", dt.Text, "#1407");
				dt.CustomFormat = @"mmmmm";
				Assert.AreEqual (@"30", dt.Text, "#1408");
				dt.CustomFormat = @"s";
				Assert.AreEqual (@"45", dt.Text, "#1409");
				dt.CustomFormat = @"ss";
				Assert.AreEqual (@"45", dt.Text, "#1410");
				dt.CustomFormat = @"sss";
				Assert.AreEqual (@"45", dt.Text, "#1411");
				dt.CustomFormat = @"ssss";
				Assert.AreEqual (@"45", dt.Text, "#1412");
				dt.CustomFormat = @"sssss";
				Assert.AreEqual (@"45", dt.Text, "#1413");
				dt.CustomFormat = @"t";
				Assert.AreEqual (@"P", dt.Text, "#1414");
				dt.CustomFormat = @"tt";
				Assert.AreEqual (@"PM", dt.Text, "#1415");
				dt.CustomFormat = @"ttt";
				Assert.AreEqual (@"PM", dt.Text, "#1416");
				dt.CustomFormat = @"tttt";
				Assert.AreEqual (@"PM", dt.Text, "#1417");
				dt.CustomFormat = @"ttttt";
				Assert.AreEqual (@"PM", dt.Text, "#1418");
				dt.CustomFormat = @"M/y";
				Assert.AreEqual (@"2/7", dt.Text, "#1419");
				dt.CustomFormat = @"yy-MM";
				Assert.AreEqual (@"07-02", dt.Text, "#1420");
				dt.CustomFormat = @"MM-yy";
				Assert.AreEqual (@"02-07", dt.Text, "#1421");
				dt.CustomFormat = @"M-y-d-h-H-m-s-t";
				Assert.AreEqual (@"2-7-8-3-15-30-45-P", dt.Text, "#1422");
				dt.CustomFormat = @" yy-MM";
				Assert.AreEqual (@" 07-02", dt.Text, "#1423");
				dt.CustomFormat = @"-yy-MM";
				Assert.AreEqual (@"-07-02", dt.Text, "#1424");
			}
		}

		[Test]
		public void CustomFormatTest ()
		{
			CheckCulture ();
			
			using (Form frm = new Form ()) {
				DateTimePicker dt = new DateTimePicker ();
				frm.Controls.Add (dt);
				frm.Show ();
				
				dt.Format = DateTimePickerFormat.Custom;
				dt.Value = new DateTime (2007, 2, 8, 15, 30, 45, 60);

				/* 
				 * This is really weird and necessary, otherwise the tests won't succeed on windows.
				 * other strings that can be used here: "a", "dddd", " ddd", "'a'", "'d'" + a probably a lot more.
				 * seems like the first non-literal must be ddd and cannot be an empty string
				 * Without this everytime a "y" or "yy" comes first in the format, it will always show as 
				 * a 4-digit string if this is not done.
				 */
				if (typeof (int).GetType ().GetType ().Name != "MonoType") {
					// Only do this for MS, not implemented in Mono.
					dt.CustomFormat = "y";
					Assert.AreEqual ("2007", dt.Text, "#msbug1?");
				}
				dt.CustomFormat = "ddd"; // Setting CustomFormat to this string (among others) seems to fix it.
					dt.CustomFormat = "y";
					Assert.AreEqual ("7", dt.Text, "#msfix1?");

				dt.CustomFormat = Thread.CurrentThread.CurrentCulture.DateTimeFormat.ShortDatePattern;
				Assert.AreEqual ("2/8/2007", dt.Text, "#01");

				dt.CustomFormat = "y-M-d-h-H-m-s-t";
				Assert.AreEqual ("7-2-8-3-15-30-45-P", dt.Text, "#02");

				dt.CustomFormat = "M/y";
				Assert.AreEqual ("2/7", dt.Text, "#03");

				dt.CustomFormat = "yyyyy";
				Assert.AreEqual ("2007", dt.Text, "#04");
				
				dt.CustomFormat = "yyyyyyyy";
				Assert.AreEqual ("2007", dt.Text, "#05");
				
				dt.CustomFormat = "HHHHHHHH";
				Assert.AreEqual ("15", dt.Text, "#06");
				
				dt.CustomFormat = "yy-MM";
				Assert.AreEqual ("07-02", dt.Text, "#07");
				
				dt.CustomFormat = "MM-yy";
				Assert.AreEqual ("02-07", dt.Text, "#08");
				
				dt.CustomFormat = "M-y-d-h-H-m-s-t";
				Assert.AreEqual ("2-7-8-3-15-30-45-P", dt.Text, "#09");
				
				dt.CustomFormat = "y";
				Assert.AreEqual ("7", dt.Text, "#10");

				dt.CustomFormat = "M";
				Assert.AreEqual ("2", dt.Text, "#11");
				
				dt.CustomFormat = "h";
				Assert.AreEqual ("3", dt.Text, "#12");
				
				dt.CustomFormat = "m";
				Assert.AreEqual ("30", dt.Text, "#13");
				
				dt.CustomFormat = "d";
				Assert.AreEqual ("8", dt.Text, "#14");
				
				dt.CustomFormat = "t";
				Assert.AreEqual ("P", dt.Text, "#15");
				
				dt.CustomFormat = "s";
				Assert.AreEqual ("45", dt.Text, "#16");
				
				dt.CustomFormat = "H";
				Assert.AreEqual ("15", dt.Text, "#17");

				dt.CustomFormat = " yy-MM";
				Assert.AreEqual (" 07-02", dt.Text, "#20");
				
				dt.CustomFormat = "-yy-MM";
				Assert.AreEqual ("-07-02", dt.Text, "#21");

				dt.CustomFormat = "-yy-MM-dd";
				Assert.AreEqual ("-07-02-08", dt.Text, "#22");
				
				dt.CustomFormat = "yy-M-d";
				Assert.AreEqual ("07-2-8", dt.Text, "#23");
				
				dt.CustomFormat = "yy-M-d gg";
				Assert.AreEqual ("07-2-8  ", dt.Text, "#24");
				
				dt.CustomFormat = "yy-M-d g";
				Assert.AreEqual ("07-2-8  ", dt.Text, "#25");
				
				dt.CustomFormat = "g";
				Assert.AreEqual (" ", dt.Text, "#26");
				
				dt.CustomFormat = "gggggg";
				Assert.AreEqual (" ", dt.Text, "#27");
			}
		}
		
		[Test]
		public void CustomFormatNullTest ()
		{
			DateTimePicker dt = new DateTimePicker ();
			dt.CustomFormat = null;
		}
		
		[Test]
		public void DefaultPropertiesTest ()
		{
			DateTimePicker dt = new DateTimePicker ();
			
			Assert.AreEqual ("Window", dt.BackColor.Name, "B1");
			Assert.AreSame (null, dt.BackgroundImage, "B2");
#if NET_2_0
			Assert.AreEqual (ImageLayout.Tile, dt.BackgroundImageLayout, "B3");
#endif

			//Assert.AreSame (null, dt.CalendarFont, "C1");
			Assert.AreEqual ("ControlText", dt.CalendarForeColor.Name, "C2");
			Assert.AreEqual ("Window", dt.CalendarMonthBackground.Name, "C3");
			Assert.AreEqual ("ActiveCaption", dt.CalendarTitleBackColor.Name, "C4");
			Assert.AreEqual ("ActiveCaptionText", dt.CalendarTitleForeColor.Name, "C5");
			Assert.AreEqual ("GrayText", dt.CalendarTrailingForeColor.Name, "C6");
			Assert.AreEqual (true, dt.Checked, "C7");
			Assert.AreEqual (null, dt.CustomFormat, "C8");

			Assert.AreEqual (LeftRightAlignment.Left, dt.DropDownAlign, "D1");
			
			Assert.AreEqual ("WindowText", dt.ForeColor.Name, "F1");
			Assert.AreEqual (DateTimePickerFormat.Long, dt.Format, "F2");
			
			Assert.AreEqual (new DateTime (9998, 12, 31, 0, 0, 0), DateTimePicker.MaxDateTime, "M2");
			Assert.AreEqual (new DateTime (1753, 1, 1), DateTimePicker.MinDateTime, "M4");
#if NET_2_0
			Assert.AreEqual (new DateTime (9998, 12, 31, 0, 0, 0), DateTimePicker.MaximumDateTime, "M5");
			Assert.AreEqual (new DateTime (1753, 1, 1), DateTimePicker.MinimumDateTime, "M6");
#endif

#if NET_2_0
			Assert.AreEqual (new Padding (0, 0, 0, 0), dt.Padding, "P1");
#endif
			// PreferredHeight is Font dependent.
			
#if NET_2_0
			Assert.AreEqual (false, dt.RightToLeftLayout, "R1");
#endif

			Assert.AreEqual (false, dt.ShowCheckBox, "S1");
			Assert.AreEqual (false, dt.ShowUpDown, "S2");
			
			Assert.AreEqual ("", dt.Text, "T1");
			
			Assert.AreEqual (DateTime.Today, dt.Value.Date, "V1");
		}

		[Test]
		public void MaxDate ()
		{
			DateTimePicker dt = new DateTimePicker ();
			Assert.AreEqual (new DateTime (9998, 12, 31), dt.MaxDate, "#1");
			dt.Value = new DateTime (2007, 8, 13);
			Assert.AreEqual (new DateTime (9998, 12, 31), dt.MaxDate, "#2");
			Assert.AreEqual (new DateTime (2007, 8, 13), dt.Value, "#3");
			dt.MaxDate = new DateTime (2010, 2, 10);
			Assert.AreEqual (new DateTime (2010, 2, 10), dt.MaxDate, "#4");
			Assert.AreEqual (new DateTime (2007, 8, 13), dt.Value, "#5");
			dt.MaxDate = new DateTime (2005, 10, 15);
			Assert.AreEqual (new DateTime (2005, 10, 15), dt.MaxDate, "#6");
			Assert.AreEqual (new DateTime (2005, 10, 15), dt.Value, "#7");
			dt.MaxDate = new DateTime (2008, 1, 4);
			Assert.AreEqual (new DateTime (2008, 1, 4), dt.MaxDate, "#8");
			Assert.AreEqual (new DateTime (2005, 10, 15), dt.Value, "#9");
			dt.MaxDate = dt.MinDate;
			Assert.AreEqual (new DateTime (1753, 1, 1), dt.MaxDate, "#10");
			Assert.AreEqual (new DateTime (1753, 1, 1), dt.Value, "#11");
			dt.MaxDate = DateTimePicker.MaxDateTime;
			Assert.AreEqual (DateTimePicker.MaxDateTime, dt.MaxDate, "#12");
			Assert.AreEqual (new DateTime (1753, 1, 1), dt.Value, "#13");
		}

		[Test]
		public void MaxDate_Invalid ()
		{
			DateTimePicker dt = new DateTimePicker ();

			// not less or equal to MaxDateTime
			try {
				dt.MaxDate = new DateTime (9999, 1, 1);
				Assert.Fail ("#A1");
#if NET_2_0
			} catch (ArgumentOutOfRangeException ex) {
				// DateTimePicker does not support dates after 12/31/9998 12:00:00 AM
				Assert.AreEqual (typeof (ArgumentOutOfRangeException), ex.GetType (), "#A2");
				Assert.IsNull (ex.InnerException, "#A3");
				Assert.IsNotNull (ex.Message, "#A4");
				Assert.IsNotNull (ex.ParamName, "#A5");
				Assert.AreEqual ("MaxDate", ex.ParamName, "#A6");
			}
#else
			} catch (ArgumentException ex) {
				// DateTimePicker does not support dates after 12/31/9998 12:00:00 AM
				Assert.AreEqual (typeof (ArgumentException), ex.GetType (), "#A2");
				Assert.IsNull (ex.InnerException, "#A3");
				Assert.IsNotNull (ex.Message, "#A4");
				Assert.IsNotNull (ex.ParamName, "#A5");
				Assert.AreEqual ("value", ex.ParamName, "#A6");
			}
#endif

			dt.MinDate = new DateTime (2007, 8, 13);

			// not less than MinDate
			try {
				dt.MaxDate = new DateTime (2007, 8, 12);
				Assert.Fail ("#B1");
#if NET_2_0
			} catch (ArgumentOutOfRangeException ex) {
				// '8/12/2007 12:00:00 AM' is not a valid value for 'MaxDate'.
				// 'MaxDate' must be greater than or equal to MinDate.
				Assert.AreEqual (typeof (ArgumentOutOfRangeException), ex.GetType (), "#B2");
				Assert.IsNull (ex.InnerException, "#B3");
				Assert.IsNotNull (ex.Message, "#B4");
				Assert.IsNotNull (ex.ParamName, "#B5");
				Assert.AreEqual ("MaxDate", ex.ParamName, "#B6");
			}
#else
			} catch (ArgumentException ex) {
				// '8/12/2007 12:00:00 AM' is not a valid value for 'MaxDate'.
				// 'MaxDate' must be greater than or equal to MinDate.
				Assert.AreEqual (typeof (ArgumentException), ex.GetType (), "#B2");
				Assert.IsNull (ex.InnerException, "#B3");
				Assert.IsNotNull (ex.Message, "#B4");
				Assert.IsNull (ex.ParamName, "#B5");
			}
#endif
		}

		[Test]
		public void MinDate ()
		{
			DateTimePicker dt = new DateTimePicker ();
			Assert.AreEqual (new DateTime (1753, 1, 1), dt.MinDate, "#1");
			dt.Value = new DateTime (2007, 8, 13);
			Assert.AreEqual (new DateTime (1753, 1, 1), dt.MinDate, "#2");
			Assert.AreEqual (new DateTime (2007, 8, 13), dt.Value, "#3");
			dt.MinDate = new DateTime (2005, 1, 15);
			Assert.AreEqual (new DateTime (2005, 1, 15), dt.MinDate, "#4");
			Assert.AreEqual (new DateTime (2007, 8, 13), dt.Value, "#5");
			dt.MinDate = new DateTime (2008, 2, 5);
			Assert.AreEqual (new DateTime (2008, 2, 5), dt.MinDate, "#6");
			Assert.AreEqual (new DateTime (2008, 2, 5), dt.Value, "#7");
			dt.MinDate = new DateTime (2004, 8, 20);
			Assert.AreEqual (new DateTime (2004, 8, 20), dt.MinDate, "#8");
			Assert.AreEqual (new DateTime (2008, 2, 5), dt.Value, "#9");
			dt.MinDate = DateTimePicker.MinDateTime;
			Assert.AreEqual (DateTimePicker.MinDateTime, dt.MinDate, "#10");
			Assert.AreEqual (new DateTime (2008, 2, 5), dt.Value, "#11");
		}

#if NET_2_0
		[Test]
		public void DateTimeMinDate ()
		{
			DateTimePicker dt = new DateTimePicker ();

			// MinDate can be set to DateTime.MinValue, which sets MinDate to
			// DateTimePicker.MinDateTime
			dt.MinDate = DateTime.MinValue;
			Assert.AreEqual (DateTimePicker.MinimumDateTime, dt.MinDate, "A1");
		}
#endif

		[Test]
		public void MinDate_Invalid ()
		{
			DateTimePicker dt = new DateTimePicker ();

			// less than MinDateTime
			try {
				dt.MinDate = new DateTime (1752, 12, 31);
				Assert.Fail ("#A1");
#if NET_2_0
			} catch (ArgumentOutOfRangeException ex) {
				// DateTimePicker does not support dates before 1/1/1753 12:00:00 AM
				Assert.AreEqual (typeof (ArgumentOutOfRangeException), ex.GetType (), "#A2");
				Assert.IsNull (ex.InnerException, "#A3");
				Assert.IsNotNull (ex.Message, "#A4");
				Assert.IsNotNull (ex.ParamName, "#A5");
				Assert.AreEqual ("MinDate", ex.ParamName, "#A6");
			}
#else
			} catch (ArgumentException ex) {
				// DateTimePicker does not support dates before 1/1/1753 12:00:00 AM
				Assert.AreEqual (typeof (ArgumentException), ex.GetType (), "#A2");
				Assert.IsNull (ex.InnerException, "#A3");
				Assert.IsNotNull (ex.Message, "#A4");
				Assert.IsNotNull (ex.ParamName, "#A5");
				Assert.AreEqual ("value", ex.ParamName, "#A6");
			}
#endif

			dt.MaxDate = new DateTime (2007, 8, 13);

			// equal to MaxDate
#if NET_2_0
			dt.MinDate = new DateTime (2007, 8, 13);
			Assert.AreEqual (new DateTime (2007, 8, 13), dt.MinDate, "#B1");
#else
			try {
				dt.MinDate = new DateTime (2007, 8, 13);
				Assert.Fail ("#B1");
			} catch (ArgumentException ex) {
				// '8/13/2007 12:00:00 AM' is not a valid value for 'MinDate'.
				// 'MinDate' must be less than MaxDate
				Assert.AreEqual (typeof (ArgumentException), ex.GetType (), "#B2");
				Assert.IsNull (ex.InnerException, "#B3");
				Assert.IsNotNull (ex.Message, "#B4");
				Assert.IsNull (ex.ParamName, "#B5");
			}
#endif

			// not less than MaxDate
			try {
				dt.MinDate = new DateTime (2007, 8, 14);
				Assert.Fail ("#C1");
#if NET_2_0
			} catch (ArgumentOutOfRangeException ex) {
				// '8/14/2007 12:00:00 AM' is not a valid value for 'MinDate'.
				// 'MinDate' must be less than MaxDate
				Assert.AreEqual (typeof (ArgumentOutOfRangeException), ex.GetType (), "#C2");
				Assert.IsNull (ex.InnerException, "#C3");
				Assert.IsNotNull (ex.Message, "#C4");
				Assert.IsNotNull (ex.ParamName, "#C5");
				Assert.AreEqual ("MinDate", ex.ParamName, "#C6");
			}
#else
			} catch (ArgumentException ex) {
				// '8/14/2007 12:00:00 AM' is not a valid value for 'MinDate'.
				// 'MinDate' must be less than MaxDate
				Assert.AreEqual (typeof (ArgumentException), ex.GetType (), "#C2");
				Assert.IsNull (ex.InnerException, "#C3");
				Assert.IsNotNull (ex.Message, "#C4");
				Assert.IsNull (ex.ParamName, "#C5");
			}
#endif
		}

		[Test]
		public void TextTest ()
		{
			CheckCulture ();

			DateTimePicker dt = new DateTimePicker ();
			EventLogger log = new EventLogger (dt);
			DateTime tomorrow = DateTime.Today.AddDays (1);

			log.Clear ();
			dt.Value = tomorrow;
			Assert.AreEqual ("", dt.Text, "#1");
			Assert.IsTrue (log.ContainsEventsOrdered ("ValueChanged"), "#1.1");

			log.Clear ();
			dt.CreateControl ();
			Assert.AreEqual (tomorrow.ToLongDateString (), dt.Text, "#2");
			Assert.IsTrue (log.ContainsEventsOrdered ("HandleCreated"), "#2.1");

			log.Clear ();
			dt.Text = "";
			Assert.AreEqual (DateTime.Today.ToLongDateString (), dt.Text, "#3");
			Assert.IsTrue (log.ContainsEventsOrdered ("ValueChanged", "TextChanged"), "#3.1");
			
			log.Clear ();
			dt.Text = null;
			Assert.AreEqual (DateTime.Today.ToLongDateString (), dt.Text, "#4");
			Assert.IsTrue (log.ContainsEventsOrdered ("ValueChanged", "TextChanged"), "#4.1, expected ValueChanged+TextChanged, got: " + log.EventsJoined ());

			log.Clear (); 
			dt.CustomFormat = "yyyy-MM-dd HH-mm-ss";
			dt.Format = DateTimePickerFormat.Custom;
			Assert.AreEqual (DateTime.Now.ToString (dt.CustomFormat), dt.Text, "#5");
			Assert.IsTrue (log.ContainsEventsOrdered ("HandleDestroyed", "HandleCreated", "FormatChanged"), "#5.1");

			log.Clear ();
			dt.CustomFormat = "yyyy-MM-dd";
			Assert.AreEqual (DateTime.Now.ToString (dt.CustomFormat), dt.Text, "#6");
			Assert.AreEqual ("", log.EventsJoined (), "#6.1");
			
			log.Clear ();
			dt.Format = DateTimePickerFormat.Short;
			Assert.AreEqual (DateTime.Now.ToShortDateString (), dt.Text, "#7");
			Assert.IsTrue (log.ContainsEventsOrdered ("HandleDestroyed", "HandleCreated", "FormatChanged"), "#7.1");

			dt.Value = DateTime.Today;
			log.Clear ();
			dt.Value = DateTime.Today;
			Assert.AreEqual (0, log.EventsRaised, "#8");

			dt.Text = null;
			log.Clear ();
			dt.Text = null;
			Assert.IsTrue (log.ContainsEventsOrdered ("ValueChanged", "TextChanged"), "#9, expected ValueChanged+TextChanged, got: " + log.EventsJoined ());
			
			dt.Text = "";
			log.Clear ();
			dt.Text = "";
			Assert.IsTrue (log.ContainsEventsOrdered ("ValueChanged", "TextChanged"), "#19, expected ValueChanged+TextChanged, got: " + log.EventsJoined ());
		}
		
		[Test]
		[ExpectedException (typeof (FormatException))]
		public void InvalidTextTest ()
		{
			DateTimePicker dt = new DateTimePicker ();
			dt.Text = "abcdef";
		}

		[Test]
#if NET_2_0
		[ExpectedException (typeof (ArgumentOutOfRangeException))]
#else
		[ExpectedException (typeof (ArgumentException))]
#endif	
		public void ValueTooBig ()
		{
			DateTimePicker dt = new DateTimePicker ();
			dt.MaxDate = DateTime.Now;

			dt.Value = DateTime.Now.AddDays (3);
		}

		[Test]
#if NET_2_0
		[ExpectedException (typeof (ArgumentOutOfRangeException))]
#else
		[ExpectedException (typeof (ArgumentException))]
#endif
		public void ValueTooSmall ()
		{
			DateTimePicker dt = new DateTimePicker ();
			dt.MinDate = DateTime.Now;

			dt.Value = DateTime.Now.AddDays (-3);
		}
	}
}
