/* mhs: A GObject wrapper for the Mozilla Mhs API
 *
 * Copyright (C) 2009  Intel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib-object.h>
#include <nsIThread.h>
#include <nsIThreadInternal.h>
#include <nsThreadUtils.h>

#include "mhs-event-source.h"

struct MhsEventSource
{
  GSource source;

  /* Mozilla's handle to the thread */
  nsCOMPtr<nsIThread> nsthread;
};

static gboolean mhs_event_source_prepare  (GSource *source, gint *timeout);
static gboolean mhs_event_source_check    (GSource *source);
static gboolean mhs_event_source_dispatch (GSource    *source,
                                              GSourceFunc callback,
                                              gpointer    user_data);
static void     mhs_event_source_finalize (GSource *source);

static GSourceFuncs mhs_event_source_funcs =
  {
    mhs_event_source_prepare,
    mhs_event_source_check,
    mhs_event_source_dispatch,
    mhs_event_source_finalize
  };

class MhsEventSourceObserver : public nsIThreadObserver
{
public:
  NS_DECL_ISUPPORTS
  NS_DECL_NSITHREADOBSERVER
};

NS_IMPL_ISUPPORTS1 (MhsEventSourceObserver, nsIThreadObserver)

NS_IMETHODIMP
MhsEventSourceObserver::OnDispatchedEvent (nsIThreadInternal *thread)
{
  /* Events can be queued from any thread so we need to wakeup the
     main loop explicitly */
  g_main_context_wakeup (g_main_context_default ());

  return NS_OK;
}

NS_IMETHODIMP
MhsEventSourceObserver::OnProcessNextEvent (nsIThreadInternal *thread,
                                               PRBool mayWait,
                                               PRUint32 recursionDepth)
{
  return NS_OK;
}

NS_IMETHODIMP
MhsEventSourceObserver::AfterProcessNextEvent (nsIThreadInternal *thread,
                                                  PRUint32 recursionDepth)
{
  return NS_OK;
}

GSource *
mhs_event_source_new (void)
{
  GSource *source = g_source_new (&mhs_event_source_funcs,
				  sizeof (MhsEventSource));
  MhsEventSource *event_source = (MhsEventSource *) source;
  nsresult rv;

  /* Use placement new to initialise the nsCOMPtr */
  new (reinterpret_cast<void *> (source)) MhsEventSource;

  /* Fill in the nsthread pointer */
  rv = NS_GetCurrentThread (getter_AddRefs (event_source->nsthread));

  if (NS_SUCCEEDED (rv))
    {
      /* Register an observer for when events are queued so that we
         can wakeup the main loop */
      nsCOMPtr<nsIThreadInternal> thread_internal
        = do_QueryInterface (event_source->nsthread, &rv);

      if (NS_SUCCEEDED (rv))
        {
          MhsEventSourceObserver *observer = new MhsEventSourceObserver;

          rv = thread_internal->SetObserver (observer);

          if (NS_FAILED (rv))
            delete observer;
        }
    }

  return source;
}

static gboolean
mhs_event_source_prepare (GSource *source, gint *delay)
{
  if (delay)
    *delay = -1;

  return mhs_event_source_check (source);
}

static gboolean
mhs_event_source_check (GSource *source)
{
  MhsEventSource *event_source = (MhsEventSource *) source;
  PRBool pending;
  nsresult rv;

  rv = event_source->nsthread->HasPendingEvents (&pending);

  return NS_SUCCEEDED (rv) && pending;
}

static gboolean
mhs_event_source_dispatch (GSource     *source,
                              GSourceFunc  callback,
                              gpointer     user_data)
{
  MhsEventSource *event_source = (MhsEventSource *) source;

  /* Process all pending events */
  while (mhs_event_source_check (source))
    {
      PRBool processed;
      event_source->nsthread->ProcessNextEvent (PR_TRUE, &processed);
    }

  return TRUE;
}

static void
mhs_event_source_finalize (GSource *source)
{
  /* Explicitly call the destructor so that the nsCOMPtr will be
     destroyed */
  ((MhsEventSource *) source)->~MhsEventSource ();
}
