/*
 * Copyright (C) 2008  Intel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License, version 2.1, as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef DCS_ACCEL_API_H
#define DCS_ACCEL_API_H

#include "DCSCommon.h"
#include "accel/AccelDef.h"

#ifdef __cplusplus
extern "C" {
#endif

/**
 * Typedef of the callback function for acceleration notifications.
 *
 * @param pData		the pointer to the accelerometer data got
 */
typedef void (CALLBACK *DCS_Accel_Callback) (PDCS_Accel_Data pData);

/**
 * Typedef of the callback function for system orientation changes.
 *
 * @param pData		the pointer to the system orientation data got
 */
typedef void (CALLBACK *DCS_SystemOrientation_Callback) (DCS_SystemOrientation *pData);

/**
 * Gets the acceleration value for gravity.
 *
 * @param pValue	the pointer to the gravity value got
 */
DCSAPI DCS_Return_Code DCS_GetAccelForGravity(CHAR *pValue);

/**
 * Gets acceleration data from driver
 *
 * @param pData		the pointer to the accelerometer got from driver
 */
DCSAPI DCS_Return_Code DCS_GetAccelData(PDCS_Accel_Data pData);

/**
 * Registers the callback function to receive acceleration data change.
 *
 * @param pCallback		the pointer to the callback function getting accelerometer data
 */
DCSAPI DCS_Return_Code DCS_RegisterAccelCallback(DCS_Accel_Callback pCallback);

/**
 * Unregisters the callback function to receive acceleration data change.
 */
DCSAPI DCS_Return_Code DCS_UnregisterAccelCallback(void);

/**
 * Registers callback function to be notified when the system orientation is changed.
 *
 * @param pCallback		the pointer to the callback function getting orientation data
 */
DCSAPI DCS_Return_Code DCS_RegisterSystemOrientationCallback(DCS_SystemOrientation_Callback pCallback);

/**
 * Unregisters and stop being notified if the system orientation is changed.
 */
DCSAPI DCS_Return_Code DCS_UnregisterSystemOrientationCallback(void);

/**
 * Enables desktop auto-rotation unconditionally.  Normal application
 * should NOT call this function.  This function will reverse the effect
 * of all previous #DCS_DisableDesktopAutoRotation calls from all
 * clients.  It may be harmful and cause unwanted effects.
 *
 * @see	DCS_UnDisableDesktopAutoRotation
 * @see	DCS_GetDesktopAutoRotationStatus
 */
DCSAPI DCS_Return_Code DCS_EnableDesktopAutoRotation(void);

/**
 * Disables desktop auto-rotation for the current client.  It can be
 * undone by #DCS_UnDisableDesktopAutoRotation.
 */
DCSAPI DCS_Return_Code DCS_DisableDesktopAutoRotation(void);

/**
 * Undoes the previous #DCS_DisableDesktopAutoRotation calls from the
 * current client.  If other clients have disabled desktop
 * auto-rotation, the system desktop auto-rotation status will remain
 * disabled.  Normally, applications should call this function to
 * restore desktop auto-rotation.
 */
DCSAPI DCS_Return_Code DCS_UnDisableDesktopAutoRotation(void);

/**
 * Gets the system enabled/disabled status of desktop auto-rotation.
 * The system orientation change notification event is sent whenever the
 * accelerometer detects a big system orientation change, and the Tablet
 * Screen Controller will check the auto-rotation status to determine
 * whether a desktop rotation should occur.
 *
 * @param pEnabled		the pointer to the value got
 */
DCSAPI DCS_Return_Code DCS_GetDesktopAutoRotationStatus(BOOL *pEnabled);

/**
 * Gets the system orientation.
 *
 * @param pData		the pointer to the orientation data got
 */
DCSAPI DCS_Return_Code DCS_GetSystemOrientation(DCS_SystemOrientation *pData);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* DCS_ACCEL_API_H */
