/*******************************************************************************
**3456789 123456789 123456789 123456789 123456789 123456789 123456789 123456789
**      10        20        30        40        50        60        70        80
**
** notify-osd
**
** button.h - a button class for snap-decisions
**
** Copyright 2012 Canonical Ltd.
**
** Authors:
**    Mirco "MacSlow" Mueller <mirco.mueller@canonical.com>
**
** This program is free software: you can redistribute it and/or modify it
** under the terms of the GNU General Public License version 3, as published
** by the Free Software Foundation.
**
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranties of
** MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
** PURPOSE.  See the GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License along
** with this program.  If not, see <http://www.gnu.org/licenses/>.
**
*******************************************************************************/

#include <pango/pango.h>

#include "bubble.h"
#include "button.h"
#include "json-parser.h"

G_DEFINE_TYPE (Button, button, G_TYPE_OBJECT);

#define GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), BUTTON_TYPE, ButtonPrivate))

struct _ButtonPrivate {
	gint             x;
	gint             y;
	gint             width;
	gint             height;
	GString*         label;
	ButtonState      state;
	gboolean         highlight;
	cairo_surface_t* normal_surface;
	cairo_surface_t* hover_surface;
	cairo_surface_t* pressed_surface;
	gboolean         uses_button_tint;
	SettingsButton*  settings;
};

/*-- internal API ------------------------------------------------------------*/

static void
button_dispose (GObject* gobject)
{
	/* chain up to the parent class */
	G_OBJECT_CLASS (button_parent_class)->dispose (gobject);
}

static void
button_finalize (GObject* gobject)
{
	Button*        button = NULL;
	ButtonPrivate* priv   = NULL;

	// sanity checks
	g_assert (gobject);
	button = BUTTON (gobject);
	g_assert (button);
	g_assert (IS_BUTTON (button));
	priv = GET_PRIVATE (button);
	g_assert (priv);

	if (priv->label != NULL)
		g_string_free (priv->label, TRUE);

	if (priv->normal_surface != NULL)
		cairo_surface_destroy (priv->normal_surface);

	if (priv->hover_surface != NULL)
		cairo_surface_destroy (priv->hover_surface);

	if (priv->pressed_surface != NULL)
		cairo_surface_destroy (priv->pressed_surface);

	/* chain up to the parent class */
	G_OBJECT_CLASS (button_parent_class)->finalize (gobject);
}

static void
button_init (Button* self)
{
	/* If you need specific construction properties to complete
	** initialization, delay initialization completion until the
	** property is set. */
}

static void
button_get_property (GObject*    gobject,
                     guint       prop,
                     GValue*     value,
                     GParamSpec* spec)
{
	switch (prop)
	{
		default :
			G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, prop, spec);
		break;
	}
}

static void
button_class_init (ButtonClass* klass)
{
	GObjectClass* gobject_class = G_OBJECT_CLASS (klass);

	g_type_class_add_private (klass, sizeof (ButtonPrivate));

	gobject_class->dispose      = button_dispose;
	gobject_class->finalize     = button_finalize;
	gobject_class->get_property = button_get_property;
}

void
_refresh_button_surfaces (Button* self)
{
	gdouble  text_normal_color[4]    = {0.0, 0.0, 0.0, 0.0};
	gdouble  text_hover_color[4]     = {0.0, 0.0, 0.0, 0.0};
	gdouble  text_pressed_color[4]   = {0.0, 0.0, 0.0, 0.0};
	gdouble  fill_normal_color[4]    = {0.0, 0.0, 0.0, 0.0};
	gdouble  fill_hover_color[4]     = {0.0, 0.0, 0.0, 0.0};
	gdouble  fill_pressed_color[4]   = {0.0, 0.0, 0.0, 0.0};
	gdouble  stroke_normal_color[4]  = {0.0, 0.0, 0.0, 0.0};
	gdouble  stroke_hover_color[4]   = {0.0, 0.0, 0.0, 0.0};
	gdouble  stroke_pressed_color[4] = {0.0, 0.0, 0.0, 0.0};
	gboolean grid_align              = FALSE;

	g_return_if_fail (self && IS_BUTTON (self));

	ButtonPrivate* priv = GET_PRIVATE (self);

	grid_align = priv->settings->line_width == 1.0;

	if (cairo_surface_status (priv->normal_surface) != CAIRO_STATUS_SUCCESS)
		return;

	if (cairo_surface_status (priv->hover_surface) != CAIRO_STATUS_SUCCESS)
		return;

	if (cairo_surface_status (priv->pressed_surface) != CAIRO_STATUS_SUCCESS)
		return;

	cairo_t* cr = NULL;

	text_normal_color[0]   = (gdouble) priv->settings->text_colors[0].red / (gdouble) 0xFFFF;
	text_normal_color[1]   = (gdouble) priv->settings->text_colors[0].green / (gdouble) 0xFFFF;
	text_normal_color[2]   = (gdouble) priv->settings->text_colors[0].blue / (gdouble) 0xFFFF;
	text_normal_color[3]   = priv->settings->text_opacities[0];
	text_hover_color[0]    = (gdouble) priv->settings->text_colors[1].red / (gdouble) 0xFFFF;
	text_hover_color[1]    = (gdouble) priv->settings->text_colors[1].green / (gdouble) 0xFFFF;
	text_hover_color[2]    = (gdouble) priv->settings->text_colors[1].blue / (gdouble) 0xFFFF;
	text_hover_color[3]    = priv->settings->text_opacities[1];
	text_pressed_color[0]  = (gdouble) priv->settings->text_colors[2].red / (gdouble) 0xFFFF;
	text_pressed_color[1]  = (gdouble) priv->settings->text_colors[2].green / (gdouble) 0xFFFF;
	text_pressed_color[2]  = (gdouble) priv->settings->text_colors[2].blue / (gdouble) 0xFFFF;
	text_pressed_color[3]  = priv->settings->text_opacities[2];

	if (priv->highlight)
	{
		if (priv->uses_button_tint)
		{
			/* use tinted colors for stroke and fill */
			fill_normal_color[0]   = (gdouble) priv->settings->fill_colors_tinted[0].red / (gdouble) 0xFFFF;
			fill_normal_color[1]   = (gdouble) priv->settings->fill_colors_tinted[0].green / (gdouble) 0xFFFF;
			fill_normal_color[2]   = (gdouble) priv->settings->fill_colors_tinted[0].blue / (gdouble) 0xFFFF;
			fill_normal_color[3]   = priv->settings->fill_opacities_tinted[0];
			fill_hover_color[0]     = (gdouble) priv->settings->fill_colors_tinted[1].red / (gdouble) 0xFFFF;
			fill_hover_color[1]     = (gdouble) priv->settings->fill_colors_tinted[1].green / (gdouble) 0xFFFF;
			fill_hover_color[2]     = (gdouble) priv->settings->fill_colors_tinted[1].blue / (gdouble) 0xFFFF;
			fill_hover_color[3]     = priv->settings->fill_opacities_tinted[1];
			fill_pressed_color[0]   = (gdouble) priv->settings->fill_colors_tinted[2].red / (gdouble) 0xFFFF;
			fill_pressed_color[1]   = (gdouble) priv->settings->fill_colors_tinted[2].green / (gdouble) 0xFFFF;
			fill_pressed_color[2]   = (gdouble) priv->settings->fill_colors_tinted[2].blue / (gdouble) 0xFFFF;
			fill_pressed_color[3]   = priv->settings->fill_opacities_tinted[2];
			stroke_normal_color[0]  = (gdouble) priv->settings->outline_colors_tinted[0].red / (gdouble) 0xFFFF;
			stroke_normal_color[1]  = (gdouble) priv->settings->outline_colors_tinted[0].green / (gdouble) 0xFFFF;
			stroke_normal_color[2]  = (gdouble) priv->settings->outline_colors_tinted[0].blue / (gdouble) 0xFFFF;
			stroke_normal_color[3]  = priv->settings->outline_opacities_tinted[0];
			stroke_hover_color[0]   = (gdouble) priv->settings->outline_colors_tinted[1].red / (gdouble) 0xFFFF;
			stroke_hover_color[1]   = (gdouble) priv->settings->outline_colors_tinted[1].green / (gdouble) 0xFFFF;
			stroke_hover_color[2]   = (gdouble) priv->settings->outline_colors_tinted[1].blue / (gdouble) 0xFFFF;
			stroke_hover_color[3]   = priv->settings->outline_opacities_tinted[1];
			stroke_pressed_color[0] = (gdouble) priv->settings->outline_colors_tinted[2].red / (gdouble) 0xFFFF;
			stroke_pressed_color[1] = (gdouble) priv->settings->outline_colors_tinted[2].green / (gdouble) 0xFFFF;
			stroke_pressed_color[2] = (gdouble) priv->settings->outline_colors_tinted[2].blue / (gdouble) 0xFFFF;
			stroke_pressed_color[3] = priv->settings->outline_opacities_tinted[2];
		}
		else
		{
			/* use regular highlight colors for stroke and fill */
			fill_normal_color[0]   = (gdouble) priv->settings->fill_colors_lit[0].red / (gdouble) 0xFFFF;
			fill_normal_color[1]   = (gdouble) priv->settings->fill_colors_lit[0].green / (gdouble) 0xFFFF;
			fill_normal_color[2]   = (gdouble) priv->settings->fill_colors_lit[0].blue / (gdouble) 0xFFFF;
			fill_normal_color[3]   = priv->settings->fill_opacities_lit[0];
			fill_hover_color[0]     = (gdouble) priv->settings->fill_colors_lit[1].red / (gdouble) 0xFFFF;
			fill_hover_color[1]     = (gdouble) priv->settings->fill_colors_lit[1].green / (gdouble) 0xFFFF;
			fill_hover_color[2]     = (gdouble) priv->settings->fill_colors_lit[1].blue / (gdouble) 0xFFFF;
			fill_hover_color[3]     = priv->settings->fill_opacities_lit[1];
			fill_pressed_color[0]   = (gdouble) priv->settings->fill_colors_lit[2].red / (gdouble) 0xFFFF;
			fill_pressed_color[1]   = (gdouble) priv->settings->fill_colors_lit[2].green / (gdouble) 0xFFFF;
			fill_pressed_color[2]   = (gdouble) priv->settings->fill_colors_lit[2].blue / (gdouble) 0xFFFF;
			fill_pressed_color[3]   = priv->settings->fill_opacities_lit[2];
			stroke_normal_color[0]  = (gdouble) priv->settings->outline_colors_lit[0].red / (gdouble) 0xFFFF;
			stroke_normal_color[1]  = (gdouble) priv->settings->outline_colors_lit[0].green / (gdouble) 0xFFFF;
			stroke_normal_color[2]  = (gdouble) priv->settings->outline_colors_lit[0].blue / (gdouble) 0xFFFF;
			stroke_normal_color[3]  = priv->settings->outline_opacities_lit[0];
			stroke_hover_color[0]   = (gdouble) priv->settings->outline_colors_lit[1].red / (gdouble) 0xFFFF;
			stroke_hover_color[1]   = (gdouble) priv->settings->outline_colors_lit[1].green / (gdouble) 0xFFFF;
			stroke_hover_color[2]   = (gdouble) priv->settings->outline_colors_lit[1].blue / (gdouble) 0xFFFF;
			stroke_hover_color[3]   = priv->settings->outline_opacities_lit[1];
			stroke_pressed_color[0] = (gdouble) priv->settings->outline_colors_lit[2].red / (gdouble) 0xFFFF;
			stroke_pressed_color[1] = (gdouble) priv->settings->outline_colors_lit[2].green / (gdouble) 0xFFFF;
			stroke_pressed_color[2] = (gdouble) priv->settings->outline_colors_lit[2].blue / (gdouble) 0xFFFF;
			stroke_pressed_color[3] = priv->settings->outline_opacities_lit[2];
		}
	}
	else
	{
		/* use clear colors for stroke and fill */
		fill_normal_color[0]   = (gdouble) priv->settings->fill_colors[0].red / (gdouble) 0xFFFF;
		fill_normal_color[1]   = (gdouble) priv->settings->fill_colors[0].green / (gdouble) 0xFFFF;
		fill_normal_color[2]   = (gdouble) priv->settings->fill_colors[0].blue / (gdouble) 0xFFFF;
		fill_normal_color[3]   = priv->settings->fill_opacities[0];
		fill_hover_color[0]     = (gdouble) priv->settings->fill_colors[1].red / (gdouble) 0xFFFF;
		fill_hover_color[1]     = (gdouble) priv->settings->fill_colors[1].green / (gdouble) 0xFFFF;
		fill_hover_color[2]     = (gdouble) priv->settings->fill_colors[1].blue / (gdouble) 0xFFFF;
		fill_hover_color[3]     = priv->settings->fill_opacities[1];
		fill_pressed_color[0]   = (gdouble) priv->settings->fill_colors[2].red / (gdouble) 0xFFFF;
		fill_pressed_color[1]   = (gdouble) priv->settings->fill_colors[2].green / (gdouble) 0xFFFF;
		fill_pressed_color[2]   = (gdouble) priv->settings->fill_colors[2].blue / (gdouble) 0xFFFF;
		fill_pressed_color[3]   = priv->settings->fill_opacities[2];
		stroke_normal_color[0]  = (gdouble) priv->settings->outline_colors[0].red / (gdouble) 0xFFFF;
		stroke_normal_color[1]  = (gdouble) priv->settings->outline_colors[0].green / (gdouble) 0xFFFF;
		stroke_normal_color[2]  = (gdouble) priv->settings->outline_colors[0].blue / (gdouble) 0xFFFF;
		stroke_normal_color[3]  = priv->settings->outline_opacities[0];
		stroke_hover_color[0]   = (gdouble) priv->settings->outline_colors[1].red / (gdouble) 0xFFFF;
		stroke_hover_color[1]   = (gdouble) priv->settings->outline_colors[1].green / (gdouble) 0xFFFF;
		stroke_hover_color[2]   = (gdouble) priv->settings->outline_colors[1].blue / (gdouble) 0xFFFF;
		stroke_hover_color[3]   = priv->settings->outline_opacities[1];
		stroke_pressed_color[0] = (gdouble) priv->settings->outline_colors[2].red / (gdouble) 0xFFFF;
		stroke_pressed_color[1] = (gdouble) priv->settings->outline_colors[2].green / (gdouble) 0xFFFF;
		stroke_pressed_color[2] = (gdouble) priv->settings->outline_colors[2].blue / (gdouble) 0xFFFF;
		stroke_pressed_color[3] = priv->settings->outline_opacities[2];
	}

	PangoWeight font_weight = PANGO_WEIGHT_NORMAL;

	if (priv->settings->text_weight == FONT_WEIGHT_LIGHT)
		font_weight = PANGO_WEIGHT_MEDIUM;
	else if (priv->settings->text_weight == FONT_WEIGHT_NORMAL)
		font_weight = PANGO_WEIGHT_NORMAL;
	else if (priv->settings->text_weight == FONT_WEIGHT_BOLD)
		font_weight = PANGO_WEIGHT_BOLD;

	/* normal-state surface - fill */
	cr = cairo_create (priv->normal_surface);
	cairo_scale (cr, 1.0, 1.0);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	draw_round_rect (cr,
	                 1.0,
	                 align (0.0, FALSE),
	                 align (0.0, FALSE),
	                 priv->settings->radius,
	                 align (priv->width, FALSE),
	                 align (priv->height, FALSE));
	cairo_set_source_rgba (cr,
	                       fill_normal_color[0],
	                       fill_normal_color[1],
	                       fill_normal_color[2],
	                       fill_normal_color[3]);
	cairo_fill (cr);

	/* normal-state surface - outline */
	cairo_set_line_width (cr, priv->settings->line_width);
	draw_round_rect (cr,
	                 1.0,
	                 align (0.0, grid_align),
	                 align (0.0, grid_align),
	                 priv->settings->radius,
	                 align (priv->width, grid_align) - 3.0 * align (0.0, grid_align),
	                 align (priv->height, grid_align) - 3.0 * align (0.0, grid_align));
	cairo_set_source_rgba (cr,
	                       stroke_normal_color[0],
	                       stroke_normal_color[1],
	                       stroke_normal_color[2],
	                       stroke_normal_color[3]);
	cairo_stroke (cr);

	/* normal-state surface - text */
	draw_button_label (cr,
	                   priv->label->str,
	                   0,
	                   0,
	                   priv->width,
	                   priv->height,
	                   priv->settings->text_margin_horiz,
	                   priv->settings->text_family,
	                   priv->settings->text_size,
	                   font_weight,
	           text_normal_color);
	cairo_destroy (cr);

	/* hover-state surface - fill */
	cr = cairo_create (priv->hover_surface);
	cairo_scale (cr, 1.0, 1.0);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	draw_round_rect (cr,
	                 1.0,
	                 align (0.0, FALSE),
	                 align (0.0, FALSE),
	                 priv->settings->radius,
	                 align (priv->width, FALSE),
	                 align (priv->height, FALSE));
	cairo_set_source_rgba (cr,
	                       fill_hover_color[0],
	                       fill_hover_color[1],
	                       fill_hover_color[2],
	                       fill_hover_color[3]);
	cairo_fill (cr);

	/* hover-state surface - outline */
	cairo_set_line_width (cr, priv->settings->line_width);
	draw_round_rect (cr,
	                 1.0,
	                 align (0.0, grid_align),
	                 align (0.0, grid_align),
	                 priv->settings->radius,
	                 align (priv->width, grid_align) - 3.0 * align (0.0, grid_align),
	                 align (priv->height, grid_align) - 3.0 * align (0.0, grid_align));
	cairo_set_source_rgba (cr,
	                       stroke_hover_color[0],
	                       stroke_hover_color[1],
	                       stroke_hover_color[2],
	                       stroke_hover_color[3]);
	cairo_stroke (cr);

	/* hover-state surface - text */
	draw_button_label (cr,
	                   priv->label->str,
	                   0,
	                   0,
	                   priv->width,
	                   priv->height,
	                   priv->settings->text_margin_horiz,
	                   priv->settings->text_family,
	                   priv->settings->text_size,
	                   font_weight,
	                   text_hover_color);
	cairo_destroy (cr);

	/* pressed-state surface - fill */
	cr = cairo_create (priv->pressed_surface);
	cairo_scale (cr, 1.0, 1.0);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	draw_round_rect (cr,
	                 1.0,
	                 align (0.0, FALSE),
	                 align (0.0, FALSE),
	                 priv->settings->radius,
	                 align (priv->width, FALSE),
	                 align (priv->height, FALSE));
	cairo_set_source_rgba (cr,
	                       fill_pressed_color[0],
	                       fill_pressed_color[1],
	                       fill_pressed_color[2],
	                       fill_pressed_color[3]);
	cairo_fill (cr);

	/* pressed-state surface - outline */
	cairo_set_line_width (cr, priv->settings->line_width);
	draw_round_rect (cr,
	                 1.0,
	                 align (0.0, grid_align),
	                 align (0.0, grid_align),
	                 priv->settings->radius,
	                 align (priv->width, grid_align) - 3.0 * align (0.0, grid_align),
	                 align (priv->height, grid_align) - 3.0 * align (0.0, grid_align));
	cairo_set_source_rgba (cr,
	                       stroke_pressed_color[0],
	                       stroke_pressed_color[1],
	                       stroke_pressed_color[2],
	                       stroke_pressed_color[3]);
	cairo_stroke (cr);

	/* pressed-state surface - text*/
	draw_button_label (cr,
	                   priv->label->str,
	                   0,
	                   0,
	                   priv->width,
	                   priv->height,
	                   priv->settings->text_margin_horiz,
	                   priv->settings->text_family,
	                   priv->settings->text_size,
	                   font_weight,
	                   text_pressed_color);
	cairo_destroy (cr);
}

/*-- public API --------------------------------------------------------------*/

Button*
button_new (gint            x,
            gint            y,
            gint            width,
            gint            height,
            gboolean        highlight,
            SettingsButton* settings,
            const gchar*    label)
{
	Button*        self = NULL;
	ButtonPrivate* priv = NULL;

	g_return_val_if_fail (width != 0 && height != 0, NULL);

	self = g_object_new (BUTTON_TYPE, NULL);
	if (!self)
		return NULL;

	priv = GET_PRIVATE (self);

	priv->x               = x;
	priv->y               = y;
	priv->width           = width;
	priv->height          = height;
	priv->label           = g_string_new (label);
	priv->state           = BUTTON_STATE_NORMAL;
	priv->highlight       = highlight;
	priv->normal_surface  = cairo_image_surface_create (CAIRO_FORMAT_ARGB32,
	                                                    width,
	                                                    height);
	priv->hover_surface   = cairo_image_surface_create (CAIRO_FORMAT_ARGB32,
	                                                    width,
	                                                    height);
	priv->pressed_surface = cairo_image_surface_create (CAIRO_FORMAT_ARGB32,
	                                                    width,
	                                                    height);
	priv->uses_button_tint = FALSE;
	priv->settings         = settings;

	_refresh_button_surfaces (self);

	return self;
}

void
button_del (Button* self)
{
	g_return_if_fail (self && IS_BUTTON (self));
	g_object_unref (self);
}

void
button_set_position (Button* self,
                     gint    x,
                     gint    y)
{
	g_return_if_fail (self && IS_BUTTON (self));

	ButtonPrivate* priv = GET_PRIVATE (self);

	priv->x = x;
	priv->y = y;
}

void
button_get_position (Button* self,
                     gint*   x,
                     gint*   y)
{
	g_return_if_fail (self && IS_BUTTON (self) && x && y);

	ButtonPrivate* priv = GET_PRIVATE (self);

	*x = priv->x;
	*y = priv->y;
}

gboolean
button_is_hit (Button* self,
               gint    x,
               gint    y)
{
	g_return_val_if_fail (self && IS_BUTTON (self), FALSE);

	gboolean              result = FALSE;
	ButtonPrivate*        priv   = NULL;
	cairo_rectangle_int_t rect   = {0, 0, 0, 0};
	cairo_region_t*       region = NULL;

	priv = GET_PRIVATE (self);

	rect.x      = priv->x;
	rect.y      = priv->y;
	rect.width  = priv->width;
	rect.height = priv->height;

	region = cairo_region_create_rectangle (&rect);
	result = cairo_region_contains_point (region, x, y);
	cairo_region_destroy (region);

	return result;
}

void
button_set_state (Button*     self,
                  ButtonState state)
{
	g_return_if_fail (self && IS_BUTTON (self) && state != BUTTON_STATE_NONE);

	GET_PRIVATE (self)->state = state;
}

ButtonState
button_get_state (Button* self)
{
	g_return_val_if_fail (self && IS_BUTTON (self), BUTTON_STATE_NONE);

	return GET_PRIVATE (self)->state;
}

void
button_set_button_tint (Button*  self,
                        gboolean uses_tint)
{
	g_return_if_fail (self && IS_BUTTON (self));

	GET_PRIVATE (self)->uses_button_tint = uses_tint;
	_refresh_button_surfaces (self);
}

gboolean
button_get_button_tint (Button* self)
{
	g_return_val_if_fail (self && IS_BUTTON (self), FALSE);

	return GET_PRIVATE (self)->uses_button_tint;
}

void
button_paint (Button*  self,
              cairo_t* cr)
{
	g_return_if_fail (self && IS_BUTTON (self) && cr);

	ButtonPrivate* priv = GET_PRIVATE (self);

	if (priv->state == BUTTON_STATE_NORMAL)
		cairo_set_source_surface (cr, priv->normal_surface, priv->x, priv->y);
	if (priv->state == BUTTON_STATE_HOVER)
		cairo_set_source_surface (cr, priv->hover_surface, priv->x, priv->y);
	if (priv->state == BUTTON_STATE_PRESSED)
		cairo_set_source_surface (cr, priv->pressed_surface, priv->x, priv->y);

	cairo_paint (cr);
}
