/*******************************************************************************
**3456789 123456789 123456789 123456789 123456789 123456789 123456789 123456789
**      10        20        30        40        50        60        70        80
**
** notify-osd
**
** close-button.h - a special close-button for snap-decisions
**
** Copyright 2012 Canonical Ltd.
**
** Authors:
**    Mirco "MacSlow" Mueller <mirco.mueller@canonical.com>
**
** This program is free software: you can redistribute it and/or modify it
** under the terms of the GNU General Public License version 3, as published
** by the Free Software Foundation.
**
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranties of
** MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
** PURPOSE.  See the GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License along
** with this program.  If not, see <http://www.gnu.org/licenses/>.
**
*******************************************************************************/

#include <pango/pango.h>

#include "bubble.h"
#include "close-button.h"
#include "json-parser.h"

G_DEFINE_TYPE (CloseButton, close_button, G_TYPE_OBJECT);

#define GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), CLOSE_BUTTON_TYPE, CloseButtonPrivate))

struct _CloseButtonPrivate {
	gint                 x;
	gint                 y;
	CloseButtonState     state;
	cairo_surface_t*     normal_surface;
	cairo_surface_t*     hover_surface;
	cairo_surface_t*     pressed_surface;
	SettingsCloseButton* settings;
};

/*-- internal API ------------------------------------------------------------*/

static void
close_button_dispose (GObject* gobject)
{
	/* chain up to the parent class */
	G_OBJECT_CLASS (close_button_parent_class)->dispose (gobject);
}

static void
close_button_finalize (GObject* gobject)
{
	CloseButton*        close_button = NULL;
	CloseButtonPrivate* priv         = NULL;

	// sanity checks
	g_assert (gobject);
	close_button = CLOSE_BUTTON (gobject);
	g_assert (close_button);
	g_assert (IS_CLOSE_BUTTON (close_button));
	priv = GET_PRIVATE (close_button);
	g_assert (priv);

	if (priv->normal_surface != NULL)
		cairo_surface_destroy (priv->normal_surface);

	if (priv->hover_surface != NULL)
		cairo_surface_destroy (priv->hover_surface);

	if (priv->pressed_surface != NULL)
		cairo_surface_destroy (priv->pressed_surface);

	/* chain up to the parent class */
	G_OBJECT_CLASS (close_button_parent_class)->finalize (gobject);
}

static void
close_button_init (CloseButton* self)
{
	/* If you need specific construction properties to complete
	** initialization, delay initialization completion until the
	** property is set. */
}

static void
close_button_get_property (GObject*    gobject,
                           guint       prop,
                           GValue*     value,
                           GParamSpec* spec)
{
	switch (prop)
	{
		default :
			G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, prop, spec);
		break;
	}
}

static void
close_button_class_init (CloseButtonClass* klass)
{
	GObjectClass* gobject_class = G_OBJECT_CLASS (klass);

	g_type_class_add_private (klass, sizeof (CloseButtonPrivate));

	gobject_class->dispose      = close_button_dispose;
	gobject_class->finalize     = close_button_finalize;
	gobject_class->get_property = close_button_get_property;
}

void
_draw (cairo_t* cr,
       gdouble  size,
       gdouble* fill_color,
       gdouble* stroke_color,
       gint     outline_thickness)
{
	cairo_scale (cr, 1.0, 1.0);
	cairo_set_operator (cr, CAIRO_OPERATOR_CLEAR);
	cairo_paint (cr);
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	cairo_arc (cr,
	           size / 2.0 + outline_thickness / 2.0,
	           size / 2.0 + outline_thickness / 2.0,
	           size / 2.0,
	           0.0,
	           G_PI / 180.0 * 360.0);
	cairo_set_source_rgba (cr,
	                       fill_color[0],
	                       fill_color[1],
	                       fill_color[2],
	                       fill_color[3]);
	cairo_fill_preserve (cr);
	cairo_set_source_rgba (cr,
	                       stroke_color[0],
	                       stroke_color[1],
	                       stroke_color[2],
	                       stroke_color[3]);
	cairo_set_line_width (cr, outline_thickness);
	cairo_stroke (cr);
	gdouble x = size / 2.0 + outline_thickness / 2.0;
	gdouble y = size / 2.0 + outline_thickness / 2.0;
	gdouble step = size / 4.0;
	cairo_move_to (cr, x - step, y - step);
	cairo_line_to (cr, x + step, y + step);
	cairo_move_to (cr, x - step, y + step);
	cairo_line_to (cr, x + step, y - step);
	cairo_stroke (cr);
}

void
_refresh_close_button_surfaces (CloseButton* self)
{
	gdouble  fill_normal_color[4]     = {0.0, 0.0, 0.0, 0.0};
	gdouble  fill_hover_color[4]      = {0.0, 0.0, 0.0, 0.0};
	gdouble  fill_pressed_color[4]    = {0.0, 0.0, 0.0, 0.0};
	gdouble  outline_normal_color[4]  = {0.0, 0.0, 0.0, 0.0};
	gdouble  outline_hover_color[4]   = {0.0, 0.0, 0.0, 0.0};
	gdouble  outline_pressed_color[4] = {0.0, 0.0, 0.0, 0.0};

	g_return_if_fail (self && IS_CLOSE_BUTTON (self));

	CloseButtonPrivate* priv = GET_PRIVATE (self);

	if (cairo_surface_status (priv->normal_surface) != CAIRO_STATUS_SUCCESS)
		return;

	if (cairo_surface_status (priv->hover_surface) != CAIRO_STATUS_SUCCESS)
		return;

	if (cairo_surface_status (priv->pressed_surface) != CAIRO_STATUS_SUCCESS)
		return;

	cairo_t* cr = NULL;

	setup_color (fill_normal_color,
	             &priv->settings->fill_colors[0],
	             priv->settings->fill_opacities[0]);
	setup_color (fill_hover_color,
	             &priv->settings->fill_colors[1],
	             priv->settings->fill_opacities[1]);
	setup_color (fill_pressed_color,
	             &priv->settings->fill_colors[2],
	             priv->settings->fill_opacities[2]);
	setup_color (outline_normal_color,
	             &priv->settings->outline_colors[0],
	             priv->settings->outline_opacities[0]);
	setup_color (outline_hover_color,
	             &priv->settings->outline_colors[1],
	             priv->settings->outline_opacities[1]);
	setup_color (outline_pressed_color,
	             &priv->settings->outline_colors[2],
	             priv->settings->outline_opacities[2]);

	/* normal-state surface */
	cr = cairo_create (priv->normal_surface);
	_draw (cr,
	       priv->settings->size,
	       fill_normal_color,
	       outline_normal_color,
	       priv->settings->outline_thickness);
	cairo_destroy (cr);
	cr = NULL;

	/* hover-state surface */
	cr = cairo_create (priv->hover_surface);
	_draw (cr,
	       priv->settings->size,
	       fill_hover_color,
	       outline_hover_color,
	       priv->settings->outline_thickness);
	cairo_destroy (cr);
	cr = NULL;

	/* pressed-state surface */
	cr = cairo_create (priv->pressed_surface);
	_draw (cr,
	       priv->settings->size,
	       fill_pressed_color,
	       outline_pressed_color,
	       priv->settings->outline_thickness);
	cairo_destroy (cr);
}

/*-- public API --------------------------------------------------------------*/

CloseButton*
close_button_new (SettingsCloseButton* settings)
{
	CloseButton*        self = NULL;
	CloseButtonPrivate* priv = NULL;

	g_return_val_if_fail (settings, NULL);

	self = g_object_new (CLOSE_BUTTON_TYPE, NULL);
	if (!self)
		return NULL;

	priv = GET_PRIVATE (self);

	gint size = settings->size + 2 * settings->outline_thickness;
	priv->x               = settings->position_x;
	priv->y               = settings->position_y;
	priv->state           = CLOSE_BUTTON_STATE_NORMAL;
	priv->normal_surface  = cairo_image_surface_create (CAIRO_FORMAT_ARGB32,
	                                                    size,
	                                                    size);
	priv->hover_surface   = cairo_image_surface_create (CAIRO_FORMAT_ARGB32,
	                                                    size,
	                                                    size);
	priv->pressed_surface = cairo_image_surface_create (CAIRO_FORMAT_ARGB32,
	                                                    size,
	                                                    size);
	priv->settings         = settings;

	_refresh_close_button_surfaces (self);

	return self;
}

void
close_button_del (CloseButton* self)
{
	g_return_if_fail (self && IS_CLOSE_BUTTON (self));
	g_object_unref (self);
}

void
close_button_set_position (CloseButton* self,
                           gint         x,
                           gint         y)
{
	g_return_if_fail (self && IS_CLOSE_BUTTON (self));

	CloseButtonPrivate* priv = GET_PRIVATE (self);

	priv->x = x;
	priv->y = y;
}

void
close_button_get_position (CloseButton* self,
                           gint*        x,
                           gint*        y)
{
	g_return_if_fail (self && IS_CLOSE_BUTTON (self) && x && y);

	CloseButtonPrivate* priv = GET_PRIVATE (self);

	*x = priv->x;
	*y = priv->y;
}

gboolean
close_button_is_hit (CloseButton* self,
                     gint         x,
                     gint         y)
{
	g_return_val_if_fail (self && IS_CLOSE_BUTTON (self), FALSE);

	gboolean              result = FALSE;
	CloseButtonPrivate*   priv   = NULL;
	cairo_rectangle_int_t rect   = {0, 0, 0, 0};
	cairo_region_t*       region = NULL;

	priv = GET_PRIVATE (self);

	rect.x      = priv->x;
	rect.y      = priv->y;
	rect.width  = priv->settings->size;
	rect.height = priv->settings->size;

	region = cairo_region_create_rectangle (&rect);
	result = cairo_region_contains_point (region, x, y);
	cairo_region_destroy (region);

	return result;
}

void
close_button_set_state (CloseButton*     self,
                        CloseButtonState state)
{
	g_return_if_fail (self && IS_CLOSE_BUTTON (self) && state != CLOSE_BUTTON_STATE_NONE);

	GET_PRIVATE (self)->state = state;
}

CloseButtonState
close_button_get_state (CloseButton* self)
{
	g_return_val_if_fail (self && IS_CLOSE_BUTTON (self),
	                      CLOSE_BUTTON_STATE_NONE);

	return GET_PRIVATE (self)->state;
}

void
close_button_paint (CloseButton*  self,
                    cairo_t*      cr)
{
	g_return_if_fail (self && IS_CLOSE_BUTTON (self) && cr);

	CloseButtonPrivate* priv = GET_PRIVATE (self);

	if (priv->state == CLOSE_BUTTON_STATE_NORMAL)
		cairo_set_source_surface (cr, priv->normal_surface, priv->x, priv->y);
	if (priv->state == CLOSE_BUTTON_STATE_HOVER)
		cairo_set_source_surface (cr, priv->hover_surface, priv->x, priv->y);
	if (priv->state == CLOSE_BUTTON_STATE_PRESSED)
		cairo_set_source_surface (cr, priv->pressed_surface, priv->x, priv->y);

	cairo_paint (cr);
}
