////////////////////////////////////////////////////////////////////////////////
//3456789 123456789 123456789 123456789 123456789 123456789 123456789 123456789
//      10        20        30        40        50        60        70        80
//
// notify-osd
//
// notification.c - notification object storing attributes like title- and body-
//                  text, value, icon, id, sender-pid etc.
//
// Copyright 2009 Canonical Ltd.
//
// Authors:
//    Mirco "MacSlow" Mueller <mirco.mueller@canonical.com>
//
// This program is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License version 3, as published
// by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranties of
// MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
// PURPOSE.  See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this program.  If not, see <http://www.gnu.org/licenses/>.
//
////////////////////////////////////////////////////////////////////////////////

#include <gtk/gtk.h>

#include "notification.h"

G_DEFINE_TYPE (Notification, notification, G_TYPE_OBJECT);

enum
{
	PROP_DUMMY = 0,
	PROP_ID,
	PROP_SUMMARY,
	PROP_BODY,
	PROP_VALUE,
	PROP_ICON_PIXBUF,
	PROP_ONSCREEN_TIME,
	PROP_DBUS_SENDER,
	PROP_SENDER_NAME,
	PROP_SENDER_PID,
	PROP_TIMESTAMP,
	PROP_URGENCY
};

#define GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), NOTIFICATION_TYPE, NotificationPrivate))

struct _NotificationPrivate {
	gint       id;                  // unique notification-id
	GString*   summary;             // summary-text strdup'ed from setter
	GString*   body;                // body-text strdup'ed from setter
	gint       value;               // range: 0..100, special: -1, 101
	GdkPixbuf* icon_pixbuf;         // from setter memcpy'ed pixbuf 
	gint       onscreen_time;       // time on-screen in ms
	GString*   dbus_sender;         // DBus sender-name
	GString*   sender_name;         // app-name, strdup'ed from setter
	gint       sender_pid;          // pid of sending application
	GTimeVal   timestamp;           // timestamp of  reception
	Urgency    urgency;             // urgency-level: low, normal, high
	gchar**    actions;             // list of labels for actions/buttons
	guint      num_actions;         // number of labels
	gboolean   visible;             // flag to indicate, that it's shown to user
	gboolean   uses_button_tint;    // flag to indicate if/if not to use tint
};

#define RETURN_GCHAR(n, string)\
	NotificationPrivate* priv;\
	g_return_val_if_fail (IS_NOTIFICATION (n), NULL);\
	priv = GET_PRIVATE (n);\
	if (!priv->string)\
		return NULL;\
	return GET_PRIVATE (n)->string->str;

#define SET_GCHAR(n, string)\
	NotificationPrivate* priv;\
	g_assert (IS_NOTIFICATION (n));\
	if (!string)\
		return;\
	priv = GET_PRIVATE (n);\
	if (priv->string)\
	{\
		g_string_free (priv->string, TRUE);\
		priv->string = NULL;\
	}\
	priv->string = g_string_new (string);

//-- private functions ---------------------------------------------------------

GdkPixbuf*
_notification_load_icon (const gchar* filename)
{
	GdkPixbuf*    buffer = NULL;
	GdkPixbuf*    pixbuf = NULL;
	GtkIconTheme* theme  = NULL; /* not ref'ed, doesn't need to be freed */
	GError*       error  = NULL;

	/* sanity check */
	g_return_val_if_fail (filename, NULL);

	/* Images referenced must always be local files. */
	if (!strncmp (filename, "file://", 7))
		filename += 7;

	if (filename[0] == '/')
	{
		/* load image into pixbuf */
		pixbuf = gdk_pixbuf_new_from_file (filename, &error);
	}
	else
	{
		/* TODO: rewrite, check for SVG support, raise apport
		** notification for low-res icons */
		theme = gtk_icon_theme_get_default ();
		buffer = gtk_icon_theme_load_icon (theme,
		                                   filename,
		                                   50,
		                                   GTK_ICON_LOOKUP_FORCE_SVG |
		                                   GTK_ICON_LOOKUP_GENERIC_FALLBACK |
		                                   GTK_ICON_LOOKUP_FORCE_SIZE,
		                                   &error);
	}

	if (error)
	{
		g_print ("%s() - loading icon '%s' caused error: '%s'\n",
		         G_STRFUNC,
		         filename,
		         error->message);
		g_error_free (error);
		error = NULL;
		pixbuf = NULL;
	}
	else
	{
		/* copy and unref buffer so on an icon-theme change old
		** icons are not kept in memory due to dangling
		** references, this also makes sure we do not need to
		** connect to GtkWidget::style-set signal for the
		** GdkPixbuf we get from gtk_icon_theme_load_icon() */
		pixbuf = gdk_pixbuf_copy (buffer);
		g_object_unref (buffer);
	}

	return pixbuf;
}

//-- internal functions --------------------------------------------------------

// this is what gets called if one does g_object_unref(bla)
static void
notification_dispose (GObject* gobject)
{
	Notification*        n;
	NotificationPrivate* priv;

	// sanity checks
	g_assert (gobject);
	n = NOTIFICATION (gobject);
	g_assert (n);
	g_assert (IS_NOTIFICATION (n));
	priv = GET_PRIVATE (n);
	g_assert (priv);

	// free any allocated resources
	if (priv->summary)
	{
		g_string_free (priv->summary, TRUE);
		priv->summary = NULL;
	}

	if (priv->body)
	{
		g_string_free (priv->body, TRUE);
		priv->body = NULL;
	}

	if (priv->icon_pixbuf)
	{
		g_object_unref (priv->icon_pixbuf);
	}

	if (priv->dbus_sender)
	{
		g_string_free (priv->dbus_sender, TRUE);
		priv->dbus_sender = NULL;
	}

	if (priv->sender_name)
	{
		g_string_free (priv->sender_name, TRUE);
		priv->sender_name = NULL;
	}

	if (priv->num_actions > 0)
	{
		for (guint i = 0; i < priv->num_actions; i++)
			g_free (priv->actions[i]);
		g_free (priv->actions);
		priv->actions = NULL;
		priv->num_actions = 0;
	}

	// chain up to the parent class
	G_OBJECT_CLASS (notification_parent_class)->dispose (gobject);
}

static void
notification_finalize (GObject* gobject)
{
	// chain up to the parent class
	G_OBJECT_CLASS (notification_parent_class)->finalize (gobject);
}

static void
notification_init (Notification* n)
{
	// nothing to be done here for now
}

static void
notification_get_property (GObject*    gobject,
			   guint       prop,
			   GValue*     value,
			   GParamSpec* spec)
{
	Notification* n = NOTIFICATION (gobject);

	// sanity checks are done in public getters

	switch (prop)
	{
		case PROP_ID:
			g_value_set_int (value, notification_get_id (n));
		break;

		case PROP_SUMMARY:
			g_value_set_string (value, notification_get_summary (n));
		break;

		case PROP_BODY:
			g_value_set_string (value, notification_get_body (n));
		break;

		case PROP_VALUE:
			g_value_set_int (value, notification_get_value (n));
		break;

		case PROP_ICON_PIXBUF:
			g_value_set_pointer (value, notification_get_icon_pixbuf (n));
		break;

		case PROP_ONSCREEN_TIME:
			g_value_set_int (value, notification_get_onscreen_time (n));
		break;

		case PROP_DBUS_SENDER:
			g_value_set_string (value, notification_get_dbus_sender (n));
		break;

		case PROP_SENDER_NAME:
			g_value_set_string (value, notification_get_sender_name (n));
		break;

		case PROP_SENDER_PID:
			g_value_set_int (value, notification_get_sender_pid (n));
		break;

		case PROP_TIMESTAMP:
			g_value_set_pointer (value, notification_get_timestamp (n));
		break;

		case PROP_URGENCY:
			g_value_set_int (value, notification_get_urgency (n));
		break;

		default :
			G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, prop, spec);
		break;
	}
}

static void
notification_set_property (GObject*      gobject,
			   guint         prop,
			   const GValue* value,
			   GParamSpec*   spec)
{
	Notification* n = NOTIFICATION (gobject);

	// sanity checks are done in the public setters

	switch (prop)
	{
		case PROP_ID:
			notification_set_id (n, g_value_get_int (value));
		break;

		case PROP_SUMMARY:
			notification_set_summary (n, g_value_get_string (value));
		break;

		case PROP_BODY:
			notification_set_body (n, g_value_get_string (value));
		break;

		case PROP_VALUE:
			notification_set_value (n, g_value_get_int (value));
		break;

		case PROP_ICON_PIXBUF:
			notification_set_icon_from_pixbuf (n, g_value_get_pointer (value));
		break;

		case PROP_ONSCREEN_TIME:
			notification_set_onscreen_time (n, g_value_get_int (value));
		break;

		case PROP_DBUS_SENDER:
			notification_set_dbus_sender (n, g_value_get_string (value));
		break;

		case PROP_SENDER_NAME:
			notification_set_sender_name (n, g_value_get_string (value));
		break;

		case PROP_SENDER_PID:
			notification_set_sender_pid (n, g_value_get_int (value));
		break;

		case PROP_TIMESTAMP:
			notification_set_timestamp (n, g_value_get_pointer (value));
		break;

		case PROP_URGENCY:
			notification_set_urgency (n, g_value_get_int (value));
		break;

		default :
			G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, prop, spec);
		break;
	}
}

static void
notification_class_init (NotificationClass* klass)
{
	GObjectClass* gobject_class = G_OBJECT_CLASS (klass);
	GParamSpec*   property_id;
	GParamSpec*   property_summary;
	GParamSpec*   property_body;
	GParamSpec*   property_value;
	GParamSpec*   property_icon_pixbuf;
	GParamSpec*   property_onscreen_time;
	GParamSpec*   property_dbus_sender;
	GParamSpec*   property_sender_name;
	GParamSpec*   property_sender_pid;
	GParamSpec*   property_timestamp;
	GParamSpec*   property_urgency;

	g_type_class_add_private (klass, sizeof (NotificationPrivate));

	gobject_class->dispose      = notification_dispose;
	gobject_class->finalize     = notification_finalize;
	gobject_class->get_property = notification_get_property;
	gobject_class->set_property = notification_set_property;

	property_id = g_param_spec_int ("id",
					"id",
					"unique notification id",
					-1,
					G_MAXINT,
					-1,
					G_PARAM_CONSTRUCT |
					G_PARAM_READWRITE);
	g_object_class_install_property (gobject_class,
					 PROP_ID,
					 property_id);

	property_summary = g_param_spec_string ("summary",
					      "summary",
					      "title-text of a notification",
					      NULL,
					      G_PARAM_CONSTRUCT |
					      G_PARAM_READWRITE);
	g_object_class_install_property (gobject_class,
					 PROP_SUMMARY,
					 property_summary);

	property_body = g_param_spec_string ("body",
					     "body",
					     "body-text of a notification",
					     NULL,
					     G_PARAM_CONSTRUCT |
					     G_PARAM_READWRITE);
	g_object_class_install_property (gobject_class,
					 PROP_BODY,
					 property_body);

	property_value = g_param_spec_int ("value",
					   "value",
					   "value between -1..101 to render",
					   -2,
					   101,
					   -2,
					   G_PARAM_CONSTRUCT |
					   G_PARAM_READWRITE);
	g_object_class_install_property (gobject_class,
					 PROP_VALUE,
					 property_value);

	property_icon_pixbuf = g_param_spec_pointer ("icon-pixbuf",
						     "icon-pixbuf",
						     "pixbuf of icon to use",
						     G_PARAM_CONSTRUCT |
						     G_PARAM_READWRITE);
	g_object_class_install_property (gobject_class,
					 PROP_ICON_PIXBUF,
					 property_icon_pixbuf);

	property_onscreen_time = g_param_spec_int ("onscreen-time",
						   "onscreen-time",
						   "time on screen sofar",
						   0,
						   G_MAXINT,
						   0,
						   G_PARAM_CONSTRUCT |
						   G_PARAM_READWRITE);
	g_object_class_install_property (gobject_class,
					 PROP_ONSCREEN_TIME,
					 property_onscreen_time);

	property_dbus_sender = g_param_spec_string (
					"dbus-sender",
					"dbus-sender",
					"name of DBus-sender",
					NULL,
					G_PARAM_CONSTRUCT |
					G_PARAM_READWRITE);
	g_object_class_install_property (gobject_class,
					 PROP_DBUS_SENDER,
					 property_dbus_sender);

	property_sender_name = g_param_spec_string (
					"sender-name",
					"sender-name",
					"name of sending application",
					NULL,
					G_PARAM_CONSTRUCT |
					G_PARAM_READWRITE);
	g_object_class_install_property (gobject_class,
					 PROP_SENDER_NAME,
					 property_sender_name);

	property_sender_pid = g_param_spec_int (
					"sender-pid",
					"sender-pid",
					"process ID of sending application",
					0,
					G_MAXSHORT,
					0,
					G_PARAM_CONSTRUCT |
					G_PARAM_READWRITE);
	g_object_class_install_property (gobject_class,
					 PROP_SENDER_PID,
					 property_sender_pid);

	property_timestamp = g_param_spec_pointer ("timestamp",
						   "timestamp",
						   "timestamp of reception",
					 	   G_PARAM_CONSTRUCT |
						   G_PARAM_READWRITE);
	g_object_class_install_property (gobject_class,
					 PROP_TIMESTAMP,
					 property_timestamp);

	property_urgency = g_param_spec_int ("urgency",
					     "urgency",
					     "urgency-level of notification",
					     URGENCY_LOW,
					     URGENCY_NONE,
					     URGENCY_NONE,
					     G_PARAM_CONSTRUCT |
					     G_PARAM_READWRITE);
	g_object_class_install_property (gobject_class,
					 PROP_URGENCY,
					 property_urgency);
}

//-- public functions ----------------------------------------------------------

Notification*
notification_new ()
{
	Notification*        self      = NULL;
	NotificationPrivate* priv      = NULL;
	const GTimeVal       timestamp = {0};

	self = g_object_new (NOTIFICATION_TYPE, "timestamp", &timestamp, NULL);
	priv                   = GET_PRIVATE (self);
	priv->num_actions      = 0;
	priv->actions          = NULL;
	priv->uses_button_tint = FALSE;

	/*priv->id             = -1;
	priv->summary          = NULL;
	priv->body             = NULL;
	priv->value            = -2;
	priv->icon_pixbuf      = NULL; 
	priv->onscreen_time    = 0;
	priv->dbus_sender      = NULL;
	priv->sender_name      = NULL;
	priv->sender_pid       = 0;
	priv->urgency          = URGENCY_NONE;
	priv->visible          = FALSE;*/

	return self;
}

gint
notification_get_id (Notification* n)
{
	g_return_val_if_fail (IS_NOTIFICATION (n), -1);

	return GET_PRIVATE (n)->id;
}

/* -1 is meant to "unset" the id, 0 is the smallest possible PID, but very
** unlikely to ever be passed around */
void
notification_set_id (Notification* n,
                     gint          id)
{
	g_assert (IS_NOTIFICATION (n));

	/* IDs can't be negative, except for -1 to indicate it's "unset" */
	if (id < -1)
		return;

	GET_PRIVATE (n)->id = id;
}

gchar*
notification_get_summary (Notification* n)
{
	RETURN_GCHAR (n, summary)
}

void
notification_set_summary (Notification* n,
			const gchar*  summary)
{
	SET_GCHAR (n, summary)
}

gchar*
notification_get_body (Notification* n)
{
	RETURN_GCHAR (n, body)
}

void
notification_set_body (Notification* n,
		       const gchar*  body)
{
	SET_GCHAR (n, body)
}

/* the allowed range for stored values is -1..101, thus a return-value of -2
** indicates an error on behalf of the caller */
gint
notification_get_value (Notification* n)
{
	g_return_val_if_fail (IS_NOTIFICATION (n), -2);

	return GET_PRIVATE (n)->value;
}

/* valid range is -1..101, -2 is used to indicate it's "unset" */
void
notification_set_value (Notification* n,
			gint          value)
{
	NotificationPrivate* priv;

	g_assert (IS_NOTIFICATION (n));

	priv = GET_PRIVATE (n);

	// this is used to indicate the "unset" state
	if (value == -2)
	{
		priv->value = value;
		return;
	}

	// don't store any values outside of allowed range -1..101
	if (value <= NOTIFICATION_VALUE_MIN_ALLOWED)
	{
		priv->value = NOTIFICATION_VALUE_MIN_ALLOWED;
		return;
	}

	if (value >= NOTIFICATION_VALUE_MAX_ALLOWED)
	{
		priv->value = NOTIFICATION_VALUE_MAX_ALLOWED;
		return;
	}

	priv->value = value;
}

void
notification_set_icon_from_pixbuf (Notification*    n,
                                   const GdkPixbuf* icon_pixbuf)
{
	g_return_if_fail (n && IS_NOTIFICATION (n));

	NotificationPrivate* priv = GET_PRIVATE (n);

	/* create a new/private copy of the supplied pixbuf */
	if (icon_pixbuf != NULL)
	{
		/* free any previous stored pixbuf */
		if (priv->icon_pixbuf)
		{
			g_object_unref (priv->icon_pixbuf);
			priv->icon_pixbuf = NULL;
		}

		priv->icon_pixbuf = gdk_pixbuf_copy (icon_pixbuf);
	}
}

void
notification_set_icon_from_path (Notification* n,
                                 const gchar*  filepath)
{
	g_return_if_fail (n &&
	                  IS_NOTIFICATION (n) &&
	                  filepath &&
	                  g_strcmp0 (filepath, ""));

	NotificationPrivate* priv = GET_PRIVATE (n);

	if (priv->icon_pixbuf)
	{
		g_object_unref (priv->icon_pixbuf);
		priv->icon_pixbuf = NULL;
	}

	GError* error = NULL;
	priv->icon_pixbuf = gdk_pixbuf_new_from_file (filepath, &error);
	if (!priv->icon_pixbuf)
	{
		g_print ("%s() - \"%s\"\n", G_STRFUNC, error->message);
		g_error_free (error);
	}
}

void
notification_set_icon (Notification* n,
                       const gchar*  filename)
{
	g_return_if_fail (n &&
	                  IS_NOTIFICATION (n) &&
	                  filename &&
	                  g_strcmp0 (filename, ""));

	NotificationPrivate* priv = GET_PRIVATE (n);

	if (priv->icon_pixbuf)
	{
		g_object_unref (priv->icon_pixbuf);
		priv->icon_pixbuf = NULL;
	}

	priv->icon_pixbuf = _notification_load_icon (filename);
}

GdkPixbuf*
notification_get_icon_pixbuf (Notification* n)
{
	g_return_val_if_fail (n && IS_NOTIFICATION (n), NULL);
	
	return GET_PRIVATE (n)->icon_pixbuf;
}

/* a return-value of -1 indicates an error on behalf of the caller, a
** return-value of 0 would indicate that a notification has not been displayed
** yet */
gint
notification_get_onscreen_time (Notification* n)
{
	g_return_val_if_fail (IS_NOTIFICATION (n), -1);

	return GET_PRIVATE (n)->onscreen_time;
}

// only onscreen_time values <= 0 are considered valid
void
notification_set_onscreen_time (Notification* n,
				gint          onscreen_time)
{
	NotificationPrivate* priv;

	g_assert (IS_NOTIFICATION (n));

	priv = GET_PRIVATE (n);

	// avoid storing negative values
	if (onscreen_time < 0)
		return;

	// onscreen-time can only increase not decrease
	if (priv->onscreen_time > onscreen_time)
		return;

	// you made it upto here, congratulations... let's store the new value
	priv->onscreen_time = onscreen_time;
}

gchar*
notification_get_dbus_sender (Notification* n)
{
	RETURN_GCHAR (n, dbus_sender)
}

void
notification_set_dbus_sender (Notification* n,
                              const gchar*  dbus_sender)
{
	SET_GCHAR (n, dbus_sender)
}

gchar*
notification_get_sender_name (Notification* n)
{
	RETURN_GCHAR (n, sender_name)
}

void
notification_set_sender_name (Notification* n,
			      const gchar*  sender_name)
{
	SET_GCHAR (n, sender_name)
}

// a return-value of 0 indicates an error on behalf of the caller, PIDs are
// never negative
gint
notification_get_sender_pid (Notification* n)
{
	g_return_val_if_fail (IS_NOTIFICATION (n), 0);

	return GET_PRIVATE (n)->sender_pid;
}

void
notification_set_sender_pid (Notification* n,
			     const gint    sender_pid)
{
	g_assert (IS_NOTIFICATION (n));

	// it's hardly possible we'll get a notification from init, but anyway
	if (sender_pid <= 0)
		return;

	GET_PRIVATE (n)->sender_pid = sender_pid;
}

GTimeVal*
notification_get_timestamp (Notification* n)
{
	NotificationPrivate* priv;

	g_return_val_if_fail (IS_NOTIFICATION (n), NULL);

	priv = GET_PRIVATE (n);

	return &priv->timestamp;
}

void
notification_set_timestamp (Notification*   n,
			    const GTimeVal* timestamp)
{
	NotificationPrivate* priv;

	// sanity checks
	g_assert (IS_NOTIFICATION (n));
	if (!timestamp)
		return;

	priv = GET_PRIVATE (n);

	// don't store older timestamp over newer one
	if (priv->timestamp.tv_sec > timestamp->tv_sec)
		return;

	if (priv->timestamp.tv_sec == timestamp->tv_sec)
		if (priv->timestamp.tv_usec > timestamp->tv_usec)
			return;

	// new timestamp certainly more current that stored one
	priv->timestamp.tv_sec  = timestamp->tv_sec;
	priv->timestamp.tv_usec = timestamp->tv_usec;
}

gint
notification_get_urgency (Notification* n)
{
	g_return_val_if_fail (IS_NOTIFICATION (n), URGENCY_NONE);

	return GET_PRIVATE (n)->urgency;
}

void
notification_set_urgency (Notification* n,
                          Urgency       urgency)
{
	NotificationPrivate* priv;

	g_assert (IS_NOTIFICATION (n));

	priv = GET_PRIVATE (n);

	// URGENCY_NONE is meant to indicate the "unset" state
	if (urgency == URGENCY_NONE)
	{
		priv->urgency = URGENCY_NONE;
		return;
	}

	// don't store any values outside of allowed range LOW..HIGH
	if (urgency != URGENCY_LOW    &&
	    urgency != URGENCY_NORMAL &&
	    urgency != URGENCY_HIGH)
		return;

	GET_PRIVATE (n)->urgency = urgency;
}

gchar**
notification_get_actions (Notification* n)
{
	g_return_val_if_fail (IS_NOTIFICATION (n), NULL);

	return GET_PRIVATE (n)->actions;
}

void
notification_set_actions (Notification* n,
                          gchar**       actions)
{
	NotificationPrivate* priv;

	g_return_if_fail (n && IS_NOTIFICATION (n) && actions);

	priv = GET_PRIVATE (n);

	/* clear current list of actions */
	if (priv->num_actions > 0)
	{
		for (guint i = 0; i < priv->num_actions; i++)
			g_free (priv->actions[i]);

		g_free (priv->actions);
		priv->actions     = NULL;
		priv->num_actions = 0;
	}

	/* copy/fill list of actions with new strings */
	if (actions != NULL)
	{
		guint i = 0;
		while (actions[i] != NULL)
			i++;
		priv->num_actions = i;

		priv->actions = (gchar*) g_malloc0 (priv->num_actions * sizeof (gchar*));
		for (i = 0; i < priv->num_actions; i++)
			priv->actions[i] = g_strdup (actions[i]);
	}
}

guint
notification_get_num_actions (Notification* n)
{
	g_assert (IS_NOTIFICATION (n));

	return GET_PRIVATE (n)->num_actions;
}

gchar*
notification_get_action_by_index (Notification* n,
                                  guint         index)
{
	g_return_val_if_fail (n &&
	                      IS_NOTIFICATION (n) &&
	                      GET_PRIVATE (n)->num_actions - 1 >= index,
	                      NULL);

	return GET_PRIVATE (n)->actions[index];
}

void
notification_set_visible (Notification* n)
{
	g_assert (n != NULL && IS_NOTIFICATION (n));

	GET_PRIVATE (n)->visible = TRUE;
}

gboolean
notification_is_visible (Notification* n)
{
	g_assert (IS_NOTIFICATION (n));

	return GET_PRIVATE (n)->visible;
}

void
notification_set_button_tint (Notification* n,
                              gboolean      uses_tint)
{
	g_return_if_fail (n && IS_NOTIFICATION (n));

	GET_PRIVATE (n)->uses_button_tint = uses_tint;
}

gboolean
notification_get_button_tint (Notification* n)
{
	g_return_val_if_fail (n && IS_NOTIFICATION (n), FALSE);

	return GET_PRIVATE (n)->uses_button_tint;
}
