/*******************************************************************************
**3456789 123456789 123456789 123456789 123456789 123456789 123456789 123456789
**      10        20        30        40        50        60        70        80
**
** notify-osd
**
** queue.c - manages the stack/queue of incoming snap-decisions
**
** Copyright 2012 Canonical Ltd.
**
** Authors:
**    Mirco "MacSlow" Mueller <mirco.mueller@canonical.com>
**
** This program is free software: you can redistribute it and/or modify it
** under the terms of the GNU General Public License version 3, as published
** by the Free Software Foundation.
**
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranties of
** MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
** PURPOSE.  See the GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License along
** with this program.  If not, see <http://www.gnu.org/licenses/>.
**
*******************************************************************************/

#include <assert.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <glib-object.h>

#include "notification.h"
#include "queue.h"

G_DEFINE_TYPE (Queue, queue, G_TYPE_OBJECT);

/*-- internal API ------------------------------------------------------------*/

static void
queue_dispose (GObject* gobject)
{
	/* chain up to the parent class */
	G_OBJECT_CLASS (queue_parent_class)->dispose (gobject);
}

static void
remove_sd (gpointer data)
{
	g_assert (data != NULL);
	Notification* n = NOTIFICATION (data);
	g_object_unref (n);
}

static void
queue_finalize (GObject* gobject)
{
	if (QUEUE(gobject)->list != NULL)
		g_list_free_full (QUEUE(gobject)->list, remove_sd);

	/* chain up to the parent class */
	G_OBJECT_CLASS (queue_parent_class)->finalize (gobject);
}

static void
queue_init (Queue* self)
{
	/* If you need specific construction properties to complete
	** initialization, delay initialization completion until the
	** property is set. */

	self->list = NULL;
}

static void
queue_get_property (GObject*    gobject,
                    guint       prop,
                    GValue*     value,
                    GParamSpec* spec)
{
	switch (prop)
	{
		default :
			G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, prop, spec);
		break;
	}
}

static void
queue_class_init (QueueClass* klass)
{
	GObjectClass* gobject_class = G_OBJECT_CLASS (klass);

	gobject_class->dispose      = queue_dispose;
	gobject_class->finalize     = queue_finalize;
	gobject_class->get_property = queue_get_property;

	/*dbus_g_object_type_install_info (G_TYPE_FROM_CLASS (klass),
					 &dbus_glib_stack_object_info);*/
}

gint
_compare_sender_name (gconstpointer a,
                      gconstpointer b)
{
	gchar* sender_name_1;
	gchar* sender_name_2;

	if (!a || !b)
		return -1;

	if (!IS_NOTIFICATION (a))
		return -1;

	sender_name_1 = notification_get_sender_name ((Notification*) a);
	sender_name_2 = ((gchar*) b);

	return g_strcmp0 (sender_name_1, sender_name_2);
}

gint
_compare_pid (gconstpointer a,
              gconstpointer b)
{
	gint  result;
	guint pid_1;
	guint pid_2;

	if (!a || !b)
		return -1;

	if (!IS_NOTIFICATION (a))
		return -1;

	pid_1 = notification_get_sender_pid ((Notification*) a);
	pid_2 = *((guint*) b);

	if (pid_1 < pid_2)
		result = -1;

	if (pid_1 == pid_2)
		result = 0;

	if (pid_1 > pid_2)
		result = 1;

	return result;
}

gint
_compare_id (gconstpointer a,
             gconstpointer b)
{
	gint  result;
	gint id_1;
	gint id_2;

	if (!a || !b)
		return -1;

	if (!IS_NOTIFICATION (a))
		return -1;

	id_1 = notification_get_id ((Notification*) a);
	id_2 = *((gint*) b);

	if (id_1 < id_2)
		result = -1;

	if (id_1 == id_2)
		result = 0;

	if (id_1 > id_2)
		result = 1;

	return result;
}

gint
_compare_urgency (gconstpointer a,
                  gconstpointer b)
{
	gint    result;
	Urgency urgency_1;
	Urgency urgency_2;

	if (!a || !b)
		return -1;

	if (!IS_NOTIFICATION (a) || !IS_NOTIFICATION (b))
		return -1;

	urgency_1 = notification_get_urgency ((Notification*) a);
	urgency_2 = notification_get_urgency ((Notification*) b);

	if (urgency_1 < urgency_2)
		result = 1;

	if (urgency_1 == urgency_2)
		result = 0;

	if (urgency_1 > urgency_2)
		result = -1;

	return result;
}

gint
_compare_visible (gconstpointer a,
                  gconstpointer b)
{
	gint    result;
	gboolean visible_1;
	gboolean visible_2;

	if (!a || !b)
		return -1;

	if (!IS_NOTIFICATION (a))
		return -1;

	visible_1 = notification_is_visible ((Notification*) a);
	visible_2 = *((gboolean*) b);

	if (visible_1 < visible_2)
		result = 1;

	if (visible_1 == visible_2)
		result = 0;

	if (visible_1 > visible_2)
		result = -1;

	return result;
}

/*-- public API --------------------------------------------------------------*/

Queue*
queue_new ()
{
	Queue* self = NULL;

	self = g_object_new (QUEUE_TYPE, NULL);
	if (!self)
		return NULL;

	self->list    = NULL;
	self->next_id = 0;

	return self;
}

void
queue_del (Queue* self)
{
	if (!self)
		return;

	g_list_free_full (self->list, remove_sd);
	self->list = NULL;
	g_object_unref (self);
}

void
queue_inject (Queue*        self,
              Notification* n)
{
	gchar* sender_name = NULL;
	GList* element     = NULL;

	/* sanity check */
	if (!self || !IS_QUEUE(self) || !n || !IS_NOTIFICATION(n))
		return;

	sender_name = notification_get_sender_name (n);
	element = g_list_find_custom (self->list,
	                              (gconstpointer) sender_name,
	                              _compare_sender_name);
	if (element)
	{
		g_print ("\"%s\" tried to inject more than snap-decision!\n",
		         notification_get_sender_name (n));
		return;
	}

	if (!queue_is_full (self))
	{
		self->list = g_list_append (self->list, (gpointer) n);
		queue_reorder (self);
	}
}

void
queue_remove (Queue*        self,
              Notification* n)
{
	GList* element = NULL;
	gint   id      = -1;

	/* sanity check */
	if (!self || !IS_QUEUE(self) || !n || !IS_NOTIFICATION(n))
		return;

	id = notification_get_id (n);

	element = g_list_find_custom (self->list, (gconstpointer) &id, _compare_id);
	if (!element)
		return;

	self->list = g_list_remove_link (self->list, element);
	g_list_free_full (element, remove_sd);
	queue_reorder (self);
}

void
queue_remove_by_pid (Queue* self,
                     guint  pid)
{
	GList* element = NULL;

	/* sanity check */
	if (!self || !IS_QUEUE(self))
		return;

	element = g_list_find_custom (self->list,
	                              (gconstpointer) &pid,
	                              _compare_pid);
	if (!element)
		return;

	self->list = g_list_remove_link (self->list, element);
	g_list_free_full (element, remove_sd);
	queue_reorder (self);
}

void
queue_remove_by_id (Queue* self,
                    gint   id)
{
	GList* element = NULL;

	/* sanity check */
	if (!self || !IS_QUEUE(self))
		return;

	element = g_list_find_custom (self->list, (gconstpointer) &id, _compare_id);
	if (element)
	{
		self->list = g_list_remove_link (self->list, element);
		g_list_free_full (element, remove_sd);
		queue_reorder (self);
	}
	else if (id != 0) /* notifications with ID 0 are not used */
		g_print ("%s() - Could not find notification with ID = %d\n",
		         G_STRFUNC,
		         id);
}

Notification*
queue_find_by_pid (Queue* self,
                   guint  pid)
{
	GList* element = NULL;

	/* sanity check */
	if (!self || !IS_QUEUE (self))
		return NULL;

	element = g_list_find_custom (self->list,
	                              (gconstpointer) &pid,
	                              _compare_pid);
	if (!element)
		return NULL;

	return NOTIFICATION (element->data);
}

Notification*
queue_find_by_id (Queue* self,
                  gint   id)
{
	GList* element = NULL;

	/* sanity check */
	if (!self || !IS_QUEUE (self))
		return NULL;

	element = g_list_find_custom (self->list, (gconstpointer) &id, _compare_id);

	if (!element)
		return NULL;

	return NOTIFICATION (element->data);
}

guint
queue_get_size (Queue* self)
{
	/* sanity check */
	if (!self || !IS_QUEUE (self))
		return 0;

	return (g_list_length (self->list));
}

gboolean
queue_is_full (Queue* self)
{
	/* sanity check */
	if (!self || !IS_QUEUE (self))
		return FALSE;

	return (g_list_length (self->list) == MAX_QUEUE_SIZE);
}

/* order by urgency */
void
queue_reorder (Queue* self)
{
	/* sanity check */
	if (!self || !IS_QUEUE (self))
		return;

	self->list = g_list_sort (self->list, _compare_urgency);
}

Notification*
queue_pop (Queue* self)
{
	/* sanity check */
	if (!self || !IS_QUEUE (self))
		return NULL;

	gboolean      flag    = FALSE;
	GList*        element = NULL;
	Notification* n       = NULL;

	element = g_list_find_custom (self->list,
	                              (gconstpointer) &flag,
	                              _compare_visible);

	if (element != NULL)
		n = NOTIFICATION (element->data);

	return n;
}
