/*******************************************************************************
**3456789 123456789 123456789 123456789 123456789 123456789 123456789 123456789
**      10        20        30        40        50        60        70        80
**
** notify-osd
**
** visual-queue.h - unit-tests for visual-queue class
**
** Copyright 2012 Canonical Ltd.
**
** Authors:
**    Mirco "MacSlow" Mueller <mirco.mueller@canonical.com>
**
** This program is free software: you can redistribute it and/or modify it
** under the terms of the GNU General Public License version 3, as published
** by the Free Software Foundation.
**
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranties of
** MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
** PURPOSE.  See the GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License along
** with this program.  If not, see <http://www.gnu.org/licenses/>.
**
*******************************************************************************/

#include <glib.h>

#include "visual-queue.h"

#define RIGHT_EDGE  1700
#define TOP_EDGE    24
#define BOTTOM_EDGE 1000

static void
test_visual_queue_new ()
{
	VisualQueue* visual_queue = NULL;
	Queue*       queue        = NULL;

	queue = queue_new ();
	visual_queue = visual_queue_new (RIGHT_EDGE, TOP_EDGE, BOTTOM_EDGE, queue);
	g_assert (visual_queue != NULL);
	g_assert (IS_VISUAL_QUEUE (visual_queue));
	g_object_unref (visual_queue);
	queue_del (queue);
}

static void
test_visual_queue_del ()
{
	VisualQueue* visual_queue = NULL;
	Queue*       queue        = NULL;

	queue = queue_new ();
	visual_queue = visual_queue_new (RIGHT_EDGE, TOP_EDGE, BOTTOM_EDGE, queue);
	g_assert (visual_queue != NULL);
	g_assert (IS_VISUAL_QUEUE (visual_queue));
	visual_queue_del (visual_queue);
	g_assert (!IS_VISUAL_QUEUE (visual_queue));
	visual_queue = NULL;
	queue_del (queue);
}

static void
test_visual_queue_pull_notification ()
{
	VisualQueue*  visual_queue = NULL;
	Queue*        queue        = NULL;
	Notification* n1           = NULL;
	Notification* n2           = NULL;
	gchar*        actions[5]   = {"accept_id",  "Accept",
								  "decline_id", "Decline",
								  NULL};	
    const gchar*  dbus_name_1 = ":1.1234";
    const gchar*  dbus_name_2 = ":1.4321";

	// create pixbuf for testing
	GdkPixbuf* pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, TRUE, 8, 100, 100);

	queue = queue_new ();
	g_assert (queue != NULL);
	g_assert (IS_QUEUE (queue));

	n1 = notification_new ();
	notification_set_icon_from_pixbuf (n1, pixbuf);
	notification_set_actions (n1, actions);
	notification_set_sender_name (n1, "application1");
	notification_set_sender_pid (n1, 42);
	notification_set_urgency (n1, URGENCY_NORMAL);
	notification_set_dbus_sender (n1, dbus_name_1);
	notification_set_summary (n1, "Title");
	notification_set_body (n1, "Foo bar laber fasel.");

	n2 = notification_new ();
	notification_set_icon_from_pixbuf (n2, pixbuf);
	notification_set_actions (n2, actions);
	notification_set_sender_name (n2, "application2");
	notification_set_sender_pid (n2, 13);
	notification_set_urgency (n2, URGENCY_NORMAL);
	notification_set_dbus_sender (n2, dbus_name_2);
	notification_set_summary (n2, "Summary");
	notification_set_body (n2, "Just some body text.");

	queue_inject (queue, n1);
	queue_inject (queue, n2);
	g_assert_cmpuint (queue_get_size (queue), ==, 2);

	visual_queue = visual_queue_new (RIGHT_EDGE, TOP_EDGE, BOTTOM_EDGE, queue);
	g_assert (visual_queue != NULL);
	g_assert (IS_VISUAL_QUEUE (visual_queue));

	g_assert_cmpint (visual_queue_get_size (visual_queue), ==, 0);
	visual_queue_pull_notification (visual_queue);
	g_assert_cmpint (visual_queue_get_size (visual_queue), ==, 1);
	visual_queue_pull_notification (visual_queue);
	g_assert_cmpint (visual_queue_get_size (visual_queue), ==, 2);

	queue_del (queue);
	g_assert (!IS_QUEUE (queue));
	queue = NULL;

	visual_queue_del (visual_queue);
	g_assert (!IS_VISUAL_QUEUE (visual_queue));
	visual_queue = NULL;

	g_object_unref (pixbuf);
	pixbuf = NULL;
}

static void
test_visual_queue_is_full ()
{
	VisualQueue*  visual_queue      = NULL;
	Queue*        queue             = NULL;
	Notification* n[MAX_QUEUE_SIZE];
	gint          i                 = 0;
	gchar*        actions[5]        = {"accept_id",  "Accept",
								       "decline_id", "Decline",
								       NULL};	
	// create pixbuf for testing
	GdkPixbuf* pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, TRUE, 8, 100, 100);

	queue = queue_new ();
	g_assert (queue != NULL);
	g_assert (IS_QUEUE (queue));

	for (i = 0; i < MAX_QUEUE_SIZE; i++)
	{
		n[i] = notification_new ();
		notification_set_icon_from_pixbuf (n[i], pixbuf);
		GString* app_name = g_string_new ("app_");
		g_string_append_printf (app_name, "%d", i);
		notification_set_sender_name (n[i], app_name->str);
		g_string_free (app_name, TRUE);
		notification_set_sender_pid (n[i], i);
		notification_set_urgency (n[i], URGENCY_NORMAL);
		notification_set_actions (n[i], actions);

		GString* dbus_name = g_string_new (":1.12");
		g_string_append_printf (dbus_name, "%d", i);
		notification_set_dbus_sender (n[i], dbus_name->str);
		g_string_free (dbus_name, TRUE);

		notification_set_summary (n[i], "Title");
		notification_set_body (n[i], "Foo bar laber fasel.");
		queue_inject (queue, n[i]);
	}

	visual_queue = visual_queue_new (RIGHT_EDGE, TOP_EDGE, BOTTOM_EDGE, queue);
	g_assert (visual_queue != NULL);
	g_assert (IS_VISUAL_QUEUE (visual_queue));

	while (!visual_queue_is_full (visual_queue))
		visual_queue_pull_notification (visual_queue);

	g_assert (visual_queue_is_full (visual_queue));
	g_assert (visual_queue_get_size (visual_queue) < queue_get_size (queue));

	g_print ("\nvisual-queue-size (%d) < queue-size (%d)\n",
	         visual_queue_get_size (visual_queue),
	         queue_get_size (queue));

	queue_del (queue);
	g_assert (!IS_QUEUE (queue));
	queue = NULL;

	visual_queue_del (visual_queue);
	g_assert (!IS_VISUAL_QUEUE (visual_queue));
	visual_queue = NULL;

	g_object_unref (pixbuf);
	pixbuf = NULL;
}

GTestSuite *
test_visual_queue_create_test_suite (void)
{
	GTestSuite *ts = NULL;

	ts = g_test_create_suite ("visual-queue");

#define TC(x) g_test_create_case(#x, 0, NULL, NULL, x, NULL)

	g_test_suite_add(ts, TC(test_visual_queue_new));
	g_test_suite_add(ts, TC(test_visual_queue_del));
	g_test_suite_add(ts, TC(test_visual_queue_pull_notification));
	g_test_suite_add(ts, TC(test_visual_queue_is_full));

	return ts;
}
