/******************************************************************************
 *
 * Copyright(c) 2005 - 2013 Intel Corporation.
 * All rights reserved.
 *
 * LICENSE PLACE HOLDER
 *
 *****************************************************************************/

#include <linux/kthread.h>

#include "iwl-fh.h"
#include "iwl-io.h"
#include "iwl-prph.h"
#include "iwl-csr.h"
#include "iwl-em-sfdb.h"
#include "trans_slave/idi_al.h"
#include "iwl-emulation.h"
#include "iwl-target-access.h"

#ifdef CPTCFG_IWLWIFI_TRANS_UT
#include <linux/interrupt.h>
#include "trans_slave/idi_UT/iwl-idi-ut.h"
#endif
/*
 * IWL SFDB Emulation Logger
 */
#define SFDB_LOG_PREFIX	"[SFDB]"

#ifdef __IWL_SFDB_LOG_ENABLED__
#define SFDB_TRACE_ENTER \
	IWL_EM_TRACE_ENTER(__SFDB_LOG_LEVEL_TRACE__, SFDB_LOG_PREFIX)
#define SFDB_TRACE_EXIT \
	IWL_EM_TRACE_EXIT(__SFDB_LOG_LEVEL_TRACE__, SFDB_LOG_PREFIX)
#define SFDB_TRACE_EXIT_RET_STR(_ret) \
	IWL_EM_LOG(__SFDB_LOG_LEVEL_TRACE__, SFDB_LOG_PREFIX, "<<< "_ret)
#define SFDB_TRACE_EXIT_RET(_ret) \
	IWL_EM_TRACE_EXIT_RET(__SFDB_LOG_LEVEL_TRACE__, SFDB_LOG_PREFIX,\
			      _ret)
#define IWL_SFDB_LOG(fmt, args ...) \
	IWL_EM_LOG(__SFDB_LOG_LEVEL_DEBUG__, SFDB_LOG_PREFIX, fmt, ## args)
#define IWL_SFDB_LOG_HEX_DUMP(msg, p, len) \
	IWL_EM_LOG_HEX_DUMP(msg, p, len)
#else

#define SFDB_TRACE_ENTER
#define SFDB_TRACE_EXIT
#define SFDB_TRACE_EXIT_RET_STR(_ret)
#define SFDB_TRACE_EXIT_RET(_ret)
#define IWL_SFDB_LOG(fmt, args ...)
#define IWL_SFDB_LOG_HEX_DUMP(msg, p, len)

#endif
/* Always print error messages */
#define IWL_SFDB_LOG_ERR(fmt, args ...) \
	IWL_EM_LOG_ERR(fmt, ## args)

/* Pointer manipulation defines */
#define SFDB_SCD_PTR_MASK	(SFDB_SCD_QUEUE_SIZE - 1)
#define SFDB_QUEUE_PTR_MASK	(SFDB_TFD_NUM_ENTRIES - 1)
#define SFDB_SCD_PTR(_p)	((_p) & SFDB_SCD_PTR_MASK)
#define SFDB_QUEUE_PTR(_p)	((_p) & SFDB_QUEUE_PTR_MASK)
#define SFDB_INC_PTR(_p)	((_p) = ((_p)+1) & SFDB_SCD_PTR_MASK)
#define SFDB_GET_QUEUE(_qid)	(&iwl_sfdb_em->tfd_queues[_qid])

/* This is the order of TFD size, size = 64 ==> order = 6 */
#define HIT_DATA_TFD_SIZE_ORDER (6)

/* dma_addr_t of allocated BC table, externed to replace AL define */
u32 IWL_SFDB_BC_ADDRESS;

#ifdef CPTCFG_IWLWIFI_TRANS_UT
static void iwl_sfdb_em_dispatch_reclaim_emu(unsigned long data);
#endif

/* Like struct iwl_dma_ptr, which is currently PCIe-specific. */
struct iwl_em_dma_ptr {
	void *addr;
	dma_addr_t dma;
	size_t size;
};

/**
 * Holds one TFD queue for the emulation.
 *
 * The PCIe TFD queue itself is dynamically allocated, and its virtual and dma
 * address is saved in 'tfds'. The queue is used by the PCIe FH.
 * The indices array is used to tell which PCIe TFD entry (in the tfds queue)
 * is a decompression of which tfd index as represented by the AL FW and mapped
 * to the SRAM.
 * Meaning that when we attach TFD x from the pool (of IDI TFDs),
 * we both decompress it to pcie_tfds.addr[write_ptr], and write 'x'
 * in lut_table[write_ptr],
 * so later we would be able to free/clean both its representations.
 **/
struct iwl_tfd_em_queue {
	int write_ptr;
	int read_ptr;

	bool need_update;
	bool is_aggregated;

	struct iwl_em_dma_ptr pci_tfds;
	u8 lut_table[SFDB_TFD_NUM_ENTRIES];
};

/**
  * Represents a byte count table per one queue.
  * For each entry in the queue holds a 2-byte count.
  * This is multiplied by 2 in order to
  * provide duplication of each entry as needed by the scheduler.
  */
struct iwl_sfdb_em_bc_tbl {
	__le16 sfdb_queue_bc_table[SFDB_TFD_NUM_ENTRIES * 2];
} __packed;

/**
  * Holds the whole TFD queues (SFDB) emulation context.
  * kw -- keep warm buffer (not present in IDI, so here we manage it)
  * tfd_queues -- array of TFD queue contexts
  * trans -- pointer to transport layer, to interact with the driver
  **/
struct iwl_sfdb_t {
	/* Connection to PCI transport */
	struct iwl_trans *trans;

	/* Internal KW and BC buffers */
	struct iwl_em_dma_ptr kw_buffer;
	struct iwl_em_dma_ptr bc_tables;

	/* TFDQs */
	struct iwl_tfd_em_queue tfd_queues[SFDB_NUM_QUEUES];

#ifdef CPTCFG_IWLWIFI_TRANS_UT
	/* Gate on the queue id given to the reclaim flow */
	spinlock_t gate;

	/* Runs the reclaim flow emulation - needs to be in tasklet context
	 * to emulate RX packet*/
	struct tasklet_struct reclaim_flow;
#endif
};

/* Global SFDB struct */
static struct iwl_sfdb_t iwl_sfdb_em_global;
static struct iwl_sfdb_t *iwl_sfdb_em = &iwl_sfdb_em_global;

/**
 * Represents IDI form of transmission buffers
 *
 * @tb _addr - The index of the TB in the SRAM TB Pool.
 * @t_len - The length of the tb.
 */
struct iwl_sfdb_idi_tb {
	u8	tb_addr;
	__le16	tb_len;
} __packed;

/**
 * Represents IDI form of transmission frame descriptor (TFD)
 */
struct iwl_sfdb_idi_tfd {
	u8			reserved[3];
	u8			numTbs;
	struct iwl_sfdb_idi_tb	tbs[SFDB_NUM_OF_TBS];
} __packed;

/*
 ************ Utils functions *************
 */

/**
 * Allocates a DMAable address and fills it to the emulation dma pointer struct.
 * The dma memory is zeroed.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
static inline int iwl_sfdb_em_alloc_dma_ptr(struct iwl_em_dma_ptr *ptr,
					    size_t size)
{
	SFDB_TRACE_ENTER;

	if (WARN_ON(ptr->addr))
		return -EINVAL;

	/* Alloc dmaable pointer */
	ptr->addr = dma_alloc_coherent(iwl_sfdb_em->trans->dev,
				       size,
				       &ptr->dma, GFP_KERNEL);
	if (!ptr->addr)
		return -ENOMEM;
	ptr->size = size;

	/* Zero allocated memory */
	memset(ptr->addr, 0, size);

	IWL_SFDB_LOG("Allocated DMA pointer: virt %p phys %lx size %zu\n",
		     ptr->addr, (unsigned long)ptr->dma, ptr->size);
	SFDB_TRACE_ENTER;
	return 0;
}

/**
 * Frees a DMAable memory location with the emulation dma pointer struct.
 * The struct is zeroed before return.
 */
static inline void iwl_sfdb_em_free_dma_ptr(struct iwl_em_dma_ptr *ptr)
{
	SFDB_TRACE_ENTER;

	if (!ptr->addr)
		return;

	IWL_SFDB_LOG("Freeing DMA pointer: virt %p phys %lx size %zu\n",
		     ptr->addr, (unsigned long)ptr->dma, ptr->size);
	dma_free_coherent(iwl_sfdb_em->trans->dev,
			  ptr->size,
			  ptr->addr,
			  ptr->dma);
	memset(ptr, 0, sizeof(*ptr));

	SFDB_TRACE_EXIT;
}

/**
 * Gets a pointer to a TFD in the IDI form which is located in the AL SRAM.
 */
static inline struct iwl_sfdb_idi_tfd *iwl_sfdb_em_get_idi_tfd(u8 tfd_index)
{
	return ((struct iwl_sfdb_idi_tfd *)((u8 *)AL_SRAM_VIRTUAL_ADDRESS
					+ IDI_AL_SFDB_TFD_POOL_OFFSET))
		+ tfd_index;
}

/**
 * Gets a pointer to a TB in the IDI form which is located in the AL SRAM.
 */
static void *iwl_sfdb_em_get_idi_tb(u8 tb_index)
{
	SFDB_TRACE_ENTER;

	return ((u8 *)AL_SRAM_VIRTUAL_ADDRESS + IDI_AL_SFDB_PAYLOAD_MEM_OFFSET)
		 + (tb_index * SFDB_TB_SIZE);

	SFDB_TRACE_EXIT;
}

/**
 * Converts a TB in the IDI form to the PCIe TB form.
 *
 * Each IDI TB is composed of one-byte index to the TB pool, and 1 byte length.
 * The PCIe TB is made of 36 bit of address, and 12 bits of length.
 *
 * Copyies the idi_tb to the pci_tb.
 */
static inline void iwl_sfdb_em_convert_tb(struct iwl_sfdb_idi_tb *idi_tb,
					  struct iwl_tfd_tb *pcie_tb,
					  int tb_index)
{
	/* Buffer is the virtual address of the idi tb
	    phys_addr is the phisical address of the idi tb */
	void *virt_addr = iwl_sfdb_em_get_idi_tb(idi_tb->tb_addr);
	u16 hi_n_len = (le16_to_cpu(idi_tb->tb_len) << 4);
	dma_addr_t phys_addr;
	SFDB_TRACE_ENTER;

	/* Calc IDI TB phisical address */
	phys_addr = (dma_addr_t)((u8 *)AL_SRAM_ADDRESS +
			((u8 *)virt_addr - (u8 *)AL_SRAM_VIRTUAL_ADDRESS));
	IWL_SFDB_LOG("IDI TB: Convertin TB - index=%d, addres=%lx, len=%u",
		     idi_tb->tb_addr, (unsigned long)phys_addr,
		     idi_tb->tb_len);

	put_unaligned_le32(phys_addr, &pcie_tb->lo);
	if (sizeof(dma_addr_t) > sizeof(u32))
		hi_n_len |= (((phys_addr >> 16) >> 16) & 0xF);
	pcie_tb->hi_n_len = cpu_to_le16(hi_n_len);

	SFDB_TRACE_EXIT;
}

/*
 ************ API functions ***************
 */

/**
 * Initializes the SFDB module in the emulation.
 *
 *@trans - transport to the pci.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_sfdb_em_init(struct iwl_trans *trans)
{
	int ret;
	SFDB_TRACE_ENTER;

	/* Zero structs */
	memset(iwl_sfdb_em, 0, sizeof(struct iwl_sfdb_t));

	/* Store trans */
	iwl_sfdb_em->trans = trans;

	/* Initialize keep warm buffers */
	IWL_SFDB_LOG("Initializing keep-warm buffer");
	ret = iwl_sfdb_em_alloc_dma_ptr(&iwl_sfdb_em->kw_buffer, IWL_KW_SIZE);
	if (ret) {
		IWL_SFDB_LOG_ERR("Keep Warm allocation failed (%d)", ret);
		goto out;
	}

	/* Initialize Byte Count tables*/
	IWL_SFDB_LOG("Initializing BC tables");
	ret = iwl_sfdb_em_alloc_dma_ptr(&iwl_sfdb_em->bc_tables,
		SFDB_NUM_QUEUES * sizeof(struct iwl_sfdb_em_bc_tbl));
	if (ret) {
		IWL_SFDB_LOG_ERR("BC tables allocation failed (%d)", ret);
		goto free_kw;
	}
	IWL_SFDB_BC_ADDRESS = (u32)&iwl_sfdb_em->bc_tables.dma;

#ifdef CPTCFG_IWLWIFI_TRANS_UT
	/* Init reclaim flow tasklet */
	tasklet_init(&(iwl_sfdb_em->reclaim_flow),
		     iwl_sfdb_em_dispatch_reclaim_emu, 0);
	spin_lock_init(&iwl_sfdb_em->gate);
#endif

	IWL_SFDB_LOG("SFDB Initialized");
	goto out;

free_kw:
	iwl_sfdb_em_free_dma_ptr(&iwl_sfdb_em->kw_buffer);
out:
	SFDB_TRACE_EXIT;
	return ret;
}

/*
 * start  the SFDB.
 * The start should be called only after sfdb init or after stop flow.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_sfdb_em_start(void)
{
	/* Write KW address to NIC */
	iwl_idi_tg_write32(iwl_sfdb_em->trans,
			   FH_KW_MEM_ADDR_REG,
			   iwl_sfdb_em->kw_buffer.dma >> 4);

#ifdef CPTCFG_IWLWIFI_TRANS_UT
	/* Re Init reclaim flow tasklet */
	tasklet_init(&(iwl_sfdb_em->reclaim_flow),
		     iwl_sfdb_em_dispatch_reclaim_emu, 0);
#endif
	return 0;
}

/*
 * Reset the queue data struct to default values.
 *
 * @queue_id - The queue id to reset in the sfdb.
 */
static void iwl_sfdb_em_reset_queue(u8 queue_id)
{
	struct iwl_tfd_em_queue *q = SFDB_GET_QUEUE(queue_id);
	SFDB_TRACE_ENTER;

	/* Q flags */
	q->is_aggregated = false;
	q->need_update = true;

	/* Read/Write pointer */
	q->read_ptr = 0;
	q->write_ptr = 0;

	/* Set all TB indexes to invalid */
	memset(q->lut_table, SFDB_INVALID_ENTRY, SFDB_TFD_NUM_ENTRIES);

	SFDB_TRACE_EXIT;
}

/*
 * Stops the SFDB.
 * Does not free the data structures just zeroes them.
 *
 * NOTE: The LMAC should be stopped before this call to prevent
 * a race condition of LMAC pulling data from zeroed ring.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_sfdb_em_stop(void)
{
	int i;
	SFDB_TRACE_ENTER;

	/* Clear the BC tables */
	memset(iwl_sfdb_em->bc_tables.addr,
	       0,
	       SFDB_NUM_QUEUES * sizeof(struct iwl_sfdb_em_bc_tbl));

	/* Clear the Queues */
	for (i = 0; i < SFDB_NUM_QUEUES; i++)
		iwl_sfdb_em_reset_queue(i);

#ifdef CPTCFG_IWLWIFI_TRANS_UT
	tasklet_kill(&iwl_sfdb_em->reclaim_flow);
#endif

	SFDB_TRACE_EXIT;
	return 0;
}

/**
 * Frees the sfdb emulation struct.
 *
 * The al emulation stopping should be called before trying to stop the SFDB.
 */
void iwl_sfdb_em_free(void)
{
	int i;
	struct iwl_tfd_em_queue *q;
	SFDB_TRACE_ENTER;

	/* Free internal structures */
	iwl_sfdb_em_free_dma_ptr(&iwl_sfdb_em->bc_tables);
	iwl_sfdb_em_free_dma_ptr(&iwl_sfdb_em->kw_buffer);

	/* Free Queues */
	for (i = 0; i < SFDB_NUM_QUEUES; i++) {
		IWL_SFDB_LOG("SFDB: Freeing Queue %d", i);
		q = SFDB_GET_QUEUE(i);
		iwl_sfdb_em_free_dma_ptr(&q->pci_tfds);
	}

#ifdef CPTCFG_IWLWIFI_TRANS_UT
	/*Stop reclaim flow interrupts */
	tasklet_kill(&iwl_sfdb_em->reclaim_flow);
#endif

	IWL_SFDB_LOG("SFDB Freed");
	SFDB_TRACE_EXIT;
}

/**
 * This function initializes a TFD queue and configures FH with the base
 * address of this queue.
 * Allocates a DMA memory for the PCIe TFD queue.
 *
 * @param queue_id - the number of queue.
 */
void iwl_sfdb_em_init_queue(u8 queue_id)
{
	int ret;
	struct iwl_tfd_em_queue *q = SFDB_GET_QUEUE(queue_id);
	SFDB_TRACE_ENTER;

	BUG_ON(queue_id >= SFDB_NUM_QUEUES);

	/* Allocate pci TFDs*/
	IWL_SFDB_LOG("Initializing queue %d - %s Allocated",
		     queue_id,
		     (!q->pci_tfds.addr) ? "Not" : "Already");

	/* If this pointer has been allocated before, just zero the memory.
	 * This pointer is zeroed on sfdb init -  so if it's not zero it means
	 * that the memory was already allocated */
	if (!q->pci_tfds.addr) {
		ret = iwl_sfdb_em_alloc_dma_ptr(&q->pci_tfds,
			sizeof(struct iwl_tfd) * SFDB_TFD_NUM_ENTRIES);
		if (ret) {
			IWL_SFDB_LOG_ERR("Failed to allocate DMA memory (%d)",
					 ret);
			WARN_ON(true);
		}
	} else {
		memset(q->pci_tfds.addr, 0,
		       sizeof(struct iwl_tfd) * SFDB_TFD_NUM_ENTRIES);
	}

	/* Reset the queue */
	iwl_sfdb_em_reset_queue(queue_id);

	/*Configure FH with the TFD ring address of the Q */
	iwl_sfdb_em_config_hw(queue_id);
	SFDB_TRACE_EXIT;
}

/**
 * Writes the given value to the queues tfd entry in the Byte Count table.
 *
 * @ param queue_id - The queue id to update the byte count value.
 * @ param tfd_entry - The tfd entry in the queue to which we update the bc.
 * @ param value - The value to write to BC_TABLE[queue_id][tfd_entry]
 */
void iwl_sfdb_em_write_bc(u8 queue_id, u8 tfd_entry, __le16 value)
{
	struct iwl_sfdb_em_bc_tbl *tbl =
		(struct iwl_sfdb_em_bc_tbl *)iwl_sfdb_em->bc_tables.addr +
		queue_id;

	SFDB_TRACE_ENTER;
	IWL_SFDB_LOG("Writing BC to queue %d, tfd %d , BC value %d",
		     queue_id, tfd_entry, value);

	tbl->sfdb_queue_bc_table[SFDB_QUEUE_PTR(tfd_entry)] = value;
	tbl->sfdb_queue_bc_table[SFDB_TFD_NUM_ENTRIES +
			 SFDB_QUEUE_PTR(tfd_entry)] = value;
	SFDB_TRACE_EXIT;
}

/**
 * Writes the given value to the LUT table in the tfd entry.
 *
 * @ param queue_id - The queue id to update the LUT value.
 * @ param tfd_entry - The tfd entry in the queue to which we update the LUT.
 * @ param value - The value to write to LUT_TABLE[queue_id][tfd_entry]
 */
void iwl_sfdb_em_write_lut_value(u8 queue_id, u8 tfd_entry, u16 value)
{
	struct iwl_tfd_em_queue *q = SFDB_GET_QUEUE(queue_id);

	SFDB_TRACE_ENTER;
	BUG_ON(queue_id >= SFDB_NUM_QUEUES);

	IWL_SFDB_LOG("Writing LUT value to queue %d, tfd %d , LUT value %d",
		     queue_id, tfd_entry, value);

	q->lut_table[tfd_entry] = value;

	SFDB_TRACE_EXIT;
}

/**
 * Frees all TFDs in the queue up to (but not including) given tfd_entry.
 * Note that tfd_entry is a pointer to the queue (like the read pointer),
 * not an index in the TFD pool.
 *
 * @param queue_id - The number of queue.
 * @param tfd_index - The index of the TFD in the TFD pool
 */
void iwl_sfdb_em_release_tfds(u8 queue_id, u8 tfd_entry)
{
	u8 read;
	struct iwl_tfd_em_queue *q = SFDB_GET_QUEUE(queue_id);

	SFDB_TRACE_ENTER;
	BUG_ON(queue_id >= SFDB_NUM_QUEUES);

	read = SFDB_QUEUE_PTR(q->read_ptr);
	IWL_SFDB_LOG(
		"Releasing TFDs: queue %d, from slot %d (read pointer) until %d (inclusive)\n",
		queue_id, read, tfd_entry);

	/* This increment will make the release until the
	original tfd_entry inclusive. */
	tfd_entry = SFDB_QUEUE_PTR(tfd_entry + 1);

	while (read != tfd_entry) {
		iwl_sfdb_em_deattach_tfd(queue_id, read);
		SFDB_INC_PTR(q->read_ptr);
		read = SFDB_QUEUE_PTR(q->read_ptr);
	}

	SFDB_TRACE_EXIT;
}

/**
 * Sets the next free entry in the lut of the queue
 * to point to IDI TFD index from the pool.
 *
 * @param queue_id - The number of queue.
 * @param tfd_index - The index of the TFD in the TFD pool
 *
 * @return void
 */
void iwl_sfdb_em_attach_tfd(u8 queue_id, u8 tfd_index)
{
	int i;
	struct iwl_tfd_em_queue *q = SFDB_GET_QUEUE(queue_id);
	u32 write_ptr = SFDB_QUEUE_PTR(q->write_ptr);
	struct iwl_tfd *pcie_tfd =
				(struct iwl_tfd *)q->pci_tfds.addr + write_ptr;
	struct iwl_sfdb_idi_tfd *idi_tfd =
		(struct iwl_sfdb_idi_tfd *)iwl_sfdb_em_get_idi_tfd(tfd_index);

	SFDB_TRACE_ENTER;
	BUG_ON(queue_id >= SFDB_NUM_QUEUES);

	IWL_SFDB_LOG("Attaching TFD %u to queue %u in slot %u",
		     tfd_index, queue_id, write_ptr);
	memset(pcie_tfd, 0, sizeof(struct iwl_tfd));

	for (i = 0; i < idi_tfd->numTbs; ++i) {
		iwl_sfdb_em_convert_tb(&idi_tfd->tbs[i], &pcie_tfd->tbs[i], i);
		pcie_tfd->num_tbs++;
	}

	SFDB_TRACE_EXIT;
}

/**
 * This function deattach TFD from the queue.
 * Because we use a PCIe queue, there is no "de-attaching" as removing the
 * TFD, but only invalidating the relevant index in tfd_indices (telling that no
 * TFD in the pool now referenced to that slot in the queue).
 *
 * @param U08 : queue_id  -- the number of queue
 * @param U08 : tfdEntry -- the entry in the TFD queue
 */
void iwl_sfdb_em_deattach_tfd(u8 queue_id, u8 tfd_entry)
{
	struct iwl_tfd_em_queue *q = SFDB_GET_QUEUE(queue_id);
	SFDB_TRACE_ENTER;
	BUG_ON(queue_id >= SFDB_NUM_QUEUES);

	IWL_SFDB_LOG("Deattaching TFD %u from queue %u",
		     tfd_entry, queue_id);
	q->lut_table[tfd_entry] = SFDB_INVALID_ENTRY;

	/* Invalidating BC */
	iwl_sfdb_em_write_bc(queue_id, tfd_entry, cpu_to_le16(1));

	SFDB_TRACE_EXIT;
}

#ifdef CPTCFG_IWLWIFI_TRANS_UT

/**
 * Get device command from the latest packet in the given queue id.
 */
static struct iwl_device_cmd *iwl_sfdb_em_get_current_cmd(u8 queue_id)
{
	struct iwl_tfd_em_queue *q = SFDB_GET_QUEUE(queue_id);
	u32 write_ptr = SFDB_QUEUE_PTR(q->write_ptr);
	struct iwl_sfdb_idi_tfd *idi_tfd;
	u8 tfd_index;
	void *virt_addr;

	SFDB_TRACE_ENTER;
	BUG_ON(queue_id >= SFDB_NUM_QUEUES);


	tfd_index = q->lut_table[write_ptr];
	idi_tfd = (struct iwl_sfdb_idi_tfd *)iwl_sfdb_em_get_idi_tfd(tfd_index);
	virt_addr = (void *)iwl_sfdb_em_get_idi_tb(idi_tfd->tbs[0].tb_addr);
	return (struct iwl_device_cmd *)virt_addr;
}

/**
 * Calls the IDI reclaim flow.
 * Tasklet context.
 */
static void iwl_sfdb_em_dispatch_reclaim_emu(unsigned long data)
{
	struct iwl_rx_cmd_buffer *rxcb = (struct iwl_rx_cmd_buffer *)data;

	/* Call the mock reclaim flow in tasklet context */
	iwl_idi_tx_handle_dispatch_emu(iwl_sfdb_em->trans, rxcb);

	/* Free variables */
	kfree(rxcb);
}

/**
 * Reclaim flow emulation for the integration.
 */
static void iwl_sfdb_em_test_reclaim(u8 queue_id)
{
	struct iwl_rx_cmd_buffer *rxcb;
	struct iwl_rx_packet *pkt;
	struct iwl_device_cmd *dev_cmd;
	struct iwl_idi_trans_rx *trans_rx =
			IWL_TRANS_GET_IDI_TRANS_RX(iwl_sfdb_em->trans);
	struct iwl_trans_slv *trans_slv =
			IWL_TRANS_GET_SLV_TRANS(iwl_sfdb_em->trans);

	/* Alloc RX command buffer */
	rxcb = kzalloc(sizeof(struct iwl_rx_cmd_buffer), GFP_KERNEL);
	if (!rxcb) {
		IWL_SFDB_LOG_ERR("Can't allocate rxcb");
		return;
	}

	/* Alloc and zero page for command buffer */
	rxcb->_page = alloc_pages((GFP_KERNEL | __GFP_COMP | __GFP_ZERO),
				  trans_slv->rx_page_order);
	if (!rxcb->_page) {
		IWL_SFDB_LOG_ERR("Can't allocate rxcb->page");
		kfree(rxcb);
		return;
	}

	/* extract last device cmd from TB */
	dev_cmd = iwl_sfdb_em_get_current_cmd(queue_id);

	/* Update packet in local rxcb */
	pkt = rxb_addr(rxcb);
	pkt->hdr.cmd = dev_cmd->hdr.cmd;
	pkt->hdr.sequence = dev_cmd->hdr.sequence;

	/* Call mock reclaim flow in tasklet context*/
	spin_lock(&iwl_sfdb_em->gate);
	tasklet_unlock_wait(&iwl_sfdb_em->reclaim_flow);
	iwl_sfdb_em->reclaim_flow.data = (unsigned long)rxcb;
	tasklet_schedule(&iwl_sfdb_em->reclaim_flow);
	spin_unlock(&iwl_sfdb_em->gate);
}
#endif /* CPTCFG_IWLWIFI_TRANS_UT */

/**
 * Increments the write pointer and updates the SCD register.
 *
 * @param queue_id - The number of queue.
 */
void iwl_sfdb_em_inc_write_ptr(u8 queue_id)
{
	struct iwl_tfd_em_queue *q = SFDB_GET_QUEUE(queue_id);

	SFDB_TRACE_ENTER;
	BUG_ON(queue_id >= SFDB_NUM_QUEUES);

	IWL_SFDB_LOG("Incrementing WRITE pointer on queue %d, Old %d",
		     queue_id,
		     iwl_sfdb_em_get_write_ptr(queue_id));

#ifdef CPTCFG_IWLWIFI_TRANS_UT
	iwl_sfdb_em_test_reclaim(queue_id);
	SFDB_INC_PTR(q->write_ptr);
#else
	SFDB_INC_PTR(q->write_ptr);
	iwl_sfdb_em_hw_set_write_ptr(queue_id, SFDB_SCD_PTR(q->write_ptr));
#endif
	SFDB_TRACE_EXIT;
}

/**
 * Gets the write pointer of the given queue.
 *
 * @param queue_id - The number of queue.
 *
 * Returns the write pointer of the queue.
 */
u32  iwl_sfdb_em_get_write_ptr(u8 queue_id)
{
	struct iwl_tfd_em_queue *q = SFDB_GET_QUEUE(queue_id);
	SFDB_TRACE_ENTER;
	BUG_ON(queue_id >= SFDB_NUM_QUEUES);

	SFDB_TRACE_EXIT;
	return SFDB_SCD_PTR(q->write_ptr);
}

/**
 * Update Write pointer of Scheduler for queue_id
 *
 * @param queue_id - The number of queue
 * @param index - The index to set the write pointer
 */
void iwl_sfdb_em_hw_set_write_ptr(u8 queue_id, u32 index)
{
	/* TODO: does the same assumptions of PCI hold here?
	  PCI code assumes that after setting CSR_GP_CNTRL this function will
	  be called again, and this time it will update the write pointer */

	SFDB_TRACE_ENTER;
	BUG_ON(queue_id >= SFDB_NUM_QUEUES);

	iwl_write32(iwl_sfdb_em->trans, HBUS_TARG_WRPTR,
		    (u32)index | ((u32)queue_id << 8));
	IWL_SFDB_LOG("HW write pointer set to %d on Queue %d, requested %d",
		     iwl_sfdb_em_get_write_ptr(queue_id),
		     queue_id,
		     index);

	SFDB_TRACE_EXIT;
}

/**
 * Returns the read pointer of the queue
 *
 * @param queue_id - the number of queue.
 *
 * Returns the read pointer of the given queue.
 */
u8 iwl_sfdb_em_get_read_ptr(u8 queue_id)
{
	struct iwl_tfd_em_queue *q = SFDB_GET_QUEUE(queue_id);
	SFDB_TRACE_ENTER;
	BUG_ON(queue_id >= SFDB_NUM_QUEUES);

	SFDB_TRACE_EXIT;
	return SFDB_SCD_PTR(q->read_ptr);
}

/**
 * Update read pointer of Scheduler for queue_id
 *
 * @param queue_id - The number of queue
 * @param index - The index to set the read pointer
 */
void iwl_sfdb_em_hw_set_read_ptr(u8 queue_id, u8 index)
{
	struct iwl_tfd_em_queue *q = SFDB_GET_QUEUE(queue_id);
	SFDB_TRACE_ENTER;
	BUG_ON(queue_id >= SFDB_NUM_QUEUES);

	IWL_SFDB_LOG("Setting READ pointer on Q %d to %d", queue_id, index);
	q->read_ptr = index;
	iwl_write_prph(iwl_sfdb_em->trans,
		       SCD_QUEUE_RDPTR(queue_id),
		       q->read_ptr);
	SFDB_TRACE_EXIT;
}

/**
 * This function updates the byte count table for this queue and index
 *
 * @param queue_id - the number of queue
 * @param byte_cnt - the number of bytes for this frame
 *
 * @return void
 */
void iwl_sfdb_em_update_byte_count(u8 queue_id, __le16 byte_cnt)
{
	struct iwl_tfd_em_queue *q = SFDB_GET_QUEUE(queue_id);
	SFDB_TRACE_ENTER;
	BUG_ON(queue_id >= SFDB_NUM_QUEUES);

	iwl_sfdb_em_write_bc(queue_id, SFDB_QUEUE_PTR(q->write_ptr), byte_cnt);
	SFDB_TRACE_EXIT;
}

/**
 * Configures the LMAC (FH/SCD) for using this queue
 *
 * @param queue_id - The number of queue
 */
void iwl_sfdb_em_config_hw(u8 queue_id)
{
	struct iwl_tfd_em_queue *q = SFDB_GET_QUEUE(queue_id);
	SFDB_TRACE_ENTER;
	BUG_ON(queue_id >= SFDB_NUM_QUEUES);

	IWL_SFDB_LOG("Configure HW to use queue %d", queue_id);
	iwl_write_direct32(iwl_sfdb_em->trans,
			   FH_MEM_CBBC_QUEUE(queue_id),
			   q->pci_tfds.dma >> 8);

	SFDB_TRACE_EXIT;
}

/**
 * This function sets the context data for the queue
 *
 * @queue_id - the number of queue
 *
 * @return void
 */
void iwl_sfdb_em_set_ctx_data(u8 queue_id, u32 scd_base_addr)
{
	iwl_trans_write_mem32(iwl_sfdb_em->trans, scd_base_addr +
			SCD_CONTEXT_QUEUE_OFFSET(queue_id), 0);
	iwl_trans_write_mem32(iwl_sfdb_em->trans, scd_base_addr +
			SCD_CONTEXT_QUEUE_OFFSET(queue_id) +
			sizeof(u32),
			((64 <<
			SCD_QUEUE_CTX_REG2_WIN_SIZE_POS) &
			SCD_QUEUE_CTX_REG2_WIN_SIZE_MSK) |
			((64 <<
			SCD_QUEUE_CTX_REG2_FRAME_LIMIT_POS) &
			SCD_QUEUE_CTX_REG2_FRAME_LIMIT_MSK));
}

/**
 * This function sets the status of this queue (to SCD status register)
 *
 * @queueId - the number of queue
 * @isActive - is the queue active
 * @isPan - is the queue for PAN
 * @txFifo - the Q->Fifo mapping
 *
 * @return void
 */
void iwl_sfdb_em_set_status(u8 queue_id, u32 is_active,
			    u32 is_pan, u32 tx_fifo)
{
	iwl_write_prph(iwl_sfdb_em->trans, SCD_QUEUE_STATUS_BITS(queue_id),
		       (is_active << SCD_QUEUE_STTS_REG_POS_ACTIVE) |
		       (tx_fifo << SCD_QUEUE_STTS_REG_POS_TXF) |
		       (1 << SCD_QUEUE_STTS_REG_POS_WSL) |
		       (is_pan << SCD_QUEUE_STTS_REG_POS_SCD_ACT_EN) |
		       SCD_QUEUE_STTS_REG_MSK);
}

/**
 * This function returns the status of this queue (from SCD status register)
 *
 * @queueId- The number of queue.
 */
u32 iwl_sfdb_em_get_status(u8 queue_id)
{
	return iwl_idi_tg_read32(iwl_sfdb_em->trans,
				 SCD_QUEUE_STATUS_BITS(queue_id));
}

/**
 * Returns if the queue is aggregated.
 *
 * @param queue_id - The number of queue
 *
 * Returns TRUE if the queue is active, FALSE otherwise.
 */
bool iwl_sfdb_em_queue_is_aggregated(u8 queue_id)
{
	struct iwl_tfd_em_queue *q = SFDB_GET_QUEUE(queue_id);
	BUG_ON(queue_id >= SFDB_NUM_QUEUES);
	return q->is_aggregated;
}
