#ifndef __iwl_sdio_tx_h__
#define __iwl_sdio_tx_h__

#include "shared.h"
#include "sdio_al.h"

#define IWL_SDIO_TB_SIZE 512
#define IWL_SDIO_TB_SIZE_MASK (IWL_SDIO_TB_SIZE - 1)
#define IWL_SDIO_MAX_TBS_NUM 7
#define IWL_SDIO_CB_QUEUE_SIZE 64
#define IWL_SDIO_CB_QUEUES_NUM 32
#define IWL_SDIO_BC_TABLE_ENTRY_SIZE_BYTES 2
#define IWL_SDIO_SRAM_TABLE_EMPTY_PTFD_CELL 0xFFFFFFFF
#define IWL_SDIO_SRAM_TABLE_EMPTY_BC_CELL 0
#define IWL_SDIO_PTFDS_ROW_SIZE 4
#define IWL_SDIO_BC_ROW_SIZE 2
#define IWL_SDIO_SEND_BUF_LEN (8 * 1024)

/**
 * struct iwl_sdio_sram_alloc - internal representation  of sram allocations
 * @ptfd:	ptfd idx
 * @tfd:	tfd idx
 * @txq_id:	tx queue id
 * @num_tbs:	number of AL SRAM TBs needed
 * @tbs:	TB indices
 */
struct iwl_sdio_sram_alloc {
	u16 ptfd;
	u16 tfd;
	u8 txq_id;
	u8 num_tbs;
	u32 tbs[IWL_SDIO_MAX_TBS_NUM];
};

/**
 * struct iwl_sdio_tfd - compressed TFD for SDTM
 * @reserved:	unused
 * @num_tbs:	number of used TBs
 * @tbs:	bits [0..19] address, bits [20..31] length in bytes
 */
struct iwl_sdio_tfd {
	u8 reserved[3];
	u8 num_tbs;
	__le32 tbs[IWL_SDIO_MAX_TBS_NUM];
} __packed;

/* SD ADMA definitions;
 * According to paragraph 1.13.4 of SD HC specification v3.00
 */

#define IWL_SDIO_ADMA_ATTR_VALID 0x01
#define IWL_SDIO_ADMA_ATTR_END	 0x02
#define IWL_SDIO_ADMA_ATTR_ACT1	 0x10
#define IWL_SDIO_ADMA_ATTR_ACT2	 0x20

struct iwl_sdio_adma_desc {
	u8 attr;
	u8 reserved;
	__le16 length;
	__le32 addr;
} __packed;

#define IWL_SDIO_ADMA_DESC_MAX_NUM 13

/**
 * struct iwl_sdio_tx_dtu_hdr - the header for DTU
 * @hdr:	sdio command header
 * @dma_desc:	destination address and length of ADMA descriptors sequence
 * @reserved:
 * @adma_list:	ADMA descriptors, one for each destination in DTU
 */
struct iwl_sdio_tx_dtu_hdr {
	struct iwl_sdio_cmd_header hdr;
	__le32 dma_desc;
	u8 reserved[8];
	u8 adma_list[0];
} __packed;

/**
 * struct iwl_sdio_tx_dtu_trailer - the trailer of DTU
 * @ptfd:	ptfd index
 * @scd_bc:	byte count in double words
 * @scd_wr_ptr:	updated scheduler write pointer
 * @padding:	aligned to the size of the SDIO block, filled with 0xAC
 */
struct iwl_sdio_tx_dtu_trailer {
	__le32 ptfd;
	__le32 scd_bc;
	__le32 scd_wr_ptr;
	u8 padding[0];
} __packed;

/**
 * struct iwl_sdio_dtu_info - internal met data for DTU
 * @sram_alloc:		allocation indices
 * @data_pad_len:	ADMA descriptor can handle only data length aligned
 *			to 4 bytes. If the length of the data to be sent is not
 *			aligned, only the last TB (and corresponding ADMA desc)
 *			will suffer; In this case, ADMA desc will be configured
 *			to move more data (up to 3 bytes). This is the length of
 *			this padding.
 * @block_pad_len:	the total length of a DTU must be aligned to SDIO block
 *			size; this is the length of this padding.
 * @adma_desc_num:	total number of ADMA descriptors
 * @ctrl_buf:	holds DTU header, trailer and TFD
 */
struct iwl_sdio_dtu_info {
	struct iwl_sdio_sram_alloc sram_alloc;
	u32 data_pad_len;
	u32 block_pad_len;
	u8 adma_desc_num;
	u8 ctrl_buf[0];
};

void iwl_sdio_tx_stop(struct iwl_trans *trans);
void iwl_sdio_tx_free(struct iwl_trans *trans);
void iwl_sdio_tx_start(struct iwl_trans *trans, u32 scd_base_addr);
int iwl_sdio_process_dtu(struct iwl_trans_slv *trans_slv, u8 txq_id);
void iwl_sdio_tx_free_dtu_mem(struct iwl_trans *trans, void **data);
void iwl_sdio_tx_free_resources(struct iwl_trans *trans, void *data);
void iwl_trans_sdio_txq_enable(struct iwl_trans *trans, int txq_id, int fifo,
			       int sta_id, int tid, int frame_limit, u16 ssn);
void iwl_trans_sdio_txq_disable(struct iwl_trans *trans, int txq_id);

#endif
