#include <linux/sched.h>
#include <linux/wait.h>
#include <linux/gfp.h>

#include "iwl-io.h"
#include "iwl-op-mode.h"
#include "iwl-trans.h"
#include "idi_internal.h"
#include "idi_utils.h"
#include "shared.h"
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
#include "iwl-idi.h"
#endif

#define IWL_IDI_RX_RES_NAME "rx fifo"

static struct iwl_trans *g_trans;

static void iwl_idi_rx_process_pages(struct work_struct *data);
static void iwl_idi_rx_sg_process(struct iwl_idi_trans_rx *trans_rx);
static void iwl_idi_rx_handler_kick0(struct iwl_idi_trans_rx *trans_rx);
static void iwl_idi_rx_handler_data(struct iwl_idi_trans_rx *trans_rx);
static void iwl_idi_rx_sm_move(struct iwl_idi_trans_rx *trans_rx);

#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
/**
* iwl_idi_rx_handler - this is the ISR for the DBB DMA Rx interrupt.
*
*/
static void __maybe_unused iwl_idi_rx_handler(__le32 chan, __le32 dir,
					      void *ctx)
{
	struct iwl_idi_trans_rx *trans_rx = (struct iwl_idi_trans_rx *)ctx;

	if (trans_rx->sm.state == IWL_IDI_RX_STOPPED)
		return;
	else if (trans_rx->sm.state == IWL_IDI_RX_KICK0)
		iwl_idi_rx_handler_kick0(trans_rx);
	else
		iwl_idi_rx_handler_data(trans_rx);
}

#else

static void iwl_idi_rx_complete(struct idi_transaction *transaction)
{
	struct iwl_idi_trans_rx *trans_rx = IWL_TRANS_GET_IDI_TRANS_RX(g_trans);

	if (transaction->status != IDI_STATUS_COMPLETE)
		return;

	if (trans_rx->sm.state == IWL_IDI_RX_STOPPED)
		return;
	else if (trans_rx->sm.state == IWL_IDI_RX_KICK0)
		iwl_idi_rx_handler_kick0(trans_rx);
	else
		iwl_idi_rx_handler_data(trans_rx);
}

int iwl_idi_rx_set_channel_config(struct idi_peripheral_device *pdev)
{
	struct idi_channel_config conf = {0};
	struct idi_resource *idi_res;
	struct resource *res;
	int ret;

	idi_res = &pdev->resources;

	res = idi_get_resource_byname(idi_res, IORESOURCE_MEM,
				      IWL_IDI_RX_RES_NAME);
	if (!res) {
		dev_err(&pdev->device, "Failed to get resource %s\n",
			IWL_IDI_RX_RES_NAME);
		return -EINVAL;
	}

	conf.tx_or_rx = 0;
	conf.priority = IDI_NORMAL_PRIORITY;
	conf.channel_opts = IDI_PRIMARY_CHANNEL;
	conf.dst_addr = res->start;
	conf.hw_fifo_size = resource_size(res);

	ret = idi_set_channel_config(pdev, &conf);
	if (ret)
		dev_err(&pdev->device, "Failed in idi_set_channel_config\n");

	return ret;
}

static int iwl_idi_rx_alloc_transaction(struct iwl_idi_trans_rx *trans_rx)
{
	struct iwl_trans *trans = IWL_TRANS_RX_GET_TRANS(trans_rx);
	trans_rx->transaction = idi_alloc_transaction(GFP_KERNEL);
	if (!trans_rx->transaction) {
		IWL_ERR(trans, "%s failed to allocate IDI transaction\n",
			__func__);
		return -ENOMEM;
	}

	trans_rx->transaction->complete = iwl_idi_rx_complete;
	trans_rx->transaction->context = trans_rx;
	trans_rx->transaction->idi_xfer.channel_opts = IDI_PRIMARY_CHANNEL;

	return 0;
}

#endif /* CPTCFG_IWLWIFI_IDI_OVER_PCI */

static inline
struct page *iwl_idi_alloc_pages(gfp_t mask, u32 page_order)
{
	struct page *page;

	if (page_order > 0)
		mask |= __GFP_COMP;
	page = alloc_pages(mask, page_order);
	return page;
}

static
struct iwl_rx_mem_buffer *iwl_idi_alloc_rxb(struct iwl_idi_trans_rx *trans_rx,
					    gfp_t flags)
{
	struct iwl_trans *trans = IWL_TRANS_RX_GET_TRANS(trans_rx);
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);

	dma_addr_t aligned_addr;
	struct iwl_rx_mem_buffer *rxb;

	rxb = kmem_cache_alloc(trans_rx->rxb_pool, flags);
	if (unlikely(!rxb))
		return NULL;

	rxb->page = iwl_idi_alloc_pages(flags, trans_slv->rx_page_order);
	if (!rxb->page) {
		kmem_cache_free(trans_rx->rxb_pool, rxb);
		return NULL;
	}

	rxb->page_dma = dma_map_page(trans->dev,
				     rxb->page, 0,
				     PAGE_SIZE << trans_slv->rx_page_order,
				     DMA_FROM_DEVICE);
	if (dma_mapping_error(trans->dev, rxb->page_dma)) {
		IWL_ERR(trans, "Failed to map DMA page.\n");
		__free_pages(rxb->page, trans_slv->rx_page_order);
		kmem_cache_free(trans_rx->rxb_pool, rxb);
		return NULL;
	}

	aligned_addr = ALIGN(rxb->page_dma, 4);
	WARN_ON(aligned_addr != rxb->page_dma);

	return rxb;
}

static void iwl_idi_rx_free_rxb(struct iwl_idi_trans_rx *trans_rx,
			struct iwl_rx_mem_buffer *rxb)
{
	struct iwl_trans *trans = IWL_TRANS_RX_GET_TRANS(trans_rx);
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);

	if (rxb == NULL)
		return;

	if (rxb->page) {
		dma_unmap_page(trans->dev,
			       rxb->page_dma,
			       PAGE_SIZE << trans_slv->rx_page_order,
			       DMA_FROM_DEVICE);
		__free_pages(rxb->page, trans_slv->rx_page_order);
	}
	kmem_cache_free(trans_rx->rxb_pool, rxb);
}

static inline void iwl_idi_rx_free_fifo_rxbs(struct iwl_idi_trans_rx *trans_rx,
				      struct kfifo *fifo)
{
	int count;
	struct iwl_rx_mem_buffer *rxb;

	while ((count = kfifo_out(fifo, (void *)&rxb, sizeof(void *))) != 0)
		iwl_idi_rx_free_rxb(trans_rx, rxb);

	kfifo_free(&trans_rx->page_fifo);
}

static inline void iwl_idi_rx_page_pool_free(struct iwl_idi_trans_rx *trans_rx)
{
	cancel_work_sync(&trans_rx->page_refill_wrk);
	iwl_idi_rx_free_fifo_rxbs(trans_rx, &trans_rx->page_fifo);
}

static
struct iwl_rx_mem_buffer *
iwl_idi_rx_page_pool_get(struct iwl_idi_trans_rx *trans_rx)
{
	struct iwl_rx_mem_buffer *rxb;
	int ret;

	ret = kfifo_out(&trans_rx->page_fifo, (void *)&rxb, sizeof(void *));
	if (!ret)
		rxb = NULL;

	if ((kfifo_len(&trans_rx->page_fifo) <
		IWL_IDI_RX_BG_POOL_TRSHLD*sizeof(void *))) {
		queue_work(trans_rx->wq, &trans_rx->page_refill_wrk);
	}

	/* try to allocate with GFP_ATOMIC in case of temporary empty pool */
	if (rxb == NULL)
		rxb = iwl_idi_alloc_rxb(trans_rx, GFP_ATOMIC);

	return rxb;
}

/**
 * iwl_idi_rx_page_pool_alloc - allocate page pool
 * @trans_rx - rx transport
 * Returns 0 only if succeeded to allocate everything - fifo, rxbs and pages.
 * Otherwise returns -ENOMEM and frees all the allocated resources.
 */
static int iwl_idi_rx_page_pool_alloc(struct iwl_idi_trans_rx *trans_rx)
{
	int i;
	struct iwl_rx_mem_buffer *rxb;
	struct iwl_trans *trans = IWL_TRANS_RX_GET_TRANS(trans_rx);

	if (kfifo_alloc(&trans_rx->page_fifo,
			IWL_IDI_RX_BG_POOL_SIZE * sizeof(void *),
			GFP_KERNEL)) {
		IWL_ERR(trans, "kfifo_alloc failed");
		return -ENOMEM;
	}
	kfifo_reset(&trans_rx->page_fifo);

	for (i = 0; i < IWL_IDI_RX_BG_POOL_SIZE; i++) {
		rxb = iwl_idi_alloc_rxb(trans_rx, GFP_KERNEL);
		if (!rxb) {
			IWL_WARN(trans, "bg pool alloc - not enough memory.");
			goto error;
		}
		kfifo_in(&trans_rx->page_fifo, (void *)&rxb, sizeof(void *));
	}

	return 0;
error:
	iwl_idi_rx_free_fifo_rxbs(trans_rx, &trans_rx->page_fifo);
	return -ENOMEM;
}

static void iwl_idi_rx_page_pool_refill(struct work_struct *data)
{
	struct iwl_rx_mem_buffer *rxb;
	struct iwl_idi_trans_rx *trans_rx =
		container_of(data, struct iwl_idi_trans_rx, page_refill_wrk);
	struct iwl_trans *trans = IWL_TRANS_RX_GET_TRANS(trans_rx);

	while (kfifo_len(&trans_rx->page_fifo) <
			 IWL_IDI_RX_BG_POOL_SIZE*sizeof(void *)) {
		rxb = iwl_idi_alloc_rxb(trans_rx, GFP_KERNEL);
		if (!rxb) {
			IWL_WARN(trans, "refill bg page pool failed.");
			return;
		}

		kfifo_in(&trans_rx->page_fifo, (void *)&rxb, sizeof(void *));
	}
}

static
struct iwl_idi_sg_list_holder *iwl_rx_get_sg_list(struct list_head *sg_lists)
{
	struct iwl_idi_sg_list_holder *cur_sg_list;

	cur_sg_list = list_first_entry(sg_lists, struct iwl_idi_sg_list_holder,
				       list);
	list_del(&cur_sg_list->list);

	return cur_sg_list;
}

static void iwl_idi_rx_free_sg_list_data(struct iwl_trans *trans,
			struct iwl_idi_sg_list_holder *sg_list_holder)
{
	struct iwl_idi_trans_rx *trans_rx = IWL_TRANS_GET_IDI_TRANS_RX(trans);
	int i;

	if (sg_list_holder == NULL)
		return;

	for (i = 0; i < SG_LIST_MAX_SIZE; i++)
		iwl_idi_rx_free_rxb(trans_rx, sg_list_holder->addr_map[i]);

	iwl_idi_free_sg_list(trans, &sg_list_holder->sg_list);
	kfree(sg_list_holder);
}

static void iwl_idi_rx_free_sg_lists(struct iwl_trans *trans)
{
	struct iwl_idi_trans_rx *trans_rx = IWL_TRANS_GET_IDI_TRANS_RX(trans);
	struct iwl_idi_sg_list_holder *cur_sg_list, *tmp;

	if (unlikely(trans_rx == NULL))
		return;

	list_for_each_entry_safe(cur_sg_list, tmp, &trans_rx->free_sg_lists,
				 list)
		iwl_idi_rx_free_sg_list_data(trans, cur_sg_list);

	list_for_each_entry_safe(cur_sg_list, tmp, &trans_rx->used_sg_lists,
				 list)
		iwl_idi_rx_free_sg_list_data(trans, cur_sg_list);

	iwl_idi_rx_free_sg_list_data(trans, trans_rx->armed_sg_list);
	iwl_idi_free_sg_list(trans, &trans_rx->sg_list_kick0);
}

/**
* iwl_idi_rx_stop - should be called from stop_device
*/
void iwl_idi_rx_stop(struct iwl_trans *trans)
{
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	int ret;
#else
	struct iwl_trans_idi *trans_idi = IWL_TRANS_GET_IDI_TRANS(trans);
#endif
	struct iwl_idi_trans_rx *trans_rx =
			IWL_TRANS_GET_IDI_TRANS_RX(trans);

	trans_rx->sm.state = IWL_IDI_RX_STOPPED;

#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	ret = iwl_al_dbb_stop(cpu_to_le32(IWL_IDI_RX_CHANNEL),
			      cpu_to_le32(IWL_IDI_DBB_RX));
	if (ret < 0)
		IWL_ERR(trans, "Rx DBB stop failed, ret = %d\n", ret);
#else
	/* flash IDI RX channel */
	idi_peripheral_flush(trans_idi->pdev, IDI_PRIMARY_CHANNEL);
#endif

	cancel_work_sync(&trans_rx->page_proc_wrk);

	clear_bit(IWL_IDI_RX_SM_MISC_DMA_ARMED, &trans_rx->sm.misc);

	iwl_idi_rx_page_pool_free(trans_rx);
}

/**
* iwl_idi_rx_free - implements rx_free callback.
*
*/
void iwl_idi_rx_free(struct iwl_trans *trans)
{
	struct iwl_idi_trans_rx *trans_rx;

	/* sanity */
	BUG_ON(trans == NULL);
	trans_rx = IWL_TRANS_GET_IDI_TRANS_RX(trans);
	BUG_ON(trans_rx == NULL);
	if (trans_rx->trans_idi == NULL)
		return;

	if (trans_rx->sm.state != IWL_IDI_RX_STOPPED) {
		IWL_WARN(trans,
			 "An attempt to free Rx transport when not STOPPED\n");
		return;
	}

	iwl_idi_rx_free_sg_lists(trans);
	iwl_idi_rx_free_fifo_rxbs(trans_rx, &trans_rx->ready_pages_fifo);

	if (trans_rx->rxb_pool) {
		kmem_cache_destroy(trans_rx->rxb_pool);
		trans_rx->rxb_pool = NULL;
	}

	if (trans_rx->wq) {
		destroy_workqueue(trans_rx->wq);
		trans_rx->wq = NULL;
	}

#ifndef CPTCFG_IWLWIFI_IDI_OVER_PCI
	if (trans_rx->transaction)
		idi_free_transaction(trans_rx->transaction);
#endif
}

static void iwl_idi_rx_sg_list_reset_kick0(struct iwl_idi_trans_rx *trans_rx)
{
	struct idi_sg_desc *virt;

	virt = (struct idi_sg_desc *)trans_rx->sg_list_kick0.addr;

	virt->next = cpu_to_le32(IWL_IDI_SG_LIST_END | IWL_IDI_SG_LIST_INT);
	virt->base = cpu_to_le32(trans_rx->sg_list_kick0.dma +
				 sizeof(struct idi_sg_desc));
	virt->size = cpu_to_le32(IWL_IDI_RX_SIG_SIZE);

}

/**
* iwl_idi_rx_sg_list_reset - reset all the pointers in the given SG list
* The same SG list should not be reset simultaneously by several threds,
* thus the sync is needed only when accessing free mem buffers list.
*
* @idx - the index of the SG list
*/
static void iwl_idi_rx_sg_list_reset(struct iwl_idi_trans_rx *trans_rx,
				struct iwl_idi_sg_list_holder *sg_list_hld)
{
	struct iwl_trans *trans = IWL_TRANS_RX_GET_TRANS(trans_rx);
	struct iwl_rx_mem_buffer *rxb;
	struct idi_sg_desc *cur_virt;
	dma_addr_t cur_dma;
	int i;

	cur_virt = (struct idi_sg_desc *)sg_list_hld->sg_list.addr;
	cur_dma = sg_list_hld->sg_list.dma;

	for (i = 0; i < SG_LIST_MAX_SIZE; i++) {
		/* this is the point where DMA finished writing. It can happen
		* only in the first DMA descriptor out of the 4 that compose
		* the logical one. Note that it cannot be the last dma
		* descriptor because of the loop stop condition.
		*/
		if (cur_virt->next & cpu_to_le32(IWL_IDI_SG_LIST_END)) {
			cur_virt->next = cpu_to_le32(cur_dma +
					sizeof(struct idi_sg_desc));
			goto sg_list_ready;
		}

		rxb = iwl_idi_rx_page_pool_get(trans_rx);
		if (!rxb) {
			IWL_WARN(trans,
			"Failed to reset SG list - no free mem bufs\n");
			return;
		}

		/* configure 1st descriptor */
		cur_virt->next = cpu_to_le32(cur_dma +
					     sizeof(struct idi_sg_desc));
		cur_virt->base = cpu_to_le32(cur_dma +
					     sizeof(struct idi_sg_desc) +
					     offsetof(struct idi_sg_desc,
					     size));
		cur_virt->size = cpu_to_le32(sizeof(u32));
		cur_virt->pad = 0;

		/* configure 2nd descriptor */
		cur_virt += 1;
		cur_dma += sizeof(struct idi_sg_desc);

		cur_virt->next = cpu_to_le32(cur_dma +
					     sizeof(struct idi_sg_desc));
		cur_virt->base = cpu_to_le32(rxb->page_dma);

		sg_list_hld->addr_map[i] = rxb;
		cur_virt->size = 0;

		/* configure 3rd descriptor */
		cur_virt += 1;
		cur_dma += sizeof(struct idi_sg_desc);

		cur_virt->next = cpu_to_le32(cur_dma +
					     sizeof(struct idi_sg_desc));
		cur_virt->base = cpu_to_le32(cur_dma +
					     sizeof(struct idi_sg_desc) +
					     offsetof(struct idi_sg_desc,
					     size));
		cur_virt->size = cpu_to_le32(sizeof(u32));
		cur_virt->pad = 0;

		/* configure 4th descriptor */
		cur_virt += 1;
		cur_dma += sizeof(struct idi_sg_desc);

		cur_virt->next = cpu_to_le32(cur_dma +
					     sizeof(struct idi_sg_desc));
		cur_virt->base = cpu_to_le32(cur_dma +
					     offsetof(struct idi_sg_desc, pad));
		cur_virt->size = 0;
		cur_virt->pad = 0;

		/* move to the next logical descriptor */
		cur_virt += 1;
		cur_dma += sizeof(struct idi_sg_desc);
	}
	cur_virt->next = 0;
	cur_virt->size = 0;

sg_list_ready:
	/* from this moment the list can be used again */
	list_add(&sg_list_hld->list, &trans_rx->free_sg_lists);

	return;
}

static int iwl_idi_rx_prepare_sg_ll(struct iwl_trans *trans)
{
	struct iwl_idi_trans_rx *trans_rx = IWL_TRANS_GET_IDI_TRANS_RX(trans);
	size_t sg_byte_size;
	int i, ret;

	/*
	 * initialize data SG lists, allocate two additional physical
	 * descriptors to keep end-of-data notification.
	 */
	sg_byte_size = (SG_LIST_MAX_SIZE*SG_LIST_DMA_DESC_NUM + 2)*
		sizeof(struct idi_sg_desc);
	for (i = 0; i < SG_LIST_NUM; i++) {
		struct iwl_idi_sg_list_holder *cur_sg_list;

		cur_sg_list = kmalloc(sizeof(struct iwl_idi_sg_list_holder),
				      GFP_KERNEL);
		if (!cur_sg_list)
			goto error;

		ret = iwl_idi_alloc_sg_list(trans, &cur_sg_list->sg_list,
					    sg_byte_size);
		if (ret) {
			IWL_WARN(trans,
				 "Rx init failed - not enough memory.");
			goto error;
		}

		memset(cur_sg_list->addr_map, 0,
		       SG_LIST_MAX_SIZE*sizeof(void *));
		iwl_idi_rx_sg_list_reset(trans_rx, cur_sg_list);
	}

	/* initilaize kick0 SG list
	 * The memory for receiving signature data is appended
	 * at the end of SG list.
	 */
	sg_byte_size = sizeof(struct idi_sg_desc) +
		       sizeof(u16) + IWL_IDI_RX_SIG_SIZE;
	ret = iwl_idi_alloc_sg_list(trans, &trans_rx->sg_list_kick0,
				    sg_byte_size);
	if (ret) {
		IWL_WARN(trans, "Rx init failed - not enough memory.");
		goto error;
	}
	iwl_idi_rx_sg_list_reset_kick0(trans_rx);

	return 0;

error:
	iwl_idi_rx_free_sg_lists(trans);
	return -ENOMEM;
}

static int iwl_idi_rx_alloc_data(struct iwl_trans *trans)
{
	int ret;
	struct iwl_idi_trans_rx *trans_rx = IWL_TRANS_GET_IDI_TRANS_RX(trans);

	trans_rx->rxb_pool = kmem_cache_create("iwl_idi_rxb",
					       sizeof(struct iwl_rx_mem_buffer),
					       sizeof(void *), 0, NULL);
	if (!trans_rx->rxb_pool) {
		IWL_ERR(trans, "IDI Rx init failed\n");
		return -ENOMEM;
	}

	if (kfifo_alloc(&trans_rx->ready_pages_fifo,
			IWL_IDI_RX_READY_PAGES_MAX_NUM * sizeof(void *),
			GFP_KERNEL)) {
		IWL_ERR(trans, "alloc ready_pages_fifo failed");
		goto err_free_cache;
	}
	kfifo_reset(&trans_rx->ready_pages_fifo);

	ret = iwl_idi_rx_page_pool_alloc(trans_rx);
	if (ret)
		goto err_free_fifo;

	ret = iwl_idi_rx_prepare_sg_ll(trans);
	if (ret)
		goto err_free_mem_rsrc;

#ifndef CPTCFG_IWLWIFI_IDI_OVER_PCI
	ret = iwl_idi_rx_alloc_transaction(trans_rx);
	if (ret) {
		IWL_ERR(trans, "Rx init failed - IDI transaction.");
		goto err_free_all;
	}
#endif

	return 0;

#ifndef CPTCFG_IWLWIFI_IDI_OVER_PCI
err_free_all:
	iwl_idi_rx_free_sg_lists(trans);
#endif

err_free_mem_rsrc:
	iwl_idi_rx_free_fifo_rxbs(trans_rx, &trans_rx->page_fifo);
err_free_fifo:
	kfifo_free(&trans_rx->ready_pages_fifo);
err_free_cache:
	kmem_cache_destroy(trans_rx->rxb_pool);
	return -ENOMEM;
}

int iwl_idi_rx_init(struct iwl_trans *trans)
{
	struct iwl_idi_trans_rx *trans_rx;
	int ret;

	BUG_ON(trans == NULL);
	BUG_ON(trans->trans_specific == NULL);

	trans_rx = IWL_TRANS_GET_IDI_TRANS_RX(trans);
	trans_rx->trans_idi = IWL_TRANS_GET_IDI_TRANS(trans);

	g_trans = trans;

	INIT_LIST_HEAD(&trans_rx->free_sg_lists);
	INIT_LIST_HEAD(&trans_rx->used_sg_lists);

	trans_rx->wq = alloc_workqueue("idi_rx_wq",
				       WQ_UNBOUND | WQ_NON_REENTRANT, 1);
	INIT_WORK(&trans_rx->page_refill_wrk, iwl_idi_rx_page_pool_refill);
	INIT_WORK(&trans_rx->page_proc_wrk, iwl_idi_rx_process_pages);

	trans_rx->sm.state = IWL_IDI_RX_STOPPED;
	set_bit(IWL_IDI_RX_SM_MISC_KICK0_IS_ON, &trans_rx->sm.misc);
	clear_bit(IWL_IDI_RX_SM_MISC_DMA_ARMED, &trans_rx->sm.misc);

	IWL_INFO(trans, "IDI Rx config: SGs %d, pages %d\n",
		 SG_LIST_NUM, IWL_IDI_RX_BG_POOL_SIZE);

	ret = iwl_idi_rx_alloc_data(trans);
	if (ret)
		return ret;

#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	iwl_al_dbb_dma_set_irq_handler(cpu_to_le32(IWL_IDI_RX_CHANNEL),
				       cpu_to_le32(IWL_IDI_DBB_RX),
				       iwl_idi_rx_handler,
				       (void *)trans_rx);
#endif
	clear_bit(IWL_IDI_RX_SM_MISC_DMA_ARMED, &trans_rx->sm.misc);

	iwl_idi_rx_sm_move(trans_rx);

	IWL_INFO(trans, "%s: finished.\n", __func__);
	return 0;
}

static inline bool iwl_idi_rx_sm_idle(struct iwl_idi_trans_rx *trans_rx)
{
	if (test_bit(IWL_IDI_RX_SM_MISC_DMA_ARMED, &trans_rx->sm.misc))
		return false;

	return true;
}

static int iwl_idi_rx_verify(struct iwl_idi_dma_ptr *sg_list_kick0)
{
#ifndef CPTCFG_IWLWIFI_IDI_OVER_PCI
	struct idi_sg_desc *sg_desc;
	u32 *payload_data;

	sg_desc = (struct idi_sg_desc *)sg_list_kick0->addr;
	if (WARN(sg_desc->size != sizeof(u32),
		 "KICK0 payload size is wrong: %d instead of %d\n",
		 sg_desc->size, sizeof(u32)))
		return -EINVAL;

	payload_data = (u32 *)phys_to_virt(sg_desc->base);
	if (WARN(payload_data[0] != RX_KCK0_SIGNATURE,
		 "KICK0 siganture is not valid: 0x%x instead of 0x%x\n",
		 payload_data[0], RX_KCK0_SIGNATURE))
		return -EINVAL;
#endif
	return 0;
}

static void iwl_idi_rx_handler_kick0(struct iwl_idi_trans_rx *trans_rx)
{
	int ret;
	struct iwl_trans *trans =
		IWL_TRANS_RX_GET_TRANS(trans_rx);

	ret = iwl_idi_rx_verify(&trans_rx->sg_list_kick0);
	if (ret) {
		/* TODO: handle KICK0 wrong signature */
		IWL_ERR(trans, "verify signature failed\n");
	}

	clear_bit(IWL_IDI_RX_SM_MISC_DMA_ARMED, &trans_rx->sm.misc);
	iwl_idi_rx_sm_move(trans_rx);
}
static void iwl_idi_rx_handler_data(struct iwl_idi_trans_rx *trans_rx)
{
	/* move the armed S/G list to used */
	if (WARN_ON(NULL == trans_rx->armed_sg_list))
		return;

	list_add_tail(&trans_rx->armed_sg_list->list,
		      &trans_rx->used_sg_lists);
	trans_rx->armed_sg_list = NULL;
	clear_bit(IWL_IDI_RX_SM_MISC_DMA_ARMED, &trans_rx->sm.misc);

	/* schedule processing of the received sg list*/
	iwl_idi_rx_sm_move(trans_rx);
	iwl_idi_rx_sg_process(trans_rx);

	return;
}

static void iwl_idi_rx_process_one_page(struct iwl_idi_trans_rx *trans_rx,
			struct iwl_rx_mem_buffer *rxb)
{
	struct iwl_trans *trans = IWL_TRANS_RX_GET_TRANS(trans_rx);
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);

	struct iwl_rx_cmd_buffer rxcb;

	dma_unmap_page(trans->dev,
		       rxb->page_dma,
		       PAGE_SIZE << trans_slv->rx_page_order,
		       DMA_FROM_DEVICE);

	rxcb._page = rxb->page;
	/*
	 * IDI currently doesn't support multiple frames in one
	 * rxb, so just resetting rxcb here.
	 */
	rxcb._offset = 0;
	rxcb._page_stolen = false;
	iwl_slv_rx_handle_dispatch(trans, &rxcb);

	if (rxcb._page_stolen) {
		__free_pages(rxb->page,
			     trans_slv->rx_page_order);
		rxb->page = NULL;
	}

	/* FIXME: reuse this page */
	if (rxb->page != NULL)
		__free_pages(rxb->page, trans_slv->rx_page_order);

	kmem_cache_free(trans_rx->rxb_pool, rxb);
}

static void iwl_idi_rx_process_pages(struct work_struct *data)
{
	struct iwl_idi_trans_rx *trans_rx =
		container_of(data, struct iwl_idi_trans_rx, page_proc_wrk);
	struct iwl_trans *trans = IWL_TRANS_RX_GET_TRANS(trans_rx);
	struct iwl_rx_mem_buffer *rxb;
	int count;

	while ((count = kfifo_out(&trans_rx->ready_pages_fifo,
				(void *)&rxb, sizeof(void *))) != 0) {
		if (WARN_ON(rxb == NULL)) {
			/* FIXME: define behaviour */
			IWL_ERR(trans, "Invalid rxb in ready_pages_fifo\n");
			continue;
		}
		iwl_idi_rx_process_one_page(trans_rx, rxb);
	}
}

/**
 * iwl_idi_rx_sg_process - processing of the Rx SG list.
 *
 * FIXME: the current implementation still supports having several
 * used SG LLs, which could happen when it was in a separate tasklet.
 * Now, this function runs in the context of the interrupt taslet,
 * thus only one used SG LL is possible. Need to consider moving
 * to a simpler implementation.
 */
static void iwl_idi_rx_sg_process(struct iwl_idi_trans_rx *trans_rx)
{
	struct idi_sg_desc *sg_virt;
	struct iwl_trans *trans = IWL_TRANS_RX_GET_TRANS(trans_rx);
	struct iwl_idi_sg_list_holder *cur_sg_list;
	struct list_head processed_lists;
	int i;

	INIT_LIST_HEAD(&processed_lists);

	/*
	 * Go over all SG lists that are back from DMA and still weren't
	 * processed. The situation when number of such lists is more
	 * than 1 could arise when the Rx interrupt with a new SG list
	 * arrives during the processing of the previous one.
	 */
	while (!list_empty(&trans_rx->used_sg_lists)) {
		cur_sg_list = iwl_rx_get_sg_list(&trans_rx->used_sg_lists);

		sg_virt = (struct idi_sg_desc *)
					cur_sg_list->sg_list.addr;

		/* push all pages from SG LL to fifo */
		for (i = 0; i < SG_LIST_MAX_SIZE; i++) {
			/* This indicates the end of the received data. */
			if (((sg_virt->next & cpu_to_le32(IWL_IDI_SG_LIST_END)) ||
			    (sg_virt->next & cpu_to_le32(IWL_IDI_SG_LIST_INT))))
				break;

			sg_virt += SG_LIST_DMA_DESC_NUM;

			if (kfifo_is_full(&trans_rx->ready_pages_fifo)) {
				IWL_ERR(trans, "Full ready_pages_fifo\n");
				/* FIXME: is it enough to discard this page? */
				iwl_idi_rx_free_rxb(trans_rx,
						    cur_sg_list->addr_map[i]);
				cur_sg_list->addr_map[i] = NULL;
				continue;
			}
			kfifo_in(&trans_rx->ready_pages_fifo,
				 (void *)&cur_sg_list->addr_map[i],
				 sizeof(void *));

			cur_sg_list->addr_map[i] = NULL;
		}

		queue_work(trans_rx->wq, &trans_rx->page_proc_wrk);

		/* optimization: when DMA is idle reset S/G list now */
		if (iwl_idi_rx_sm_idle(trans_rx)) {
			iwl_idi_rx_sg_list_reset(trans_rx, cur_sg_list);
			iwl_idi_rx_sm_move(trans_rx);
		} else {
			list_add_tail(&cur_sg_list->list, &processed_lists);
		}
	}

	while (!list_empty(&processed_lists)) {
		cur_sg_list = iwl_rx_get_sg_list(&processed_lists);

		iwl_idi_rx_sg_list_reset(trans_rx, cur_sg_list);
		iwl_idi_rx_sm_move(trans_rx);
	}

	/*
	* An optimization which moves the state machine in case this run of
	* the tasklet didn't processed any SG list.
	*/
	if (iwl_idi_rx_sm_idle(trans_rx))
		iwl_idi_rx_sm_move(trans_rx);
}

/**
* iwl_idi_rx_dma_arm - set a new Rx SG list to DMA a kick.
*
*/
static int iwl_idi_rx_dma_arm(struct iwl_idi_trans_rx *trans_rx)
{
	struct iwl_trans *trans = IWL_TRANS_RX_GET_TRANS(trans_rx);
	struct iwl_idi_sg_list_holder *cur_sg_list;
	struct iwl_idi_dma_ptr *sg_list = NULL;
	int ret;

	if (trans_rx->sm.state == IWL_IDI_RX_KICK0) {
		sg_list = &trans_rx->sg_list_kick0;
	} else if (!list_empty(&trans_rx->free_sg_lists)) {
		cur_sg_list =
			iwl_rx_get_sg_list(&trans_rx->free_sg_lists);
		trans_rx->armed_sg_list = cur_sg_list;
		sg_list = &cur_sg_list->sg_list;
	}

	if (sg_list == NULL) {
		IWL_ERR(trans, "No free S/G list.");
		return -ENOMEM;
	}
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	ret = iwl_al_dbb_set_sg(cpu_to_le32(IWL_IDI_RX_CHANNEL),
			cpu_to_le32(IWL_IDI_DBB_RX),
			(struct idi_sg_desc *)sg_list->addr);
	ret = iwl_al_dbb_start(cpu_to_le32(IWL_IDI_RX_CHANNEL),
			       cpu_to_le32(IWL_IDI_DBB_RX));
#else
	/* The API of IDI driver will be updated, should be phys addr */
	trans_rx->transaction->idi_xfer.desc = sg_list->dma;
	ret = idi_async_read(trans_rx->trans_idi->pdev, trans_rx->transaction);
	if (ret) {
		IWL_ERR(trans, "IDI async read failed.");
		return ret;
	}
#endif
	return 0;
}

static void iwl_idi_rx_sm_move(struct iwl_idi_trans_rx *trans_rx)
{
	struct iwl_trans *trans = IWL_TRANS_RX_GET_TRANS(trans_rx);

	if (test_and_set_bit(IWL_IDI_RX_SM_MISC_DMA_ARMED, &trans_rx->sm.misc))
		return;

	switch (trans_rx->sm.state) {
	case IWL_IDI_RX_STOPPED:
		if (test_bit(IWL_IDI_RX_SM_MISC_KICK0_IS_ON,
			     &trans_rx->sm.misc)) {
			trans_rx->sm.state = IWL_IDI_RX_KICK0;
			iwl_idi_rx_dma_arm(trans_rx);
		} else if (!list_empty(&trans_rx->free_sg_lists)) {
			trans_rx->sm.state = IWL_IDI_RX_DATA;
			iwl_idi_rx_dma_arm(trans_rx);
		} else {
			IWL_DEBUG_RX(trans, "moving to no SG list state.");
			trans_rx->sm.state = IWL_IDI_RX_NO_SG_LIST;
			clear_bit(IWL_IDI_RX_SM_MISC_DMA_ARMED,
				  &trans_rx->sm.misc);
		}
		break;

	case IWL_IDI_RX_KICK0:
		if (!list_empty(&trans_rx->free_sg_lists)) {
			trans_rx->sm.state = IWL_IDI_RX_DATA;
			iwl_idi_rx_dma_arm(trans_rx);
		} else {
			IWL_DEBUG_RX(trans, "moving to no SG list state.");
			trans_rx->sm.state = IWL_IDI_RX_NO_SG_LIST;
			clear_bit(IWL_IDI_RX_SM_MISC_DMA_ARMED,
				  &trans_rx->sm.misc);
		}
		break;

	case IWL_IDI_RX_DATA:
		if (test_bit(IWL_IDI_RX_SM_MISC_KICK0_IS_ON,
			     &trans_rx->sm.misc)) {
			trans_rx->sm.state = IWL_IDI_RX_KICK0;
			iwl_idi_rx_dma_arm(trans_rx);
		} else if (!list_empty(&trans_rx->free_sg_lists)) {
			/* no kick0 - remain in the current state */
			iwl_idi_rx_dma_arm(trans_rx);
		} else {
			IWL_DEBUG_RX(trans, "moving to no SG list state.");
			trans_rx->sm.state = IWL_IDI_RX_NO_SG_LIST;
			clear_bit(IWL_IDI_RX_SM_MISC_DMA_ARMED,
				  &trans_rx->sm.misc);
		}

		break;

	case IWL_IDI_RX_NO_SG_LIST:
		if (!list_empty(&trans_rx->free_sg_lists)) {
			IWL_DEBUG_RX(trans, "moving out of no SG list state.");
			trans_rx->sm.state = IWL_IDI_RX_DATA;
			iwl_idi_rx_dma_arm(trans_rx);
		}
		break;

	default:
		IWL_ERR(trans, "IDI Rx SM reached invalid state.\n");
		break;
	}
}
