#ifndef __SMS4_H__
#define __SMS4_H__

#include <linux/types.h>

#define SMS4_BLOCK_SIZE     16
#define SMS4_RK_NUM         32
#define SMS4_MAX_ENCRYPTED_TEXT_SIZE 2294

struct sms4_ctx {
	u32 rk_enc[SMS4_RK_NUM];
	u32 rk_dec[SMS4_RK_NUM];
};

/*
 * sms4_set_key - installs a key (in_key) in the given sms4_ctx
 *   Assumption: in_key is of size SMS4_BLOCK_SIZE (defined above to be 16)
 */
void sms4_set_key(struct sms4_ctx *ctx, const u8 *in_key);

/*
 * sms4_encrypt - encrypts a single block of plaintext (in) into a single
 *                block of ciphertext (out), using the given sms4_ctx.
 *   Assumptions: 1. A key was previously installed into ctx
 *                2. Both in and out are of size SMS4_BLOCK_SIZE
 */
void sms4_encrypt(const struct sms4_ctx *ctx, u8 *out, const u8 *in);

/*
 * sms4_decrypt - decrypts a single block of ciphertext (in) into a single
 *                block of plaintext (out), using the given sms4_ctx.
 *   Assumptions: 1. A key was previously installed into ctx
 *                2. Both in and out are of size SMS4_BLOCK_SIZE
 */
void sms4_decrypt(const struct sms4_ctx *ctx, u8 *out, const u8 *in);

/*
 * sms4_crypt_buf_by_ofb - encrypts/decrypts (it's the same process) the given
 *                         buffer (buf) of the given size, using the given
 *                         sms4_ctx and the given initial vector (iv_le).
 *                         'iv_le' stands for iv little endian. this means the
 *                         given buffer starts (iv_le[0]) with the LSB of iv.
 *   Assumption: 1. A key was previously installed into ctx
 *               2. iv_le is of size SMS4_BLOCK_SIZE
 *   Note: output overwrites the initial data in buf.
 */
void sms4_crypt_buf_by_ofb(struct sms4_ctx *ctx, const u8 *iv_le,
	u8 *buf, size_t buf_size);

/*
 * sms4_calculate_mic_by_cbc_mac - calculates the MIC value of the given buffer
 *                         (integrity_check_data) of the given size, using the
 *                         given sms4_ctx and the given initial vector (iv_le).
 *                         'iv_le' stands for iv little endian. this means the
 *                         given buffer starts (iv_le[o]) with the LSB of iv.
 *  Assumption: 1. A key was previously installed into ctx
 *               2. Both iv_le and mic are of size SMS4_BLOCK_SIZE
 */
void sms4_calculate_mic_by_cbc_mac(struct sms4_ctx *ctx, const u8 *iv_le,
	const u8 *integrity_check_data, size_t icd_size, u8 *mic);

#endif
