
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: filelist.h 1555 2006-11-15 09:04:35Z mschwerin $
 *
 */

#ifndef HAVE_FILELIST_H
#define HAVE_FILELIST_H
#ifdef __cplusplus
extern "C" {
#endif

#include "list.h"
#include "mutex.h"
#include "types.h"

#define FILELIST_TOPLEVEL_MRL           "computer:///"
#ifdef HAVE_VDR
#define FILELIST_VDR_RECORDINGS_MRL     "vdr-recordings://"
#endif

typedef enum {
    FILE_TYPE_UNKNOWN,
    FILE_TYPE_REGULAR,          // a normal file
    FILE_TYPE_DIRECTORY,        // a normal directory
    FILE_TYPE_MOUNTPOINT,       // a mountpoint
    FILE_TYPE_MEDIAMARKS,       // a oxine mediamarks file
    FILE_TYPE_M3U,              // a m3u playlist
    FILE_TYPE_AUTODISC,
#ifdef HAVE_VDR
    FILE_TYPE_VDR_RECORDING     // vdr recordings
#endif
} fileitem_type_t;

typedef enum {
    ALLOW_FILES_MUSIC,          // music files only
    ALLOW_FILES_VIDEO,          // video files only
    ALLOW_FILES_SUBTITLE,       // spu files only
    ALLOW_FILES_MULTIMEDIA,     // all extensions xine-lib knows about
    ALLOW_FILES_ALL             // any file
} fileitem_allowed_t;

typedef struct filelist_s filelist_t;
typedef struct fileitem_s fileitem_t;

struct fileitem_s {
    /* The title of this item. */
    char *title;

    /* The MRL of this item. */
    char *mrl;

    /* A description of this fileitem. */
    char *description;

    /* A thumbnail of this fileitem. */
    char *thumbnail_mrl;

    /* User defined data. */
    void *user_data;

    /* The type of this fileitem. */
    fileitem_type_t type;

    /* The list of items below this item. */
    filelist_t *sublist;

    /* The parent of this fileitem. */
    filelist_t *parent;

    /* The number of times this item has already been listened to. */
    // This is only used for the favorites.
    int listened_count;
};

struct filelist_s {
    l_list_t *list;

    /* The title of this list. */
    char *title;

    /* The MRL of this list. */
    char *mrl;

    /* The parent of this list. */
    filelist_t *parent;

    /* The type of files allowed in this filelist. */
    fileitem_allowed_t allowed_filetypes;

    /* A mutex to protect the list. */
    pthread_mutex_t mutex;
    pthread_mutexattr_t mutex_attr;

    /* This is used to save the top position of list when displaying it. */
    int top_position;
    /* This is used to save the current position in the list when displaying it. */
    int cur_position;

    /* The number of references to this filelist. */
    int reference_count;
};


/* 
 * ***************************************************************************
 * Name:            filelist_extensions_init
 * Access:          public
 *
 * Description:     Initializes the list of allowed fileextensions. This is
 *                  done by comparing a list of known audio, video and
 *                  subtitle extensions to the extensions xine-lib knows
 *                  about. Only entries that appear in both lists are added to
 *                  the allowed-lists.
 * ***************************************************************************
 */
void filelist_extensions_init (void);


/* 
 * ***************************************************************************
 * Name:            filelist_extensions_init
 * Access:          public
 *
 * Description:     Frees the allowed-extension-lists.
 * ***************************************************************************
 */
void filelist_extensions_free (void);


/* 
 * ***************************************************************************
 * Name:            filelist_new
 * Access:          public
 *
 * Description:     Creates a new filelist. Only files of the allowed type
 *                  will be added when expanding a directory.
 * ***************************************************************************
 */
filelist_t *filelist_new (filelist_t * parent, const char *title,
                          const char *mrl,
                          fileitem_allowed_t allowed_filetypes);


/* 
 * ***************************************************************************
 * Name:            filelist_clear
 * Access:          public
 *
 * Description:     Removes all fileitems from the list.
 * ***************************************************************************
 */
void filelist_clear (filelist_t * filelist);


/* 
 * ***************************************************************************
 * Name:            filelist_ref_set
 * Access:          public
 *
 * Description:     This function is used to keep track of the number of
 *                  references that point at a filelist. The first parameter
 *                  is a pointer to a filelist pointer, the second the value
 *                  that is to be assigned to that pointer. As soon as the
 *                  reference count reaches 0 the filelist will be freed.
 * ***************************************************************************
 */
void filelist_ref_set (filelist_t ** p, filelist_t * d);


/* 
 * ***************************************************************************
 * Name:            filelist_sort
 * Access:          public
 *
 * Description:     Sorts the list. swap_cb is a callback function, that is
 *                  used to determin if two entries must be swapped.
 * ***************************************************************************
 */
void filelist_sort (filelist_t * filelist, l_swap_cb_t swap_cb);


/* 
 * ***************************************************************************
 * Name:            filelist_add
 * Access:          public
 *
 * Description:     Adds a new entry to the list.
 * ***************************************************************************
 */
fileitem_t *filelist_add (filelist_t * filelist, const char *title,
                          const char *mrl, fileitem_type_t type);


/* 
 * ***************************************************************************
 * Name:            filelist_remove
 * Access:          public
 *
 * Description:     Removes the fileitem from the list.
 * ***************************************************************************
 */
void filelist_remove (filelist_t * filelist, fileitem_t * fileitem);


/* 
 * ***************************************************************************
 * Name:            filelist_expand
 * Access:          public
 *
 * Description:     Depending on the type of the fileitem it is expanded.
 *                  Expanding means that any directory, playlist or
 *                  mediamarks-file is read and added to the sublist of the
 *                  fileitem.
 * ***************************************************************************
 */
void filelist_expand (fileitem_t * fileitem);


/* 
 * ***************************************************************************
 * Name:            filelist_first
 * Access:          public
 *
 * Description:     Returns the first fileitem.
 * ***************************************************************************
 */
fileitem_t *filelist_first (filelist_t * filelist);


/* 
 * ***************************************************************************
 * Name:            filelist_next
 * Access:          public
 *
 * Description:     Returns the next fileitem.
 * ***************************************************************************
 */
fileitem_t *filelist_next (filelist_t * filelist, fileitem_t * fileitem);


/* 
 * ***************************************************************************
 * Name:            filelist_get_by_mrl
 * Access:          public
 *
 * Description:     Returns (if its there) the fileitem with the passed MRL.
 * ***************************************************************************
 */
fileitem_t *filelist_get_by_mrl (filelist_t * filelist, const char *mrl);


/* 
 * ***************************************************************************
 * Name:            filelist_contains
 * Access:          public
 *
 * Description:     Determines, if the list contains an entry with the given
 *                  MRL.
 * ***************************************************************************
 */
bool filelist_contains (filelist_t * filelist, const char *mrl);


/* 
 * ***************************************************************************
 * Name:            filelist_length
 * Access:          public
 *
 * Description:     Returns the number of fileitems in this list.
 * ***************************************************************************
 */
int filelist_length (filelist_t * filelist);


/* 
 * ***************************************************************************
 * Name:            create_title
 * Access:          public
 *
 * Description:     Creates a title from the MRL. The returned string must be
 *                  freed when not used any more.
 * ***************************************************************************
 */
char *create_title (const char *mrl);


/* 
 * ***************************************************************************
 * Name:            is_file_allowed
 * Access:          public
 *
 * Description:     Determines if the file is to be added depending on the
 *                  currently allowed filetypes. The suffix of the file is
 *                  compared to all suffixes of the allowed type.
 * ***************************************************************************
 */
bool is_file_allowed (const char *mrl, int allowed_filetypes);
#ifdef __cplusplus
}
#endif
#endif /* HAVE_FILELIST_H */
