#
# This file is part of Python Terra
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import logging
from logging import *
from logging import addLevelName, getLevelName, basicConfig, \
    shutdown, getLogger, getLoggerClass, \
    logProcesses, logThreads, makeLogRecord, root, setLoggerClass, \
    critical, debug, error, exception, fatal, info, log, warn, warning

DEBUG_ERROR = DEBUG + 1
DEBUG_WARNING = DEBUG + 2

class Logger(logging.Logger):
    def debug_error(self, msg, *args, **kwargs):
        level = DEBUG_ERROR
        if self.manager.disable >= level:
            return
        if self.isEnabledFor(level):
            apply(self._log, (level, msg, args), kwargs)

    def debug_warning(self, msg, *args, **kwargs):
        level = DEBUG_WARNING
        if self.manager.disable >= level:
            return
        if self.isEnabledFor(level):
            apply(self._log, (level, msg, args), kwargs)


class Formatter(logging.Formatter):
    def __init__(self):
        fmt = ("%(created).4f %(levelcolor)s%(levelname)-5.5s\033[0m "
               "\033[1m%(module)s:%(lineno)d:%(funcName)s()\033[0m "
               "%(message)s")
        logging.Formatter.__init__(self, fmt, "%H:%M:%S")

    _level_color = {CRITICAL: "\033[1;5;37;41m",
                    ERROR: "\033[1;31m",
                    WARNING: "\033[1;33m",
                    INFO: "\033[36m",
                    DEBUG: "\033[1;30m",
                    DEBUG_ERROR: "\033[31m",
                    DEBUG_WARNING: "\033[33m",
                    }
    def format(self, record):
        record.levelcolor = self._level_color.get(record.levelno, "")
        return logging.Formatter.format(self, record)


log_handler = logging.StreamHandler()
log_handler.setFormatter(Formatter())
log_root = logging.getLogger()
log_root.addHandler(log_handler)

def setLevel(level):
    log_root.setLevel(level)

def debug_error(msg, *args, **kwargs):
    apply(log_root.debug_error, (msg,)+args, kwargs)

def debug_warning(msg, *args, **kwargs):
    apply(log_root.debug_warning, (msg,)+args, kwargs)

# Change global module to use our system
logging.DEBUG_ERROR = DEBUG_ERROR
logging.DEBUG_WARNING = DEBUG_WARNING
logging.debug_error = debug_error
logging.debug_warning = debug_warning
logging.addLevelName(DEBUG_ERROR, "DBG_ERROR")
logging.addLevelName(DEBUG_WARNING, "DBG_WARNING")
logging.setLoggerClass(Logger)

import types
def _add_log_meth(logger, name, level):
    def _log_meth(self, msg, *args, **kwargs):
        if self.manager.disable >= level:
            return
        if self.isEnabledFor(level):
            apply(self._log, (level, msg, args), kwargs)

    meth = types.MethodType(_log_meth, logger, type(logger))
    setattr(logger, name, meth)

# no way to choose root logger, so hack it.
_add_log_meth(log_root, "debug_error", DEBUG_ERROR)
_add_log_meth(log_root, "debug_warning", DEBUG_WARNING)

del types
del _add_log_meth
