#
# This file is part of Python Terra
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import logging
from dispatcher import send_to_thread_async

from threads import ThreadsManager


log = logging.getLogger("terra.core.threaded_func")


class ThreadedFunction(object):
    (STATE_NONE,
     STATE_RUNNING,
     STATE_COMPLETED) = xrange(3)

    def __init__(self, finished_cb, func, *args, **kargs):
        self._func = func
        self._args = args
        self._kargs = kargs
        self._finished_cb = finished_cb
        self._retval = None
        self._exception = None
        self._state = self.STATE_NONE

        self._thr_mger = ThreadsManager()
        self._thr = None
        self._terra_thr_name = self._thr_mger.thr_name
        self._terra_disp = self._thr_mger.disp

    def _check_completed(self):
        return self._state == self.STATE_COMPLETED

    is_completed = property(_check_completed)

    def _check_running(self):
        return self._state == self.STATE_RUNNING

    is_running = property(_check_running)

    def start(self):
        if self.is_running:
            return
        if self.is_completed:
            if self._finished_cb is not None:
                self._finished_cb(self._exception, self._retval)
            return

        self._state = self.STATE_RUNNING
        self._thr = self._thr_mger.next_worker_thread(self._run)
        self._thr.start()

    def _terminate(self):
        self._thr.join(0.3)
        if self._thr.isAlive():
            log.debug_error("thread %r of %r didn't terminate!",
                            self._thr.getName(), self)

    @send_to_thread_async("_terra_thr_name", "_terra_disp")
    def _finished(self):
        self._state = self.STATE_COMPLETED
        self._terminate()
        if self._finished_cb is not None:
            self._finished_cb(self._exception, self._retval)

    ## Method executed in plugin's thread
    def _run(self):
        try:
            self._retval = self._func(*self._args, **self._kargs)
        except Exception, e:
            log.debug("threaded func %r raised an exception %r",
                      self, e, exc_info=True)
            self._exception = e

        self._finished()

    def __str__(self):
        return "%s(finished_cb=%r, func=%r, args=%r, kargs=%r)" % \
               (self.__class__.__name__, self._finished_cb,
                self._func, self._args, self._kargs)

    __repr__ = __str__
