#
# This file is part of Python Terra
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import ecore
import logging
from evas import ClippedSmartObject
from kinetic import KineticMouse

__all__ = ("RowRenderer", "BaseList", "KineticList")

log = logging.getLogger("terra.ui.list")


class RowRenderer(object):
    def __init__(self, canvas):
        pass

    def value_set(self, v):
        raise NotImplemented("value_set")

    STATE_FIRST = 1
    STATE_DEFAULT = 2
    STATE_LAST = 3
    def state_set(self, state):
        pass


class BaseList(ClippedSmartObject):
    """Basic vertical list of items.

    Row renderers should implement the interface provided by L{RowRenderer},
    that is, a C{value_set()} method that will get the model it should
    use. These renderers should be returned by C{renderer_new}, that will
    be called with the current L{Canvas} as parameter.

    This is a base class and implements no event management.

    @note: this list does not move items by itself, it rely on another
       element to call L{List.move_offset()} or other means to manipulate
       the current position, like L{KineticList}.

    @see: L{KineticList}
    """
    def __init__(self, canvas, renderer_new, elements, renderer_height=32):
        if not callable(renderer_new):
            raise TypeError("renderer_new must be callable")
        ClippedSmartObject.__init__(self, canvas)
        self.clipper.move(0, 0)
        self.elements = elements
        if elements:
            self.current = 0
        else:
            self.current = None
        self._dirty = True
        self._frozen = 0
        self.renderer_new = renderer_new
        self.renderers = []
        self.renderer_height = renderer_height
        self.spare_renderers = 2
        self._position_changed_cb = None
        self._clicked_cb = None
        self.last_top_visible = None
        self.offset = -renderer_height
        self.last_offset = None

    def freeze(self):
        self._frozen += 1

    def thaw(self):
        if self._frozen > 1:
            self._frozen -= 1
        elif self._frozen == 1:
            self._frozen = 0
            if self._dirty:
                self._reconfigure_renderers(*self.geometry_get())
        else:
            log.warning("thaw more than freeze!")

    def model_updated(self):
        "Notifies that model was updated and list need to be redrawn."
        self.request_reconfigure(*self.geometry_get())

    def request_reconfigure(self, x, y, w, h):
        "Request object to reconfigure its children to fit given geometry."
        self._dirty = True
        if self._frozen == 0:
            self._reconfigure_renderers(x, y, w, h)

    def resize(self, w, h):
        x, y, old_w, old_h = self.geometry_get()
        if (old_w, old_h) == (w, h):
            return
        self.clipper.resize(w, h)
        self.request_reconfigure(x, y, w, h)

    def position_changed_cb_set(self, cb, *a, **ka):
        """Set callback used to notify position changes.

        Signature: C{function(list, percent, *a, **ka)}
        """
        if cb is None:
            self._position_changed_cb = None
        elif callable(cb):
            self._position_changed_cb = (cb, a, ka)
        else:
            raise TypeError("cb must be callable or None")

    def position_changed_cb_get(self):
        return self._position_changed_cb

    position_changed_cb = property(position_changed_cb_get,
                                   position_changed_cb_set)

    def emit_position_changed(self):
        "Emit position_changed."
        if self._position_changed_cb is None or len(self.elements) <= 1:
            return

        percent = self.position_get()
        cb, a, ka = self._position_changed_cb
        cb(self, percent, *a, **ka)

    def position_get(self):
        "Get position in list, from 0.0 (top) to 1.0 (bottom)."
        if self.last_top_visible < 1:
            return 0.0
        else:
            rel_offset = self.offset + self.renderer_height
            pos = self.current + float(rel_offset) / -self.renderer_height
            return float(pos) / self.last_top_visible

    def position_set(self, value):
        if value < 0.0:
            log.warning("value (%s) < 0.0, set to 0.0", value)
            value = 0.0
        elif value > 1.0:
            log.warning("value (%s) > 1.0, set to 1.0", value)
            value = 1.0

        max_y = self.last_top_visible * self.renderer_height
        y = int(max_y * value)
        self.current = y / self.renderer_height
        self.offset = -self.renderer_height - (y % self.renderer_height)
        self._reposition_renderers()
        self._refill_renderers()
        self.emit_position_changed()

    def clicked_cb_set(self, cb, *a, **ka):
        """Set callback used to notify click events.

        Signature: C{function(list, index, *a, **ka)}
        """
        if cb is None:
            self._clicked_cb = None
        elif callable(cb):
            self._clicked_cb = (cb, a, ka)
        else:
            raise TypeError("cb must be callable or None")

    def clicked_cb_get(self):
        return self._clicked_cb

    clicked_cb = property(clicked_cb_get, clicked_cb_set)

    def emit_clicked(self, y):
        "Emit clicked at the given vertical position (relative to canvas)."
        if not self._clicked_cb:
            return

        y += - self.pos[1] - self.offset
        idx = y / self.renderer_height - 1 + self.current
        if 0 <= idx < len(self.elements):
            cb, a, ka = self._clicked_cb
            cb(self, idx, *a, **ka)

    def visible_rows_count(self):
        "Return the number of visible rows."
        r = len(self.renderers)
        e = len(self.elements)
        if r == e:
            return e - 1
        elif r > e:
            return e
        else:
            return max(r - self.spare_renderers, 0)

    def visible_rows_scale_get(self):
        "Return the scale of visible rows compared to total number of children."
        v = self.visible_rows_count()
        e = len(self.elements)
        if v >= e or e == 0:
            return 1.0
        else:
            return float(v) / e

    def renderer_for_index(self, index):
        """Get the renderer of the element at index or None if it's not visible.

        @return: row renderer or None if index is not visible.
        """
        if self.current is None:
            return None
        if not (0 <= index < len(self.elements)):
            return None
        base = self.current
        top = base + self.visible_rows_count()
        if base <= index <= top:
            return self.renderers[index - base + 1]
        else:
            return None

    def index_at_y(self, y):
        """Return the index of the element being displayed at given position.

        @return: -1 if out of visible area or index otherwise.
        """
        ox, oy, ow, oh = self.geometry
        dy = y - oy
        if dy < 0 or dy > oh:
            return -1

        n_items = (dy - self.offset) / self.renderer_height - 1
        idx = self.current + n_items
        if idx < len(self.elements):
            return idx
        else:
            return -1

    def _renderer_new_get(self):
        o = self.renderer_new(self.evas)
        self.member_add(o)
        return o

    def _reconfigure_renderers(self, x, y, w, h):
        """Reconfigure renderers based on new geometry.

        Create, delete or change renderers to fit the new geometry.

        @precondition: C{self._dirty == True}
        """
        if not self._dirty:
            return

        c_height = self.renderer_height
        n_items = int(h / c_height)

        if n_items >= len(self.elements):
            self.current = 0
            self.offset = -c_height

        self.last_top_visible = len(self.elements) - n_items
        if h % c_height:
            self.spare_renderers = 3
        else:
            self.spare_renderers = 2
        n_items += self.spare_renderers

        if self.last_top_visible < 0:
            self.last_top_visible = 0

        # delete unneeded
        if n_items < len(self.renderers):
            while n_items < len(self.renderers):
                c = self.renderers.pop()
                c.delete()

        # resize existing
        for c in self.renderers:
            c.resize(w, c_height)

        # create required
        if n_items > len(self.renderers):
            y += len(self.renderers) * c_height + self.offset
            while n_items > len(self.renderers):
                c = self._renderer_new_get()
                c.geometry_set(x, y, w, c_height)
                self.renderers.append(c)
                y += c_height

        self._refill_renderers()
        self._dirty = False

    def _refill_renderers(self):
        """Setup renderers to use the correct models.

        This synchronizes the view and model, also taking care to hide
        renderers not being used.
        """
        if not self.renderers:
            return

        if self.current is None:
            for c in self.renderers:
                c.hide()
            return

        first = self.renderers[0]
        if self.current > 0:
            first.value_set(self.elements[self.current - 1])
            first.show()
        else:
            first.hide()

        start = self.current
        end = start + len(self.renderers) - 1
        elements = self.elements[start:end]
        for i, t in enumerate(elements):
            c = self.renderers[i + 1]
            c.value_set(t)
            c.show()

        start = len(elements) + 1
        for c in self.renderers[start:]:
            c.hide()

        rc = self.visible_rows_count()
        if rc > 1:
            if self.current == 0:
                self.renderers[1].state_set(RowRenderer.STATE_FIRST)
                start = 2
            else:
                start = 0

            if self.current == self.last_top_visible:
                end = rc
                self.renderers[end].state_set(RowRenderer.STATE_LAST)
            else:
                end = len(self.renderers)

            for r in self.renderers[start:end]:
                r.state_set(RowRenderer.STATE_DEFAULT)

    def _reposition_renderers(self):
        """Reposition renderers based on new offset.

        This uses self.offset to reposition renderers.
        """
        if self.offset == self.last_offset:
            return

        self.last_offset = self.offset
        x, y = self.clipper.pos
        y += self.offset
        for renderer in self.renderers:
            renderer.move(x, y)
            y += self.renderer_height

    def move_offset(self, offset):
        """Move (scroll) contents by the given offset.

        @precondition: C{offset != 0 and self.current is not None}
        @return: True if operation was successful or False if the value was
           out of boundaries. The offset will be restricted to respect
           these limits.
        """
        if offset == 0 or self.current is None:
            return False

        max_y = self.last_top_visible * self.renderer_height
        old_y = (self.current - 1) * self.renderer_height - self.offset
        y = old_y - offset

        if y < 0 or y > max_y:
            if y < 0:
                self.current = 0
            else:
                self.current = self.last_top_visible
            self.offset = -self.renderer_height
            self._refill_renderers()
            self._reposition_renderers()
            self.emit_position_changed()
            return False

        idx = y / self.renderer_height
        items_over = idx - self.current
        if items_over == 0:
            self.offset += offset
        elif items_over in (-1, 1):
            self.current += items_over
            self.offset = (self.current - 1) * self.renderer_height - y
            if items_over == 1:
                renderer = self.renderers.pop(0)
                self.renderers.append(renderer)
                new_idx = self.current + len(self.renderers) - 2
            else:
                renderer = self.renderers.pop()
                self.renderers.insert(0, renderer)
                new_idx = self.current - 1

            if 0 <= new_idx < len(self.elements):
                renderer.value_set(self.elements[new_idx])
                renderer.show()
            else:
                renderer.hide()

            rc = self.visible_rows_count()
            if rc > 1:
                if self.current == 0:
                    self.renderers[1].state_set(RowRenderer.STATE_FIRST)
                else:
                    renderer.state_set(RowRenderer.STATE_DEFAULT)

                if self.current == self.last_top_visible:
                    self.renderers[rc].state_set(RowRenderer.STATE_LAST)
                else:
                    renderer.state_set(RowRenderer.STATE_DEFAULT)


        else:
            self.current += items_over
            self.offset = (self.current - 1) * self.renderer_height - y
            self._refill_renderers()

        self._reposition_renderers()
        self.emit_position_changed()
        return True


# TODO: stop aligned to top?
class KineticList(BaseList):
    """List using kinetics.

    Events will not be handled to row renderers, they will be handled by
    event_area, an object that will be laid out on top of every renderer.

    Behavior is based on B{click_constant}: maximum vertical motion to
    consider as click, values greater that (in absolute terms) are considered
    drag.
    """
    click_constant = 20
    move_constant = 12
    click_init_time = 0.4
    click_block_time = 0.5

    def __init__(self, canvas, renderer_new, elements, renderer_height=32):
        BaseList.__init__(self, canvas, renderer_new, elements, renderer_height)
        self.event_area = self.Rectangle(color=(0, 0, 0, 0))
        self.event_area.show()
        self.event_area.on_mouse_down_add(self._cb_on_mouse_down)
        self.event_area.on_mouse_up_add(self._cb_on_mouse_up)
        self.event_area.on_mouse_move_add(self._cb_on_mouse_move)
        self.kinetic = KineticMouse(self.move_offset)
        self.is_drag = False
        self.mouse_down_pos = None
        self.click_timer = None
        self.actual_pos_y = None
        self.click_time_finished = False
        self.clicks_blocked = False
        self._unblock_clicks_timer = None

    def _cb_on_init_click(self, obj, event):
        pass

    def _renderer_new_get(self):
        o = BaseList._renderer_new_get(self)
        return o

    def resize(self, w, h):
        BaseList.resize(self, w, h)
        self.event_area.resize(w, h)

    def remove_click_timer(self):
        if self.click_timer is not None:
            self.click_timer.delete()
            self.click_timer = None

    def _cb_on_mouse_down(self, obj, event):
        if event.button == 1:
            y = event.position.canvas.y
            self.mouse_down_pos = y
            self.is_drag = not self.kinetic.mouse_down(y)

            self.actual_pos_y = y
            self.mouse_down = True

            def cb_init_click():
                if self.mouse_down:
                    self.click_time_finished = True
                    self._cb_on_init_click(obj, event)

            self.remove_click_timer()
            self.click_time_finished = False
            self.click_timer = ecore.timer_add(self.click_init_time, cb_init_click)

    def _is_click_possible(self, y):
        if self.is_drag or self.mouse_down_pos is None:
            return False
        else:
            return abs(y - self.mouse_down_pos) <= self.click_constant

    def _get_pos_smooth(self, y):
        if abs(self.mouse_down_pos - y) <= self.move_constant:
            return y
        elif self.mouse_down_pos - y < 0:
            return y - self.move_constant
        else:
            return y + self.move_constant

    def _cb_on_mouse_up(self, obj, event):
        self.mouse_down = False
        if event.button == 1 and self.mouse_down_pos is not None:
            y = event.position.canvas.y
            if self._is_click_possible(y):
                if not self.clicks_blocked:
                    self.emit_clicked(y)
                self.kinetic.mouse_cancel()
            else:
                self.kinetic.mouse_up(self._get_pos_smooth(y))

            self.remove_click_timer()

    def _cb_on_mouse_move(self, obj, event):
        if event.buttons == 1 and self.mouse_down_pos is not None:
            y = self.actual_pos_y = event.position.canvas.y

            if not self._is_click_possible(y):
                self.is_drag = True
                self.remove_click_timer()

            if abs(self.mouse_down_pos - y) > self.move_constant:
                self.kinetic.mouse_move(self._get_pos_smooth(y))

    def emit_position_changed(self):
        BaseList.emit_position_changed(self)

        if self.is_drag:
            if self._unblock_clicks_timer is not None:
                self._unblock_clicks_timer.delete()

            def cb_unblock_clicks():
                self.clicks_blocked = False

            self.clicks_blocked = True
            self._unblock_clicks_timer = ecore.timer_add(self.click_block_time,
                                                         cb_unblock_clicks)
